// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.




/******************************************************************************

Module Name:

    codeman.h

Abstract:

    Wrapper to facilitate multiple JITcompiler support in the CLR

    The ExecutionManager is responsible for managing the RangeSections.
    Given an IP, it can find the RangeSection which holds that IP.

    RangeSections contain the JITed codes. Each RangeSection knows the
    IJitManager which created it.

    An IJitManager knows about which method bodies live in each RangeSection.
    It can handle methods of one given CodeType. It can map a method body to
    a MethodDesc. It knows where the GCInfo about the method lives.
    Today, we have two IJitManagers:
    1. EEJitManager for JITcompiled code generated by clrjit.dll
    2. ReadyToRunJitManager for version resiliant ReadyToRun code

    An ICodeManager knows how to crack a specific format of GCInfo. There is
    a default format (handled by ExecutionManager::GetDefaultCodeManager())
    which can be shared by different IJitManagers/IJitCompilers.

    An ICorJitCompiler knows how to generate code for a method IL, and produce
    GCInfo in a format which the corresponding IJitManager's ICodeManager
    can handle.

                                               ExecutionManager
                                                       |
                           +-----------+---------------+---------------+-----------+--- ...
                           |           |                               |           |
                        CodeType       |                            CodeType       |
                           |           |                               |           |
                           v           v                               v           v
+---------------+      +--------+<---- R    +---------------+      +--------+<---- R
|ICorJitCompiler|<---->|IJitMan |<---- R    |ICorJitCompiler|<---->|IJitMan |<---- R
+---------------+      +--------+<---- R    +---------------+      +--------+<---- R
                           |       x   .                               |       x   .
                           |        \  .                               |        \  .
                           v         \ .                               v         \ .
                       +--------+      R                           +--------+      R
                       |ICodeMan|                                  |ICodeMan|     (RangeSections)
                       +--------+                                  +--------+

******************************************************************************/

#ifndef __CODEMAN_HPP__

#define __CODEMAN_HPP__

#include "crst.h"
#include "eetwain.h"
#include "ceeload.h"
#include "jitinterface.h"
#include "debuginfostore.h"
#include "shash.h"
#include "pedecoder.h"
#include "gcinfo.h"
#include "eexcp.h"
#ifdef TARGET_X86
#include "gc_unwind_x86.h"
#endif

class MethodDesc;
class ICorJitCompiler;
class IJitManager;
class EEJitManager;
class ReadyToRunJitManager;
class ExecutionManager;
class Thread;
class CrawlFrame;
struct EE_ILEXCEPTION;
struct EE_ILEXCEPTION_CLAUSE;
typedef struct
{
    unsigned iCurrentPos;
    TADDR pExceptionClauseArray;
} EH_CLAUSE_ENUMERATOR;
class EECodeInfo;

#define ROUND_DOWN_TO_PAGE(x)   ( (size_t) (x)                        & ~((size_t)GetOsPageSize()-1))
#define ROUND_UP_TO_PAGE(x)     (((size_t) (x) + (GetOsPageSize()-1)) & ~((size_t)GetOsPageSize()-1))


enum StubCodeBlockKind : int
{
    STUB_CODE_BLOCK_UNKNOWN = 0,
    STUB_CODE_BLOCK_JUMPSTUB = 1,
    UNUSED = 2,
    STUB_CODE_BLOCK_DYNAMICHELPER = 3,
    STUB_CODE_BLOCK_STUBPRECODE = 4,
    STUB_CODE_BLOCK_FIXUPPRECODE = 5,
#ifdef FEATURE_VIRTUAL_STUB_DISPATCH
    STUB_CODE_BLOCK_VSD_DISPATCH_STUB = 6,
    STUB_CODE_BLOCK_VSD_RESOLVE_STUB = 7,
    STUB_CODE_BLOCK_VSD_LOOKUP_STUB = 8,
    STUB_CODE_BLOCK_VSD_VTABLE_STUB = 9,
#endif // FEATURE_VIRTUAL_STUB_DISPATCH
    // Last valid value. Note that the definition is duplicated in debug\daccess\fntableaccess.cpp
    STUB_CODE_BLOCK_LAST = 0xF,
    // Placeholders returned by code:GetStubCodeBlockKind
    STUB_CODE_BLOCK_NOCODE = 0x10,
    STUB_CODE_BLOCK_MANAGED = 0x11,
    STUB_CODE_BLOCK_STUBLINK = 0x12,
    // Placeholdes used by ReadyToRun images
    STUB_CODE_BLOCK_METHOD_CALL_THUNK = 0x13,
};

inline const char *GetStubCodeBlockKindString(StubCodeBlockKind kind)
{
    switch (kind)
    {
    case STUB_CODE_BLOCK_JUMPSTUB:
        return "JumpStub";
    case STUB_CODE_BLOCK_STUBLINK:
        return "StubLinkStub";
    case STUB_CODE_BLOCK_MANAGED:
        return "Managed";
    case STUB_CODE_BLOCK_METHOD_CALL_THUNK:
        return "MethodCallThunk";
    case STUB_CODE_BLOCK_DYNAMICHELPER:
        return "MethodCallThunk";
    case STUB_CODE_BLOCK_FIXUPPRECODE:
        return "MethodCallThunk";
#ifdef FEATURE_VIRTUAL_STUB_DISPATCH
    case STUB_CODE_BLOCK_VSD_DISPATCH_STUB:
        return "VSD_DispatchStub";
    case STUB_CODE_BLOCK_VSD_RESOLVE_STUB:
        return "VSD_ResolveStub";
    case STUB_CODE_BLOCK_VSD_LOOKUP_STUB:
        return "VSD_LookupStub";
    case STUB_CODE_BLOCK_VSD_VTABLE_STUB:
        return "VSD_VTableStub";
#endif // FEATURE_VIRTUAL_STUB_DISPATCH
    default:
        return "Unknown";
    }
}

void ReportStubBlock(void* start, size_t size, StubCodeBlockKind kind);
#ifndef FEATURE_PERFMAP
inline void ReportStubBlock(void* start, size_t size, StubCodeBlockKind kind)
{
}
#endif

//-----------------------------------------------------------------------------
// Method header which exists just before the code.
// Every IJitManager could have its own format for the header.
// Today CodeHeader is used by the EEJitManager.
// The GCInfo version is always current GCINFO_VERSION in this header.

typedef DPTR(struct RealCodeHeader) PTR_RealCodeHeader;
typedef DPTR(struct CodeHeader) PTR_CodeHeader;

struct RealCodeHeader
{
public:
    PTR_BYTE            phdrDebugInfo;

    // Note - *(&(pCodeHeader->phdrJitEHInfo) - sizeof(size_t))
    // contains the number of EH clauses, See CEECodeGenInfo::setEHcountWorker
    PTR_EE_ILEXCEPTION  phdrJitEHInfo;
    PTR_BYTE            phdrJitGCInfo;

#if defined(FEATURE_GDBJIT)
    VOID*            pCalledMethods;
#endif

    PTR_MethodDesc      phdrMDesc;

#ifdef FEATURE_EH_FUNCLETS
    DWORD               nUnwindInfos;
    T_RUNTIME_FUNCTION  unwindInfos[0];
#endif // FEATURE_EH_FUNCLETS
};

struct InterpreterRealCodeHeader
{
public:
    PTR_BYTE            phdrDebugInfo;

    // Note - *(&(pCodeHeader->phdrJitEHInfo) - sizeof(size_t))
    // contains the number of EH clauses, See CEECodeGenInfo::setEHcountWorker
    PTR_EE_ILEXCEPTION  phdrJitEHInfo;
    PTR_BYTE            phdrJitGCInfo;
    PTR_MethodDesc      phdrMDesc;
};

struct CodeHeader
{
    PTR_RealCodeHeader   pRealCodeHeader;

public:
    PTR_BYTE                GetDebugInfo()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrDebugInfo;
    }
    PTR_EE_ILEXCEPTION      GetEHInfo()
    {
        return pRealCodeHeader->phdrJitEHInfo;
    }
    PTR_BYTE                GetGCInfo()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrJitGCInfo;

    }
    PTR_MethodDesc          GetMethodDesc()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrMDesc;
    }
#if defined(FEATURE_GDBJIT)
    VOID*                GetCalledMethods()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->pCalledMethods;
    }
#endif
    TADDR                   GetCodeStartAddress()
    {
        SUPPORTS_DAC;
        return dac_cast<PCODE>(dac_cast<PTR_CodeHeader>(this) + 1);
    }
    StubCodeBlockKind       GetStubCodeBlockKind()
    {
        SUPPORTS_DAC;
        return (StubCodeBlockKind)dac_cast<TADDR>(pRealCodeHeader);
    }
    BOOL                    IsStubCodeBlock()
    {
        SUPPORTS_DAC;
        // Note that it is important for this comparison to be unsigned
        return dac_cast<TADDR>(pRealCodeHeader) <= (TADDR)STUB_CODE_BLOCK_LAST;
    }

    void SetRealCodeHeader(BYTE* pRCH)
    {
        pRealCodeHeader = PTR_RealCodeHeader((RealCodeHeader*)pRCH);
    }

    void SetDebugInfo(PTR_BYTE pDI)
    {
        pRealCodeHeader->phdrDebugInfo = pDI;
    }
    void SetEHInfo(PTR_EE_ILEXCEPTION pEH)
    {
        pRealCodeHeader->phdrJitEHInfo = pEH;
    }
    void SetGCInfo(PTR_BYTE pGC)
    {
        pRealCodeHeader->phdrJitGCInfo = pGC;
    }
    void SetMethodDesc(PTR_MethodDesc pMD)
    {
        pRealCodeHeader->phdrMDesc = pMD;
    }
#if defined(FEATURE_GDBJIT)
    void SetCalledMethods(VOID* pCM)
    {
        pRealCodeHeader->pCalledMethods = pCM;
    }
#endif
    void SetStubCodeBlockKind(StubCodeBlockKind kind)
    {
        pRealCodeHeader = (PTR_RealCodeHeader)kind;
    }

#if defined(FEATURE_EH_FUNCLETS)
    UINT                    GetNumberOfUnwindInfos()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->nUnwindInfos;
    }

    bool MayHaveFunclets()
    {
        return GetNumberOfUnwindInfos() != 1;
    }
    void                    SetNumberOfUnwindInfos(UINT nUnwindInfos)
    {
        LIMITED_METHOD_CONTRACT;
        pRealCodeHeader->nUnwindInfos = nUnwindInfos;
    }
    PTR_RUNTIME_FUNCTION    GetUnwindInfo(UINT iUnwindInfo)
    {
        SUPPORTS_DAC;
        _ASSERTE(iUnwindInfo < GetNumberOfUnwindInfos());
        return dac_cast<PTR_RUNTIME_FUNCTION>(
            PTR_TO_MEMBER_TADDR(RealCodeHeader, pRealCodeHeader, unwindInfos) + iUnwindInfo * sizeof(T_RUNTIME_FUNCTION));
    }
#endif // FEATURE_EH_FUNCLETS


#ifdef DACCESS_COMPILE
    void EnumMemoryRegions(CLRDataEnumMemoryFlags flags, IJitManager* pJitMan);
#endif  // DACCESS_COMPILE

};

typedef DPTR(RealCodeHeader) PTR_RealCodeHeader;
typedef DPTR(InterpreterRealCodeHeader) PTR_InterpreterRealCodeHeader;

#ifdef FEATURE_INTERPRETER

struct InterpreterCodeHeader
{
    PTR_InterpreterRealCodeHeader   pRealCodeHeader;

public:
    PTR_BYTE GetDebugInfo()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrDebugInfo;
    }
    PTR_EE_ILEXCEPTION GetEHInfo()
    {
        return pRealCodeHeader->phdrJitEHInfo;
    }
    PTR_BYTE GetGCInfo()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrJitGCInfo;

    }
    PTR_MethodDesc GetMethodDesc()
    {
        SUPPORTS_DAC;
        return pRealCodeHeader->phdrMDesc;
    }
    TADDR GetCodeStartAddress()
    {
        SUPPORTS_DAC;
        return dac_cast<PCODE>(dac_cast<PTR_CodeHeader>(this) + 1);
    }

    void SetRealCodeHeader(BYTE* pRCH)
    {
        pRealCodeHeader = PTR_InterpreterRealCodeHeader((InterpreterRealCodeHeader*)pRCH);
    }

    void SetDebugInfo(PTR_BYTE pDI)
    {
        pRealCodeHeader->phdrDebugInfo = pDI;
    }
    void SetEHInfo(PTR_EE_ILEXCEPTION pEH)
    {
        pRealCodeHeader->phdrJitEHInfo = pEH;
    }
    void SetGCInfo(PTR_BYTE pGC)
    {
        pRealCodeHeader->phdrJitGCInfo = pGC;
    }
    void SetMethodDesc(PTR_MethodDesc pMD)
    {
        pRealCodeHeader->phdrMDesc = pMD;
    }

    BOOL IsStubCodeBlock()
    {
        return FALSE;
    }

#if defined(FEATURE_EH_FUNCLETS)
    bool MayHaveFunclets()
    {
        LIMITED_METHOD_CONTRACT;
        return true;
    }
    // Used during initialization of the EECodeInfo if MayHaveFunclets returns false.
    // As that can't happen, the implementaiton here is meaningless.
    PTR_RUNTIME_FUNCTION    GetUnwindInfo(UINT iUnwindInfo)
    {
        _ASSERTE(!"Unexpected call to GetUnwindInfoZero");
        return NULL;
    }
#endif

#ifdef DACCESS_COMPILE
    void EnumMemoryRegions(CLRDataEnumMemoryFlags flags, IJitManager* pJitMan);
#endif  // DACCESS_COMPILE
};


typedef DPTR(InterpreterCodeHeader) PTR_InterpreterCodeHeader;

#endif // FEATURE_INTERPRETER

//-----------------------------------------------------------------------------
// This is a structure used to consolidate the information that we
// need we creating new code heaps.
// When creating new JumpStubs we have a constarint that the address used
// should be in the range [loAddr..hiAddr]
//
struct CodeHeapRequestInfo
{
    MethodDesc * m_pMD;
    LoaderAllocator* m_pAllocator;
    const BYTE * m_loAddr;          // lowest address to use to satisfy our request (0 -- don't care)
    const BYTE * m_hiAddr;          // hihest address to use to satisfy our request (0 -- don't care)
    size_t       m_requestSize;     // minimum size that must be made available
    size_t       m_reserveSize;     // Amount that VirtualAlloc will reserved
    size_t       m_reserveForJumpStubs; // Amount to reserve for jump stubs (won't be allocated)
    bool         m_isDynamicDomain;
    bool         m_isCollectible;
    bool         m_isInterpreted;
    bool         m_throwOnOutOfMemoryWithinRange;

    bool   IsDynamicDomain()                    { return m_isDynamicDomain;    }
    void   SetDynamicDomain()                   { m_isDynamicDomain = true;    }

    bool   IsCollectible()                      { return m_isCollectible;      }

    bool   IsInterpreted()                      { return m_isInterpreted;       }
    void   SetInterpreted()                     { m_isInterpreted = true;      }

    size_t getRequestSize()                     { return m_requestSize;        }
    void   setRequestSize(size_t requestSize)   { m_requestSize = requestSize; }

    size_t getReserveSize()                     { return m_reserveSize;        }
    void   setReserveSize(size_t reserveSize)   { m_reserveSize = reserveSize; }

    size_t getReserveForJumpStubs()             { return m_reserveForJumpStubs; }
    void   setReserveForJumpStubs(size_t size)  { m_reserveForJumpStubs = size; }

    bool   getThrowOnOutOfMemoryWithinRange()   { return m_throwOnOutOfMemoryWithinRange; }
    void   setThrowOnOutOfMemoryWithinRange(bool value) { m_throwOnOutOfMemoryWithinRange = value; }

    void   Init();

    CodeHeapRequestInfo(MethodDesc *pMD)
        : m_pMD(pMD), m_pAllocator(0),
          m_loAddr(0), m_hiAddr(0),
          m_requestSize(0), m_reserveSize(0), m_reserveForJumpStubs(0)
        , m_isInterpreted(false)
    { WRAPPER_NO_CONTRACT;   Init(); }

    CodeHeapRequestInfo(MethodDesc *pMD, LoaderAllocator* pAllocator,
                        BYTE * loAddr, BYTE * hiAddr)
        : m_pMD(pMD), m_pAllocator(pAllocator),
          m_loAddr(loAddr), m_hiAddr(hiAddr),
          m_requestSize(0), m_reserveSize(0), m_reserveForJumpStubs(0)
        , m_isInterpreted(false)
    { WRAPPER_NO_CONTRACT;   Init(); }
};

//-----------------------------------------------------------------------------
//
// A CodeHeap is the abstraction the IJitManager uses to allocate memory
// needed to the jitting of a method.
// The CodeHeap works together with the HeapList to manage a contiguous block of memory.
// The CodeHeap is a non growable chunk of memory (it can be reserved and
// committed on demand).
//
// A CodeHeap is naturally protected from multiple threads by the code heap
// critical section - m_CodeHeapLock - so if the implementation of the heap
// is only for the code manager, no locking needs to occur.
// It's important however that a delete operation on the CodeHeap (if any) happens
// via a EECodeGenManager and not directly.
//
// The heap to be created depends on the MethodDesc that is being compiled.
// Standard code uses the LoaderCodeHeap, a heap based on the LoaderHeap.
// DynamicMethods - and only those - use a HostCodeHeap, a heap that does
// normal Alloc/Free so reclamation can be performed.
//
// The convention is that every heap implementation would have a static create
// function that returns a HeapList. The HeapList *must* be properly initialized
// on return except for the next pointer
//

typedef VPTR(class CodeHeap) PTR_CodeHeap;

class CodeHeap
{
    VPTR_BASE_VTABLE_CLASS(CodeHeap)

public:
    CodeHeap() = default;

    // virtual dtor. Clean up heap
    virtual ~CodeHeap() = default;

    // Alloc the specified numbers of bytes for code. Returns NULL if the request does not fit
    // Space for header is reserved immediately before. It is not included in size.
    virtual void* AllocMemForCode_NoThrow(size_t header, size_t size, DWORD alignment, size_t reserveForJumpStubs) = 0;

#ifdef DACCESS_COMPILE
    virtual void EnumMemoryRegions(CLRDataEnumMemoryFlags flags) = 0;
#endif
};

//-----------------------------------------------------------------------------
// The HeapList works together with the CodeHeap to manage a contiguous block of memory.
//
// A single HeapList contains code only for a single AppDomain. EEJitManager uses
// EEJitManager::DomainCodeHeapList to keep a list of HeapLists for each AppDomain.

// The number of code heaps at which we increase the size of new code heaps.
#define CODE_HEAP_SIZE_INCREASE_THRESHOLD 5

typedef DPTR(struct HeapList) PTR_HeapList;

struct HeapList
{
    PTR_HeapList        hpNext;

    PTR_CodeHeap        pHeap;

    TADDR               startAddress;
    TADDR               endAddress;     // the current end of the used portion of the Heap

    TADDR               mapBase;        // "startAddress" rounded down to GetOsPageSize(). pHdrMap is relative to this address
    PTR_DWORD           pHdrMap;        // bit array used to find the start of methods

    size_t              maxCodeHeapSize;// Size of the entire contiguous block of memory
    size_t              reserveForJumpStubs; // Amount of memory reserved for jump stubs in this block

    PTR_LoaderAllocator pLoaderAllocator; // LoaderAllocator of HeapList
#if defined(TARGET_64BIT)
    BYTE*               CLRPersonalityRoutine;  // jump thunk to personality routine, NULL if there is no personality routine (e.g. interpreter code heap)
#endif

    TADDR GetModuleBase()
    {
#if defined(TARGET_64BIT)
        return (CLRPersonalityRoutine != NULL) ? (TADDR)CLRPersonalityRoutine : (TADDR)mapBase;
#else
        return (TADDR)mapBase;
#endif
    }

    PTR_HeapList GetNext()
    { SUPPORTS_DAC; return hpNext; }

    void SetNext(PTR_HeapList next)
    { hpNext = next; }

};

//-----------------------------------------------------------------------------
// Implementation of the standard CodeHeap.
// Use the ExplicitControlLoaderHeap for allocations
// (Check the base class above - CodeHeap - for comments on the functions)
//
typedef VPTR(class LoaderCodeHeap) PTR_LoaderCodeHeap;

class LoaderCodeHeap final : public CodeHeap
{
#ifdef DACCESS_COMPILE
    friend class ClrDataAccess;
#endif

    VPTR_VTABLE_CLASS(LoaderCodeHeap, CodeHeap)

private:
    ExplicitControlLoaderHeap m_LoaderHeap;
    SSIZE_T m_cbMinNextPad;

#ifndef DACCESS_COMPILE
    LoaderCodeHeap(bool fMakeExecutable);
#endif

public:
    static HeapList* CreateCodeHeap(CodeHeapRequestInfo *pInfo, LoaderHeap *pJitMetaHeap);

public:
    virtual ~LoaderCodeHeap() = default;

    virtual void* AllocMemForCode_NoThrow(size_t header, size_t size, DWORD alignment, size_t reserveForJumpStubs) DAC_EMPTY_RET(NULL);

#ifdef DACCESS_COMPILE
    virtual void EnumMemoryRegions(CLRDataEnumMemoryFlags flags)
    {
        WRAPPER_NO_CONTRACT;
        m_LoaderHeap.EnumMemoryRegions(flags);
    }
#endif
};

typedef DPTR(class UnwindInfoTable) PTR_UnwindInfoTable;
// On Windows x64, publish OS UnwindInfo (accessed from RUNTIME_FUNCTION
// structures) to support the ability unwind the stack. Unfortunately the pre-Win8
// APIs defined a callback API for publishing this data dynamically that ETW does
// not use (and really can't because the walk happens in the kernel). In Win8
// new APIs were defined that allow incremental publishing via a table.
//
// UnwindInfoTable is a class that wraps the OS APIs that we use to publish
// this table. Its job is to allocate the table, deallocate it when we are
// done and allow us to add new entries one at a time (AddToUnwindInfoTable).
//
// Each _rangesection has a UnwindInfoTable's which hold the
// RUNTIME_FUNCTION array as well as other bookkeeping (the current and maximum
// size of the array, and the handle used to publish it to the OS.
//
class UnwindInfoTable final
{
public:
    // All public functions are thread-safe.

    // These are wrapper functions over the UnwindInfoTable functions that are specific to JIT compile code
    static void PublishUnwindInfoForMethod(TADDR baseAddress, T_RUNTIME_FUNCTION* unwindInfo, int unwindInfoCount);
    static void UnpublishUnwindInfoForMethod(TADDR entryPoint);

    static void Initialize();

#if defined(TARGET_AMD64) && defined(TARGET_WINDOWS)
private:
    // These are lower level functions that assume you have found the list of UnwindInfoTable entries
    // These are used by the high-level method functions above
    static void AddToUnwindInfoTable(UnwindInfoTable** unwindInfoPtr, T_RUNTIME_FUNCTION* data, TADDR rangeStart, TADDR rangeEnd);
    static void RemoveFromUnwindInfoTable(UnwindInfoTable** unwindInfoPtr, TADDR baseAddress, TADDR entryPoint);

public:
    ~UnwindInfoTable();

private:
    void UnRegister();
    void Register();
    UnwindInfoTable(ULONG_PTR rangeStart, ULONG_PTR rangeEnd, ULONG size);

private:
    PVOID               hHandle;          // OS handle for a published RUNTIME_FUNCTION table
    ULONG_PTR           iRangeStart;      // Start of memory described by this table
    ULONG_PTR           iRangeEnd;        // End of memory described by this table
    T_RUNTIME_FUNCTION* pTable;           // The actual list of method unwind info, sorted by address
    ULONG               cTableCurCount;
    ULONG               cTableMaxCount;
    int                 cDeletedEntries;    // Number of slots we removed.
#endif // defined(TARGET_AMD64) && defined(TARGET_WINDOWS)
};

//-----------------------------------------------------------------------------
// The ExecutionManager uses RangeSection as the abstraction of a contiguous
// address range to track the code heaps.

typedef DPTR(struct RangeSection) PTR_RangeSection;
typedef VPTR(class CodeRangeMapRangeList) PTR_CodeRangeMapRangeList;

class RangeSectionMap;

class Range
{
    // [begin,end) (This is an inclusive range)
    TADDR begin;
    TADDR end;

public:
    Range(TADDR begin, TADDR end) : begin(begin), end(end)
    {
        assert(end >= begin);
    }

    bool IsInRange(TADDR address) const
    {
        return address >= begin && address < end;
    }

    TADDR RangeSize() const
    {
        return end - begin;
    }

    TADDR RangeStart() const
    {
        return begin;
    }

    TADDR RangeEnd() const
    {
        assert(RangeSize() > 0);
        return end - 1;
    }

    TADDR RangeEndOpen() const
    {
        return end;
    }

    friend struct ::cdac_data<RangeSection>;
    friend struct ::cdac_data<RangeSectionMap>;
};

struct RangeSection
{
    friend class RangeSectionMap;
    enum RangeSectionFlags
    {
        RANGE_SECTION_NONE          = 0x0,
        RANGE_SECTION_COLLECTIBLE   = 0x1,
        RANGE_SECTION_CODEHEAP      = 0x2,
        RANGE_SECTION_RANGELIST     = 0x4,
        RANGE_SECTION_INTERPRETER   = 0x8,
    };

#ifdef FEATURE_READYTORUN
    RangeSection(Range range, IJitManager* pJit, RangeSectionFlags flags, PTR_Module pR2RModule) :
        _range(range),
        _flags(flags),
        _pjit(pJit),
        _pR2RModule(pR2RModule),
        _pHeapList(dac_cast<PTR_HeapList>((TADDR)0)),
        _pRangeList(dac_cast<PTR_CodeRangeMapRangeList>((TADDR)0))
#if defined(TARGET_AMD64)
        , _pUnwindInfoTable(dac_cast<PTR_UnwindInfoTable>((TADDR)0))
#endif
    {
        assert(!(flags & RANGE_SECTION_COLLECTIBLE));
        assert(pR2RModule != NULL);
    }
#endif

    RangeSection(Range range, IJitManager* pJit, RangeSectionFlags flags, PTR_HeapList pHeapList) :
        _range(range),
        _flags(flags),
        _pjit(pJit),
        _pR2RModule(dac_cast<PTR_Module>((TADDR)0)),
        _pHeapList(pHeapList),
        _pRangeList(dac_cast<PTR_CodeRangeMapRangeList>((TADDR)0))
#if defined(TARGET_AMD64)
        , _pUnwindInfoTable(dac_cast<PTR_UnwindInfoTable>((TADDR)0))
#endif
    {}

    RangeSection(Range range, IJitManager* pJit, RangeSectionFlags flags, PTR_CodeRangeMapRangeList pRangeList) :
        _range(range),
        _flags(flags),
        _pjit(pJit),
        _pR2RModule(dac_cast<PTR_Module>((TADDR)0)),
        _pHeapList(dac_cast<PTR_HeapList>((TADDR)0)),
        _pRangeList(pRangeList)
#if defined(TARGET_AMD64)
        , _pUnwindInfoTable(dac_cast<PTR_UnwindInfoTable>((TADDR)0))
#endif
    {}

#ifdef DACCESS_COMPILE
    void EnumMemoryRegions(CLRDataEnumMemoryFlags flags);
#endif

    const Range _range;
    const RangeSectionFlags _flags;
    const PTR_IJitManager _pjit;
    const PTR_Module _pR2RModule;
    const PTR_HeapList _pHeapList;
    const PTR_CodeRangeMapRangeList _pRangeList;

#if defined(TARGET_AMD64)
    PTR_UnwindInfoTable _pUnwindInfoTable; // Points to unwind information for this memory range.
#endif // defined(TARGET_AMD64)


    RangeSection* _pRangeSectionNextForDelete = NULL; // Used for adding to the cleanup list

    friend struct ::cdac_data<RangeSection>;
};

template<> struct cdac_data<RangeSection>
{
    static constexpr size_t RangeBegin = offsetof(RangeSection, _range.begin);
    static constexpr size_t RangeEndOpen = offsetof(RangeSection, _range.end);
    static constexpr size_t NextForDelete = offsetof(RangeSection, _pRangeSectionNextForDelete);
    static constexpr size_t JitManager = offsetof(RangeSection, _pjit);
    static constexpr size_t Flags = offsetof(RangeSection, _flags);
    static constexpr size_t HeapList = offsetof(RangeSection, _pHeapList);
    static constexpr size_t R2RModule = offsetof(RangeSection, _pR2RModule);
};

enum class RangeSectionLockState
{
    None,
    NeedsLock,
    ReaderLocked,
    WriteLocked,
};

// For 64bit, we work with 8KB chunks of memory holding pointers to the next level. This provides 10 bits of address resolution per level.
// For *reasons* the X64 hardware is limited to 57bits of addressable address space, and to make the math work out nicely, the minimum granularity that
// is 128KB (or every 2^17 bits) for the tree structure.
//  Similarly the Arm64 specification requires addresses to use at most 52 bits. Thus we use the maximum addressable range of X64 to provide the real max range
// So the first level is bits [56:49] -> L5
// Then                       [48:41] -> L4
//                            [40:33] -> L3
//                            [32:25] -> L2
//                            [24:17] -> L1
// This leaves 17 bits of the address to be handled by the RangeSectionFragment linked list
//
// For 32bit VA processes, use 1KB chunks holding pointers to the next level. This provides 8 bits of address resolution per level.    [31:24] and [23:16].
// For the 32bit processes, only the last 16bits are handled by the RangeSectionFragment linked list.

// Each level of the tree may be considered collectible or non-collectible (except for the top level, which is ALWAYS non-collectible)
// The locking model of the tree structure is as follows.
// Adding a newly allocated level requires holding the Reader lock. Multiple threads may add a level in parallel.
// Removing a level requires holding the Writer lock.
// No level which refers to a non-collectible fragment may be considered collectible.
// A level may be upgraded from collectible to non-collectible, by changing the pointer which points at it to not have the sentinel bit.
// A level may NOT ever be downgraded from non-collectible to collectible.
// When a level becomes empty, it may be freed, and the pointer which points at it may be nulled out.
//
// Within the linked list of RangeSectionFragments, there are effectively 2 lists.
// - The non-collectible list, which are always found first.
// - The collectible list, which follows the non-collectible list.
//
// Insertion into the map uses atomic updates and fully pre-initialized RangeSection structures, so that insertions can be lock-free with regards to each other.
// However, they are not lock-free with regards to removals, so the insertions use a Reader lock.
//
// Reading from the non-collectible data (the non-collectible portion of tree structure + the non-collectible list) does not require any locking at all.
// Reading from the collectible data will require a ReaderLock. There is a scheme using the RangeSectionLockState where when there is an attempt to read without the
// lock and we find collectible data, which will cause the runtime to upgrade to using the Reader lock in that situation.
//
// Reading this code you will ALSO find that the ReaderLock logic used here is intertwined with the GC mode of the process. In particular,
// in cooperative mode and during GC stackwalking, the ReaderLock is always considered to be held.
class RangeSectionMap
{
    class RangeSectionFragment;
    class RangeSectionFragmentPointer;
    typedef DPTR(RangeSectionFragment) PTR_RangeSectionFragment;
    typedef DPTR(RangeSectionFragmentPointer) PTR_RangeSectionFragmentPointer;

    // Helper structure which forces all access to the various pointers to be handled via volatile/interlocked operations
    // The copy/move constructors are all deleted to forbid accidental reads into temporaries, etc.
    class RangeSectionFragmentPointer
    {
    private:
        TADDR _ptr;

        static TADDR FragmentToPtr(RangeSectionFragment* fragment)
        {
            TADDR ptr = dac_cast<TADDR>(fragment);
            if (ptr == 0)
                return ptr;

            if (fragment->isCollectibleRangeSectionFragment)
            {
                ptr += 1;
            }

            return ptr;
        }

        RangeSectionFragmentPointer() { _ptr = 0; }
    public:

        RangeSectionFragmentPointer(RangeSectionFragmentPointer &) = delete;
        RangeSectionFragmentPointer(RangeSectionFragmentPointer &&) = delete;
        RangeSectionFragmentPointer& operator=(const RangeSectionFragmentPointer&) = delete;

        bool PointerIsCollectible()
        {
            return ((_ptr & 1) == 1);
        }

        bool IsNull()
        {
            return _ptr == 0;
        }

        PTR_RangeSectionFragment VolatileLoadWithoutBarrier(RangeSectionLockState *pLockState)
        {
            TADDR ptr = ::VolatileLoadWithoutBarrier(&_ptr);
            if ((ptr & 1) == 1)
            {
                if ((*pLockState == RangeSectionLockState::None) || (*pLockState == RangeSectionLockState::NeedsLock))
                {
                    *pLockState = RangeSectionLockState::NeedsLock;
                    return NULL;
                }
                return dac_cast<PTR_RangeSectionFragment>(ptr - 1);
            }
            else
            {
                return dac_cast<PTR_RangeSectionFragment>(ptr);
            }
        }

#ifndef DACCESS_COMPILE
        void VolatileStore(RangeSectionFragment* fragment)
        {
            ::VolatileStore(&_ptr, FragmentToPtr(fragment));
        }

        bool AtomicReplace(RangeSectionFragment* newFragment, RangeSectionFragment* oldFragment)
        {
            TADDR oldPtr = FragmentToPtr(oldFragment);
            TADDR newPtr = FragmentToPtr(newFragment);

            return oldPtr == InterlockedCompareExchangeT(&_ptr, newPtr, oldPtr);
        }
#endif // DACCESS_COMPILE
    };

    // Helper structure which forces all access to the various pointers to be handled via volatile/interlocked operations
    // The copy/move constructors are all deleted to forbid accidental reads into temporaries, etc.
    template <class TPtr>
    class RangeSectionLevelPointer
    {
    private:
        TADDR _ptr;

        static TADDR LevelToPtr(TPtr level, bool collectible)
        {
            TADDR ptr = dac_cast<TADDR>(level);
            if (ptr == 0)
                return ptr;

            if (collectible)
            {
                ptr += 1;
            }

            return ptr;
        }

        RangeSectionLevelPointer() { _ptr = 0; }
    public:

        RangeSectionLevelPointer(RangeSectionLevelPointer<TPtr> &) = delete;
        RangeSectionLevelPointer(RangeSectionLevelPointer<TPtr> &&) = delete;
        RangeSectionLevelPointer<TPtr>& operator=(const RangeSectionLevelPointer<TPtr>&) = delete;

        bool PointerIsCollectible()
        {
            return ((::VolatileLoadWithoutBarrier(&_ptr) & 1) == 1);
        }

        bool IsNull()
        {
            return _ptr == 0;
        }

        TPtr VolatileLoadWithoutBarrier(RangeSectionLockState *pLockState)
        {
            TADDR ptr = ::VolatileLoadWithoutBarrier(&_ptr);
            if ((ptr & 1) == 1)
            {
                if ((*pLockState == RangeSectionLockState::None) || (*pLockState == RangeSectionLockState::NeedsLock))
                {
                    *pLockState = RangeSectionLockState::NeedsLock;
                    return NULL;
                }
                return dac_cast<TPtr>(ptr - 1);
            }
            else
            {
                return dac_cast<TPtr>(ptr);
            }
        }

        TPtr VolatileLoad(RangeSectionLockState *pLockState)
        {
            TADDR ptr = ::VolatileLoad(&_ptr);
            if ((ptr & 1) == 1)
            {
                if ((*pLockState == RangeSectionLockState::None) || (*pLockState == RangeSectionLockState::NeedsLock))
                {
                    *pLockState = RangeSectionLockState::NeedsLock;
                    return NULL;
                }
                return dac_cast<TPtr>(ptr - 1);
            }
            else
            {
                return dac_cast<TPtr>(ptr);
            }
        }

#ifndef DACCESS_COMPILE

        void UpgradeToNonCollectible()
        {
            TADDR ptr = ::VolatileLoadWithoutBarrier(&_ptr);
            if ((ptr & 1) == 1)
            {
                // Upgrade to non-collectible
#ifdef _DEBUG
                TADDR initialValue =
#endif
                InterlockedCompareExchangeT(&_ptr, ptr - 1, ptr);
                assert(initialValue == ptr || initialValue == (ptr - 1));
            }
        }

        // Install a newly allocated level pointer. Return true if the new buffer is installed.
        // Return false if a buffer is already installed.
        bool Install(TPtr level, bool collectible)
        {
            TADDR initialPointerStoreAttempt = LevelToPtr(level, collectible);
            if (0 == InterlockedCompareExchangeT(&_ptr, initialPointerStoreAttempt, (TADDR)0))
            {
                return true;
            }
            else if (!collectible)
            {
                // In this case we update the already stored level to be pointed at via a non-collectible pointer
                // But since we don't actually install the newly passed in pointer, we still return false.
                UpgradeToNonCollectible();
            }

            return false;
        }

        void Uninstall()
        {
            ::VolatileStore(&_ptr, (TADDR)0);
        }
#endif // DACCESS_COMPILE
    };

    // Unlike a RangeSection, a RangeSectionFragment cannot span multiple elements of the last level of the RangeSectionMap
    // Always allocated via memset/free
    class RangeSectionFragment
    {
    public:
        RangeSectionFragmentPointer pRangeSectionFragmentNext;
        Range _range;
        PTR_RangeSection pRangeSection;
        bool InRange(TADDR address) { return _range.IsInRange(address) && pRangeSection->_pRangeSectionNextForDelete == NULL; }
        bool isPrimaryRangeSectionFragment; // RangeSectionFragment are allocated in arrays, but we only need to free the first allocated one. It will be marked with this flag.
        bool isCollectibleRangeSectionFragment; // RangeSectionFragments
    };

#ifdef TARGET_64BIT
    static constexpr uintptr_t entriesPerMapLevel = 256;
#else
    static constexpr uintptr_t entriesPerMapLevel = 256;
#endif

    typedef RangeSectionFragmentPointer RangeSectionList;
    typedef RangeSectionList RangeSectionL1[entriesPerMapLevel];
    typedef RangeSectionLevelPointer<DPTR(RangeSectionL1)> RangeSectionL2[entriesPerMapLevel];
    typedef RangeSectionLevelPointer<DPTR(RangeSectionL2)> RangeSectionL3[entriesPerMapLevel];
    typedef RangeSectionLevelPointer<DPTR(RangeSectionL3)> RangeSectionL4[entriesPerMapLevel];
    typedef RangeSectionLevelPointer<DPTR(RangeSectionL4)> RangeSectionL5[entriesPerMapLevel];

#ifdef TARGET_64BIT
    typedef RangeSectionL5 RangeSectionTopLevel;
    static constexpr uintptr_t mapLevels = 5;
    static constexpr uintptr_t maxSetBit = 56; // This is 0 indexed
    static constexpr uintptr_t bitsPerLevel = 8;
#else
    typedef RangeSectionL2 RangeSectionTopLevel;
    static constexpr uintptr_t mapLevels = 2;
    static constexpr uintptr_t maxSetBit = 31; // This is 0 indexed
    static constexpr uintptr_t bitsPerLevel = 8;
#endif

    BYTE _topLevelData[sizeof(RangeSectionTopLevel)];
    RangeSectionTopLevel &GetTopLevel()
    {
        return *(RangeSectionTopLevel*)&_topLevelData;
    }

    RangeSection* _pCleanupList;

    static constexpr uintptr_t bitsAtLastLevel = maxSetBit - (bitsPerLevel * mapLevels) + 1;
    static constexpr uintptr_t bytesAtLastLevel = (((uintptr_t)1) << bitsAtLastLevel);

    RangeSection* EndOfCleanupListMarker() { return (RangeSection*)1; }

    void* AllocateLevel()
    {
        size_t size = entriesPerMapLevel * sizeof(void*);
        void *buf = malloc(size);
        if (buf == NULL)
            return NULL;
        memset(buf, 0, size);
        return buf;
    }

    uintptr_t EffectiveBitsForLevel(TADDR address, uintptr_t level)
    {
        TADDR addressAsInt = address;
        TADDR addressBitsUsedInMap = addressAsInt >> (maxSetBit + 1 - (mapLevels * bitsPerLevel));
        TADDR addressBitsShifted = addressBitsUsedInMap >> ((level - 1) * bitsPerLevel);
        TADDR addressBitsUsedInLevel = (entriesPerMapLevel - 1) & addressBitsShifted;
        return addressBitsUsedInLevel;
    }

#ifndef DACCESS_COMPILE
    template<class T>
    auto EnsureLevel(TADDR address, T* outerLevel, uintptr_t level, bool collectible) -> decltype(&((*outerLevel->VolatileLoad(NULL))[0]))
    {
        RangeSectionLockState lockState = RangeSectionLockState::ReaderLocked; // This function may only be called while the lock is held at least at ReaderLocked
        uintptr_t index = EffectiveBitsForLevel(address, level);
        auto levelToGetPointerIn = outerLevel->VolatileLoad(&lockState);

        if (levelToGetPointerIn == NULL)
        {
            auto levelNew = static_cast<decltype(&(outerLevel->VolatileLoad(NULL))[0])>(AllocateLevel());
            if (levelNew == NULL)
                return NULL;

            if (!outerLevel->Install(levelNew, collectible))
            {
                // Handle race where another thread grew the table
                levelToGetPointerIn = outerLevel->VolatileLoad(&lockState);
                free(levelNew);
            }
            else
            {
                levelToGetPointerIn = levelNew;
            }
            assert(levelToGetPointerIn != NULL);
        }
        else if (!collectible && outerLevel->PointerIsCollectible())
        {
            outerLevel->UpgradeToNonCollectible();
        }

        return &((*levelToGetPointerIn)[index]);
    }
    // Returns pointer to address in last level map that actually points at RangeSection space.
    RangeSectionFragmentPointer* EnsureMapsForAddress(TADDR address, bool collectible)
    {
        uintptr_t level = mapLevels + 1;
        uintptr_t topLevelIndex = EffectiveBitsForLevel(address, --level);
        auto nextLevelAddress = &(GetTopLevel()[topLevelIndex]);
#ifdef TARGET_64BIT
        auto rangeSectionL4 = nextLevelAddress;
        auto rangeSectionL3 = EnsureLevel(address, rangeSectionL4, --level, collectible);
        if (rangeSectionL3 == NULL)
            return NULL; // Failure case
        auto rangeSectionL2 = EnsureLevel(address, rangeSectionL3, --level, collectible);
        if (rangeSectionL2 == NULL)
            return NULL; // Failure case
        auto rangeSectionL1 = EnsureLevel(address, rangeSectionL2, --level, collectible);
        if (rangeSectionL1 == NULL)
            return NULL; // Failure case
#else
        auto rangeSectionL1 = nextLevelAddress;
#endif
        auto result = EnsureLevel(address, rangeSectionL1, --level, collectible);
        if (result == NULL)
            return NULL; // Failure case

        return result;
    }
#endif // DACCESS_COMPILE

    void* GetRangeSectionMapLevelForAddress(TADDR address, uintptr_t level, RangeSectionLockState *pLockState)
    {
        uintptr_t topLevelIndex = EffectiveBitsForLevel(address, mapLevels);
        auto nextLevelAddress = &(GetTopLevel()[topLevelIndex]);
#ifdef TARGET_64BIT
        if (level == 4)
            return nextLevelAddress;

        auto rangeSectionL4 = nextLevelAddress->VolatileLoad(pLockState);
        if (rangeSectionL4 == NULL)
            return NULL;
        auto rangeSectionL3Ptr = &((*rangeSectionL4)[EffectiveBitsForLevel(address, 4)]);
        if (level == 3)
            return rangeSectionL3Ptr;

        auto rangeSectionL3 = rangeSectionL3Ptr->VolatileLoadWithoutBarrier(pLockState);
        if (rangeSectionL3 == NULL)
            return NULL;

        auto rangeSectionL2Ptr = &((*rangeSectionL3)[EffectiveBitsForLevel(address, 3)]);
        if (level == 2)
            return rangeSectionL2Ptr;

        auto rangeSectionL2 = rangeSectionL2Ptr->VolatileLoadWithoutBarrier(pLockState);
        if (rangeSectionL2 == NULL)
            return NULL;

        auto rangeSectionL1Ptr = &((*rangeSectionL2)[EffectiveBitsForLevel(address, 2)]);
        if (level == 1)
            return rangeSectionL1Ptr;
#else
        if (level == 1)
        {
            return nextLevelAddress;
        }
#endif
        assert(!"Unexpected level searched for");
        return NULL;
    }

    PTR_RangeSectionFragment GetRangeSectionForAddress(TADDR address, RangeSectionLockState *pLockState)
    {
        uintptr_t topLevelIndex = EffectiveBitsForLevel(address, mapLevels);
        auto nextLevelAddress = &(GetTopLevel()[topLevelIndex]);
#ifdef TARGET_64BIT
        auto rangeSectionL4 = nextLevelAddress->VolatileLoad(pLockState);
        if (rangeSectionL4 == NULL)
            return NULL;
        auto rangeSectionL3 = (*rangeSectionL4)[EffectiveBitsForLevel(address, 4)].VolatileLoadWithoutBarrier(pLockState);
        if (rangeSectionL3 == NULL)
            return NULL;
        auto rangeSectionL2 = (*rangeSectionL3)[EffectiveBitsForLevel(address, 3)].VolatileLoadWithoutBarrier(pLockState);
        if (rangeSectionL2 == NULL)
            return NULL;
        auto rangeSectionL1 = (*rangeSectionL2)[EffectiveBitsForLevel(address, 2)].VolatileLoadWithoutBarrier(pLockState);
#else
        auto rangeSectionL1 = nextLevelAddress->VolatileLoad(pLockState);
#endif
        if (rangeSectionL1 == NULL)
            return NULL;

        return ((*rangeSectionL1)[EffectiveBitsForLevel(address, 1)]).VolatileLoadWithoutBarrier(pLockState);
    }

    static uintptr_t RangeSectionFragmentCount(PTR_RangeSection pRangeSection)
    {
        uintptr_t rangeSize = pRangeSection->_range.RangeSize();
        if (rangeSize == 0)
            return 0;

        // Account for the range not starting at the beginning of a last level fragment
        rangeSize += pRangeSection->_range.RangeStart() & (bytesAtLastLevel - 1);

        uintptr_t fragmentCount = ((rangeSize - 1) / bytesAtLastLevel) + 1;
        return fragmentCount;
    }

    static TADDR IncrementAddressByMaxSizeOfFragment(TADDR input)
    {
        return input + bytesAtLastLevel;
    }

#ifndef DACCESS_COMPILE
    bool AttachRangeSectionToMap(PTR_RangeSection pRangeSection, RangeSectionLockState *pLockState)
    {
        assert(*pLockState == RangeSectionLockState::ReaderLocked); // Must be locked so that the cannot fail case, can't fail. NOTE: This only needs the reader lock, as the attach process can happen in parallel to reads.

        // Currently all use of the RangeSection should be with aligned addresses, so validate that the start and end are at aligned boundaries
        assert((pRangeSection->_range.RangeStart() & 0xF) == 0);
        assert((pRangeSection->_range.RangeEnd() & 0xF) == 0xF);
        assert((pRangeSection->_range.RangeEndOpen() & 0xF) == 0);

        uintptr_t rangeSectionFragmentCount = RangeSectionFragmentCount(pRangeSection);
        size_t fragmentsSize = rangeSectionFragmentCount * sizeof(RangeSectionFragment);
        void* fragmentsMemory = (RangeSectionFragment*)malloc(fragmentsSize);
        if (fragmentsMemory == NULL)
        {
            return false;
        }
        memset(fragmentsMemory, 0, fragmentsSize);

        RangeSectionFragment* fragments = (RangeSectionFragment*)fragmentsMemory;


        size_t entryUpdateSize = rangeSectionFragmentCount * sizeof(RangeSectionFragmentPointer*);
        RangeSectionFragmentPointer** entriesInMapToUpdate = (RangeSectionFragmentPointer**)malloc(entryUpdateSize);
        if (entriesInMapToUpdate == NULL)
        {
            free(fragments);
            return false;
        }

        memset(entriesInMapToUpdate, 0, entryUpdateSize);

        fragments[0].isPrimaryRangeSectionFragment = true;

        TADDR addressToPrepForUpdate = pRangeSection->_range.RangeStart();

        // Assert that range is not already mapped in any way
        assert(LookupRangeSection(addressToPrepForUpdate, pLockState) == NULL);
        assert(LookupRangeSection(pRangeSection->_range.RangeEnd(), pLockState) == NULL);
        for (TADDR fragmentAddress = addressToPrepForUpdate; pRangeSection->_range.IsInRange(fragmentAddress); fragmentAddress = IncrementAddressByMaxSizeOfFragment(fragmentAddress))
        {
            assert(LookupRangeSection(fragmentAddress, pLockState) == NULL);
        }

        bool collectible = !!(pRangeSection->_flags & RangeSection::RANGE_SECTION_COLLECTIBLE);

        for (uintptr_t iFragment = 0; iFragment < rangeSectionFragmentCount; iFragment++)
        {
            fragments[iFragment].pRangeSection = pRangeSection;
            fragments[iFragment]._range = pRangeSection->_range;
            fragments[iFragment].isCollectibleRangeSectionFragment = collectible;
            RangeSectionFragmentPointer* entryInMapToUpdate = EnsureMapsForAddress(addressToPrepForUpdate, collectible);
            if (entryInMapToUpdate == NULL)
            {
                free(fragments);
                free(entriesInMapToUpdate);
                return false;
            }

            entriesInMapToUpdate[iFragment] = entryInMapToUpdate;
            addressToPrepForUpdate = IncrementAddressByMaxSizeOfFragment(addressToPrepForUpdate);
        }

        // At this point all the needed memory is allocated, and it is no longer possible to fail.
        for (uintptr_t iFragment = 0; iFragment < rangeSectionFragmentCount; iFragment++)
        {
            RangeSectionFragmentPointer* pFragmentPointerToUpdate = entriesInMapToUpdate[iFragment];
            do
            {
                RangeSectionFragment* initialFragmentInMap = pFragmentPointerToUpdate->VolatileLoadWithoutBarrier(pLockState);

                // When inserting collectible elements into the range section map, ALWAYS put them after any non-collectible
                // fragments. This is so that when looking up ReadyToRun data, we never will need to take the ReaderLock for real.
                while (initialFragmentInMap != NULL && collectible && !initialFragmentInMap->isCollectibleRangeSectionFragment)
                {
                    pFragmentPointerToUpdate = &initialFragmentInMap->pRangeSectionFragmentNext;
                    initialFragmentInMap = pFragmentPointerToUpdate->VolatileLoadWithoutBarrier(pLockState);
                }

                fragments[iFragment].pRangeSectionFragmentNext.VolatileStore(initialFragmentInMap);
                if (pFragmentPointerToUpdate->AtomicReplace(&(fragments[iFragment]), initialFragmentInMap))
                    break;
            } while (true);
        }

        // Assert that range is now found via lookup
        assert(LookupRangeSection(pRangeSection->_range.RangeStart(), pLockState) == pRangeSection);
        assert(LookupRangeSection(pRangeSection->_range.RangeEnd(), pLockState) == pRangeSection);
        for (TADDR fragmentAddress = pRangeSection->_range.RangeStart(); pRangeSection->_range.IsInRange(fragmentAddress); fragmentAddress = IncrementAddressByMaxSizeOfFragment(fragmentAddress))
        {
            assert(LookupRangeSection(fragmentAddress, pLockState) == pRangeSection);
        }

        // entriesInMapToUpdate was just a temporary allocation
        free(entriesInMapToUpdate);

        return true;
    }
#endif // DACCESS_COMPILE


public:
    RangeSectionMap() : _pCleanupList(EndOfCleanupListMarker())
    {
        memset(&_topLevelData, 0, sizeof(_topLevelData));
    }

#ifndef DACCESS_COMPILE

#ifdef FEATURE_READYTORUN
    RangeSection *AllocateRange(Range range, IJitManager* pJit, RangeSection::RangeSectionFlags flags, PTR_Module pR2RModule, RangeSectionLockState* pLockState)
    {
        PTR_RangeSection pSection(new(nothrow)RangeSection(range, pJit, flags, pR2RModule));
        if (pSection == NULL)
            return NULL;

        if (!AttachRangeSectionToMap(pSection, pLockState))
        {
            delete pSection;
            return NULL;
        }
        return pSection;
    }
#endif

    RangeSection *AllocateRange(Range range, IJitManager* pJit, RangeSection::RangeSectionFlags flags, PTR_HeapList pHeapList, RangeSectionLockState* pLockState)
    {
        PTR_RangeSection pSection(new(nothrow)RangeSection(range, pJit, flags, pHeapList));
        if (pSection == NULL)
            return NULL;

        if (!AttachRangeSectionToMap(pSection, pLockState))
        {
            delete pSection;
            return NULL;
        }
        return pSection;
    }

    RangeSection *AllocateRange(Range range, IJitManager* pJit, RangeSection::RangeSectionFlags flags, PTR_CodeRangeMapRangeList pRangeList, RangeSectionLockState* pLockState)
    {
        PTR_RangeSection pSection(new(nothrow)RangeSection(range, pJit, flags, pRangeList));
        if (pSection == NULL)
            return NULL;

        if (!AttachRangeSectionToMap(pSection, pLockState))
        {
            delete pSection;
            return NULL;
        }
        return pSection;
    }
#endif // DACCESS_COMPILE

    PTR_RangeSection LookupRangeSection(TADDR address, RangeSectionLockState *pLockState)
    {
        PTR_RangeSectionFragment fragment = GetRangeSectionForAddress(address, pLockState);
        if (fragment == NULL)
            return NULL;

        while ((fragment != NULL) && !fragment->InRange(address))
        {
            fragment = fragment->pRangeSectionFragmentNext.VolatileLoadWithoutBarrier(pLockState);
        }

        if (fragment != NULL)
        {
            if (fragment->pRangeSection->_pRangeSectionNextForDelete != NULL)
                return NULL;
            return fragment->pRangeSection;
        }

        return NULL;
    }

#ifndef DACCESS_COMPILE
    void RemoveRangeSection(RangeSection* pRangeSection)
    {
        assert(pRangeSection->_pRangeSectionNextForDelete == NULL);
        assert(pRangeSection->_flags & RangeSection::RANGE_SECTION_COLLECTIBLE);
#ifdef FEATURE_READYTORUN
        assert(pRangeSection->_pR2RModule == NULL);
#endif

        // Removal is implemented by placing onto the cleanup linked list. This is then processed later during cleanup
        RangeSection* pLatestRemovedRangeSection;
        do
        {
            pLatestRemovedRangeSection = VolatileLoad(&_pCleanupList);
            VolatileStore(&pRangeSection->_pRangeSectionNextForDelete, pLatestRemovedRangeSection);
        } while (InterlockedCompareExchangeT(&_pCleanupList, pRangeSection, pLatestRemovedRangeSection) != pLatestRemovedRangeSection);
    }

    void CleanupRangeSections(RangeSectionLockState *pLockState)
    {
        assert(*pLockState == RangeSectionLockState::WriteLocked);

        while (this->_pCleanupList != EndOfCleanupListMarker())
        {
            PTR_RangeSection pRangeSectionToCleanup(this->_pCleanupList);
            RangeSectionFragment* pRangeSectionFragmentToFree = NULL;
            this->_pCleanupList = pRangeSectionToCleanup->_pRangeSectionNextForDelete;

            uintptr_t rangeSectionFragmentCount = RangeSectionFragmentCount(pRangeSectionToCleanup);

            TADDR addressToPrepForCleanup = pRangeSectionToCleanup->_range.RangeStart();

            assert(LookupRangeSection(addressToPrepForCleanup, pLockState) == NULL);
            assert(LookupRangeSection(pRangeSectionToCleanup->_range.RangeEnd(), pLockState) == NULL);
            for (TADDR fragmentAddress = addressToPrepForCleanup; pRangeSectionToCleanup->_range.IsInRange(fragmentAddress); fragmentAddress = IncrementAddressByMaxSizeOfFragment(fragmentAddress))
            {
                assert(LookupRangeSection(fragmentAddress, pLockState) == NULL);
            }

            // Remove fragments from each of the fragment linked lists
            for (uintptr_t iFragment = 0; iFragment < rangeSectionFragmentCount; iFragment++)
            {
                RangeSectionFragmentPointer* entryInMapToUpdate = EnsureMapsForAddress(addressToPrepForCleanup, true /* collectible */);
                assert(entryInMapToUpdate != NULL);

#ifdef _DEBUG
                bool seenCollectibleRangeList = false;
#endif
                while ((entryInMapToUpdate->VolatileLoadWithoutBarrier(pLockState))->pRangeSection != pRangeSectionToCleanup)
                {
#ifdef _DEBUG
                    if (entryInMapToUpdate->VolatileLoadWithoutBarrier(pLockState)->isCollectibleRangeSectionFragment)
                    {
                        seenCollectibleRangeList = true;
                    }
                    else
                    {
                        // Since the fragment linked lists are sorted such that the collectible ones are always after the non-collectible ones, this should never happen.
                        assert(!seenCollectibleRangeList);
                    }
#endif
                    entryInMapToUpdate = &(entryInMapToUpdate->VolatileLoadWithoutBarrier(pLockState))->pRangeSectionFragmentNext;
                }

                RangeSectionFragment* fragment = entryInMapToUpdate->VolatileLoadWithoutBarrier(pLockState);

                // The fragment associated with the start of the range has the address that was allocated earlier
                if (iFragment == 0)
                {
                    pRangeSectionFragmentToFree = fragment;
                    assert(pRangeSectionFragmentToFree->isPrimaryRangeSectionFragment);
                }

                auto fragmentThatRemains = fragment->pRangeSectionFragmentNext.VolatileLoadWithoutBarrier(pLockState);
                entryInMapToUpdate->VolatileStore(fragmentThatRemains);

                // Now determine if we need to actually free portions of the map structure
                if (fragmentThatRemains == NULL)
                {
                    for (uintptr_t level = 1; level < mapLevels; level++)
                    {
                        // Note that the type here is actually not necessarily correct, but its close enough
                        auto pointerToLevelData = (RangeSectionLevelPointer<DPTR(RangeSectionL2)>*)GetRangeSectionMapLevelForAddress(addressToPrepForCleanup, level, pLockState);
                        if (pointerToLevelData == NULL)
                            break;
                        auto &rawData = *pointerToLevelData->VolatileLoad(pLockState);
                        bool foundMeaningfulValue = false;

                        for (uintptr_t i = 0; i < entriesPerMapLevel; i++)
                        {
                            if (!rawData[i].IsNull())
                            {
                                foundMeaningfulValue = true;
                                break;
                            }
                        }

                        if (foundMeaningfulValue)
                            break;

                        // This level is completely empty. Free it, and then null out the pointer to it.
                        pointerToLevelData->Uninstall();
#if defined(__GNUC__)
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wfree-nonheap-object" // The compiler can't tell that this pointer always comes from a malloc call.
                        free((void*)rawData);
#pragma GCC diagnostic pop
#endif
                    }
                }

                addressToPrepForCleanup = IncrementAddressByMaxSizeOfFragment(addressToPrepForCleanup);
            }

            // Free the array of fragments
            delete pRangeSectionToCleanup;
            free(pRangeSectionFragmentToFree);
        }
    }
#endif // DACCESS_COMPILE

#ifdef DACCESS_COMPILE
    void EnumMemoryRangeSectionMapLevel(CLRDataEnumMemoryFlags flags, RangeSectionFragmentPointer& fragmentPointer, RangeSectionLockState* pLockState)
    {
        PTR_RangeSectionFragment fragment = fragmentPointer.VolatileLoadWithoutBarrier(pLockState);
        while (fragment != NULL)
        {
            if (!DacEnumMemoryRegion(dac_cast<TADDR>(fragment), sizeof(RangeSectionFragment)))
                return;

            fragment->pRangeSection->EnumMemoryRegions(flags);

            fragment = fragment->pRangeSectionFragmentNext.VolatileLoadWithoutBarrier(pLockState);
        }
    }

    void EnumMemoryRangeSectionMapLevel(CLRDataEnumMemoryFlags flags, RangeSectionL1& level, RangeSectionLockState* pLockState)
    {
        if (!DacEnumMemoryRegion(dac_cast<TADDR>(&level), sizeof(level)))
            return;

        for (uintptr_t i = 0; i < entriesPerMapLevel; i++)
        {
            if (!level[i].IsNull())
            {
                EnumMemoryRangeSectionMapLevel(flags, level[i], pLockState);
            }
        }
    }

    template<class T>
    void EnumMemoryRangeSectionMapLevel(CLRDataEnumMemoryFlags flags, T& level, RangeSectionLockState* pLockState)
    {
        if (!DacEnumMemoryRegion(dac_cast<TADDR>(&level), sizeof(level)))
            return;

        for (uintptr_t i = 0; i < entriesPerMapLevel; i++)
        {
            if (level[i].IsNull())
            {
                EnumMemoryRangeSectionMapLevel(flags, *level[i].VolatileLoad(pLockState), pLockState);
            }
        }
    }

    void EnumMemoryRegions(CLRDataEnumMemoryFlags flags)
    {
        if (!DacEnumMemoryRegion(dac_cast<TADDR>(this), sizeof(*this)))
            return;

        // Always assume we are locked when enumerating
        RangeSectionLockState lockState = RangeSectionLockState::ReaderLocked;
        EnumMemoryRangeSectionMapLevel(flags, GetTopLevel(), &lockState);
    }
#endif// DACCESS_COMPILE

    friend struct ::cdac_data<RangeSectionMap>;
};

template<>
struct cdac_data<RangeSectionMap>
{
    static constexpr size_t TopLevelData = offsetof(RangeSectionMap, _topLevelData);

    struct RangeSectionFragment
    {
        static constexpr size_t RangeBegin = offsetof(RangeSectionMap::RangeSectionFragment, _range.begin);
        static constexpr size_t RangeEndOpen = offsetof(RangeSectionMap::RangeSectionFragment, _range.end);
        static constexpr size_t RangeSection = offsetof(RangeSectionMap::RangeSectionFragment, pRangeSection);
        static constexpr size_t Next = offsetof(RangeSectionMap::RangeSectionFragment, pRangeSectionFragmentNext);
    };
};

struct RangeSectionMapData
{
    BYTE Data[sizeof(RangeSectionMap)];
};

/*****************************************************************************/


//
// A simple linked-list based allocator to expose code heap as loader heap for allocation of precodes.
// The loader heap like interface is necessary to support backout. It is also conveniently used to reduce space overhead
// for small blocks that are common for precodes.
//
// Allocating precodes on code heap makes them close to other code, it reduces need for jump stubs and thus chance
// that we run into bogus OOM because of not being able to allocate memory in particular memory range.
//
class CodeFragmentHeap : public ILoaderHeapBackout
{
    PTR_LoaderAllocator m_pAllocator;

    struct FreeBlock
    {
        DPTR(FreeBlock) m_pNext;    // Next block
        void  *m_pBlock;
        SIZE_T m_dwSize;            // Size of this block (includes size of FreeBlock)
    };
    typedef DPTR(FreeBlock) PTR_FreeBlock;

    PTR_FreeBlock       m_pFreeBlocks;
    StubCodeBlockKind   m_kind;

    Crst                m_Lock;

    void AddBlock(VOID * pMem, size_t dwSize);
    void RemoveBlock(FreeBlock ** ppBlock);

public:
    CodeFragmentHeap(LoaderAllocator * pAllocator, StubCodeBlockKind kind);
    virtual ~CodeFragmentHeap();

    TaggedMemAllocPtr RealAllocAlignedMem(size_t  dwRequestedSize
                                         ,unsigned  dwAlignment
#ifdef _DEBUG
                                         ,_In_ _In_z_ const char *szFile
                                         ,int  lineNum
#endif
                                         );

    virtual void RealBackoutMem(void *pMem
                        , size_t dwSize
#ifdef _DEBUG
                        , _In_ _In_z_ const char *szFile
                        , int lineNum
                        , _In_ _In_z_ const char *szAllocFile
                        , int allocLineNum
#endif
                        ) DAC_EMPTY();

#ifdef DACCESS_COMPILE
    void EnumMemoryRegions(enum CLRDataEnumMemoryFlags flags)
    {
        WRAPPER_NO_CONTRACT;
        DAC_ENUM_DTHIS();
    }
#endif
};

typedef DPTR(class CodeFragmentHeap) PTR_CodeFragmentHeap;

//-----------------------------------------------------------------------------
//
// Manages the CodeHeap for some of the RangeSections in the ExecutionManager
//
//-----------------------------------------------------------------------------

class IJitManager
{
    VPTR_BASE_VTABLE_CLASS(IJitManager)

public:
    struct MethodRegionInfo
    {
        TADDR hotStartAddress;
        size_t hotSize;
        TADDR coldStartAddress;
        size_t coldSize;
    };

#ifndef DACCESS_COMPILE
    IJitManager();
#endif // !DACCESS_COMPILE

    virtual DWORD GetCodeType() = 0;

    // Used to read debug info.
    // 1) Caller passes an allocator which these functions use to allocate memory.
    //    This is b/c the store may need to decompress the information just to figure out the size.
    // 2) Note that these methods use Uncompressed (Normal) jit data.
    //    Compression is just an implementation detail.
    // 3) These throw on OOM (exceptional case), and may return a
    //    failing HR if no data is available (not exceptional)

    virtual BOOL GetBoundariesAndVars(
        const DebugInfoRequest & request,
        IN FP_IDS_NEW fpNew,
        IN void * pNewData,
        BoundsType boundsType,
        OUT ULONG32 * pcMap,
        OUT ICorDebugInfo::OffsetMapping **ppMap,
        OUT ULONG32 * pcVars,
        OUT ICorDebugInfo::NativeVarInfo **ppVars) = 0;

    virtual size_t WalkILOffsets(
        const DebugInfoRequest & request,
        BoundsType boundsType,
        void* pContext,
        size_t (*pfnWalkILOffsets)(ICorDebugInfo::OffsetMapping *pOffsetMapping, void *pContext)) = 0;

    virtual BOOL GetRichDebugInfo(
        const DebugInfoRequest& request,
        IN FP_IDS_NEW fpNew, IN void* pNewData,
        OUT ICorDebugInfo::InlineTreeNode** ppInlineTree,
        OUT ULONG32* pNumInlineTree,
        OUT ICorDebugInfo::RichOffsetMapping** ppRichMappings,
        OUT ULONG32* pNumRichMappings) = 0;

    virtual PCODE GetCodeAddressForRelOffset(const METHODTOKEN& MethodToken, DWORD relOffset) = 0;

    virtual BOOL JitCodeToMethodInfo(
        RangeSection * pRangeSection,
        PCODE currentPC,
        MethodDesc** ppMethodDesc,
        OUT EECodeInfo * pCodeInfo) = 0;

    virtual TADDR       JitTokenToStartAddress(const METHODTOKEN& MethodToken)=0;
    virtual void        JitTokenToMethodRegionInfo(const METHODTOKEN& MethodToken, MethodRegionInfo *methodRegionInfo) = 0;
    virtual unsigned    InitializeEHEnumeration(const METHODTOKEN& MethodToken, EH_CLAUSE_ENUMERATOR* pEnumState)=0;
    virtual PTR_EXCEPTION_CLAUSE_TOKEN GetNextEHClause(EH_CLAUSE_ENUMERATOR* pEnumState,
                                 EE_ILEXCEPTION_CLAUSE* pEHclause)=0;
#ifndef DACCESS_COMPILE
    virtual TypeHandle  ResolveEHClause(EE_ILEXCEPTION_CLAUSE* pEHClause,
                                        CrawlFrame *pCf)=0;
#endif // #ifndef DACCESS_COMPILE

    virtual GCInfoToken GetGCInfoToken(const METHODTOKEN& MethodToken)=0;
    PTR_VOID GetGCInfo(const METHODTOKEN& MethodToken)
    {
        return GetGCInfoToken(MethodToken).Info;
    }

    TADDR JitTokenToModuleBase(const METHODTOKEN& MethodToken);

#if defined(FEATURE_EH_FUNCLETS)
    virtual PTR_RUNTIME_FUNCTION LazyGetFunctionEntry(EECodeInfo * pCodeInfo) = 0;

    // GetFuncletStartAddress returns the starting address of the function or funclet indicated by the EECodeInfo address.
    virtual TADDR GetFuncletStartAddress(EECodeInfo * pCodeInfo);

    virtual DWORD GetFuncletStartOffsets(const METHODTOKEN& MethodToken, DWORD* pStartFuncletOffsets, DWORD dwLength) = 0;

    virtual BOOL LazyIsFunclet(EECodeInfo * pCodeInfo);
    virtual BOOL IsFilterFunclet(EECodeInfo * pCodeInfo);
#endif // FEATURE_EH_FUNCLETS

    virtual StubCodeBlockKind   GetStubCodeBlockKind(RangeSection * pRangeSection, PCODE currentPC) = 0;

    // DAC-specific virtual functions.
    // Note that these MUST occur below any other virtual function definitions to ensure that the vtable in
    // DAC builds is compatible with the non-DAC one so that DAC virtual dispatch will work correctly.
#if defined(DACCESS_COMPILE)
    virtual void EnumMemoryRegions(CLRDataEnumMemoryFlags flags);
    virtual void EnumMemoryRegionsForMethodDebugInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo) = 0;
#if defined(FEATURE_EH_FUNCLETS)
    // Enumerate the memory necessary to retrieve the unwind info for a specific method
    virtual void EnumMemoryRegionsForMethodUnwindInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo) = 0;
#endif // FEATURE_EH_FUNCLETS
#endif // DACCESS_COMPILE

#ifndef DACCESS_COMPILE
    void SetCodeManager(ICodeManager *codeMgr)
    {
        LIMITED_METHOD_CONTRACT;

        m_runtimeSupport = codeMgr;
    }
#endif // !DACCESS_COMPILE

    ICodeManager *GetCodeManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_runtimeSupport;
    }

protected:
    PTR_ICodeManager m_runtimeSupport;
};

#if !defined(DACCESS_COMPILE)
class CodeHeapIterator final
{
    // Captured state for each heap in the iterator.
    struct HeapListState
    {
        HeapList* Heap;
        void* MapBase;
        void* HdrMap;
        size_t MaxCodeHeapSize;
    };

    EECodeGenManager* m_manager;
    MethodSectionIterator m_Iterator;
    CUnorderedArray<HeapListState, 64> m_Heaps;
    int32_t m_HeapsIndexNext;
    LoaderAllocator* m_pLoaderAllocatorFilter;
    MethodDesc* m_pCurrent;

public:
    CodeHeapIterator(EECodeGenManager* manager, HeapList* heapList, LoaderAllocator* pLoaderAllocatorFilter);
    ~CodeHeapIterator();

    CodeHeapIterator(CodeHeapIterator const&) = delete;
    CodeHeapIterator& operator=(CodeHeapIterator const&) = delete;
    CodeHeapIterator(CodeHeapIterator&&) = default;
    CodeHeapIterator& operator=(CodeHeapIterator&&) = default;

    bool Next();

    MethodDesc* GetMethod()
    {
        LIMITED_METHOD_CONTRACT;
        return m_pCurrent;
    }

    TADDR GetMethodCode()
    {
        LIMITED_METHOD_CONTRACT;
        return (TADDR)m_Iterator.GetMethodCode();
    }

private:
    bool NextMethodSectionIterator();
};
#endif // !DACCESS_COMPILE

class HostCodeHeap;
typedef VPTR(class HostCodeHeap) PTR_HostCodeHeap;

class EECodeGenManager : public IJitManager
{
#ifdef DACCESS_COMPILE
    friend class ClrDataAccess;
    friend class DacDbiInterfaceImpl;
#endif
    friend class CheckDuplicatedStructLayouts;
    friend class EECodeInfo;
    friend class CodeHeapIterator;

    VPTR_ABSTRACT_VTABLE_CLASS(EECodeGenManager, IJitManager)

    struct DomainCodeHeapList
    {
        LoaderAllocator *m_pAllocator;
        CDynArray<HeapList *> m_CodeHeapList;
        DomainCodeHeapList(LoaderAllocator* allocator);
        ~DomainCodeHeapList() = default;
    };

public:

    EECodeGenManager();

    virtual ICorJitCompiler* GetCompiler() = 0;
    virtual ICorJitCompiler* GetAltCompiler() = 0;

    virtual PTR_EXCEPTION_CLAUSE_TOKEN GetNextEHClause(EH_CLAUSE_ENUMERATOR* pEnumState,
                                                       EE_ILEXCEPTION_CLAUSE* pEHclause);

    static TADDR FindMethodCode(RangeSection * pRangeSection, PCODE currentPC);
    static TADDR FindMethodCode(PCODE currentPC);

    bool IsStoringRichDebugInfo()
    {
        LIMITED_METHOD_CONTRACT;
        return m_storeRichDebugInfo;
    }

protected:
    BOOL GetBoundariesAndVarsWorker(
        PTR_BYTE pDebugInfo,
        IN FP_IDS_NEW fpNew,
        IN void * pNewData,
        BoundsType boundsType,
        OUT ULONG32 * pcMap,
        OUT ICorDebugInfo::OffsetMapping **ppMap,
        OUT ULONG32 * pcVars,
        OUT ICorDebugInfo::NativeVarInfo **ppVars);

    size_t WalkILOffsetsWorker(
        PTR_BYTE pDebugInfo,
        BoundsType boundsType,
        void* pContext,
        size_t (*pfnWalkILOffsets)(ICorDebugInfo::OffsetMapping *pOffsetMapping, void *pContext));

    BOOL GetRichDebugInfoWorker(
        PTR_BYTE pDebugInfo,
        IN FP_IDS_NEW fpNew, IN void* pNewData,
        OUT ICorDebugInfo::InlineTreeNode** ppInlineTree,
        OUT ULONG32* pNumInlineTree,
        OUT ICorDebugInfo::RichOffsetMapping** ppRichMappings,
        OUT ULONG32* pNumRichMappings);

    template<typename TCodeHeader>
    BOOL JitCodeToMethodInfoWorker(
        RangeSection * pRangeSection,
        PCODE currentPC,
        MethodDesc ** ppMethodDesc,
        EECodeInfo * pCodeInfo);

public:
#ifndef DACCESS_COMPILE
    virtual TypeHandle ResolveEHClause(EE_ILEXCEPTION_CLAUSE* pEHClause,
                                       CrawlFrame *pCf);

    void Unload(LoaderAllocator* pAllocator);

public:
    void CleanupCodeHeaps();

    template<typename TCodeHeader>
    void AllocCode(MethodDesc* pMD, size_t blockSize, size_t reserveForJumpStubs, CorJitAllocMemFlag flag, void** ppCodeHeader, void** ppCodeHeaderRW,
                   size_t* pAllocatedSize, HeapList** ppCodeHeap , BYTE** ppRealHeader
#ifdef FEATURE_EH_FUNCLETS
                 , UINT nUnwindInfos
#endif
                  );

    BYTE *AllocFromJitMetaHeap(MethodDesc *pMD, size_t blockSize);

    // Heap Management functions
    void NibbleMapSet(HeapList * pHp, TADDR pCode, size_t codeSize);
    void AddToCleanupList(HostCodeHeap* pCodeHeap);
    bool TryFreeHostCodeHeapMemory(HostCodeHeap* pCodeHeap, void* codeStart);
    CodeHeapIterator GetCodeHeapIterator(LoaderAllocator* pLoaderAllocatorFilter = NULL);

private:
    void AddRefIterator();
    void ReleaseIterator();

protected:
    virtual void DeleteFunctionTable(PVOID pvTableID) = 0;

    void* AllocCodeWorker(CodeHeapRequestInfo *pInfo, size_t header, size_t blockSize, unsigned align, HeapList ** ppCodeHeap);
    void NibbleMapSetUnlocked(HeapList * pHp, TADDR pCode, size_t codeSize);
    void NibbleMapDeleteUnlocked(HeapList* pHp, TADDR pCode);

private:
    HeapList* NewCodeHeap(CodeHeapRequestInfo *pInfo, DomainCodeHeapList *pADHeapList);
    void DeleteCodeHeap(HeapList *pHeapList);
    void RemoveFromCleanupList(HostCodeHeap *pCodeHeap);
    void RemoveCodeHeapFromDomainList(CodeHeap *pHeap, LoaderAllocator *pAllocator);
    bool CanUseCodeHeap(CodeHeapRequestInfo *pInfo, HeapList *pCodeHeap);

    DomainCodeHeapList* GetCodeHeapList(CodeHeapRequestInfo *pInfo, LoaderAllocator *pAllocator, BOOL fDynamicOnly = FALSE);
    DomainCodeHeapList* CreateCodeHeapList(CodeHeapRequestInfo *pInfo);
    LoaderHeap* GetJitMetaHeap(MethodDesc *pMD);

    void UnloadWorker(LoaderAllocator* pAllocator);
    void FreeHostCodeHeapMemoryWorker(HostCodeHeap* pCodeHeap, void* codeStart);
#else // !DACCESS_COMPILE
    virtual void EnumMemoryRegions(CLRDataEnumMemoryFlags flags);
protected:
    template<typename TCodeHeader>
    void EnumMemoryRegionsForMethodDebugInfoWorker(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo);
#endif // !DACCESS_COMPILE

private:
    PTR_HeapList m_pAllCodeHeaps; // Includes all code heaps, including dynamic ones.
    PTR_HostCodeHeap m_cleanupList;

    // must hold critical section to access this structure.
    CUnorderedArray<DomainCodeHeapList*, 5> m_DomainCodeHeaps;
    CUnorderedArray<DomainCodeHeapList*, 5> m_DynamicDomainCodeHeaps;
    CUnorderedArray<LoaderAllocator*, 4> m_delayUnload;

protected:
    Crst m_CodeHeapLock;
    ULONG m_iteratorCount;
    bool m_storeRichDebugInfo;
};

//-----------------------------------------------------------------------------

typedef VPTR(class EEJitManager) PTR_EEJitManager;
typedef VPTR(class EECodeGenManager) PTR_EECodeGenManager;
typedef VPTR(class ReadyToRunJitManager) PTR_ReadyToRunJitManager;
typedef VPTR(class InterpreterJitManager) PTR_InterpreterJitManager;

struct JumpStubBlockHeader
{
    JumpStubBlockHeader *  m_next;
    UINT32                 m_used;
    UINT32                 m_allocated;

    LoaderAllocator* GetLoaderAllocator()
    {
        _ASSERTE(m_zero == 0);
        return m_Allocator;
    }

    void SetLoaderAllocator(LoaderAllocator * loaderAllocator)
    {
        m_zero = 0;
        m_Allocator = loaderAllocator;
    }

    HostCodeHeap* GetHostCodeHeap()
    {
        WRAPPER_NO_CONTRACT;
        _ASSERTE(m_zero == -1);
        return m_CodeHeap;
    }

    void SetHostCodeHeap(HostCodeHeap * hostCodeHeap)
    {
        m_zero = -1;
        m_CodeHeap = hostCodeHeap;
    }

private:
    union {
        HostCodeHeap    *m_CodeHeap;
        LoaderAllocator *m_Allocator;
    };

    INT64   m_zero; // 0 for normal methods and -1 for LCG methods
};


/*****************************************************************************/

class EEJitManager final : public EECodeGenManager
{
#ifdef DACCESS_COMPILE
    friend class ClrDataAccess;
    friend class DacDbiInterfaceImpl;
#endif
    friend class CheckDuplicatedStructLayouts;

    VPTR_VTABLE_CLASS(EEJitManager, EECodeGenManager)

public:
    virtual ICorJitCompiler* GetCompiler()
    {
        return m_jit;
    }

    virtual ICorJitCompiler* GetAltCompiler()
    {
        return m_alternateJit;
    }

    // Failing to load the main JIT is a failure.
    // If the user requested an altjit and we failed to load an altjit, that is also a failure.
    BOOL IsJitLoaded()
    {
        LIMITED_METHOD_CONTRACT;

        return (m_jit != NULL)
#ifdef ALLOW_SXS_JIT
            && (!m_AltJITRequired || (m_alternateJit != NULL))
#endif // ALLOW_SXS_JIT
            ;
    }

#ifdef ALLOW_SXS_JIT
    BOOL IsMainJitLoaded()
    {
        LIMITED_METHOD_CONTRACT;

        return (m_jit != NULL);
    }

    BOOL IsAltJitLoaded()
    {
        LIMITED_METHOD_CONTRACT;

        return (m_alternateJit != NULL);
    }
#endif // ALLOW_SXS_JIT

#if !defined DACCESS_COMPILE
    EEJitManager();
    ~EEJitManager() = default;
#endif // !DACCESS_COMPILE

    virtual DWORD GetCodeType()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return (miManaged | miIL);
    }

    // Used to read debug info.
    virtual BOOL GetBoundariesAndVars(
        const DebugInfoRequest & request,
        IN FP_IDS_NEW fpNew,
        IN void * pNewData,
        BoundsType boundsType,
        OUT ULONG32 * pcMap,
        OUT ICorDebugInfo::OffsetMapping **ppMap,
        OUT ULONG32 * pcVars,
        OUT ICorDebugInfo::NativeVarInfo **ppVars);

    virtual size_t WalkILOffsets(
        const DebugInfoRequest & request,
        BoundsType boundsType,
        void* pContext,
        size_t (*pfnWalkILOffsets)(ICorDebugInfo::OffsetMapping *pOffsetMapping, void *pContext));

    virtual BOOL GetRichDebugInfo(
        const DebugInfoRequest& request,
        IN FP_IDS_NEW fpNew, IN void* pNewData,
        OUT ICorDebugInfo::InlineTreeNode** ppInlineTree,
        OUT ULONG32* pNumInlineTree,
        OUT ICorDebugInfo::RichOffsetMapping** ppRichMappings,
        OUT ULONG32* pNumRichMappings);

    virtual PCODE GetCodeAddressForRelOffset(const METHODTOKEN& MethodToken, DWORD relOffset);

    virtual BOOL JitCodeToMethodInfo(RangeSection * pRangeSection,
                                     PCODE currentPC,
                                     MethodDesc ** ppMethodDesc,
                                     EECodeInfo * pCodeInfo);

    virtual TADDR       JitTokenToStartAddress(const METHODTOKEN& MethodToken);
    virtual void        JitTokenToMethodRegionInfo(const METHODTOKEN& MethodToken, MethodRegionInfo *methodRegionInfo);

    virtual unsigned    InitializeEHEnumeration(const METHODTOKEN& MethodToken, EH_CLAUSE_ENUMERATOR* pEnumState);

    GCInfoToken         GetGCInfoToken(const METHODTOKEN& MethodToken);

#ifdef DACCESS_COMPILE
    virtual void EnumMemoryRegionsForMethodDebugInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo);
#endif // DACCESS_COMPILE

#if !defined DACCESS_COMPILE
    BOOL                LoadJIT();

    JumpStubBlockHeader* AllocJumpStubBlock(MethodDesc* pMD, DWORD numJumps,
                                            BYTE * loAddr, BYTE * hiAddr,
                                            LoaderAllocator *pLoaderAllocator,
                                            bool throwOnOutOfMemoryWithinRange);

    void *              AllocCodeFragmentBlock(size_t blockSize, unsigned alignment, LoaderAllocator *pLoaderAllocator, StubCodeBlockKind kind);
#endif // !DACCESS_COMPILE

    static CodeHeader * GetCodeHeader(const METHODTOKEN& MethodToken);
    static CodeHeader * GetCodeHeaderFromStartAddress(TADDR methodStartAddress);

#if defined(FEATURE_EH_FUNCLETS)
    // Compute function entry lazily. Do not call directly. Use EECodeInfo::GetFunctionEntry instead.
    virtual PTR_RUNTIME_FUNCTION    LazyGetFunctionEntry(EECodeInfo * pCodeInfo);

    virtual DWORD                   GetFuncletStartOffsets(const METHODTOKEN& MethodToken, DWORD* pStartFuncletOffsets, DWORD dwLength);
#endif // FEATURE_EH_FUNCLETS

    virtual StubCodeBlockKind       GetStubCodeBlockKind(RangeSection * pRangeSection, PCODE currentPC);

#if defined(FEATURE_EH_FUNCLETS)
    // Enumerate the memory necessary to retrieve the unwind info for a specific method
    virtual void EnumMemoryRegionsForMethodUnwindInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo)
    {
        // We don't need to do explicitly enumerate the memory for unwind information for JITted methods because
        // it is stored using the Win64 standard dynamic function table mechanism, and dump generation code knows
        // it needs to call our code:OutOfProcessFunctionTableCallback in order to save the function table including
        // unwind information at dump generation time (since it's dynamic, it will not be otherwise
        // available at debug time).
    }
#endif // FEATURE_EH_FUNCLETS

#if !defined DACCESS_COMPILE
protected:
    virtual void DeleteFunctionTable(PVOID pvTableID);
#endif // !DACCESS_COMPILE

private:
    CORJIT_FLAGS m_CPUCompileFlags;

#if !defined DACCESS_COMPILE
    void SetCpuInfo();
#endif

public:
    inline CORJIT_FLAGS GetCPUCompileFlags()
    {
        LIMITED_METHOD_CONTRACT;
        return m_CPUCompileFlags;
    }

private :
    Crst                m_JitLoadLock;

#ifdef TARGET_AMD64
private:
    //
    // List of reserved memory blocks to be used for jump stub allocation if no suitable memory block is found
    // via the regular mechanism
    //
    struct EmergencyJumpStubReserve
    {
        EmergencyJumpStubReserve * m_pNext;
        BYTE *   m_ptr;
        SIZE_T   m_size;
        SIZE_T   m_free;
    };
    EmergencyJumpStubReserve * m_pEmergencyJumpStubReserveList;

public:
    BYTE * AllocateFromEmergencyJumpStubReserve(const BYTE * loAddr, const BYTE * hiAddr, SIZE_T * pReserveSize);
    VOID EnsureJumpStubReserve(BYTE * pImageBase, SIZE_T imageSize, SIZE_T reserveSize);
#endif

public:
    ICorJitCompiler *   m_jit;
    HINSTANCE           m_JITCompiler;
#if defined(TARGET_X86) || defined(TARGET_AMD64)
    HINSTANCE           m_JITCompilerOther; // Stores the handle of the legacy JIT, if one is loaded.
#endif

#ifdef ALLOW_SXS_JIT
    //put these at the end so that we don't mess up the offsets in the DAC.
    ICorJitCompiler *   m_alternateJit;
    HINSTANCE           m_AltJITCompiler;
    bool                m_AltJITRequired;
#endif //ALLOW_SXS_JIT
};

//*****************************************************************************
//
// This class manages IJitManagers and ICorJitCompilers.  It has only static
// members.  It should never be constructed.
//
//*****************************************************************************

class ExecutionManager
{
    friend class CorExternalDataAccess;
    friend struct _DacGlobals;

#ifdef DACCESS_COMPILE
    friend class ClrDataAccess;
#endif

public:
    static void Init();

    enum ScanFlag
    {
        // When this is passed to a function, it must directly acquire a reader lock
        // before it may continue
        ScanReaderLock,

        // This means the function need not directly acquire a reader lock; however, it
        // may call other functions that may require other reader locks (e.g.,
        // ExecutionManager::FindJitMan may be called with ScanNoReaderLock, but
        // still calls IJitManager::JitCodeToMethodInfo which acquires its own
        // IJitManager reader lock)
        ScanNoReaderLock
    };

    // Returns default scan flag for current thread
    static ScanFlag GetScanFlags();

    // Returns whether currentPC is in managed code. Returns false for jump stubs on WIN64.
    static BOOL IsManagedCode(PCODE currentPC);

    // Returns true if currentPC is ready to run codegen
    static BOOL IsReadyToRunCode(PCODE currentPC);

    // Returns method's start address for a given PC
    static PCODE GetCodeStartAddress(PCODE currentPC);

    static NativeCodeVersion GetNativeCodeVersion(PCODE currentPC);

    // Returns methodDesc for given PC
    static MethodDesc * GetCodeMethodDesc(PCODE currentPC);

    static IJitManager* FindJitMan(PCODE currentPC)
    {
        CONTRACTL {
            NOTHROW;
            GC_NOTRIGGER;
            SUPPORTS_DAC;
        } CONTRACTL_END;

        RangeSection * pRange = FindCodeRange(currentPC, GetScanFlags());
        return (pRange != NULL) ? pRange->_pjit : NULL;
    }

    static RangeSection * FindCodeRange(PCODE currentPC, ScanFlag scanFlag);

    static BOOL IsCollectibleMethod(const METHODTOKEN& MethodToken);

    class ReaderLockHolder
    {
    public:
        ReaderLockHolder();
        ~ReaderLockHolder();

        BOOL Acquired();
    };

#if defined(TARGET_64BIT)
    static ULONG          GetCLRPersonalityRoutineValue()
    {
        LIMITED_METHOD_CONTRACT;
        return 0;
    }
#endif // TARGET_64BIT

    static EEJitManager * GetEEJitManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_pEEJitManager;
    }

#ifdef FEATURE_READYTORUN
    static ReadyToRunJitManager * GetReadyToRunJitManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_pReadyToRunJitManager;
    }
#endif

#ifdef FEATURE_INTERPRETER
    static InterpreterJitManager * GetInterpreterJitManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_pInterpreterJitManager;
    }
#endif

    static LPCWSTR         GetJitName();

#ifdef FEATURE_INTERPRETER
    static LPCWSTR         GetInterpreterName();

    static ICodeManager*   GetInterpreterCodeManager()
    {
        LIMITED_METHOD_CONTRACT;
        return (ICodeManager *)m_pInterpreterCodeMan;
    }
#endif // FEATURE_INTERPRETER

    static void           Unload(LoaderAllocator *pLoaderAllocator);

    static void           AddCodeRange(TADDR StartRange, TADDR EndRange,
                                       IJitManager* pJit,
                                       RangeSection::RangeSectionFlags flags,
                                       PTR_CodeRangeMapRangeList pRangeList);

    static void           AddCodeRange(TADDR StartRange, TADDR EndRange,
                                       IJitManager* pJit,
                                       RangeSection::RangeSectionFlags flags,
                                       PTR_HeapList pHp);

    static void           AddCodeRange(TADDR StartRange, TADDR EndRange,
                                       IJitManager* pJit,
                                       RangeSection::RangeSectionFlags flags,
                                       PTR_Module pModule);

    static void           DeleteRange(TADDR StartRange);

    static void           CleanupCodeHeaps();

    static ICodeManager*  GetDefaultCodeManager()
    {
        LIMITED_METHOD_CONTRACT;
        return (ICodeManager *)m_pDefaultCodeMan;
    }

    static PTR_Module FindReadyToRunModule(TADDR currentData);

    // FindReadyToRunModule flavor to be used during GC to find GCRefMap
    static PTR_Module FindModuleForGCRefMap(TADDR currentData);

#ifdef DACCESS_COMPILE
    static void EnumMemoryRegions(CLRDataEnumMemoryFlags flags);
#endif

#ifndef DACCESS_COMPILE
    static PCODE jumpStub(MethodDesc* pMD,
                          PCODE target,
                          BYTE * loAddr,
                          BYTE * hiAddr,
                          LoaderAllocator *pLoaderAllocator = NULL,
                          bool throwOnOutOfMemoryWithinRange = true);
#endif

private:
    static RangeSection * FindCodeRangeWithLock(PCODE currentPC);

    static BOOL IsManagedCodeWithLock(PCODE currentPC);
    static BOOL IsManagedCodeWorker(PCODE currentPC, RangeSectionLockState *pLockState);

    static RangeSection* GetRangeSection(TADDR addr, RangeSectionLockState *pLockState);

    SPTR_DECL(EECodeManager, m_pDefaultCodeMan);

    SPTR_DECL(EEJitManager, m_pEEJitManager);
#ifdef FEATURE_READYTORUN
    SPTR_DECL(ReadyToRunJitManager, m_pReadyToRunJitManager);
#endif
#ifdef FEATURE_INTERPRETER
    SPTR_DECL(InterpreterJitManager, m_pInterpreterJitManager);
    SPTR_DECL(InterpreterCodeManager, m_pInterpreterCodeMan);
#endif

    static CrstStatic       m_JumpStubCrst;
    static CrstStatic       m_RangeCrst;        // Acquire before writing into m_CodeRangeList and m_DataRangeList

    // Make the CodeRangeMap a global, initialized as the process starts up.
    // The odd formulation of a BYTE array is used to avoid an extra memory indirection
    // that would be needed if the memory for the CodeRangeMap was dynamically allocated.
    SVAL_DECL(RangeSectionMapData,  g_codeRangeMap);
    static PTR_RangeSectionMap GetCodeRangeMap()
    {
        TADDR codeRangeMapAddress = dac_cast<TADDR>(&(g_codeRangeMap));
        return dac_cast<PTR_RangeSectionMap>(codeRangeMapAddress);
    }

    // infrastructure to manage readers so we can lock them out and delete domain data
    // make ReaderCount volatile because we have order dependency in READER_INCREMENT
    VOLATILE_SVAL_DECL(LONG, m_dwReaderCount);
    VOLATILE_SVAL_DECL(LONG, m_dwWriterLock);

#ifndef DACCESS_COMPILE
    class WriterLockHolder
    {
    public:
        WriterLockHolder();
        ~WriterLockHolder();
    };
#endif

#if defined(_DEBUG)
    // The LOCK_TAKEN/RELEASED macros need a "pointer" to the lock object to do
    // comparisons between takes & releases (and to provide debugging info to the
    // developer).  Since Inc/Dec Reader/Writer are static, there's no object to
    // use.  So we just use the pointer to m_dwReaderCount.  Note that both
    // readers & writers use this same pointer, which follows the general convention
    // of other ReaderWriter locks in the EE code base: each reader/writer locking object
    // instance protects only 1 piece of data or code.  Readers & writers both access the
    // same locking object & shared resource, so conceptually they would share the same
    // lock pointer.
    static void * GetPtrForLockContract()
    {
        return (void *) &m_dwReaderCount;
    }
#endif // defined(_DEBUG)

#ifndef DACCESS_COMPILE
    static PCODE getNextJumpStub(MethodDesc* pMD,
                                 PCODE target,
                                 BYTE * loAddr,  BYTE * hiAddr,
                                 LoaderAllocator *pLoaderAllocator,
                                 bool throwOnOutOfMemoryWithinRange);
#endif

private:
    // ***************************************************************************
    // Hashtable for JumpStubs for jitted code

    struct JumpStubEntry {
        PCODE m_target;
        PCODE m_jumpStub;
    };

    class JumpStubTraits : public DefaultSHashTraits<JumpStubEntry>
    {
    public:
        typedef PCODE key_t;

        static key_t GetKey(element_t e)
        {
            LIMITED_METHOD_CONTRACT;
            return e.m_target;
        }
        static BOOL Equals(key_t k1, key_t k2)
        {
            LIMITED_METHOD_CONTRACT;
            return k1 == k2;
        }
        static count_t Hash(key_t k)
        {
            LIMITED_METHOD_CONTRACT;
#ifdef HOST_64BIT
            return (count_t) ((size_t) k ^ ((size_t) k >> 32));
#else
            return (count_t)(size_t)k;
#endif
        }

        static const element_t Null() { LIMITED_METHOD_CONTRACT; JumpStubEntry e; e.m_target = 0; e.m_jumpStub = 0; return e; }
        static bool IsNull(const element_t &e) { LIMITED_METHOD_CONTRACT; return e.m_target == 0; }
        static const element_t Deleted() { LIMITED_METHOD_CONTRACT; JumpStubEntry e; e.m_target = (PCODE)-1; e.m_jumpStub = 0; return e; }
        static bool IsDeleted(const element_t &e) { LIMITED_METHOD_CONTRACT; return e.m_target == (PCODE)-1; }
    };
    typedef SHash<JumpStubTraits> JumpStubTable;

    static unsigned m_normal_JumpStubLookup;
    static unsigned m_normal_JumpStubUnique;
    static unsigned m_normal_JumpStubBlockAllocCount;
    static unsigned m_normal_JumpStubBlockFullCount;

    static unsigned m_LCG_JumpStubLookup;
    static unsigned m_LCG_JumpStubUnique;
    static unsigned m_LCG_JumpStubBlockAllocCount;
    static unsigned m_LCG_JumpStubBlockFullCount;

public:

    static void DumpExecutionManagerUsage()
    {
        minipal_log_print_info("JumpStub usage count:\nNormal: %u, LCG: %u\n", m_normal_JumpStubLookup, m_LCG_JumpStubLookup);
    }

    struct JumpStubCache
    {
        JumpStubCache()
            : m_pBlocks(NULL)
        {
            LIMITED_METHOD_CONTRACT;
        }

        JumpStubBlockHeader * m_pBlocks;
        JumpStubTable m_Table;
    };

    friend struct ::cdac_data<ExecutionManager>;
};

#ifndef DACCESS_COMPILE
template<>
struct cdac_data<ExecutionManager>
{
    static constexpr void* const CodeRangeMapAddress = (void*)&ExecutionManager::g_codeRangeMap.Data[0];
};
#endif

inline CodeHeader * EEJitManager::GetCodeHeader(const METHODTOKEN& MethodToken)
{
    LIMITED_METHOD_DAC_CONTRACT;
    _ASSERTE(!MethodToken.IsNull());
    return dac_cast<PTR_CodeHeader>(MethodToken.m_pCodeHeader);
}

inline CodeHeader * EEJitManager::GetCodeHeaderFromStartAddress(TADDR methodStartAddress)
{
    LIMITED_METHOD_DAC_CONTRACT;
    _ASSERTE(methodStartAddress != (TADDR)NULL);
    ARM_ONLY(_ASSERTE((methodStartAddress & THUMB_CODE) == 0));
    return dac_cast<PTR_CodeHeader>(methodStartAddress - sizeof(CodeHeader));
}

inline TADDR EEJitManager::JitTokenToStartAddress(const METHODTOKEN& MethodToken)
{
    CONTRACTL {
        NOTHROW;
        GC_NOTRIGGER;
        SUPPORTS_DAC;
    } CONTRACTL_END;

    CodeHeader * pCH = GetCodeHeader(MethodToken);
    return pCH->GetCodeStartAddress();
}

inline void EEJitManager::JitTokenToMethodRegionInfo(const METHODTOKEN& MethodToken,
                                                     MethodRegionInfo * methodRegionInfo)
{
    CONTRACTL {
        NOTHROW;
        GC_NOTRIGGER;
        SUPPORTS_DAC;
        PRECONDITION(methodRegionInfo != NULL);
    } CONTRACTL_END;

    methodRegionInfo->hotStartAddress  = JitTokenToStartAddress(MethodToken);
    methodRegionInfo->hotSize          = GetCodeManager()->GetFunctionSize(GetGCInfoToken(MethodToken));
    methodRegionInfo->coldStartAddress = 0;
    methodRegionInfo->coldSize         = 0;
}

#if defined(FEATURE_READYTORUN)

class NativeExceptionInfoLookupTable
{
public:
    static DWORD LookupExceptionInfoRVAForMethod(PTR_CORCOMPILE_EXCEPTION_LOOKUP_TABLE pTable,
                                                                   COUNT_T numLookupEntries,
                                                                   DWORD methodStartRVA,
                                                                   COUNT_T* pSize);
};

class NativeUnwindInfoLookupTable
{
public:
    static int LookupUnwindInfoForMethod(DWORD codeOffset,
                                         PTR_RUNTIME_FUNCTION pRuntimeFunctionTable,
                                         int StartIndex,
                                         int EndIndex);
};

class HotColdMappingLookupTable
{
public:
    // ***************************************************************************
    // Binary searches pInfo->m_pHotColdMap for the given hot/cold MethodIndex, and
    // returns the index in pInfo->m_pHotColdMap of its corresponding cold/hot MethodIndex.
    // If MethodIndex is cold and at index i, returns i + 1.
    // If MethodIndex is hot and at index i, returns i - 1.
    // If MethodIndex is not in pInfo->m_pHotColdMap, returns -1.
    //
    static int LookupMappingForMethod(ReadyToRunInfo* pInfo, ULONG MethodIndex);
};

#endif // FEATURE_READYTORUN

#ifdef FEATURE_READYTORUN

class ReadyToRunJitManager final : public IJitManager
{
    VPTR_VTABLE_CLASS(ReadyToRunJitManager, IJitManager)

public:
#ifndef DACCESS_COMPILE
    ReadyToRunJitManager();
#endif // #ifndef DACCESS_COMPILE

    virtual DWORD GetCodeType()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return (miManaged | miNative);
    }

    // Used to read debug info.
    virtual BOOL GetBoundariesAndVars(
        const DebugInfoRequest & request,
        IN FP_IDS_NEW fpNew,
        IN void * pNewData,
        BoundsType boundsType,
        OUT ULONG32 * pcMap,
        OUT ICorDebugInfo::OffsetMapping **ppMap,
        OUT ULONG32 * pcVars,
        OUT ICorDebugInfo::NativeVarInfo **ppVars);

    virtual size_t WalkILOffsets(
        const DebugInfoRequest & request,
        BoundsType boundsType,
        void* pContext,
        size_t (*pfnWalkILOffsets)(ICorDebugInfo::OffsetMapping *pOffsetMapping, void *pContext));

    virtual BOOL GetRichDebugInfo(
        const DebugInfoRequest & request,
        IN FP_IDS_NEW fpNew, IN void * pNewData,
        OUT ICorDebugInfo::InlineTreeNode**    ppInlineTree,
        OUT ULONG32*                           pNumInlineTree,
        OUT ICorDebugInfo::RichOffsetMapping** ppRichMappings,
        OUT ULONG32*                           pNumRichMappings);

    virtual BOOL JitCodeToMethodInfo(RangeSection * pRangeSection,
                                     PCODE currentPC,
                                     MethodDesc** ppMethodDesc,
                                     OUT EECodeInfo * pCodeInfo);

    virtual PCODE GetCodeAddressForRelOffset(const METHODTOKEN& MethodToken, DWORD relOffset);

    static ReadyToRunInfo * JitTokenToReadyToRunInfo(const METHODTOKEN& MethodToken);
    static UINT32 JitTokenToGCInfoVersion(const METHODTOKEN& MethodToken);

    static PTR_RUNTIME_FUNCTION JitTokenToRuntimeFunction(const METHODTOKEN& MethodToken);

    virtual TADDR       JitTokenToStartAddress(const METHODTOKEN& MethodToken);
    virtual void        JitTokenToMethodRegionInfo(const METHODTOKEN& MethodToken, MethodRegionInfo * methodRegionInfo);

    virtual unsigned    InitializeEHEnumeration(const METHODTOKEN& MethodToken, EH_CLAUSE_ENUMERATOR* pEnumState);

    virtual PTR_EXCEPTION_CLAUSE_TOKEN  GetNextEHClause(EH_CLAUSE_ENUMERATOR* pEnumState,
                                        EE_ILEXCEPTION_CLAUSE* pEHclause);

#ifndef DACCESS_COMPILE
    virtual TypeHandle  ResolveEHClause(EE_ILEXCEPTION_CLAUSE* pEHClause,
                                        CrawlFrame *pCf);
#endif // #ifndef DACCESS_COMPILE

    virtual GCInfoToken  GetGCInfoToken(const METHODTOKEN& MethodToken);

#if defined(FEATURE_EH_FUNCLETS)
    virtual PTR_RUNTIME_FUNCTION    LazyGetFunctionEntry(EECodeInfo * pCodeInfo);

    virtual TADDR                   GetFuncletStartAddress(EECodeInfo * pCodeInfo);
    virtual DWORD                   GetFuncletStartOffsets(const METHODTOKEN& MethodToken, DWORD* pStartFuncletOffsets, DWORD dwLength);
    virtual BOOL                    LazyIsFunclet(EECodeInfo * pCodeInfo);
    virtual BOOL                    IsFilterFunclet(EECodeInfo * pCodeInfo);
#endif // FEATURE_EH_FUNCLETS

    virtual StubCodeBlockKind       GetStubCodeBlockKind(RangeSection * pRangeSection, PCODE currentPC);

#if defined(DACCESS_COMPILE)
    virtual void EnumMemoryRegions(CLRDataEnumMemoryFlags flags);
    virtual void EnumMemoryRegionsForMethodDebugInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo);
#if defined(FEATURE_EH_FUNCLETS)
    // Enumerate the memory necessary to retrieve the unwind info for a specific method
    virtual void EnumMemoryRegionsForMethodUnwindInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo);
#endif //FEATURE_EH_FUNCLETS
#endif //DACCESS_COMPILE
};

#endif

#ifdef FEATURE_INTERPRETER

class InterpreterJitManager final : public EECodeGenManager
{
    VPTR_VTABLE_CLASS(InterpreterJitManager, EECodeGenManager)
public:
    ICorJitCompiler *   m_interpreter;
    HINSTANCE           m_interpreterHandle;

    InterpreterJitManager();

    virtual ICorJitCompiler* GetCompiler()
    {
        return m_interpreter;
    }

    virtual ICorJitCompiler* GetAltCompiler()
    {
        return NULL;
    }

    BOOL LoadInterpreter();

    BOOL IsInterpreterLoaded()
    {
        LIMITED_METHOD_CONTRACT;
        return m_interpreter != NULL;
    }

    virtual DWORD GetCodeType()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        // Interpreter-TODO: consider adding some extra flag for the interpreter
        return (miManaged | miIL);
    }

    GCInfoToken GetGCInfoToken(const METHODTOKEN& MethodToken);
    virtual unsigned InitializeEHEnumeration(const METHODTOKEN& MethodToken, EH_CLAUSE_ENUMERATOR* pEnumState);
    virtual PCODE GetCodeAddressForRelOffset(const METHODTOKEN& MethodToken, DWORD relOffset);

    virtual TADDR JitTokenToStartAddress(const METHODTOKEN& MethodToken);

    virtual void JitTokenToMethodRegionInfo(const METHODTOKEN& MethodToken, MethodRegionInfo * methodRegionInfo);

    static InterpreterCodeHeader * GetCodeHeaderFromStartAddress(TADDR methodStartAddress);
    static InterpreterCodeHeader * GetCodeHeader(const METHODTOKEN& MethodToken);

    virtual BOOL JitCodeToMethodInfo(RangeSection * pRangeSection,
        PCODE currentPC,
        MethodDesc ** ppMethodDesc,
        EECodeInfo * pCodeInfo);

    // Used to read debug info.
    virtual BOOL GetBoundariesAndVars(
        const DebugInfoRequest & request,
        IN FP_IDS_NEW fpNew,
        IN void * pNewData,
        BoundsType boundsType,
        OUT ULONG32 * pcMap,
        OUT ICorDebugInfo::OffsetMapping **ppMap,
        OUT ULONG32 * pcVars,
        OUT ICorDebugInfo::NativeVarInfo **ppVars);

    virtual size_t WalkILOffsets(
        const DebugInfoRequest & request,
        BoundsType boundsType,
        void* pContext,
        size_t (*pfnWalkILOffsets)(ICorDebugInfo::OffsetMapping *pOffsetMapping, void *pContext));

    virtual BOOL GetRichDebugInfo(
        const DebugInfoRequest& request,
        IN FP_IDS_NEW fpNew, IN void* pNewData,
        OUT ICorDebugInfo::InlineTreeNode** ppInlineTree,
        OUT ULONG32* pNumInlineTree,
        OUT ICorDebugInfo::RichOffsetMapping** ppRichMappings,
        OUT ULONG32* pNumRichMappings);

#if defined(FEATURE_EH_FUNCLETS)
    virtual PTR_RUNTIME_FUNCTION LazyGetFunctionEntry(EECodeInfo * pCodeInfo)
    {
        // Not used for the interpreter
        _ASSERTE(FALSE);
        return PTR_NULL;
    }

    virtual TADDR GetFuncletStartAddress(EECodeInfo * pCodeInfo);
    virtual DWORD GetFuncletStartOffsets(const METHODTOKEN& MethodToken, DWORD* pStartFuncletOffsets, DWORD dwLength);

#if !defined DACCESS_COMPILE
protected:
    virtual void DeleteFunctionTable(PVOID pvTableID)
    {
        // Nothing to do for the interpreter
    }

public:
#endif // !DACCESS_COMPILE

#endif // FEATURE_EH_FUNCLETS

    virtual StubCodeBlockKind GetStubCodeBlockKind(RangeSection * pRangeSection, PCODE currentPC)
    {
        return STUB_CODE_BLOCK_UNKNOWN;
    }

#if defined(DACCESS_COMPILE)

    virtual void EnumMemoryRegionsForMethodDebugInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo);

#if defined(FEATURE_EH_FUNCLETS)
    virtual void EnumMemoryRegionsForMethodUnwindInfo(CLRDataEnumMemoryFlags flags, EECodeInfo * pCodeInfo)
    {
        // Not used for the interpreter
        _ASSERTE(FALSE);
    }

#endif // FEATURE_EH_FUNCLETS
#endif // DACCESS_COMPILE
private :
    Crst m_interpreterLoadLock;
};

#endif // FEATURE_INTERPRETER

//*****************************************************************************
// EECodeInfo provides information about code at particular address:
//  - Start of the method and relative offset
//  - GC Info of the method
//  etc.
//
// EECodeInfo caches information from IJitManager and thus avoids
// quering IJitManager repeatedly for same data.
//
class EECodeInfo
{
    friend BOOL EECodeGenManager::JitCodeToMethodInfoWorker<CodeHeader>(RangeSection * pRangeSection, PCODE currentPC, MethodDesc** ppMethodDesc, EECodeInfo * pCodeInfo);
#ifdef FEATURE_INTERPRETER
    friend BOOL EECodeGenManager::JitCodeToMethodInfoWorker<InterpreterCodeHeader>(RangeSection * pRangeSection, PCODE currentPC, MethodDesc** ppMethodDesc, EECodeInfo * pCodeInfo);
#endif
#ifdef FEATURE_READYTORUN
    friend BOOL ReadyToRunJitManager::JitCodeToMethodInfo(RangeSection * pRangeSection, PCODE currentPC, MethodDesc** ppMethodDesc, EECodeInfo * pCodeInfo);
#endif

#if defined(FEATURE_EH_FUNCLETS)
    enum class IsFuncletCache : uint32_t
    {
        NotSet = 2,
        IsFunclet = 1,
        IsNotFunclet = 0
    };
#endif // FEATURE_EH_FUNCLETS

public:
    EECodeInfo();

    EECodeInfo(PCODE codeAddress)
    {
        Init(codeAddress);
    }

    // Explicit initialization
    void Init(PCODE codeAddress);
    void Init(PCODE codeAddress, ExecutionManager::ScanFlag scanFlag);

    TADDR       GetSavedMethodCode();

    TADDR       GetStartAddress() const;

    BOOL        IsValid()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_pJM != NULL;
    }

    IJitManager* GetJitManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        _ASSERTE(m_pJM != NULL);
        return m_pJM;
    }

    ICodeManager* GetCodeManager()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return GetJitManager()->GetCodeManager();
    }

    const METHODTOKEN& GetMethodToken()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_methodToken;
    }

    // This returns a pointer to the start of an instruction; conceptually, a PINSTR.
    TADDR       GetCodeAddress() const
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return PCODEToPINSTR(m_codeAddress);
    }

    NativeCodeVersion GetNativeCodeVersion() const;

    MethodDesc * GetMethodDesc() const
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_pMD;
    }

    DWORD       GetRelOffset()
    {
        LIMITED_METHOD_DAC_CONTRACT;
        return m_relOffset;
    }

    GCInfoToken  GetGCInfoToken()
    {
        WRAPPER_NO_CONTRACT;
        return GetJitManager()->GetGCInfoToken(GetMethodToken());
    }

    PTR_VOID GetGCInfo()
    {
        WRAPPER_NO_CONTRACT;
        return GetGCInfoToken().Info;
    }

    void        GetMethodRegionInfo(IJitManager::MethodRegionInfo *methodRegionInfo)
    {
        WRAPPER_NO_CONTRACT;
        return GetJitManager()->JitTokenToMethodRegionInfo(GetMethodToken(), methodRegionInfo);
    }

    TADDR       GetModuleBase()
    {
        WRAPPER_NO_CONTRACT;
        return GetJitManager()->JitTokenToModuleBase(GetMethodToken());
    }

#ifdef FEATURE_EH_FUNCLETS
    PTR_RUNTIME_FUNCTION GetFunctionEntry();
    BOOL        IsFunclet();
    EECodeInfo  GetMainFunctionInfo();
#endif // FEATURE_EH_FUNCLETS

#if defined(TARGET_X86)
    ULONG       GetFixedStackSize()
    {
        WRAPPER_NO_CONTRACT;
        return GetCodeManager()->GetFrameSize(GetGCInfoToken());
    }

    FORCEINLINE PTR_CBYTE DecodeGCHdrInfo(hdrInfo   ** infoPtr)
    {
        if (m_hdrInfoTable == NULL)
        {
            return DecodeGCHdrInfoHelper(infoPtr);
        }

        *infoPtr = &m_hdrInfoBody;
        return m_hdrInfoTable;
    }

    PTR_CBYTE DecodeGCHdrInfo(hdrInfo * infoPtr, DWORD relOffset);

private:
    PTR_CBYTE   DecodeGCHdrInfoHelper(hdrInfo   ** infoPtr);
#endif // TARGET_X86

#if defined(TARGET_WASM)
ULONG       GetFixedStackSize();
#endif

#if defined(TARGET_AMD64)
    BOOL        HasFrameRegister();
    ULONG       GetFixedStackSize();

    void         GetOffsetsFromUnwindInfo(ULONG* pRSPOffset, ULONG* pRBPOffset);
#endif // TARGET_AMD64

private:
    PCODE               m_codeAddress;
    METHODTOKEN         m_methodToken;
    MethodDesc         *m_pMD;
    IJitManager        *m_pJM;
    DWORD               m_relOffset;
#ifdef FEATURE_EH_FUNCLETS
    IsFuncletCache      m_isFuncletCache;
    PTR_RUNTIME_FUNCTION m_pFunctionEntry;
#endif // FEATURE_EH_FUNCLETS

#ifdef TARGET_X86
    PTR_CBYTE           m_hdrInfoTable;
    hdrInfo             m_hdrInfoBody;
#endif

#ifdef TARGET_AMD64
    // Simple helper to return a pointer to the UNWIND_INFO given the offset to the unwind info.
    UNWIND_INFO * GetUnwindInfoHelper(ULONG unwindInfoOffset);
#endif // TARGET_AMD64

};

#ifdef FEATURE_INTERPRETER

inline InterpreterCodeHeader* InterpreterJitManager::GetCodeHeader(const METHODTOKEN& MethodToken)
{
    LIMITED_METHOD_DAC_CONTRACT;
    _ASSERTE(!MethodToken.IsNull());
    return dac_cast<PTR_InterpreterCodeHeader>(MethodToken.m_pCodeHeader);
}

inline InterpreterCodeHeader * InterpreterJitManager::GetCodeHeaderFromStartAddress(TADDR methodStartAddress)
{
    LIMITED_METHOD_DAC_CONTRACT;
    _ASSERTE(methodStartAddress != (TADDR)NULL);
    return dac_cast<PTR_InterpreterCodeHeader>(methodStartAddress - sizeof(InterpreterCodeHeader));
}

inline TADDR InterpreterJitManager::JitTokenToStartAddress(const METHODTOKEN& MethodToken)
{
    CONTRACTL {
        NOTHROW;
        GC_NOTRIGGER;
        SUPPORTS_DAC;
    } CONTRACTL_END;

    InterpreterCodeHeader * pCH = GetCodeHeader(MethodToken);
    return pCH->GetCodeStartAddress();
}

#endif // FEATURE_INTERPRETER

#include "codeman.inl"

void ThrowOutOfMemoryWithinRange();

#endif // !__CODEMAN_HPP__
