﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iotwireless/IoTWirelessRequest.h>
#include <aws/iotwireless/IoTWireless_EXPORTS.h>
#include <aws/iotwireless/model/Tag.h>
#include <aws/iotwireless/model/UpdateWirelessGatewayTaskCreate.h>

#include <utility>

namespace Aws {
namespace IoTWireless {
namespace Model {

/**
 */
class CreateWirelessGatewayTaskDefinitionRequest : public IoTWirelessRequest {
 public:
  AWS_IOTWIRELESS_API CreateWirelessGatewayTaskDefinitionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateWirelessGatewayTaskDefinition"; }

  AWS_IOTWIRELESS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Whether to automatically create tasks using this task definition for all
   * gateways with the specified current version. If <code>false</code>, the task
   * must me created by calling <code>CreateWirelessGatewayTask</code>.</p>
   */
  inline bool GetAutoCreateTasks() const { return m_autoCreateTasks; }
  inline bool AutoCreateTasksHasBeenSet() const { return m_autoCreateTasksHasBeenSet; }
  inline void SetAutoCreateTasks(bool value) {
    m_autoCreateTasksHasBeenSet = true;
    m_autoCreateTasks = value;
  }
  inline CreateWirelessGatewayTaskDefinitionRequest& WithAutoCreateTasks(bool value) {
    SetAutoCreateTasks(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the new resource.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateWirelessGatewayTaskDefinitionRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the gateways to update.</p>
   */
  inline const UpdateWirelessGatewayTaskCreate& GetUpdate() const { return m_update; }
  inline bool UpdateHasBeenSet() const { return m_updateHasBeenSet; }
  template <typename UpdateT = UpdateWirelessGatewayTaskCreate>
  void SetUpdate(UpdateT&& value) {
    m_updateHasBeenSet = true;
    m_update = std::forward<UpdateT>(value);
  }
  template <typename UpdateT = UpdateWirelessGatewayTaskCreate>
  CreateWirelessGatewayTaskDefinitionRequest& WithUpdate(UpdateT&& value) {
    SetUpdate(std::forward<UpdateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Each resource must have a unique client request token. The client token is
   * used to implement idempotency. It ensures that the request completes no more
   * than one time. If you retry a request with the same token and the same
   * parameters, the request will complete successfully. However, if you try to
   * create a new resource using the same token but different parameters, an HTTP 409
   * conflict occurs. If you omit this value, AWS SDKs will automatically generate a
   * unique client request. For more information about idempotency, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
   * idempotency in Amazon EC2 API requests</a>.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CreateWirelessGatewayTaskDefinitionRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to attach to the specified resource. Tags are metadata that you can
   * use to manage a resource.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateWirelessGatewayTaskDefinitionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateWirelessGatewayTaskDefinitionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_autoCreateTasks{false};

  Aws::String m_name;

  UpdateWirelessGatewayTaskCreate m_update;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Vector<Tag> m_tags;
  bool m_autoCreateTasksHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_updateHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTWireless
}  // namespace Aws
