﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/frauddetector/FraudDetectorRequest.h>
#include <aws/frauddetector/FraudDetector_EXPORTS.h>
#include <aws/frauddetector/model/ListUpdateMode.h>

#include <utility>

namespace Aws {
namespace FraudDetector {
namespace Model {

/**
 */
class UpdateListRequest : public FraudDetectorRequest {
 public:
  AWS_FRAUDDETECTOR_API UpdateListRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateList"; }

  AWS_FRAUDDETECTOR_API Aws::String SerializePayload() const override;

  AWS_FRAUDDETECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The name of the list to update. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateListRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> One or more list elements to add or replace. If you are providing the
   * elements, make sure to specify the <code>updateMode</code> to use. </p> <p>If
   * you are deleting all elements from the list, use <code>REPLACE</code> for the
   * <code>updateMode</code> and provide an empty list (0 elements).</p>
   */
  inline const Aws::Vector<Aws::String>& GetElements() const { return m_elements; }
  inline bool ElementsHasBeenSet() const { return m_elementsHasBeenSet; }
  template <typename ElementsT = Aws::Vector<Aws::String>>
  void SetElements(ElementsT&& value) {
    m_elementsHasBeenSet = true;
    m_elements = std::forward<ElementsT>(value);
  }
  template <typename ElementsT = Aws::Vector<Aws::String>>
  UpdateListRequest& WithElements(ElementsT&& value) {
    SetElements(std::forward<ElementsT>(value));
    return *this;
  }
  template <typename ElementsT = Aws::String>
  UpdateListRequest& AddElements(ElementsT&& value) {
    m_elementsHasBeenSet = true;
    m_elements.emplace_back(std::forward<ElementsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The new description. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateListRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The update mode (type). </p> <ul> <li> <p>Use <code>APPEND</code> if you are
   * adding elements to the list.</p> </li> <li> <p>Use <code>REPLACE</code> if you
   * replacing existing elements in the list.</p> </li> <li> <p>Use
   * <code>REMOVE</code> if you are removing elements from the list.</p> </li> </ul>
   */
  inline ListUpdateMode GetUpdateMode() const { return m_updateMode; }
  inline bool UpdateModeHasBeenSet() const { return m_updateModeHasBeenSet; }
  inline void SetUpdateMode(ListUpdateMode value) {
    m_updateModeHasBeenSet = true;
    m_updateMode = value;
  }
  inline UpdateListRequest& WithUpdateMode(ListUpdateMode value) {
    SetUpdateMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The variable type you want to assign to the list. </p>  <p>You cannot
   * update a variable type of a list that already has a variable type assigned to
   * it. You can assign a variable type to a list only if the list does not already
   * have a variable type.</p>
   */
  inline const Aws::String& GetVariableType() const { return m_variableType; }
  inline bool VariableTypeHasBeenSet() const { return m_variableTypeHasBeenSet; }
  template <typename VariableTypeT = Aws::String>
  void SetVariableType(VariableTypeT&& value) {
    m_variableTypeHasBeenSet = true;
    m_variableType = std::forward<VariableTypeT>(value);
  }
  template <typename VariableTypeT = Aws::String>
  UpdateListRequest& WithVariableType(VariableTypeT&& value) {
    SetVariableType(std::forward<VariableTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::Vector<Aws::String> m_elements;

  Aws::String m_description;

  ListUpdateMode m_updateMode{ListUpdateMode::NOT_SET};

  Aws::String m_variableType;
  bool m_nameHasBeenSet = false;
  bool m_elementsHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_updateModeHasBeenSet = false;
  bool m_variableTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace FraudDetector
}  // namespace Aws
