//! PKCS#1 encoding tests

#![cfg(feature = "encoding")]

use crypto_bigint::{BoxedUint, CtEq};
use hex_literal::hex;
use rsa::{
    pkcs1::{DecodeRsaPrivateKey, DecodeRsaPublicKey, EncodeRsaPrivateKey, EncodeRsaPublicKey},
    traits::{PrivateKeyParts, PublicKeyParts},
    RsaPrivateKey, RsaPublicKey,
};

#[cfg(feature = "encoding")]
use rsa::pkcs1::LineEnding;

/// RSA-2048 PKCS#1 private key encoded as ASN.1 DER.
///
/// Note: this key is extracted from the corresponding `rsa2048-priv.der`
/// example key in the `pkcs8` crate.
const RSA_2048_PRIV_DER: &[u8] = include_bytes!("examples/pkcs1/rsa2048-priv.der");

/// RSA-4096 PKCS#1 private key encoded as ASN.1 DER
const RSA_4096_PRIV_DER: &[u8] = include_bytes!("examples/pkcs1/rsa4096-priv.der");

/// RSA-2048 PKCS#1 public key encoded as ASN.1 DER.
///
/// Note: this key is extracted from the corresponding `rsa2048-priv.der`
/// example key in the `pkcs8` crate.
const RSA_2048_PUB_DER: &[u8] = include_bytes!("examples/pkcs1/rsa2048-pub.der");

/// RSA-4096 PKCS#1 public key encoded as ASN.1 DER
const RSA_4096_PUB_DER: &[u8] = include_bytes!("examples/pkcs1/rsa4096-pub.der");

/// RSA-2048 PKCS#1 private key encoded as PEM
#[cfg(feature = "encoding")]
const RSA_2048_PRIV_PEM: &str = include_str!("examples/pkcs1/rsa2048-priv.pem");

/// RSA-4096 PKCS#1 private key encoded as PEM
#[cfg(feature = "encoding")]
const RSA_4096_PRIV_PEM: &str = include_str!("examples/pkcs1/rsa4096-priv.pem");

/// RSA-2048 PKCS#1 public key encoded as PEM
#[cfg(feature = "encoding")]
const RSA_2048_PUB_PEM: &str = include_str!("examples/pkcs1/rsa2048-pub.pem");

/// RSA-4096 PKCS#1 public key encoded as PEM
#[cfg(feature = "encoding")]
const RSA_4096_PUB_PEM: &str = include_str!("examples/pkcs1/rsa4096-pub.pem");

#[test]
fn decode_rsa2048_priv_der() {
    let key = RsaPrivateKey::from_pkcs1_der(RSA_2048_PRIV_DER).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa2048-priv.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "B6C42C515F10A6AAF282C63EDBE24243A170F3FA2633BD4833637F47CA4F6F36"
            "E03A5D29EFC3191AC80F390D874B39E30F414FCEC1FCA0ED81E547EDC2CD382C"
            "76F61C9018973DB9FA537972A7C701F6B77E0982DFC15FC01927EE5E7CD94B4F"
            "599FF07013A7C8281BDF22DCBC9AD7CABB7C4311C982F58EDB7213AD4558B332"
            "266D743AED8192D1884CADB8B14739A8DADA66DC970806D9C7AC450CB13D0D7C"
            "575FB198534FC61BC41BC0F0574E0E0130C7BBBFBDFDC9F6A6E2E3E2AFF1CBEA"
            "C89BA57884528D55CFB08327A1E8C89F4E003CF2888E933241D9D695BCBBACDC"
            "90B44E3E095FA37058EA25B13F5E295CBEAC6DE838AB8C50AF61E298975B872F"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
    assert_eq!(
        &key.d().to_be_bytes()[..],
        &hex!(
            "7ECC8362C0EDB0741164215E22F74AB9D91BA06900700CF63690E5114D8EE6BD"
            "CFBB2E3F9614692A677A083F168A5E52E5968E6407B9D97C6E0E4064F82DA0B7"
            "58A14F17B9B7D41F5F48E28D6551704F56E69E7AA9FA630FC76428C06D25E455"
            "DCFC55B7AC2B4F76643FDED3FE15FF78ABB27E65ACC4AAD0BDF6DB27EF60A691"
            "0C5C4A085ED43275AB19C1D997A32C6EFFCE7DF2D1935F6E601EEDE161A12B5C"
            "C27CA21F81D2C99C3D1EA08E90E3053AB09BEFA724DEF0D0C3A3C1E9740C0D9F"
            "76126A149EC0AA7D8078205484254D951DB07C4CF91FB6454C096588FD5924DB"
            "ABEB359CA2025268D004F9D66EB3D6F7ADC1139BAD40F16DDE639E11647376C1"
        )
    );
    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "DCC061242D4E92AFAEE72AC513CA65B9F77036F9BD7E0E6E61461A7EF7654225"
            "EC153C7E5C31A6157A6E5A13FF6E178E8758C1CB33D9D6BBE3179EF18998E422"
            "ECDCBED78F4ECFDBE5F4FCD8AEC2C9D0DC86473CA9BD16D9D238D21FB5DDEFBE"
            "B143CA61D0BD6AA8D91F33A097790E9640DBC91085DC5F26343BA3138F6B2D67"
        ),
        1024,
    )
    .unwrap();
    assert!(bool::from(key.primes()[0].ct_eq(&expected_prime)));

    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "D3F314757E40E954836F92BE24236AF2F0DA04A34653C180AF67E960086D93FD"
            "E65CB23EFD9D09374762F5981E361849AF68CDD75394FF6A4E06EB69B209E422"
            "8DB2DFA70E40F7F9750A528176647B788D0E5777A2CB8B22E3CD267FF70B4F3B"
            "02D3AAFB0E18C590A564B03188B0AA5FC48156B07622214243BD1227EFA7F2F9"
        ),
        1024,
    )
    .unwrap();
    assert!(bool::from(key.primes()[1].ct_eq(&expected_prime)));
}

#[test]
fn decode_rsa4096_priv_der() {
    let key = RsaPrivateKey::from_pkcs1_der(RSA_4096_PRIV_DER).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa4096-priv.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "A7A74572811EA2617E49E85BD730DDE30F103F7D88EE3F765E540D3DD993BBB0"
            "BA140002859D0B40897436637F58B828EA74DF8321634077F99D4AA2D54CA375"
            "852EF597661D3713CE1EF3B4FD6A8E220238E467668A2C7EE3861D2212AE6A1E"
            "BDDFA88B62DF10F6BCF79EFF4AC298FB2563DF1B8764381AF9B1FB0CCD085E02"
            "6B0AD9F6721A235177D0396B48754AD4A75242250A873BF2F6E7EE3C75DD613E"
            "365BA4F3210A6CC66B90A2FA3F762CA6884087B6BF8161EB144819F0F572F21F"
            "6C8E273E70D45A365B8B2819CE734613CC23B01329A17901F17078403861F54C"
            "52A051E2A58C75C2D9D80091BB9808A106C1F7ECB4034E15058BEEC725C5F919"
            "D62EAA234B62628D346C60BB919E70851DAB38571E6F0ED7634129F994EA368F"
            "EE7373DFDEC04445EBCA47FA20ED1540A860C948BABC98DA591CA1DE2E2E2554"
            "0EF9B7CB353F60213B814A45D359EFA9B811EEFF08C65993BF8A85C2BFEAAA7E"
            "D5E6B43E18AE604464CE5F96150136E7D09F8B24FAD43D7870118CFA7BC24875"
            "506EBBC321B977E0861AEA50128620121F0B394A9CDD0A42411A1350C0770D97"
            "5D71B00A90436240C967A0C3A5C20A0F6DE77F3F2CAFDA94ED0143C1F6E34F73"
            "E0CAC279EEEB7C637723A2B026C82802E1A4AEBAA8846DF98E7919498773E0D4"
            "F319956F4DE3AAD00EFB9A147D66B3AC1A01D35B2CFB48D400B0E7A80DC97551"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
    assert_eq!(
        &key.d().to_be_bytes()[..],
        &hex!(
            "9FE3097B2322B90FAB6606C017A095EBE640C39C100BCEE02F238FA14DAFF38E"
            "9E57568F1127ED4436126B904631B127EC395BB3EE127EB82C88D2562A7FB55F"
            "ED8D1450B7E4E2D2F37F5742636FCC6F289963522D5B5706082CADFA01C0EE99"
            "B4D0E9274D3A992E06974CBE01694686356962AC1959FD9BD447E5B9968C0543"
            "DF1BF134742AF345CDB2FA1F9371B0D4CF61C68D16D653D8E999D4FD3A16CF97"
            "8A35AA40E860CDCE09655DD8B4CF19D4141B1E92AD5E51A8E4A5C27FA745611D"
            "90E49D0E9282222AB6F126643E1C77578816FCE3B98F321D2549F294A470DF84"
            "53446BF36F985DF25ED8FDE9FDF3073FB27727DF48E9E1FC7056BC78965090B7"
            "850126406462C8253051EF84E34EE3C3CEB8F96C658C38BE45558D2F64E29D22"
            "3350555FC1EFA28EC1F4AFB5BA4080F09A86CDC3538C1AD7C972E6D7A3612E68"
            "45BA9AFBDF19F09060D1A779DE9635E2D2F8E0C510BA24C6C44B30C9BDFAF85B"
            "E917AEC5D43AFAB1AA3ADD33CC83DA93CAC69218F6A36EB47F199D5424C95FD9"
            "ED7B1E8BE2AEAA6433B227241316C20EE792650CEB48BFD634446B19D286B4EA"
            "1722498DA1A36973210EC3824751A5808D9AAEF59C449E19A5077CFECA126BD9"
            "A8DD4996561D4E27B3609FF82C5B1B21E627845D44961B33B875D5C4FA9FF357"
            "EF6BE3364969E1337C91B29A07B9A913CDE40CE2D5530C900E73751685E65431"
        )
    );
    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "D0213A79425B665B719118448893EC3275600F63DBF85B77F4E8E99EF302F6E8"
            "2596048F6DCA772DE6BBF1124DB84B0AFE61B03A8604AB0079ED53F3304797AD"
            "01B38C44FE27A5A45E378483A804B56A4A967F48F01A866E721E67E4C9A1048A"
            "F68927FAA43D6A85D93E7BF7074DBA797563FCABE12309B76653C6DB614DC231"
            "CC556D9F25AC4841A02D31CDF3015B212307F9D0C79FEB5D3956CE53CC8FA165"
            "1BE60761F19F74672489EAF9F215409F39956E77A82183F1F72BB2FEDDF1B9FB"
            "FC4AD89EA445809DDBD5BD595277990C0BE9366FBB2ECF7B057CC1C3DC8FB77B"
            "F8456D07BBC95B3C1815F48E62B81468C3D4D9D96C0F48DAB04993BE8D91EDE5"
        ),
        2048,
    )
    .unwrap();
    assert!(bool::from(key.primes()[0].ct_eq(&expected_prime)));

    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "CE36C6810522ABE5D6465F36EB137DA3B9EA4A5F1D27C6614729EB8E5E2E5CB8"
            "8E3EF1A473A21944B66557B3DC2CE462E4BF3446CB4990037E5672B1705CBAE8"
            "1B65BAF967A266DC18EFE80F4DBBFE1A59063205CE2943CADF421CCE74AF7063"
            "FD1A83AF3C39AF84525F59BDC1FF54815F52AFD1E8D4862B2C3654F6CFA83DC0"
            "8E2A9D52B9F833C646AF7694467DFC5F7D7AD7B441895FCB7FFBED526324B015"
            "4A15823F5107C89548EDDCB61DA5308C6CC834D4A0C16DFA6CA1D67B61A65677"
            "EB1719CD125D0EF0DB8802FB76CFC17577BCB2510AE294E1BF8A9173A2B85C16"
            "A6B508C98F2D770B7F3DE48D9E720C53E263680B57E7109410015745570652FD"
        ),
        2048,
    )
    .unwrap();
    assert!(bool::from(key.primes()[1].ct_eq(&expected_prime)));
}

#[test]
fn decode_rsa2048_pub_der() {
    let key = RsaPublicKey::from_pkcs1_der(RSA_2048_PUB_DER).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa2048-pub.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "B6C42C515F10A6AAF282C63EDBE24243A170F3FA2633BD4833637F47CA4F6F36"
            "E03A5D29EFC3191AC80F390D874B39E30F414FCEC1FCA0ED81E547EDC2CD382C"
            "76F61C9018973DB9FA537972A7C701F6B77E0982DFC15FC01927EE5E7CD94B4F"
            "599FF07013A7C8281BDF22DCBC9AD7CABB7C4311C982F58EDB7213AD4558B332"
            "266D743AED8192D1884CADB8B14739A8DADA66DC970806D9C7AC450CB13D0D7C"
            "575FB198534FC61BC41BC0F0574E0E0130C7BBBFBDFDC9F6A6E2E3E2AFF1CBEA"
            "C89BA57884528D55CFB08327A1E8C89F4E003CF2888E933241D9D695BCBBACDC"
            "90B44E3E095FA37058EA25B13F5E295CBEAC6DE838AB8C50AF61E298975B872F"

        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
}

#[test]
fn decode_rsa4096_pub_der() {
    let key = RsaPublicKey::from_pkcs1_der(RSA_4096_PUB_DER).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa4096-pub.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "A7A74572811EA2617E49E85BD730DDE30F103F7D88EE3F765E540D3DD993BBB0"
            "BA140002859D0B40897436637F58B828EA74DF8321634077F99D4AA2D54CA375"
            "852EF597661D3713CE1EF3B4FD6A8E220238E467668A2C7EE3861D2212AE6A1E"
            "BDDFA88B62DF10F6BCF79EFF4AC298FB2563DF1B8764381AF9B1FB0CCD085E02"
            "6B0AD9F6721A235177D0396B48754AD4A75242250A873BF2F6E7EE3C75DD613E"
            "365BA4F3210A6CC66B90A2FA3F762CA6884087B6BF8161EB144819F0F572F21F"
            "6C8E273E70D45A365B8B2819CE734613CC23B01329A17901F17078403861F54C"
            "52A051E2A58C75C2D9D80091BB9808A106C1F7ECB4034E15058BEEC725C5F919"
            "D62EAA234B62628D346C60BB919E70851DAB38571E6F0ED7634129F994EA368F"
            "EE7373DFDEC04445EBCA47FA20ED1540A860C948BABC98DA591CA1DE2E2E2554"
            "0EF9B7CB353F60213B814A45D359EFA9B811EEFF08C65993BF8A85C2BFEAAA7E"
            "D5E6B43E18AE604464CE5F96150136E7D09F8B24FAD43D7870118CFA7BC24875"
            "506EBBC321B977E0861AEA50128620121F0B394A9CDD0A42411A1350C0770D97"
            "5D71B00A90436240C967A0C3A5C20A0F6DE77F3F2CAFDA94ED0143C1F6E34F73"
            "E0CAC279EEEB7C637723A2B026C82802E1A4AEBAA8846DF98E7919498773E0D4"
            "F319956F4DE3AAD00EFB9A147D66B3AC1A01D35B2CFB48D400B0E7A80DC97551"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
}

#[test]
fn encode_rsa2048_priv_der() {
    let key = RsaPrivateKey::from_pkcs1_der(RSA_2048_PRIV_DER).unwrap();
    let der = key.to_pkcs1_der().unwrap();
    assert_eq!(der.as_bytes(), RSA_2048_PRIV_DER)
}

#[test]
fn encode_rsa4096_priv_der() {
    let key = RsaPrivateKey::from_pkcs1_der(RSA_4096_PRIV_DER).unwrap();
    let der = key.to_pkcs1_der().unwrap();
    assert_eq!(der.as_bytes(), RSA_4096_PRIV_DER)
}

#[test]
fn encode_rsa2048_pub_der() {
    let key = RsaPublicKey::from_pkcs1_der(RSA_2048_PUB_DER).unwrap();
    let der = key.to_pkcs1_der().unwrap();
    assert_eq!(der.as_ref(), RSA_2048_PUB_DER)
}

#[test]
fn encode_rsa4096_pub_der() {
    let key = RsaPublicKey::from_pkcs1_der(RSA_4096_PUB_DER).unwrap();
    let der = key.to_pkcs1_der().unwrap();
    assert_eq!(der.as_ref(), RSA_4096_PUB_DER)
}

#[test]
#[cfg(feature = "encoding")]
fn decode_rsa2048_priv_pem() {
    let key = RsaPrivateKey::from_pkcs1_pem(RSA_2048_PRIV_PEM).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa2048-priv.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "B6C42C515F10A6AAF282C63EDBE24243A170F3FA2633BD4833637F47CA4F6F36"
            "E03A5D29EFC3191AC80F390D874B39E30F414FCEC1FCA0ED81E547EDC2CD382C"
            "76F61C9018973DB9FA537972A7C701F6B77E0982DFC15FC01927EE5E7CD94B4F"
            "599FF07013A7C8281BDF22DCBC9AD7CABB7C4311C982F58EDB7213AD4558B332"
            "266D743AED8192D1884CADB8B14739A8DADA66DC970806D9C7AC450CB13D0D7C"
            "575FB198534FC61BC41BC0F0574E0E0130C7BBBFBDFDC9F6A6E2E3E2AFF1CBEA"
            "C89BA57884528D55CFB08327A1E8C89F4E003CF2888E933241D9D695BCBBACDC"
            "90B44E3E095FA37058EA25B13F5E295CBEAC6DE838AB8C50AF61E298975B872F"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
    assert_eq!(
        &key.d().to_be_bytes()[..],
        &hex!(
            "7ECC8362C0EDB0741164215E22F74AB9D91BA06900700CF63690E5114D8EE6BD"
            "CFBB2E3F9614692A677A083F168A5E52E5968E6407B9D97C6E0E4064F82DA0B7"
            "58A14F17B9B7D41F5F48E28D6551704F56E69E7AA9FA630FC76428C06D25E455"
            "DCFC55B7AC2B4F76643FDED3FE15FF78ABB27E65ACC4AAD0BDF6DB27EF60A691"
            "0C5C4A085ED43275AB19C1D997A32C6EFFCE7DF2D1935F6E601EEDE161A12B5C"
            "C27CA21F81D2C99C3D1EA08E90E3053AB09BEFA724DEF0D0C3A3C1E9740C0D9F"
            "76126A149EC0AA7D8078205484254D951DB07C4CF91FB6454C096588FD5924DB"
            "ABEB359CA2025268D004F9D66EB3D6F7ADC1139BAD40F16DDE639E11647376C1"
        )
    );
    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "DCC061242D4E92AFAEE72AC513CA65B9F77036F9BD7E0E6E61461A7EF7654225"
            "EC153C7E5C31A6157A6E5A13FF6E178E8758C1CB33D9D6BBE3179EF18998E422"
            "ECDCBED78F4ECFDBE5F4FCD8AEC2C9D0DC86473CA9BD16D9D238D21FB5DDEFBE"
            "B143CA61D0BD6AA8D91F33A097790E9640DBC91085DC5F26343BA3138F6B2D67"
        ),
        1024,
    )
    .unwrap();
    assert!(bool::from(key.primes()[0].ct_eq(&expected_prime)));

    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "D3F314757E40E954836F92BE24236AF2F0DA04A34653C180AF67E960086D93FD"
            "E65CB23EFD9D09374762F5981E361849AF68CDD75394FF6A4E06EB69B209E422"
            "8DB2DFA70E40F7F9750A528176647B788D0E5777A2CB8B22E3CD267FF70B4F3B"
            "02D3AAFB0E18C590A564B03188B0AA5FC48156B07622214243BD1227EFA7F2F9"
        ),
        1024,
    )
    .unwrap();
    assert!(bool::from(key.primes()[1].ct_eq(&expected_prime)));
}

#[test]
#[cfg(feature = "encoding")]
fn decode_rsa4096_priv_pem() {
    let key = RsaPrivateKey::from_pkcs1_pem(RSA_4096_PRIV_PEM).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa4096-priv.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "A7A74572811EA2617E49E85BD730DDE30F103F7D88EE3F765E540D3DD993BBB0"
            "BA140002859D0B40897436637F58B828EA74DF8321634077F99D4AA2D54CA375"
            "852EF597661D3713CE1EF3B4FD6A8E220238E467668A2C7EE3861D2212AE6A1E"
            "BDDFA88B62DF10F6BCF79EFF4AC298FB2563DF1B8764381AF9B1FB0CCD085E02"
            "6B0AD9F6721A235177D0396B48754AD4A75242250A873BF2F6E7EE3C75DD613E"
            "365BA4F3210A6CC66B90A2FA3F762CA6884087B6BF8161EB144819F0F572F21F"
            "6C8E273E70D45A365B8B2819CE734613CC23B01329A17901F17078403861F54C"
            "52A051E2A58C75C2D9D80091BB9808A106C1F7ECB4034E15058BEEC725C5F919"
            "D62EAA234B62628D346C60BB919E70851DAB38571E6F0ED7634129F994EA368F"
            "EE7373DFDEC04445EBCA47FA20ED1540A860C948BABC98DA591CA1DE2E2E2554"
            "0EF9B7CB353F60213B814A45D359EFA9B811EEFF08C65993BF8A85C2BFEAAA7E"
            "D5E6B43E18AE604464CE5F96150136E7D09F8B24FAD43D7870118CFA7BC24875"
            "506EBBC321B977E0861AEA50128620121F0B394A9CDD0A42411A1350C0770D97"
            "5D71B00A90436240C967A0C3A5C20A0F6DE77F3F2CAFDA94ED0143C1F6E34F73"
            "E0CAC279EEEB7C637723A2B026C82802E1A4AEBAA8846DF98E7919498773E0D4"
            "F319956F4DE3AAD00EFB9A147D66B3AC1A01D35B2CFB48D400B0E7A80DC97551"

        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
    assert_eq!(
        &key.d().to_be_bytes()[..],
        &hex!(
            "9FE3097B2322B90FAB6606C017A095EBE640C39C100BCEE02F238FA14DAFF38E"
            "9E57568F1127ED4436126B904631B127EC395BB3EE127EB82C88D2562A7FB55F"
            "ED8D1450B7E4E2D2F37F5742636FCC6F289963522D5B5706082CADFA01C0EE99"
            "B4D0E9274D3A992E06974CBE01694686356962AC1959FD9BD447E5B9968C0543"
            "DF1BF134742AF345CDB2FA1F9371B0D4CF61C68D16D653D8E999D4FD3A16CF97"
            "8A35AA40E860CDCE09655DD8B4CF19D4141B1E92AD5E51A8E4A5C27FA745611D"
            "90E49D0E9282222AB6F126643E1C77578816FCE3B98F321D2549F294A470DF84"
            "53446BF36F985DF25ED8FDE9FDF3073FB27727DF48E9E1FC7056BC78965090B7"
            "850126406462C8253051EF84E34EE3C3CEB8F96C658C38BE45558D2F64E29D22"
            "3350555FC1EFA28EC1F4AFB5BA4080F09A86CDC3538C1AD7C972E6D7A3612E68"
            "45BA9AFBDF19F09060D1A779DE9635E2D2F8E0C510BA24C6C44B30C9BDFAF85B"
            "E917AEC5D43AFAB1AA3ADD33CC83DA93CAC69218F6A36EB47F199D5424C95FD9"
            "ED7B1E8BE2AEAA6433B227241316C20EE792650CEB48BFD634446B19D286B4EA"
            "1722498DA1A36973210EC3824751A5808D9AAEF59C449E19A5077CFECA126BD9"
            "A8DD4996561D4E27B3609FF82C5B1B21E627845D44961B33B875D5C4FA9FF357"
            "EF6BE3364969E1337C91B29A07B9A913CDE40CE2D5530C900E73751685E65431"
        )
    );
    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "D0213A79425B665B719118448893EC3275600F63DBF85B77F4E8E99EF302F6E8"
            "2596048F6DCA772DE6BBF1124DB84B0AFE61B03A8604AB0079ED53F3304797AD"
            "01B38C44FE27A5A45E378483A804B56A4A967F48F01A866E721E67E4C9A1048A"
            "F68927FAA43D6A85D93E7BF7074DBA797563FCABE12309B76653C6DB614DC231"
            "CC556D9F25AC4841A02D31CDF3015B212307F9D0C79FEB5D3956CE53CC8FA165"
            "1BE60761F19F74672489EAF9F215409F39956E77A82183F1F72BB2FEDDF1B9FB"
            "FC4AD89EA445809DDBD5BD595277990C0BE9366FBB2ECF7B057CC1C3DC8FB77B"
            "F8456D07BBC95B3C1815F48E62B81468C3D4D9D96C0F48DAB04993BE8D91EDE5"
        ),
        2048,
    )
    .unwrap();
    assert!(bool::from(key.primes()[0].ct_eq(&expected_prime)));

    let expected_prime = BoxedUint::from_be_slice(
        &hex!(
            "CE36C6810522ABE5D6465F36EB137DA3B9EA4A5F1D27C6614729EB8E5E2E5CB8"
            "8E3EF1A473A21944B66557B3DC2CE462E4BF3446CB4990037E5672B1705CBAE8"
            "1B65BAF967A266DC18EFE80F4DBBFE1A59063205CE2943CADF421CCE74AF7063"
            "FD1A83AF3C39AF84525F59BDC1FF54815F52AFD1E8D4862B2C3654F6CFA83DC0"
            "8E2A9D52B9F833C646AF7694467DFC5F7D7AD7B441895FCB7FFBED526324B015"
            "4A15823F5107C89548EDDCB61DA5308C6CC834D4A0C16DFA6CA1D67B61A65677"
            "EB1719CD125D0EF0DB8802FB76CFC17577BCB2510AE294E1BF8A9173A2B85C16"
            "A6B508C98F2D770B7F3DE48D9E720C53E263680B57E7109410015745570652FD"
        ),
        2048,
    )
    .unwrap();
    assert!(bool::from(key.primes()[1].ct_eq(&expected_prime)));
}

#[test]
#[cfg(feature = "encoding")]
fn decode_rsa2048_pub_pem() {
    let key = RsaPublicKey::from_pkcs1_pem(RSA_2048_PUB_PEM).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa2048-pub.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "B6C42C515F10A6AAF282C63EDBE24243A170F3FA2633BD4833637F47CA4F6F36"
            "E03A5D29EFC3191AC80F390D874B39E30F414FCEC1FCA0ED81E547EDC2CD382C"
            "76F61C9018973DB9FA537972A7C701F6B77E0982DFC15FC01927EE5E7CD94B4F"
            "599FF07013A7C8281BDF22DCBC9AD7CABB7C4311C982F58EDB7213AD4558B332"
            "266D743AED8192D1884CADB8B14739A8DADA66DC970806D9C7AC450CB13D0D7C"
            "575FB198534FC61BC41BC0F0574E0E0130C7BBBFBDFDC9F6A6E2E3E2AFF1CBEA"
            "C89BA57884528D55CFB08327A1E8C89F4E003CF2888E933241D9D695BCBBACDC"
            "90B44E3E095FA37058EA25B13F5E295CBEAC6DE838AB8C50AF61E298975B872F"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
}

#[test]
#[cfg(feature = "encoding")]
fn decode_rsa4096_pub_pem() {
    let key = RsaPublicKey::from_pkcs1_pem(RSA_4096_PUB_PEM).unwrap();

    // Extracted using:
    // $ openssl asn1parse -in tests/examples/pkcs1/rsa4096-pub.pem
    assert_eq!(
        &key.n().to_be_bytes()[..],
        &hex!(
            "A7A74572811EA2617E49E85BD730DDE30F103F7D88EE3F765E540D3DD993BBB0"
            "BA140002859D0B40897436637F58B828EA74DF8321634077F99D4AA2D54CA375"
            "852EF597661D3713CE1EF3B4FD6A8E220238E467668A2C7EE3861D2212AE6A1E"
            "BDDFA88B62DF10F6BCF79EFF4AC298FB2563DF1B8764381AF9B1FB0CCD085E02"
            "6B0AD9F6721A235177D0396B48754AD4A75242250A873BF2F6E7EE3C75DD613E"
            "365BA4F3210A6CC66B90A2FA3F762CA6884087B6BF8161EB144819F0F572F21F"
            "6C8E273E70D45A365B8B2819CE734613CC23B01329A17901F17078403861F54C"
            "52A051E2A58C75C2D9D80091BB9808A106C1F7ECB4034E15058BEEC725C5F919"
            "D62EAA234B62628D346C60BB919E70851DAB38571E6F0ED7634129F994EA368F"
            "EE7373DFDEC04445EBCA47FA20ED1540A860C948BABC98DA591CA1DE2E2E2554"
            "0EF9B7CB353F60213B814A45D359EFA9B811EEFF08C65993BF8A85C2BFEAAA7E"
            "D5E6B43E18AE604464CE5F96150136E7D09F8B24FAD43D7870118CFA7BC24875"
            "506EBBC321B977E0861AEA50128620121F0B394A9CDD0A42411A1350C0770D97"
            "5D71B00A90436240C967A0C3A5C20A0F6DE77F3F2CAFDA94ED0143C1F6E34F73"
            "E0CAC279EEEB7C637723A2B026C82802E1A4AEBAA8846DF98E7919498773E0D4"
            "F319956F4DE3AAD00EFB9A147D66B3AC1A01D35B2CFB48D400B0E7A80DC97551"
        )
    );
    let expected_e = BoxedUint::from_be_slice(&hex!("010001"), 128).unwrap();
    assert!(bool::from(key.e().ct_eq(&expected_e)));
}

#[test]
#[cfg(feature = "encoding")]
fn encode_rsa2048_priv_pem() {
    let key = RsaPrivateKey::from_pkcs1_pem(RSA_2048_PRIV_PEM).unwrap();
    let pem = key.to_pkcs1_pem(LineEnding::LF).unwrap();
    assert_eq!(&*pem, RSA_2048_PRIV_PEM)
}

#[test]
#[cfg(feature = "encoding")]
fn encode_rsa4096_priv_pem() {
    let key = RsaPrivateKey::from_pkcs1_pem(RSA_4096_PRIV_PEM).unwrap();
    let pem = key.to_pkcs1_pem(LineEnding::LF).unwrap();
    assert_eq!(&*pem, RSA_4096_PRIV_PEM)
}

#[test]
#[cfg(feature = "encoding")]
fn encode_rsa2048_pub_pem() {
    let key = RsaPublicKey::from_pkcs1_pem(RSA_2048_PUB_PEM).unwrap();
    let pem = key.to_pkcs1_pem(LineEnding::LF).unwrap();
    assert_eq!(&*pem, RSA_2048_PUB_PEM)
}

#[test]
#[cfg(feature = "encoding")]
fn encode_rsa4096_pub_pem() {
    let key = RsaPublicKey::from_pkcs1_pem(RSA_4096_PUB_PEM).unwrap();
    let pem = key.to_pkcs1_pem(LineEnding::LF).unwrap();
    assert_eq!(&*pem, RSA_4096_PUB_PEM)
}
