#[derive(Debug)]
/// Errors generated by the PQCP API.
pub enum PQCPError {
    /// An error during key generation.
    KeyGeneration,
    /// An error during encapsulation.
    Encapsulation,
    /// An error during decapsulation.
    Decapsulation,
    /// An error during public key deserialization.
    InvalidPublicKey,
    /// An error during private key deserialization.
    InvalidPrivateKey,
}

/// This macro implements the "packed" PQCP common API.
///
/// - `$use_mod` must be the path to the module that will be generated by
///   this macro. This way, the doc test can `use` the required constants.
///   (Unfortunate to have to include this as a string, but doing it in
///   another way inserts an ugly linebreak in the doc output,
///   cf. https://stackoverflow.com/questions/60905060/prevent-line-break-in-doc-test)
/// - `$trait_implementer` must be a type implementing the
///   `libcrux_traits::kem::arrayref::Kem` trait.
/// - `$variant` must be a string containing the numeric parameter set
///   identifier with a space in front and one after, e.g. " 512 ". It is
///   used only for generating documentation.
macro_rules! pqcp_api {
    ($use_mod:literal, $trait_implementer:ty, $variant:literal) => {
        /// Common APIs shared between PQCP implementations.
        ///
        /// ```
        /// use rand::RngCore;
        #[doc = $use_mod]
        ///
        /// // Key Generation
        /// let mut pk = [0u8; PK_LEN];
        /// let mut sk = [0u8; SK_LEN];
        /// let mut coins = [0u8; KEYGEN_SEED_LEN];
        ///
        /// let mut rng = rand::rng();
        /// rng.fill_bytes(&mut coins);
        ///
        /// crypto_kem_keypair_derand(&mut pk, &mut sk, coins).unwrap();
        ///
        /// // Encapsulation
        /// let mut encaps_coins = [0u8; ENCAPS_SEED_LEN];
        /// let mut ct = [0u8; CT_LEN];
        /// let mut ss_encaps = [0u8; SS_LEN];
        ///
        /// rng.fill_bytes(&mut encaps_coins);
        ///
        /// crypto_kem_enc_derand(&mut ct, &mut ss_encaps, &pk, encaps_coins).unwrap();
        ///
        /// // Decapsulation
        /// let mut ss_decaps = [0u8; SS_LEN];
        ///
        /// crypto_kem_dec(&mut ss_decaps, &ct, &sk).unwrap();
        ///
        /// assert_eq!(ss_encaps, ss_decaps);
        /// ```
        #[cfg(all(not(eurydice), feature = "pqcp"))]
        pub mod pqcp {
            use crate::pqcp::PQCPError;
            use libcrux_traits::kem::arrayref::Kem;

            #[cfg(feature = "rand")]
            use ::rand::CryptoRng;

            use super::*;

            /// Length of the public key in bytes.
            pub const PK_LEN: usize = CPA_PKE_PUBLIC_KEY_SIZE;
            /// Length of the private key in bytes.
            pub const SK_LEN: usize = SECRET_KEY_SIZE;
            /// Length of the key generation randomness in bytes.
            pub const KEYGEN_SEED_LEN: usize = KEY_GENERATION_SEED_SIZE;
            /// Length of the key generation randomness in bytes.
            pub const ENCAPS_SEED_LEN: usize = ENCAPS_SEED_SIZE;
            /// Length of the ciphertext in bytes.
            pub const CT_LEN: usize = CPA_PKE_CIPHERTEXT_SIZE;
            /// Length of the shared secret in bytes.
            pub const SS_LEN: usize = SHARED_SECRET_SIZE;

            // NIST APIs

            /// Generate ML-KEM
            #[doc = $variant]
            /// Key Pair (randomness generated externally)
            pub fn crypto_kem_keypair_derand(
                pk: &mut [u8; PK_LEN],
                sk: &mut [u8; SK_LEN],
                coins: [u8; KEYGEN_SEED_LEN],
            ) -> Result<(), PQCPError> {
                <$trait_implementer>::keygen(pk, sk, &coins).map_err(|_| PQCPError::KeyGeneration)
            }

            #[doc = "Generate ML-KEM "]
            #[doc = $variant]
            #[doc = " Key Pair (randomness generated internally)"]
            #[cfg(feature = "rand")]
            pub fn crypto_kem_keypair(
                pk: &mut [u8; PK_LEN],
                sk: &mut [u8; SK_LEN],
                rng: &mut impl CryptoRng,
            ) -> Result<(), PQCPError> {
                let mut rand = [0u8; KEYGEN_SEED_LEN];
                rng.fill_bytes(&mut rand);
                <$trait_implementer>::keygen(pk, sk, &rand).map_err(|_| PQCPError::KeyGeneration)
            }

            /// Encapsulate ML-KEM
            #[doc = $variant]
            /// shared secret (randomness generated externally)
            ///
            /// Does not perform public key validation.
            pub fn crypto_kem_enc_derand(
                ct: &mut [u8; CT_LEN],
                ss: &mut [u8; SS_LEN],
                pk: &[u8; PK_LEN],
                coins: [u8; ENCAPS_SEED_LEN],
            ) -> Result<(), PQCPError> {
                <$trait_implementer>::encaps(ct, ss, pk, &coins)
                    .map_err(|_| PQCPError::Encapsulation)
            }

            #[cfg(feature = "rand")]
            /// Encapsulate ML-KEM
            #[doc = $variant]
            /// shared secret (randomness generated internally)
            ///
            /// Does not perform public key validation.
            pub fn crypto_kem_enc(
                ct: &mut [u8; CT_LEN],
                ss: &mut [u8; SS_LEN],
                pk: &[u8; PK_LEN],
                rng: &mut impl CryptoRng,
            ) -> Result<(), PQCPError> {
                let mut rand = [0u8; ENCAPS_SEED_LEN];
                rng.fill_bytes(&mut rand);
                <$trait_implementer>::encaps(ct, ss, pk, &rand)
                    .map_err(|_| PQCPError::Encapsulation)
            }

            /// Decapsulate ML-KEM
            #[doc = $variant]
            /// shared secret
            ///
            /// Does not perform private key validation.
            pub fn crypto_kem_dec(
                ss: &mut [u8; SS_LEN],
                ct: &[u8; CT_LEN],
                sk: &[u8; SK_LEN],
            ) -> Result<(), PQCPError> {
                <$trait_implementer>::decaps(ss, ct, sk).map_err(|_| PQCPError::Decapsulation)
            }
        }
    };
}

/// This macro implements the "unpacked" PQCP common API.
///
/// - `$sk_type_unpacked` should be the (platform dependent) unpacked
///   key pair type.
/// - `$pk_type_unpacked` should be the (platform dependent) unpacked
///   public key type.
/// - `$sk_type_packed` should be the packed private key type.
/// - `$pk_type_packed` should be the packed public key type.
/// - `$ct_type` should be the ciphertext type.
/// - `$variant` should be the numeric parameter set identifier with a
///   space in front and one after, e.g. " 512 ". It is used only for
///   generating documentation.
macro_rules! pqcp_unpacked_api {
    ($sk_type_unpacked:ty, $pk_type_unpacked:ty, $sk_type_packed:ty, $pk_type_packed:ty, $ct_type:ty, $variant:literal) => {
        /// Common APIs shared between PQCP implementations.
        #[cfg(all(not(eurydice), feature = "pqcp"))]
        pub mod pqcp {
            pub use super::super::pqcp::{
                CT_LEN, ENCAPS_SEED_LEN, KEYGEN_SEED_LEN, PK_LEN, SK_LEN, SS_LEN,
            };
            use super::*;
            use crate::pqcp::PQCPError;

            /// Generate ML-KEM
            #[doc = $variant]
            /// Key Pair in \"unpacked\" form (randomness generated externally)
            pub fn crypto_kem_keypair_derand_struct(
                key_pair: &mut $sk_type_unpacked,
                coins: [u8; KEYGEN_SEED_LEN],
            ) {
                generate_key_pair_mut(coins, key_pair)
            }

            #[cfg(feature = "rand")]
            /// Generate ML-KEM
            #[doc = $variant]
            /// Key Pair in \"unpacked\" form (randomness generated internally)
            pub fn crypto_kem_keypair_struct(
                key_pair: &mut $sk_type_unpacked,
                rng: &mut impl ::rand::CryptoRng,
            ) {
                let mut randomness = [0u8; KEYGEN_SEED_LEN];
                rng.fill_bytes(&mut randomness);
                generate_key_pair_mut(randomness, key_pair)
            }

            /// Serialize an unpacked public key
            pub fn crypto_kem_marshal_pk(
                pk: &mut [u8; PK_LEN],
                pks: &$pk_type_unpacked,
            ) -> Result<(), PQCPError> {
                let mut packed_key = <$pk_type_packed>::default();

                pks.serialized_mut::<PK_LEN>(&mut packed_key);

                pk.copy_from_slice(packed_key.as_slice());
                Ok(())
            }

            /// Serialize an unpacked private key.
            pub fn crypto_kem_marshal_sk(
                sk: &mut [u8; SK_LEN],
                sks: &$sk_type_unpacked,
            ) -> Result<(), PQCPError> {
                let mut packed_key = <$sk_type_packed>::default();

                key_pair_serialized_private_key_mut(sks, &mut packed_key);

                sk.copy_from_slice(packed_key.as_slice());
                Ok(())
            }

            /// Validate and deserialize an unpacked public key.
            pub fn crypto_kem_parse_pk(
                pk: &[u8; PK_LEN],
                pks: &mut $pk_type_unpacked,
            ) -> Result<(), PQCPError> {
                let packed_key = <$pk_type_packed>::from(pk);

                if !validate_public_key(&packed_key) {
                    Err(PQCPError::InvalidPublicKey)
                } else {
                    unpacked_public_key(&packed_key, pks);
                    Ok(())
                }
            }

            /// Validate and deserialize an unpacked private key.
            pub fn crypto_kem_parse_sk(
                sk: &[u8; SK_LEN],
                sks: &mut $sk_type_unpacked,
            ) -> Result<(), PQCPError> {
                let packed_key = <$sk_type_packed>::from(sk);

                if !p::validate_private_key_only::<RANK, SK_LEN>(&packed_key) {
                    Err(PQCPError::InvalidPrivateKey)
                } else {
                    key_pair_from_private_mut(&packed_key, sks);
                    Ok(())
                }
            }

            /// Encapsulate using an unpacked public key (randomness generated externally)
            ///
            /// Does not perform public key validation, since
            /// the public key must have been validated on
            /// deserialization.
            pub fn crypto_kem_enc_derand_struct(
                ct: &mut [u8; CT_LEN],
                ss: &mut [u8; SS_LEN],
                pk: &$pk_type_unpacked,
                coins: [u8; ENCAPS_SEED_LEN],
            ) -> Result<(), PQCPError> {
                let (ciphertext, shared_secret) = super::encapsulate(pk, coins);
                ct.copy_from_slice(ciphertext.as_slice());
                ss.copy_from_slice(shared_secret.as_slice());
                Ok(())
            }

            #[cfg(feature = "rand")]
            /// Encapsulate using an unpacked public key (randomness generated internally)
            ///
            /// Does not perform public key validation, since
            /// the public key must have been validated on
            /// deserialization.
            pub fn crypto_kem_enc_struct(
                ct: &mut [u8; CT_LEN],
                ss: &mut [u8; SS_LEN],
                pk: &$pk_type_unpacked,
                rng: &mut impl ::rand::CryptoRng,
            ) -> Result<(), PQCPError> {
                let mut randomness = [0u8; ENCAPS_SEED_LEN];
                rng.fill_bytes(&mut randomness);
                let (ciphertext, shared_secret) = super::encapsulate(pk, randomness);
                ct.copy_from_slice(ciphertext.as_slice());
                ss.copy_from_slice(shared_secret.as_slice());
                Ok(())
            }

            /// Decapsulate using an unpacked private key
            ///
            /// Does not perform key validation, since the
            /// keys must have been validated on
            /// deserialization.
            pub fn crypto_kem_dec_struct(
                ss: &mut [u8; SS_LEN],
                ct: &[u8; CT_LEN],
                sk: &$sk_type_unpacked,
            ) -> Result<(), PQCPError> {
                let ciphertext = <$ct_type>::from(ct);

                let shared_secret = super::decapsulate(sk, &ciphertext);
                ss.copy_from_slice(shared_secret.as_slice());

                Ok(())
            }

            /// Alias for `crypto_kem_keypair_derand_struct`
            pub fn crypto_kem_sk_from_seed(
                key_pair: &mut $sk_type_unpacked,
                randomness: [u8; KEYGEN_SEED_LEN],
            ) {
                crypto_kem_keypair_derand_struct(key_pair, randomness)
            }

            /// Generate an unpacked ML-KEM 512 public key from the private key
            pub fn crypto_kem_pk_from_sk(pk: &mut $pk_type_unpacked, sk: &$sk_type_unpacked) {
                *pk = sk.public_key().clone()
            }
        }
    };
}

pub(crate) use pqcp_api;
pub(crate) use pqcp_unpacked_api;
