/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* Generated by Together */

package ghidra.program.model.lang;

/**
 * Helper class for testing operand related flags in an integer.
 */
public final class OperandType {

	/**
	 * Bit set if operand refers to an address being read
	 */
	public final static int READ = 0x00000001;
	/**
	 * Bit set if operand refers to an address being written to
	 */
	public final static int WRITE = 0x00000002;

	/**
	 * Bit set if operand is an indirect reference.
	 */
	public final static int INDIRECT = 0x00000004;
	/**
	 * Bit set if operand is an immediate value.
	 */
	public final static int IMMEDIATE = 0x00000008;
	/**
	 * Bit set if operand depends on the instruction's address.
	 */
	public final static int RELATIVE = 0x00000010;
	/**
	 * Bit set if operand is implicit.
	 */
	public final static int IMPLICIT = 0x00000020;

	/**
	 * Bit set it the address referred to contains code.
	 */
	public final static int CODE = 0x00000040;
	/**
	 * Bit set if the address referred to contains data.
	 */
	public final static int DATA = 0x00000080;

	/**
	 * Bit set if the operand is a port.
	 */
	public final static int PORT = 0x00000100;
	/**
	 * Bit set if the operand is a register.
	 */
	public final static int REGISTER = 0x00000200;
	/**
	 * Bit set if the operand is a list.
	 */
	public final static int LIST = 0x00000400;
	/**
	 * Bit set if the operand is a flag.
	 */
	public final static int FLAG = 0x00000800;
	/**
	 * Bit set if the operand is text.
	 */
	public final static int TEXT = 0x00001000;

	/**
	 * Bit set if the operand is used as an address.
	 *  If this is not set, assume it is a scalar value.
	 */
	public final static int ADDRESS = 0x00002000;
	/**
	 * Bit set if the operand is a scalar value
	 */
	public final static int SCALAR = 0x00004000;
	/**
	 * Bit set if the operand is a bit value
	 */
	public final static int BIT = 0x00008000;
	/**
	 * Bit set if the operand is a byte value
	 */
	public final static int BYTE = 0x00010000;
	/**
	 * Bit set if the operand is a 2 byte value
	 */
	public final static int WORD = 0x00020000;
	/**
	 * Bit set if the operand is a 8 byte value
	 */
	public final static int QUADWORD = 0x00040000;
	/**
	 * Bit set if the operand is a signed value
	 */
	public final static int SIGNED = 0x00080000;
	/**
	 * Bit set if the operand is a float value
	 */
	public final static int FLOAT = 0x00100000;
	/**
	 * Bit set if the operand is a co-processor value
	 */
	public final static int COP = 0x00200000;
	/**
	 * Bit set if the operand is dynamically defined given some processorContext.
	 * If bit is set then the SCALAR or ADDRESS bit must be set.
	 */
	public final static int DYNAMIC = 0x00400000;

	/**
	 * check the READ flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the READ flag is set.
	 */
	public static boolean doesRead(int operandType) {
		return (operandType & READ) != 0;
	}

	/**
	 * check the WRITE flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the WRITE flag is set.
	 */
	public static boolean doesWrite(int operandType) {
		return (operandType & WRITE) != 0;
	}

	/**
	 * check the INDIRECT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the INDIRECT flag is set.
	 */
	public static boolean isIndirect(int operandType) {
		return (operandType & INDIRECT) != 0;
	}

	/**
	 * check the IMMEDIATE flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the IMMEDIATE flag is set.
	 */
	public static boolean isImmediate(int operandType) {
		return (operandType & IMMEDIATE) != 0;
	}

	/**
	 * check the RELATIVE flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the RELATIVE flag is set.
	 */
	public static boolean isRelative(int operandType) {
		return (operandType & RELATIVE) != 0;
	}

	/**
	 * check the IMPLICIT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the IMPLICIT flag is set.
	 */
	public static boolean isImplicit(int operandType) {
		return (operandType & IMPLICIT) != 0;
	}

	/**
	 * check the CODE flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the CODE flag is set.
	 */
	public static boolean isCodeReference(int operandType) {
		return (operandType & CODE) != 0;
	}

	/**
	 * check the DATA flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the DATA flag is set.
	 */
	public static boolean isDataReference(int operandType) {
		return (operandType & DATA) != 0;
	}

	/**
	 * check the PORT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the PORT flag is set.
	 */
	public static boolean isPort(int operandType) {
		return (operandType & PORT) != 0;
	}

	/**
	 * check the REGISTER flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the REGISTER flag is set.
	 */
	public static boolean isRegister(int operandType) {
		return (operandType & REGISTER) != 0;
	}

	/**
	 * check the LIST flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the LIST flag is set.
	 */
	public static boolean isList(int operandType) {
		return (operandType & LIST) != 0;
	}

	/**
	 * check the CONDITION FLAG flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the CONDITION flag is set.
	 */
	public static boolean isFlag(int operandType) {
		return (operandType & FLAG) != 0;
	}

	/**
	 * check the TEXT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the TEXT flag is set.
	 */
	public static boolean isText(int operandType) {
		return (operandType & TEXT) != 0;
	}

	/**
	 * check ADDRESS flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the ADDRESS flag is set
	 */
	public static boolean isAddress(int operandType) {
		return (operandType & ADDRESS) != 0;
	}

	/**
	 * check SCALAR flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the SCALAR flag is set
	 */
	public static boolean isScalar(int operandType) {
		return (operandType & SCALAR) != 0;
	}

	/**
	 * check the BIT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the BIT flag is set.
	 */
	public static boolean isBit(int operandType) {
		return (operandType & BIT) != 0;
	}

	/**
	 * check the BYTE flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the BYTE flag is set.
	 */
	public static boolean isByte(int operandType) {
		return (operandType & BYTE) != 0;
	}

	/**
	 * check the WORD flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the WORD flag is set.
	 */
	public static boolean isWord(int operandType) {
		return (operandType & WORD) != 0;
	}

	/**
	 * check the QUADWORD flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the QUADWORD flag is set.
	 */
	public static boolean isQuadWord(int operandType) {
		return (operandType & QUADWORD) != 0;
	}

	/**
	 * check the SIGNED flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the SIGNED flag is set.
	 */
	public static boolean isSigned(int operandType) {
		return (operandType & SIGNED) != 0;
	}

	/**
	 * check the FLOAT flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the FLOAT flag is set.
	 */
	public static boolean isFloat(int operandType) {
		return (operandType & FLOAT) != 0;
	}

	/**
	 * check the COPROCESSOR flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the COPROCESSOR flag is set.
	 */
	public static boolean isCoProcessor(int operandType) {
		return (operandType & COP) != 0;
	}

	/**
	 * check the DYNAMIC flag.
	 * @param operandType the bit field to examine.
	 *
	 * @return true if the DYNAMIC flag is set.
	 */
	public static boolean isDynamic(int operandType) {
		return (operandType & DYNAMIC) != 0;
	}

	/**
	 * check if both a scalar and an address
	 * @param operandType the bit field to examine.
	 * @return true if it is both a scalar and an address
	 */
	public static boolean isScalarAsAddress(int operandType) {
		return (isAddress(operandType) && isScalar(operandType));
	}

	/**
	 * returns a string representation of the given operandType
	 * @param operandType the operandType bits
	 * @return the string rep
	 */
	public static String toString(int operandType) {
		StringBuffer buf = new StringBuffer();

		//*******************************

		if (isAddress(operandType)) {
			append(buf, "ADDR");
		}

		if (isScalar(operandType)) {
			append(buf, "SCAL");
		}

		if (isPort(operandType)) {
			append(buf, "PORT");
		}

		if (isRegister(operandType)) {
			append(buf, "REG ");
		}

		if (isList(operandType)) {
			append(buf, "LIST");
		}

		if (isFlag(operandType)) {
			append(buf, "FLAG");
		}

		if (isText(operandType)) {
			append(buf, "TEXT");
		}

		//*******************************

		if (isCodeReference(operandType)) {
			append(buf, "CODE");
		}

		if (isDataReference(operandType)) {
			append(buf, "DATA");
		}

		//*******************************

		if (isBit(operandType)) {
			append(buf, "BIT ");
		}

		if (isByte(operandType)) {
			append(buf, "BYTE");
		}

		if (isWord(operandType)) {
			append(buf, "WORD");
		}

		if (isQuadWord(operandType)) {
			append(buf, "QUAD");
		}

		if (isSigned(operandType)) {
			append(buf, "SIGN");
		}
		if (isFloat(operandType)) {
			append(buf, "FLT ");
		}

		//*******************************

		if (isIndirect(operandType)) {
			append(buf, "IND ");
		}

		if (isImmediate(operandType)) {
			append(buf, "IMM ");
		}

		if (isRelative(operandType)) {
			append(buf, "REL ");
		}

		if (isImplicit(operandType)) {
			append(buf, "IMPL");
		}

		//*******************************

		if (doesRead(operandType)) {
			append(buf, "READ");
		}

		if (doesWrite(operandType)) {
			append(buf, "WRTE");
		}

		//*******************************

		if (isCoProcessor(operandType)) {
			append(buf, "COP ");
		}

		if (isDynamic(operandType)) {
			append(buf, "DYN ");
		}

		return buf.toString();
	}

	private static void append(StringBuffer buf, String str) {
		if (buf.length() > 0) {
			buf.append(" | ");
		}
		buf.append(str);
	}
}
