/*
 * Decompiled with CFR 0.152.
 */
package org.sleuthkit.datamodel;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.sleuthkit.datamodel.CaseDbConnectionInfo;
import org.sleuthkit.datamodel.CaseDbSchemaVersionNumber;
import org.sleuthkit.datamodel.SQLHelper;
import org.sleuthkit.datamodel.SleuthkitCase;
import org.sleuthkit.datamodel.SleuthkitJNI;
import org.sleuthkit.datamodel.TskCoreException;

class CaseDatabaseFactory {
    private static final Logger logger = Logger.getLogger(CaseDatabaseFactory.class.getName());
    private final SQLHelper dbQueryHelper;
    private final DbCreationHelper dbCreationHelper;
    static final String SSL_NONVERIFY_URL = "?ssl=true&sslfactory=org.postgresql.ssl.NonValidatingFactory&sslmode=require";
    static final String SSL_VERIFY_DEFAULT_URL = "?ssl=true&sslfactory=org.postgresql.ssl.DefaultJavaSSLFactory&sslmode=verify-ca";

    static String getCustomPostrgesSslVerificationUrl(String customSslValidationClassName) {
        return "?ssl=true&sslfactory=" + customSslValidationClassName + "&sslmode=verify-ca";
    }

    CaseDatabaseFactory(String dbPath) {
        this.dbQueryHelper = new SQLHelper.SQLiteHelper();
        this.dbCreationHelper = new SQLiteDbCreationHelper(this, dbPath);
    }

    CaseDatabaseFactory(String caseName, CaseDbConnectionInfo info) {
        this.dbQueryHelper = new SQLHelper.PostgreSQLHelper();
        this.dbCreationHelper = new PostgreSQLDbCreationHelper(this, caseName, info);
    }

    void createCaseDatabase() throws TskCoreException {
        this.createDatabase();
        this.initializeSchema();
    }

    private void createDatabase() throws TskCoreException {
        this.dbCreationHelper.createDatabase();
    }

    private void initializeSchema() throws TskCoreException {
        try (Connection conn = this.dbCreationHelper.getConnection();){
            this.dbCreationHelper.performPreInitialization(conn);
            this.addDbInfo(conn);
            this.addTables(conn);
            this.dbCreationHelper.performPostTableInitialization(conn);
            this.addIndexes(conn);
        }
        catch (SQLException ex) {
            throw new TskCoreException("Error initializing case database", ex);
        }
    }

    private void addDbInfo(Connection conn) throws TskCoreException {
        CaseDbSchemaVersionNumber version = SleuthkitCase.CURRENT_DB_SCHEMA_VERSION;
        long tskVersionNum = SleuthkitJNI.getSleuthkitVersion();
        try (Statement stmt = conn.createStatement();){
            stmt.execute("CREATE TABLE tsk_db_info (schema_ver INTEGER, tsk_ver INTEGER, schema_minor_ver INTEGER)");
            stmt.execute("INSERT INTO tsk_db_info (schema_ver, tsk_ver, schema_minor_ver) VALUES (" + version.getMajor() + ", " + tskVersionNum + ", " + version.getMinor() + ");");
            stmt.execute("CREATE TABLE tsk_db_info_extended (name TEXT PRIMARY KEY, value TEXT NOT NULL);");
            stmt.execute("INSERT INTO tsk_db_info_extended (name, value) VALUES ('TSK_VERSION', '" + tskVersionNum + "');");
            stmt.execute("INSERT INTO tsk_db_info_extended (name, value) VALUES ('SCHEMA_MAJOR_VERSION', '" + version.getMajor() + "');");
            stmt.execute("INSERT INTO tsk_db_info_extended (name, value) VALUES ('SCHEMA_MINOR_VERSION', '" + version.getMinor() + "');");
            stmt.execute("INSERT INTO tsk_db_info_extended (name, value) VALUES ('CREATION_SCHEMA_MAJOR_VERSION', '" + version.getMajor() + "');");
            stmt.execute("INSERT INTO tsk_db_info_extended (name, value) VALUES ('CREATION_SCHEMA_MINOR_VERSION', '" + version.getMinor() + "');");
        }
        catch (SQLException ex) {
            throw new TskCoreException("Error initializing db_info tables", ex);
        }
    }

    private void addTables(Connection conn) throws TskCoreException {
        try (Statement stmt = conn.createStatement();){
            this.createTskObjects(stmt);
            this.createHostTables(stmt);
            this.createAccountTables(stmt);
            this.createFileTables(stmt);
            this.createArtifactTables(stmt);
            this.createAnalysisResultsTables(stmt);
            this.createTagTables(stmt);
            this.createIngestTables(stmt);
            this.createEventTables(stmt);
            this.createAttributeTables(stmt);
            this.createAccountInstancesAndArtifacts(stmt);
        }
        catch (SQLException ex) {
            throw new TskCoreException("Error initializing tables", ex);
        }
    }

    private void createTskObjects(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_objects (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, par_obj_id " + this.dbQueryHelper.getBigIntType() + ", type INTEGER NOT NULL, UNIQUE (obj_id), FOREIGN KEY (par_obj_id) REFERENCES tsk_objects (obj_id) ON DELETE CASCADE)");
    }

    private void createFileTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_image_info (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, type INTEGER, ssize INTEGER, tzone TEXT, size " + this.dbQueryHelper.getBigIntType() + ", md5 TEXT, sha1 TEXT, sha256 TEXT, display_name TEXT, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_image_names (obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, name TEXT NOT NULL, sequence INTEGER NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_vs_info (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, vs_type INTEGER NOT NULL, img_offset " + this.dbQueryHelper.getBigIntType() + " NOT NULL, block_size " + this.dbQueryHelper.getBigIntType() + " NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_vs_parts (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, addr " + this.dbQueryHelper.getBigIntType() + " NOT NULL, start " + this.dbQueryHelper.getBigIntType() + " NOT NULL, length " + this.dbQueryHelper.getBigIntType() + " NOT NULL, " + this.dbQueryHelper.getVSDescColName() + " TEXT, flags INTEGER NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE);");
        stmt.execute("CREATE TABLE tsk_pool_info (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, pool_type INTEGER NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE);");
        stmt.execute("CREATE TABLE data_source_info (obj_id " + this.dbQueryHelper.getBigIntType() + " PRIMARY KEY, device_id TEXT NOT NULL, time_zone TEXT NOT NULL, acquisition_details TEXT, added_date_time " + this.dbQueryHelper.getBigIntType() + ", acquisition_tool_settings TEXT, acquisition_tool_name TEXT, acquisition_tool_version TEXT, host_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, FOREIGN KEY(host_id) REFERENCES tsk_hosts(id), FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_fs_info (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, img_offset " + this.dbQueryHelper.getBigIntType() + " NOT NULL, fs_type INTEGER NOT NULL, block_size " + this.dbQueryHelper.getBigIntType() + " NOT NULL, block_count " + this.dbQueryHelper.getBigIntType() + " NOT NULL, root_inum " + this.dbQueryHelper.getBigIntType() + " NOT NULL, first_inum " + this.dbQueryHelper.getBigIntType() + " NOT NULL, last_inum " + this.dbQueryHelper.getBigIntType() + " NOT NULL, display_name TEXT, FOREIGN KEY(data_source_obj_id) REFERENCES data_source_info(obj_id) ON DELETE CASCADE, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE file_collection_status_types (collection_status_type INTEGER PRIMARY KEY, name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE tsk_files (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, fs_obj_id " + this.dbQueryHelper.getBigIntType() + ", data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, attr_type INTEGER, attr_id INTEGER, name TEXT NOT NULL, meta_addr " + this.dbQueryHelper.getBigIntType() + ", meta_seq " + this.dbQueryHelper.getBigIntType() + ", type INTEGER, has_layout INTEGER, has_path INTEGER, dir_type INTEGER, meta_type INTEGER, dir_flags INTEGER, meta_flags INTEGER, size " + this.dbQueryHelper.getBigIntType() + ", ctime " + this.dbQueryHelper.getBigIntType() + ", crtime " + this.dbQueryHelper.getBigIntType() + ", atime " + this.dbQueryHelper.getBigIntType() + ", mtime " + this.dbQueryHelper.getBigIntType() + ", mode INTEGER, uid INTEGER, gid INTEGER, md5 TEXT, sha256 TEXT, sha1 TEXT,known INTEGER, parent_path TEXT, mime_type TEXT, extension TEXT, owner_uid TEXT DEFAULT NULL, os_account_obj_id " + this.dbQueryHelper.getBigIntType() + " DEFAULT NULL, collected INTEGER NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(fs_obj_id) REFERENCES tsk_fs_info(obj_id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES data_source_info(obj_id) ON DELETE CASCADE, FOREIGN KEY(os_account_obj_id) REFERENCES tsk_os_accounts(os_account_obj_id) ON DELETE SET NULL, FOREIGN KEY(collected) REFERENCES file_collection_status_types (collection_status_type))");
        stmt.execute("CREATE TABLE file_encoding_types (encoding_type INTEGER PRIMARY KEY, name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE tsk_files_path (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, path TEXT NOT NULL, encoding_type INTEGER NOT NULL, FOREIGN KEY(encoding_type) references file_encoding_types(encoding_type), FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_files_derived (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, derived_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, rederive TEXT, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_files_derived_method (derived_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, tool_name TEXT NOT NULL, tool_version TEXT NOT NULL, other TEXT)");
        stmt.execute("CREATE TABLE tsk_file_layout (obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, byte_start " + this.dbQueryHelper.getBigIntType() + " NOT NULL, byte_len " + this.dbQueryHelper.getBigIntType() + " NOT NULL, sequence INTEGER NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE);");
        stmt.execute("CREATE TABLE reports (obj_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, path TEXT NOT NULL, crtime INTEGER NOT NULL, src_module_name TEXT NOT NULL, report_name TEXT NOT NULL, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE);");
    }

    private void createArtifactTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE blackboard_artifact_types (artifact_type_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, type_name TEXT NOT NULL, display_name TEXT,category_type INTEGER DEFAULT 0)");
        stmt.execute("CREATE TABLE blackboard_attribute_types (attribute_type_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, type_name TEXT NOT NULL, display_name TEXT, value_type INTEGER NOT NULL)");
        stmt.execute("CREATE TABLE review_statuses (review_status_id INTEGER PRIMARY KEY, review_status_name TEXT NOT NULL, display_name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE blackboard_artifacts (artifact_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, artifact_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + ", artifact_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, review_status_id INTEGER NOT NULL, UNIQUE (artifact_obj_id),FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(artifact_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(artifact_type_id) REFERENCES blackboard_artifact_types(artifact_type_id), FOREIGN KEY(review_status_id) REFERENCES review_statuses(review_status_id))");
        stmt.execute("CREATE TABLE blackboard_attributes (artifact_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, artifact_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, source TEXT, context TEXT, attribute_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, value_type INTEGER NOT NULL, value_byte " + this.dbQueryHelper.getBlobType() + ", value_text TEXT, value_int32 INTEGER, value_int64 " + this.dbQueryHelper.getBigIntType() + ", value_double NUMERIC(20, 10), FOREIGN KEY(artifact_id) REFERENCES blackboard_artifacts(artifact_id) ON DELETE CASCADE, FOREIGN KEY(artifact_type_id) REFERENCES blackboard_artifact_types(artifact_type_id), FOREIGN KEY(attribute_type_id) REFERENCES blackboard_attribute_types(attribute_type_id))");
    }

    private void createAnalysisResultsTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_analysis_results (artifact_obj_id " + this.dbQueryHelper.getBigIntType() + " PRIMARY KEY, conclusion TEXT, significance INTEGER NOT NULL, priority INTEGER NOT NULL, configuration TEXT, justification TEXT, ignore_score INTEGER DEFAULT 0, FOREIGN KEY(artifact_obj_id) REFERENCES blackboard_artifacts(artifact_obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_aggregate_score( obj_id " + this.dbQueryHelper.getBigIntType() + " PRIMARY KEY, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + ", significance INTEGER NOT NULL, priority INTEGER NOT NULL, UNIQUE (obj_id),FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE )");
    }

    private void createTagTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_tag_sets (tag_set_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, name TEXT UNIQUE)");
        stmt.execute("CREATE TABLE tag_names (tag_name_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, display_name TEXT UNIQUE, description TEXT NOT NULL, color TEXT NOT NULL, knownStatus INTEGER NOT NULL, tag_set_id " + this.dbQueryHelper.getBigIntType() + ", rank INTEGER, FOREIGN KEY(tag_set_id) REFERENCES tsk_tag_sets(tag_set_id) ON DELETE SET NULL)");
        stmt.execute("CREATE TABLE tsk_examiners (examiner_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, login_name TEXT NOT NULL, display_name TEXT, UNIQUE(login_name))");
        stmt.execute("CREATE TABLE content_tags (tag_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, tag_name_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, comment TEXT NOT NULL, begin_byte_offset " + this.dbQueryHelper.getBigIntType() + " NOT NULL, end_byte_offset " + this.dbQueryHelper.getBigIntType() + " NOT NULL, examiner_id " + this.dbQueryHelper.getBigIntType() + ", FOREIGN KEY(examiner_id) REFERENCES tsk_examiners(examiner_id) ON DELETE CASCADE, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(tag_name_id) REFERENCES tag_names(tag_name_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE blackboard_artifact_tags (tag_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, artifact_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, tag_name_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, comment TEXT NOT NULL,  examiner_id " + this.dbQueryHelper.getBigIntType() + ", FOREIGN KEY(examiner_id) REFERENCES tsk_examiners(examiner_id) ON DELETE CASCADE, FOREIGN KEY(artifact_id) REFERENCES blackboard_artifacts(artifact_id) ON DELETE CASCADE, FOREIGN KEY(tag_name_id) REFERENCES tag_names(tag_name_id) ON DELETE CASCADE)");
    }

    private void addIndexes(Connection conn) throws TskCoreException {
        try (Statement stmt = conn.createStatement();){
            stmt.execute("CREATE INDEX parObjId ON tsk_objects(par_obj_id)");
            stmt.execute("CREATE INDEX layout_objID ON tsk_file_layout(obj_id)");
            stmt.execute("CREATE INDEX artifact_objID ON blackboard_artifacts(obj_id)");
            stmt.execute("CREATE INDEX artifact_artifact_objID ON blackboard_artifacts(artifact_obj_id)");
            stmt.execute("CREATE INDEX artifact_typeID ON blackboard_artifacts(artifact_type_id)");
            stmt.execute("CREATE INDEX attrsArtifactID ON blackboard_attributes(artifact_id)");
            stmt.execute("CREATE INDEX mime_type ON tsk_files(dir_type,mime_type,type)");
            stmt.execute("CREATE INDEX file_extension ON tsk_files(extension)");
            stmt.execute("CREATE INDEX relationships_account1 ON account_relationships(account1_id)");
            stmt.execute("CREATE INDEX relationships_account2 ON account_relationships(account2_id)");
            stmt.execute("CREATE INDEX relationships_relationship_source_obj_id ON account_relationships(relationship_source_obj_id)");
            stmt.execute("CREATE INDEX relationships_date_time ON account_relationships(date_time)");
            stmt.execute("CREATE INDEX relationships_relationship_type ON account_relationships(relationship_type)");
            stmt.execute("CREATE INDEX relationships_data_source_obj_id ON account_relationships(data_source_obj_id)");
            stmt.execute("CREATE INDEX events_data_source_obj_id ON tsk_event_descriptions(data_source_obj_id)");
            stmt.execute("CREATE INDEX events_content_obj_id ON tsk_event_descriptions(content_obj_id)");
            stmt.execute("CREATE INDEX events_artifact_id ON tsk_event_descriptions(artifact_id)");
            stmt.execute("CREATE INDEX events_sub_type_time ON tsk_events(event_type_id,  time)");
            stmt.execute("CREATE INDEX events_time ON tsk_events(time)");
            stmt.execute("CREATE INDEX score_significance_priority ON tsk_aggregate_score(significance, priority)");
            stmt.execute("CREATE INDEX score_datasource_obj_id ON tsk_aggregate_score(data_source_obj_id)");
            stmt.execute("CREATE INDEX tsk_file_attributes_obj_id ON tsk_file_attributes(obj_id)");
            stmt.execute("CREATE INDEX tsk_os_accounts_login_name_idx  ON tsk_os_accounts(login_name, db_status, realm_id)");
            stmt.execute("CREATE INDEX tsk_os_accounts_addr_idx  ON tsk_os_accounts(addr, db_status, realm_id)");
            stmt.execute("CREATE INDEX tsk_os_account_realms_realm_name_idx  ON tsk_os_account_realms(realm_name)");
            stmt.execute("CREATE INDEX tsk_os_account_realms_realm_addr_idx  ON tsk_os_account_realms(realm_addr)");
        }
        catch (SQLException ex) {
            throw new TskCoreException("Error initializing db_info tables", ex);
        }
    }

    private void createIngestTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE ingest_module_types (type_id INTEGER PRIMARY KEY, type_name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE ingest_job_status_types (type_id INTEGER PRIMARY KEY, type_name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE ingest_modules (ingest_module_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, display_name TEXT NOT NULL, unique_name TEXT UNIQUE NOT NULL, type_id INTEGER NOT NULL, version TEXT NOT NULL, FOREIGN KEY(type_id) REFERENCES ingest_module_types(type_id) ON DELETE CASCADE);");
        stmt.execute("CREATE TABLE ingest_jobs (ingest_job_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, host_name TEXT NOT NULL, start_date_time " + this.dbQueryHelper.getBigIntType() + " NOT NULL, end_date_time " + this.dbQueryHelper.getBigIntType() + " NOT NULL, status_id INTEGER NOT NULL, settings_dir TEXT, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(status_id) REFERENCES ingest_job_status_types(type_id) ON DELETE CASCADE);");
        stmt.execute("CREATE TABLE ingest_job_modules (ingest_job_id INTEGER, ingest_module_id INTEGER, pipeline_position INTEGER, PRIMARY KEY(ingest_job_id, ingest_module_id), FOREIGN KEY(ingest_job_id) REFERENCES ingest_jobs(ingest_job_id) ON DELETE CASCADE, FOREIGN KEY(ingest_module_id) REFERENCES ingest_modules(ingest_module_id) ON DELETE CASCADE);");
    }

    private void createHostTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_persons (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, name TEXT NOT NULL, UNIQUE(name)) ");
        stmt.execute("CREATE TABLE tsk_hosts (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, name TEXT NOT NULL, db_status INTEGER DEFAULT 0, person_id INTEGER, merged_into " + this.dbQueryHelper.getBigIntType() + ", FOREIGN KEY(person_id) REFERENCES tsk_persons(id) ON DELETE SET NULL, FOREIGN KEY(merged_into) REFERENCES tsk_hosts(id) ON DELETE CASCADE, UNIQUE(name)) ");
        stmt.execute("CREATE TABLE  tsk_host_addresses (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, address_type INTEGER NOT NULL, address TEXT NOT NULL, UNIQUE(address_type, address)) ");
        stmt.execute("CREATE TABLE tsk_host_address_map  (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, host_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, addr_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, source_obj_id " + this.dbQueryHelper.getBigIntType() + ", time " + this.dbQueryHelper.getBigIntType() + ", UNIQUE(host_id, addr_obj_id, time), FOREIGN KEY(host_id) REFERENCES tsk_hosts(id) ON DELETE CASCADE, FOREIGN KEY(addr_obj_id) REFERENCES tsk_host_addresses(id), FOREIGN KEY(source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE SET NULL )");
        stmt.execute("CREATE TABLE tsk_host_address_dns_ip_map (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, dns_address_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, ip_address_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, source_obj_id " + this.dbQueryHelper.getBigIntType() + ", time " + this.dbQueryHelper.getBigIntType() + ", UNIQUE(dns_address_id, ip_address_id, time), FOREIGN KEY(dns_address_id) REFERENCES tsk_host_addresses(id) ON DELETE CASCADE, FOREIGN KEY(ip_address_id) REFERENCES tsk_host_addresses(id) ON DELETE CASCADE,FOREIGN KEY(source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE SET NULL )");
        stmt.execute("CREATE TABLE  tsk_host_address_usage (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, addr_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, UNIQUE(addr_obj_id, obj_id), FOREIGN KEY(addr_obj_id) REFERENCES tsk_host_addresses(id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE )");
    }

    private void createAccountTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE account_types (account_type_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, type_name TEXT UNIQUE NOT NULL, display_name TEXT NOT NULL)");
        stmt.execute("CREATE TABLE accounts (account_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, account_type_id INTEGER NOT NULL, account_unique_identifier TEXT NOT NULL, UNIQUE(account_type_id, account_unique_identifier), FOREIGN KEY(account_type_id) REFERENCES account_types(account_type_id))");
        stmt.execute("CREATE TABLE account_relationships (relationship_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, account1_id INTEGER NOT NULL, account2_id INTEGER NOT NULL, relationship_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, date_time " + this.dbQueryHelper.getBigIntType() + ", relationship_type INTEGER NOT NULL, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, UNIQUE(account1_id, account2_id, relationship_source_obj_id), FOREIGN KEY(account1_id) REFERENCES accounts(account_id), FOREIGN KEY(account2_id) REFERENCES accounts(account_id), FOREIGN KEY(relationship_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE)");
        stmt.execute("CREATE TABLE tsk_os_account_realms (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, realm_name TEXT DEFAULT NULL, realm_addr TEXT DEFAULT NULL, realm_signature TEXT NOT NULL, scope_host_id " + this.dbQueryHelper.getBigIntType() + " DEFAULT NULL, scope_confidence INTEGER, db_status INTEGER DEFAULT 0, merged_into " + this.dbQueryHelper.getBigIntType() + " DEFAULT NULL, UNIQUE(realm_signature), FOREIGN KEY(scope_host_id) REFERENCES tsk_hosts(id) ON DELETE CASCADE,FOREIGN KEY(merged_into) REFERENCES tsk_os_account_realms(id) ON DELETE CASCADE )");
        stmt.execute("CREATE TABLE tsk_os_accounts (os_account_obj_id " + this.dbQueryHelper.getBigIntType() + " PRIMARY KEY, login_name TEXT DEFAULT NULL, full_name TEXT DEFAULT NULL, realm_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, addr TEXT DEFAULT NULL, signature TEXT NOT NULL, status INTEGER, type INTEGER, created_date " + this.dbQueryHelper.getBigIntType() + " DEFAULT NULL, db_status INTEGER DEFAULT 0, merged_into " + this.dbQueryHelper.getBigIntType() + " DEFAULT NULL, UNIQUE(signature, realm_id), FOREIGN KEY(os_account_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE, FOREIGN KEY(realm_id) REFERENCES tsk_os_account_realms(id) ON DELETE CASCADE,FOREIGN KEY(merged_into) REFERENCES tsk_os_accounts(os_account_obj_id) ON DELETE CASCADE )");
    }

    private void createAccountInstancesAndArtifacts(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_os_account_attributes (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, os_account_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, host_id " + this.dbQueryHelper.getBigIntType() + ", source_obj_id " + this.dbQueryHelper.getBigIntType() + ", attribute_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, value_type INTEGER NOT NULL, value_byte " + this.dbQueryHelper.getBlobType() + ", value_text TEXT, value_int32 INTEGER, value_int64 " + this.dbQueryHelper.getBigIntType() + ", value_double NUMERIC(20, 10), FOREIGN KEY(os_account_obj_id) REFERENCES tsk_os_accounts(os_account_obj_id) ON DELETE CASCADE, FOREIGN KEY(host_id) REFERENCES tsk_hosts(id) ON DELETE CASCADE, FOREIGN KEY(source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE SET NULL, FOREIGN KEY(attribute_type_id) REFERENCES blackboard_attribute_types(attribute_type_id))");
        stmt.execute("CREATE TABLE tsk_os_account_instances (id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, os_account_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, instance_type INTEGER NOT NULL, UNIQUE(os_account_obj_id, data_source_obj_id, instance_type), FOREIGN KEY(os_account_obj_id) REFERENCES tsk_os_accounts(os_account_obj_id) ON DELETE CASCADE, FOREIGN KEY(data_source_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE ) ");
        stmt.execute("CREATE TABLE tsk_data_artifacts ( artifact_obj_id " + this.dbQueryHelper.getBigIntType() + " PRIMARY KEY, os_account_obj_id " + this.dbQueryHelper.getBigIntType() + ", FOREIGN KEY(artifact_obj_id) REFERENCES blackboard_artifacts(artifact_obj_id) ON DELETE CASCADE, FOREIGN KEY(os_account_obj_id) REFERENCES tsk_os_accounts(os_account_obj_id) ON DELETE SET NULL) ");
    }

    private void createEventTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_event_types ( event_type_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, display_name TEXT UNIQUE NOT NULL ,  super_type_id INTEGER REFERENCES tsk_event_types(event_type_id) )");
        stmt.execute("CREATE TABLE tsk_event_descriptions (  event_description_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY,  full_description TEXT NOT NULL,  med_description TEXT,  short_description TEXT, data_source_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL,  content_obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL,  artifact_id " + this.dbQueryHelper.getBigIntType() + ",  hash_hit INTEGER NOT NULL,  tagged INTEGER NOT NULL,  FOREIGN KEY(data_source_obj_id) REFERENCES data_source_info(obj_id) ON DELETE CASCADE,  FOREIGN KEY(content_obj_id) REFERENCES tsk_objects(obj_id) ON DELETE CASCADE,  FOREIGN KEY(artifact_id) REFERENCES blackboard_artifacts(artifact_id) ON DELETE CASCADE, UNIQUE (full_description, content_obj_id, artifact_id))");
        stmt.execute("CREATE TABLE tsk_events ( event_id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY,  event_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL REFERENCES tsk_event_types(event_type_id) , event_description_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL REFERENCES tsk_event_descriptions(event_description_id) ON DELETE CASCADE , time " + this.dbQueryHelper.getBigIntType() + " NOT NULL ,  UNIQUE (event_type_id, event_description_id, time))");
    }

    private void createAttributeTables(Statement stmt) throws SQLException {
        stmt.execute("CREATE TABLE tsk_file_attributes ( id " + this.dbQueryHelper.getPrimaryKey() + " PRIMARY KEY, obj_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, attribute_type_id " + this.dbQueryHelper.getBigIntType() + " NOT NULL, value_type INTEGER NOT NULL, value_byte " + this.dbQueryHelper.getBlobType() + ", value_text TEXT, value_int32 INTEGER, value_int64 " + this.dbQueryHelper.getBigIntType() + ", value_double NUMERIC(20, 10), FOREIGN KEY(obj_id) REFERENCES tsk_files(obj_id) ON DELETE CASCADE, FOREIGN KEY(attribute_type_id) REFERENCES blackboard_attribute_types(attribute_type_id))");
    }

    private class SQLiteDbCreationHelper
    extends DbCreationHelper {
        private static final String PRAGMA_SYNC_OFF = "PRAGMA synchronous = OFF";
        private static final String PRAGMA_READ_UNCOMMITTED_TRUE = "PRAGMA read_uncommitted = True";
        private static final String PRAGMA_ENCODING_UTF8 = "PRAGMA encoding = 'UTF-8'";
        private static final String PRAGMA_PAGE_SIZE_4096 = "PRAGMA page_size = 4096";
        private static final String PRAGMA_FOREIGN_KEYS_ON = "PRAGMA foreign_keys = ON";
        private static final String JDBC_DRIVER = "org.sqlite.JDBC";
        private static final String JDBC_BASE_URI = "jdbc:sqlite:";
        String dbPath;

        SQLiteDbCreationHelper(CaseDatabaseFactory caseDatabaseFactory, String dbPath) {
            super(caseDatabaseFactory);
            this.dbPath = dbPath;
        }

        @Override
        void createDatabase() throws TskCoreException {
            File dbFile = new File(this.dbPath);
            if (dbFile.exists()) {
                throw new TskCoreException("Case database already exists : " + this.dbPath);
            }
            if (dbFile.getParentFile() != null && !dbFile.getParentFile().exists()) {
                throw new TskCoreException("Case database folder does not exist : " + dbFile.getParent());
            }
        }

        @Override
        Connection getConnection() throws TskCoreException {
            Connection conn;
            StringBuilder url = new StringBuilder();
            url.append(JDBC_BASE_URI).append(this.dbPath);
            try {
                Class.forName(JDBC_DRIVER);
                conn = DriverManager.getConnection(url.toString());
            }
            catch (ClassNotFoundException | SQLException ex) {
                throw new TskCoreException("Failed to acquire ephemeral connection SQLite database " + this.dbPath, ex);
            }
            return conn;
        }

        @Override
        void performPreInitialization(Connection conn) throws TskCoreException {
            try (Statement stmt = conn.createStatement();){
                stmt.execute(PRAGMA_SYNC_OFF);
                stmt.execute(PRAGMA_READ_UNCOMMITTED_TRUE);
                stmt.execute(PRAGMA_ENCODING_UTF8);
                stmt.execute(PRAGMA_PAGE_SIZE_4096);
                stmt.execute(PRAGMA_FOREIGN_KEYS_ON);
            }
            catch (SQLException ex) {
                throw new TskCoreException("Error setting pragmas", ex);
            }
        }

        @Override
        void performPostTableInitialization(Connection conn) throws TskCoreException {
            try (Statement stmt = conn.createStatement();){
                stmt.execute("CREATE INDEX tsk_files_datasrc_md5_size_index ON tsk_files(data_source_obj_id, md5, size)");
            }
            catch (SQLException ex) {
                throw new TskCoreException("Error performing SQLite post table initialization", ex);
            }
        }
    }

    private abstract class DbCreationHelper {
        private DbCreationHelper(CaseDatabaseFactory caseDatabaseFactory) {
        }

        abstract void createDatabase() throws TskCoreException;

        abstract Connection getConnection() throws TskCoreException;

        abstract void performPreInitialization(Connection var1) throws TskCoreException;

        abstract void performPostTableInitialization(Connection var1) throws TskCoreException;
    }

    private class PostgreSQLDbCreationHelper
    extends DbCreationHelper {
        private static final String JDBC_BASE_URI = "jdbc:postgresql://";
        private static final String JDBC_DRIVER = "org.postgresql.Driver";
        private final String caseName;
        private final CaseDbConnectionInfo info;

        PostgreSQLDbCreationHelper(CaseDatabaseFactory caseDatabaseFactory, String caseName, CaseDbConnectionInfo info) {
            super(caseDatabaseFactory);
            this.caseName = caseName;
            this.info = info;
        }

        @Override
        void createDatabase() throws TskCoreException {
            try (Connection conn = this.getPostgresConnection();
                 Statement stmt = conn.createStatement();){
                stmt.execute("CREATE DATABASE \"" + this.caseName + "\" WITH ENCODING='UTF8'");
            }
            catch (SQLException ex) {
                throw new TskCoreException("Error creating PostgreSQL case " + this.caseName, ex);
            }
        }

        @Override
        Connection getConnection() throws TskCoreException {
            return this.getConnection(this.caseName);
        }

        Connection getPostgresConnection() throws TskCoreException {
            return this.getConnection("postgres");
        }

        Connection getConnection(String databaseName) throws TskCoreException {
            Connection conn;
            String encodedDbName;
            try {
                encodedDbName = URLEncoder.encode(databaseName, "UTF-8");
            }
            catch (UnsupportedEncodingException ex) {
                logger.log(Level.WARNING, "Error encoding database name " + databaseName, ex);
                encodedDbName = databaseName;
            }
            StringBuilder url = new StringBuilder();
            url.append(JDBC_BASE_URI).append(this.info.getHost()).append(":").append(this.info.getPort()).append('/').append(encodedDbName);
            if (this.info.isSslEnabled()) {
                if (this.info.isSslVerify()) {
                    if (this.info.getCustomSslValidationClassName().isBlank()) {
                        url.append(CaseDatabaseFactory.SSL_VERIFY_DEFAULT_URL);
                    } else {
                        url.append(CaseDatabaseFactory.getCustomPostrgesSslVerificationUrl(this.info.getCustomSslValidationClassName()));
                    }
                } else {
                    url.append(CaseDatabaseFactory.SSL_NONVERIFY_URL);
                }
            }
            try {
                Properties props = new Properties();
                props.setProperty("user", this.info.getUserName());
                props.setProperty("password", this.info.getPassword());
                Class.forName(JDBC_DRIVER);
                conn = DriverManager.getConnection(url.toString(), props);
            }
            catch (ClassNotFoundException | SQLException ex) {
                throw new TskCoreException("Failed to acquire ephemeral connection to PostgreSQL database " + databaseName, ex);
            }
            return conn;
        }

        @Override
        void performPreInitialization(Connection conn) throws TskCoreException {
        }

        @Override
        void performPostTableInitialization(Connection conn) throws TskCoreException {
            try (Statement stmt = conn.createStatement();){
                stmt.execute("ALTER SEQUENCE blackboard_artifacts_artifact_id_seq minvalue -9223372036854775808 restart with -9223372036854775808");
                stmt.execute("CREATE INDEX tsk_files_datasrc_md5_size_partial_index ON tsk_files(data_source_obj_id, md5, size) WHERE md5 IS NOT NULL AND size > 0");
            }
            catch (SQLException ex) {
                throw new TskCoreException("Error performing PostgreSQL post table initialization", ex);
            }
        }
    }
}

