#!/usr/bin/env python3
# badKarma - network reconnaissance toolkit
#
# Copyright (C) 2018 <Giuseppe `r3vn` Corti>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import gi
import base64
import subprocess
import threading

gi.require_version('Gtk', '3.0')

from gi.repository import Gtk
from gi.repository.GdkPixbuf import Pixbuf
from gi.repository import Gio
from gi.repository import GLib

from core.extensions import base_ext

class karma_ext(base_ext):
	
	menu = { 
					"service" : ["http","X11","ms-wbt-server", "vnc", "rtsp"],
					"label"   : "Take Screenshot",
				}

	name = "screenshoter"
	log = True

	def task(self, config):
		""" take the screenshot and return the base64 of the image """

		host        = config["rhost"]
		port        = config["rport"]
		proxychains = config["proxychains"]

		cmd = ""

		if proxychains:
			cmd += "proxychains"

		if config["service"] == "http":
			# HTTP screenshot
			url = host+":"+port

			cmd += "phantomjs --ignore-ssl-errors=true " + config["path_script"]+"/http_screenshot.js "

			if "443" in port:
				cmd += "https://"+url

			else:
				cmd += "http://"+url

		elif config["service"] == "ms-wbt-server":
			# RDP screenshot
			cmd += "bash " + config["path_script"]+"/rdp_screenshot.sh " + config["rhost"] + " " + config["rport"]
				
		elif config["service"] == "X11":
			# X11 screenshot
			cmd += "bash " + config["path_script"]+"/x11_screenshot.sh "+ config["rhost"] + " 0"

		elif config["service"] == "vnc":
			# VNC screenshot
			cmd += "python3 " + config["path_script"]+"/vnc_screenshot.py " + config["rhost"] +" "+ config["rport"] 

		elif config["service"] == "rtsp":
			# RTSP screenshot
			cmd += "bash " + config["path_script"]+"/rtsp_screenshot.sh " + config["rhost"] + " " + config["rport"]


		self.p = subprocess.Popen(cmd, stdout=subprocess.PIPE, shell=True)
		
		
		return self.p, self.p.pid


	def read(self, out):

		download_thread = threading.Thread(target=self.read_screen)
		download_thread.start()
		
		return self.get_log("""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""")

	def call_end(self, screen):
		self.emit('end_task', screen)
		
		try:
			self.image.set_from_pixbuf( Pixbuf.new_from_stream(Gio.MemoryInputStream.new_from_data(base64.b64decode(screen), None) , None) )
		except: pass
		
		return False


	def get_log(self,out):
		base64image = out

		try:
			raw_data = base64.b64decode(base64image)
			input_stream = Gio.MemoryInputStream.new_from_data(raw_data, None) 
			pixbuf = Pixbuf.new_from_stream(input_stream, None) 
		except:
			# error image
			raw_data = base64.b64decode("""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""")
			input_stream = Gio.MemoryInputStream.new_from_data(raw_data, None) 
			pixbuf = Pixbuf.new_from_stream(input_stream, None) 

		self.image = Gtk.Image() 
		self.image.set_from_pixbuf(pixbuf)
		self.image.show()
		scrolledwindow = Gtk.ScrolledWindow()
		scrolledwindow.set_hexpand(True)
		scrolledwindow.set_vexpand(True)
		scrolledwindow.add(self.image)
		scrolledwindow.show()

		return scrolledwindow
		
	def read_screen(self):

		screen = self.p.stdout.read().decode()
		GLib.idle_add(self.call_end, screen)
		