#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright: (c) 2021, [ Hitachi Vantara ]
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function

__metaclass__ = type


DOCUMENTATION = """
---
module: hv_session
short_description: Manages sessions on VSP block storage systems.
description:
  - This module allows to generate and discard session.
  - For examples go to URL
    U(https://github.com/hitachi-vantara/vspone-block-ansible/blob/main/playbooks/vsp_direct/session.yml)
version_added: '4.6.0'
author:
  - Hitachi Vantara LTD (@hitachi-vantara)
requirements:
  - python >= 3.9
attributes:
  check_mode:
    description: Determines if the module should run in check mode.
    support: none
extends_documentation_fragment:
  - hitachivantara.vspone_block.common.connection_with_type
options:
  state:
    description: State of the session.
    required: false
    type: str
    choices: ['present', 'absent']
    default: 'present'
  spec:
    description: Specification for Session.
    required: false
    type: dict
    suboptions:
      id:
        description: Session ID. Required for delete operation.
        type: int
        required: false
      alive_time_in_seconds:
        description: The idle time in seconds after which a session times out. Valid for create session operation.
          If omitted, 300 is applied. If specified value must be between 1 and 300.
        type: int
        required: false
      authentication_timeout:
        description: Timeout value for authentication processing (in seconds). Valid for create session operation.
          Specify this value if an external authentication server is being used to authenticate users.
          Change the value according to the external authentication settings of the storage system.
          If omitted, 120 is applied. If specified value must be between 1 and 900.
        type: int
        required: false
      token:
        description: Authentication token for the session. Required for delete operation.
        type: str
        required: false
      force:
        description: Specify whether to force discarding of the session generated by other users, in addition to the session you generated.
          Only a user who belongs to the Administrator user group (built-in user group) can specify this attribute.
          When true, forces discarding of the session generated by other users, in addition to the session you generated.
          When false, forces discarding of only the session you generated. If you omit this value, false is used.
        type: bool
        required: false
        default: false
"""

EXAMPLES = """
- name: Create a session
  hitachivantara.vspone_block.sds_block.hv_sds_block_session:
    connection_info:
      address: sdsb.company.com
      username: "admin"
      password: "password"
    spec:
      alive_time_in_seconds: 100

- name: Delete a session
  hitachivantara.vspone_block.sds_block.hv_sds_block_session:
    connection_info:
      address: sdsb.company.com
      username: "admin"
      password: "password"
      state: "absent"
      spec:
        id: "ae0f247c-dc56-491c-9cb9-4b2b6d33b345"
"""

RETURN = r"""
session:
  description: Session information returned by the module.
  returned: always
  type: dict
  contains:
    session_id:
      description: Unique identifier of the session.
      type: int
      sample: 15561
    token:
      description: Authentication token associated with the session.
      type: str
      sample: d7c7bd35d3854a9f973d934800fb70ab
"""

from ansible.module_utils.basic import AnsibleModule
from ansible_collections.hitachivantara.vspone_block.plugins.module_utils.reconciler.vsp_session import (
    VSPSessionReconciler,
)
from ansible_collections.hitachivantara.vspone_block.plugins.module_utils.common.hv_log import (
    Log,
)
from ansible_collections.hitachivantara.vspone_block.plugins.module_utils.common.vsp_utils import (
    VSPSessionArguments,
    VSPParametersManager,
)
from ansible_collections.hitachivantara.vspone_block.plugins.module_utils.common.ansible_common import (
    validate_ansible_product_registration,
)
from ansible_collections.hitachivantara.vspone_block.plugins.module_utils.common.hv_log import (
    Log,
)


class VSPSessionManager:
    def __init__(self):

        self.logger = Log()
        self.argument_spec = VSPSessionArguments().session()
        self.module = AnsibleModule(
            argument_spec=self.argument_spec,
            supports_check_mode=False,
        )

        parameter_manager = VSPParametersManager(self.module.params)
        self.state = parameter_manager.get_state()
        self.connection_info = parameter_manager.get_connection_info()
        self.spec = parameter_manager.get_session_spec()

    def apply(self):
        self.logger.writeInfo("=== Start of VSP Session Operation ===")
        registration_message = validate_ansible_product_registration()

        try:
            vsp_reconciler = VSPSessionReconciler(self.connection_info)
            session = vsp_reconciler.reconcile_session(self.state, self.spec)

            # self.logger.writeDebug(f"MOD:hv_session:session= {session}")

        except Exception as e:
            self.logger.writeException(e)
            self.logger.writeInfo("=== End of VSP Session Operation ===")
            self.module.fail_json(msg=str(e))

        response = {
            "changed": self.connection_info.changed,
            "session": session,
        }

        if registration_message:
            response["user_consent_required"] = registration_message
        self.logger.writeInfo("=== End of VSP Session Operation ===")
        self.module.exit_json(**response)


def main(module=None):
    obj_store = VSPSessionManager()
    obj_store.apply()


if __name__ == "__main__":
    main()
