import { jsxs, jsx } from 'react/jsx-runtime';
import { useState, useEffect } from 'react';
import { css } from '@emotion/css';
import '@grafana/data';
import { stylesFactory, Button, DataLinkInput, LegacyForms } from '@grafana/ui';
import { usePrevious } from 'react-use';
import { DataSourcePicker } from '../DataSourcePicker/DataSourcePicker.js';

const { FormField, Switch } = LegacyForms;
const getStyles = stylesFactory(() => ({
  firstRow: css`
    display: flex;
  `,
  nameField: css`
    flex: 2;
  `,
  regexField: css`
    flex: 3;
  `,
  row: css`
    display: flex;
    align-items: baseline;
  `
}));
const DataLink = (props) => {
  const { value, onChange, onDelete, suggestions, className } = props;
  const styles = getStyles();
  const [showInternalLink, setShowInternalLink] = useInternalLink(value.datasourceUid);
  const handleChange = (field) => (event) => {
    onChange({
      ...value,
      [field]: event.currentTarget.value
    });
  };
  return /* @__PURE__ */ jsxs("div", { className, children: [
    /* @__PURE__ */ jsxs("div", { className: styles.firstRow + " gf-form", children: [
      /* @__PURE__ */ jsx(
        FormField,
        {
          className: styles.nameField,
          labelWidth: 6,
          inputWidth: null,
          label: "Field",
          type: "text",
          value: value.field,
          tooltip: "Can be exact field name or a regex pattern that will match on the field name.",
          onChange: handleChange("field")
        }
      ),
      /* @__PURE__ */ jsx(
        Button,
        {
          variant: "destructive",
          title: "Remove field",
          icon: "times",
          onClick: (event) => {
            event.preventDefault();
            onDelete();
          }
        }
      )
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "gf-form", children: [
      /* @__PURE__ */ jsx(
        FormField,
        {
          className: styles.nameField,
          inputWidth: null,
          label: "Label",
          type: "text",
          value: value.label,
          onChange: handleChange("label"),
          tooltip: "Use to provide a meaningful label to the data matched in the regex"
        }
      ),
      /* @__PURE__ */ jsx(
        FormField,
        {
          className: styles.regexField,
          inputWidth: null,
          label: "Regex",
          type: "text",
          value: value.matcherRegex,
          onChange: handleChange("matcherRegex"),
          tooltip: "Use to parse and capture some part of the log message. You can use the captured groups in the template."
        }
      )
    ] }),
    /* @__PURE__ */ jsx("div", { className: "gf-form", children: /* @__PURE__ */ jsx(
      FormField,
      {
        label: showInternalLink ? "Query" : "URL",
        labelWidth: 6,
        inputEl: /* @__PURE__ */ jsx(
          DataLinkInput,
          {
            placeholder: showInternalLink ? "${__value.raw}" : "http://example.com/${__value.raw}",
            value: value.url || "",
            onChange: (newValue) => onChange({
              ...value,
              url: newValue
            }),
            suggestions
          }
        ),
        className: css`
            width: 100%;
          `
      }
    ) }),
    /* @__PURE__ */ jsxs("div", { className: styles.row, children: [
      /* @__PURE__ */ jsx(
        Switch,
        {
          labelClass: "width-6",
          label: "Internal link",
          checked: showInternalLink,
          onChange: () => {
            if (showInternalLink) {
              onChange({
                ...value,
                datasourceUid: undefined
              });
            }
            setShowInternalLink(!showInternalLink);
          }
        }
      ),
      showInternalLink && /* @__PURE__ */ jsx(
        DataSourcePicker,
        {
          onChange: (ds) => {
            onChange({
              ...value,
              datasourceUid: ds.uid
            });
          },
          current: value.datasourceUid
        }
      )
    ] })
  ] });
};
function useInternalLink(datasourceUid) {
  const [showInternalLink, setShowInternalLink] = useState(!!datasourceUid);
  const previousUid = usePrevious(datasourceUid);
  useEffect(() => {
    if (!previousUid && datasourceUid && !showInternalLink) {
      setShowInternalLink(true);
    }
    if (previousUid && !datasourceUid && showInternalLink) {
      setShowInternalLink(false);
    }
  }, [previousUid, datasourceUid, showInternalLink]);
  return [showInternalLink, setShowInternalLink];
}

export { DataLink };
//# sourceMappingURL=DataLink.js.map
