﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/route53-recovery-readiness/Route53RecoveryReadinessRequest.h>
#include <aws/route53-recovery-readiness/Route53RecoveryReadiness_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Route53RecoveryReadiness {
namespace Model {

/**
 */
class CreateRecoveryGroupRequest : public Route53RecoveryReadinessRequest {
 public:
  AWS_ROUTE53RECOVERYREADINESS_API CreateRecoveryGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRecoveryGroup"; }

  AWS_ROUTE53RECOVERYREADINESS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A list of the cell Amazon Resource Names (ARNs) in the recovery group.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCells() const { return m_cells; }
  inline bool CellsHasBeenSet() const { return m_cellsHasBeenSet; }
  template <typename CellsT = Aws::Vector<Aws::String>>
  void SetCells(CellsT&& value) {
    m_cellsHasBeenSet = true;
    m_cells = std::forward<CellsT>(value);
  }
  template <typename CellsT = Aws::Vector<Aws::String>>
  CreateRecoveryGroupRequest& WithCells(CellsT&& value) {
    SetCells(std::forward<CellsT>(value));
    return *this;
  }
  template <typename CellsT = Aws::String>
  CreateRecoveryGroupRequest& AddCells(CellsT&& value) {
    m_cellsHasBeenSet = true;
    m_cells.emplace_back(std::forward<CellsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the recovery group to create.</p>
   */
  inline const Aws::String& GetRecoveryGroupName() const { return m_recoveryGroupName; }
  inline bool RecoveryGroupNameHasBeenSet() const { return m_recoveryGroupNameHasBeenSet; }
  template <typename RecoveryGroupNameT = Aws::String>
  void SetRecoveryGroupName(RecoveryGroupNameT&& value) {
    m_recoveryGroupNameHasBeenSet = true;
    m_recoveryGroupName = std::forward<RecoveryGroupNameT>(value);
  }
  template <typename RecoveryGroupNameT = Aws::String>
  CreateRecoveryGroupRequest& WithRecoveryGroupName(RecoveryGroupNameT&& value) {
    SetRecoveryGroupName(std::forward<RecoveryGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRecoveryGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRecoveryGroupRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_cells;

  Aws::String m_recoveryGroupName;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_cellsHasBeenSet = false;
  bool m_recoveryGroupNameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Route53RecoveryReadiness
}  // namespace Aws
