﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2Request.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2_EXPORTS.h>
#include <aws/pinpoint-sms-voice-v2/model/MessageType.h>
#include <aws/pinpoint-sms-voice-v2/model/Tag.h>

#include <utility>

namespace Aws {
namespace PinpointSMSVoiceV2 {
namespace Model {

/**
 */
class RequestSenderIdRequest : public PinpointSMSVoiceV2Request {
 public:
  AWS_PINPOINTSMSVOICEV2_API RequestSenderIdRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RequestSenderId"; }

  AWS_PINPOINTSMSVOICEV2_API Aws::String SerializePayload() const override;

  AWS_PINPOINTSMSVOICEV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The sender ID string to request.</p>
   */
  inline const Aws::String& GetSenderId() const { return m_senderId; }
  inline bool SenderIdHasBeenSet() const { return m_senderIdHasBeenSet; }
  template <typename SenderIdT = Aws::String>
  void SetSenderId(SenderIdT&& value) {
    m_senderIdHasBeenSet = true;
    m_senderId = std::forward<SenderIdT>(value);
  }
  template <typename SenderIdT = Aws::String>
  RequestSenderIdRequest& WithSenderId(SenderIdT&& value) {
    SetSenderId(std::forward<SenderIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The two-character code, in ISO 3166-1 alpha-2 format, for the country or
   * region.</p>
   */
  inline const Aws::String& GetIsoCountryCode() const { return m_isoCountryCode; }
  inline bool IsoCountryCodeHasBeenSet() const { return m_isoCountryCodeHasBeenSet; }
  template <typename IsoCountryCodeT = Aws::String>
  void SetIsoCountryCode(IsoCountryCodeT&& value) {
    m_isoCountryCodeHasBeenSet = true;
    m_isoCountryCode = std::forward<IsoCountryCodeT>(value);
  }
  template <typename IsoCountryCodeT = Aws::String>
  RequestSenderIdRequest& WithIsoCountryCode(IsoCountryCodeT&& value) {
    SetIsoCountryCode(std::forward<IsoCountryCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of message. Valid values are TRANSACTIONAL for messages that are
   * critical or time-sensitive and PROMOTIONAL for messages that aren't critical or
   * time-sensitive.</p>
   */
  inline const Aws::Vector<MessageType>& GetMessageTypes() const { return m_messageTypes; }
  inline bool MessageTypesHasBeenSet() const { return m_messageTypesHasBeenSet; }
  template <typename MessageTypesT = Aws::Vector<MessageType>>
  void SetMessageTypes(MessageTypesT&& value) {
    m_messageTypesHasBeenSet = true;
    m_messageTypes = std::forward<MessageTypesT>(value);
  }
  template <typename MessageTypesT = Aws::Vector<MessageType>>
  RequestSenderIdRequest& WithMessageTypes(MessageTypesT&& value) {
    SetMessageTypes(std::forward<MessageTypesT>(value));
    return *this;
  }
  inline RequestSenderIdRequest& AddMessageTypes(MessageType value) {
    m_messageTypesHasBeenSet = true;
    m_messageTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>By default this is set to false. When set to true the sender ID can't be
   * deleted.</p>
   */
  inline bool GetDeletionProtectionEnabled() const { return m_deletionProtectionEnabled; }
  inline bool DeletionProtectionEnabledHasBeenSet() const { return m_deletionProtectionEnabledHasBeenSet; }
  inline void SetDeletionProtectionEnabled(bool value) {
    m_deletionProtectionEnabledHasBeenSet = true;
    m_deletionProtectionEnabled = value;
  }
  inline RequestSenderIdRequest& WithDeletionProtectionEnabled(bool value) {
    SetDeletionProtectionEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of tags (key and value pairs) to associate with the sender ID.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  RequestSenderIdRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  RequestSenderIdRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier that you provide to ensure the idempotency
   * of the request. If you don't specify a client token, a randomly generated token
   * is used for the request to ensure idempotency.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  RequestSenderIdRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_senderId;

  Aws::String m_isoCountryCode;

  Aws::Vector<MessageType> m_messageTypes;

  bool m_deletionProtectionEnabled{false};

  Aws::Vector<Tag> m_tags;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_senderIdHasBeenSet = false;
  bool m_isoCountryCodeHasBeenSet = false;
  bool m_messageTypesHasBeenSet = false;
  bool m_deletionProtectionEnabledHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace PinpointSMSVoiceV2
}  // namespace Aws
