﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/opensearch/OpenSearchServiceRequest.h>
#include <aws/opensearch/OpenSearchService_EXPORTS.h>
#include <aws/opensearch/model/DirectQueryDataSourceType.h>
#include <aws/opensearch/model/Tag.h>

#include <utility>

namespace Aws {
namespace OpenSearchService {
namespace Model {

/**
 */
class AddDirectQueryDataSourceRequest : public OpenSearchServiceRequest {
 public:
  AWS_OPENSEARCHSERVICE_API AddDirectQueryDataSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddDirectQueryDataSource"; }

  AWS_OPENSEARCHSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> A unique, user-defined label to identify the data source within your
   * OpenSearch Service environment. </p>
   */
  inline const Aws::String& GetDataSourceName() const { return m_dataSourceName; }
  inline bool DataSourceNameHasBeenSet() const { return m_dataSourceNameHasBeenSet; }
  template <typename DataSourceNameT = Aws::String>
  void SetDataSourceName(DataSourceNameT&& value) {
    m_dataSourceNameHasBeenSet = true;
    m_dataSourceName = std::forward<DataSourceNameT>(value);
  }
  template <typename DataSourceNameT = Aws::String>
  AddDirectQueryDataSourceRequest& WithDataSourceName(DataSourceNameT&& value) {
    SetDataSourceName(std::forward<DataSourceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The supported Amazon Web Services service that you want to use as the source
   * for direct queries in OpenSearch Service. </p>
   */
  inline const DirectQueryDataSourceType& GetDataSourceType() const { return m_dataSourceType; }
  inline bool DataSourceTypeHasBeenSet() const { return m_dataSourceTypeHasBeenSet; }
  template <typename DataSourceTypeT = DirectQueryDataSourceType>
  void SetDataSourceType(DataSourceTypeT&& value) {
    m_dataSourceTypeHasBeenSet = true;
    m_dataSourceType = std::forward<DataSourceTypeT>(value);
  }
  template <typename DataSourceTypeT = DirectQueryDataSourceType>
  AddDirectQueryDataSourceRequest& WithDataSourceType(DataSourceTypeT&& value) {
    SetDataSourceType(std::forward<DataSourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An optional text field for providing additional context and details about
   * the data source. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  AddDirectQueryDataSourceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A list of Amazon Resource Names (ARNs) for the OpenSearch collections that
   * are associated with the direct query data source. </p>
   */
  inline const Aws::Vector<Aws::String>& GetOpenSearchArns() const { return m_openSearchArns; }
  inline bool OpenSearchArnsHasBeenSet() const { return m_openSearchArnsHasBeenSet; }
  template <typename OpenSearchArnsT = Aws::Vector<Aws::String>>
  void SetOpenSearchArns(OpenSearchArnsT&& value) {
    m_openSearchArnsHasBeenSet = true;
    m_openSearchArns = std::forward<OpenSearchArnsT>(value);
  }
  template <typename OpenSearchArnsT = Aws::Vector<Aws::String>>
  AddDirectQueryDataSourceRequest& WithOpenSearchArns(OpenSearchArnsT&& value) {
    SetOpenSearchArns(std::forward<OpenSearchArnsT>(value));
    return *this;
  }
  template <typename OpenSearchArnsT = Aws::String>
  AddDirectQueryDataSourceRequest& AddOpenSearchArns(OpenSearchArnsT&& value) {
    m_openSearchArnsHasBeenSet = true;
    m_openSearchArns.emplace_back(std::forward<OpenSearchArnsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Vector<Tag>& GetTagList() const { return m_tagList; }
  inline bool TagListHasBeenSet() const { return m_tagListHasBeenSet; }
  template <typename TagListT = Aws::Vector<Tag>>
  void SetTagList(TagListT&& value) {
    m_tagListHasBeenSet = true;
    m_tagList = std::forward<TagListT>(value);
  }
  template <typename TagListT = Aws::Vector<Tag>>
  AddDirectQueryDataSourceRequest& WithTagList(TagListT&& value) {
    SetTagList(std::forward<TagListT>(value));
    return *this;
  }
  template <typename TagListT = Tag>
  AddDirectQueryDataSourceRequest& AddTagList(TagListT&& value) {
    m_tagListHasBeenSet = true;
    m_tagList.emplace_back(std::forward<TagListT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dataSourceName;

  DirectQueryDataSourceType m_dataSourceType;

  Aws::String m_description;

  Aws::Vector<Aws::String> m_openSearchArns;

  Aws::Vector<Tag> m_tagList;
  bool m_dataSourceNameHasBeenSet = false;
  bool m_dataSourceTypeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_openSearchArnsHasBeenSet = false;
  bool m_tagListHasBeenSet = false;
};

}  // namespace Model
}  // namespace OpenSearchService
}  // namespace Aws
