﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/directory-service-data/DirectoryServiceDataRequest.h>
#include <aws/directory-service-data/DirectoryServiceData_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DirectoryServiceData {
namespace Model {

/**
 */
class DescribeUserRequest : public DirectoryServiceDataRequest {
 public:
  AWS_DIRECTORYSERVICEDATA_API DescribeUserRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeUser"; }

  AWS_DIRECTORYSERVICEDATA_API Aws::String SerializePayload() const override;

  AWS_DIRECTORYSERVICEDATA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> The identifier (ID) of the directory that's associated with the user. </p>
   */
  inline const Aws::String& GetDirectoryId() const { return m_directoryId; }
  inline bool DirectoryIdHasBeenSet() const { return m_directoryIdHasBeenSet; }
  template <typename DirectoryIdT = Aws::String>
  void SetDirectoryId(DirectoryIdT&& value) {
    m_directoryIdHasBeenSet = true;
    m_directoryId = std::forward<DirectoryIdT>(value);
  }
  template <typename DirectoryIdT = Aws::String>
  DescribeUserRequest& WithDirectoryId(DirectoryIdT&& value) {
    SetDirectoryId(std::forward<DirectoryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> One or more attribute names to be returned for the user. A key is an
   * attribute name, and the value is a list of maps. For a list of supported
   * attributes, see <a
   * href="https://docs.aws.amazon.com/directoryservice/latest/admin-guide/ad_data_attributes.html">Directory
   * Service Data Attributes</a>. </p>
   */
  inline const Aws::Vector<Aws::String>& GetOtherAttributes() const { return m_otherAttributes; }
  inline bool OtherAttributesHasBeenSet() const { return m_otherAttributesHasBeenSet; }
  template <typename OtherAttributesT = Aws::Vector<Aws::String>>
  void SetOtherAttributes(OtherAttributesT&& value) {
    m_otherAttributesHasBeenSet = true;
    m_otherAttributes = std::forward<OtherAttributesT>(value);
  }
  template <typename OtherAttributesT = Aws::Vector<Aws::String>>
  DescribeUserRequest& WithOtherAttributes(OtherAttributesT&& value) {
    SetOtherAttributes(std::forward<OtherAttributesT>(value));
    return *this;
  }
  template <typename OtherAttributesT = Aws::String>
  DescribeUserRequest& AddOtherAttributes(OtherAttributesT&& value) {
    m_otherAttributesHasBeenSet = true;
    m_otherAttributes.emplace_back(std::forward<OtherAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The domain name that's associated with the user. </p>  <p> This
   * parameter is optional, so you can return users outside your Managed Microsoft AD
   * domain. When no value is defined, only your Managed Microsoft AD users are
   * returned. </p> <p> This value is case insensitive. </p>
   */
  inline const Aws::String& GetRealm() const { return m_realm; }
  inline bool RealmHasBeenSet() const { return m_realmHasBeenSet; }
  template <typename RealmT = Aws::String>
  void SetRealm(RealmT&& value) {
    m_realmHasBeenSet = true;
    m_realm = std::forward<RealmT>(value);
  }
  template <typename RealmT = Aws::String>
  DescribeUserRequest& WithRealm(RealmT&& value) {
    SetRealm(std::forward<RealmT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the user. </p>
   */
  inline const Aws::String& GetSAMAccountName() const { return m_sAMAccountName; }
  inline bool SAMAccountNameHasBeenSet() const { return m_sAMAccountNameHasBeenSet; }
  template <typename SAMAccountNameT = Aws::String>
  void SetSAMAccountName(SAMAccountNameT&& value) {
    m_sAMAccountNameHasBeenSet = true;
    m_sAMAccountName = std::forward<SAMAccountNameT>(value);
  }
  template <typename SAMAccountNameT = Aws::String>
  DescribeUserRequest& WithSAMAccountName(SAMAccountNameT&& value) {
    SetSAMAccountName(std::forward<SAMAccountNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryId;

  Aws::Vector<Aws::String> m_otherAttributes;

  Aws::String m_realm;

  Aws::String m_sAMAccountName;
  bool m_directoryIdHasBeenSet = false;
  bool m_otherAttributesHasBeenSet = false;
  bool m_realmHasBeenSet = false;
  bool m_sAMAccountNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectoryServiceData
}  // namespace Aws
