//
// Copyright Aliaksei Levin (levlam@telegram.org), Arseny Smirnov (arseny30@gmail.com) 2014-2024
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
#include "td/telegram/MessagesManager.h"

#include "td/telegram/AccountManager.h"
#include "td/telegram/AuthManager.h"
#include "td/telegram/BackgroundInfo.hpp"
#include "td/telegram/BlockListId.h"
#include "td/telegram/BusinessBotManageBar.h"
#include "td/telegram/ChainId.h"
#include "td/telegram/ChannelType.h"
#include "td/telegram/ChatManager.h"
#include "td/telegram/ChatReactions.hpp"
#include "td/telegram/Dependencies.h"
#include "td/telegram/DialogAction.h"
#include "td/telegram/DialogActionBar.h"
#include "td/telegram/DialogActionManager.h"
#include "td/telegram/DialogDb.h"
#include "td/telegram/DialogFilter.h"
#include "td/telegram/DialogFilterManager.h"
#include "td/telegram/DialogManager.h"
#include "td/telegram/DialogNotificationSettings.hpp"
#include "td/telegram/DialogParticipantManager.h"
#include "td/telegram/DownloadManager.h"
#include "td/telegram/DraftMessage.h"
#include "td/telegram/DraftMessage.hpp"
#include "td/telegram/FileReferenceManager.h"
#include "td/telegram/files/FileId.hpp"
#include "td/telegram/files/FileLocation.h"
#include "td/telegram/files/FileManager.h"
#include "td/telegram/files/FileType.h"
#include "td/telegram/ForumTopicManager.h"
#include "td/telegram/Global.h"
#include "td/telegram/GroupCallManager.h"
#include "td/telegram/InlineQueriesManager.h"
#include "td/telegram/InputDialogId.h"
#include "td/telegram/InputMessageText.h"
#include "td/telegram/LinkManager.h"
#include "td/telegram/Location.h"
#include "td/telegram/logevent/LogEvent.h"
#include "td/telegram/MessageContent.h"
#include "td/telegram/MessageDb.h"
#include "td/telegram/MessageEntity.h"
#include "td/telegram/MessageEntity.hpp"
#include "td/telegram/MessageForwardInfo.h"
#include "td/telegram/MessageForwardInfo.hpp"
#include "td/telegram/MessageOrigin.hpp"
#include "td/telegram/MessageQuote.h"
#include "td/telegram/MessageReaction.h"
#include "td/telegram/MessageReaction.hpp"
#include "td/telegram/MessageReplyInfo.hpp"
#include "td/telegram/MessageSender.h"
#include "td/telegram/misc.h"
#include "td/telegram/MissingInvitee.h"
#include "td/telegram/net/DcId.h"
#include "td/telegram/NotificationGroupInfo.hpp"
#include "td/telegram/NotificationGroupType.h"
#include "td/telegram/NotificationManager.h"
#include "td/telegram/NotificationObjectId.h"
#include "td/telegram/NotificationSettingsManager.h"
#include "td/telegram/NotificationSound.h"
#include "td/telegram/NotificationType.h"
#include "td/telegram/OptionManager.h"
#include "td/telegram/Photo.h"
#include "td/telegram/PollId.h"
#include "td/telegram/PublicDialogType.h"
#include "td/telegram/QuickReplyManager.h"
#include "td/telegram/ReactionManager.h"
#include "td/telegram/RepliedMessageInfo.hpp"
#include "td/telegram/ReplyMarkup.h"
#include "td/telegram/ReplyMarkup.hpp"
#include "td/telegram/SavedMessagesManager.h"
#include "td/telegram/SecretChatsManager.h"
#include "td/telegram/SponsoredMessageManager.h"
#include "td/telegram/StickerType.h"
#include "td/telegram/StoryId.h"
#include "td/telegram/StoryManager.h"
#include "td/telegram/Td.h"
#include "td/telegram/TdDb.h"
#include "td/telegram/telegram_api.h"
#include "td/telegram/TopDialogCategory.h"
#include "td/telegram/TopDialogManager.h"
#include "td/telegram/TranslationManager.h"
#include "td/telegram/UpdatesManager.h"
#include "td/telegram/UserManager.h"
#include "td/telegram/Usernames.h"
#include "td/telegram/Version.h"
#include "td/telegram/WebPageId.h"

#include "td/db/binlog/BinlogEvent.h"
#include "td/db/binlog/BinlogHelper.h"
#include "td/db/SqliteKeyValue.h"
#include "td/db/SqliteKeyValueAsync.h"

#include "td/actor/SleepActor.h"

#include "td/utils/algorithm.h"
#include "td/utils/format.h"
#include "td/utils/logging.h"
#include "td/utils/misc.h"
#include "td/utils/Random.h"
#include "td/utils/Slice.h"
#include "td/utils/SliceBuilder.h"
#include "td/utils/Time.h"
#include "td/utils/tl_helpers.h"
#include "td/utils/utf8.h"

#include <algorithm>
#include <limits>
#include <tuple>
#include <type_traits>
#include <unordered_set>
#include <utility>

namespace td {

class GetDialogQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getPeerDialogs(
            td_->dialog_manager_->get_input_dialog_peers({dialog_id}, AccessRights::Read)),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetDialogQuery: " << to_string(result);

    td_->user_manager_->on_get_users(std::move(result->users_), "GetDialogQuery");
    td_->chat_manager_->on_get_chats(std::move(result->chats_), "GetDialogQuery");
    td_->messages_manager_->on_get_dialogs(FolderId(), std::move(result->dialogs_), -1, std::move(result->messages_),
                                           PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(),
                                                                   dialog_id = dialog_id_](Result<> result) {
                                             send_closure(actor_id, &MessagesManager::on_get_dialog_query_finished,
                                                          dialog_id,
                                                          result.is_error() ? result.move_as_error() : Status::OK());
                                           }));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogQuery");
    td_->messages_manager_->on_get_dialog_query_finished(dialog_id_, std::move(status));
  }
};

class GetPinnedDialogsQuery final : public Td::ResultHandler {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetPinnedDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  NetQueryRef send(FolderId folder_id) {
    folder_id_ = folder_id;
    auto query =
        G()->net_query_creator().create(telegram_api::messages_getPinnedDialogs(folder_id.get()), {{folder_id}});
    auto result = query.get_weak();
    send_query(std::move(query));
    return result;
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive pinned chats in " << folder_id_ << ": " << to_string(result);

    td_->user_manager_->on_get_users(std::move(result->users_), "GetPinnedDialogsQuery");
    td_->chat_manager_->on_get_chats(std::move(result->chats_), "GetPinnedDialogsQuery");
    td_->messages_manager_->on_get_dialogs(folder_id_, std::move(result->dialogs_), -2, std::move(result->messages_),
                                           std::move(promise_));
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetDialogUnreadMarksQuery final : public Td::ResultHandler {
 public:
  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_getDialogUnreadMarks()));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogUnreadMarks>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto results = result_ptr.move_as_ok();
    for (auto &result : results) {
      td_->messages_manager_->on_update_dialog_is_marked_as_unread(DialogId(result), true);
    }

    G()->td_db()->get_binlog_pmc()->set("fetched_marks_as_unread", "1");
  }

  void on_error(Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for GetDialogUnreadMarksQuery: " << status;
    }
    status.ignore();
  }
};

class GetDiscussionMessageQuery final : public Td::ResultHandler {
  Promise<MessageThreadInfo> promise_;
  DialogId dialog_id_;
  MessageId message_id_;
  DialogId expected_dialog_id_;
  MessageId expected_message_id_;

 public:
  explicit GetDiscussionMessageQuery(Promise<MessageThreadInfo> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    expected_dialog_id_ = expected_dialog_id;
    expected_message_id_ = expected_message_id;
    CHECK(expected_dialog_id_.is_valid());
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getDiscussionMessage(std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDiscussionMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    td_->messages_manager_->process_discussion_message(result_ptr.move_as_ok(), dialog_id_, message_id_,
                                                       expected_dialog_id_, expected_message_id_, std::move(promise_));
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_get_message_error(dialog_id_, message_id_, status, "GetDiscussionMessageQuery");
    promise_.set_error(std::move(status));
  }
};

class GetMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids) {
    send_query(G()->net_query_creator().create(telegram_api::messages_getMessages(std::move(message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, DialogId(), result_ptr.move_as_ok(), "GetMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages) << "Receive channel messages in GetMessagesQuery";
    td_->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, false,
                                            std::move(promise_), "GetMessagesQuery");
  }

  void on_error(Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId last_new_message_id_;
  bool can_be_inaccessible_ = false;

 public:
  explicit GetChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel,
            vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids, MessageId last_new_message_id) {
    channel_id_ = channel_id;
    last_new_message_id_ = last_new_message_id;
    can_be_inaccessible_ = message_ids.size() == 1 && message_ids[0]->get_id() != telegram_api::inputMessageID::ID;
    CHECK(input_channel != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::channels_getMessages(std::move(input_channel), std::move(message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, DialogId(channel_id_), result_ptr.move_as_ok(), "GetChannelMessagesQuery");
    LOG_IF(ERROR, !info.is_channel_messages) << "Receive ordinary messages in GetChannelMessagesQuery";
    // messages with invalid big identifiers can be received as messageEmpty
    // bots can receive messageEmpty because of their privacy mode
    if (last_new_message_id_.is_valid() && !td_->auth_manager_->is_bot()) {
      vector<MessageId> empty_message_ids;
      for (auto &message : info.messages) {
        if (message->get_id() == telegram_api::messageEmpty::ID) {
          auto message_id = MessageId::get_message_id(message, false);
          if (message_id.is_valid() && message_id <= last_new_message_id_) {
            empty_message_ids.push_back(message_id);
          }
        }
      }
      td_->messages_manager_->on_get_empty_messages(DialogId(channel_id_), empty_message_ids);
    }
    const char *source = can_be_inaccessible_ ? "GetRepliedChannelMessageQuery" : "GetChannelMessagesQuery";
    td_->messages_manager_->get_channel_difference_if_needed(
        DialogId(channel_id_), std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), source,
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_messages, std::move(info.messages),
                         info.is_channel_messages, false, std::move(promise), source);
          }
        }),
        source);
  }

  void on_error(Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td_->chat_manager_->on_get_channel_error(channel_id_, status, "GetChannelMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit GetScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> &&input_peer, vector<int32> &&message_ids) {
    dialog_id_ = dialog_id;
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getScheduledMessages(std::move(input_peer), std::move(message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "GetScheduledMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages != (dialog_id_.get_type() == DialogType::Channel))
        << "Receive wrong messages constructor in GetScheduledMessagesQuery";
    td_->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, true,
                                            std::move(promise_), "GetScheduledMessagesQuery");
  }

  void on_error(Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogPinnedMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId message_id_;

 public:
  explicit UpdateDialogPinnedMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, bool is_unpin, bool disable_notification, bool only_for_self) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't update pinned message in " << dialog_id;
      return on_error(Status::Error(400, "Can't update pinned message"));
    }

    int32 flags = 0;
    if (disable_notification) {
      flags |= telegram_api::messages_updatePinnedMessage::SILENT_MASK;
    }
    if (is_unpin) {
      flags |= telegram_api::messages_updatePinnedMessage::UNPIN_MASK;
    }
    if (only_for_self) {
      flags |= telegram_api::messages_updatePinnedMessage::PM_ONESIDE_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_updatePinnedMessage(flags, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                                   std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_updatePinnedMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for UpdateDialogPinnedMessageQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_get_message_error(dialog_id_, message_id_, status, "UpdateDialogPinnedMessageQuery");
    promise_.set_error(std::move(status));
  }
};

class UnpinAllMessagesQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  DialogId dialog_id_;
  MessageId top_thread_message_id_;

 public:
  explicit UnpinAllMessagesQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id) {
    dialog_id_ = dialog_id;
    top_thread_message_id_ = top_thread_message_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't unpin all messages in " << dialog_id_;
      return on_error(Status::Error(400, "Can't unpin all messages"));
    }

    int32 flags = 0;
    if (top_thread_message_id.is_valid()) {
      flags |= telegram_api::messages_unpinAllMessages::TOP_MSG_ID_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_unpinAllMessages(
        flags, std::move(input_peer), top_thread_message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_unpinAllMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_get_message_error(dialog_id_, top_thread_message_id_, status, "UnpinAllMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetOutboxReadDateQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::MessageReadDate>> promise_;
  DialogId dialog_id_;
  MessageId message_id_;

 public:
  explicit GetOutboxReadDateQuery(Promise<td_api::object_ptr<td_api::MessageReadDate>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getOutboxReadDate(std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getOutboxReadDate>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    promise_.set_value(td_api::make_object<td_api::messageReadDateRead>(ptr->date_));
  }

  void on_error(Status status) final {
    if (status.message() == "USER_PRIVACY_RESTRICTED") {
      return promise_.set_value(td_api::make_object<td_api::messageReadDateUserPrivacyRestricted>());
    }
    if (status.message() == "YOUR_PRIVACY_RESTRICTED") {
      return promise_.set_value(td_api::make_object<td_api::messageReadDateMyPrivacyRestricted>());
    }
    if (status.message() == "MESSAGE_TOO_OLD") {
      return promise_.set_value(td_api::make_object<td_api::messageReadDateTooOld>());
    }

    td_->messages_manager_->on_get_message_error(dialog_id_, message_id_, status, "GetOutboxReadDateQuery");
    promise_.set_error(std::move(status));
  }
};

class GetMessageReadParticipantsQuery final : public Td::ResultHandler {
  Promise<MessageViewers> promise_;
  DialogId dialog_id_;
  MessageId message_id_;

 public:
  explicit GetMessageReadParticipantsQuery(Promise<MessageViewers> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_getMessageReadParticipants(
        std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessageReadParticipants>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(MessageViewers(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_get_message_error(dialog_id_, message_id_, status, "GetMessageReadParticipantsQuery");
    promise_.set_error(std::move(status));
  }
};

class ExportChannelMessageLinkQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId message_id_;
  bool for_group_ = false;
  bool ignore_result_ = false;

 public:
  explicit ExportChannelMessageLinkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId message_id, bool for_group, bool ignore_result) {
    channel_id_ = channel_id;
    message_id_ = message_id;
    for_group_ = for_group;
    ignore_result_ = ignore_result;
    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }
    int32 flags = 0;
    if (for_group) {
      flags |= telegram_api::channels_exportMessageLink::GROUPED_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::channels_exportMessageLink(flags, false /*ignored*/, false /*ignored*/, std::move(input_channel),
                                                 message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_exportMessageLink>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(DEBUG) << "Receive result for ExportChannelMessageLinkQuery: " << to_string(ptr);
    if (!ignore_result_) {
      td_->messages_manager_->on_get_public_message_link({DialogId(channel_id_), message_id_}, for_group_,
                                                         std::move(ptr->link_), std::move(ptr->html_));
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!ignore_result_) {
      td_->messages_manager_->on_get_message_error(DialogId(channel_id_), message_id_, status,
                                                   "ExportChannelMessageLinkQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogListQuery final : public Td::ResultHandler {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetDialogListQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, int32 offset_date, ServerMessageId offset_message_id, DialogId offset_dialog_id,
            int32 limit) {
    folder_id_ = folder_id;
    auto input_peer = DialogManager::get_input_peer_force(offset_dialog_id);
    CHECK(input_peer != nullptr);

    int32 flags =
        telegram_api::messages_getDialogs::EXCLUDE_PINNED_MASK | telegram_api::messages_getDialogs::FOLDER_ID_MASK;
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getDialogs(flags, false /*ignored*/, folder_id.get(), offset_date,
                                          offset_message_id.get(), std::move(input_peer), limit, 0),
        {{folder_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive chats from chat list of " << folder_id_ << ": " << to_string(ptr);
    switch (ptr->get_id()) {
      case telegram_api::messages_dialogs::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogs>(ptr);
        td_->user_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListQuery");
        td_->chat_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListQuery");
        td_->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_),
                                               narrow_cast<int32>(dialogs->dialogs_.size()),
                                               std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsSlice::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogsSlice>(ptr);
        td_->user_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListQuery slice");
        td_->chat_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListQuery slice");
        td_->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_), max(dialogs->count_, 0),
                                               std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsNotModified::ID:
        LOG(ERROR) << "Receive " << to_string(ptr);
        return on_error(Status::Error(500, "Receive wrong server response messages.dialogsNotModified"));
      default:
        UNREACHABLE();
    }
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class SearchPublicDialogsQuery final : public Td::ResultHandler {
  string query_;

 public:
  void send(const string &query) {
    query_ = query;
    send_query(
        G()->net_query_creator().create(telegram_api::contacts_search(query, 20 /* mostly ignored server-side */)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto dialogs = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SearchPublicDialogsQuery: " << to_string(dialogs);
    td_->user_manager_->on_get_users(std::move(dialogs->users_), "SearchPublicDialogsQuery");
    td_->chat_manager_->on_get_chats(std::move(dialogs->chats_), "SearchPublicDialogsQuery");
    td_->messages_manager_->on_get_public_dialogs_search_result(query_, std::move(dialogs->my_results_),
                                                                std::move(dialogs->results_));
  }

  void on_error(Status status) final {
    if (!G()->is_expected_error(status)) {
      if (status.message() == "QUERY_TOO_SHORT") {
        return td_->messages_manager_->on_get_public_dialogs_search_result(query_, {}, {});
      }
      LOG(ERROR) << "Receive error for SearchPublicDialogsQuery: " << status;
    }
    td_->messages_manager_->on_failed_public_dialogs_search(query_, std::move(status));
  }
};

class GetBlockedDialogsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messageSenders>> promise_;
  int32 offset_;
  int32 limit_;

 public:
  explicit GetBlockedDialogsQuery(Promise<td_api::object_ptr<td_api::messageSenders>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(BlockListId block_list_id, int32 offset, int32 limit) {
    offset_ = offset;
    limit_ = limit;

    int32 flags = 0;
    if (block_list_id == BlockListId::stories()) {
      flags |= telegram_api::contacts_getBlocked::MY_STORIES_FROM_MASK;
    }

    send_query(
        G()->net_query_creator().create(telegram_api::contacts_getBlocked(flags, false /*ignored*/, offset, limit)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_getBlocked>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetBlockedDialogsQuery: " << to_string(ptr);

    switch (ptr->get_id()) {
      case telegram_api::contacts_blocked::ID: {
        auto blocked_peers = move_tl_object_as<telegram_api::contacts_blocked>(ptr);

        td_->user_manager_->on_get_users(std::move(blocked_peers->users_), "GetBlockedDialogsQuery");
        td_->chat_manager_->on_get_chats(std::move(blocked_peers->chats_), "GetBlockedDialogsQuery");
        td_->messages_manager_->on_get_blocked_dialogs(offset_, limit_,
                                                       narrow_cast<int32>(blocked_peers->blocked_.size()),
                                                       std::move(blocked_peers->blocked_), std::move(promise_));
        break;
      }
      case telegram_api::contacts_blockedSlice::ID: {
        auto blocked_peers = move_tl_object_as<telegram_api::contacts_blockedSlice>(ptr);

        td_->user_manager_->on_get_users(std::move(blocked_peers->users_), "GetBlockedDialogsQuery slice");
        td_->chat_manager_->on_get_chats(std::move(blocked_peers->chats_), "GetBlockedDialogsQuery slice");
        td_->messages_manager_->on_get_blocked_dialogs(offset_, limit_, blocked_peers->count_,
                                                       std::move(blocked_peers->blocked_), std::move(promise_));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class SetChatAvailableReactionsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetChatAvailableReactionsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const ChatReactions &available_reactions) {
    dialog_id_ = dialog_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_setChatAvailableReactions(
        std::move(input_peer), available_reactions.get_input_chat_reactions())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setChatAvailableReactions>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SetChatAvailableReactionsQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td_->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SetChatAvailableReactionsQuery");
      td_->dialog_manager_->reload_dialog_info_full(dialog_id_, "SetChatAvailableReactionsQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SetChatThemeQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetChatThemeQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &theme_name) {
    dialog_id_ = dialog_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_setChatTheme(std::move(input_peer), theme_name)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setChatTheme>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SetChatThemeQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td_->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SetChatThemeQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SetHistoryTtlQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetHistoryTtlQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 period) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::messages_setHistoryTTL(std::move(input_peer), period)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setHistoryTTL>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SetHistoryTtlQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td_->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SetHistoryTtlQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogPinQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_pinned_;

 public:
  explicit ToggleDialogPinQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_pinned) {
    dialog_id_ = dialog_id;
    is_pinned_ = is_pinned;

    auto input_peer = td_->dialog_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (is_pinned) {
      flags |= telegram_api::messages_toggleDialogPin::PINNED_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_toggleDialogPin(flags, false /*ignored*/, std::move(input_peer)), {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_toggleDialogPin>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(Status::Error(400, "Toggle dialog pin failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogPinQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogPinQuery: " << status;
    }
    td_->messages_manager_->on_update_pinned_dialogs(FolderId::main());
    td_->messages_manager_->on_update_pinned_dialogs(FolderId::archive());
    promise_.set_error(std::move(status));
  }
};

class ReorderPinnedDialogsQuery final : public Td::ResultHandler {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit ReorderPinnedDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, const vector<DialogId> &dialog_ids) {
    folder_id_ = folder_id;
    int32 flags = telegram_api::messages_reorderPinnedDialogs::FORCE_MASK;
    send_query(G()->net_query_creator().create(telegram_api::messages_reorderPinnedDialogs(
        flags, true /*ignored*/, folder_id.get(),
        td_->dialog_manager_->get_input_dialog_peers(dialog_ids, AccessRights::Read))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_reorderPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.move_as_ok();
    if (!result) {
      return on_error(Status::Error(400, "Result is false"));
    }
    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for ReorderPinnedDialogsQuery: " << status;
    }
    td_->messages_manager_->on_update_pinned_dialogs(folder_id_);
    promise_.set_error(std::move(status));
  }
};

class ToggleViewForumAsMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool view_as_messages_;

 public:
  explicit ToggleViewForumAsMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool view_as_messages) {
    dialog_id_ = dialog_id;
    view_as_messages_ = view_as_messages;

    CHECK(dialog_id.get_type() == DialogType::Channel);
    auto input_channel = td_->chat_manager_->get_input_channel(dialog_id.get_channel_id());
    CHECK(input_channel != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::channels_toggleViewForumAsMessages(std::move(input_channel), view_as_messages), {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_toggleViewForumAsMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for ToggleViewForumAsMessagesQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ToggleViewForumAsMessagesQuery")) {
      LOG(ERROR) << "Receive error for ToggleViewForumAsMessagesQuery: " << status;
    }
    if (!G()->close_flag()) {
      td_->messages_manager_->on_update_dialog_view_as_messages(dialog_id_, !view_as_messages_);
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogUnreadMarkQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_marked_as_unread_;

 public:
  explicit ToggleDialogUnreadMarkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_marked_as_unread) {
    dialog_id_ = dialog_id;
    is_marked_as_unread_ = is_marked_as_unread;

    auto input_peer = td_->dialog_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (is_marked_as_unread) {
      flags |= telegram_api::messages_markDialogUnread::UNREAD_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_markDialogUnread(flags, false /*ignored*/, std::move(input_peer)), {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_markDialogUnread>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(Status::Error(400, "Toggle dialog mark failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogUnreadMarkQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogUnreadMarkQuery: " << status;
    }
    if (!G()->close_flag()) {
      td_->messages_manager_->on_update_dialog_is_marked_as_unread(dialog_id_, !is_marked_as_unread_);
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogTranslationsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_translatable_;

 public:
  explicit ToggleDialogTranslationsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_translatable) {
    dialog_id_ = dialog_id;
    is_translatable_ = is_translatable;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (!is_translatable) {
      flags |= telegram_api::messages_togglePeerTranslations::DISABLED_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_togglePeerTranslations(flags, false /*ignored*/, std::move(input_peer)), {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_togglePeerTranslations>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(Status::Error(400, "Toggle dialog translations failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogTranslationsQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogTranslationsQuery: " << status;
    }
    if (!G()->close_flag()) {
      td_->messages_manager_->on_update_dialog_is_translatable(dialog_id_, !is_translatable_);
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogIsBlockedQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ToggleDialogIsBlockedQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_blocked, bool is_blocked_for_stories) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Know);
    CHECK(input_peer != nullptr && input_peer->get_id() != telegram_api::inputPeerEmpty::ID);

    int32 flags = 0;
    if (is_blocked_for_stories) {
      flags |= telegram_api::contacts_block::MY_STORIES_FROM_MASK;
    }
    vector<ChainId> chain_ids{{dialog_id, MessageContentType::Photo}, {dialog_id, MessageContentType::Text}};
    auto query =
        is_blocked || is_blocked_for_stories
            ? G()->net_query_creator().create(
                  telegram_api::contacts_block(flags, false /*ignored*/, std::move(input_peer)), std::move(chain_ids))
            : G()->net_query_creator().create(
                  telegram_api::contacts_unblock(flags, false /*ignored*/, std::move(input_peer)),
                  std::move(chain_ids));
    send_query(std::move(query));
  }

  void on_result(BufferSlice packet) final {
    static_assert(
        std::is_same<telegram_api::contacts_block::ReturnType, telegram_api::contacts_unblock::ReturnType>::value, "");
    auto result_ptr = fetch_result<telegram_api::contacts_block>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(WARNING, !result) << "Block/Unblock " << dialog_id_ << " has failed";

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogIsBlockedQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogIsBlockedQuery: " << status;
    }
    if (!G()->close_flag()) {
      td_->dialog_manager_->get_dialog_info_full(dialog_id_, Auto(), "ToggleDialogIsBlockedQuery");
      td_->messages_manager_->reget_dialog_action_bar(dialog_id_, "ToggleDialogIsBlockedQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class GetMessagesViewsQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

 public:
  void send(DialogId dialog_id, vector<MessageId> &&message_ids, bool increment_view_counter) {
    dialog_id_ = dialog_id;
    message_ids_ = std::move(message_ids);

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_getMessagesViews(
        std::move(input_peer), MessageId::get_server_message_ids(message_ids_), increment_view_counter)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessagesViews>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    auto interaction_infos = std::move(result->views_);
    if (message_ids_.size() != interaction_infos.size()) {
      return on_error(Status::Error(500, "Wrong number of message views returned"));
    }
    td_->user_manager_->on_get_users(std::move(result->users_), "GetMessagesViewsQuery");
    td_->chat_manager_->on_get_chats(std::move(result->chats_), "GetMessagesViewsQuery");
    for (size_t i = 0; i < message_ids_.size(); i++) {
      MessageFullId message_full_id{dialog_id_, message_ids_[i]};

      auto *info = interaction_infos[i].get();
      auto flags = info->flags_;
      auto view_count = (flags & telegram_api::messageViews::VIEWS_MASK) != 0 ? info->views_ : 0;
      auto forward_count = (flags & telegram_api::messageViews::FORWARDS_MASK) != 0 ? info->forwards_ : 0;
      td_->messages_manager_->on_update_message_interaction_info(message_full_id, view_count, forward_count, true,
                                                                 std::move(info->replies_));
    }
    td_->messages_manager_->finish_get_message_views(dialog_id_, message_ids_);
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetMessagesViewsQuery")) {
      LOG(ERROR) << "Receive error for GetMessagesViewsQuery: " << status;
    }
    td_->messages_manager_->finish_get_message_views(dialog_id_, message_ids_);
  }
};

class GetExtendedMediaQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

 public:
  void send(DialogId dialog_id, vector<MessageId> &&message_ids) {
    dialog_id_ = dialog_id;
    message_ids_ = std::move(message_ids);

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_getExtendedMedia(
        std::move(input_peer), MessageId::get_server_message_ids(message_ids_))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getExtendedMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetExtendedMediaQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
    td_->messages_manager_->finish_get_message_extended_media(dialog_id_, message_ids_);
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetExtendedMediaQuery");
    td_->messages_manager_->finish_get_message_extended_media(dialog_id_, message_ids_);
  }
};

class ReadMessagesContentsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ReadMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<MessageId> &&message_ids) {
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readMessageContents(MessageId::get_server_message_ids(message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for ReadMessagesContentsQuery: " << to_string(affected_messages);

    if (affected_messages->pts_count_ > 0) {
      td_->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                    affected_messages->pts_count_, Time::now(), Promise<Unit>(),
                                                    "read messages content query");
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for read message contents: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelMessagesContentsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<MessageId> &&message_ids) {
    channel_id_ = channel_id;

    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Have no input channel for " << channel_id;
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    send_query(
        G()->net_query_creator().create(telegram_api::channels_readMessageContents(
                                            std::move(input_channel), MessageId::get_server_message_ids(message_ids)),
                                        {{channel_id_}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(ERROR, !result) << "Read channel messages contents failed";

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->chat_manager_->on_get_channel_error(channel_id_, status, "ReadChannelMessagesContentsQuery")) {
      LOG(ERROR) << "Receive error for read messages contents in " << channel_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogMessageByDateQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 date_;
  int64 random_id_;

 public:
  explicit GetDialogMessageByDateQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 date, int64 random_id) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    date_ = date;
    random_id_ = random_id;

    send_query(G()->net_query_creator().create(
        telegram_api::messages_getHistory(std::move(input_peer), 0, date, -3, 5, 0, 0, 0)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "GetDialogMessageByDateQuery");
    td_->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), dialog_id = dialog_id_, date = date_,
                                random_id = random_id_,
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_dialog_message_by_date_success, dialog_id, date, random_id,
                         std::move(info.messages), std::move(promise));
          }
        }),
        "GetDialogMessageByDateQuery");
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogMessageByDateQuery")) {
      LOG(ERROR) << "Receive error for GetDialogMessageByDateQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
    td_->messages_manager_->on_get_dialog_message_by_date_fail(random_id_);
  }
};

class GetHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId from_message_id_;
  MessageId old_last_new_message_id_;
  int32 offset_;
  int32 limit_;
  bool from_the_end_;

 public:
  explicit GetHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId from_message_id, MessageId old_last_new_message_id, int32 offset,
            int32 limit) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }
    CHECK(offset < 0);

    dialog_id_ = dialog_id;
    from_message_id_ = from_message_id;
    old_last_new_message_id_ = old_last_new_message_id;
    offset_ = offset;
    limit_ = limit;
    from_the_end_ = false;
    send_query(G()->net_query_creator().create(telegram_api::messages_getHistory(
        std::move(input_peer), from_message_id.get_server_message_id().get(), 0, offset, limit, 0, 0, 0)));
  }

  void send_get_from_the_end(DialogId dialog_id, MessageId old_last_new_message_id, int32 limit) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    old_last_new_message_id_ = old_last_new_message_id;
    offset_ = 0;
    limit_ = limit;
    from_the_end_ = true;
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getHistory(std::move(input_peer), 0, 0, 0, limit, 0, 0, 0)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "GetHistoryQuery");
    td_->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), dialog_id = dialog_id_,
                                from_message_id = from_message_id_, old_last_new_message_id = old_last_new_message_id_,
                                offset = offset_, limit = limit_, from_the_end = from_the_end_,
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            // TODO use info.total_count, info.pts
            send_closure(actor_id, &MessagesManager::on_get_history, dialog_id, from_message_id,
                         old_last_new_message_id, offset, limit, from_the_end, std::move(info.messages),
                         std::move(promise));
          }
        }),
        "GetHistoryQuery");
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetHistoryQuery")) {
      LOG(ERROR) << "Receive error for GetHistoryQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readHistory(std::move(input_peer), max_message_id.get_server_message_id().get()),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for ReadHistoryQuery: " << to_string(affected_messages);

    if (affected_messages->pts_count_ > 0) {
      td_->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                    affected_messages->pts_count_, Time::now(), Promise<Unit>(),
                                                    "read history query");
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ReadHistoryQuery")) {
      LOG(ERROR) << "Receive error for ReadHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id) {
    channel_id_ = channel_id;
    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(
        telegram_api::channels_readHistory(std::move(input_channel), max_message_id.get_server_message_id().get()),
        {{DialogId(channel_id)}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    if (!td_->chat_manager_->on_get_channel_error(channel_id_, status, "ReadChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for ReadChannelHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadDiscussionQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadDiscussionQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id, MessageId max_message_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readDiscussion(std::move(input_peer),
                                              top_thread_message_id.get_server_message_id().get(),
                                              max_message_id.get_server_message_id().get()),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readDiscussion>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ReadDiscussionQuery");
    promise_.set_error(std::move(status));
  }
};

class GetSearchResultCalendarQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  SavedMessagesTopicId saved_messages_topic_id_;
  MessageId from_message_id_;
  MessageSearchFilter filter_;
  int64 random_id_;

 public:
  explicit GetSearchResultCalendarQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageId from_message_id,
            MessageSearchFilter filter, int64 random_id) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    saved_messages_topic_id_ = saved_messages_topic_id;
    from_message_id_ = from_message_id;
    filter_ = filter;
    random_id_ = random_id;

    int32 flags = 0;
    telegram_api::object_ptr<telegram_api::InputPeer> saved_input_peer;
    if (saved_messages_topic_id.is_valid()) {
      flags |= telegram_api::messages_getSearchResultsCalendar::SAVED_PEER_ID_MASK;
      saved_input_peer = saved_messages_topic_id.get_input_peer(td_);
      CHECK(saved_input_peer != nullptr);
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_getSearchResultsCalendar(
        flags, std::move(input_peer), std::move(saved_input_peer), get_input_messages_filter(filter),
        from_message_id.get_server_message_id().get(), 0)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchResultsCalendar>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetSearchResultCalendarQuery: " << to_string(result);
    td_->user_manager_->on_get_users(std::move(result->users_), "GetSearchResultCalendarQuery");
    td_->chat_manager_->on_get_chats(std::move(result->chats_), "GetSearchResultCalendarQuery");

    // unused: inexact:flags.0?true min_date:int min_msg_id:int offset_id_offset:flags.1?int

    MessagesInfo info;
    info.messages = std::move(result->messages_);
    info.total_count = result->count_;
    info.is_channel_messages = dialog_id_.get_type() == DialogType::Channel;

    td_->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), dialog_id = dialog_id_,
                                saved_messages_topic_id = saved_messages_topic_id_, from_message_id = from_message_id_,
                                filter = filter_, random_id = random_id_, periods = std::move(result->periods_),
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_message_search_result_calendar, dialog_id,
                         saved_messages_topic_id, from_message_id, filter, random_id, info.total_count,
                         std::move(info.messages), std::move(periods), std::move(promise));
          }
        }),
        "GetSearchResultCalendarQuery");
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetSearchResultCalendarQuery");
    td_->messages_manager_->on_failed_get_message_search_result_calendar(random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetMessagePositionQuery final : public Td::ResultHandler {
  Promise<int32> promise_;
  DialogId dialog_id_;
  MessageId message_id_;
  MessageId top_thread_message_id_;
  SavedMessagesTopicId saved_messages_topic_id_;
  MessageSearchFilter filter_;

 public:
  explicit GetMessagePositionQuery(Promise<int32> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, MessageSearchFilter filter, MessageId top_thread_message_id,
            SavedMessagesTopicId saved_messages_topic_id) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    message_id_ = message_id;
    top_thread_message_id_ = top_thread_message_id;
    saved_messages_topic_id_ = saved_messages_topic_id;
    filter_ = filter;

    if (filter == MessageSearchFilter::Empty && !top_thread_message_id.is_valid()) {
      if (saved_messages_topic_id.is_valid()) {
        send_query(G()->net_query_creator().create(telegram_api::messages_getSavedHistory(
            saved_messages_topic_id.get_input_peer(td_), message_id.get_server_message_id().get(), 0, -1, 1, 0, 0, 0)));
      } else {
        send_query(G()->net_query_creator().create(telegram_api::messages_getHistory(
            std::move(input_peer), message_id.get_server_message_id().get(), 0, -1, 1, 0, 0, 0)));
      }
    } else {
      int32 flags = 0;
      tl_object_ptr<telegram_api::InputPeer> saved_input_peer;
      if (saved_messages_topic_id.is_valid()) {
        flags |= telegram_api::messages_search::SAVED_PEER_ID_MASK;
        saved_input_peer = saved_messages_topic_id.get_input_peer(td_);
        CHECK(saved_input_peer != nullptr);
      }
      if (top_thread_message_id.is_valid()) {
        flags |= telegram_api::messages_search::TOP_MSG_ID_MASK;
      }
      send_query(G()->net_query_creator().create(telegram_api::messages_search(
          flags, std::move(input_peer), string(), nullptr, std::move(saved_input_peer), Auto(),
          top_thread_message_id.get_server_message_id().get(), get_input_messages_filter(filter), 0,
          std::numeric_limits<int32>::max(), message_id.get_server_message_id().get(), -1, 1,
          std::numeric_limits<int32>::max(), 0, 0)));
    }
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto messages_ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetMessagePositionQuery: " << to_string(messages_ptr);
    switch (messages_ptr->get_id()) {
      case telegram_api::messages_messages::ID: {
        auto messages = move_tl_object_as<telegram_api::messages_messages>(messages_ptr);
        if (messages->messages_.size() != 1 ||
            MessageId::get_message_id(messages->messages_[0], false) != message_id_) {
          return promise_.set_error(Status::Error(400, "Message not found by the filter"));
        }
        return promise_.set_value(narrow_cast<int32>(messages->messages_.size()));
      }
      case telegram_api::messages_messagesSlice::ID: {
        auto messages = move_tl_object_as<telegram_api::messages_messagesSlice>(messages_ptr);
        if (messages->messages_.size() != 1 ||
            MessageId::get_message_id(messages->messages_[0], false) != message_id_) {
          return promise_.set_error(Status::Error(400, "Message not found by the filter"));
        }
        if (messages->offset_id_offset_ <= 0) {
          LOG(ERROR) << "Failed to receive position for " << message_id_ << " in thread of " << top_thread_message_id_
                     << " and in " << saved_messages_topic_id_ << " in " << dialog_id_ << " by " << filter_;
          return promise_.set_error(Status::Error(400, "Message position is unknown"));
        }
        return promise_.set_value(std::move(messages->offset_id_offset_));
      }
      case telegram_api::messages_channelMessages::ID: {
        auto messages = move_tl_object_as<telegram_api::messages_channelMessages>(messages_ptr);
        if (messages->messages_.size() != 1 ||
            MessageId::get_message_id(messages->messages_[0], false) != message_id_) {
          return promise_.set_error(Status::Error(400, "Message not found by the filter"));
        }
        if (messages->offset_id_offset_ <= 0) {
          LOG(ERROR) << "Failed to receive position for " << message_id_ << " in " << dialog_id_ << " by " << filter_;
          return promise_.set_error(Status::Error(500, "Message position is unknown"));
        }
        return promise_.set_value(std::move(messages->offset_id_offset_));
      }
      case telegram_api::messages_messagesNotModified::ID:
        LOG(ERROR) << "Server returned messagesNotModified in response to GetMessagePositionQuery";
        return promise_.set_error(Status::Error(500, "Receive invalid response"));
      default:
        UNREACHABLE();
        break;
    }
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetMessagePositionQuery");
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  SavedMessagesTopicId saved_messages_topic_id_;
  string query_;
  DialogId sender_dialog_id_;
  MessageId from_message_id_;
  int32 offset_;
  int32 limit_;
  MessageSearchFilter filter_;
  MessageId top_thread_message_id_;
  ReactionType tag_;
  int64 random_id_;
  bool handle_errors_ = true;

 public:
  explicit SearchMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, const string &query,
            DialogId sender_dialog_id, MessageId from_message_id, int32 offset, int32 limit, MessageSearchFilter filter,
            MessageId top_thread_message_id, const ReactionType &tag, int64 random_id) {
    auto input_peer = dialog_id.is_valid() ? td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read)
                                           : make_tl_object<telegram_api::inputPeerEmpty>();
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    saved_messages_topic_id_ = saved_messages_topic_id;
    query_ = query;
    sender_dialog_id_ = sender_dialog_id;
    from_message_id_ = from_message_id;
    offset_ = offset;
    limit_ = limit;
    filter_ = filter;
    top_thread_message_id_ = top_thread_message_id;
    tag_ = tag;
    random_id_ = random_id;

    auto top_msg_id = top_thread_message_id.get_server_message_id().get();
    auto offset_id = from_message_id.get_server_message_id().get();
    if (filter == MessageSearchFilter::UnreadMention) {
      CHECK(!saved_messages_topic_id.is_valid());
      CHECK(tag_.is_empty());
      int32 flags = 0;
      if (top_thread_message_id.is_valid()) {
        flags |= telegram_api::messages_getUnreadMentions::TOP_MSG_ID_MASK;
      }
      send_query(G()->net_query_creator().create(telegram_api::messages_getUnreadMentions(
          flags, std::move(input_peer), top_msg_id, offset_id, offset, limit, std::numeric_limits<int32>::max(), 0)));
    } else if (filter == MessageSearchFilter::UnreadReaction) {
      CHECK(!saved_messages_topic_id.is_valid());
      CHECK(tag_.is_empty());
      int32 flags = 0;
      if (top_thread_message_id.is_valid()) {
        flags |= telegram_api::messages_getUnreadReactions::TOP_MSG_ID_MASK;
      }
      send_query(G()->net_query_creator().create(telegram_api::messages_getUnreadReactions(
          flags, std::move(input_peer), top_msg_id, offset_id, offset, limit, std::numeric_limits<int32>::max(), 0)));
    } else if (top_thread_message_id.is_valid() && query.empty() && !sender_dialog_id.is_valid() &&
               filter == MessageSearchFilter::Empty) {
      CHECK(!saved_messages_topic_id.is_valid());
      CHECK(tag_.is_empty());
      handle_errors_ = dialog_id.get_type() != DialogType::Channel ||
                       !td_->chat_manager_->is_broadcast_channel(dialog_id.get_channel_id());
      send_query(G()->net_query_creator().create(telegram_api::messages_getReplies(
          std::move(input_peer), top_msg_id, offset_id, 0, offset, limit, std::numeric_limits<int32>::max(), 0, 0)));
    } else {
      int32 flags = 0;
      tl_object_ptr<telegram_api::InputPeer> sender_input_peer;
      if (sender_dialog_id.is_valid()) {
        flags |= telegram_api::messages_search::FROM_ID_MASK;
        sender_input_peer = td_->dialog_manager_->get_input_peer(sender_dialog_id, AccessRights::Know);
        CHECK(sender_input_peer != nullptr);
      }
      tl_object_ptr<telegram_api::InputPeer> saved_input_peer;
      if (saved_messages_topic_id.is_valid()) {
        flags |= telegram_api::messages_search::SAVED_PEER_ID_MASK;
        saved_input_peer = saved_messages_topic_id.get_input_peer(td_);
        CHECK(saved_input_peer != nullptr);
      }
      vector<telegram_api::object_ptr<telegram_api::Reaction>> saved_reactions;
      if (!tag.is_empty()) {
        flags |= telegram_api::messages_search::SAVED_REACTION_MASK;
        saved_reactions.push_back(tag.get_input_reaction());
      }
      if (top_thread_message_id.is_valid()) {
        flags |= telegram_api::messages_search::TOP_MSG_ID_MASK;
      }

      send_query(G()->net_query_creator().create(telegram_api::messages_search(
          flags, std::move(input_peer), query, std::move(sender_input_peer), std::move(saved_input_peer),
          std::move(saved_reactions), top_msg_id, get_input_messages_filter(filter), 0,
          std::numeric_limits<int32>::max(), offset_id, offset, limit, std::numeric_limits<int32>::max(), 0, 0)));
    }
  }

  void on_result(BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_getUnreadMentions::ReturnType,
                               telegram_api::messages_search::ReturnType>::value,
                  "");
    static_assert(std::is_same<telegram_api::messages_getUnreadReactions::ReturnType,
                               telegram_api::messages_search::ReturnType>::value,
                  "");
    static_assert(
        std::is_same<telegram_api::messages_getReplies::ReturnType, telegram_api::messages_search::ReturnType>::value,
        "");
    auto result_ptr = fetch_result<telegram_api::messages_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "SearchMessagesQuery");
    td_->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda(
            [actor_id = td_->messages_manager_actor_.get(), dialog_id = dialog_id_,
             saved_messages_topic_id = saved_messages_topic_id_, query = std::move(query_),
             sender_dialog_id = sender_dialog_id_, from_message_id = from_message_id_, offset = offset_, limit = limit_,
             filter = filter_, top_thread_message_id = top_thread_message_id_, tag = std::move(tag_),
             random_id = random_id_, promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
              if (result.is_error()) {
                promise.set_error(result.move_as_error());
              } else {
                auto info = result.move_as_ok();
                send_closure(actor_id, &MessagesManager::on_get_dialog_messages_search_result, dialog_id,
                             saved_messages_topic_id, query, sender_dialog_id, from_message_id, offset, limit, filter,
                             top_thread_message_id, tag, random_id, info.total_count, std::move(info.messages),
                             std::move(promise));
              }
            }),
        "SearchMessagesQuery");
  }

  void on_error(Status status) final {
    if (handle_errors_) {
      td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SearchMessagesQuery");
    }
    td_->messages_manager_->on_failed_dialog_messages_search(dialog_id_, random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetSearchResultPositionsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messagePositions>> promise_;
  DialogId dialog_id_;
  SavedMessagesTopicId saved_messages_topic_id_;
  MessageSearchFilter filter_;

 public:
  explicit GetSearchResultPositionsQuery(Promise<td_api::object_ptr<td_api::messagePositions>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageSearchFilter filter,
            MessageId from_message_id, int32 limit) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }
    dialog_id_ = dialog_id;
    saved_messages_topic_id_ = saved_messages_topic_id;
    filter_ = filter;

    int32 flags = 0;
    telegram_api::object_ptr<telegram_api::InputPeer> saved_input_peer;
    if (saved_messages_topic_id.is_valid()) {
      flags |= telegram_api::messages_getSearchResultsPositions::SAVED_PEER_ID_MASK;
      saved_input_peer = saved_messages_topic_id.get_input_peer(td_);
      CHECK(saved_input_peer != nullptr);
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_getSearchResultsPositions(
        flags, std::move(input_peer), std::move(saved_input_peer), get_input_messages_filter(filter),
        from_message_id.get_server_message_id().get(), limit)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchResultsPositions>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    td_->messages_manager_->on_get_dialog_sparse_message_positions(dialog_id_, saved_messages_topic_id_, filter_,
                                                                   result_ptr.move_as_ok(), std::move(promise_));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetSearchResultPositionsQuery");
    promise_.set_error(std::move(status));
  }
};

class GetSearchCountersQuery final : public Td::ResultHandler {
  Promise<int32> promise_;
  DialogId dialog_id_;
  SavedMessagesTopicId saved_messages_topic_id_;
  MessageSearchFilter filter_;

 public:
  explicit GetSearchCountersQuery(Promise<int32> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageSearchFilter filter) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    saved_messages_topic_id_ = saved_messages_topic_id;
    filter_ = filter;

    CHECK(filter != MessageSearchFilter::Empty);
    CHECK(filter != MessageSearchFilter::UnreadMention);
    CHECK(filter != MessageSearchFilter::FailedToSend);
    CHECK(filter != MessageSearchFilter::UnreadReaction);
    vector<telegram_api::object_ptr<telegram_api::MessagesFilter>> filters;
    filters.push_back(get_input_messages_filter(filter));

    int32 flags = 0;
    telegram_api::object_ptr<telegram_api::InputPeer> saved_input_peer;
    if (saved_messages_topic_id.is_valid()) {
      flags |= telegram_api::messages_getSearchCounters::SAVED_PEER_ID_MASK;
      saved_input_peer = saved_messages_topic_id.get_input_peer(td_);
      CHECK(saved_input_peer != nullptr);
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_getSearchCounters(
        flags, std::move(input_peer), std::move(saved_input_peer), 0, std::move(filters))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchCounters>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    if (result.size() != 1 || result[0]->filter_->get_id() != get_input_messages_filter(filter_)->get_id()) {
      LOG(ERROR) << "Receive unexpected response for get message count in " << dialog_id_ << " with filter " << filter_
                 << ": " << to_string(result);
      return on_error(Status::Error(500, "Receive wrong response"));
    }

    td_->messages_manager_->on_get_dialog_message_count(dialog_id_, saved_messages_topic_id_, filter_,
                                                        result[0]->count_, std::move(promise_));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetSearchCountersQuery");
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesGlobalQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  string query_;
  int32 offset_date_;
  DialogId offset_dialog_id_;
  MessageId offset_message_id_;
  int32 limit_;
  MessageSearchFilter filter_;
  int32 min_date_;
  int32 max_date_;
  int64 random_id_;

 public:
  explicit SearchMessagesGlobalQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, bool ignore_folder_id, const string &query, int32 offset_date,
            DialogId offset_dialog_id, MessageId offset_message_id, int32 limit, MessageSearchFilter filter,
            int32 min_date, int32 max_date, int64 random_id) {
    query_ = query;
    offset_date_ = offset_date;
    offset_dialog_id_ = offset_dialog_id;
    offset_message_id_ = offset_message_id;
    limit_ = limit;
    random_id_ = random_id;
    filter_ = filter;
    min_date_ = min_date;
    max_date_ = max_date;

    auto input_peer = DialogManager::get_input_peer_force(offset_dialog_id);
    CHECK(input_peer != nullptr);

    int32 flags = 0;
    if (!ignore_folder_id) {
      flags |= telegram_api::messages_searchGlobal::FOLDER_ID_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_searchGlobal(
        flags, folder_id.get(), query, get_input_messages_filter(filter), min_date_, max_date_, offset_date_,
        std::move(input_peer), offset_message_id.get_server_message_id().get(), limit)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_searchGlobal>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, DialogId(), result_ptr.move_as_ok(), "SearchMessagesGlobalQuery");
    td_->messages_manager_->get_channel_differences_if_needed(
        std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), query = std::move(query_),
                                offset_date = offset_date_, offset_dialog_id = offset_dialog_id_,
                                offset_message_id = offset_message_id_, limit = limit_, filter = std::move(filter_),
                                min_date = min_date_, max_date = max_date_, random_id = random_id_,
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_messages_search_result, query, offset_date,
                         offset_dialog_id, offset_message_id, limit, filter, min_date, max_date, random_id,
                         info.total_count, std::move(info.messages), info.next_rate, std::move(promise));
          }
        }),
        "SearchMessagesGlobalQuery");
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_failed_messages_search(random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetAllScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  uint32 generation_;

 public:
  explicit GetAllScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 hash, uint32 generation) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    generation_ = generation;

    send_query(
        G()->net_query_creator().create(telegram_api::messages_getScheduledHistory(std::move(input_peer), hash)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    if (result_ptr.ok()->get_id() == telegram_api::messages_messagesNotModified::ID) {
      td_->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, Auto(), true);
    } else {
      auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "GetAllScheduledMessagesQuery");
      td_->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, std::move(info.messages),
                                                               false);
    }

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetAllScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class SearchSentMediaQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::foundMessages>> promise_;

 public:
  explicit SearchSentMediaQuery(Promise<td_api::object_ptr<td_api::foundMessages>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(const string &query, int32 limit) {
    send_query(G()->net_query_creator().create(telegram_api::messages_searchSentMedia(
        query, telegram_api::make_object<telegram_api::inputMessagesFilterDocument>(), limit)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_searchSentMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, DialogId(), result_ptr.move_as_ok(), "SearchSentMediaQuery");
    td_->messages_manager_->get_channel_differences_if_needed(
        std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(),
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_outgoing_document_messages, std::move(info.messages),
                         std::move(promise));
          }
        }),
        "SearchSentMediaQuery");
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetRecentLocationsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messages>> promise_;
  DialogId dialog_id_;
  int32 limit_;

 public:
  explicit GetRecentLocationsQuery(Promise<td_api::object_ptr<td_api::messages>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 limit) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Chat is not accessible"));
    }

    dialog_id_ = dialog_id;
    limit_ = limit;

    send_query(
        G()->net_query_creator().create(telegram_api::messages_getRecentLocations(std::move(input_peer), limit, 0)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getRecentLocations>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto info = get_messages_info(td_, dialog_id_, result_ptr.move_as_ok(), "GetRecentLocationsQuery");
    td_->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td_->messages_manager_actor_.get(), dialog_id = dialog_id_, limit = limit_,
                                promise = std::move(promise_)](Result<MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_recent_locations, dialog_id, limit, info.total_count,
                         std::move(info.messages), std::move(promise));
          }
        }),
        "GetRecentLocationsQuery");
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetRecentLocationsQuery");
    promise_.set_error(std::move(status));
  }
};

class HidePromoDataQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::help_hidePromoData(std::move(input_peer))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::help_hidePromoData>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    // we are not interested in the result
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "HidePromoDataQuery")) {
      LOG(ERROR) << "Receive error for sponsored chat hiding: " << status;
    }
  }
};

class DeleteHistoryQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  DialogId dialog_id_;

 public:
  explicit DeleteHistoryQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id, bool remove_from_dialog_list, bool revoke) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = 0;
    if (!remove_from_dialog_list) {
      flags |= telegram_api::messages_deleteHistory::JUST_CLEAR_MASK;
    }
    if (revoke) {
      flags |= telegram_api::messages_deleteHistory::REVOKE_MASK;
    }

    send_query(G()->net_query_creator().create(
        telegram_api::messages_deleteHistory(flags, false /*ignored*/, false /*ignored*/, std::move(input_peer),
                                             max_message_id.get_server_message_id().get(), 0, 0)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "DeleteHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class DeleteTopicHistoryQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  ChannelId channel_id_;
  MessageId top_thread_message_id_;

 public:
  explicit DeleteTopicHistoryQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id) {
    CHECK(dialog_id.get_type() == DialogType::Channel);
    channel_id_ = dialog_id.get_channel_id();
    top_thread_message_id_ = top_thread_message_id;

    auto input_channel = td_->chat_manager_->get_input_channel(channel_id_);
    if (input_channel == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(telegram_api::channels_deleteTopicHistory(
        std::move(input_channel), top_thread_message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteTopicHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->messages_manager_->on_get_message_error(DialogId(channel_id_), top_thread_message_id_, status,
                                                 "DeleteTopicHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId max_message_id_;
  bool allow_error_;

 public:
  explicit DeleteChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id, bool allow_error, bool revoke) {
    channel_id_ = channel_id;
    max_message_id_ = max_message_id;
    allow_error_ = allow_error;
    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (revoke) {
      flags |= telegram_api::channels_deleteHistory::FOR_EVERYONE_MASK;
    }

    send_query(G()->net_query_creator().create(telegram_api::channels_deleteHistory(
        flags, false /*ignored*/, std::move(input_channel), max_message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteChannelHistoryQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (!td_->chat_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for DeleteChannelHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteMessagesByDateQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  DialogId dialog_id_;

 public:
  explicit DeleteMessagesByDateQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 min_date, int32 max_date, bool revoke) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = telegram_api::messages_deleteHistory::JUST_CLEAR_MASK |
                  telegram_api::messages_deleteHistory::MIN_DATE_MASK |
                  telegram_api::messages_deleteHistory::MAX_DATE_MASK;
    if (revoke) {
      flags |= telegram_api::messages_deleteHistory::REVOKE_MASK;
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_deleteHistory(
        flags, false /*ignored*/, false /*ignored*/, std::move(input_peer), 0, min_date, max_date)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "DeleteMessagesByDateQuery");
    promise_.set_error(std::move(status));
  }
};

class DeletePhoneCallHistoryQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;

 public:
  explicit DeletePhoneCallHistoryQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(bool revoke) {
    int32 flags = 0;
    if (revoke) {
      flags |= telegram_api::messages_deletePhoneCallHistory::REVOKE_MASK;
    }
    send_query(
        G()->net_query_creator().create(telegram_api::messages_deletePhoneCallHistory(flags, false /*ignored*/)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deletePhoneCallHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    if (!affected_messages->messages_.empty()) {
      td_->messages_manager_->process_pts_update(
          make_tl_object<telegram_api::updateDeleteMessages>(std::move(affected_messages->messages_), 0, 0));
    }
    promise_.set_value(AffectedHistory(std::move(affected_messages)));
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class BlockFromRepliesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit BlockFromRepliesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(MessageId message_id, bool need_delete_message, bool need_delete_all_messages, bool report_spam) {
    int32 flags = 0;
    if (need_delete_message) {
      flags |= telegram_api::contacts_blockFromReplies::DELETE_MESSAGE_MASK;
    }
    if (need_delete_all_messages) {
      flags |= telegram_api::contacts_blockFromReplies::DELETE_HISTORY_MASK;
    }
    if (report_spam) {
      flags |= telegram_api::contacts_blockFromReplies::REPORT_SPAM_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::contacts_blockFromReplies(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, message_id.get_server_message_id().get())));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_blockFromReplies>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for BlockFromRepliesQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class DeleteParticipantHistoryQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  ChannelId channel_id_;
  DialogId sender_dialog_id_;

 public:
  explicit DeleteParticipantHistoryQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, DialogId sender_dialog_id) {
    channel_id_ = channel_id;
    sender_dialog_id_ = sender_dialog_id;

    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }
    auto input_peer = td_->dialog_manager_->get_input_peer(sender_dialog_id, AccessRights::Know);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Message sender is not accessible"));
    }

    send_query(G()->net_query_creator().create(
        telegram_api::channels_deleteParticipantHistory(std::move(input_channel), std::move(input_peer))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteParticipantHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    if (sender_dialog_id_.get_type() != DialogType::Channel) {
      td_->chat_manager_->on_get_channel_error(channel_id_, status, "DeleteParticipantHistoryQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class ReadMentionsQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadMentionsQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = 0;
    if (top_thread_message_id.is_valid()) {
      flags |= telegram_api::messages_readMentions::TOP_MSG_ID_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readMentions(flags, std::move(input_peer),
                                            top_thread_message_id.get_server_message_id().get()),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readMentions>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ReadMentionsQuery");
    promise_.set_error(std::move(status));
  }
};

class ReadReactionsQuery final : public Td::ResultHandler {
  Promise<AffectedHistory> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadReactionsQuery(Promise<AffectedHistory> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = 0;
    if (top_thread_message_id.is_valid()) {
      flags |= telegram_api::messages_readReactions::TOP_MSG_ID_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readReactions(flags, std::move(input_peer),
                                             top_thread_message_id.get_server_message_id().get()),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readReactions>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    promise_.set_value(AffectedHistory(result_ptr.move_as_ok()));
  }

  void on_error(Status status) final {
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ReadReactionsQuery");
    promise_.set_error(std::move(status));
  }
};

class SaveDefaultSendAsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit SaveDefaultSendAsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, DialogId send_as_dialog_id) {
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    auto send_as_input_peer = td_->dialog_manager_->get_input_peer(send_as_dialog_id, AccessRights::Read);
    CHECK(send_as_input_peer != nullptr);

    send_query(G()->net_query_creator().create(
        telegram_api::messages_saveDefaultSendAs(std::move(input_peer), std::move(send_as_input_peer)),
        {{dialog_id, MessageContentType::Photo}, {dialog_id, MessageContentType::Text}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_saveDefaultSendAs>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto success = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SaveDefaultSendAsQuery: " << success;

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    // td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SaveDefaultSendAsQuery");
    promise_.set_error(std::move(status));
  }
};

class SendMessageQuery final : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> as_input_peer,
            const MessageInputReplyTo &input_reply_to, MessageId top_thread_message_id, int32 schedule_date,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text, bool is_copy,
            int64 random_id, NetQueryRef *send_query_ref) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    auto reply_to = input_reply_to.get_input_reply_to(td_, top_thread_message_id);

    if (reply_to != nullptr) {
      flags |= telegram_api::messages_sendMessage::REPLY_TO_MASK;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (as_input_peer != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_SEND_AS;
    }

    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendMessage(
            flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/,
            false /*ignored*/, false /*ignored*/, std::move(input_peer), std::move(reply_to), text, random_id,
            std::move(reply_markup), std::move(entities), schedule_date, std::move(as_input_peer), nullptr),
        {{dialog_id, MessageContentType::Text},
         {dialog_id, is_copy ? MessageContentType::Photo : MessageContentType::Text}});
    if (td_->option_manager_->get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda([random_id](Result<Unit> result) {
        if (result.is_ok()) {
          send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
        }
      });
    }
    *send_query_ref = query.get_weak();
    send_query(std::move(query));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMessageQuery for " << random_id_ << ": " << to_string(ptr);

    auto constructor_id = ptr->get_id();
    if (constructor_id != telegram_api::updateShortSentMessage::ID) {
      td_->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMessage");
      return td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
    }
    auto sent_message = move_tl_object_as<telegram_api::updateShortSentMessage>(ptr);
    td_->messages_manager_->on_update_sent_text_message(random_id_, std::move(sent_message->media_),
                                                        std::move(sent_message->entities_));

    auto message_id = MessageId(ServerMessageId(sent_message->id_));
    auto ttl_period = sent_message->ttl_period_;
    auto update = make_tl_object<updateSentMessage>(random_id_, message_id, sent_message->date_, ttl_period);
    if (dialog_id_.get_type() == DialogType::Channel) {
      td_->messages_manager_->add_pending_channel_update(dialog_id_, std::move(update), sent_message->pts_,
                                                         sent_message->pts_count_, Promise<Unit>(),
                                                         "send message actor");
      return;
    }

    td_->updates_manager_->add_pending_pts_update(std::move(update), sent_message->pts_, sent_message->pts_count_,
                                                  Time::now(), Promise<Unit>(), "send message actor");
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendMessage: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message will be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendMessageQuery");
    td_->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class StartBotQuery final : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(tl_object_ptr<telegram_api::InputUser> bot_input_user, DialogId dialog_id,
                   tl_object_ptr<telegram_api::InputPeer> input_peer, const string &parameter, int64 random_id) {
    CHECK(bot_input_user != nullptr);
    CHECK(input_peer != nullptr);
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto query = G()->net_query_creator().create(
        telegram_api::messages_startBot(std::move(bot_input_user), std::move(input_peer), random_id, parameter),
        {{dialog_id, MessageContentType::Text}, {dialog_id, MessageContentType::Photo}});
    if (td_->option_manager_->get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda([random_id](Result<Unit> result) {
        if (result.is_ok()) {
          send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
        }
      });
    }
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_startBot>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for StartBotQuery for " << random_id_ << ": " << to_string(ptr);
    // Result may contain messageActionChatAddUser
    // td_->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "StartBot");
    td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for StartBotQuery: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message should be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "StartBotQuery");
    td_->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendInlineBotResultQuery final : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(int32 flags, DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> as_input_peer,
                   const MessageInputReplyTo &input_reply_to, MessageId top_thread_message_id, int32 schedule_date,
                   int64 random_id, int64 query_id, const string &result_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto reply_to = input_reply_to.get_input_reply_to(td_, top_thread_message_id);

    if (reply_to != nullptr) {
      flags |= telegram_api::messages_sendInlineBotResult::REPLY_TO_MASK;
    }
    if (as_input_peer != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_SEND_AS;
    }

    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendInlineBotResult(
            flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
            std::move(reply_to), random_id, query_id, result_id, schedule_date, std::move(as_input_peer), nullptr),
        {{dialog_id, MessageContentType::Text}, {dialog_id, MessageContentType::Photo}});
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendInlineBotResult>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendInlineBotResultQuery for " << random_id_ << ": " << to_string(ptr);
    td_->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendInlineBotResult");
    td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendInlineBotResultQuery: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message will be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendInlineBotResultQuery");
    td_->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendMultiMediaQuery final : public Td::ResultHandler {
  vector<FileId> file_ids_;
  vector<string> file_references_;
  vector<int64> random_ids_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> as_input_peer,
            const MessageInputReplyTo &input_reply_to, MessageId top_thread_message_id, int32 schedule_date,
            vector<FileId> &&file_ids, vector<tl_object_ptr<telegram_api::inputSingleMedia>> &&input_single_media,
            bool is_copy) {
    for (auto &single_media : input_single_media) {
      random_ids_.push_back(single_media->random_id_);
      CHECK(FileManager::extract_was_uploaded(single_media->media_) == false);
      file_references_.push_back(FileManager::extract_file_reference(single_media->media_));
    }
    dialog_id_ = dialog_id;
    file_ids_ = std::move(file_ids);
    CHECK(file_ids_.size() == random_ids_.size());

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    auto reply_to = input_reply_to.get_input_reply_to(td_, top_thread_message_id);

    if (reply_to != nullptr) {
      flags |= telegram_api::messages_sendMultiMedia::REPLY_TO_MASK;
    }
    if (as_input_peer != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_SEND_AS;
    }

    // no quick ack, because file reference errors are very likely to happen
    send_query(G()->net_query_creator().create(
        telegram_api::messages_sendMultiMedia(flags, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                              false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                              std::move(input_peer), std::move(reply_to), std::move(input_single_media),
                                              schedule_date, std::move(as_input_peer), nullptr),
        {{dialog_id, is_copy ? MessageContentType::Text : MessageContentType::Photo},
         {dialog_id, MessageContentType::Photo}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMultiMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMultiMedia for " << format::as_array(random_ids_) << ": " << to_string(ptr);

    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td_->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not sent"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (DialogId::get_message_dialog_id(sent_message.first) != dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for SendMultiMediaQuery with random_ids " << format::as_array(random_ids_)
                 << " to " << dialog_id_ << ": " << oneline(to_string(ptr));
      td_->updates_manager_->schedule_get_difference("Wrong sendMultiMedia result");
    }

    td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendMultiMedia: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message will be re-sent after restart
      return;
    }
    if (!td_->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      auto pos = FileReferenceManager::get_file_reference_error_pos(status);
      if (1 <= pos && pos <= file_ids_.size() && file_ids_[pos - 1].is_valid()) {
        VLOG(file_references) << "Receive " << status << " for " << file_ids_[pos - 1];
        td_->file_manager_->delete_file_reference(file_ids_[pos - 1], file_references_[pos - 1]);
        td_->messages_manager_->on_send_media_group_file_reference_error(dialog_id_, std::move(random_ids_));
        return;
      } else {
        LOG(ERROR) << "Receive file reference error " << status << ", but file_ids = " << file_ids_
                   << ", message_count = " << file_ids_.size();
      }
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendMultiMediaQuery");
    for (auto &random_id : random_ids_) {
      td_->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
  }
};

class SendMediaQuery final : public Td::ResultHandler {
  int64 random_id_ = 0;
  FileId file_id_;
  FileId thumbnail_file_id_;
  DialogId dialog_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(FileId file_id, FileId thumbnail_file_id, int32 flags, DialogId dialog_id,
            tl_object_ptr<telegram_api::InputPeer> as_input_peer, const MessageInputReplyTo &input_reply_to,
            MessageId top_thread_message_id, int32 schedule_date,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text,
            tl_object_ptr<telegram_api::InputMedia> &&input_media, MessageContentType content_type, bool is_copy,
            int64 random_id, NetQueryRef *send_query_ref) {
    random_id_ = random_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    dialog_id_ = dialog_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    auto reply_to = input_reply_to.get_input_reply_to(td_, top_thread_message_id);

    if (reply_to != nullptr) {
      flags |= telegram_api::messages_sendMedia::REPLY_TO_MASK;
    }
    if (!entities.empty()) {
      flags |= telegram_api::messages_sendMedia::ENTITIES_MASK;
    }
    if (as_input_peer != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_SEND_AS;
    }

    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendMedia(
            flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/,
            false /*ignored*/, std::move(input_peer), std::move(reply_to), std::move(input_media), text, random_id,
            std::move(reply_markup), std::move(entities), schedule_date, std::move(as_input_peer), nullptr),
        {{dialog_id, content_type}, {dialog_id, is_copy ? MessageContentType::Text : content_type}});
    if (td_->option_manager_->get_option_boolean("use_quick_ack") && was_uploaded_) {
      query->quick_ack_promise_ = PromiseCreator::lambda([random_id](Result<Unit> result) {
        if (result.is_ok()) {
          send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
        }
      });
    }
    *send_query_ref = query.get_weak();
    send_query(std::move(query));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMediaQuery for " << random_id_ << ": " << to_string(ptr);
    td_->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMedia");
    td_->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      td_->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendMedia: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message will be re-sent after restart
      return;
    }
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td_->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      auto bad_parts = FileManager::get_missing_file_parts(status);
      if (!bad_parts.empty()) {
        td_->messages_manager_->on_send_message_file_parts_missing(random_id_, std::move(bad_parts));
        return;
      } else {
        td_->file_manager_->delete_partial_remote_location_if_needed(file_id_, status);
      }
    } else if (!td_->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      if (file_id_.is_valid() && !was_uploaded_) {
        VLOG(file_references) << "Receive " << status << " for " << file_id_;
        td_->file_manager_->delete_file_reference(file_id_, file_reference_);
        td_->messages_manager_->on_send_message_file_reference_error(random_id_);
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but file_id = " << file_id_
                   << ", was_uploaded = " << was_uploaded_;
      }
    }

    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendMediaQuery");
    td_->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class UploadMediaQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  MessageId message_id_;
  FileId file_id_;
  FileId thumbnail_file_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(DialogId dialog_id, MessageId message_id, FileId file_id, FileId thumbnail_file_id,
            tl_object_ptr<telegram_api::InputMedia> &&input_media) {
    CHECK(input_media != nullptr);
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    int32 flags = 0;
    send_query(G()->net_query_creator().create(
        telegram_api::messages_uploadMedia(flags, string(), std::move(input_peer), std::move(input_media))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_uploadMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      td_->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for UploadMediaQuery for " << message_id_ << " in " << dialog_id_ << ": "
              << to_string(ptr);
    td_->messages_manager_->on_upload_message_media_success(dialog_id_, message_id_, std::move(ptr));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for UploadMediaQuery for " << message_id_ << " in " << dialog_id_ << ": " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, message will be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "UploadMediaQuery");
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td_->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      auto bad_parts = FileManager::get_missing_file_parts(status);
      if (!bad_parts.empty()) {
        td_->messages_manager_->on_upload_message_media_file_parts_missing(dialog_id_, message_id_,
                                                                           std::move(bad_parts));
        return;
      } else {
        td_->file_manager_->delete_partial_remote_location_if_needed(file_id_, status);
      }
    } else if (FileReferenceManager::is_file_reference_error(status)) {
      LOG(ERROR) << "Receive file reference error for UploadMediaQuery";
    }
    td_->messages_manager_->on_upload_message_media_fail(dialog_id_, message_id_, std::move(status));
  }
};

class SendScheduledMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SendScheduledMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    int32 server_message_id = message_id.get_scheduled_server_message_id().get();
    send_query(G()->net_query_creator().create(
        telegram_api::messages_sendScheduledMessages(std::move(input_peer), {server_message_id}),
        {{dialog_id, MessageContentType::Text}, {dialog_id, MessageContentType::Photo}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScheduledMessageQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendScheduledMessageQuery: " << status;
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendScheduledMessageQuery");
    promise_.set_error(std::move(status));
  }
};

class EditMessageQuery final : public Td::ResultHandler {
  Promise<int32> promise_;
  DialogId dialog_id_;
  MessageId message_id_;

 public:
  explicit EditMessageQuery(Promise<Unit> &&promise) {
    promise_ = PromiseCreator::lambda([promise = std::move(promise)](Result<int32> result) mutable {
      if (result.is_error()) {
        promise.set_error(result.move_as_error());
      } else {
        promise.set_value(Unit());
      }
    });
  }
  explicit EditMessageQuery(Promise<int32> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId dialog_id, MessageId message_id, const string &text,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media, bool invert_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup, int32 schedule_date) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;

    if (input_media != nullptr && false) {
      return on_error(Status::Error(400, "FILE_PART_1_MISSING"));
    }

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editMessage::MEDIA_MASK;
    }
    if (invert_media) {
      flags |= telegram_api::messages_editMessage::INVERT_MEDIA_MASK;
    }
    if (schedule_date != 0) {
      flags |= telegram_api::messages_editMessage::SCHEDULE_DATE_MASK;
    }

    int32 server_message_id = schedule_date != 0 ? message_id.get_scheduled_server_message_id().get()
                                                 : message_id.get_server_message_id().get();
    send_query(G()->net_query_creator().create(
        telegram_api::messages_editMessage(flags, false /*ignored*/, false /*ignored*/, std::move(input_peer),
                                           server_message_id, text, std::move(input_media), std::move(reply_markup),
                                           std::move(entities), schedule_date, 0),
        {{dialog_id}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_editMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditMessageQuery: " << to_string(ptr);
    auto pts = td_->updates_manager_->get_update_edit_message_pts(ptr.get(), {dialog_id_, message_id_});
    auto promise = PromiseCreator::lambda(
        [promise = std::move(promise_), pts](Result<Unit> result) mutable { promise.set_value(std::move(pts)); });
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise));
  }

  void on_error(Status status) final {
    if (status.code() != 403 && !(status.code() == 500 && G()->close_flag())) {
      LOG(WARNING) << "Failed to edit " << MessageFullId{dialog_id_, message_id_} << " with the error "
                   << status.message();
    } else {
      LOG(INFO) << "Receive error for EditMessageQuery: " << status;
    }
    if (!td_->auth_manager_->is_bot() && status.message() == "MESSAGE_NOT_MODIFIED") {
      return promise_.set_value(0);
    }
    td_->messages_manager_->on_get_message_error(dialog_id_, message_id_, status, "EditMessageQuery");
    promise_.set_error(std::move(status));
  }
};

class EditInlineMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit EditInlineMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, tl_object_ptr<telegram_api::InputBotInlineMessageID> input_bot_inline_message_id,
            const string &text, vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media, bool invert_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup) {
    CHECK(input_bot_inline_message_id != nullptr);

    // file in an inline message can't be uploaded to another datacenter,
    // so only previously uploaded files or URLs can be used in the InputMedia
    CHECK(!FileManager::extract_was_uploaded(input_media));

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editInlineBotMessage::MEDIA_MASK;
    }
    if (invert_media) {
      flags |= telegram_api::messages_editInlineBotMessage::INVERT_MEDIA_MASK;
    }

    auto dc_id = DcId::internal(InlineQueriesManager::get_inline_message_dc_id(input_bot_inline_message_id));
    send_query(G()->net_query_creator().create(
        telegram_api::messages_editInlineBotMessage(
            flags, false /*ignored*/, false /*ignored*/, std::move(input_bot_inline_message_id), text,
            std::move(input_media), std::move(reply_markup), std::move(entities)),
        {}, dc_id));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_editInlineBotMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    LOG_IF(ERROR, !result_ptr.ok()) << "Receive false in result of editInlineMessage";

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for EditInlineMessageQuery: " << status;
    promise_.set_error(std::move(status));
  }
};

class ForwardMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  vector<int64> random_ids_;
  DialogId from_dialog_id_;
  DialogId to_dialog_id_;
  MessageId message_id_;

 public:
  explicit ForwardMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId to_dialog_id, MessageId top_thread_message_id, DialogId from_dialog_id,
            tl_object_ptr<telegram_api::InputPeer> as_input_peer, const vector<MessageId> &message_ids,
            vector<int64> &&random_ids, int32 schedule_date) {
    random_ids_ = random_ids;
    from_dialog_id_ = from_dialog_id;
    to_dialog_id_ = to_dialog_id;
    if (message_ids.size() == 1) {
      message_id_ = message_ids[0];
    }

    auto to_input_peer = td_->dialog_manager_->get_input_peer(to_dialog_id, AccessRights::Write);
    if (to_input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    auto from_input_peer = td_->dialog_manager_->get_input_peer(from_dialog_id, AccessRights::Read);
    if (from_input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat to forward messages from"));
    }

    if (as_input_peer != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_SEND_AS;
    }
    if (top_thread_message_id.is_valid()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_IS_FROM_THREAD;
    }

    auto query = G()->net_query_creator().create(
        telegram_api::messages_forwardMessages(
            flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/,
            false /*ignored*/, std::move(from_input_peer), MessageId::get_server_message_ids(message_ids),
            std::move(random_ids), std::move(to_input_peer), top_thread_message_id.get_server_message_id().get(),
            schedule_date, std::move(as_input_peer), nullptr),
        {{to_dialog_id, MessageContentType::Text}, {to_dialog_id, MessageContentType::Photo}});
    if (td_->option_manager_->get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda([random_ids = random_ids_](Result<Unit> result) {
        if (result.is_ok()) {
          for (auto random_id : random_ids) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          }
        }
      });
    }
    send_query(std::move(query));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_forwardMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for ForwardMessagesQuery for " << format::as_array(random_ids_) << ": "
              << to_string(ptr);
    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td_->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not forwarded"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (DialogId::get_message_dialog_id(sent_message.first) != to_dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for forwarding messages with random_ids " << format::as_array(random_ids_)
                 << " to " << to_dialog_id_ << ": " << oneline(to_string(ptr));
      td_->updates_manager_->schedule_get_difference("Wrong forwardMessages result");
    }

    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for forward messages: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, messages will be re-sent after restart
      return;
    }
    // no on_get_dialog_error call, because two dialogs are involved
    if (status.code() == 400 && status.message() == CSlice("CHAT_FORWARDS_RESTRICTED")) {
      td_->dialog_manager_->reload_dialog_info(from_dialog_id_, Promise<Unit>());
    }
    if (status.code() == 400 && status.message() == CSlice("SEND_AS_PEER_INVALID")) {
      td_->dialog_manager_->reload_dialog_info_full(to_dialog_id_, "SEND_AS_PEER_INVALID");
    }
    if (message_id_.is_valid() && status.message() == CSlice("MESSAGE_ID_INVALID")) {
      td_->messages_manager_->get_message_from_server({from_dialog_id_, message_id_}, Promise<Unit>(),
                                                      "ForwardMessagesQuery");
    }
    for (auto &random_id : random_ids_) {
      td_->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
    promise_.set_error(std::move(status));
  }
};

class SendQuickReplyMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  vector<int64> random_ids_;
  DialogId dialog_id_;
  QuickReplyShortcutId shortcut_id_;

 public:
  explicit SendQuickReplyMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, QuickReplyShortcutId shortcut_id, const vector<MessageId> &message_ids,
            vector<int64> &&random_ids) {
    random_ids_ = random_ids;
    dialog_id_ = dialog_id;
    shortcut_id_ = shortcut_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id_, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Have no write access to the chat"));
    }

    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendQuickReplyMessages(std::move(input_peer), shortcut_id.get(),
                                                      MessageId::get_server_message_ids(message_ids),
                                                      std::move(random_ids)),
        {{dialog_id, MessageContentType::Text}, {dialog_id, MessageContentType::Photo}});
    if (td_->option_manager_->get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda([random_ids = random_ids_](Result<Unit> result) {
        if (result.is_ok()) {
          for (auto random_id : random_ids) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          }
        }
      });
    }
    send_query(std::move(query));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendQuickReplyMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendQuickReplyMessagesQuery for " << format::as_array(random_ids_) << ": "
              << to_string(ptr);
    auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    if (random_ids_.size() != sent_messages.size() || random_ids_.size() != sent_random_ids.size()) {
      is_result_wrong = true;
    }
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        is_result_wrong = true;
      }
    }
    for (auto &sent_message : sent_messages) {
      if (DialogId::get_message_dialog_id(sent_message.first) != dialog_id_) {
        is_result_wrong = true;
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for sending quick reply messages with random_ids "
                 << format::as_array(random_ids_) << " to " << dialog_id_ << ": " << oneline(to_string(ptr));
      td_->updates_manager_->schedule_get_difference("Wrong sendQuickReplyMessages result");
      for (auto &random_id : random_ids_) {
        td_->messages_manager_->on_send_message_fail(random_id, Status::Error(500, "Receive invalid response"));
      }
    }
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendQuickReplyMessagesQuery: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, messages will be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendQuickReplyMessagesQuery");
    if (status.code() == 400 && status.message() == CSlice("MESSAGE_IDS_MISMATCH")) {
      td_->quick_reply_manager_->reload_quick_reply_messages(shortcut_id_, Auto());
    }
    for (auto &random_id : random_ids_) {
      td_->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
    promise_.set_error(std::move(status));
  }
};

class SendScreenshotNotificationQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;
  DialogId dialog_id_;

 public:
  explicit SendScreenshotNotificationQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 random_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(
        telegram_api::messages_sendScreenshotNotification(
            std::move(input_peer),
            telegram_api::make_object<telegram_api::inputReplyToMessage>(0, 0, 0, nullptr, string(), Auto(), 0),
            random_id),
        {{dialog_id, MessageContentType::Text}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendScreenshotNotification>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScreenshotNotificationQuery for " << random_id_ << ": " << to_string(ptr);
    td_->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(),
                                                      "SendScreenshotNotificationQuery");
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for SendScreenshotNotificationQuery: " << status;
    if (G()->close_flag() && G()->use_message_database()) {
      // do not send error, messages will be re-sent after restart
      return;
    }
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "SendScreenshotNotificationQuery");
    td_->messages_manager_->on_send_message_fail(random_id_, status.clone());
    promise_.set_error(std::move(status));
  }
};

class SendBotRequestedPeerQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit SendBotRequestedPeerQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(MessageFullId message_full_id, int32 button_id, vector<DialogId> &&requested_dialog_ids) {
    auto dialog_id = message_full_id.get_dialog_id();
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }
    vector<telegram_api::object_ptr<telegram_api::InputPeer>> requested_peers;
    for (auto requested_dialog_id : requested_dialog_ids) {
      auto requested_peer = td_->dialog_manager_->get_input_peer(requested_dialog_id, AccessRights::Read);
      if (requested_peer == nullptr) {
        return on_error(Status::Error(400, "Can't access the chosen chat"));
      }
      requested_peers.push_back(std::move(requested_peer));
    }

    send_query(G()->net_query_creator().create(
        telegram_api::messages_sendBotRequestedPeer(std::move(input_peer),
                                                    message_full_id.get_message_id().get_server_message_id().get(),
                                                    button_id, std::move(requested_peers)),
        {{dialog_id, MessageContentType::Text}}));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendBotRequestedPeer>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendBotRequestedPeerQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    promise_.set_error(std::move(status));
  }
};

class DeleteMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  vector<int32> server_message_ids_;

 public:
  explicit DeleteMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, vector<int32> &&server_message_ids, bool revoke) {
    dialog_id_ = dialog_id;
    server_message_ids_ = server_message_ids;

    int32 flags = 0;
    if (revoke) {
      flags |= telegram_api::messages_deleteMessages::REVOKE_MASK;
    }

    send_query(G()->net_query_creator().create(
        telegram_api::messages_deleteMessages(flags, false /*ignored*/, std::move(server_message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteMessagesQuery: " << to_string(affected_messages);
    td_->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                  affected_messages->pts_count_, Time::now(), std::move(promise_),
                                                  "delete messages query");
  }

  void on_error(Status status) final {
    if (!G()->is_expected_error(status)) {
      // MESSAGE_DELETE_FORBIDDEN can be returned in group chats when administrator rights were removed
      // MESSAGE_DELETE_FORBIDDEN can be returned in private chats for bots when revoke time limit exceeded
      if (status.message() != "MESSAGE_DELETE_FORBIDDEN" ||
          (dialog_id_.get_type() == DialogType::User && !td_->auth_manager_->is_bot())) {
        LOG(ERROR) << "Receive error for delete messages: " << status;
      }
    }
    td_->messages_manager_->on_failed_message_deletion(dialog_id_, server_message_ids_);
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  vector<int32> server_message_ids_;

 public:
  explicit DeleteChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<int32> &&server_message_ids) {
    channel_id_ = channel_id;
    server_message_ids_ = server_message_ids;

    auto input_channel = td_->chat_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }
    send_query(G()->net_query_creator().create(
        telegram_api::channels_deleteMessages(std::move(input_channel), std::move(server_message_ids))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteChannelMessagesQuery: " << to_string(affected_messages);
    td_->messages_manager_->add_pending_channel_update(DialogId(channel_id_), make_tl_object<dummyUpdate>(),
                                                       affected_messages->pts_, affected_messages->pts_count_,
                                                       std::move(promise_), "DeleteChannelMessagesQuery");
  }

  void on_error(Status status) final {
    if (!td_->chat_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelMessagesQuery")) {
      if (status.message() != "MESSAGE_DELETE_FORBIDDEN") {
        LOG(ERROR) << "Receive error for delete channel messages: " << status;
      }
    }
    td_->messages_manager_->on_failed_message_deletion(DialogId(channel_id_), server_message_ids_);
    promise_.set_error(std::move(status));
  }
};

class DeleteScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

 public:
  explicit DeleteScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, vector<MessageId> &&message_ids) {
    dialog_id_ = dialog_id;
    message_ids_ = std::move(message_ids);

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(Status::Error(400, "Can't access the chat"));
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_deleteScheduledMessages(
        std::move(input_peer), MessageId::get_scheduled_server_message_ids(message_ids_))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteScheduledMessagesQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "DeleteScheduledMessagesQuery")) {
      LOG(ERROR) << "Receive error for delete scheduled messages: " << status;
    }
    td_->messages_manager_->on_failed_scheduled_message_deletion(dialog_id_, message_ids_);
    promise_.set_error(std::move(status));
  }
};

class GetPeerSettingsQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::messages_getPeerSettings(std::move(input_peer))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerSettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    td_->user_manager_->on_get_users(std::move(ptr->users_), "GetPeerSettingsQuery");
    td_->chat_manager_->on_get_chats(std::move(ptr->chats_), "GetPeerSettingsQuery");
    td_->messages_manager_->on_get_peer_settings(dialog_id_, std::move(ptr->settings_));
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for get peer settings: " << status;
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetPeerSettingsQuery");
  }
};

class UpdatePeerSettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdatePeerSettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_spam_dialog) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_value(Unit());
    }

    if (is_spam_dialog) {
      send_query(G()->net_query_creator().create(telegram_api::messages_reportSpam(std::move(input_peer))));
    } else {
      send_query(G()->net_query_creator().create(telegram_api::messages_hidePeerSettingsBar(std::move(input_peer))));
    }
  }

  void on_result(BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_reportSpam::ReturnType,
                               telegram_api::messages_hidePeerSettingsBar::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_reportSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    td_->messages_manager_->on_get_peer_settings(dialog_id_, make_tl_object<telegram_api::peerSettings>(), true);

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for update peer settings: " << status;
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "UpdatePeerSettingsQuery");
    td_->messages_manager_->reget_dialog_action_bar(dialog_id_, "UpdatePeerSettingsQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportEncryptedSpamQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReportEncryptedSpamQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_encrypted_chat(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::messages_reportEncryptedSpam(std::move(input_peer))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_reportEncryptedSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    td_->messages_manager_->on_get_peer_settings(dialog_id_, make_tl_object<telegram_api::peerSettings>(), true);

    promise_.set_value(Unit());
  }

  void on_error(Status status) final {
    LOG(INFO) << "Receive error for report encrypted spam: " << status;
    td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "ReportEncryptedSpamQuery");
    td_->messages_manager_->reget_dialog_action_bar(
        DialogId(td_->user_manager_->get_secret_chat_user_id(dialog_id_.get_secret_chat_id())),
        "ReportEncryptedSpamQuery");
    promise_.set_error(std::move(status));
  }
};

class EditPeerFoldersQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditPeerFoldersQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FolderId folder_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    vector<telegram_api::object_ptr<telegram_api::inputFolderPeer>> input_folder_peers;
    input_folder_peers.push_back(
        telegram_api::make_object<telegram_api::inputFolderPeer>(std::move(input_peer), folder_id.get()));
    send_query(G()->net_query_creator().create(telegram_api::folders_editPeerFolders(std::move(input_folder_peers))));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::folders_editPeerFolders>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditPeerFoldersQuery: " << to_string(ptr);
    td_->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "EditPeerFoldersQuery")) {
      LOG(INFO) << "Receive error for EditPeerFoldersQuery: " << status;
    }

    // trying to repair folder ID for this dialog
    td_->dialog_manager_->get_dialog_info_full(dialog_id_, Auto(), "EditPeerFoldersQuery");

    promise_.set_error(std::move(status));
  }
};

class GetChannelDifferenceQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  int32 pts_;
  int32 limit_;

 public:
  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel, int32 pts, int32 limit,
            bool force) {
    CHECK(pts >= 0);
    dialog_id_ = dialog_id;
    pts_ = pts;
    limit_ = limit;
    CHECK(input_channel != nullptr);

    int32 flags = 0;
    if (force) {
      flags |= telegram_api::updates_getChannelDifference::FORCE_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::updates_getChannelDifference(
        flags, false /*ignored*/, std::move(input_channel), make_tl_object<telegram_api::channelMessagesFilterEmpty>(),
        pts, limit)));
  }

  void on_result(BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::updates_getChannelDifference>(packet);
    if (result_ptr.is_error()) {
      return on_error(result_ptr.move_as_error());
    }

    td_->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, result_ptr.move_as_ok(), Status::OK());
  }

  void on_error(Status status) final {
    if (!td_->dialog_manager_->on_get_dialog_error(dialog_id_, status, "GetChannelDifferenceQuery") &&
        status.message() != "PERSISTENT_TIMESTAMP_INVALID") {
      LOG(ERROR) << "Receive error for GetChannelDifferenceQuery for " << dialog_id_ << " with PTS " << pts_
                 << " and limit " << limit_ << ": " << status;
    }
    td_->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, nullptr, std::move(status));
  }
};

class MessagesManager::UploadMediaCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, std::move(input_file),
                       nullptr);
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, nullptr,
                       std::move(input_file));
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media_error, file_id, std::move(error));
  }
};

class MessagesManager::UploadThumbnailCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, nullptr);
  }
};

template <class StorerT>
void MessagesManager::Message::store(StorerT &storer) const {
  using td::store;
  bool has_sender = sender_user_id.is_valid();
  bool has_edit_date = edit_date > 0;
  bool has_random_id = random_id != 0;
  bool is_reply_to_random_id = reply_to_random_id != 0;
  bool is_via_bot = via_bot_user_id.is_valid();
  bool has_view_count = view_count > 0;
  bool has_reply_markup = reply_markup != nullptr;
  bool has_ttl = ttl.is_valid();
  bool has_author_signature = !author_signature.empty();
  bool has_media_album_id = media_album_id != 0;
  bool has_send_date = message_id.is_yet_unsent() && send_date != 0;
  bool has_flags2 = true;
  bool has_notification_id = notification_id.is_valid();
  bool has_send_error_code = send_error_code != 0;
  bool has_real_forward_from = real_forward_from_dialog_id.is_valid() && real_forward_from_message_id.is_valid();
  bool has_legacy_layer = legacy_layer != 0;
  bool has_restriction_reasons = !restriction_reasons.empty();
  bool has_forward_count = forward_count > 0;
  bool has_reply_info = !reply_info.is_empty();
  bool has_sender_dialog_id = sender_dialog_id.is_valid();
  bool has_top_thread_message_id = top_thread_message_id.is_valid();
  bool has_thread_draft_message = thread_draft_message != nullptr;
  bool has_local_thread_message_ids = !local_thread_message_ids.empty();
  bool has_linked_top_thread_message_id = linked_top_thread_message_id.is_valid();
  bool has_interaction_info_update_date = interaction_info_update_date != 0;
  bool has_send_emoji = !send_emoji.empty();
  bool has_ttl_period = ttl_period != 0;
  bool has_max_reply_media_timestamp = max_reply_media_timestamp >= 0;
  bool are_message_media_timestamp_entities_found = true;
  bool has_flags3 = true;
  bool has_reactions = reactions != nullptr;
  bool has_available_reactions_generation = available_reactions_generation != 0;
  bool has_history_generation = history_generation != 0;
  bool is_reply_to_story = reply_to_story_full_id != StoryFullId();
  bool has_input_reply_to = !message_id.is_any_server() && input_reply_to.is_valid();
  bool has_replied_message_info = !replied_message_info.is_empty();
  bool has_forward_info = forward_info != nullptr;
  bool has_saved_messages_topic_id = saved_messages_topic_id.is_valid();
  bool has_initial_top_thread_message_id = !message_id.is_any_server() && initial_top_thread_message_id.is_valid();
  bool has_sender_boost_count = sender_boost_count != 0;
  bool has_via_business_bot_user_id = via_business_bot_user_id.is_valid();
  BEGIN_STORE_FLAGS();
  STORE_FLAG(is_channel_post);
  STORE_FLAG(is_outgoing);
  STORE_FLAG(is_failed_to_send);
  STORE_FLAG(disable_notification);
  STORE_FLAG(contains_mention);
  STORE_FLAG(from_background);
  STORE_FLAG(disable_web_page_preview);
  STORE_FLAG(clear_draft);
  STORE_FLAG(false);
  STORE_FLAG(false);
  STORE_FLAG(has_sender);
  STORE_FLAG(has_edit_date);
  STORE_FLAG(has_random_id);
  STORE_FLAG(false);
  STORE_FLAG(false);
  STORE_FLAG(is_reply_to_random_id);
  STORE_FLAG(is_via_bot);
  STORE_FLAG(has_view_count);
  STORE_FLAG(has_reply_markup);
  STORE_FLAG(has_ttl);
  STORE_FLAG(has_author_signature);
  STORE_FLAG(false);
  STORE_FLAG(had_reply_markup);
  STORE_FLAG(contains_unread_mention);
  STORE_FLAG(has_media_album_id);
  STORE_FLAG(false);
  STORE_FLAG(in_game_share);
  STORE_FLAG(is_content_secret);
  STORE_FLAG(has_send_date);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();
  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_notification_id);
    STORE_FLAG(is_mention_notification_disabled);
    STORE_FLAG(had_forward_info);
    STORE_FLAG(false);
    STORE_FLAG(has_send_error_code);
    STORE_FLAG(hide_via_bot);
    STORE_FLAG(is_bot_start_message);
    STORE_FLAG(has_real_forward_from);
    STORE_FLAG(has_legacy_layer);
    STORE_FLAG(hide_edit_date);
    STORE_FLAG(has_restriction_reasons);
    STORE_FLAG(is_from_scheduled);
    STORE_FLAG(is_copy);
    STORE_FLAG(false);
    STORE_FLAG(has_forward_count);
    STORE_FLAG(has_reply_info);
    STORE_FLAG(has_sender_dialog_id);
    STORE_FLAG(false);
    STORE_FLAG(has_top_thread_message_id);
    STORE_FLAG(has_thread_draft_message);
    STORE_FLAG(has_local_thread_message_ids);
    STORE_FLAG(has_linked_top_thread_message_id);
    STORE_FLAG(is_pinned);
    STORE_FLAG(has_interaction_info_update_date);
    STORE_FLAG(has_send_emoji);
    STORE_FLAG(false);
    STORE_FLAG(has_ttl_period);
    STORE_FLAG(has_max_reply_media_timestamp);
    STORE_FLAG(are_message_media_timestamp_entities_found);
    STORE_FLAG(has_flags3);
    END_STORE_FLAGS();
  }
  if (has_flags3) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(noforwards);
    STORE_FLAG(has_explicit_sender);
    STORE_FLAG(has_reactions);
    STORE_FLAG(has_available_reactions_generation);
    STORE_FLAG(update_stickersets_order);
    STORE_FLAG(is_topic_message);
    STORE_FLAG(has_history_generation);
    STORE_FLAG(is_reply_to_story);
    STORE_FLAG(false);
    STORE_FLAG(invert_media);
    STORE_FLAG(has_input_reply_to);
    STORE_FLAG(has_replied_message_info);
    STORE_FLAG(has_forward_info);
    STORE_FLAG(has_saved_messages_topic_id);
    STORE_FLAG(has_initial_top_thread_message_id);
    STORE_FLAG(has_sender_boost_count);
    STORE_FLAG(has_via_business_bot_user_id);
    STORE_FLAG(is_from_offline);
    END_STORE_FLAGS();
  }

  store(message_id, storer);
  if (has_sender) {
    store(sender_user_id, storer);
  }
  store(date, storer);
  if (has_edit_date) {
    store(edit_date, storer);
  }
  if (has_send_date) {
    store(send_date, storer);
  }
  if (has_random_id) {
    store(random_id, storer);
  }
  if (has_forward_info) {
    store(forward_info, storer);
  }
  if (has_real_forward_from) {
    store(real_forward_from_dialog_id, storer);
    store(real_forward_from_message_id, storer);
  }
  if (is_reply_to_random_id) {
    store(reply_to_random_id, storer);
  }
  if (is_via_bot) {
    store(via_bot_user_id, storer);
  }
  if (has_view_count) {
    store(view_count, storer);
  }
  if (has_forward_count) {
    store(forward_count, storer);
  }
  if (has_reply_info) {
    store(reply_info, storer);
  }
  if (has_ttl) {
    store(ttl, storer);
    store_time(ttl_expires_at, storer);
  }
  if (has_send_error_code) {
    store(send_error_code, storer);
    store(send_error_message, storer);
    if (send_error_code == 429) {
      store_time(try_resend_at, storer);
    }
  }
  if (has_author_signature) {
    store(author_signature, storer);
  }
  if (has_media_album_id) {
    store(media_album_id, storer);
  }
  if (has_notification_id) {
    store(notification_id, storer);
  }
  if (has_legacy_layer) {
    store(legacy_layer, storer);
  }
  if (has_restriction_reasons) {
    store(restriction_reasons, storer);
  }
  if (has_sender_dialog_id) {
    store(sender_dialog_id, storer);
  }
  if (has_top_thread_message_id) {
    store(top_thread_message_id, storer);
  }
  if (has_thread_draft_message) {
    store(thread_draft_message, storer);
  }
  if (has_local_thread_message_ids) {
    store(local_thread_message_ids, storer);
  }
  if (has_linked_top_thread_message_id) {
    store(linked_top_thread_message_id, storer);
  }
  if (has_interaction_info_update_date) {
    store(interaction_info_update_date, storer);
  }
  if (has_send_emoji) {
    store(send_emoji, storer);
  }
  store_message_content(content.get(), storer);
  if (has_reply_markup) {
    store(reply_markup, storer);
  }
  if (has_ttl_period) {
    store(ttl_period, storer);
  }
  if (has_max_reply_media_timestamp) {
    store(max_reply_media_timestamp, storer);
  }
  if (has_reactions) {
    store(reactions, storer);
  }
  if (has_available_reactions_generation) {
    store(available_reactions_generation, storer);
  }
  if (has_history_generation) {
    store(history_generation, storer);
  }
  if (is_reply_to_story) {
    store(reply_to_story_full_id, storer);
  }
  if (has_input_reply_to) {
    store(input_reply_to, storer);
  }
  if (has_replied_message_info) {
    store(replied_message_info, storer);
  }
  if (has_saved_messages_topic_id) {
    store(saved_messages_topic_id, storer);
  }
  if (has_initial_top_thread_message_id) {
    store(initial_top_thread_message_id, storer);
  }
  if (has_sender_boost_count) {
    store(sender_boost_count, storer);
  }
  if (has_via_business_bot_user_id) {
    store(via_business_bot_user_id, storer);
  }
}

// do not forget to resolve message dependencies
template <class ParserT>
void MessagesManager::Message::parse(ParserT &parser) {
  using td::parse;
  bool legacy_have_previous;
  bool legacy_have_next;
  bool has_sender;
  bool has_edit_date;
  bool has_random_id;
  bool legacy_is_forwarded;
  bool legacy_is_reply;
  bool is_reply_to_random_id;
  bool is_via_bot;
  bool has_view_count;
  bool has_reply_markup;
  bool has_ttl;
  bool has_author_signature;
  bool legacy_has_forward_author_signature;
  bool has_media_album_id;
  bool legacy_has_forward_from;
  bool has_send_date;
  bool has_flags2;
  bool has_notification_id = false;
  bool legacy_has_forward_sender_name = false;
  bool has_send_error_code = false;
  bool has_real_forward_from = false;
  bool has_legacy_layer = false;
  bool has_restriction_reasons = false;
  bool legacy_has_forward_psa_type = false;
  bool has_forward_count = false;
  bool has_reply_info = false;
  bool has_sender_dialog_id = false;
  bool legacy_has_reply_in_dialog_id = false;
  bool has_top_thread_message_id = false;
  bool has_thread_draft_message = false;
  bool has_local_thread_message_ids = false;
  bool has_linked_top_thread_message_id = false;
  bool has_interaction_info_update_date = false;
  bool has_send_emoji = false;
  bool legacy_is_imported = false;
  bool has_ttl_period = false;
  bool has_max_reply_media_timestamp = false;
  bool has_flags3 = false;
  bool has_reactions = false;
  bool has_available_reactions_generation = false;
  bool has_history_generation = false;
  bool is_reply_to_story = false;
  bool legacy_has_forward_origin = false;
  bool has_input_reply_to = false;
  bool has_replied_message_info = false;
  bool has_forward_info = false;
  bool has_saved_messages_topic_id = false;
  bool has_initial_top_thread_message_id = false;
  bool has_sender_boost_count = false;
  bool has_via_business_bot_user_id = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(is_channel_post);
  PARSE_FLAG(is_outgoing);
  PARSE_FLAG(is_failed_to_send);
  PARSE_FLAG(disable_notification);
  PARSE_FLAG(contains_mention);
  PARSE_FLAG(from_background);
  PARSE_FLAG(disable_web_page_preview);
  PARSE_FLAG(clear_draft);
  PARSE_FLAG(legacy_have_previous);
  PARSE_FLAG(legacy_have_next);
  PARSE_FLAG(has_sender);
  PARSE_FLAG(has_edit_date);
  PARSE_FLAG(has_random_id);
  PARSE_FLAG(legacy_is_forwarded);
  PARSE_FLAG(legacy_is_reply);
  PARSE_FLAG(is_reply_to_random_id);
  PARSE_FLAG(is_via_bot);
  PARSE_FLAG(has_view_count);
  PARSE_FLAG(has_reply_markup);
  PARSE_FLAG(has_ttl);
  PARSE_FLAG(has_author_signature);
  PARSE_FLAG(legacy_has_forward_author_signature);
  PARSE_FLAG(had_reply_markup);
  PARSE_FLAG(contains_unread_mention);
  PARSE_FLAG(has_media_album_id);
  PARSE_FLAG(legacy_has_forward_from);
  PARSE_FLAG(in_game_share);
  PARSE_FLAG(is_content_secret);
  PARSE_FLAG(has_send_date);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();
  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_notification_id);
    PARSE_FLAG(is_mention_notification_disabled);
    PARSE_FLAG(had_forward_info);
    PARSE_FLAG(legacy_has_forward_sender_name);
    PARSE_FLAG(has_send_error_code);
    PARSE_FLAG(hide_via_bot);
    PARSE_FLAG(is_bot_start_message);
    PARSE_FLAG(has_real_forward_from);
    PARSE_FLAG(has_legacy_layer);
    PARSE_FLAG(hide_edit_date);
    PARSE_FLAG(has_restriction_reasons);
    PARSE_FLAG(is_from_scheduled);
    PARSE_FLAG(is_copy);
    PARSE_FLAG(legacy_has_forward_psa_type);
    PARSE_FLAG(has_forward_count);
    PARSE_FLAG(has_reply_info);
    PARSE_FLAG(has_sender_dialog_id);
    PARSE_FLAG(legacy_has_reply_in_dialog_id);
    PARSE_FLAG(has_top_thread_message_id);
    PARSE_FLAG(has_thread_draft_message);
    PARSE_FLAG(has_local_thread_message_ids);
    PARSE_FLAG(has_linked_top_thread_message_id);
    PARSE_FLAG(is_pinned);
    PARSE_FLAG(has_interaction_info_update_date);
    PARSE_FLAG(has_send_emoji);
    PARSE_FLAG(legacy_is_imported);
    PARSE_FLAG(has_ttl_period);
    PARSE_FLAG(has_max_reply_media_timestamp);
    PARSE_FLAG(are_media_timestamp_entities_found);
    PARSE_FLAG(has_flags3);
    END_PARSE_FLAGS();
  }
  if (has_flags3) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(noforwards);
    PARSE_FLAG(has_explicit_sender);
    PARSE_FLAG(has_reactions);
    PARSE_FLAG(has_available_reactions_generation);
    PARSE_FLAG(update_stickersets_order);
    PARSE_FLAG(is_topic_message);
    PARSE_FLAG(has_history_generation);
    PARSE_FLAG(is_reply_to_story);
    PARSE_FLAG(legacy_has_forward_origin);
    PARSE_FLAG(invert_media);
    PARSE_FLAG(has_input_reply_to);
    PARSE_FLAG(has_replied_message_info);
    PARSE_FLAG(has_forward_info);
    PARSE_FLAG(has_saved_messages_topic_id);
    PARSE_FLAG(has_initial_top_thread_message_id);
    PARSE_FLAG(has_sender_boost_count);
    PARSE_FLAG(has_via_business_bot_user_id);
    PARSE_FLAG(is_from_offline);
    END_PARSE_FLAGS();
  }

  parse(message_id, parser);
  if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
    return parser.set_error("Invalid message identifier");
  }
  if (has_sender) {
    parse(sender_user_id, parser);
  }
  parse(date, parser);
  if (has_edit_date) {
    parse(edit_date, parser);
  }
  if (has_send_date) {
    if (!message_id.is_yet_unsent()) {
      return parser.set_error("Unexpected send date");
    }
    parse(send_date, parser);
  } else if (message_id.is_valid() && message_id.is_yet_unsent()) {
    send_date = date;  // for backward compatibility
  }
  if (has_random_id) {
    parse(random_id, parser);
  }
  if (has_forward_info) {
    parse(forward_info, parser);
  } else if (legacy_is_forwarded) {
    MessageOrigin forward_origin;
    int32 forward_date;
    if (legacy_has_forward_origin) {
      parse(forward_origin, parser);
      parse(forward_date, parser);
    } else {
      UserId forward_sender_user_id;
      DialogId forward_sender_dialog_id;
      MessageId forward_message_id;
      string forward_author_signature;
      string forward_sender_name;
      parse(forward_sender_user_id, parser);
      parse(forward_date, parser);
      parse(forward_sender_dialog_id, parser);
      parse(forward_message_id, parser);
      if (legacy_has_forward_author_signature) {
        parse(forward_author_signature, parser);
      }
      if (legacy_has_forward_sender_name) {
        parse(forward_sender_name, parser);
      }
      forward_origin = MessageOrigin(forward_sender_user_id, forward_sender_dialog_id, forward_message_id,
                                     std::move(forward_author_signature), std::move(forward_sender_name));
    }
    LastForwardedMessageInfo last_message_info;
    if (legacy_has_forward_from) {
      DialogId forward_from_dialog_id;
      MessageId forward_from_message_id;
      parse(forward_from_dialog_id, parser);
      parse(forward_from_message_id, parser);
      last_message_info =
          LastForwardedMessageInfo(forward_from_dialog_id, forward_from_message_id, DialogId(), string(), 0, false);
    }
    string psa_type;
    if (legacy_has_forward_psa_type) {
      parse(psa_type, parser);
    }
    forward_info = td::make_unique<MessageForwardInfo>(
        std::move(forward_origin), forward_date, std::move(last_message_info), std::move(psa_type), legacy_is_imported);
  }
  if (has_real_forward_from) {
    parse(real_forward_from_dialog_id, parser);
    parse(real_forward_from_message_id, parser);
  }
  MessageId legacy_reply_to_message_id;
  if (legacy_is_reply) {
    parse(legacy_reply_to_message_id, parser);
  }
  if (is_reply_to_random_id) {
    parse(reply_to_random_id, parser);
  }
  if (is_via_bot) {
    parse(via_bot_user_id, parser);
  }
  if (has_view_count) {
    parse(view_count, parser);
  }
  if (has_forward_count) {
    parse(forward_count, parser);
  }
  if (has_reply_info) {
    parse(reply_info, parser);
  }
  if (has_ttl) {
    parse(ttl, parser);
    parse_time(ttl_expires_at, parser);
  }
  if (has_send_error_code) {
    parse(send_error_code, parser);
    parse(send_error_message, parser);
    if (send_error_code == 429) {
      parse_time(try_resend_at, parser);
    }
  }
  if (has_author_signature) {
    parse(author_signature, parser);
  }
  if (has_media_album_id) {
    parse(media_album_id, parser);
  }
  if (has_notification_id) {
    parse(notification_id, parser);
  }
  if (has_legacy_layer) {
    parse(legacy_layer, parser);
  }
  if (has_restriction_reasons) {
    parse(restriction_reasons, parser);
  }
  if (has_sender_dialog_id) {
    parse(sender_dialog_id, parser);
  }
  DialogId legacy_reply_in_dialog_id;
  if (legacy_has_reply_in_dialog_id) {
    parse(legacy_reply_in_dialog_id, parser);
  }
  if (has_top_thread_message_id) {
    parse(top_thread_message_id, parser);
  }
  if (has_thread_draft_message) {
    parse(thread_draft_message, parser);
  }
  if (has_local_thread_message_ids) {
    parse(local_thread_message_ids, parser);
  }
  if (has_linked_top_thread_message_id) {
    parse(linked_top_thread_message_id, parser);
  }
  if (has_interaction_info_update_date) {
    parse(interaction_info_update_date, parser);
  }
  if (has_send_emoji) {
    parse(send_emoji, parser);
  }
  parse_message_content(content, parser);
  if (has_reply_markup) {
    parse(reply_markup, parser);
  }
  if (has_ttl_period) {
    parse(ttl_period, parser);
  }
  if (has_max_reply_media_timestamp) {
    parse(max_reply_media_timestamp, parser);
  }
  if (has_reactions) {
    parse(reactions, parser);
  }
  if (has_available_reactions_generation) {
    parse(available_reactions_generation, parser);
  }
  if (has_history_generation) {
    parse(history_generation, parser);
  }
  if (is_reply_to_story) {
    parse(reply_to_story_full_id, parser);
  }
  if (has_input_reply_to) {
    parse(input_reply_to, parser);
  } else if (!message_id.is_any_server()) {
    if (reply_to_story_full_id.is_valid()) {
      input_reply_to = MessageInputReplyTo(reply_to_story_full_id);
    } else if (legacy_reply_to_message_id.is_valid()) {
      input_reply_to = MessageInputReplyTo{legacy_reply_to_message_id, DialogId(), MessageQuote()};
    }
  }
  if (has_replied_message_info) {
    parse(replied_message_info, parser);
  } else {
    replied_message_info = RepliedMessageInfo::legacy(legacy_reply_to_message_id, legacy_reply_in_dialog_id);
  }
  if (has_saved_messages_topic_id) {
    parse(saved_messages_topic_id, parser);
  }
  if (has_initial_top_thread_message_id) {
    parse(initial_top_thread_message_id, parser);
  }
  if (has_sender_boost_count) {
    parse(sender_boost_count, parser);
  }
  if (has_via_business_bot_user_id) {
    parse(via_business_bot_user_id, parser);
  }

  CHECK(content != nullptr);
  is_content_secret |= ttl.is_secret_message_content(content->get_type());  // repair is_content_secret for old messages
  if (hide_edit_date && content->get_type() == MessageContentType::LiveLocation) {
    hide_edit_date = false;
  }
}

template <class StorerT>
void MessagesManager::Dialog::store(StorerT &storer) const {
  using td::store;
  const Message *last_database_message = nullptr;
  if (last_database_message_id.is_valid()) {
    last_database_message = get_message(this, last_database_message_id);
  }

  auto dialog_type = dialog_id.get_type();
  bool has_draft_message = draft_message != nullptr;
  bool has_last_database_message = last_database_message != nullptr;
  bool has_first_database_message_id = first_database_message_id.is_valid();
  bool has_first_database_message_id_by_index = true;
  bool has_message_count_by_index = true;
  bool has_client_data = !client_data.empty();
  bool has_last_read_all_mentions_message_id = last_read_all_mentions_message_id.is_valid();
  bool has_max_unavailable_message_id = max_unavailable_message_id.is_valid();
  bool has_local_unread_count = local_unread_count != 0;
  bool has_deleted_last_message = delete_last_message_date > 0;
  bool has_last_clear_history_message_id = last_clear_history_message_id.is_valid();
  bool has_last_database_message_id = !has_last_database_message && last_database_message_id.is_valid();
  bool has_message_notification_group =
      notification_info != nullptr && notification_info->message_notification_group_.is_active();
  bool has_mention_notification_group =
      notification_info != nullptr && notification_info->mention_notification_group_.is_active();
  bool has_new_secret_chat_notification_id =
      notification_info != nullptr && notification_info->new_secret_chat_notification_id_.is_valid();
  bool has_pinned_message_notification =
      notification_info != nullptr && notification_info->pinned_message_notification_message_id_.is_valid();
  bool has_last_pinned_message_id = last_pinned_message_id.is_valid();
  bool has_flags2 = true;
  bool has_max_push_notification_message_id =
      notification_info != nullptr && notification_info->max_push_notification_message_id_.is_valid() &&
      notification_info->max_push_notification_message_id_ > last_new_message_id;
  bool has_folder_id = folder_id != FolderId();
  bool has_pending_read_channel_inbox = pending_read_channel_inbox_pts != 0;
  bool has_last_yet_unsent_message = last_message_id.is_valid() && last_message_id.is_yet_unsent();
  bool has_active_group_call_id = active_group_call_id.is_valid();
  bool has_message_ttl = !message_ttl.is_empty();
  bool has_default_join_group_call_as_dialog_id = default_join_group_call_as_dialog_id.is_valid();
  bool store_has_bots = dialog_type == DialogType::Chat || dialog_type == DialogType::Channel;
  bool has_theme_name = !theme_name.empty();
  bool has_flags3 = true;
  bool has_pending_join_requests = pending_join_request_count != 0;
  bool has_action_bar = action_bar != nullptr;
  bool has_default_send_message_as_dialog_id = default_send_message_as_dialog_id.is_valid();
  bool has_legacy_available_reactions = false;
  bool has_available_reactions_generation = available_reactions_generation != 0;
  bool has_have_full_history_source = have_full_history && have_full_history_source != 0;
  bool has_available_reactions = !available_reactions.empty();
  bool has_history_generation = history_generation != 0;
  bool has_background = background_info.is_valid();
  bool has_business_bot_manage_bar = business_bot_manage_bar != nullptr;
  BEGIN_STORE_FLAGS();
  STORE_FLAG(has_draft_message);
  STORE_FLAG(has_last_database_message);
  STORE_FLAG(false);  // legacy_know_can_report_spam
  STORE_FLAG(false);  // action_bar->can_report_spam
  STORE_FLAG(has_first_database_message_id);
  STORE_FLAG(false);  // legacy_is_pinned
  STORE_FLAG(has_first_database_message_id_by_index);
  STORE_FLAG(has_message_count_by_index);
  STORE_FLAG(has_client_data);
  STORE_FLAG(need_restore_reply_markup);
  STORE_FLAG(have_full_history);
  STORE_FLAG(has_last_read_all_mentions_message_id);
  STORE_FLAG(has_max_unavailable_message_id);
  STORE_FLAG(is_last_read_inbox_message_id_inited);
  STORE_FLAG(is_last_read_outbox_message_id_inited);
  STORE_FLAG(has_local_unread_count);
  STORE_FLAG(has_deleted_last_message);
  STORE_FLAG(has_last_clear_history_message_id);
  STORE_FLAG(is_last_message_deleted_locally);
  STORE_FLAG(has_contact_registered_message);
  STORE_FLAG(has_last_database_message_id);
  STORE_FLAG(need_repair_server_unread_count);
  STORE_FLAG(is_marked_as_unread);
  STORE_FLAG(has_message_notification_group);
  STORE_FLAG(has_mention_notification_group);
  STORE_FLAG(has_new_secret_chat_notification_id);
  STORE_FLAG(has_pinned_message_notification);
  STORE_FLAG(has_last_pinned_message_id);
  STORE_FLAG(is_last_pinned_message_id_inited);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();

  store(dialog_id, storer);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_max_push_notification_message_id);
    STORE_FLAG(has_folder_id);
    STORE_FLAG(is_folder_id_inited);
    STORE_FLAG(has_pending_read_channel_inbox);
    STORE_FLAG(know_action_bar);
    STORE_FLAG(false);  // action_bar->can_add_contact
    STORE_FLAG(false);  // action_bar->can_block_user
    STORE_FLAG(false);  // action_bar->can_share_phone_number
    STORE_FLAG(false);  // action_bar->can_report_location
    STORE_FLAG(has_scheduled_server_messages);
    STORE_FLAG(has_scheduled_database_messages);
    STORE_FLAG(need_repair_channel_server_unread_count);
    STORE_FLAG(false);  // action_bar->can_unarchive
    STORE_FLAG(false);  // action_bar_has_distance
    STORE_FLAG(has_outgoing_messages);
    STORE_FLAG(has_last_yet_unsent_message);
    STORE_FLAG(is_blocked);
    STORE_FLAG(is_is_blocked_inited);
    STORE_FLAG(has_active_group_call);
    STORE_FLAG(is_group_call_empty);
    STORE_FLAG(has_active_group_call_id);
    STORE_FLAG(false);  // action_bar->can_invite_members
    STORE_FLAG(has_message_ttl);
    STORE_FLAG(is_message_ttl_inited);
    STORE_FLAG(has_default_join_group_call_as_dialog_id);
    STORE_FLAG(store_has_bots ? has_bots : false);
    STORE_FLAG(store_has_bots ? is_has_bots_inited : false);
    STORE_FLAG(is_theme_name_inited);
    STORE_FLAG(has_theme_name);
    STORE_FLAG(has_flags3);
    END_STORE_FLAGS();
  }
  if (has_flags3) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_pending_join_requests);
    STORE_FLAG(need_repair_action_bar);
    STORE_FLAG(has_action_bar);
    STORE_FLAG(has_default_send_message_as_dialog_id);
    STORE_FLAG(need_drop_default_send_message_as_dialog_id);
    STORE_FLAG(has_legacy_available_reactions);
    STORE_FLAG(is_available_reactions_inited);
    STORE_FLAG(has_available_reactions_generation);
    STORE_FLAG(has_have_full_history_source);
    STORE_FLAG(has_available_reactions);
    STORE_FLAG(has_history_generation);
    STORE_FLAG(need_repair_unread_reaction_count);
    STORE_FLAG(is_translatable);
    STORE_FLAG(need_repair_unread_mention_count);
    STORE_FLAG(is_background_inited);
    STORE_FLAG(has_background);
    STORE_FLAG(is_blocked_for_stories);
    STORE_FLAG(is_is_blocked_for_stories_inited);
    STORE_FLAG(view_as_messages);
    STORE_FLAG(is_view_as_messages_inited);
    STORE_FLAG(is_forum);
    STORE_FLAG(is_saved_messages_view_as_messages_inited);
    STORE_FLAG(has_business_bot_manage_bar);
    END_STORE_FLAGS();
  }

  store(last_new_message_id, storer);
  store(server_unread_count, storer);
  if (has_local_unread_count) {
    store(local_unread_count, storer);
  }
  store(last_read_inbox_message_id, storer);
  store(last_read_outbox_message_id, storer);
  store(reply_markup_message_id, storer);
  store(notification_settings, storer);
  if (has_draft_message) {
    store(draft_message, storer);
  }
  store(last_clear_history_date, storer);
  store(order, storer);
  if (has_last_database_message) {
    store(*last_database_message, storer);
  }
  if (has_first_database_message_id) {
    store(first_database_message_id, storer);
  }
  if (has_deleted_last_message) {
    store(delete_last_message_date, storer);
    store(deleted_last_message_id, storer);
  }
  if (has_last_clear_history_message_id) {
    store(last_clear_history_message_id, storer);
  }

  if (has_first_database_message_id_by_index) {
    store(static_cast<int32>(first_database_message_id_by_index.size()), storer);
    for (auto first_message_id : first_database_message_id_by_index) {
      store(first_message_id, storer);
    }
  }
  if (has_message_count_by_index) {
    store(static_cast<int32>(message_count_by_index.size()), storer);
    for (auto message_count : message_count_by_index) {
      store(message_count, storer);
    }
  }
  if (has_client_data) {
    store(client_data, storer);
  }
  if (has_last_read_all_mentions_message_id) {
    store(last_read_all_mentions_message_id, storer);
  }
  if (has_max_unavailable_message_id) {
    store(max_unavailable_message_id, storer);
  }
  if (has_last_database_message_id) {
    store(last_database_message_id, storer);
  }
  if (has_message_notification_group) {
    store(notification_info->message_notification_group_, storer);
  }
  if (has_mention_notification_group) {
    store(notification_info->mention_notification_group_, storer);
  }
  if (has_new_secret_chat_notification_id) {
    store(notification_info->new_secret_chat_notification_id_, storer);
  }
  if (has_pinned_message_notification) {
    store(notification_info->pinned_message_notification_message_id_, storer);
  }
  if (has_last_pinned_message_id) {
    store(last_pinned_message_id, storer);
  }
  if (has_max_push_notification_message_id) {
    store(notification_info->max_push_notification_message_id_, storer);
  }
  if (has_folder_id) {
    store(folder_id, storer);
  }
  if (has_pending_read_channel_inbox) {
    store(pending_read_channel_inbox_pts, storer);
    store(pending_read_channel_inbox_max_message_id, storer);
    store(pending_read_channel_inbox_server_unread_count, storer);
  }
  if (has_active_group_call_id) {
    store(active_group_call_id, storer);
  }
  if (has_message_ttl) {
    store(message_ttl, storer);
  }
  if (has_default_join_group_call_as_dialog_id) {
    store(default_join_group_call_as_dialog_id, storer);
  }
  if (has_theme_name) {
    store(theme_name, storer);
  }
  if (has_pending_join_requests) {
    store(pending_join_request_count, storer);
    store(pending_join_request_user_ids, storer);
  }
  if (has_action_bar) {
    store(action_bar, storer);
  }
  if (has_default_send_message_as_dialog_id) {
    store(default_send_message_as_dialog_id, storer);
  }
  if (has_available_reactions) {
    store(available_reactions, storer);
  }
  if (has_available_reactions_generation) {
    store(available_reactions_generation, storer);
  }
  if (has_have_full_history_source) {
    store(have_full_history_source, storer);
  }
  if (has_history_generation) {
    store(history_generation, storer);
  }
  if (has_background) {
    store(background_info, storer);
  }
  if (has_business_bot_manage_bar) {
    store(business_bot_manage_bar, storer);
  }
}

// do not forget to resolve dialog dependencies including dependencies of last_message
template <class ParserT>
void MessagesManager::Dialog::parse(ParserT &parser) {
  using td::parse;
  bool has_draft_message;
  bool has_last_database_message;
  bool legacy_know_can_report_spam;
  bool has_first_database_message_id;
  bool legacy_is_pinned;
  bool has_first_database_message_id_by_index;
  bool has_message_count_by_index;
  bool has_client_data;
  bool has_last_read_all_mentions_message_id;
  bool has_max_unavailable_message_id;
  bool has_local_unread_count;
  bool has_deleted_last_message;
  bool has_last_clear_history_message_id;
  bool has_last_database_message_id;
  bool has_message_notification_group;
  bool has_mention_notification_group;
  bool has_new_secret_chat_notification_id;
  bool has_pinned_message_notification;
  bool has_last_pinned_message_id;
  bool has_flags2;
  bool has_max_push_notification_message_id = false;
  bool has_folder_id = false;
  bool has_pending_read_channel_inbox = false;
  bool has_active_group_call_id = false;
  bool has_message_ttl = false;
  bool has_default_join_group_call_as_dialog_id = false;
  bool has_theme_name = false;
  bool has_flags3 = false;
  bool has_pending_join_requests = false;
  bool action_bar_can_report_spam = false;
  bool action_bar_can_add_contact = false;
  bool action_bar_can_block_user = false;
  bool action_bar_can_share_phone_number = false;
  bool action_bar_can_report_location = false;
  bool action_bar_can_unarchive = false;
  bool action_bar_has_distance = false;
  bool action_bar_can_invite_members = false;
  bool has_action_bar = false;
  bool has_default_send_message_as_dialog_id = false;
  bool has_legacy_available_reactions = false;
  bool has_available_reactions_generation = false;
  bool has_have_full_history_source = false;
  bool has_available_reactions = false;
  bool has_history_generation = false;
  bool has_background = false;
  bool has_business_bot_manage_bar = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(has_draft_message);
  PARSE_FLAG(has_last_database_message);
  PARSE_FLAG(legacy_know_can_report_spam);
  PARSE_FLAG(action_bar_can_report_spam);
  PARSE_FLAG(has_first_database_message_id);
  PARSE_FLAG(legacy_is_pinned);
  PARSE_FLAG(has_first_database_message_id_by_index);
  PARSE_FLAG(has_message_count_by_index);
  PARSE_FLAG(has_client_data);
  PARSE_FLAG(need_restore_reply_markup);
  PARSE_FLAG(have_full_history);
  PARSE_FLAG(has_last_read_all_mentions_message_id);
  PARSE_FLAG(has_max_unavailable_message_id);
  PARSE_FLAG(is_last_read_inbox_message_id_inited);
  PARSE_FLAG(is_last_read_outbox_message_id_inited);
  PARSE_FLAG(has_local_unread_count);
  PARSE_FLAG(has_deleted_last_message);
  PARSE_FLAG(has_last_clear_history_message_id);
  PARSE_FLAG(is_last_message_deleted_locally);
  PARSE_FLAG(has_contact_registered_message);
  PARSE_FLAG(has_last_database_message_id);
  PARSE_FLAG(need_repair_server_unread_count);
  PARSE_FLAG(is_marked_as_unread);
  PARSE_FLAG(has_message_notification_group);
  PARSE_FLAG(has_mention_notification_group);
  PARSE_FLAG(has_new_secret_chat_notification_id);
  PARSE_FLAG(has_pinned_message_notification);
  PARSE_FLAG(has_last_pinned_message_id);
  PARSE_FLAG(is_last_pinned_message_id_inited);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();

  parse(dialog_id, parser);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_max_push_notification_message_id);
    PARSE_FLAG(has_folder_id);
    PARSE_FLAG(is_folder_id_inited);
    PARSE_FLAG(has_pending_read_channel_inbox);
    PARSE_FLAG(know_action_bar);
    PARSE_FLAG(action_bar_can_add_contact);
    PARSE_FLAG(action_bar_can_block_user);
    PARSE_FLAG(action_bar_can_share_phone_number);
    PARSE_FLAG(action_bar_can_report_location);
    PARSE_FLAG(has_scheduled_server_messages);
    PARSE_FLAG(has_scheduled_database_messages);
    PARSE_FLAG(need_repair_channel_server_unread_count);
    PARSE_FLAG(action_bar_can_unarchive);
    PARSE_FLAG(action_bar_has_distance);
    PARSE_FLAG(has_outgoing_messages);
    PARSE_FLAG(had_last_yet_unsent_message);
    PARSE_FLAG(is_blocked);
    PARSE_FLAG(is_is_blocked_inited);
    PARSE_FLAG(has_active_group_call);
    PARSE_FLAG(is_group_call_empty);
    PARSE_FLAG(has_active_group_call_id);
    PARSE_FLAG(action_bar_can_invite_members);
    PARSE_FLAG(has_message_ttl);
    PARSE_FLAG(is_message_ttl_inited);
    PARSE_FLAG(has_default_join_group_call_as_dialog_id);
    PARSE_FLAG(has_bots);
    PARSE_FLAG(is_has_bots_inited);
    PARSE_FLAG(is_theme_name_inited);
    PARSE_FLAG(has_theme_name);
    PARSE_FLAG(has_flags3);
    END_PARSE_FLAGS();
  } else {
    is_folder_id_inited = false;
    has_scheduled_server_messages = false;
    has_scheduled_database_messages = false;
    need_repair_channel_server_unread_count = false;
    has_outgoing_messages = false;
    had_last_yet_unsent_message = false;
    is_blocked = false;
    is_is_blocked_inited = false;
    has_active_group_call = false;
    is_group_call_empty = false;
    is_message_ttl_inited = false;
    has_bots = false;
    is_has_bots_inited = false;
    is_theme_name_inited = false;
  }
  if (has_flags3) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_pending_join_requests);
    PARSE_FLAG(need_repair_action_bar);
    PARSE_FLAG(has_action_bar);
    PARSE_FLAG(has_default_send_message_as_dialog_id);
    PARSE_FLAG(need_drop_default_send_message_as_dialog_id);
    PARSE_FLAG(has_legacy_available_reactions);
    PARSE_FLAG(is_available_reactions_inited);
    PARSE_FLAG(has_available_reactions_generation);
    PARSE_FLAG(has_have_full_history_source);
    PARSE_FLAG(has_available_reactions);
    PARSE_FLAG(has_history_generation);
    PARSE_FLAG(need_repair_unread_reaction_count);
    PARSE_FLAG(is_translatable);
    PARSE_FLAG(need_repair_unread_mention_count);
    PARSE_FLAG(is_background_inited);
    PARSE_FLAG(has_background);
    PARSE_FLAG(is_blocked_for_stories);
    PARSE_FLAG(is_is_blocked_for_stories_inited);
    PARSE_FLAG(view_as_messages);
    PARSE_FLAG(is_view_as_messages_inited);
    PARSE_FLAG(is_forum);
    PARSE_FLAG(is_saved_messages_view_as_messages_inited);
    PARSE_FLAG(has_business_bot_manage_bar);
    END_PARSE_FLAGS();
  } else {
    need_repair_action_bar = false;
    is_available_reactions_inited = false;
    is_background_inited = false;
    is_blocked_for_stories = false;
    is_is_blocked_for_stories_inited = false;
    view_as_messages = false;
    is_view_as_messages_inited = false;
    is_forum = false;
    is_saved_messages_view_as_messages_inited = false;
  }

  parse(last_new_message_id, parser);
  parse(server_unread_count, parser);
  if (has_local_unread_count) {
    parse(local_unread_count, parser);
  }
  parse(last_read_inbox_message_id, parser);
  if (last_read_inbox_message_id.is_valid()) {
    is_last_read_inbox_message_id_inited = true;
  }
  parse(last_read_outbox_message_id, parser);
  if (last_read_outbox_message_id.is_valid()) {
    is_last_read_outbox_message_id_inited = true;
  }
  parse(reply_markup_message_id, parser);
  parse(notification_settings, parser);
  if (has_draft_message) {
    parse(draft_message, parser);
  }
  parse(last_clear_history_date, parser);
  parse(order, parser);
  if (has_last_database_message) {
    unique_ptr<Message> last_database_message;
    parse(last_database_message, parser);
    auto loaded_last_database_message_id = last_database_message->message_id;
    if (loaded_last_database_message_id.is_valid()) {
      messages.set(loaded_last_database_message_id, std::move(last_database_message));
    }
  }
  if (has_first_database_message_id) {
    parse(first_database_message_id, parser);
  }
  if (legacy_is_pinned) {
    int64 legacy_pinned_order;
    parse(legacy_pinned_order, parser);
  }
  if (has_deleted_last_message) {
    parse(delete_last_message_date, parser);
    parse(deleted_last_message_id, parser);
  }
  if (has_last_clear_history_message_id) {
    parse(last_clear_history_message_id, parser);
  }

  if (has_first_database_message_id_by_index) {
    int32 size;
    parse(size, parser);
    if (static_cast<size_t>(size) > first_database_message_id_by_index.size()) {
      // the log event is broken
      return parser.set_error("Wrong first_database_message_id_by_index table size");
    }
    for (int32 i = 0; i < size; i++) {
      parse(first_database_message_id_by_index[i], parser);
    }
  }
  if (has_message_count_by_index) {
    int32 size;
    parse(size, parser);
    if (static_cast<size_t>(size) > message_count_by_index.size()) {
      // the log event is broken
      return parser.set_error("Wrong message_count_by_index table size");
    }
    for (int32 i = 0; i < size; i++) {
      parse(message_count_by_index[i], parser);
    }
  }
  unread_mention_count = message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadMention)];
  LOG(INFO) << "Set unread mention message count in " << dialog_id << " to " << unread_mention_count;
  if (unread_mention_count < 0) {
    unread_mention_count = 0;
  }
  unread_reaction_count = message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadReaction)];
  LOG(INFO) << "Set unread reaction count in " << dialog_id << " to " << unread_reaction_count;
  if (unread_reaction_count < 0) {
    unread_reaction_count = 0;
  }
  if (has_client_data) {
    parse(client_data, parser);
  }
  if (has_last_read_all_mentions_message_id) {
    parse(last_read_all_mentions_message_id, parser);
  }
  if (has_max_unavailable_message_id) {
    parse(max_unavailable_message_id, parser);
  }
  if (has_last_database_message_id) {
    parse(last_database_message_id, parser);
  }
  if (has_message_notification_group) {
    parse(add_dialog_notification_info(this)->message_notification_group_, parser);
  }
  if (has_mention_notification_group) {
    parse(add_dialog_notification_info(this)->mention_notification_group_, parser);
  }
  if (has_new_secret_chat_notification_id) {
    parse(add_dialog_notification_info(this)->new_secret_chat_notification_id_, parser);
  }
  if (has_pinned_message_notification) {
    parse(add_dialog_notification_info(this)->pinned_message_notification_message_id_, parser);
  }
  if (has_last_pinned_message_id) {
    parse(last_pinned_message_id, parser);
  }
  if (has_max_push_notification_message_id) {
    parse(add_dialog_notification_info(this)->max_push_notification_message_id_, parser);
  }
  if (has_folder_id) {
    parse(folder_id, parser);
  }
  if (has_pending_read_channel_inbox) {
    parse(pending_read_channel_inbox_pts, parser);
    parse(pending_read_channel_inbox_max_message_id, parser);
    parse(pending_read_channel_inbox_server_unread_count, parser);
  }
  int32 action_bar_distance = -1;
  if (action_bar_has_distance) {
    parse(action_bar_distance, parser);
  }
  if (has_active_group_call_id) {
    parse(active_group_call_id, parser);
  }
  if (has_message_ttl) {
    parse(message_ttl, parser);
  }
  if (has_default_join_group_call_as_dialog_id) {
    parse(default_join_group_call_as_dialog_id, parser);
  }
  if (has_theme_name) {
    parse(theme_name, parser);
  }
  if (has_pending_join_requests) {
    parse(pending_join_request_count, parser);
    parse(pending_join_request_user_ids, parser);
  }
  if (has_action_bar) {
    parse(action_bar, parser);
  }
  if (has_default_send_message_as_dialog_id) {
    parse(default_send_message_as_dialog_id, parser);
  }
  if (has_available_reactions) {
    parse(available_reactions, parser);
  } else if (has_legacy_available_reactions) {
    vector<ReactionType> legacy_available_reaction_types;
    parse(legacy_available_reaction_types, parser);
    available_reactions = ChatReactions(std::move(legacy_available_reaction_types));
  }
  if (has_available_reactions_generation) {
    parse(available_reactions_generation, parser);
  }
  if (has_have_full_history_source) {
    parse(have_full_history_source, parser);
  }
  if (has_history_generation) {
    parse(history_generation, parser);
  }
  if (has_background) {
    parse(background_info, parser);
  }
  if (has_business_bot_manage_bar) {
    parse(business_bot_manage_bar, parser);
  }

  (void)legacy_know_can_report_spam;
  if (know_action_bar && !has_action_bar) {
    action_bar = DialogActionBar::create(
        action_bar_can_report_spam, action_bar_can_add_contact, action_bar_can_block_user,
        action_bar_can_share_phone_number, action_bar_can_report_location, action_bar_can_unarchive,
        has_outgoing_messages ? -1 : action_bar_distance, action_bar_can_invite_members, string(), false, 0);
  }
}

template <class StorerT>
void MessagesManager::CallsDbState::store(StorerT &storer) const {
  using td::store;
  store(static_cast<int32>(first_calls_database_message_id_by_index.size()), storer);
  for (auto first_message_id : first_calls_database_message_id_by_index) {
    store(first_message_id, storer);
  }
  store(static_cast<int32>(message_count_by_index.size()), storer);
  for (auto message_count : message_count_by_index) {
    store(message_count, storer);
  }
}

template <class ParserT>
void MessagesManager::CallsDbState::parse(ParserT &parser) {
  using td::parse;
  int32 size;
  parse(size, parser);
  if (static_cast<size_t>(size) > first_calls_database_message_id_by_index.size()) {
    return parser.set_error("Wrong first_calls_database_message_id_by_index table size");
  }
  for (int32 i = 0; i < size; i++) {
    parse(first_calls_database_message_id_by_index[i], parser);
  }
  parse(size, parser);
  if (static_cast<size_t>(size) > message_count_by_index.size()) {
    return parser.set_error("Wrong message_count_by_index table size");
  }
  for (int32 i = 0; i < size; i++) {
    parse(message_count_by_index[i], parser);
  }
}

void MessagesManager::load_calls_db_state() {
  if (!G()->use_message_database()) {
    return;
  }
  std::fill(calls_db_state_.message_count_by_index.begin(), calls_db_state_.message_count_by_index.end(), -1);
  auto value = G()->td_db()->get_sqlite_sync_pmc()->get("calls_db_state");
  if (value.empty()) {
    return;
  }
  log_event_parse(calls_db_state_, value).ensure();
  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
}

void MessagesManager::save_calls_db_state() {
  if (!G()->use_message_database()) {
    return;
  }

  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
  G()->td_db()->get_sqlite_pmc()->set("calls_db_state", log_event_store(calls_db_state_).as_slice().str(), Auto());
}

MessagesManager::MessagesManager(Td *td, ActorShared<> parent)
    : recently_found_dialogs_{td, "recently_found", MAX_RECENT_DIALOGS}
    , recently_opened_dialogs_{td, "recently_opened", MAX_RECENT_DIALOGS}
    , td_(td)
    , parent_(std::move(parent)) {
  upload_media_callback_ = std::make_shared<UploadMediaCallback>();
  upload_thumbnail_callback_ = std::make_shared<UploadThumbnailCallback>();

  channel_get_difference_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_timeout_.set_callback_data(static_cast<void *>(this));

  channel_get_difference_retry_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_retry_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_views_timeout_.set_callback(on_pending_message_views_timeout_callback);
  pending_message_views_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_live_location_view_timeout_.set_callback(on_pending_message_live_location_view_timeout_callback);
  pending_message_live_location_view_timeout_.set_callback_data(static_cast<void *>(this));

  pending_draft_message_timeout_.set_callback(on_pending_draft_message_timeout_callback);
  pending_draft_message_timeout_.set_callback_data(static_cast<void *>(this));

  pending_read_history_timeout_.set_callback(on_pending_read_history_timeout_callback);
  pending_read_history_timeout_.set_callback_data(static_cast<void *>(this));

  pending_updated_dialog_timeout_.set_callback(on_pending_updated_dialog_timeout_callback);
  pending_updated_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  pending_unload_dialog_timeout_.set_callback(on_pending_unload_dialog_timeout_callback);
  pending_unload_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  dialog_unmute_timeout_.set_callback(on_dialog_unmute_timeout_callback);
  dialog_unmute_timeout_.set_callback_data(static_cast<void *>(this));

  pending_send_dialog_action_timeout_.set_callback(on_pending_send_dialog_action_timeout_callback);
  pending_send_dialog_action_timeout_.set_callback_data(static_cast<void *>(this));

  preload_folder_dialog_list_timeout_.set_callback(on_preload_folder_dialog_list_timeout_callback);
  preload_folder_dialog_list_timeout_.set_callback_data(static_cast<void *>(this));

  update_viewed_messages_timeout_.set_callback(on_update_viewed_messages_timeout_callback);
  update_viewed_messages_timeout_.set_callback_data(static_cast<void *>(this));

  send_update_chat_read_inbox_timeout_.set_callback(on_send_update_chat_read_inbox_timeout_callback);
  send_update_chat_read_inbox_timeout_.set_callback_data(static_cast<void *>(this));
}

MessagesManager::~MessagesManager() {
  Scheduler::instance()->destroy_on_scheduler(
      G()->get_gc_scheduler_id(), ttl_nodes_, ttl_heap_, being_sent_messages_, update_message_ids_,
      update_scheduled_message_ids_, message_id_to_dialog_id_, last_clear_history_message_id_to_dialog_id_, dialogs_,
      postponed_chat_read_inbox_updates_, found_public_dialogs_, found_on_server_dialogs_, message_embedding_codes_[0],
      message_embedding_codes_[1], message_to_replied_media_timestamp_messages_,
      story_to_replied_media_timestamp_messages_, notification_group_id_to_dialog_id_, pending_get_channel_differences_,
      active_get_channel_differences_, get_channel_difference_to_log_event_id_, channel_get_difference_retry_timeouts_,
      is_channel_difference_finished_, expected_channel_pts_, expected_channel_max_message_id_,
      dialog_bot_command_message_ids_, message_full_id_to_file_source_id_, last_outgoing_forwarded_message_date_,
      dialog_viewed_messages_, previous_repaired_read_inbox_max_message_id_, failed_to_load_dialogs_);
}

MessagesManager::AddDialogData::AddDialogData(int32 dependent_dialog_count, unique_ptr<Message> &&last_message,
                                              unique_ptr<DraftMessage> &&draft_message)
    : dependent_dialog_count_(dependent_dialog_count)
    , last_message_(std::move(last_message))
    , draft_message_(std::move(draft_message)) {
}

void MessagesManager::on_channel_get_difference_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_channel_get_difference_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_views_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_pending_message_views_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_live_location_view_timeout_callback(void *messages_manager_ptr,
                                                                             int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::view_message_live_location_on_server, task_id);
}

void MessagesManager::on_pending_draft_message_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::save_dialog_draft_message_on_server, DialogId(dialog_id_int));
}

void MessagesManager::on_pending_read_history_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::do_read_history_on_server,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_updated_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  // no check for G()->close_flag() to save dialogs even while closing

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  // TODO it is unsafe to save dialog to database before binlog is flushed

  // no send_closure_later, because messages_manager can be not an actor while closing
  messages_manager->save_dialog_to_database(DialogId(dialog_id_int));
}

void MessagesManager::on_pending_unload_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::unload_dialog,
                     DialogId(dialog_id_int), -1);
}

void MessagesManager::on_dialog_unmute_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_dialog_unmute,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_send_dialog_action_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_send_dialog_action_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_preload_folder_dialog_list_timeout_callback(void *messages_manager_ptr, int64 folder_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::preload_folder_dialog_list,
                     FolderId(narrow_cast<int32>(folder_id_int)));
}

void MessagesManager::on_update_viewed_messages_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_update_viewed_messages_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_send_update_chat_read_inbox_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::on_send_update_chat_read_inbox_timeout, DialogId(dialog_id_int));
}

BufferSlice MessagesManager::get_dialog_database_value(const Dialog *d) {
  // can't use log_event_store, because it tries to parse stored Dialog
  LogEventStorerCalcLength storer_calc_length;
  store(*d, storer_calc_length);

  BufferSlice value_buffer{storer_calc_length.get_length()};
  auto value = value_buffer.as_mutable_slice();

  LogEventStorerUnsafe storer_unsafe(value.ubegin());
  store(*d, storer_unsafe);
  return value_buffer;
}

void MessagesManager::save_dialog_to_database(DialogId dialog_id) {
  CHECK(G()->use_message_database());
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Save " << dialog_id << " to database";
  vector<NotificationGroupKey> changed_group_keys;
  if (d->notification_info != nullptr) {
    d->notification_info->message_notification_group_.add_group_key_if_changed(changed_group_keys, dialog_id);
    d->notification_info->mention_notification_group_.add_group_key_if_changed(changed_group_keys, dialog_id);
  }
  bool can_reuse_notification_group = false;
  for (auto &group_key : changed_group_keys) {
    if (group_key.dialog_id == DialogId()) {
      can_reuse_notification_group = true;
    }
  }
  G()->td_db()->get_dialog_db_async()->add_dialog(
      dialog_id, d->folder_id, d->order, get_dialog_database_value(d), std::move(changed_group_keys),
      PromiseCreator::lambda([dialog_id, can_reuse_notification_group](Result<> result) {
        send_closure(G()->messages_manager(), &MessagesManager::on_save_dialog_to_database, dialog_id,
                     can_reuse_notification_group, result.is_ok());
      }));
}

void MessagesManager::on_save_dialog_to_database(DialogId dialog_id, bool can_reuse_notification_group, bool success) {
  LOG(INFO) << "Successfully saved " << dialog_id << " to database";

  if (success && can_reuse_notification_group && !G()->close_flag()) {
    auto d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    if (d->notification_info != nullptr) {
      try_reuse_notification_group(d->notification_info->message_notification_group_);
      try_reuse_notification_group(d->notification_info->mention_notification_group_);
    }
  }

  // TODO erase some events from binlog
}

void MessagesManager::try_reuse_notification_group(NotificationGroupInfo &group_info) {
  auto group_id = group_info.get_reused_group_id();
  if (group_id.is_valid()) {
    send_closure_later(G()->notification_manager(), &NotificationManager::try_reuse_notification_group_id, group_id);
    notification_group_id_to_dialog_id_.erase(group_id);
  }
}

void MessagesManager::invalidate_message_indexes(Dialog *d) {
  CHECK(d != nullptr);
  bool is_secret = d->dialog_id.get_type() == DialogType::SecretChat;
  for (size_t i = 0; i < d->message_count_by_index.size(); i++) {
    if (is_secret || i == static_cast<size_t>(message_search_filter_index(MessageSearchFilter::FailedToSend))) {
      // always know all messages
      d->first_database_message_id_by_index[i] = MessageId::min();
      // keep the count
    } else {
      // some messages are unknown; drop first_database_message_id and count
      d->first_database_message_id_by_index[i] = MessageId();
      d->message_count_by_index[i] = -1;
    }
  }
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, const Message *m) {
  auto index_mask = get_message_index_mask(d->dialog_id, m);
  index_mask &= ~message_search_filter_index_mask(
      MessageSearchFilter::UnreadMention);  // unread mention count has been already manually updated
  index_mask &= ~message_search_filter_index_mask(
      MessageSearchFilter::UnreadReaction);  // unread reaction count has been already manually updated

  update_message_count_by_index(d, diff, index_mask);
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, int32 index_mask) {
  if (index_mask == 0) {
    return;
  }

  LOG(INFO) << "Update message count by " << diff << " in index mask " << index_mask;
  int i = 0;
  for (auto &message_count : d->message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat ||
            i == message_search_filter_index(MessageSearchFilter::FailedToSend)) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      on_dialog_updated(d->dialog_id, "update_message_count_by_index");
    }
    i++;
  }

  i = static_cast<int>(MessageSearchFilter::Call) - 1;
  for (auto &message_count : calls_db_state_.message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      save_calls_db_state();
    }
    i++;
  }
}

int32 MessagesManager::get_message_index_mask(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || m->message_id.is_scheduled() || m->message_id.is_yet_unsent()) {
    return 0;
  }
  if (m->is_failed_to_send) {
    return message_search_filter_index_mask(MessageSearchFilter::FailedToSend);
  }
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  if (!m->message_id.is_server() && !is_secret) {
    return 0;
  }

  int32 index_mask = 0;
  if (m->is_pinned) {
    index_mask |= message_search_filter_index_mask(MessageSearchFilter::Pinned);
  }
  // retain second condition just in case
  if (m->is_content_secret || (!m->ttl.is_empty() && !is_secret)) {
    return index_mask;
  }
  index_mask |= get_message_content_index_mask(m->content.get(), td_, m->is_outgoing);
  if (m->contains_mention) {
    index_mask |= message_search_filter_index_mask(MessageSearchFilter::Mention);
    if (m->contains_unread_mention) {
      index_mask |= message_search_filter_index_mask(MessageSearchFilter::UnreadMention);
    }
  }
  if (has_unread_message_reactions(dialog_id, m)) {
    index_mask |= message_search_filter_index_mask(MessageSearchFilter::UnreadReaction);
  }
  LOG(INFO) << "Have index mask " << index_mask << " for " << m->message_id << " in " << dialog_id;
  return index_mask;
}

void MessagesManager::update_reply_count_by_message(Dialog *d, int diff, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || !m->top_thread_message_id.is_valid() ||
      m->top_thread_message_id == m->message_id || !m->message_id.is_valid() || !m->message_id.is_server()) {
    return;
  }

  update_message_reply_count(d, m->top_thread_message_id, get_message_sender(m), m->message_id,
                             diff < 0 ? G()->unix_time() : m->date, diff);
}

void MessagesManager::update_message_reply_count(Dialog *d, MessageId message_id, DialogId replier_dialog_id,
                                                 MessageId reply_message_id, int32 update_date, int diff,
                                                 bool is_recursive) {
  if (d == nullptr) {
    return;
  }

  Message *m = get_message(d, message_id);
  if (m == nullptr || !is_active_message_reply_info(d->dialog_id, m->reply_info)) {
    return;
  }
  LOG(INFO) << "Update reply count to " << message_id << " in " << d->dialog_id << " by " << diff << " from "
            << reply_message_id << " sent by " << replier_dialog_id;
  if (m->interaction_info_update_date < update_date &&
      m->reply_info.add_reply(replier_dialog_id, reply_message_id, diff)) {
    on_message_reply_info_changed(d->dialog_id, m);
    on_message_changed(d, m, true, "update_message_reply_count_by_message");
  }

  if (!is_recursive && is_discussion_message(d->dialog_id, m)) {
    auto linked_message_full_id = m->forward_info->get_last_message_full_id();
    update_message_reply_count(get_dialog(linked_message_full_id.get_dialog_id()),
                               linked_message_full_id.get_message_id(), replier_dialog_id, reply_message_id,
                               update_date, diff, true);
  }
}

bool MessagesManager::have_dialog_scheduled_messages_in_memory(const Dialog *d) {
  return d->scheduled_messages != nullptr && !d->scheduled_messages->scheduled_messages_.empty();
}

bool MessagesManager::is_allowed_useless_update(const tl_object_ptr<telegram_api::Update> &update) {
  auto constructor_id = update->get_id();
  if (constructor_id == dummyUpdate::ID) {
    // allow dummyUpdate just in case
    return true;
  }
  if (constructor_id == telegram_api::updateNewMessage::ID ||
      constructor_id == telegram_api::updateNewChannelMessage::ID) {
    // new outgoing messages are received again if random_id coincide
    return true;
  }

  return false;
}

void MessagesManager::skip_old_pending_pts_update(tl_object_ptr<telegram_api::Update> &&update, int32 new_pts,
                                                  int32 old_pts, int32 pts_count, const char *source) {
  LOG(DEBUG) << "Skip old update with PTS = " << new_pts << ", current PTS = " << old_pts;
  if (update->get_id() == telegram_api::updateNewMessage::ID) {
    auto update_new_message = static_cast<telegram_api::updateNewMessage *>(update.get());
    auto message_full_id = MessageFullId::get_message_full_id(update_new_message->message_, false);
    if (update_message_ids_.count(message_full_id) > 0) {
      // apply the sent message anyway, even it could have been deleted or edited already

      CHECK(message_full_id.get_dialog_id().get_type() == DialogType::User ||
            message_full_id.get_dialog_id().get_type() == DialogType::Chat);  // checked in check_pts_update
      delete_messages_from_updates({message_full_id.get_message_id()}, false);

      auto added_message_full_id = on_get_message(std::move(update_new_message->message_), true, false, false,
                                                  "updateNewMessage with an awaited message");
      if (added_message_full_id != message_full_id) {
        LOG(ERROR) << "Failed to add an awaited " << message_full_id << " from " << source;
      }
      return;
    }
  }
  if (update->get_id() == updateSentMessage::ID) {
    auto update_sent_message = static_cast<updateSentMessage *>(update.get());
    if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
      // apply the sent message anyway, even it could have been deleted or edited already
      delete_messages_from_updates({update_sent_message->message_id_}, false);
      on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                              update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                              "process old updateSentMessage");
      return;
    }
    return;
  }

  // very old or unuseful update
  LOG_IF(WARNING, new_pts == old_pts && pts_count == 0 && !is_allowed_useless_update(update))
      << "Receive useless update " << oneline(to_string(update)) << " from " << source;
}

MessagesManager::Dialog *MessagesManager::get_service_notifications_dialog() {
  UserId service_notifications_user_id = td_->user_manager_->add_service_notifications_user();
  DialogId service_notifications_dialog_id(service_notifications_user_id);
  force_create_dialog(service_notifications_dialog_id, "get_service_notifications_dialog");
  return get_dialog(service_notifications_dialog_id);
}

void MessagesManager::extract_authentication_codes(DialogId dialog_id, const Message *m,
                                                   vector<string> &authentication_codes) {
  CHECK(m != nullptr);
  if (dialog_id != DialogId(UserManager::get_service_notifications_user_id()) || !m->message_id.is_valid() ||
      !m->message_id.is_server() || m->content->get_type() != MessageContentType::Text || m->is_outgoing) {
    return;
  }
  auto *formatted_text = get_message_content_text(m->content.get());
  CHECK(formatted_text != nullptr);
  const string &text = formatted_text->text;
  for (size_t i = 0; i < text.size(); i++) {
    if (is_digit(text[i])) {
      string code;
      do {
        if (is_digit(text[i])) {
          code += text[i++];
          continue;
        }
        if (text[i] == '-') {
          i++;
          continue;
        }
        break;
      } while (true);
      if (5 <= code.size() && code.size() <= 7) {
        authentication_codes.push_back(code);
      }
    }
  }
}

void MessagesManager::save_auth_notification_ids() {
  auto min_date = G()->unix_time() - AUTH_NOTIFICATION_ID_CACHE_TIME;
  vector<string> stored_ids;
  for (const auto &it : auth_notification_id_date_) {
    auto date = it.second;
    if (date < min_date) {
      continue;
    }
    stored_ids.push_back(it.first);
    stored_ids.push_back(to_string(date));
  }

  if (stored_ids.empty()) {
    G()->td_db()->get_binlog_pmc()->erase("auth_notification_ids");
    return;
  }

  G()->td_db()->get_binlog_pmc()->set("auth_notification_ids", implode(stored_ids, ','));
}

void MessagesManager::on_update_service_notification(tl_object_ptr<telegram_api::updateServiceNotification> &&update,
                                                     bool skip_new_entities, Promise<Unit> &&promise) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  bool has_date = (update->flags_ & telegram_api::updateServiceNotification::INBOX_DATE_MASK) != 0;
  auto date = has_date ? update->inbox_date_ : G()->unix_time();
  if (date <= 0) {
    LOG(ERROR) << "Receive message date " << date << " in " << to_string(update);
    return;
  }
  bool is_auth_notification = update->type_.size() >= 5 && begins_with(update->type_, "auth");
  if (is_auth_notification) {
    auto &old_date = auth_notification_id_date_[update->type_.substr(4)];
    if (date <= old_date) {
      LOG(INFO) << "Skip already applied " << to_string(update);
      return;
    }
    old_date = date;

    if (auth_notification_id_date_.size() > MAX_SAVED_AUTH_NOTIFICATION_IDS) {
      auto min_date = date + 1;
      const string *min_key = nullptr;
      for (const auto &it : auth_notification_id_date_) {
        if (it.second < min_date) {
          min_date = it.second;
          min_key = &it.first;
        }
      }
      CHECK(min_key != nullptr);
      auth_notification_id_date_.erase(*min_key);
    }
  }

  bool is_authorized = td_->auth_manager_->is_authorized();
  bool is_user = is_authorized && !td_->auth_manager_->is_bot();
  auto user_manager = is_authorized ? td_->user_manager_.get() : nullptr;
  auto message_text = get_message_text(user_manager, std::move(update->message_), std::move(update->entities_),
                                       skip_new_entities, !is_user, date, false, "on_update_service_notification");
  DialogId owner_dialog_id = is_user ? get_service_notifications_dialog()->dialog_id : DialogId();
  MessageSelfDestructType ttl;
  bool disable_web_page_preview = false;
  auto content = get_message_content(td_, std::move(message_text), std::move(update->media_), owner_dialog_id, date,
                                     false, UserId(), &ttl, &disable_web_page_preview, "updateServiceNotification");
  bool is_content_secret = ttl.is_secret_message_content(content->get_type());

  if (update->popup_) {
    send_closure(
        G()->td(), &Td::send_update,
        td_api::make_object<td_api::updateServiceNotification>(
            update->type_, get_message_content_object(content.get(), td_, owner_dialog_id, date, is_content_secret,
                                                      true, -1, update->invert_media_, disable_web_page_preview)));
  }
  if (has_date && is_user) {
    Dialog *d = get_service_notifications_dialog();
    CHECK(d != nullptr);
    auto dialog_id = d->dialog_id;
    CHECK(dialog_id.get_type() == DialogType::User);

    auto new_message = make_unique<Message>();
    new_message->message_id = get_next_local_message_id(d);
    new_message->sender_user_id = dialog_id.get_user_id();
    new_message->date = date;
    new_message->ttl = ttl;
    new_message->disable_web_page_preview = disable_web_page_preview;
    new_message->is_content_secret = is_content_secret;
    new_message->invert_media = update->invert_media_;
    new_message->content = std::move(content);

    bool need_update = true;
    bool need_update_dialog_pos = false;

    Dependencies dependencies;
    add_message_dependencies(dependencies, new_message.get());
    for (auto dependent_dialog_id : dependencies.get_dialog_ids()) {
      force_create_dialog(dependent_dialog_id, "on_update_service_notification", true);
    }

    const Message *m = add_message_to_dialog(d, std::move(new_message), false, true, &need_update,
                                             &need_update_dialog_pos, "on_update_service_notification");
    if (m != nullptr && need_update) {
      send_update_new_message(d, m);
    }
    register_new_local_message_id(d, m);

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "on_update_service_notification");
    }
  }
  promise.set_value(Unit());

  if (is_auth_notification) {
    save_auth_notification_ids();
  }
}

void MessagesManager::on_update_read_channel_inbox(tl_object_ptr<telegram_api::updateReadChannelInbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelInbox";
    return;
  }
  on_update_dialog_folder_id(DialogId(channel_id), FolderId(update->folder_id_));
  on_read_channel_inbox(channel_id, MessageId(ServerMessageId(update->max_id_)), update->still_unread_count_,
                        update->pts_, "updateReadChannelInbox");
}

void MessagesManager::on_update_read_channel_outbox(tl_object_ptr<telegram_api::updateReadChannelOutbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelOutbox";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  read_history_outbox(dialog_id, MessageId(ServerMessageId(update->max_id_)));
}

void MessagesManager::on_update_read_channel_messages_contents(
    tl_object_ptr<telegram_api::updateChannelReadMessagesContents> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelReadMessagesContents";
    return;
  }
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  DialogId dialog_id = DialogId(channel_id);

  Dialog *d = get_dialog_force(dialog_id, "on_update_read_channel_messages_contents");
  if (d == nullptr) {
    LOG(INFO) << "Receive read channel messages contents update in unknown " << dialog_id;
    return;
  }

  if ((update->flags_ & telegram_api::updateChannelReadMessagesContents::TOP_MSG_ID_MASK) != 0) {
    // TODO
    return;
  }

  for (auto &server_message_id : update->messages_) {
    read_channel_message_content_from_updates(d, MessageId(ServerMessageId(server_message_id)));
  }
}

void MessagesManager::on_update_read_message_comments(DialogId dialog_id, MessageId message_id,
                                                      MessageId max_message_id, MessageId last_read_inbox_message_id,
                                                      MessageId last_read_outbox_message_id, int32 unread_count) {
  Dialog *d = get_dialog_force(dialog_id, "on_update_read_message_comments");
  if (d == nullptr) {
    LOG(INFO) << "Ignore update of read message comments in unknown " << dialog_id << " in updateReadDiscussion";
    return;
  }

  auto m = get_message_force(d, message_id, "on_update_read_message_comments");
  if (m == nullptr || !m->message_id.is_server() || m->top_thread_message_id != m->message_id) {
    return;
  }
  if (m->is_topic_message) {
    td_->forum_topic_manager_->on_update_forum_topic_unread(
        dialog_id, message_id, max_message_id, last_read_inbox_message_id, last_read_outbox_message_id, unread_count);
  }
  if (!is_active_message_reply_info(dialog_id, m->reply_info)) {
    return;
  }
  if (m->reply_info.update_max_message_ids(max_message_id, last_read_inbox_message_id, last_read_outbox_message_id)) {
    on_message_reply_info_changed(dialog_id, m);
    on_message_changed(d, m, true, "on_update_read_message_comments");
  }
}

void MessagesManager::on_update_channel_too_long(tl_object_ptr<telegram_api::updateChannelTooLong> &&update,
                                                 bool force_apply) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelTooLong";
    return;
  }
  if (!td_->chat_manager_->have_channel_force(channel_id, "on_update_channel_too_long")) {
    LOG(INFO) << "Skip updateChannelTooLong about unknown " << channel_id;
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  auto d = get_dialog_force(dialog_id, "on_update_channel_too_long 4");
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      d = add_dialog(dialog_id, "on_update_channel_too_long 5");
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, "on_update_channel_too_long 6");
    }
  }

  if (d != nullptr) {
    if (update->pts_ == 0 || update->pts_ > d->pts) {
      get_channel_difference(dialog_id, d->pts, update->pts_, MessageId(), true, "on_update_channel_too_long 1");
    }
  } else {
    if (force_apply) {
      get_channel_difference(dialog_id, -1, update->pts_, MessageId(), true, "on_update_channel_too_long 2");
    } else {
      td_->updates_manager_->schedule_get_difference("on_update_channel_too_long 3");
    }
  }
}

void MessagesManager::on_update_message_view_count(MessageFullId message_full_id, int32 view_count) {
  if (view_count < 0) {
    LOG(ERROR) << "Receive " << view_count << " views in updateChannelMessageViews for " << message_full_id;
    return;
  }
  update_message_interaction_info(message_full_id, view_count, -1, false, nullptr, false, nullptr);
}

void MessagesManager::on_update_message_forward_count(MessageFullId message_full_id, int32 forward_count) {
  if (forward_count < 0) {
    LOG(ERROR) << "Receive " << forward_count << " forwards in updateChannelMessageForwards for " << message_full_id;
    return;
  }
  update_message_interaction_info(message_full_id, -1, forward_count, false, nullptr, false, nullptr);
}

void MessagesManager::on_update_message_reactions(MessageFullId message_full_id,
                                                  tl_object_ptr<telegram_api::messageReactions> &&reactions,
                                                  Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto new_reactions = MessageReactions::get_message_reactions(td_, std::move(reactions), td_->auth_manager_->is_bot());
  if (!have_message_force(message_full_id, "on_update_message_reactions")) {
    auto dialog_id = message_full_id.get_dialog_id();
    if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
      LOG(INFO) << "Ignore updateMessageReaction in inaccessible " << message_full_id;
      promise.set_value(Unit());
      return;
    }
    Dialog *d = get_dialog(dialog_id);
    if (d == nullptr) {
      LOG(INFO) << "Ignore updateMessageReaction in unknown " << dialog_id;
      promise.set_value(Unit());
      return;
    }

    // there is no message, so the update can be ignored
    if ((new_reactions != nullptr && !new_reactions->unread_reactions_.empty()) || d->unread_reaction_count > 0) {
      // but if there are unread reactions or the chat has unread reactions,
      // then number of unread reactions could have been changed, so reload the number of unread reactions
      repair_dialog_unread_reaction_count(d, std::move(promise), "on_update_message_reactions");
    } else {
      promise.set_value(Unit());
    }
    return;
  }

  update_message_interaction_info(message_full_id, -1, -1, false, nullptr, true, std::move(new_reactions));
  promise.set_value(Unit());
}

void MessagesManager::update_message_reactions(MessageFullId message_full_id,
                                               unique_ptr<MessageReactions> &&reactions) {
  update_message_interaction_info(message_full_id, -1, -1, false, nullptr, true, std::move(reactions));
}

void MessagesManager::on_get_message_reaction_list(
    MessageFullId message_full_id, const ReactionType &reaction_type,
    FlatHashMap<ReactionType, vector<DialogId>, ReactionTypeHash> reaction_types, int32 total_count) {
  const Message *m = get_message_force(message_full_id, "on_get_message_reaction_list");
  if (m == nullptr || m->reactions == nullptr) {
    return;
  }

  // it's impossible to use received reactions to update message reactions, because there is no way to find,
  // which reactions are chosen by the current user, so just reload message reactions for consistency
  if (m->reactions->are_consistent_with_list(reaction_type, std::move(reaction_types), total_count)) {
    return;
  }

  LOG(INFO) << "Need reload reactions in " << message_full_id << " for consistency";

  auto it = pending_reactions_.find(message_full_id);
  if (it != pending_reactions_.end()) {
    it->second.was_updated = true;
  } else {
    queue_message_reactions_reload(message_full_id);
  }
}

void MessagesManager::on_update_message_interaction_info(MessageFullId message_full_id, int32 view_count,
                                                         int32 forward_count, bool has_reply_info,
                                                         tl_object_ptr<telegram_api::messageReplies> &&reply_info) {
  if (view_count < 0 || forward_count < 0) {
    LOG(ERROR) << "Receive " << view_count << "/" << forward_count << " interaction counters for " << message_full_id;
    return;
  }
  update_message_interaction_info(message_full_id, view_count, forward_count, has_reply_info, std::move(reply_info),
                                  false, nullptr);
}

void MessagesManager::on_pending_message_views_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = pending_message_views_.find(dialog_id);
  if (it == pending_message_views_.end()) {
    return;
  }
  auto &pending_viewed_message_ids = it->second.message_ids_;
  bool increment_view_counter = it->second.increment_view_counter_;

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  const size_t MAX_MESSAGE_VIEWS = 100;  // server side limit
  vector<MessageId> message_ids;
  message_ids.reserve(min(pending_viewed_message_ids.size(), MAX_MESSAGE_VIEWS));
  for (auto message_id : pending_viewed_message_ids) {
    auto *m = get_message(d, message_id);
    if (m == nullptr) {
      continue;
    }
    if (m->has_get_message_views_query) {
      if (!increment_view_counter || m->need_view_counter_increment) {
        continue;
      }
      m->need_view_counter_increment = true;
    } else {
      m->has_get_message_views_query = true;
      m->need_view_counter_increment = increment_view_counter;
    }
    message_ids.push_back(message_id);
    if (message_ids.size() >= MAX_MESSAGE_VIEWS) {
      td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), increment_view_counter);
      message_ids.clear();
    }
  }
  if (!message_ids.empty()) {
    td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), increment_view_counter);
  }
  pending_message_views_.erase(it);
}

void MessagesManager::update_message_interaction_info(MessageFullId message_full_id, int32 view_count,
                                                      int32 forward_count, bool has_reply_info,
                                                      tl_object_ptr<telegram_api::messageReplies> &&reply_info,
                                                      bool has_reactions, unique_ptr<MessageReactions> &&reactions) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "update_message_interaction_info");
  if (d == nullptr) {
    return;
  }
  auto message_id = message_full_id.get_message_id();
  Message *m = get_message_force(d, message_id, "update_message_interaction_info");
  if (m == nullptr) {
    LOG(INFO) << "Ignore message interaction info about unknown " << message_full_id;
    if (!message_id.is_scheduled() && d->last_new_message_id.is_valid() && message_id > d->last_new_message_id &&
        dialog_id.get_type() == DialogType::Channel) {
      get_channel_difference(dialog_id, d->pts, 0, message_id, true, "update_message_interaction_info");
    }
    return;
  }

  if (view_count < 0) {
    view_count = m->view_count;
  }
  if (forward_count < 0) {
    forward_count = m->forward_count;
  }
  bool is_empty_reply_info = reply_info == nullptr;
  MessageReplyInfo new_reply_info(td_, std::move(reply_info), td_->auth_manager_->is_bot());
  if (new_reply_info.is_empty() && !is_empty_reply_info) {
    has_reply_info = false;
  }

  if (update_message_interaction_info(d, m, view_count, forward_count, has_reply_info, std::move(new_reply_info),
                                      has_reactions, std::move(reactions), "update_message_interaction_info")) {
    on_message_changed(d, m, true, "update_message_interaction_info");
  }
}

bool MessagesManager::is_thread_message(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  return is_thread_message(dialog_id, m->message_id, m->reply_info, m->content->get_type());
}

bool MessagesManager::is_thread_message(DialogId dialog_id, MessageId message_id, const MessageReplyInfo &reply_info,
                                        MessageContentType content_type) const {
  if (dialog_id.get_type() != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
    return false;
  }
  if (!message_id.is_valid() || !message_id.is_server()) {
    return false;
  }
  return !reply_info.is_empty() || reply_info.was_dropped() || content_type == MessageContentType::TopicCreate;
}

bool MessagesManager::is_active_message_reply_info(DialogId dialog_id, const MessageReplyInfo &reply_info) const {
  if (reply_info.is_empty()) {
    return false;
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    return false;
  }

  if (!reply_info.is_comment_) {
    return true;
  }
  if (!td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
    return true;
  }

  auto channel_id = dialog_id.get_channel_id();
  if (!td_->chat_manager_->get_channel_has_linked_channel(channel_id)) {
    return false;
  }

  auto linked_channel_id =
      td_->chat_manager_->get_channel_linked_channel_id(channel_id, "is_active_message_reply_info");
  if (!linked_channel_id.is_valid()) {
    // keep the comment button while linked channel is unknown
    send_closure_later(G()->chat_manager(), &ChatManager::load_channel_full, channel_id, false, Promise<Unit>(),
                       "is_active_message_reply_info");
    return true;
  }

  if (linked_channel_id != reply_info.channel_id_) {
    return false;
  }

  return true;
}

bool MessagesManager::is_visible_message_reply_info(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (!m->message_id.is_valid()) {
    return false;
  }
  bool is_broadcast = td_->dialog_manager_->is_broadcast_channel(dialog_id);
  if (!m->message_id.is_server() && !(is_broadcast && m->message_id.is_yet_unsent())) {
    return false;
  }
  if (is_broadcast && (m->had_reply_markup || m->reply_markup != nullptr)) {
    return false;
  }
  if (!is_active_message_reply_info(dialog_id, m->reply_info)) {
    return false;
  }
  if (m->reply_info.is_comment_ && is_broadcast &&
      td_->chat_manager_->have_channel_force(m->reply_info.channel_id_, "is_visible_message_reply_info") &&
      !td_->chat_manager_->have_input_peer_channel(m->reply_info.channel_id_, AccessRights::Read)) {
    // keep the comment button while have no information about the linked channel
    return false;
  }
  return true;
}

bool MessagesManager::is_visible_message_reactions(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return false;
  }

  const Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (get_message_active_reactions(d, m).empty()) {
    return false;
  }
  if (m->available_reactions_generation != d->available_reactions_generation) {
    return false;
  }
  return true;
}

bool MessagesManager::has_unread_message_reactions(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }
  CHECK(m != nullptr);
  return m->reactions != nullptr && !m->reactions->unread_reactions_.empty() &&
         is_visible_message_reactions(dialog_id, m);
}

void MessagesManager::on_message_reply_info_changed(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (is_visible_message_reply_info(dialog_id, m)) {
    send_update_message_interaction_info(dialog_id, m);
  }
}

td_api::object_ptr<td_api::messageInteractionInfo> MessagesManager::get_message_interaction_info_object(
    DialogId dialog_id, const Message *m) const {
  bool is_visible_reply_info = is_visible_message_reply_info(dialog_id, m);
  bool has_reactions =
      m->reactions != nullptr && !m->reactions->reactions_.empty() && is_visible_message_reactions(dialog_id, m);
  if (m->view_count == 0 && m->forward_count == 0 && !is_visible_reply_info && !has_reactions) {
    return nullptr;
  }
  if (m->message_id.is_scheduled() &&
      (m->forward_info == nullptr || td_->dialog_manager_->is_broadcast_channel(dialog_id))) {
    return nullptr;
  }
  if (m->message_id.is_local() && m->forward_info == nullptr) {
    return nullptr;
  }

  td_api::object_ptr<td_api::messageReplyInfo> reply_info;
  if (is_visible_reply_info) {
    auto expected_dialog_id = m->reply_info.is_comment_ ? DialogId(m->reply_info.channel_id_) : dialog_id;
    const Dialog *d = get_dialog(expected_dialog_id);
    reply_info =
        m->reply_info.get_message_reply_info_object(td_, d != nullptr ? d->last_read_inbox_message_id : MessageId());
    CHECK(reply_info != nullptr);
  }

  td_api::object_ptr<td_api::messageReactions> reactions;
  if (has_reactions) {
    UserId my_user_id;
    UserId peer_user_id;
    if (dialog_id.get_type() == DialogType::User) {
      my_user_id = td_->user_manager_->get_my_id();
      peer_user_id = dialog_id.get_user_id();
    }
    reactions = m->reactions->get_message_reactions_object(td_, my_user_id, peer_user_id);
  }

  return td_api::make_object<td_api::messageInteractionInfo>(m->view_count, m->forward_count, std::move(reply_info),
                                                             std::move(reactions));
}

vector<td_api::object_ptr<td_api::unreadReaction>> MessagesManager::get_unread_reactions_object(
    DialogId dialog_id, const Message *m) const {
  if (!has_unread_message_reactions(dialog_id, m)) {
    return {};
  }

  vector<td_api::object_ptr<td_api::unreadReaction>> unread_reactions;
  for (const auto &unread_reaction : m->reactions->unread_reactions_) {
    auto unread_reaction_object = unread_reaction.get_unread_reaction_object(td_);
    if (unread_reaction_object != nullptr) {
      unread_reactions.push_back(std::move(unread_reaction_object));
    }
  }
  return unread_reactions;
}

bool MessagesManager::update_message_interaction_info(Dialog *d, Message *m, int32 view_count, int32 forward_count,
                                                      bool has_reply_info, MessageReplyInfo &&reply_info,
                                                      bool has_reactions, unique_ptr<MessageReactions> &&reactions,
                                                      const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  CHECK(d != nullptr);
  CHECK(m != nullptr);
  auto dialog_id = d->dialog_id;
  m->interaction_info_update_date = G()->unix_time();  // doesn't force message saving
  if (m->message_id.is_valid_scheduled()) {
    has_reply_info = false;
    has_reactions = false;
  }
  bool need_update_reply_info = has_reply_info && m->reply_info.need_update_to(reply_info);
  if (has_reply_info && m->reply_info.channel_id_ == reply_info.channel_id_) {
    if (need_update_reply_info) {
      reply_info.update_max_message_ids(m->reply_info);
    } else {
      if (m->reply_info.update_max_message_ids(reply_info) && view_count <= m->view_count &&
          forward_count <= m->forward_count) {
        on_message_reply_info_changed(dialog_id, m);
        on_message_changed(d, m, true, "update_message_interaction_info");
      }
    }
  }
  MessageFullId message_full_id{dialog_id, m->message_id};
  if (has_reactions) {
    auto it = pending_reactions_.find(message_full_id);
    if (it != pending_reactions_.end()) {
      LOG(INFO) << "Ignore reactions for " << message_full_id << ", because they are being changed";
      has_reactions = false;
      it->second.was_updated = true;
    }
    if (has_reactions && pending_read_reactions_.count(message_full_id) > 0) {
      LOG(INFO) << "Ignore reactions for " << message_full_id << ", because they are being read";
      has_reactions = false;
    }
  }
  if (has_reactions && reactions != nullptr) {
    if (m->reactions != nullptr) {
      reactions->update_from(*m->reactions);
    }
    reactions->sort_reactions(active_reaction_pos_);
    reactions->fix_chosen_reaction();
    reactions->fix_my_recent_chooser_dialog_id(td_->dialog_manager_->get_my_dialog_id());
  }
  bool need_update_reactions =
      has_reactions && MessageReactions::need_update_message_reactions(m->reactions.get(), reactions.get());
  bool need_update_unread_reactions =
      has_reactions && MessageReactions::need_update_unread_reactions(m->reactions.get(), reactions.get());
  bool need_update_chosen_reaction_order = has_reactions && reactions != nullptr && m->reactions != nullptr &&
                                           m->reactions->chosen_reaction_order_ != reactions->chosen_reaction_order_;
  if (view_count > m->view_count || forward_count > m->forward_count || need_update_reply_info ||
      need_update_reactions || need_update_unread_reactions || need_update_chosen_reaction_order) {
    LOG(DEBUG) << "Update interaction info of " << message_full_id << " from " << m->view_count << '/'
               << m->forward_count << '/' << m->reply_info << '/' << m->reactions << " to " << view_count << '/'
               << forward_count << '/' << reply_info << '/' << reactions
               << ", need_update_reply_info = " << need_update_reply_info
               << ", need_update_reactions = " << need_update_reactions
               << ", need_update_unread_reactions = " << need_update_unread_reactions
               << ", need_update_chosen_reaction_order = " << need_update_chosen_reaction_order;
    bool need_update = false;
    if (view_count > m->view_count) {
      m->view_count = view_count;
      need_update = true;
    }
    if (forward_count > m->forward_count) {
      m->forward_count = forward_count;
      need_update = true;
    }
    if (need_update_reply_info) {
      if (m->reply_info.channel_id_ != reply_info.channel_id_) {
        if (m->reply_info.channel_id_.is_valid() && reply_info.channel_id_.is_valid() && m->message_id.is_server()) {
          LOG(ERROR) << "Reply info of " << message_full_id << " changed from " << m->reply_info << " to " << reply_info
                     << " from " << source;
        }
      }
      m->reply_info = std::move(reply_info);
      if (!m->top_thread_message_id.is_valid() && is_thread_message(dialog_id, m)) {
        m->top_thread_message_id = m->message_id;
      }
      need_update |= is_visible_message_reply_info(dialog_id, m);
    }
    int32 new_dialog_unread_reaction_count = -1;
    if (need_update_reactions || need_update_unread_reactions) {
      CHECK(m->message_id.is_valid());

      auto old_chosen_tags = get_chosen_tags(m->reactions);
      int32 unread_reaction_diff = 0;
      unread_reaction_diff -= (has_unread_message_reactions(dialog_id, m) ? 1 : 0);
      m->reactions = std::move(reactions);
      m->available_reactions_generation = d->available_reactions_generation;
      unread_reaction_diff += (has_unread_message_reactions(dialog_id, m) ? 1 : 0);
      auto new_chosen_tags = get_chosen_tags(m->reactions);

      td_->reaction_manager_->update_saved_messages_tags(m->saved_messages_topic_id, old_chosen_tags, new_chosen_tags);

      if (is_visible_message_reactions(dialog_id, m)) {
        need_update |= need_update_reactions;
        if (need_update_unread_reactions) {
          if (unread_reaction_diff != 0) {
            // remove_message_notification_id(d, m, true, true);

            if (d->unread_reaction_count + unread_reaction_diff < 0) {
              if (is_dialog_inited(d)) {
                LOG(ERROR) << "Unread reaction count of " << dialog_id << " became negative from " << source;
              }
            } else {
              set_dialog_unread_reaction_count(d, d->unread_reaction_count + unread_reaction_diff);
              on_dialog_updated(dialog_id, "update_message_interaction_info");
            }
          }

          if (unread_reaction_diff < 0) {
            send_update_message_unread_reactions(dialog_id, m, d->unread_reaction_count);
          } else {
            new_dialog_unread_reaction_count = d->unread_reaction_count;
          }
        }
      }
    } else if (has_reactions) {
      bool is_changed = false;
      if (m->available_reactions_generation != d->available_reactions_generation) {
        m->available_reactions_generation = d->available_reactions_generation;
        is_changed = true;
      }
      if (need_update_chosen_reaction_order) {
        m->reactions->chosen_reaction_order_ = std::move(reactions->chosen_reaction_order_);
        is_changed = true;
      }
      if (is_changed) {
        on_message_changed(d, m, false, "update_message_interaction_info");
      }
    }
    if (need_update) {
      send_update_message_interaction_info(dialog_id, m);
    }
    if (new_dialog_unread_reaction_count >= 0) {
      send_update_message_unread_reactions(dialog_id, m, new_dialog_unread_reaction_count);
    }
    return true;
  } else if (has_reactions && m->available_reactions_generation != d->available_reactions_generation) {
    m->available_reactions_generation = d->available_reactions_generation;
    on_message_changed(d, m, false, "update_message_interaction_info");
  }
  return false;
}

void MessagesManager::on_update_live_location_viewed(MessageFullId message_full_id) {
  LOG(DEBUG) << "Live location was viewed in " << message_full_id;
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(PromiseCreator::lambda([actor_id = actor_id(this), message_full_id](Unit result) {
      send_closure(actor_id, &MessagesManager::on_update_live_location_viewed, message_full_id);
    }));
    return;
  }

  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  if (!td::contains(active_live_location_message_ids, message_full_id)) {
    LOG(DEBUG) << "Can't find " << message_full_id << " in " << active_live_location_message_ids;
    return;
  }

  send_update_message_live_location_viewed(message_full_id);
}

void MessagesManager::on_update_some_live_location_viewed(Promise<Unit> &&promise) {
  LOG(DEBUG) << "Some live location was viewed";
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(
        PromiseCreator::lambda([actor_id = actor_id(this), promise = std::move(promise)](Unit result) mutable {
          send_closure(actor_id, &MessagesManager::on_update_some_live_location_viewed, std::move(promise));
        }));
    return;
  }

  // update all live locations, because it is unknown, which exactly was viewed
  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  for (const auto &message_full_id : active_live_location_message_ids) {
    send_update_message_live_location_viewed(message_full_id);
  }

  promise.set_value(Unit());
}

void MessagesManager::on_update_message_extended_media(
    MessageFullId message_full_id, telegram_api::object_ptr<telegram_api::MessageExtendedMedia> extended_media) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "on_update_message_extended_media");
  if (d == nullptr) {
    LOG(INFO) << "Ignore update of message extended media in unknown " << dialog_id;
    return;
  }

  auto m = get_message_force(d, message_full_id.get_message_id(), "on_update_message_extended_media");
  if (m == nullptr) {
    LOG(INFO) << "Ignore update of message extended media in unknown " << message_full_id;
    return;
  }

  auto content = m->content.get();
  auto content_type = content->get_type();
  if (content_type != MessageContentType::Invoice) {
    if (content_type != MessageContentType::Unsupported) {
      LOG(ERROR) << "Receive updateMessageExtendedMedia for " << message_full_id << " of type " << content_type;
    }
    return;
  }
  if (update_message_content_extended_media(content, std::move(extended_media), dialog_id, td_)) {
    send_update_message_content(d, m, true, "on_update_message_extended_media");
    on_message_changed(d, m, true, "on_update_message_extended_media");
    on_message_notification_changed(d, m, "on_update_message_extended_media");  // usually a no-op
  }
}

bool MessagesManager::need_skip_bot_commands(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  if (m != nullptr && m->message_id.is_scheduled()) {
    return true;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return (d->is_has_bots_inited && !d->has_bots) || td_->dialog_manager_->is_broadcast_channel(dialog_id);
}

void MessagesManager::on_external_update_message_content(MessageFullId message_full_id) {
  Dialog *d = get_dialog(message_full_id.get_dialog_id());
  CHECK(d != nullptr);
  Message *m = get_message(d, message_full_id.get_message_id());
  CHECK(m != nullptr);
  send_update_message_content(d, m, true, "on_external_update_message_content");
  // must not call on_message_changed, because the message itself wasn't changed
  send_update_last_message_if_needed(d, m, "on_external_update_message_content");
  on_message_notification_changed(d, m, "on_external_update_message_content");
}

void MessagesManager::on_update_message_content(MessageFullId message_full_id) {
  Dialog *d = get_dialog(message_full_id.get_dialog_id());
  CHECK(d != nullptr);
  Message *m = get_message(d, message_full_id.get_message_id());
  CHECK(m != nullptr);
  send_update_message_content(d, m, true, "on_update_message_content");
  on_message_changed(d, m, true, "on_update_message_content");
  on_message_notification_changed(d, m, "on_update_message_content");
}

bool MessagesManager::update_message_contains_unread_mention(Dialog *d, Message *m, bool contains_unread_mention,
                                                             const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  if (!contains_unread_mention && m->contains_unread_mention) {
    remove_message_notification_id(d, m, true, true);  // must be called before contains_unread_mention is updated

    m->contains_unread_mention = false;
    if (d->unread_mention_count == 0) {
      if (is_dialog_inited(d)) {
        LOG(ERROR) << "Unread mention count of " << d->dialog_id << " became negative from " << source;
      }
    } else {
      set_dialog_unread_mention_count(d, d->unread_mention_count - 1);
      on_dialog_updated(d->dialog_id, "update_message_contains_unread_mention");
    }
    LOG(INFO) << "Update unread mention message count in " << d->dialog_id << " to " << d->unread_mention_count
              << " by reading " << m->message_id << " from " << source;

    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateMessageMentionRead>(
                     get_chat_id_object(d->dialog_id, "updateMessageMentionRead"), m->message_id.get(),
                     d->unread_mention_count));
    return true;
  }
  return false;
}

bool MessagesManager::remove_message_unread_reactions(Dialog *d, Message *m, const char *source) {
  CHECK(m != nullptr);
  CHECK(!m->message_id.is_scheduled());
  if (!has_unread_message_reactions(d->dialog_id, m)) {
    return false;
  }
  // remove_message_notification_id(d, m, true, true);

  m->reactions->unread_reactions_.clear();
  if (d->unread_reaction_count == 0) {
    if (is_dialog_inited(d)) {
      LOG(ERROR) << "Unread reaction count of " << d->dialog_id << " became negative from " << source;
    }
  } else {
    set_dialog_unread_reaction_count(d, d->unread_reaction_count - 1);
    on_dialog_updated(d->dialog_id, "remove_message_unread_reactions");
  }
  LOG(INFO) << "Update unread reaction count in " << d->dialog_id << " to " << d->unread_reaction_count
            << " by reading " << m->message_id << " from " << source;

  send_update_message_unread_reactions(d->dialog_id, m, d->unread_reaction_count);

  return true;
}

void MessagesManager::on_read_channel_inbox(ChannelId channel_id, MessageId max_message_id, int32 server_unread_count,
                                            int32 pts, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!max_message_id.is_scheduled());
  if (!max_message_id.is_valid() && server_unread_count <= 0) {
    return;
  }

  DialogId dialog_id(channel_id);
  Dialog *d = get_dialog_force(dialog_id, source);
  if (d == nullptr) {
    LOG(INFO) << "Receive read inbox in unknown " << dialog_id << " from " << source;
    return;
  }

  /*
  // dropping unread count can make things worse, so don't drop it
  if (server_unread_count > 0 && G()->use_message_database() && d->is_last_read_inbox_message_id_inited) {
    server_unread_count = -1;
  }
  */

  if (d->pts == pts) {
    read_history_inbox(d, max_message_id, server_unread_count, source);
  } else if (d->pts > pts) {
    // outdated update, need to repair server_unread_count from the server
    repair_channel_server_unread_count(d);
  } else {
    // update from the future, keep it until it can be applied
    if (pts >= d->pending_read_channel_inbox_pts) {
      if (d->pending_read_channel_inbox_pts == 0) {
        schedule_get_channel_difference(dialog_id, pts, MessageId(), 0.001, "on_read_channel_inbox");
      }
      d->pending_read_channel_inbox_pts = pts;
      d->pending_read_channel_inbox_max_message_id = max_message_id;
      d->pending_read_channel_inbox_server_unread_count = server_unread_count;
      on_dialog_updated(dialog_id, "on_read_channel_inbox");
    }
  }
}

void MessagesManager::on_read_channel_outbox(ChannelId channel_id, MessageId max_message_id) {
  DialogId dialog_id(channel_id);
  CHECK(!max_message_id.is_scheduled());
  if (max_message_id.is_valid()) {
    read_history_outbox(dialog_id, max_message_id);
  }
}

void MessagesManager::on_update_channel_max_unavailable_message_id(ChannelId channel_id,
                                                                   MessageId max_unavailable_message_id,
                                                                   const char *source) {
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive max_unavailable_message_id in invalid " << channel_id << " from " << source;
    return;
  }

  DialogId dialog_id(channel_id);
  CHECK(!max_unavailable_message_id.is_scheduled());
  if (!max_unavailable_message_id.is_valid() && max_unavailable_message_id != MessageId()) {
    LOG(ERROR) << "Receive wrong max_unavailable_message_id: " << max_unavailable_message_id << " from " << source;
    max_unavailable_message_id = MessageId();
  }
  set_dialog_max_unavailable_message_id(dialog_id, max_unavailable_message_id, true, source);
}

void MessagesManager::on_update_delete_scheduled_messages(DialogId dialog_id,
                                                          vector<ScheduledServerMessageId> &&server_message_ids) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive deleted scheduled messages in invalid " << dialog_id;
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "on_update_delete_scheduled_messages");
  if (d == nullptr) {
    LOG(INFO) << "Skip updateDeleteScheduledMessages in unknown " << dialog_id;
    return;
  }

  vector<int64> deleted_message_ids;
  for (auto server_message_id : server_message_ids) {
    if (!server_message_id.is_valid()) {
      LOG(ERROR) << "Incoming update tries to delete scheduled message " << server_message_id.get();
      continue;
    }

    auto message = do_delete_scheduled_message(d, MessageId(server_message_id, std::numeric_limits<int32>::max()), true,
                                               "on_update_delete_scheduled_messages");
    if (message != nullptr) {
      deleted_message_ids.push_back(message->message_id.get());
    }
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true);

  send_update_chat_has_scheduled_messages(d, true);
}

void MessagesManager::on_update_created_public_broadcasts(vector<ChannelId> channel_ids) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (created_public_broadcasts_inited_ && created_public_broadcasts_ == channel_ids) {
    return;
  }

  LOG(INFO) << "Update create public channels to " << channel_ids;
  for (auto channel_id : channel_ids) {
    force_create_dialog(DialogId(channel_id), "on_update_created_public_broadcasts");
  }

  created_public_broadcasts_inited_ = true;
  created_public_broadcasts_ = std::move(channel_ids);
}

void MessagesManager::on_dialog_speaking_action(DialogId dialog_id, DialogId speaking_dialog_id, int32 date) {
  const Dialog *d = get_dialog_force(dialog_id, "on_dialog_speaking_action");
  if (d != nullptr && d->active_group_call_id.is_valid()) {
    auto group_call_id = td_->group_call_manager_->get_group_call_id(d->active_group_call_id, dialog_id);
    td_->group_call_manager_->on_user_speaking_in_group_call(group_call_id, speaking_dialog_id, false, date);
  }
}

void MessagesManager::on_message_animated_emoji_clicked(MessageFullId message_full_id, string &&emoji, string &&data) {
  const auto *m = get_message_force(message_full_id, "on_message_animated_emoji_clicked");
  if (m != nullptr) {
    on_message_content_animated_emoji_clicked(m->content.get(), message_full_id, td_, std::move(emoji),
                                              std::move(data));
  }
}

void MessagesManager::cancel_dialog_action(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || m->forward_info != nullptr || m->had_forward_info ||
      m->via_bot_user_id.is_valid() || m->hide_via_bot || m->is_channel_post || m->message_id.is_scheduled()) {
    return;
  }

  td_->dialog_action_manager_->on_dialog_action(dialog_id, MessageId() /*ignored*/, get_message_sender(m),
                                                DialogAction(), m->date, m->content->get_type());
}

void MessagesManager::add_postponed_channel_update(DialogId dialog_id, tl_object_ptr<telegram_api::Update> &&update,
                                                   int32 new_pts, int32 pts_count, Promise<Unit> &&promise) {
  postponed_channel_updates_[dialog_id].emplace(
      new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count, std::move(promise)));
}

void MessagesManager::add_pending_channel_update(DialogId dialog_id, tl_object_ptr<telegram_api::Update> &&update,
                                                 int32 new_pts, int32 pts_count, Promise<Unit> &&promise,
                                                 const char *source, bool is_postponed_update) {
  LOG(INFO) << "Receive from " << source << " pending " << to_string(update);
  CHECK(update != nullptr);
  if (dialog_id.get_type() != DialogType::Channel) {
    LOG(ERROR) << "Receive channel update in invalid " << dialog_id << " from " << source << ": "
               << oneline(to_string(update));
    promise.set_value(Unit());
    return;
  }
  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive channel update from " << source << " with wrong PTS = " << new_pts
               << " or pts_count = " << pts_count << ": " << oneline(to_string(update));
    promise.set_value(Unit());
    return;
  }

  auto channel_id = dialog_id.get_channel_id();
  if (!td_->chat_manager_->have_channel(channel_id) && td_->chat_manager_->have_min_channel(channel_id)) {
    td_->updates_manager_->schedule_get_difference("add_pending_channel_update 1");
    promise.set_value(Unit());
    return;
  }

  // TODO need to save all updates that can change result of running queries not associated with PTS (for example
  // getHistory) and apply them to result of these queries

  Dialog *d = get_dialog_force(dialog_id, "add_pending_channel_update 2");
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      if (!td_->chat_manager_->have_channel(channel_id)) {
        // do not create dialog if there is no info about the channel
        LOG(INFO) << "There is no info about " << channel_id << ", so ignore " << oneline(to_string(update));
        promise.set_value(Unit());
        return;
      }

      d = add_dialog(dialog_id, "add_pending_channel_update 4");
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, "add_pending_channel_update 5");
    }
  }
  if (d == nullptr) {
    // if there is no dialog, it can be created by the update
    LOG(INFO) << "Receive pending update from " << source << " about unknown " << dialog_id;
    if (running_get_channel_difference(dialog_id)) {
      add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
      return;
    }
  } else {
    int32 old_pts = d->pts;
    if (new_pts <= old_pts) {  // very old or unuseful update
      if (update->get_id() == telegram_api::updateNewChannelMessage::ID) {
        auto update_new_channel_message = static_cast<telegram_api::updateNewChannelMessage *>(update.get());
        auto message_id = MessageId::get_message_id(update_new_channel_message->message_, false);
        MessageFullId message_full_id(dialog_id, message_id);
        if (update_message_ids_.count(message_full_id) > 0) {
          // apply sent channel message
          auto added_message_full_id = on_get_message(std::move(update_new_channel_message->message_), true, true,
                                                      false, "updateNewChannelMessage with an awaited message");
          if (added_message_full_id != message_full_id) {
            LOG(ERROR) << "Failed to add an awaited " << message_full_id << " from " << source;
          }
          promise.set_value(Unit());
          return;
        }
      }
      if (update->get_id() == updateSentMessage::ID) {
        auto update_sent_message = static_cast<updateSentMessage *>(update.get());
        if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
          // apply sent channel message
          on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                                  update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                                  "process old updateSentChannelMessage");
          promise.set_value(Unit());
          return;
        }
      }

      LOG_IF(WARNING, new_pts == old_pts && pts_count == 0 && !is_allowed_useless_update(update))
          << "Receive from " << source << " useless channel update " << oneline(to_string(update));
      LOG(INFO) << "Skip already applied channel update";
      promise.set_value(Unit());
      return;
    }

    if (running_get_channel_difference(dialog_id)) {
      LOG(INFO) << "Postpone channel update, because getChannelDifference is run";
      add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
      return;
    }

    if (old_pts == 0) {
      old_pts = new_pts - pts_count;
      LOG(INFO) << "Receive first update in " << dialog_id << " with PTS = " << new_pts << " from " << source;
    } else if (old_pts != new_pts - pts_count) {
      LOG(INFO) << "Found a gap in the " << dialog_id << " with PTS = " << old_pts << ". new_pts = " << new_pts
                << ", pts_count = " << pts_count << " in update from " << source;
      if (d->was_opened || td_->chat_manager_->get_channel_status(channel_id).is_member() || is_dialog_sponsored(d)) {
        add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
        get_channel_difference(dialog_id, old_pts, new_pts, MessageId(), true,
                               "add_pending_channel_update PTS mismatch");
      } else {
        promise.set_value(Unit());
      }
      return;
    }
    CHECK(old_pts + pts_count == new_pts);  // the update can be applied
  }

  if (d == nullptr || pts_count > 0) {
    if (!process_channel_update(std::move(update)) &&
        channel_get_difference_retry_timeout_.has_timeout(dialog_id.get())) {
      promise.set_value(Unit());
      return;
    }
    LOG_CHECK(!running_get_channel_difference(dialog_id)) << '"' << active_get_channel_differences_[dialog_id] << '"';
  } else {
    LOG_IF(INFO, update->get_id() != dummyUpdate::ID)
        << "Skip useless channel update from " << source << ": " << to_string(update);
  }

  if (d == nullptr) {
    d = get_dialog(dialog_id);
    if (d == nullptr) {
      LOG(INFO) << "Update didn't created " << dialog_id;
      promise.set_value(Unit());
      return;
    }
  }

  CHECK(new_pts > d->pts);
  set_channel_pts(d, new_pts, source);
  promise.set_value(Unit());
}

bool MessagesManager::is_old_channel_update(DialogId dialog_id, int32 new_pts) {
  CHECK(dialog_id.get_type() == DialogType::Channel);

  const Dialog *d = get_dialog_force(dialog_id, "is_old_channel_update");
  return new_pts <= (d == nullptr ? load_channel_pts(dialog_id) : d->pts);
}

void MessagesManager::process_pts_update(tl_object_ptr<telegram_api::Update> &&update_ptr) {
  switch (update_ptr->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case telegram_api::updateNewMessage::ID: {
      auto update = move_tl_object_as<telegram_api::updateNewMessage>(update_ptr);
      LOG(INFO) << "Process updateNewMessage";
      on_get_message(std::move(update->message_), true, false, false, "updateNewMessage");
      break;
    }
    case updateSentMessage::ID: {
      auto update = move_tl_object_as<updateSentMessage>(update_ptr);
      LOG(INFO) << "Process updateSentMessage " << update->random_id_;
      on_send_message_success(update->random_id_, update->message_id_, update->date_, update->ttl_period_, FileId(),
                              "process updateSentMessage");
      break;
    }
    case telegram_api::updateReadMessagesContents::ID: {
      auto update = move_tl_object_as<telegram_api::updateReadMessagesContents>(update_ptr);
      LOG(INFO) << "Process updateReadMessageContents";
      for (auto &message_id : update->messages_) {
        read_message_content_from_updates(MessageId(ServerMessageId(message_id)), update->date_);
      }
      break;
    }
    case telegram_api::updateEditMessage::ID: {
      auto update = move_tl_object_as<telegram_api::updateEditMessage>(update_ptr);
      LOG(INFO) << "Process updateEditMessage";
      bool had_message =
          have_message_force(MessageFullId::get_message_full_id(update->message_, false), "updateEditMessage");
      auto message_full_id = on_get_message(std::move(update->message_), false, false, false, "updateEditMessage");
      on_message_edited(message_full_id, update->pts_, had_message);
      break;
    }
    case telegram_api::updateDeleteMessages::ID: {
      auto update = move_tl_object_as<telegram_api::updateDeleteMessages>(update_ptr);
      LOG(INFO) << "Process updateDeleteMessages";
      vector<MessageId> message_ids;
      for (auto message : update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }
      delete_messages_from_updates(message_ids, true);
      break;
    }
    case telegram_api::updateReadHistoryInbox::ID: {
      auto update = move_tl_object_as<telegram_api::updateReadHistoryInbox>(update_ptr);
      LOG(INFO) << "Process updateReadHistoryInbox";
      DialogId dialog_id(update->peer_);
      on_update_dialog_folder_id(dialog_id, FolderId(update->folder_id_));
      read_history_inbox(dialog_id, MessageId(ServerMessageId(update->max_id_)), -1 /*update->still_unread_count*/,
                         "updateReadHistoryInbox");
      break;
    }
    case telegram_api::updateReadHistoryOutbox::ID: {
      auto update = move_tl_object_as<telegram_api::updateReadHistoryOutbox>(update_ptr);
      LOG(INFO) << "Process updateReadHistoryOutbox";
      read_history_outbox(DialogId(update->peer_), MessageId(ServerMessageId(update->max_id_)));
      break;
    }
    case telegram_api::updatePinnedMessages::ID: {
      auto update = move_tl_object_as<telegram_api::updatePinnedMessages>(update_ptr);
      LOG(INFO) << "Process updatePinnedMessages";
      vector<MessageId> message_ids;
      for (auto message : update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }
      update_dialog_pinned_messages_from_updates(DialogId(update->peer_), message_ids, update->pinned_);
      break;
    }
    default:
      UNREACHABLE();
  }
  update_ptr = nullptr;
  CHECK(!td_->updates_manager_->running_get_difference());
}

bool MessagesManager::process_channel_update(tl_object_ptr<telegram_api::Update> &&update_ptr) {
  switch (update_ptr->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case updateSentMessage::ID: {
      auto update = move_tl_object_as<updateSentMessage>(update_ptr);
      LOG(INFO) << "Process updateSentMessage " << update->random_id_;
      on_send_message_success(update->random_id_, update->message_id_, update->date_, update->ttl_period_, FileId(),
                              "process updateSentChannelMessage");
      break;
    }
    case telegram_api::updateNewChannelMessage::ID: {
      auto update = move_tl_object_as<telegram_api::updateNewChannelMessage>(update_ptr);
      LOG(INFO) << "Process updateNewChannelMessage";
      on_get_message(std::move(update->message_), true, true, false, "updateNewChannelMessage");
      break;
    }
    case telegram_api::updateDeleteChannelMessages::ID: {
      auto update = move_tl_object_as<telegram_api::updateDeleteChannelMessages>(update_ptr);
      LOG(INFO) << "Process updateDeleteChannelMessages";
      ChannelId channel_id(update->channel_id_);
      if (!channel_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << channel_id;
        break;
      }

      vector<MessageId> message_ids;
      for (auto &message : update->messages_) {
        auto message_id = MessageId(ServerMessageId(message));
        if (message_id.is_valid()) {
          message_ids.push_back(message_id);
        } else {
          LOG(ERROR) << "Receive updateDeleteChannelMessages with message " << message << " in " << channel_id;
        }
      }

      delete_dialog_messages(DialogId(channel_id), message_ids, true, "updateDeleteChannelMessages");
      break;
    }
    case telegram_api::updateEditChannelMessage::ID: {
      auto update = move_tl_object_as<telegram_api::updateEditChannelMessage>(update_ptr);
      LOG(INFO) << "Process updateEditChannelMessage";
      bool had_message =
          have_message_force(MessageFullId::get_message_full_id(update->message_, false), "updateEditChannelMessage");
      auto message_full_id =
          on_get_message(std::move(update->message_), false, true, false, "updateEditChannelMessage");
      if (message_full_id == MessageFullId()) {
        return false;
      }
      on_message_edited(message_full_id, update->pts_, had_message);
      break;
    }
    case telegram_api::updatePinnedChannelMessages::ID: {
      auto update = move_tl_object_as<telegram_api::updatePinnedChannelMessages>(update_ptr);
      LOG(INFO) << "Process updatePinnedChannelMessages";
      ChannelId channel_id(update->channel_id_);
      if (!channel_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << channel_id;
        break;
      }

      vector<MessageId> message_ids;
      for (auto &message : update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }

      update_dialog_pinned_messages_from_updates(DialogId(channel_id), message_ids, update->pinned_);
      break;
    }
    default:
      UNREACHABLE();
  }
  return true;
}

void MessagesManager::on_message_edited(MessageFullId message_full_id, int32 pts, bool had_message) {
  if (message_full_id == MessageFullId()) {
    return;
  }

  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  Message *m = get_message(d, message_full_id.get_message_id());
  CHECK(m != nullptr);
  m->last_edit_pts = pts;
  d->last_edited_message_id = m->message_id;
  if (td_->auth_manager_->is_bot()) {
    send_update_message_edited(dialog_id, m);
  }
  update_used_hashtags(dialog_id, m);

  if (!had_message &&
      ((m->reactions != nullptr && !m->reactions->unread_reactions_.empty()) || d->unread_reaction_count > 0)) {
    // if new message with unread reactions was added or the chat has unread reactions,
    // then number of unread reactions could have been changed, so reload the number of unread reactions
    repair_dialog_unread_reaction_count(d, Promise<Unit>(), "on_message_edited");
  }
}

bool MessagesManager::update_dialog_notification_settings(DialogId dialog_id,
                                                          DialogNotificationSettings *current_settings,
                                                          DialogNotificationSettings &&new_settings) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return false;
  }

  auto need_update = need_update_dialog_notification_settings(current_settings, new_settings);
  if (need_update.are_changed) {
    Dialog *d = get_dialog(dialog_id);
    LOG_CHECK(d != nullptr) << "Wrong " << dialog_id << " in update_dialog_notification_settings";
    bool was_dialog_mentions_disabled = is_dialog_mention_notifications_disabled(d);

    VLOG(notifications) << "Update notification settings in " << dialog_id << " from " << *current_settings << " to "
                        << new_settings;

    update_dialog_unmute_timeout(d, current_settings->use_default_mute_until, current_settings->mute_until,
                                 new_settings.use_default_mute_until, new_settings.mute_until);

    *current_settings = std::move(new_settings);
    on_dialog_updated(dialog_id, "update_dialog_notification_settings");

    if (is_dialog_muted(d)) {
      // no check for was_muted to clean pending message notifications in chats with unsynchronized settings
      remove_all_dialog_notifications(d, false, "update_dialog_notification_settings 2");
    }
    if (is_dialog_pinned_message_notifications_disabled(d) && d->notification_info != nullptr &&
        d->notification_info->mention_notification_group_.is_valid() &&
        d->notification_info->pinned_message_notification_message_id_.is_valid()) {
      remove_dialog_pinned_message_notification(d, "update_dialog_notification_settings 3");
    }
    if (was_dialog_mentions_disabled != is_dialog_mention_notifications_disabled(d)) {
      if (was_dialog_mentions_disabled) {
        update_dialog_mention_notification_count(d);
      } else {
        remove_dialog_mention_notifications(d);
      }
    }

    if (need_update.need_update_server || need_update.need_update_local) {
      send_closure(G()->td(), &Td::send_update,
                   td_api::make_object<td_api::updateChatNotificationSettings>(
                       get_chat_id_object(dialog_id, "updateChatNotificationSettings"),
                       get_chat_notification_settings_object(current_settings)));
    }
  }
  return need_update.need_update_server;
}

void MessagesManager::schedule_dialog_unmute(DialogId dialog_id, bool use_default, int32 mute_until, int32 unix_time) {
  if (!use_default && mute_until >= unix_time && mute_until < unix_time + 366 * 86400) {
    dialog_unmute_timeout_.set_timeout_in(dialog_id.get(), mute_until - unix_time + 1);
  } else {
    dialog_unmute_timeout_.cancel_timeout(dialog_id.get());
  }
}

void MessagesManager::update_dialog_unmute_timeout(Dialog *d, bool &old_use_default, int32 &old_mute_until,
                                                   bool new_use_default, int32 new_mute_until) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (old_use_default == new_use_default && old_mute_until == new_mute_until) {
    return;
  }
  CHECK(d != nullptr);
  CHECK(old_mute_until >= 0);

  schedule_dialog_unmute(d->dialog_id, new_use_default, new_mute_until, G()->unix_time());

  auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id);
  auto scope_mute_until = td_->notification_settings_manager_->get_scope_mute_until(scope);
  bool was_muted = (old_use_default ? scope_mute_until : old_mute_until) != 0;
  bool is_muted = (new_use_default ? scope_mute_until : new_mute_until) != 0;
  if (was_muted != is_muted && need_unread_counter(d->order)) {
    auto unread_count = d->server_unread_count + d->local_unread_count;
    if (unread_count != 0 || d->is_marked_as_unread) {
      for (auto &list : get_dialog_lists(d)) {
        if (unread_count != 0 && list.is_message_unread_count_inited_) {
          int32 delta = was_muted ? -unread_count : unread_count;
          list.unread_message_muted_count_ += delta;
          send_update_unread_message_count(list, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
        if (list.is_dialog_unread_count_inited_) {
          int32 delta = was_muted ? -1 : 1;
          list.unread_dialog_muted_count_ += delta;
          if (unread_count == 0 && d->is_marked_as_unread) {
            list.unread_dialog_muted_marked_count_ += delta;
          }
          send_update_unread_chat_count(list, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
      }
    }
  }

  old_use_default = new_use_default;
  old_mute_until = new_mute_until;

  if (was_muted != is_muted && td_->dialog_filter_manager_->have_dialog_filters()) {
    update_dialog_lists(d, get_dialog_positions(d), true, false, "update_dialog_unmute_timeout");
  }
}

void MessagesManager::on_update_notification_scope_is_muted(NotificationSettingsScope scope, bool is_muted) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }
  if (G()->use_message_database()) {
    std::unordered_map<DialogListId, int32, DialogListIdHash> delta;
    std::unordered_map<DialogListId, int32, DialogListIdHash> total_count;
    std::unordered_map<DialogListId, int32, DialogListIdHash> marked_count;
    std::unordered_set<DialogListId, DialogListIdHash> dialog_list_ids;
    dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
      Dialog *d = dialog.get();
      if (need_unread_counter(d->order) && d->notification_settings.use_default_mute_until &&
          td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id) == scope) {
        int32 unread_count = d->server_unread_count + d->local_unread_count;
        if (unread_count != 0) {
          for (auto dialog_list_id : get_dialog_list_ids(d)) {
            delta[dialog_list_id] += unread_count;
            total_count[dialog_list_id]++;
            dialog_list_ids.insert(dialog_list_id);
          }
        } else if (d->is_marked_as_unread) {
          for (auto dialog_list_id : get_dialog_list_ids(d)) {
            total_count[dialog_list_id]++;
            marked_count[dialog_list_id]++;
            dialog_list_ids.insert(dialog_list_id);
          }
        }
      }
    });
    for (auto dialog_list_id : dialog_list_ids) {
      auto *list = get_dialog_list(dialog_list_id);
      CHECK(list != nullptr);
      if (delta[dialog_list_id] != 0 && list->is_message_unread_count_inited_) {
        if (is_muted) {
          list->unread_message_muted_count_ += delta[dialog_list_id];
        } else {
          list->unread_message_muted_count_ -= delta[dialog_list_id];
        }
        send_update_unread_message_count(*list, DialogId(), true, "on_update_notification_scope_is_muted");
      }
      if (total_count[dialog_list_id] != 0 && list->is_dialog_unread_count_inited_) {
        if (is_muted) {
          list->unread_dialog_muted_count_ += total_count[dialog_list_id];
          list->unread_dialog_muted_marked_count_ += marked_count[dialog_list_id];
        } else {
          list->unread_dialog_muted_count_ -= total_count[dialog_list_id];
          list->unread_dialog_muted_marked_count_ -= marked_count[dialog_list_id];
        }
        send_update_unread_chat_count(*list, DialogId(), true, "on_update_notification_scope_is_muted");
      }
    }
  }

  if (td_->dialog_filter_manager_->have_dialog_filters()) {
    dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
      Dialog *d = dialog.get();
      if (need_unread_counter(d->order) && d->notification_settings.use_default_mute_until &&
          td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id) == scope) {
        update_dialog_lists(d, get_dialog_positions(d), true, false, "on_update_notification_scope_is_muted");
      }
    });
  }

  if (is_muted) {
    dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
      Dialog *d = dialog.get();
      if (need_unread_counter(d->order) && d->notification_settings.use_default_mute_until &&
          td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id) == scope) {
        remove_all_dialog_notifications(d, false, "on_update_notification_scope_is_muted");
      }
    });
  }
}

void MessagesManager::on_dialog_unmute(DialogId dialog_id) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (d->notification_settings.use_default_mute_until) {
    return;
  }
  if (d->notification_settings.mute_until == 0) {
    return;
  }

  auto unix_time = G()->unix_time();
  if (d->notification_settings.mute_until > unix_time) {
    LOG(INFO) << "Failed to unmute " << dialog_id << " in " << unix_time << ", will be unmuted in "
              << d->notification_settings.mute_until;
    schedule_dialog_unmute(dialog_id, false, d->notification_settings.mute_until, unix_time);
    return;
  }

  LOG(INFO) << "Unmute " << dialog_id;
  update_dialog_unmute_timeout(d, d->notification_settings.use_default_mute_until, d->notification_settings.mute_until,
                               false, 0);
  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateChatNotificationSettings>(
                   get_chat_id_object(dialog_id, "updateChatNotificationSettings 2"),
                   get_chat_notification_settings_object(&d->notification_settings)));
  on_dialog_updated(dialog_id, "on_dialog_unmute");
}

void MessagesManager::on_update_dialog_notify_settings(
    DialogId dialog_id, tl_object_ptr<telegram_api::peerNotifySettings> &&peer_notify_settings, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  VLOG(notifications) << "Receive notification settings for " << dialog_id << " from " << source << ": "
                      << to_string(peer_notify_settings);

  DialogNotificationSettings *current_settings = get_dialog_notification_settings(dialog_id, true);
  if (current_settings == nullptr) {
    return;
  }

  DialogNotificationSettings notification_settings =
      ::td::get_dialog_notification_settings(std::move(peer_notify_settings), current_settings);
  if (!notification_settings.is_synchronized) {
    return;
  }

  update_dialog_notification_settings(dialog_id, current_settings, std::move(notification_settings));
}

void MessagesManager::on_update_dialog_available_reactions(
    DialogId dialog_id, telegram_api::object_ptr<telegram_api::ChatReactions> &&available_reactions) {
  Dialog *d = get_dialog_force(dialog_id, "on_update_dialog_available_reactions");
  if (d == nullptr) {
    return;
  }

  set_dialog_available_reactions(d, ChatReactions(std::move(available_reactions)));
}

void MessagesManager::set_dialog_available_reactions(Dialog *d, ChatReactions &&available_reactions) {
  CHECK(d != nullptr);
  switch (d->dialog_id.get_type()) {
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::User:
    case DialogType::SecretChat:
    default:
      UNREACHABLE();
      break;
  }
  if (d->available_reactions == available_reactions) {
    if (!d->is_available_reactions_inited) {
      d->is_available_reactions_inited = true;
      on_dialog_updated(d->dialog_id, "set_dialog_available_reactions");
    }
    return;
  }

  LOG(INFO) << "Update available reactions in " << d->dialog_id << " to " << available_reactions;

  auto old_active_reactions = get_active_reactions(d->available_reactions);
  auto new_active_reactions = get_active_reactions(available_reactions);
  bool need_update = old_active_reactions != new_active_reactions;
  bool need_update_message_reactions_visibility =
      old_active_reactions.empty() != new_active_reactions.empty() && !td_->auth_manager_->is_bot();

  d->available_reactions = std::move(available_reactions);
  d->is_available_reactions_inited = true;
  if (need_update_message_reactions_visibility) {
    if (!old_active_reactions.empty()) {
      hide_dialog_message_reactions(d);
    }

    set_dialog_next_available_reactions_generation(d, d->available_reactions_generation);
  }
  on_dialog_updated(d->dialog_id, "set_dialog_available_reactions");

  if (need_update) {
    send_update_chat_available_reactions(d);
  }
}

void MessagesManager::set_dialog_next_available_reactions_generation(Dialog *d, uint32 generation) {
  CHECK(d != nullptr);
  switch (d->dialog_id.get_type()) {
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::User:
    case DialogType::SecretChat:
    default:
      UNREACHABLE();
      break;
  }
  if (get_active_reactions(d->available_reactions).empty()) {
    // 0 -> 1
    // 1 -> 3
    generation = generation + (generation & 1) + 1;
  } else {
    // 0 -> 2
    // 1 -> 2
    generation = generation - (generation & 1) + 2;
  }
  LOG(INFO) << "Change available reactions generation from " << d->available_reactions_generation << " to "
            << generation << " in " << d->dialog_id;
  d->available_reactions_generation = generation;
}

void MessagesManager::hide_dialog_message_reactions(Dialog *d) {
  CHECK(!td_->auth_manager_->is_bot());
  auto dialog_type = d->dialog_id.get_type();
  CHECK(dialog_type == DialogType::Chat || dialog_type == DialogType::Channel);
  auto message_ids = find_dialog_messages(
      d, [](const Message *m) { return m->reactions != nullptr && !m->reactions->reactions_.empty(); });
  for (auto message_id : message_ids) {
    Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->reactions != nullptr);
    bool need_update_unread_reactions = !m->reactions->unread_reactions_.empty();
    m->reactions = nullptr;
    // don't resave all messages to the database
    if (need_update_unread_reactions) {
      send_update_message_unread_reactions(d->dialog_id, m, d->unread_reaction_count);
    }
    send_update_message_interaction_info(d->dialog_id, m);
  }
  if (d->unread_reaction_count != 0) {
    set_dialog_unread_reaction_count(d, 0);
  }
}

void MessagesManager::set_active_reactions(vector<ReactionType> active_reaction_types) {
  if (active_reaction_types == active_reaction_types_) {
    return;
  }

  LOG(INFO) << "Set active reactions to " << active_reaction_types;
  bool is_changed = active_reaction_types != active_reaction_types_;
  active_reaction_types_ = std::move(active_reaction_types);

  auto old_active_reaction_pos_ = std::move(active_reaction_pos_);
  active_reaction_pos_.clear();
  for (size_t i = 0; i < active_reaction_types_.size(); i++) {
    CHECK(!active_reaction_types_[i].is_empty());
    active_reaction_pos_[active_reaction_types_[i]] = i;
  }

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    Dialog *d = dialog.get();
    switch (dialog_id.get_type()) {
      case DialogType::User:
        if (is_changed) {
          send_update_chat_available_reactions(d);
        }
        break;
      case DialogType::Chat:
      case DialogType::Channel: {
        auto old_reactions = d->available_reactions.get_active_reactions(old_active_reaction_pos_);
        auto new_reactions = d->available_reactions.get_active_reactions(active_reaction_pos_);
        if (old_reactions != new_reactions) {
          if (old_reactions.empty() != new_reactions.empty()) {
            if (!old_reactions.empty()) {
              hide_dialog_message_reactions(d);
            }
            set_dialog_next_available_reactions_generation(d, d->available_reactions_generation);
            on_dialog_updated(d->dialog_id, "set_active_reactions");
          }
          send_update_chat_available_reactions(d);
        }
        break;
      }
      case DialogType::SecretChat:
        break;
      default:
        UNREACHABLE();
        break;
    }
  });
}

ChatReactions MessagesManager::get_active_reactions(const ChatReactions &available_reactions) const {
  if (td_->auth_manager_->is_bot()) {
    return available_reactions;
  }
  return available_reactions.get_active_reactions(active_reaction_pos_);
}

ChatReactions MessagesManager::get_dialog_active_reactions(const Dialog *d) const {
  CHECK(d != nullptr);
  switch (d->dialog_id.get_type()) {
    case DialogType::User:
      return ChatReactions(true, true);
    case DialogType::Chat:
    case DialogType::Channel:
      return get_active_reactions(d->available_reactions);
    case DialogType::SecretChat:
      return {};
    default:
      UNREACHABLE();
      return {};
  }
}

// affects all users
ChatReactions MessagesManager::get_message_active_reactions(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  if (is_service_message_content(m->content->get_type()) || !m->ttl.is_empty() || !m->message_id.is_valid() ||
      !m->message_id.is_server()) {
    return ChatReactions();
  }
  auto dialog_id = d->dialog_id;
  if (is_discussion_message(dialog_id, m)) {
    auto linked_dialog_id = m->forward_info->get_last_dialog_id();
    d = get_dialog(linked_dialog_id);
    if (d == nullptr) {
      LOG(ERROR) << "Failed to find linked " << linked_dialog_id << " to determine correct active reactions";
      return ChatReactions();
    }
  }
  return get_dialog_active_reactions(d);
}

bool MessagesManager::need_poll_dialog_message_reactions(const Dialog *d) {
  CHECK(d != nullptr);
  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::SecretChat:
      return false;
    case DialogType::Chat:
    case DialogType::Channel:
      return (d->available_reactions_generation & 1) == 0;
    default:
      UNREACHABLE();
      return {};
  }
}

bool MessagesManager::need_poll_message_reactions(const Dialog *d, const Message *m) {
  CHECK(m != nullptr);
  if (!m->message_id.is_valid() || !m->message_id.is_server()) {
    return false;
  }
  if (!need_poll_dialog_message_reactions(d)) {
    return false;
  }
  if (m->reactions != nullptr) {
    return true;
  }
  if (m->available_reactions_generation == d->available_reactions_generation) {
    return false;
  }
  if (is_service_message_content(m->content->get_type())) {
    return false;
  }
  return true;
}

void MessagesManager::queue_message_reactions_reload(MessageFullId message_full_id) {
  auto dialog_id = message_full_id.get_dialog_id();
  CHECK(dialog_id.is_valid());
  auto message_id = message_full_id.get_message_id();
  CHECK(message_id.is_valid());
  being_reloaded_reactions_[dialog_id].message_ids.insert(message_id);
  try_reload_message_reactions(dialog_id, false);
}

void MessagesManager::queue_message_reactions_reload(DialogId dialog_id, const vector<MessageId> &message_ids) {
  LOG(INFO) << "Queue reload of reactions in " << message_ids << " in " << dialog_id;
  auto &message_ids_to_reload = being_reloaded_reactions_[dialog_id].message_ids;
  for (auto &message_id : message_ids) {
    CHECK(message_id.is_valid());
    message_ids_to_reload.insert(message_id);
  }
  try_reload_message_reactions(dialog_id, false);
}

void MessagesManager::try_reload_message_reactions(DialogId dialog_id, bool is_finished) {
  if (G()->close_flag()) {
    return;
  }

  auto it = being_reloaded_reactions_.find(dialog_id);
  if (it == being_reloaded_reactions_.end()) {
    return;
  }
  if (is_finished) {
    CHECK(it->second.is_request_sent);
    it->second.is_request_sent = false;

    if (it->second.message_ids.empty()) {
      being_reloaded_reactions_.erase(it);
      return;
    }
  } else if (it->second.is_request_sent) {
    return;
  }

  CHECK(!it->second.message_ids.empty());
  CHECK(!it->second.is_request_sent);

  it->second.is_request_sent = true;

  static constexpr size_t MAX_MESSAGE_IDS = 100;  // server-side limit
  vector<MessageId> message_ids;
  for (auto message_id_it = it->second.message_ids.begin();
       message_id_it != it->second.message_ids.end() && message_ids.size() < MAX_MESSAGE_IDS; ++message_id_it) {
    auto message_id = *message_id_it;
    if (pending_read_reactions_.count({dialog_id, message_id}) == 0) {
      message_ids.push_back(message_id);
    }
  }
  for (auto message_id : message_ids) {
    it->second.message_ids.erase(message_id);
  }
  reload_message_reactions(td_, dialog_id, std::move(message_ids));
}

bool MessagesManager::update_dialog_silent_send_message(Dialog *d, bool silent_send_message) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return false;
  }

  CHECK(d != nullptr);
  LOG_IF(WARNING, !d->notification_settings.is_synchronized)
      << "Have unknown notification settings in " << d->dialog_id;
  if (d->notification_settings.silent_send_message == silent_send_message) {
    return false;
  }

  LOG(INFO) << "Update silent send message in " << d->dialog_id << " to " << silent_send_message;
  d->notification_settings.silent_send_message = silent_send_message;

  on_dialog_updated(d->dialog_id, "update_dialog_silent_send_message");

  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateChatDefaultDisableNotification>(
                   get_chat_id_object(d->dialog_id, "updateChatDefaultDisableNotification"), silent_send_message));
  return true;
}

void MessagesManager::reget_dialog_action_bar(DialogId dialog_id, const char *source, bool is_repair) {
  if (G()->close_flag() || !dialog_id.is_valid() || td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, source);
  if (d == nullptr) {
    return;
  }

  if (is_repair && !d->need_repair_action_bar) {
    d->need_repair_action_bar = true;
    on_dialog_updated(dialog_id, source);
  }

  LOG(INFO) << "Reget action bar in " << dialog_id << " from " << source;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      td_->user_manager_->reload_user_full(dialog_id.get_user_id(), Auto(), source);
      return;
    case DialogType::Chat:
    case DialogType::Channel:
      if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
        return;
      }

      return td_->create_handler<GetPeerSettingsQuery>()->send(dialog_id);
    case DialogType::SecretChat:
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::repair_dialog_action_bar(Dialog *d, const char *source) {
  CHECK(d != nullptr);
  auto dialog_id = d->dialog_id;
  d->need_repair_action_bar = true;
  if (td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    create_actor<SleepActor>(
        "RepairChatActionBarActor", 1.0,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, source](Result<Unit> result) {
          send_closure(actor_id, &MessagesManager::reget_dialog_action_bar, dialog_id, source, true);
        }))
        .release();
  }
  // there is no need to change action bar
  on_dialog_updated(dialog_id, source);
}

void MessagesManager::hide_dialog_action_bar(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "hide_dialog_action_bar");
  if (d == nullptr) {
    return;
  }
  hide_dialog_action_bar(d);
}

void MessagesManager::hide_dialog_action_bar(Dialog *d) {
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  if (!d->know_action_bar) {
    return;
  }
  if (d->need_repair_action_bar) {
    d->need_repair_action_bar = false;
    on_dialog_updated(d->dialog_id, "hide_dialog_action_bar");
  }
  if (d->action_bar == nullptr) {
    return;
  }

  d->action_bar = nullptr;
  send_update_chat_action_bar(d);
}

void MessagesManager::remove_dialog_action_bar(DialogId dialog_id, Promise<Unit> &&promise) {
  TRY_RESULT_PROMISE(promise, d, check_dialog_access(dialog_id, true, AccessRights::Read, "remove_dialog_action_bar"));

  if (dialog_id.get_type() == DialogType::SecretChat) {
    dialog_id = DialogId(td_->user_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
    TRY_RESULT_PROMISE_ASSIGN(promise, d,
                              check_dialog_access(dialog_id, false, AccessRights::Read, "remove_dialog_action_bar 2"));
  }

  if (!d->know_action_bar) {
    return promise.set_error(Status::Error(400, "Can't update chat action bar"));
  }
  if (d->need_repair_action_bar) {
    d->need_repair_action_bar = false;
    on_dialog_updated(dialog_id, "remove_dialog_action_bar");
  }
  if (d->action_bar == nullptr) {
    return promise.set_value(Unit());
  }

  d->action_bar = nullptr;
  send_update_chat_action_bar(d);

  toggle_dialog_report_spam_state_on_server(dialog_id, false, 0, std::move(promise));
}

void MessagesManager::hide_all_business_bot_manager_bars() {
  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    Dialog *d = dialog.get();
    if (d->business_bot_manage_bar != nullptr) {
      d->business_bot_manage_bar = nullptr;
      send_update_chat_business_bot_manage_bar(d);
    }
  });
}

void MessagesManager::repair_dialog_active_group_call_id(DialogId dialog_id) {
  if (td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    LOG(INFO) << "Repair active voice chat ID in " << dialog_id;
    create_actor<SleepActor>("RepairChatActiveVoiceChatId", 1.0,
                             PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Result<Unit> result) {
                               send_closure(actor_id, &MessagesManager::do_repair_dialog_active_group_call_id,
                                            dialog_id);
                             }))
        .release();
  }
}

void MessagesManager::do_repair_dialog_active_group_call_id(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  bool need_repair_active_group_call_id = d->has_active_group_call && !d->active_group_call_id.is_valid();
  bool need_repair_expected_group_call_id =
      d->has_expected_active_group_call_id && d->active_group_call_id != d->expected_active_group_call_id;
  d->has_expected_active_group_call_id = false;
  if (!need_repair_active_group_call_id && !need_repair_expected_group_call_id) {
    return;
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return;
  }

  td_->dialog_manager_->reload_dialog_info_full(dialog_id, "do_repair_dialog_active_group_call_id");
}

class MessagesManager::ToggleDialogReportSpamStateOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_spam_dialog_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(is_spam_dialog_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(is_spam_dialog_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_report_spam_state_on_server_log_event(DialogId dialog_id,
                                                                                 bool is_spam_dialog) {
  ToggleDialogReportSpamStateOnServerLogEvent log_event{dialog_id, is_spam_dialog};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogReportSpamStateOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_report_spam_state_on_server(DialogId dialog_id, bool is_spam_dialog,
                                                                uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_report_spam_state_on_server_log_event(dialog_id, is_spam_dialog);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(dialog_id, is_spam_dialog);
    case DialogType::SecretChat:
      if (is_spam_dialog) {
        return td_->create_handler<ReportEncryptedSpamQuery>(std::move(promise))->send(dialog_id);
      } else {
        auto user_id = td_->user_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
        if (!user_id.is_valid()) {
          return promise.set_error(Status::Error(400, "Peer user not found"));
        }
        return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(DialogId(user_id), false);
      }
    case DialogType::None:
    default:
      UNREACHABLE();
      return;
  }
}

void MessagesManager::on_get_peer_settings(DialogId dialog_id,
                                           tl_object_ptr<telegram_api::peerSettings> &&peer_settings,
                                           bool ignore_privacy_exception) {
  CHECK(peer_settings != nullptr);
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (dialog_id.get_type() == DialogType::User && !ignore_privacy_exception) {
    td_->user_manager_->on_update_user_need_phone_number_privacy_exception(dialog_id.get_user_id(),
                                                                           peer_settings->need_contacts_exception_);
  }

  Dialog *d = get_dialog_force(dialog_id, "on_get_peer_settings");
  if (d == nullptr) {
    return;
  }

  auto business_bot_manage_bar = BusinessBotManageBar::create(
      peer_settings->business_bot_paused_, peer_settings->business_bot_can_reply_,
      UserId(peer_settings->business_bot_id_), std::move(peer_settings->business_bot_manage_url_));
  fix_dialog_business_bot_manage_bar(dialog_id, business_bot_manage_bar.get());
  if (d->business_bot_manage_bar != business_bot_manage_bar) {
    d->business_bot_manage_bar = std::move(business_bot_manage_bar);
    send_update_chat_business_bot_manage_bar(d);
  }

  auto distance =
      (peer_settings->flags_ & telegram_api::peerSettings::GEO_DISTANCE_MASK) != 0 ? peer_settings->geo_distance_ : -1;
  if (distance < -1 || d->has_outgoing_messages) {
    distance = -1;
  }
  auto action_bar =
      DialogActionBar::create(peer_settings->report_spam_, peer_settings->add_contact_, peer_settings->block_contact_,
                              peer_settings->share_contact_, peer_settings->report_geo_, peer_settings->autoarchived_,
                              distance, peer_settings->invite_members_, peer_settings->request_chat_title_,
                              peer_settings->request_chat_broadcast_, peer_settings->request_chat_date_);

  fix_dialog_action_bar(d, action_bar.get());

  if (d->action_bar == action_bar) {
    if (!d->know_action_bar || d->need_repair_action_bar) {
      d->know_action_bar = true;
      d->need_repair_action_bar = false;
      on_dialog_updated(d->dialog_id, "on_get_peer_settings");
    }
    return;
  }

  d->know_action_bar = true;
  d->need_repair_action_bar = false;
  d->action_bar = std::move(action_bar);

  send_update_chat_action_bar(d);
}

void MessagesManager::fix_dialog_action_bar(const Dialog *d, DialogActionBar *action_bar) {
  if (action_bar == nullptr) {
    return;
  }

  CHECK(d != nullptr);
  action_bar->fix(td_, d->dialog_id, d->is_blocked, d->folder_id);
}

void MessagesManager::fix_dialog_business_bot_manage_bar(DialogId dialog_id,
                                                         BusinessBotManageBar *business_bot_manage_bar) {
  if (business_bot_manage_bar == nullptr) {
    return;
  }

  business_bot_manage_bar->fix(dialog_id);
}

Result<string> MessagesManager::get_login_button_url(MessageFullId message_full_id, int64 button_id) {
  TRY_RESULT(d,
             check_dialog_access(message_full_id.get_dialog_id(), false, AccessRights::Read, "get_login_button_url"));

  auto m = get_message_force(d, message_full_id.get_message_id(), "get_login_button_url");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }
  if (m->reply_markup == nullptr || m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
    return Status::Error(400, "Message has no inline keyboard");
  }
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Can't use login buttons from scheduled messages");
  }
  if (!m->message_id.is_server()) {
    // it shouldn't have UrlAuth buttons anyway
    return Status::Error(400, "Message is not server");
  }
  if (button_id < std::numeric_limits<int32>::min() || button_id > std::numeric_limits<int32>::max()) {
    return Status::Error(400, "Invalid button identifier specified");
  }

  for (auto &row : m->reply_markup->inline_keyboard) {
    for (auto &button : row) {
      if (button.type == InlineKeyboardButton::Type::UrlAuth && button.id == button_id) {
        return button.data;
      }
    }
  }

  return Status::Error(400, "Button not found");
}

void MessagesManager::load_secret_thumbnail(FileId thumbnail_file_id) {
  class Callback final : public FileManager::DownloadCallback {
   public:
    explicit Callback(Promise<Unit> download_promise) : download_promise_(std::move(download_promise)) {
    }

    void on_download_ok(FileId file_id) final {
      download_promise_.set_value(Unit());
    }
    void on_download_error(FileId file_id, Status error) final {
      download_promise_.set_error(std::move(error));
    }

   private:
    Promise<Unit> download_promise_;
  };

  auto thumbnail_promise = PromiseCreator::lambda([actor_id = actor_id(this),
                                                   thumbnail_file_id](Result<BufferSlice> r_thumbnail) {
    BufferSlice thumbnail_slice;
    if (r_thumbnail.is_ok()) {
      thumbnail_slice = r_thumbnail.move_as_ok();
    }
    send_closure(actor_id, &MessagesManager::on_load_secret_thumbnail, thumbnail_file_id, std::move(thumbnail_slice));
  });

  auto download_promise = PromiseCreator::lambda(
      [thumbnail_file_id, thumbnail_promise = std::move(thumbnail_promise)](Result<Unit> r_download) mutable {
        if (r_download.is_error()) {
          thumbnail_promise.set_error(r_download.move_as_error());
          return;
        }
        send_closure(G()->file_manager(), &FileManager::get_content, thumbnail_file_id, std::move(thumbnail_promise));
      });

  send_closure(G()->file_manager(), &FileManager::download, thumbnail_file_id,
               std::make_shared<Callback>(std::move(download_promise)), 1, -1, -1,
               Promise<td_api::object_ptr<td_api::file>>());
}

void MessagesManager::on_upload_media(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file,
                                      tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was canceled
    return;
  }

  auto message_full_id = it->second.first;
  auto thumbnail_file_id = it->second.second;

  being_uploaded_files_.erase(it);

  Message *m = get_message(message_full_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // file upload should be already canceled in cancel_send_message_query, it shouldn't happen
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();
  auto dialog_id = message_full_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the file or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status;

    fail_send_message(message_full_id, std::move(can_send_status));
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      if (input_file && thumbnail_file_id.is_valid()) {
        // TODO: download thumbnail if needed (like in secret chats)
        LOG(INFO) << "Ask to upload thumbnail " << thumbnail_file_id;
        bool is_inserted =
            being_uploaded_thumbnails_
                .emplace(thumbnail_file_id, UploadedThumbnailInfo{message_full_id, file_id, std::move(input_file)})
                .second;
        CHECK(is_inserted);
        td_->file_manager_->upload(thumbnail_file_id, upload_thumbnail_callback_, 32, m->message_id.get());
      } else {
        do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), nullptr);
      }
      break;
    case DialogType::SecretChat:
      if (thumbnail_file_id.is_valid()) {
        LOG(INFO) << "Ask to load thumbnail " << thumbnail_file_id;
        bool is_inserted = being_loaded_secret_thumbnails_
                               .emplace(thumbnail_file_id, UploadedSecretThumbnailInfo{message_full_id, file_id,
                                                                                       std::move(input_encrypted_file)})
                               .second;
        CHECK(is_inserted);

        load_secret_thumbnail(thumbnail_file_id);
      } else {
        do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_encrypted_file), BufferSlice());
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::do_send_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                    tl_object_ptr<telegram_api::InputFile> input_file,
                                    tl_object_ptr<telegram_api::InputFile> input_thumbnail) {
  CHECK(m != nullptr);

  bool have_input_file = input_file != nullptr;
  bool have_input_thumbnail = input_thumbnail != nullptr;
  LOG(INFO) << "Do send media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file << ", have_input_thumbnail = " << have_input_thumbnail
            << ", self-destruct time = " << m->ttl;

  const MessageContent *content = nullptr;
  if (m->message_id.is_any_server()) {
    content = m->edited_content.get();
    if (content == nullptr) {
      LOG(ERROR) << "Message has no edited content";
      return;
    }
  } else {
    content = m->content.get();
  }

  auto input_media = get_input_media(content, td_, std::move(input_file), std::move(input_thumbnail), file_id,
                                     thumbnail_file_id, m->ttl, m->send_emoji, true);
  LOG_CHECK(input_media != nullptr) << to_string(get_message_object(dialog_id, m, "do_send_media")) << ' '
                                    << have_input_file << ' ' << have_input_thumbnail << ' ' << file_id << ' '
                                    << thumbnail_file_id << ' ' << m->ttl;

  on_message_media_uploaded(dialog_id, m, std::move(input_media), file_id, thumbnail_file_id);
}

void MessagesManager::do_send_secret_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                           tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file,
                                           BufferSlice thumbnail) {
  CHECK(dialog_id.get_type() == DialogType::SecretChat);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(m->message_id.is_yet_unsent());

  bool have_input_file = input_encrypted_file != nullptr;
  LOG(INFO) << "Do send secret media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file;

  auto layer = td_->user_manager_->get_secret_chat_layer(dialog_id.get_secret_chat_id());
  on_secret_message_media_uploaded(
      dialog_id, m,
      get_secret_input_media(m->content.get(), td_, std::move(input_encrypted_file), std::move(thumbnail), layer),
      file_id, thumbnail_file_id);
}

void MessagesManager::on_upload_media_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(WARNING) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was canceled
    return;
  }

  auto message_full_id = it->second.first;

  being_uploaded_files_.erase(it);

  bool is_edit = message_full_id.get_message_id().is_any_server();
  if (is_edit) {
    fail_edit_message_media(message_full_id, std::move(status));
  } else {
    fail_send_message(message_full_id, std::move(status));
  }
}

void MessagesManager::on_load_secret_thumbnail(FileId thumbnail_file_id, BufferSlice thumbnail) {
  if (G()->close_flag()) {
    // do not send secret media if closing, thumbnail may be wrong
    return;
  }

  LOG(INFO) << "SecretThumbnail " << thumbnail_file_id << " has been loaded with size " << thumbnail.size();

  auto it = being_loaded_secret_thumbnails_.find(thumbnail_file_id);
  if (it == being_loaded_secret_thumbnails_.end()) {
    // just in case, as in on_upload_thumbnail
    return;
  }

  auto message_full_id = it->second.message_full_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_loaded_secret_thumbnails_.erase(it);

  Message *m = get_message(message_full_id);
  if (m == nullptr) {
    // message has already been deleted by the user, do not need to send it
    // cancel file upload of the main file to allow next upload with the same file to succeed
    LOG(INFO) << "Message with a media has already been deleted";
    cancel_upload_file(file_id, "on_load_secret_thumbnail");
    return;
  }
  CHECK(m->message_id.is_yet_unsent());

  if (thumbnail.empty()) {
    delete_message_content_thumbnail(m->content.get(), td_);
  }

  auto dialog_id = message_full_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (can_send_status.is_error()) {
    // secret chat was closed during load of the file
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status;

    fail_send_message(message_full_id, std::move(can_send_status));
    return;
  }

  do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail));
}

void MessagesManager::on_upload_thumbnail(FileId thumbnail_file_id,
                                          tl_object_ptr<telegram_api::InputFile> thumbnail_input_file) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "Thumbnail " << thumbnail_file_id << " has been uploaded as " << to_string(thumbnail_input_file);

  auto it = being_uploaded_thumbnails_.find(thumbnail_file_id);
  if (it == being_uploaded_thumbnails_.end()) {
    // callback may be called just before the thumbnail upload was canceled
    return;
  }

  auto message_full_id = it->second.message_full_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_uploaded_thumbnails_.erase(it);

  Message *m = get_message(message_full_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // thumbnail file upload should be already canceled in cancel_send_message_query
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();

  if (thumbnail_input_file == nullptr) {
    delete_message_content_thumbnail(is_edit ? m->edited_content.get() : m->content.get(), td_);
  }

  auto dialog_id = message_full_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the thumbnail or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status;

    fail_send_message(message_full_id, std::move(can_send_status));
    return;
  }

  do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail_input_file));
}

void MessagesManager::before_get_difference() {
  running_get_difference_ = true;

  // scheduled messages are not returned in getDifference, so we must always reget them after it
  scheduled_messages_sync_generation_++;
}

void MessagesManager::after_get_difference() {
  CHECK(!td_->updates_manager_->running_get_difference());

  running_get_difference_ = false;

  if (!pending_on_get_dialogs_.empty()) {
    LOG(INFO) << "Apply postponed results of getDialogs";
    for (auto &res : pending_on_get_dialogs_) {
      on_get_dialogs(res.folder_id, std::move(res.dialogs), res.total_count, std::move(res.messages),
                     std::move(res.promise));
    }
    pending_on_get_dialogs_.clear();
  }

  if (!postponed_chat_read_inbox_updates_.empty()) {
    LOG(INFO) << "Send postponed chat read inbox updates";
    auto dialog_ids = std::move(postponed_chat_read_inbox_updates_);
    for (auto dialog_id : dialog_ids) {
      send_update_chat_read_inbox(get_dialog(dialog_id), false, "after_get_difference");
    }
  }
  while (!postponed_unread_message_count_updates_.empty()) {
    auto *list = get_dialog_list(*postponed_unread_message_count_updates_.begin());
    CHECK(list != nullptr);
    send_update_unread_message_count(*list, DialogId(), true, "after_get_difference");
  }
  while (!postponed_unread_chat_count_updates_.empty()) {
    auto *list = get_dialog_list(*postponed_unread_chat_count_updates_.begin());
    CHECK(list != nullptr);
    send_update_unread_chat_count(*list, DialogId(), true, "after_get_difference");
  }

  vector<MessageFullId> update_message_ids_to_delete;
  for (auto &it : update_message_ids_) {
    // there can be unhandled updateMessageId updates after getDifference even for ordinary chats,
    // because despite updates coming during getDifference have already been applied,
    // some of them could be postponed because of PTS gap
    auto message_full_id = it.first;
    auto dialog_id = message_full_id.get_dialog_id();
    auto message_id = message_full_id.get_message_id();
    auto old_message_id = it.second;
    CHECK(message_id.is_valid());
    CHECK(message_id.is_server());
    switch (dialog_id.get_type()) {
      case DialogType::Channel:
        // get channel difference may prevent updates from being applied
        if (running_get_channel_difference(dialog_id)) {
          break;
        }
      // fallthrough
      case DialogType::User:
      case DialogType::Chat: {
        if (!have_message_force({dialog_id, old_message_id}, "after get difference")) {
          // The sent message has already been deleted by the user or sent to inaccessible channel.
          // The sent message may never be received, but we will need updateMessageId in case the message is received
          // to delete it from the server and not add to the chat.
          // But if the chat is inaccessible or the message is in an inaccessible chat part, then we will not be able to
          // add the message or delete it from the server. In this case we forget updateMessageId for such messages in
          // order to not check them over and over.
          const Dialog *d = get_dialog(dialog_id);
          if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read) ||
              (d != nullptr &&
               message_id <= td::max(d->last_clear_history_message_id, d->max_unavailable_message_id))) {
            update_message_ids_to_delete.push_back(message_full_id);
          }
          break;
        }

        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        if (dialog_id.get_type() == DialogType::Channel || message_id <= d->last_new_message_id) {
          LOG(ERROR) << "Receive updateMessageId from " << old_message_id << " to " << message_full_id
                     << " but not receive corresponding message, last_new_message_id = " << d->last_new_message_id;
        }
        if (message_id <= d->last_new_message_id) {
          get_message_from_server(
              message_full_id,
              PromiseCreator::lambda([actor_id = actor_id(this), message_full_id, old_message_id](Result<Unit> result) {
                send_closure(actor_id, &MessagesManager::on_restore_missing_message_after_get_difference,
                             message_full_id, old_message_id, std::move(result));
              }),
              "get missing");
        } else if (dialog_id.get_type() == DialogType::Channel) {
          schedule_get_channel_difference(dialog_id, 0, message_id, 0.001, "after_get_difference");
        }
        break;
      }
      case DialogType::SecretChat:
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }
  for (const auto &message_full_id : update_message_ids_to_delete) {
    update_message_ids_.erase(message_full_id);
  }

  if (!td_->auth_manager_->is_bot()) {
    if (!G()->td_db()->get_binlog_pmc()->isset("fetched_marks_as_unread")) {
      td_->create_handler<GetDialogUnreadMarksQuery>()->send();
    }

    auto dialog_list_id = DialogListId(FolderId::archive());
    auto *list = get_dialog_list(dialog_list_id);
    CHECK(list != nullptr);
    if (!list->is_dialog_unread_count_inited_) {
      int32 limit = list->are_pinned_dialogs_inited_ ? static_cast<int32>(list->pinned_dialogs_.size())
                                                     : get_pinned_dialogs_limit(dialog_list_id);
      LOG(INFO) << "Loading chat list in " << dialog_list_id << " to init total unread count";
      get_dialogs_from_list(dialog_list_id, limit + 2, Auto());
    }
  }
}

void MessagesManager::on_restore_missing_message_after_get_difference(MessageFullId message_full_id,
                                                                      MessageId old_message_id, Result<Unit> result) {
  if (result.is_error()) {
    LOG(WARNING) << "Failed to get missing " << message_full_id << " for " << old_message_id << ": " << result.error();
  } else {
    LOG(WARNING) << "Successfully get missing " << message_full_id << " for " << old_message_id;

    bool have_message = have_message_force(message_full_id, "on_restore_missing_message_after_get_difference");
    if (!have_message && update_message_ids_.count(message_full_id)) {
      LOG(ERROR) << "Receive messageEmpty instead of missing " << message_full_id << " for " << old_message_id;

      delete_dialog_messages(message_full_id.get_dialog_id(), {old_message_id}, false,
                             "on_restore_missing_message_after_get_difference");

      update_message_ids_.erase(message_full_id);
    }
  }
}

void MessagesManager::on_get_empty_messages(DialogId dialog_id, const vector<MessageId> &empty_message_ids) {
  if (!empty_message_ids.empty()) {
    delete_dialog_messages(dialog_id, empty_message_ids, false, "on_get_empty_messages");
  }
}

void MessagesManager::get_channel_difference_if_needed(DialogId dialog_id, MessageId message_id, const char *source) {
  if (!need_channel_difference_to_add_message(dialog_id, message_id)) {
    return;
  }
  const Dialog *d = get_dialog(dialog_id);
  get_channel_difference(dialog_id, d == nullptr ? load_channel_pts(dialog_id) : d->pts, 0, message_id, true, source);
}

void MessagesManager::get_channel_difference_if_needed(DialogId dialog_id, MessagesInfo &&messages_info,
                                                       Promise<MessagesInfo> &&promise, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return promise.set_value(std::move(messages_info));
  }
  if (!dialog_id.is_valid()) {
    return get_channel_differences_if_needed(std::move(messages_info), std::move(promise), source);
  }
  for (auto &message : messages_info.messages) {
    if (need_channel_difference_to_add_message(dialog_id, message)) {
      auto max_message_id = MessageId::get_max_message_id(messages_info.messages);
      return run_after_channel_difference(
          dialog_id, max_message_id,
          PromiseCreator::lambda([messages_info = std::move(messages_info), promise = std::move(promise)](
                                     Unit ignored) mutable { promise.set_value(std::move(messages_info)); }),
          source);
    }
  }
  promise.set_value(std::move(messages_info));
}

void MessagesManager::get_channel_differences_if_needed(MessagesInfo &&messages_info, Promise<MessagesInfo> &&promise,
                                                        const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return promise.set_value(std::move(messages_info));
  }
  MultiPromiseActorSafe mpas{"GetChannelDifferencesIfNeededMultiPromiseActor"};
  mpas.add_promise(Promise<Unit>());
  mpas.set_ignore_errors(true);
  auto lock = mpas.get_promise();
  for (auto &message : messages_info.messages) {
    if (message == nullptr) {
      continue;
    }

    auto dialog_id = DialogId::get_message_dialog_id(message);
    if (need_channel_difference_to_add_message(dialog_id, message)) {
      run_after_channel_difference(dialog_id, MessageId::get_message_id(message, false), mpas.get_promise(), source);
    }
  }
  // must be added after messages_info is checked
  mpas.add_promise(PromiseCreator::lambda([messages_info = std::move(messages_info), promise = std::move(promise)](
                                              Unit ignored) mutable { promise.set_value(std::move(messages_info)); }));
  lock.set_value(Unit());
}

void MessagesManager::get_channel_differences_if_needed(
    const vector<const telegram_api::object_ptr<telegram_api::Message> *> &messages, Promise<Unit> &&promise,
    const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return promise.set_value(Unit());
  }
  MultiPromiseActorSafe mpas{"GetChannelDifferencesIfNeededGenericMultiPromiseActor"};
  mpas.add_promise(std::move(promise));
  mpas.set_ignore_errors(true);
  auto lock = mpas.get_promise();
  for (const auto &message : messages) {
    if (message == nullptr) {
      continue;
    }
    auto dialog_id = DialogId::get_message_dialog_id(*message);
    if (need_channel_difference_to_add_message(dialog_id, *message)) {
      run_after_channel_difference(dialog_id, MessageId::get_message_id(*message, false), mpas.get_promise(), source);
    }
  }
  lock.set_value(Unit());
}

void MessagesManager::on_get_messages(vector<tl_object_ptr<telegram_api::Message>> &&messages, bool is_channel_message,
                                      bool is_scheduled, Promise<Unit> &&promise, const char *source) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  for (auto &message : messages) {
    LOG(INFO) << "Receive " << to_string(message);
    on_get_message(std::move(message), false, is_channel_message, is_scheduled, source);
  }
  promise.set_value(Unit());
}

bool MessagesManager::delete_newer_server_messages_at_the_end(Dialog *d, MessageId max_message_id) {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(!max_message_id.is_scheduled());
  auto message_ids = d->ordered_messages.find_newer_messages(max_message_id);
  if (message_ids.empty()) {
    return false;
  }

  vector<MessageId> server_message_ids;
  vector<MessageId> kept_message_ids;
  for (auto message_id : message_ids) {
    CHECK(message_id > max_message_id);
    if (message_id.is_server()) {
      server_message_ids.push_back(message_id);
    } else {
      kept_message_ids.push_back(message_id);
    }
  }

  delete_dialog_messages(d, server_message_ids, false, "delete_newer_server_messages_at_the_end");

  // connect all messages with ID > max_message_id
  for (size_t i = 0; i + 1 < kept_message_ids.size(); i++) {
    d->ordered_messages.attach_message_to_next(kept_message_ids[i], "delete_newer_server_messages_at_the_end");
  }

  return !kept_message_ids.empty();
}

void MessagesManager::on_get_history(DialogId dialog_id, MessageId from_message_id, MessageId old_last_new_message_id,
                                     int32 offset, int32 limit, bool from_the_end,
                                     vector<tl_object_ptr<telegram_api::Message>> &&messages, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " history messages " << (from_the_end ? "from the end " : "") << "in "
            << dialog_id << " from " << from_message_id << " with offset " << offset << " and limit " << limit;
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  if (from_the_end) {
    CHECK(from_message_id == MessageId());
  } else {
    CHECK(!from_message_id.is_scheduled());
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  MessageId last_received_message_id = messages.empty() ? MessageId() : MessageId::get_message_id(messages[0], false);
  if (old_last_new_message_id < d->last_new_message_id && (from_the_end || old_last_new_message_id < from_message_id) &&
      last_received_message_id < d->last_new_message_id) {
    // new server messages were added to the dialog since the request was sent, but weren't received
    // they should have been received, so we must repeat the request to get them
    get_history_impl(d, from_message_id, offset, limit, false, false, std::move(promise), "on_get_history");
    return;
  }

  // the server can return less messages than requested if some of messages are deleted during request
  // but if it happens, it is likely that there are no more messages on the server
  bool have_full_history = from_the_end && narrow_cast<int32>(messages.size()) < limit && messages.size() <= 1;

  if (messages.empty()) {
    if (have_full_history) {
      d->have_full_history = true;
      d->have_full_history_source = 1;
      on_dialog_updated(dialog_id, "set have_full_history");
    }

    if (from_the_end && d->have_full_history && d->ordered_messages.empty()) {
      if (!d->last_database_message_id.is_valid()) {
        set_dialog_is_empty(d, "on_get_history empty");
      } else {
        LOG(INFO) << "Skip marking " << dialog_id << " as empty, because it probably has messages from "
                  << d->first_database_message_id << " to " << d->last_database_message_id << " in the database";
      }
    }

    // be aware that in some cases an empty answer may be returned, because of the race of getHistory and deleteMessages
    // and not because there are no more messages
    promise.set_value(Unit());
    return;
  }

  if (messages.size() > 1) {
    // check that messages are received in decreasing message_id order
    MessageId cur_message_id = MessageId::max();
    for (const auto &message : messages) {
      MessageId message_id = MessageId::get_message_id(message, false);
      if (message_id >= cur_message_id) {
        string error = PSTRING() << "Receive messages in the wrong order in history of " << dialog_id << " from "
                                 << from_message_id << " with offset " << offset << ", limit " << limit
                                 << ", from_the_end = " << from_the_end << ": ";
        for (const auto &debug_message : messages) {
          error += to_string(debug_message);
        }
        LOG(ERROR) << error;
        promise.set_value(Unit());
        return;
      }
      cur_message_id = message_id;
    }
  }

  // be aware that returned messages are guaranteed to be consecutive messages, but if !from_the_end they
  // may be older (if some messages were deleted) or newer (if some messages were added) than an expected answer
  // be aware that any subset of the returned messages may be already deleted and returned as MessageEmpty
  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  MessageId first_added_message_id;
  MessageId last_added_message_id;
  bool have_next = false;

  if (narrow_cast<int32>(messages.size()) < limit + offset && messages.size() <= 1) {
    MessageId first_received_message_id = MessageId::get_message_id(messages.back(), false);
    if (first_received_message_id >= from_message_id && d->first_database_message_id.is_valid() &&
        first_received_message_id >= d->first_database_message_id) {
      // it is likely that there are no more history messages on the server
      have_full_history = true;
    }
  }

  if (d->last_new_message_id.is_valid()) {
    // remove too new messages from response
    while (!messages.empty()) {
      if (DialogId::get_message_dialog_id(messages[0]) == dialog_id &&
          MessageId::get_message_id(messages[0], false) <= d->last_new_message_id) {
        // the message is old enough
        break;
      }

      LOG(INFO) << "Ignore too new " << MessageId::get_message_id(messages[0], false);
      messages.erase(messages.begin());
      if (messages.empty()) {
        // received no suitable messages; try again
        return promise.set_value(Unit());
      } else {
        last_received_message_id = MessageId::get_message_id(messages[0], false);
      }
    }
  }

  bool prev_have_full_history = d->have_full_history;
  MessageId prev_last_new_message_id = d->last_new_message_id;
  MessageId prev_first_database_message_id = d->first_database_message_id;
  MessageId prev_last_database_message_id = d->last_database_message_id;
  MessageId prev_last_message_id = d->last_message_id;
  if (from_the_end) {
    // delete all server messages with ID > last_received_message_id
    // there were no new messages received after the getHistory request was sent, so they are already deleted message
    if (last_received_message_id.is_valid() && delete_newer_server_messages_at_the_end(d, last_received_message_id)) {
      have_next = true;
    }
  }

  for (auto &message : messages) {
    auto expected_message_id = MessageId::get_message_id(message, false);
    if (!have_next && from_the_end && expected_message_id < d->last_message_id) {
      // last message in the dialog should be attached to the next message if there is some
      have_next = true;
    }

    auto message_dialog_id = DialogId::get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive " << expected_message_id << " in wrong " << message_dialog_id << " instead of "
                 << dialog_id << ": " << oneline(to_string(message));
      continue;
    }

    auto message_full_id = on_get_message(std::move(message), false, is_channel_message, false, "get history");
    auto message_id = message_full_id.get_message_id();
    if (message_id.is_valid()) {
      CHECK(message_id == expected_message_id);
      if (have_next) {
        d->ordered_messages.attach_message_to_next(message_id, "on_get_history");
      }
      if (!last_added_message_id.is_valid()) {
        last_added_message_id = message_id;
      }

      if (!have_next) {
        have_next = true;
      } else if (first_added_message_id.is_valid()) {
        d->ordered_messages.attach_message_to_previous(first_added_message_id, "on_get_history");
      }
      first_added_message_id = message_id;
    }
  }

  if (from_the_end && last_added_message_id.is_valid() && last_added_message_id != last_received_message_id) {
    CHECK(last_added_message_id < last_received_message_id);
    delete_newer_server_messages_at_the_end(d, last_added_message_id);
  }

  if (have_full_history) {
    d->have_full_history = true;
    d->have_full_history_source = 2;
    on_dialog_updated(dialog_id, "set have_full_history 2");
  }

  if (from_the_end && !d->last_new_message_id.is_valid()) {
    set_dialog_last_new_message_id(
        d, last_added_message_id.is_valid() ? last_added_message_id : last_received_message_id, "on_get_history");
  }
  bool intersect_last_database_message_ids =
      last_added_message_id >= d->first_database_message_id && d->last_database_message_id >= first_added_message_id;
  bool need_update_database_message_ids =
      last_added_message_id.is_valid() && (from_the_end || intersect_last_database_message_ids);
  if (from_the_end && last_added_message_id.is_valid() && last_added_message_id > d->last_message_id) {
    CHECK(d->last_new_message_id.is_valid());
    set_dialog_last_message_id(d, last_added_message_id, "on_get_history");
    send_update_chat_last_message(d, "on_get_history");
  }

  if (need_update_database_message_ids) {
    if (from_the_end && !intersect_last_database_message_ids && d->last_database_message_id.is_valid()) {
      if (d->last_database_message_id < first_added_message_id || last_added_message_id == d->last_message_id) {
        set_dialog_first_database_message_id(d, MessageId(), "on_get_history 1");
        set_dialog_last_database_message_id(d, MessageId(), "on_get_history 1");
      } else {
        auto min_message_id = td::min(d->first_database_message_id, d->last_message_id);
        LOG_CHECK(last_added_message_id < min_message_id)
            << need_update_database_message_ids << ' ' << first_added_message_id << ' ' << last_added_message_id << ' '
            << d->first_database_message_id << ' ' << d->last_database_message_id << ' ' << d->last_new_message_id
            << ' ' << d->last_message_id << ' ' << prev_first_database_message_id << ' '
            << prev_last_database_message_id << ' ' << prev_last_new_message_id << ' ' << prev_last_message_id;
        if (min_message_id <= last_added_message_id.get_next_message_id(MessageType::Server)) {
          // connect local messages with last received server message
          set_dialog_first_database_message_id(d, last_added_message_id, "on_get_history 2");
        } else {
          LOG(WARNING) << "Have last " << d->last_message_id << " and first database " << d->first_database_message_id
                       << " in " << dialog_id << ", but received history from the end only up to "
                       << last_added_message_id;
          // can't connect messages, because there can be unknown server messages after last_added_message_id
        }
      }
    }
    if (!d->last_database_message_id.is_valid()) {
      CHECK(d->last_message_id.is_valid());
      auto it = d->ordered_messages.get_const_iterator(d->last_message_id);
      MessageId new_first_database_message_id;
      while (*it != nullptr) {
        auto message_id = (*it)->get_message_id();
        if (message_id.is_server() || message_id.is_local()) {
          if (!d->last_database_message_id.is_valid()) {
            set_dialog_last_database_message_id(d, message_id, "on_get_history");
          }
          new_first_database_message_id = message_id;
          try_restore_dialog_reply_markup(d, get_message(d, message_id));
        }
        --it;
      }
      if (new_first_database_message_id.is_valid()) {
        set_dialog_first_database_message_id(d, new_first_database_message_id, "on_get_history");
      }
    } else {
      LOG_CHECK(d->last_new_message_id.is_valid())
          << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " "
          << d->last_database_message_id << " " << first_added_message_id << " " << last_added_message_id << " "
          << d->last_message_id << " " << d->last_new_message_id << " " << d->have_full_history << " "
          << prev_last_new_message_id << " " << prev_first_database_message_id << " " << prev_last_database_message_id
          << " " << prev_last_message_id << " " << prev_have_full_history << " " << d->debug_last_new_message_id << " "
          << d->debug_first_database_message_id << " " << d->debug_last_database_message_id << " " << from_message_id
          << " " << offset << " " << limit << " " << messages.size() << " " << last_received_message_id << " "
          << d->debug_set_dialog_last_database_message_id;
      CHECK(d->first_database_message_id.is_valid());
      {
        auto it = d->ordered_messages.get_const_iterator(d->first_database_message_id);
        if (*it != nullptr && ((*it)->get_message_id() == d->first_database_message_id || (*it)->have_next())) {
          MessageId new_first_database_message_id = d->first_database_message_id;
          while (*it != nullptr) {
            auto message_id = (*it)->get_message_id();
            if ((message_id.is_server() || message_id.is_local()) && message_id < new_first_database_message_id) {
              new_first_database_message_id = message_id;
              try_restore_dialog_reply_markup(d, get_message(d, message_id));
            }
            --it;
          }
          if (new_first_database_message_id != d->first_database_message_id) {
            set_dialog_first_database_message_id(d, new_first_database_message_id, "on_get_history 2");
          }
        }
      }
      {
        auto it = d->ordered_messages.get_const_iterator(d->last_database_message_id);
        if (*it != nullptr && ((*it)->get_message_id() == d->last_database_message_id || (*it)->have_next())) {
          MessageId new_last_database_message_id = d->last_database_message_id;
          while (*it != nullptr) {
            auto message_id = (*it)->get_message_id();
            if ((message_id.is_server() || message_id.is_local()) && message_id > new_last_database_message_id) {
              new_last_database_message_id = message_id;
            }
            ++it;
          }
          if (new_last_database_message_id != d->last_database_message_id) {
            set_dialog_last_database_message_id(d, new_last_database_message_id, "on_get_history 2");
          }
        }
      }
    }
    LOG_CHECK(d->first_database_message_id.is_valid())
        << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " " << d->last_database_message_id
        << " " << first_added_message_id << " " << last_added_message_id << " " << d->last_message_id << " "
        << d->last_new_message_id << " " << d->have_full_history << " " << prev_last_new_message_id << " "
        << prev_first_database_message_id << " " << prev_last_database_message_id << " " << prev_last_message_id << " "
        << prev_have_full_history << " " << d->debug_last_new_message_id << " " << d->debug_first_database_message_id
        << " " << d->debug_last_database_message_id << " " << from_message_id << " " << offset << " " << limit << " "
        << messages.size() << " " << last_received_message_id << " " << d->debug_set_dialog_last_database_message_id;
    CHECK(d->last_database_message_id.is_valid());

    for (auto &first_message_id : d->first_database_message_id_by_index) {
      if (first_added_message_id < first_message_id && first_message_id <= last_added_message_id) {
        first_message_id = first_added_message_id;
      }
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_get_public_dialogs_search_result(const string &query,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&my_peers,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&peers) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  CHECK(!query.empty());
  found_public_dialogs_[query] = td_->dialog_manager_->get_peers_dialog_ids(std::move(peers));
  found_on_server_dialogs_[query] = td_->dialog_manager_->get_peers_dialog_ids(std::move(my_peers));

  set_promises(promises);
}

void MessagesManager::on_failed_public_dialogs_search(const string &query, Status &&error) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  found_public_dialogs_[query];     // negative cache
  found_on_server_dialogs_[query];  // negative cache

  fail_promises(promises, std::move(error));
}

void MessagesManager::on_get_message_search_result_calendar(
    DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageId from_message_id,
    MessageSearchFilter filter, int64 random_id, int32 total_count,
    vector<tl_object_ptr<telegram_api::Message>> &&messages,
    vector<tl_object_ptr<telegram_api::searchResultsCalendarPeriod>> &&periods, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());

  int32 received_message_count = 0;
  for (auto &message : messages) {
    auto new_message_full_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, "on_get_message_search_result_calendar");
    if (new_message_full_id == MessageFullId()) {
      total_count--;
      continue;
    }

    if (new_message_full_id.get_dialog_id() != dialog_id) {
      LOG(ERROR) << "Receive " << new_message_full_id << " instead of a message in " << dialog_id;
      total_count--;
      continue;
    }

    received_message_count++;
  }
  if (total_count < received_message_count) {
    LOG(ERROR) << "Receive " << received_message_count << " valid messages out of " << total_count << " in "
               << messages.size() << " messages";
    total_count = received_message_count;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (!saved_messages_topic_id.is_valid()) {
    auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
    if (old_message_count != total_count) {
      old_message_count = total_count;
      on_dialog_updated(dialog_id, "on_get_message_search_result_calendar");
    }
  }

  vector<td_api::object_ptr<td_api::messageCalendarDay>> days;
  for (auto &period : periods) {
    auto message_id = MessageId(ServerMessageId(period->min_msg_id_));
    const auto *m = get_message(d, message_id);
    if (m == nullptr) {
      LOG(ERROR) << "Failed to find " << message_id;
      continue;
    }
    if (period->count_ <= 0) {
      LOG(ERROR) << "Receive " << to_string(period);
      continue;
    }
    days.push_back(td_api::make_object<td_api::messageCalendarDay>(
        period->count_, get_message_object(dialog_id, m, "on_get_message_search_result_calendar")));
  }
  it->second = td_api::make_object<td_api::messageCalendar>(total_count, std::move(days));
  promise.set_value(Unit());
}

void MessagesManager::on_failed_get_message_search_result_calendar(int64 random_id) {
  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());
  found_dialog_message_calendars_.erase(it);
}

void MessagesManager::on_get_dialog_messages_search_result(
    DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, const string &query, DialogId sender_dialog_id,
    MessageId from_message_id, int32 offset, int32 limit, MessageSearchFilter filter, MessageId top_thread_message_id,
    const ReactionType &tag, int64 random_id, int32 total_count,
    vector<tl_object_ptr<telegram_api::Message>> &&messages, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " found messages in " << dialog_id;
  if (!dialog_id.is_valid()) {
    CHECK(query.empty());
    CHECK(!sender_dialog_id.is_valid());
    CHECK(!top_thread_message_id.is_valid());
    CHECK(!saved_messages_topic_id.is_valid());
    CHECK(tag.is_empty());
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());

    MessageId first_added_message_id;
    if (messages.empty()) {
      // messages may be empty because there are no more messages or they can't be found due to global limit
      // anyway pretend that there are no more messages
      first_added_message_id = MessageId::min();
    }

    auto &result = it->second.message_full_ids;
    CHECK(result.empty());
    int32 added_message_count = 0;
    MessageId next_offset_message_id;
    for (auto &message : messages) {
      auto message_id = MessageId::get_message_id(message, false);
      if (message_id.is_valid() && (!next_offset_message_id.is_valid() || message_id < next_offset_message_id)) {
        next_offset_message_id = message_id;
      }
      auto new_message_full_id = on_get_message(std::move(message), false, false, false, "search call messages");
      if (new_message_full_id == MessageFullId()) {
        continue;
      }

      result.push_back(new_message_full_id);
      added_message_count++;

      CHECK(message_id == new_message_full_id.get_message_id());
      CHECK(message_id.is_valid());
      if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
        first_added_message_id = message_id;
      }
    }
    if (total_count < added_message_count) {
      LOG(ERROR) << "Receive total_count = " << total_count << ", but added " << added_message_count
                 << " messages out of " << messages.size();
      total_count = added_message_count;
    }
    if (G()->use_message_database()) {
      bool update_state = false;

      auto &old_message_count = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];
      if (old_message_count != total_count) {
        LOG(INFO) << "Update calls database message count to " << total_count;
        old_message_count = total_count;
        update_state = true;
      }

      auto &old_first_db_message_id =
          calls_db_state_.first_calls_database_message_id_by_index[call_message_search_filter_index(filter)];
      bool from_the_end = !from_message_id.is_valid() || from_message_id >= MessageId::max();
      LOG(INFO) << "Have from_the_end = " << from_the_end << ", old_first_db_message_id = " << old_first_db_message_id
                << ", first_added_message_id = " << first_added_message_id << ", from_message_id = " << from_message_id;
      if ((from_the_end || (old_first_db_message_id.is_valid() && old_first_db_message_id <= from_message_id)) &&
          (!old_first_db_message_id.is_valid() || first_added_message_id < old_first_db_message_id)) {
        LOG(INFO) << "Update calls database first message to " << first_added_message_id;
        old_first_db_message_id = first_added_message_id;
        update_state = true;
      }
      if (update_state) {
        save_calls_db_state();
      }
    }
    it->second.total_count = total_count;
    if (next_offset_message_id.is_valid()) {
      it->second.next_offset = PSTRING() << next_offset_message_id.get_server_message_id().get();
    }
    promise.set_value(Unit());
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());

  auto &result = it->second.message_ids;
  CHECK(result.empty());
  MessageId first_added_message_id;
  if (messages.empty()) {
    // messages may be empty because there are no more messages or they can't be found due to global limit
    // anyway pretend that there are no more messages
    first_added_message_id = MessageId::min();
  }
  bool can_be_in_different_dialog =
      top_thread_message_id.is_valid() && td_->dialog_manager_->is_broadcast_channel(dialog_id);
  DialogId real_dialog_id;
  MessageId next_from_message_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  for (auto &message : messages) {
    auto message_id = MessageId::get_message_id(message, false);
    if (message_id.is_valid() && (!next_from_message_id.is_valid() || message_id < next_from_message_id)) {
      next_from_message_id = message_id;
    }
    auto new_message_full_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, "on_get_dialog_messages_search_result");
    if (new_message_full_id == MessageFullId()) {
      total_count--;
      continue;
    }

    if (new_message_full_id.get_dialog_id() != dialog_id) {
      if (!can_be_in_different_dialog) {
        LOG(ERROR) << "Receive " << new_message_full_id << " instead of a message in " << dialog_id;
        total_count--;
        continue;
      } else {
        if (!real_dialog_id.is_valid()) {
          real_dialog_id = new_message_full_id.get_dialog_id();
          found_dialog_messages_dialog_id_[random_id] = real_dialog_id;
        } else if (new_message_full_id.get_dialog_id() != real_dialog_id) {
          LOG(ERROR) << "Receive " << new_message_full_id << " instead of a message in " << real_dialog_id << " or "
                     << dialog_id;
          total_count--;
          continue;
        }
      }
    }

    CHECK(message_id == new_message_full_id.get_message_id());
    if (filter == MessageSearchFilter::UnreadMention && message_id <= d->last_read_all_mentions_message_id &&
        !real_dialog_id.is_valid()) {
      total_count--;
      continue;
    }

    if (filter != MessageSearchFilter::Empty) {
      const Message *m = get_message(new_message_full_id);
      CHECK(m != nullptr);
      auto index_mask = get_message_index_mask(new_message_full_id.get_dialog_id(), m);
      if ((message_search_filter_index_mask(filter) & index_mask) == 0) {
        LOG(INFO) << "Skip " << new_message_full_id << " of unexpected type";
        total_count--;
        continue;
      }
    }

    // TODO check that messages are returned in decreasing message_id order
    if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
      first_added_message_id = message_id;
    }
    result.push_back(message_id);
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  if (query.empty() && !sender_dialog_id.is_valid() && filter != MessageSearchFilter::Empty &&
      !top_thread_message_id.is_valid() && !saved_messages_topic_id.is_valid() && tag.is_empty()) {
    bool from_the_end = !from_message_id.is_valid() ||
                        (d->last_message_id != MessageId() && from_message_id > d->last_message_id) ||
                        from_message_id >= MessageId::max();
    bool update_dialog = false;

    auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
    if (old_message_count != total_count) {
      old_message_count = total_count;
      if (filter == MessageSearchFilter::UnreadMention) {
        d->unread_mention_count = old_message_count;
        update_dialog_mention_notification_count(d);
        send_update_chat_unread_mention_count(d);
      }
      if (filter == MessageSearchFilter::UnreadReaction) {
        d->unread_reaction_count = old_message_count;
        // update_dialog_mention_notification_count(d);
        send_update_chat_unread_reaction_count(d, "on_get_dialog_messages_search_result");
      }
      update_dialog = true;
    }

    auto &old_first_database_message_id = d->first_database_message_id_by_index[message_search_filter_index(filter)];
    if ((from_the_end ||
         (old_first_database_message_id.is_valid() && old_first_database_message_id <= from_message_id)) &&
        (!old_first_database_message_id.is_valid() || first_added_message_id < old_first_database_message_id)) {
      old_first_database_message_id = first_added_message_id;
      update_dialog = true;
    }
    if (update_dialog) {
      on_dialog_updated(dialog_id, "search results");
    }

    if (from_the_end && filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, result.empty() ? MessageId() : result[0]);
    }
  }

  it->second.total_count = total_count;
  it->second.next_from_message_id = next_from_message_id;
  promise.set_value(Unit());
}

void MessagesManager::on_failed_dialog_messages_search(DialogId dialog_id, int64 random_id) {
  if (!dialog_id.is_valid()) {
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());
    found_call_messages_.erase(it);
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  found_dialog_messages_.erase(it);
}

void MessagesManager::on_get_dialog_message_count(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id,
                                                  MessageSearchFilter filter, int32 total_count,
                                                  Promise<int32> &&promise) {
  LOG(INFO) << "Receive " << total_count << " message count in " << dialog_id << " with filter " << filter;
  if (total_count < 0) {
    LOG(ERROR) << "Receive total message count = " << total_count << " in " << dialog_id << " with "
               << saved_messages_topic_id << " and filter " << filter;
    total_count = 0;
  }

  if (saved_messages_topic_id.is_valid()) {
    return promise.set_value(std::move(total_count));
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  CHECK(filter != MessageSearchFilter::Empty);
  CHECK(filter != MessageSearchFilter::UnreadMention);
  CHECK(filter != MessageSearchFilter::UnreadReaction);
  CHECK(filter != MessageSearchFilter::FailedToSend);

  auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
  if (old_message_count != total_count) {
    old_message_count = total_count;
    on_dialog_updated(dialog_id, "on_get_dialog_message_count");
  }

  if (total_count == 0) {
    auto &old_first_database_message_id = d->first_database_message_id_by_index[message_search_filter_index(filter)];
    if (old_first_database_message_id != MessageId::min()) {
      old_first_database_message_id = MessageId::min();
      on_dialog_updated(dialog_id, "on_get_dialog_message_count");
    }
    if (filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, MessageId());
    }
  }
  promise.set_value(std::move(total_count));
}

void MessagesManager::on_get_messages_search_result(const string &query, int32 offset_date, DialogId offset_dialog_id,
                                                    MessageId offset_message_id, int32 limit,
                                                    MessageSearchFilter filter, int32 min_date, int32 max_date,
                                                    int64 random_id, int32 total_count,
                                                    vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                    int32 next_rate, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " found messages";
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());

  auto &result = it->second.message_full_ids;
  CHECK(result.empty());
  int32 last_message_date = 0;
  MessageId last_message_id;
  DialogId last_dialog_id;
  for (auto &message : messages) {
    auto message_date = get_message_date(message);
    auto message_id = MessageId::get_message_id(message, false);
    auto dialog_id = DialogId::get_message_dialog_id(message);
    if (message_date > 0 && message_id.is_valid() && dialog_id.is_valid()) {
      last_message_date = message_date;
      last_message_id = message_id;
      last_dialog_id = dialog_id;
    }

    auto new_message_full_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, "search messages");
    if (new_message_full_id != MessageFullId()) {
      CHECK(dialog_id == new_message_full_id.get_dialog_id());
      result.push_back(new_message_full_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  it->second.total_count = total_count;
  if (!result.empty()) {
    if (next_rate > 0) {
      last_message_date = next_rate;
    }
    it->second.next_offset = PSTRING() << last_message_date << ',' << last_dialog_id.get() << ','
                                       << last_message_id.get_server_message_id().get();
  }
  promise.set_value(Unit());
}

void MessagesManager::on_failed_messages_search(int64 random_id) {
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());
  found_messages_.erase(it);
}

void MessagesManager::on_get_outgoing_document_messages(vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                        Promise<td_api::object_ptr<td_api::foundMessages>> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  FoundMessages found_messages;
  for (auto &message : messages) {
    auto dialog_id = DialogId::get_message_dialog_id(message);
    auto message_full_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel, false,
                                          "on_get_outgoing_document_messages");
    if (message_full_id != MessageFullId()) {
      CHECK(dialog_id == message_full_id.get_dialog_id());
      found_messages.message_full_ids.push_back(message_full_id);
    }
  }
  auto result = get_found_messages_object(found_messages, "on_get_outgoing_document_messages");
  td::remove_if(result->messages_,
                [](const auto &message) { return message->content_->get_id() != td_api::messageDocument::ID; });
  result->total_count_ = narrow_cast<int32>(result->messages_.size());
  promise.set_value(std::move(result));
}

void MessagesManager::on_get_scheduled_server_messages(DialogId dialog_id, uint32 generation,
                                                       vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                       bool is_not_modified) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (generation < d->scheduled_messages_sync_generation) {
    LOG(INFO) << "Ignore scheduled messages with old generation " << generation << " instead of "
              << d->scheduled_messages_sync_generation << " in " << dialog_id;
    return;
  }
  d->scheduled_messages_sync_generation = generation;

  if (is_not_modified) {
    LOG(INFO) << "Scheduled messages are mot modified in " << dialog_id;
    return;
  }

  vector<MessageId> old_message_ids;
  if (d->scheduled_messages != nullptr) {
    for (const auto &it : d->scheduled_messages->scheduled_messages_) {
      old_message_ids.push_back(it.first);
    };
  }
  FlatHashMap<ScheduledServerMessageId, MessageId, ScheduledServerMessageIdHash> old_server_message_ids;
  for (auto &message_id : old_message_ids) {
    if (message_id.is_scheduled_server()) {
      old_server_message_ids[message_id.get_scheduled_server_message_id()] = message_id;
    }
  }

  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  bool has_scheduled_server_messages = false;
  for (auto &message : messages) {
    auto message_dialog_id = DialogId::get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      // server can send messageEmpty for deleted scheduled messages
      auto message_id = MessageId::get_message_id(message, true);
      if (message_id.is_valid() || message_dialog_id.is_valid()) {
        LOG(ERROR) << "Receive " << message_id << " in wrong " << message_dialog_id << " instead of " << dialog_id
                   << ": " << oneline(to_string(message));
      }
      continue;
    }

    auto message_full_id = on_get_message(std::move(message), d->sent_scheduled_messages, is_channel_message, true,
                                          "on_get_scheduled_server_messages");
    auto message_id = message_full_id.get_message_id();
    if (message_id.is_valid_scheduled()) {
      CHECK(message_id.is_scheduled_server());
      old_server_message_ids.erase(message_id.get_scheduled_server_message_id());
      has_scheduled_server_messages = true;
    }
  }
  on_update_dialog_has_scheduled_server_messages(dialog_id, has_scheduled_server_messages);

  for (const auto &it : old_server_message_ids) {
    auto message_id = it.second;
    auto message = do_delete_scheduled_message(d, message_id, true, "on_get_scheduled_server_messages");
    CHECK(message != nullptr);
    send_update_delete_messages(dialog_id, {message->message_id.get()}, true);
  }

  send_update_chat_has_scheduled_messages(d, false);
}

void MessagesManager::on_get_recent_locations(DialogId dialog_id, int32 limit, int32 total_count,
                                              vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                              Promise<td_api::object_ptr<td_api::messages>> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " recent locations in " << dialog_id;
  vector<MessageId> result;
  for (auto &message : messages) {
    auto new_message_full_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, "get recent locations");
    if (new_message_full_id != MessageFullId()) {
      if (new_message_full_id.get_dialog_id() != dialog_id) {
        LOG(ERROR) << "Receive " << new_message_full_id << " instead of a message in " << dialog_id;
        total_count--;
        continue;
      }
      auto m = get_message(new_message_full_id);
      CHECK(m != nullptr);
      if (m->content->get_type() != MessageContentType::LiveLocation) {
        LOG(ERROR) << "Receive a message of wrong type " << m->content->get_type() << " in on_get_recent_locations in "
                   << dialog_id;
        total_count--;
        continue;
      }

      result.push_back(m->message_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  promise.set_value(get_messages_object(total_count, dialog_id, result, true, "on_get_recent_locations"));
}

void MessagesManager::delete_messages_from_updates(const vector<MessageId> &message_ids, bool is_permanent) {
  FlatHashMap<DialogId, vector<int64>, DialogIdHash> deleted_message_ids;
  FlatHashMap<DialogId, bool, DialogIdHash> need_update_dialog_pos;
  vector<unique_ptr<Message>> deleted_messages;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || !message_id.is_server()) {
      LOG(ERROR) << "Incoming update tries to delete " << message_id;
      continue;
    }

    Dialog *d = get_dialog_by_message_id(message_id);
    if (d != nullptr) {
      auto message = delete_message(d, message_id, is_permanent, &need_update_dialog_pos[d->dialog_id],
                                    "delete_messages_from_updates");
      CHECK(message != nullptr);
      LOG_CHECK(message->message_id == message_id) << message_id << " " << message->message_id << " " << d->dialog_id;
      deleted_message_ids[d->dialog_id].push_back(message->message_id.get());
      deleted_messages.push_back(std::move(message));
    }
    if (last_clear_history_message_id_to_dialog_id_.count(message_id)) {
      d = get_dialog(last_clear_history_message_id_to_dialog_id_[message_id]);
      CHECK(d != nullptr);
      auto message = delete_message(d, message_id, is_permanent, &need_update_dialog_pos[d->dialog_id],
                                    "delete_messages_from_updates");
      CHECK(message == nullptr);
    }
  }
  if (deleted_messages.size() >= MIN_DELETED_ASYNCHRONOUSLY_MESSAGES) {
    Scheduler::instance()->destroy_on_scheduler(G()->get_gc_scheduler_id(), deleted_messages);
  }
  for (auto &it : need_update_dialog_pos) {
    if (it.second) {
      auto dialog_id = it.first;
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      send_update_chat_last_message(d, "delete_messages_from_updates");
    }
  }
  for (auto &it : deleted_message_ids) {
    auto dialog_id = it.first;
    send_update_delete_messages(dialog_id, std::move(it.second), is_permanent);
  }
}

void MessagesManager::delete_dialog_messages(DialogId dialog_id, const vector<MessageId> &message_ids,
                                             bool force_update_for_not_found_messages, const char *source) {
  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_messages");
  if (d == nullptr) {
    LOG(INFO) << "Ignore deleteChannelMessages for unknown " << dialog_id << " from " << source;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    return;
  }

  delete_dialog_messages(d, message_ids, force_update_for_not_found_messages, source);
}

void MessagesManager::delete_dialog_messages(Dialog *d, const vector<MessageId> &message_ids,
                                             bool force_update_for_not_found_messages, const char *source) {
  vector<unique_ptr<Message>> deleted_messages;
  vector<int64> deleted_message_ids;
  bool need_update_dialog_pos = false;
  bool need_update_chat_has_scheduled_messages = false;
  for (auto message_id : message_ids) {
    CHECK(message_id.is_valid() || message_id.is_valid_scheduled());

    bool force_update = force_update_for_not_found_messages && !is_deleted_message(d, message_id);
    auto message = delete_message(d, message_id, true, &need_update_dialog_pos, source);
    if (message == nullptr) {
      if (force_update) {
        deleted_message_ids.push_back(message_id.get());
      }
    } else {
      need_update_chat_has_scheduled_messages |= message->message_id.is_scheduled();
      deleted_message_ids.push_back(message->message_id.get());
      deleted_messages.push_back(std::move(message));
    }
  }
  if (deleted_messages.size() >= MIN_DELETED_ASYNCHRONOUSLY_MESSAGES) {
    Scheduler::instance()->destroy_on_scheduler(G()->get_gc_scheduler_id(), deleted_messages);
  }
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, source);
  }
  send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), true);

  if (need_update_chat_has_scheduled_messages) {
    send_update_chat_has_scheduled_messages(d, true);
  }
}

void MessagesManager::update_dialog_pinned_messages_from_updates(DialogId dialog_id,
                                                                 const vector<MessageId> &message_ids, bool is_pin) {
  Dialog *d = get_dialog_force(dialog_id, "update_dialog_pinned_messages_from_updates");
  if (d == nullptr) {
    LOG(INFO) << "Ignore updatePinnedMessages for unknown " << dialog_id;
    return;
  }

  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || (!message_id.is_server() && dialog_id.get_type() != DialogType::SecretChat)) {
      LOG(ERROR) << "Incoming update tries to pin/unpin " << message_id << " in " << dialog_id;
      continue;
    }

    Message *m = get_message_force(d, message_id, "update_dialog_pinned_messages_from_updates");
    if (m != nullptr && update_message_is_pinned(d, m, is_pin, "update_dialog_pinned_messages_from_updates")) {
      on_message_changed(d, m, true, "update_dialog_pinned_messages_from_updates");
    }
  }
}

bool MessagesManager::update_message_is_pinned(Dialog *d, Message *m, bool is_pinned, const char *source) {
  CHECK(m != nullptr);
  CHECK(!m->message_id.is_scheduled());
  if (m->is_pinned == is_pinned) {
    return false;
  }

  LOG(INFO) << "Update message is_pinned of " << m->message_id << " in " << d->dialog_id << " to " << is_pinned
            << " from " << source;
  auto old_index_mask = get_message_index_mask(d->dialog_id, m);
  m->is_pinned = is_pinned;
  auto new_index_mask = get_message_index_mask(d->dialog_id, m);
  update_message_count_by_index(d, -1, old_index_mask & ~new_index_mask);
  update_message_count_by_index(d, +1, new_index_mask & ~old_index_mask);

  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateMessageIsPinned>(
                   get_chat_id_object(d->dialog_id, "updateMessageIsPinned"), m->message_id.get(), is_pinned));
  if (is_pinned) {
    if (d->is_last_pinned_message_id_inited && m->message_id > d->last_pinned_message_id) {
      set_dialog_last_pinned_message_id(d, m->message_id);
    }
  } else {
    if (d->is_last_pinned_message_id_inited && m->message_id == d->last_pinned_message_id) {
      if (!td_->auth_manager_->is_bot() &&
          d->message_count_by_index[message_search_filter_index(MessageSearchFilter::Pinned)] == 0) {
        set_dialog_last_pinned_message_id(d, MessageId());
      } else {
        drop_dialog_last_pinned_message_id(d);
      }
    }
  }
  return true;
}

string MessagesManager::get_message_search_text(const Message *m) const {
  if (m->is_content_secret) {
    return string();
  }
  return get_message_content_search_text(td_, m->content.get());
}

bool MessagesManager::can_forward_message(DialogId from_dialog_id, const Message *m) {
  if (m == nullptr) {
    return false;
  }
  if (!m->ttl.is_empty()) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  switch (from_dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }

  return can_forward_message_content(m->content.get());
}

bool MessagesManager::can_save_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr || m->noforwards || m->is_content_secret) {
    return false;
  }
  return !td_->dialog_manager_->get_dialog_has_protected_content(dialog_id);
}

bool MessagesManager::can_get_message_statistics(MessageFullId message_full_id) {
  return can_get_message_statistics(message_full_id.get_dialog_id(),
                                    get_message_force(message_full_id, "can_get_message_statistics"));
}

bool MessagesManager::can_get_message_statistics(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot() || dialog_id.get_type() != DialogType::Channel) {
    return false;
  }
  if (m == nullptr || m->message_id.is_scheduled() || !m->message_id.is_server() || m->view_count == 0 ||
      m->had_forward_info || (m->forward_info != nullptr && m->forward_info->get_origin().is_channel_post())) {
    return false;
  }
  return td_->chat_manager_->can_get_channel_message_statistics(dialog_id.get_channel_id());
}

bool MessagesManager::can_delete_channel_message(const DialogParticipantStatus &status, const Message *m, bool is_bot) {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local() || m->message_id.is_yet_unsent()) {
    return true;
  }
  if (m->message_id.is_scheduled()) {
    if (m->is_channel_post) {
      return status.can_post_messages();
    }
    return true;
  }

  if (is_bot && G()->unix_time() >= m->date + 2 * 86400) {
    // bots can't delete messages older than 2 days
    return false;
  }

  CHECK(m->message_id.is_server());
  if (m->message_id.get_server_message_id().get() == 1) {
    return false;
  }
  auto content_type = m->content->get_type();
  if (content_type == MessageContentType::ChannelMigrateFrom || content_type == MessageContentType::ChannelCreate ||
      content_type == MessageContentType::TopicCreate) {
    return false;
  }

  if (status.can_delete_messages()) {
    return true;
  }

  if (!m->is_outgoing) {
    return false;
  }

  if (m->is_channel_post || is_service_message_content(content_type)) {
    return status.can_post_messages();
  }

  return true;
}

bool MessagesManager::can_delete_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local() || m->message_id.is_yet_unsent()) {
    return true;
  }
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return true;
    case DialogType::Chat:
      return true;
    case DialogType::Channel: {
      auto dialog_status = td_->chat_manager_->get_channel_permissions(dialog_id.get_channel_id());
      return can_delete_channel_message(dialog_status, m, td_->auth_manager_->is_bot());
    }
    case DialogType::SecretChat:
      return true;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

bool MessagesManager::can_revoke_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local()) {
    return false;
  }
  if (dialog_id == td_->dialog_manager_->get_my_dialog_id()) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  if (m->message_id.is_yet_unsent()) {
    return true;
  }
  CHECK(m->message_id.is_server());

  const int32 DEFAULT_REVOKE_TIME_LIMIT = td_->auth_manager_->is_bot() ? 2 * 86400 : std::numeric_limits<int32>::max();
  auto content_type = m->content->get_type();
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      bool can_revoke_incoming = td_->option_manager_->get_option_boolean("revoke_pm_inbox", true);
      int64 revoke_time_limit =
          td_->option_manager_->get_option_integer("revoke_pm_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      if (G()->unix_time() - m->date < 86400 && content_type == MessageContentType::Dice) {
        return false;
      }
      return ((m->is_outgoing && !is_service_message_content(content_type)) ||
              (can_revoke_incoming && content_type != MessageContentType::ScreenshotTaken)) &&
             G()->unix_time() - m->date <= revoke_time_limit;
    }
    case DialogType::Chat: {
      bool is_appointed_administrator = td_->chat_manager_->is_appointed_chat_administrator(dialog_id.get_chat_id());
      int64 revoke_time_limit =
          td_->option_manager_->get_option_integer("revoke_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      return ((m->is_outgoing && !is_service_message_content(content_type)) || is_appointed_administrator) &&
             G()->unix_time() - m->date <= revoke_time_limit;
    }
    case DialogType::Channel:
      return true;  // any server message that can be deleted will be deleted for all participants
    case DialogType::SecretChat:
      // all non-service messages will be deleted for everyone if secret chat is active
      return td_->user_manager_->get_secret_chat_state(dialog_id.get_secret_chat_id()) == SecretChatState::Active &&
             !is_service_message_content(content_type);
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::delete_messages(DialogId dialog_id, const vector<MessageId> &input_message_ids, bool revoke,
                                      Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  Dialog *d = get_dialog_force(dialog_id, "delete_messages");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat is not found"));
  }

  if (input_message_ids.empty()) {
    return promise.set_value(Unit());
  }

  auto dialog_type = dialog_id.get_type();
  bool is_secret = dialog_type == DialogType::SecretChat;

  vector<MessageId> message_ids;
  message_ids.reserve(input_message_ids.size());
  vector<MessageId> deleted_server_message_ids;

  vector<MessageId> deleted_scheduled_server_message_ids;
  for (auto message_id : input_message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      return promise.set_error(Status::Error(400, "Invalid message identifier"));
    }

    message_id = get_persistent_message_id(d, message_id);
    message_ids.push_back(message_id);
    auto m = get_message_force(d, message_id, "delete_messages");
    if (m != nullptr) {
      if (m->message_id.is_scheduled()) {
        if (m->message_id.is_scheduled_server()) {
          deleted_scheduled_server_message_ids.push_back(m->message_id);
        }
      } else {
        if (m->message_id.is_server() || is_secret) {
          deleted_server_message_ids.push_back(m->message_id);
        }
      }
    }
  }

  bool is_bot = td_->auth_manager_->is_bot();
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    if (!can_delete_message(dialog_id, m)) {
      return promise.set_error(Status::Error(400, "Message can't be deleted"));
    }
    if (is_bot && !message_id.is_scheduled() && message_id.is_server() && !can_revoke_message(dialog_id, m)) {
      return promise.set_error(Status::Error(400, "Message can't be deleted for everyone"));
    }
  }

  MultiPromiseActorSafe mpas{"DeleteMessagesMultiPromiseActor"};
  mpas.add_promise(std::move(promise));

  auto lock = mpas.get_promise();
  delete_messages_on_server(dialog_id, std::move(deleted_server_message_ids), revoke, 0, mpas.get_promise());
  delete_scheduled_messages_on_server(dialog_id, std::move(deleted_scheduled_server_message_ids), 0,
                                      mpas.get_promise());
  lock.set_value(Unit());

  delete_dialog_messages(d, message_ids, false, DELETE_MESSAGE_USER_REQUEST_SOURCE);
}

void MessagesManager::erase_delete_messages_log_event(uint64 log_event_id) {
  if (!G()->close_flag()) {
    binlog_erase(G()->td_db()->get_binlog(), log_event_id);
  }
}

void MessagesManager::delete_sent_message_on_server(DialogId dialog_id, MessageId message_id,
                                                    MessageId old_message_id) {
  // this would be a no-op, because replies have already been removed in cancel_send_message_query
  // update_reply_to_message_id(dialog_id, old_message_id, message_id, false, "delete_sent_message_on_server");

  // being sent message was deleted by the user or is in an inaccessible channel
  // don't need to send an update to the user, because the message has already been deleted
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    LOG(INFO) << "Ignore sent " << message_id << " in inaccessible " << dialog_id;
    return;
  }

  LOG(INFO) << "Delete already deleted sent " << message_id << " in " << dialog_id << " from server";
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (get_message_force(d, message_id, "delete_sent_message_on_server") != nullptr) {
    delete_messages(dialog_id, {message_id}, true, Auto());
  } else {
    if (message_id.is_valid()) {
      CHECK(message_id.is_server());
      delete_messages_on_server(dialog_id, {message_id}, true, 0, Auto());
    } else {
      CHECK(message_id.is_scheduled_server());
      delete_scheduled_messages_on_server(dialog_id, {message_id}, 0, Auto());
    }

    bool need_update_dialog_pos = false;
    auto message = delete_message(d, message_id, true, &need_update_dialog_pos, "delete_sent_message_on_server");
    CHECK(message == nullptr);
    if (need_update_dialog_pos) {  // last_clear_history_message_id might be removed
      update_dialog_pos(d, "delete_sent_message_on_server");
    }
  }
}

class MessagesManager::DeleteMessagesOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_messages_on_server_log_event(DialogId dialog_id,
                                                                 const vector<MessageId> &message_ids, bool revoke) {
  DeleteMessagesOnServerLogEvent log_event{dialog_id, message_ids, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_messages_on_server(DialogId dialog_id, vector<MessageId> message_ids, bool revoke,
                                                uint64 log_event_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << (revoke ? "Revoke " : "Delete ") << format::as_array(message_ids) << " in " << dialog_id
            << " from server";

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_delete_messages_on_server_log_event(dialog_id, message_ids, revoke);
  }

  MultiPromiseActorSafe mpas{"DeleteMessagesOnServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));
  if (log_event_id != 0) {
    mpas.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), log_event_id](Unit) {
      send_closure(actor_id, &MessagesManager::erase_delete_messages_log_event, log_event_id);
    }));
  }
  auto lock = mpas.get_promise();
  auto dialog_type = dialog_id.get_type();
  switch (dialog_type) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel: {
      auto server_message_ids = MessageId::get_server_message_ids(message_ids);
      const size_t MAX_SLICE_SIZE = 100;  // server side limit
      for (size_t i = 0; i < server_message_ids.size(); i += MAX_SLICE_SIZE) {
        auto end_i = i + MAX_SLICE_SIZE;
        auto end = end_i < server_message_ids.size() ? server_message_ids.begin() + end_i : server_message_ids.end();
        if (dialog_type != DialogType::Channel) {
          td_->create_handler<DeleteMessagesQuery>(mpas.get_promise())
              ->send(dialog_id, {server_message_ids.begin() + i, end}, revoke);
        } else {
          td_->create_handler<DeleteChannelMessagesQuery>(mpas.get_promise())
              ->send(dialog_id.get_channel_id(), {server_message_ids.begin() + i, end});
        }
      }
      break;
    }
    case DialogType::SecretChat: {
      vector<int64> random_ids;
      auto d = get_dialog_force(dialog_id, "delete_messages_on_server");
      CHECK(d != nullptr);
      for (auto &message_id : message_ids) {
        auto *m = get_message(d, message_id);
        if (m != nullptr) {
          random_ids.push_back(m->random_id);
        }
      }
      if (!random_ids.empty()) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_messages, dialog_id.get_secret_chat_id(),
                     std::move(random_ids), mpas.get_promise());
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
  lock.set_value(Unit());
}

class MessagesManager::DeleteScheduledMessagesOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_scheduled_messages_on_server_log_event(DialogId dialog_id,
                                                                           const vector<MessageId> &message_ids) {
  DeleteScheduledMessagesOnServerLogEvent log_event{dialog_id, message_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteScheduledMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_scheduled_messages_on_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                          uint64 log_event_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << "Delete " << format::as_array(message_ids) << " in " << dialog_id << " from server";

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_delete_scheduled_messages_on_server_log_event(dialog_id, message_ids);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  td_->create_handler<DeleteScheduledMessagesQuery>(std::move(promise))->send(dialog_id, std::move(message_ids));
}

void MessagesManager::on_failed_message_deletion(DialogId dialog_id, const vector<int32> &server_message_ids) {
  if (G()->close_flag()) {
    return;
  }
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  vector<MessageFullId> message_full_ids;
  for (auto &server_message_id : server_message_ids) {
    auto message_id = MessageId(ServerMessageId(server_message_id));
    d->deleted_message_ids.erase(message_id);
    message_full_ids.emplace_back(dialog_id, message_id);
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return;
  }
  get_messages_from_server(std::move(message_full_ids), Promise<Unit>(), "on_failed_message_deletion");
}

void MessagesManager::on_failed_scheduled_message_deletion(DialogId dialog_id, const vector<MessageId> &message_ids) {
  if (G()->close_flag()) {
    return;
  }
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (d->scheduled_messages != nullptr) {
    for (auto &message_id : message_ids) {
      d->scheduled_messages->deleted_scheduled_server_message_ids_.erase(message_id.get_scheduled_server_message_id());
    }
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return;
  }
  if (td_->dialog_manager_->is_broadcast_channel(dialog_id) &&
      !td_->chat_manager_->get_channel_status(dialog_id.get_channel_id()).can_post_messages()) {
    return;
  }
  load_dialog_scheduled_messages(dialog_id, false, 0, Promise<Unit>());
}

MessagesManager::CanDeleteDialog MessagesManager::can_delete_dialog(const Dialog *d) const {
  if (is_dialog_sponsored(d)) {
    auto chat_source = sponsored_dialog_source_.get_chat_source_object();
    if (chat_source != nullptr) {
      switch (chat_source->get_id()) {
        case td_api::chatSourcePublicServiceAnnouncement::ID:
          // can delete for self (but only while removing from dialog list)
          return {true, false};
        default:
          return {false, false};
      }
    }
  }
  if (td_->auth_manager_->is_bot() || !td_->dialog_manager_->have_input_peer(d->dialog_id, true, AccessRights::Read)) {
    return {false, false};
  }

  switch (d->dialog_id.get_type()) {
    case DialogType::User:
      if (d->dialog_id == td_->dialog_manager_->get_my_dialog_id() ||
          td_->user_manager_->is_user_deleted(d->dialog_id.get_user_id()) ||
          td_->user_manager_->is_user_bot(d->dialog_id.get_user_id())) {
        return {true, false};
      }
      return {true, td_->option_manager_->get_option_boolean("revoke_pm_inbox", true)};
    case DialogType::Chat:
      // chats can be deleted only for self and can be deleted for everyone by their creator
      return {true, td_->chat_manager_->get_chat_status(d->dialog_id.get_chat_id()).is_creator()};
    case DialogType::Channel: {
      // private non-forum joined supergroups can be deleted for self
      auto channel_id = d->dialog_id.get_channel_id();
      return {!td_->chat_manager_->is_broadcast_channel(channel_id) &&
                  !td_->chat_manager_->is_channel_public(channel_id) &&
                  !td_->chat_manager_->is_forum_channel(channel_id) &&
                  td_->chat_manager_->get_channel_status(channel_id).is_member(),
              td_->chat_manager_->get_channel_can_be_deleted(channel_id)};
    }
    case DialogType::SecretChat:
      if (td_->user_manager_->get_secret_chat_state(d->dialog_id.get_secret_chat_id()) == SecretChatState::Closed) {
        // in a closed secret chats there is no way to delete messages for both users
        return {true, false};
      }
      // active secret chats can be deleted only for both users
      return {false, true};
    case DialogType::None:
    default:
      UNREACHABLE();
      return {false, false};
  }
}

void MessagesManager::delete_dialog_history(DialogId dialog_id, bool remove_from_dialog_list, bool revoke,
                                            Promise<Unit> &&promise) {
  LOG(INFO) << "Receive deleteChatHistory request to delete all messages in " << dialog_id
            << ", remove_from_chat_list is " << remove_from_dialog_list << ", revoke is " << revoke;

  TRY_RESULT_PROMISE(promise, d, check_dialog_access(dialog_id, true, AccessRights::Read, "delete_dialog_history"));

  if (is_dialog_sponsored(d)) {
    auto chat_source = sponsored_dialog_source_.get_chat_source_object();
    if (chat_source == nullptr || chat_source->get_id() != td_api::chatSourcePublicServiceAnnouncement::ID) {
      return promise.set_error(Status::Error(400, "Can't delete the chat"));
    }
    if (!remove_from_dialog_list) {
      return promise.set_error(Status::Error(400, "Can't delete chat history without removing the chat"));
    }

    removed_sponsored_dialog_id_ = dialog_id;
    remove_sponsored_dialog();

    if (dialog_id.get_type() != DialogType::SecretChat) {
      td_->create_handler<HidePromoDataQuery>()->send(dialog_id);
    }
    promise.set_value(Unit());
    return;
  }

  auto can_delete = can_delete_dialog(d);
  if (revoke) {
    if (!can_delete.for_all_users_) {
      if (!can_delete.for_self_) {
        return promise.set_error(Status::Error(400, "Chat history can't be deleted"));
      }

      LOG(INFO) << "Can't delete history of " << dialog_id << " for everyone; delete it only for self";
      revoke = false;
    }
  } else {
    if (!can_delete.for_self_) {
      return promise.set_error(
          Status::Error(400, PSLICE() << "Can't delete history of " << dialog_id << " only for self"));
    }
  }

  auto last_new_message_id = d->last_new_message_id;
  if (dialog_id.get_type() != DialogType::SecretChat && last_new_message_id == MessageId()) {
    // TODO get dialog from the server and delete history from last message identifier
  }

  bool allow_error = d->messages.empty();
  auto old_order = d->order;

  delete_all_dialog_messages(d, remove_from_dialog_list, true);

  if (last_new_message_id.is_valid() && last_new_message_id == d->max_unavailable_message_id && !revoke &&
      !(old_order != DEFAULT_ORDER && remove_from_dialog_list)) {
    // history has already been cleared, nothing to do
    promise.set_value(Unit());
    return;
  }

  set_dialog_max_unavailable_message_id(dialog_id, last_new_message_id, false, "delete_dialog_history");

  delete_dialog_history_on_server(dialog_id, last_new_message_id, remove_from_dialog_list, revoke, allow_error, 0,
                                  std::move(promise));
}

class MessagesManager::DeleteDialogHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;
  bool remove_from_dialog_list_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(remove_from_dialog_list_);
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(remove_from_dialog_list_);
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

uint64 MessagesManager::save_delete_dialog_history_on_server_log_event(DialogId dialog_id, MessageId max_message_id,
                                                                       bool remove_from_dialog_list, bool revoke) {
  DeleteDialogHistoryOnServerLogEvent log_event{dialog_id, max_message_id, remove_from_dialog_list, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteDialogHistoryOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_dialog_history_on_server(DialogId dialog_id, MessageId max_message_id,
                                                      bool remove_from_dialog_list, bool revoke, bool allow_error,
                                                      uint64 log_event_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Delete history in " << dialog_id << " up to " << max_message_id << " from server";

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id =
        save_delete_dialog_history_on_server_log_event(dialog_id, max_message_id, remove_from_dialog_list, revoke);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat: {
      AffectedHistoryQuery query = [td = td_, max_message_id, remove_from_dialog_list, revoke](
                                       DialogId dialog_id, Promise<AffectedHistory> &&query_promise) {
        td->create_handler<DeleteHistoryQuery>(std::move(query_promise))
            ->send(dialog_id, max_message_id, remove_from_dialog_list, revoke);
      };
      run_affected_history_query_until_complete(dialog_id, std::move(query), false, std::move(promise));
      break;
    }
    case DialogType::Channel:
      td_->create_handler<DeleteChannelHistoryQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), max_message_id, allow_error, revoke);
      break;
    case DialogType::SecretChat:
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_all_messages,
                   dialog_id.get_secret_chat_id(), std::move(promise));
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::delete_topic_history(DialogId dialog_id, MessageId top_thread_message_id,
                                           Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(
      promise, td_->dialog_manager_->check_dialog_access(dialog_id, false, AccessRights::Read, "delete_topic_history"));

  // auto old_order = d->order;
  // delete_all_dialog_topic_messages(d, top_thread_message_id);

  delete_topic_history_on_server(dialog_id, top_thread_message_id, 0, std::move(promise));
}

class MessagesManager::DeleteTopicHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId top_thread_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    END_STORE_FLAGS();
    td::store(dialog_id_, storer);
    td::store(top_thread_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    END_PARSE_FLAGS();
    td::parse(dialog_id_, parser);
    td::parse(top_thread_message_id_, parser);
  }
};

uint64 MessagesManager::save_delete_topic_history_on_server_log_event(DialogId dialog_id,
                                                                      MessageId top_thread_message_id) {
  DeleteTopicHistoryOnServerLogEvent log_event{dialog_id, top_thread_message_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteTopicHistoryOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_topic_history_on_server(DialogId dialog_id, MessageId top_thread_message_id,
                                                     uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_delete_topic_history_on_server_log_event(dialog_id, top_thread_message_id);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  AffectedHistoryQuery query = [td = td_, top_thread_message_id](DialogId dialog_id,
                                                                 Promise<AffectedHistory> &&query_promise) {
    td->create_handler<DeleteTopicHistoryQuery>(std::move(query_promise))->send(dialog_id, top_thread_message_id);
  };
  run_affected_history_query_until_complete(dialog_id, std::move(query), true, std::move(promise));
}

void MessagesManager::delete_all_call_messages(bool revoke, Promise<Unit> &&promise) {
  delete_all_call_messages_on_server(revoke, 0, std::move(promise));
}

class MessagesManager::DeleteAllCallMessagesOnServerLogEvent {
 public:
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();
  }
};

uint64 MessagesManager::save_delete_all_call_messages_on_server_log_event(bool revoke) {
  DeleteAllCallMessagesOnServerLogEvent log_event{revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteAllCallMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_all_call_messages_on_server(bool revoke, uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0) {
    log_event_id = save_delete_all_call_messages_on_server_log_event(revoke);
  }

  AffectedHistoryQuery query = [td = td_, revoke](DialogId /*dialog_id*/, Promise<AffectedHistory> &&query_promise) {
    td->create_handler<DeletePhoneCallHistoryQuery>(std::move(query_promise))->send(revoke);
  };
  run_affected_history_query_until_complete(DialogId(), std::move(query), false,
                                            get_erase_log_event_promise(log_event_id, std::move(promise)));
}

vector<MessageId> MessagesManager::find_dialog_messages(const Dialog *d,
                                                        const std::function<bool(const Message *)> &condition) {
  vector<MessageId> message_ids;
  d->messages.foreach([&](const MessageId &message_id, const unique_ptr<Message> &message) {
    CHECK(message_id == message->message_id);
    if (condition(message.get())) {
      message_ids.push_back(message_id);
    }
  });
  return message_ids;
}

vector<MessageId> MessagesManager::find_unloadable_messages(const Dialog *d, int32 unload_before_date,
                                                            bool &has_left_to_unload_messages) const {
  vector<MessageId> message_ids;
  for (auto it = d->message_lru_list.next; it != &d->message_lru_list; it = it->next) {
    if (message_ids.size() >= MAX_UNLOADED_MESSAGES) {
      has_left_to_unload_messages = true;
      break;
    }
    const auto *m = static_cast<const Message *>(it);
    if (can_unload_message(d, m)) {
      if (m->last_access_date <= unload_before_date) {
        message_ids.push_back(m->message_id);
      } else {
        has_left_to_unload_messages = true;
      }
    }
    if (has_left_to_unload_messages && m->date > unload_before_date) {
      // we aren't interested in unloading too new messages
      break;
    }
  }
  return message_ids;
}

void MessagesManager::delete_dialog_messages_by_sender(DialogId dialog_id, DialogId sender_dialog_id,
                                                       Promise<Unit> &&promise) {
  bool is_bot = td_->auth_manager_->is_bot();
  CHECK(!is_bot);

  TRY_RESULT_PROMISE(promise, d,
                     check_dialog_access(dialog_id, true, AccessRights::Write, "delete_dialog_messages_by_sender"));

  if (!td_->dialog_manager_->have_input_peer(sender_dialog_id, false, AccessRights::Know)) {
    return promise.set_error(Status::Error(400, "Message sender not found"));
  }

  ChannelId channel_id;
  DialogParticipantStatus channel_status = DialogParticipantStatus::Left();
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      return promise.set_error(
          Status::Error(400, "All messages from a sender can be deleted only in supergroup chats"));
    case DialogType::Channel: {
      channel_id = dialog_id.get_channel_id();
      if (!td_->chat_manager_->is_megagroup_channel(channel_id)) {
        return promise.set_error(Status::Error(400, "The method is available only in supergroup chats"));
      }
      channel_status = td_->chat_manager_->get_channel_permissions(channel_id);
      if (!channel_status.can_delete_messages()) {
        return promise.set_error(Status::Error(400, "Need delete messages administator right in the supergroup chat"));
      }
      channel_id = dialog_id.get_channel_id();
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
  CHECK(channel_id.is_valid());

  if (sender_dialog_id.get_type() == DialogType::SecretChat) {
    return promise.set_value(Unit());
  }

  if (G()->use_message_database()) {
    LOG(INFO) << "Delete all messages from " << sender_dialog_id << " in " << dialog_id << " from database";
    G()->td_db()->get_message_db_async()->delete_dialog_messages_by_sender(dialog_id, sender_dialog_id,
                                                                           Auto());  // TODO Promise
  }

  vector<MessageId> message_ids = find_dialog_messages(d, [sender_dialog_id, channel_status, is_bot](const Message *m) {
    return sender_dialog_id == get_message_sender(m) && can_delete_channel_message(channel_status, m, is_bot);
  });

  delete_dialog_messages(d, message_ids, false, DELETE_MESSAGE_USER_REQUEST_SOURCE);

  delete_all_channel_messages_by_sender_on_server(channel_id, sender_dialog_id, 0, std::move(promise));
}

class MessagesManager::DeleteAllChannelMessagesFromSenderOnServerLogEvent {
 public:
  ChannelId channel_id_;
  DialogId sender_dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(channel_id_, storer);
    td::store(sender_dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(channel_id_, parser);
    if (parser.version() >= static_cast<int32>(Version::AddKeyboardButtonFlags)) {
      td::parse(sender_dialog_id_, parser);
    } else {
      UserId user_id;
      td::parse(user_id, parser);
      sender_dialog_id_ = DialogId(user_id);
    }
  }
};

uint64 MessagesManager::save_delete_all_channel_messages_by_sender_on_server_log_event(ChannelId channel_id,
                                                                                       DialogId sender_dialog_id) {
  DeleteAllChannelMessagesFromSenderOnServerLogEvent log_event{channel_id, sender_dialog_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteAllChannelMessagesFromSenderOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_all_channel_messages_by_sender_on_server(ChannelId channel_id, DialogId sender_dialog_id,
                                                                      uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_chat_info_database()) {
    log_event_id = save_delete_all_channel_messages_by_sender_on_server_log_event(channel_id, sender_dialog_id);
  }

  AffectedHistoryQuery query = [td = td_, sender_dialog_id](DialogId dialog_id,
                                                            Promise<AffectedHistory> &&query_promise) {
    td->create_handler<DeleteParticipantHistoryQuery>(std::move(query_promise))
        ->send(dialog_id.get_channel_id(), sender_dialog_id);
  };
  run_affected_history_query_until_complete(DialogId(channel_id), std::move(query),
                                            sender_dialog_id.get_type() != DialogType::User,
                                            get_erase_log_event_promise(log_event_id, std::move(promise)));
}

Status MessagesManager::fix_delete_message_min_max_dates(int32 &min_date, int32 &max_date) {
  if (min_date > max_date) {
    return Status::Error(400, "Wrong date interval specified");
  }

  const int32 telegram_launch_date = 1376438400;
  if (max_date < telegram_launch_date) {
    max_date = 0;
    min_date = 0;
    return Status::OK();
  }
  if (min_date < telegram_launch_date) {
    min_date = telegram_launch_date;
  }

  auto current_date = max(G()->unix_time(), 1635000000);
  if (min_date >= current_date - 30) {
    max_date = 0;
    min_date = 0;
    return Status::OK();
  }
  if (max_date >= current_date - 30) {
    max_date = current_date - 31;
  }
  CHECK(min_date <= max_date);
  return Status::OK();
}

void MessagesManager::delete_dialog_messages_by_date(DialogId dialog_id, int32 min_date, int32 max_date, bool revoke,
                                                     Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  TRY_RESULT_PROMISE(promise, d,
                     check_dialog_access(dialog_id, true, AccessRights::Read, "delete_dialog_messages_by_date"));
  TRY_STATUS_PROMISE(promise, fix_delete_message_min_max_dates(min_date, max_date));
  if (max_date == 0) {
    return promise.set_value(Unit());
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      if (revoke) {
        return promise.set_error(Status::Error(400, "Bulk message revocation is unsupported in basic group chats"));
      }
      break;
    case DialogType::Channel:
      return promise.set_error(Status::Error(400, "Bulk message deletion is unsupported in supergroup chats"));
    case DialogType::SecretChat:
      return promise.set_error(Status::Error(400, "Bulk message deletion is unsupported in secret chats"));
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }

  // TODO delete in database by dates

  auto message_ids = d->ordered_messages.find_messages_by_date(min_date, max_date, get_get_message_date(d));

  delete_dialog_messages(d, message_ids, false, DELETE_MESSAGE_USER_REQUEST_SOURCE);

  delete_dialog_messages_by_date_on_server(dialog_id, min_date, max_date, revoke, 0, std::move(promise));
}

class MessagesManager::DeleteDialogMessagesByDateOnServerLogEvent {
 public:
  DialogId dialog_id_;
  int32 min_date_;
  int32 max_date_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();
    td::store(dialog_id_, storer);
    td::store(min_date_, storer);
    td::store(max_date_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();
    td::parse(dialog_id_, parser);
    td::parse(min_date_, parser);
    td::parse(max_date_, parser);
  }
};

uint64 MessagesManager::save_delete_dialog_messages_by_date_on_server_log_event(DialogId dialog_id, int32 min_date,
                                                                                int32 max_date, bool revoke) {
  DeleteDialogMessagesByDateOnServerLogEvent log_event{dialog_id, min_date, max_date, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteDialogMessagesByDateOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_dialog_messages_by_date_on_server(DialogId dialog_id, int32 min_date, int32 max_date,
                                                               bool revoke, uint64 log_event_id,
                                                               Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_chat_info_database()) {
    log_event_id = save_delete_dialog_messages_by_date_on_server_log_event(dialog_id, min_date, max_date, revoke);
  }

  AffectedHistoryQuery query = [td = td_, min_date, max_date, revoke](DialogId dialog_id,
                                                                      Promise<AffectedHistory> &&query_promise) {
    td->create_handler<DeleteMessagesByDateQuery>(std::move(query_promise))
        ->send(dialog_id, min_date, max_date, revoke);
  };
  run_affected_history_query_until_complete(dialog_id, std::move(query), true,
                                            get_erase_log_event_promise(log_event_id, std::move(promise)));
}

int32 MessagesManager::get_unload_dialog_delay() const {
  constexpr int32 DIALOG_UNLOAD_DELAY = 60;        // seconds
  constexpr int32 DIALOG_UNLOAD_BOT_DELAY = 1800;  // seconds

  CHECK(is_message_unload_enabled());
  auto default_unload_delay = td_->auth_manager_->is_bot() ? DIALOG_UNLOAD_BOT_DELAY : DIALOG_UNLOAD_DELAY;
  return narrow_cast<int32>(td_->option_manager_->get_option_integer("message_unload_delay", default_unload_delay));
}

double MessagesManager::get_next_unload_dialog_delay(Dialog *d) const {
  if (d->unload_dialog_delay_seed == 0) {
    d->unload_dialog_delay_seed = Random::fast(1, 1000000000);
  }
  auto delay = get_unload_dialog_delay() / 4;
  return delay + delay * 1e-9 * d->unload_dialog_delay_seed;
}

void MessagesManager::unload_dialog(DialogId dialog_id, int32 delay) {
  if (G()->close_flag()) {
    return;
  }
  if (delay < 0) {
    delay = get_unload_dialog_delay() - 2;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!d->has_unload_timeout) {
    LOG(INFO) << "Don't need to unload " << dialog_id;
    // possible right after the dialog was opened
    return;
  }

  if (!is_message_unload_enabled()) {
    // just in case
    LOG(INFO) << "Message unload is disabled in " << dialog_id;
    d->has_unload_timeout = false;
    return;
  }

  bool has_left_to_unload_messages = false;
  auto to_unload_message_ids = find_unloadable_messages(d, G()->unix_time() - delay, has_left_to_unload_messages);

  vector<int64> unloaded_message_ids;
  vector<unique_ptr<Message>> unloaded_messages;
  for (auto message_id : to_unload_message_ids) {
    auto message = unload_message(d, message_id);
    CHECK(message != nullptr);
    if (message->is_update_sent) {
      unloaded_message_ids.push_back(message->message_id.get());
    }
    unloaded_messages.push_back(std::move(message));
  }
  if (unloaded_messages.size() >= MIN_DELETED_ASYNCHRONOUSLY_MESSAGES) {
    Scheduler::instance()->destroy_on_scheduler(G()->get_gc_scheduler_id(), unloaded_messages);
  }

  if (!to_unload_message_ids.empty() && !G()->use_message_database() && !d->is_empty) {
    d->have_full_history = false;
    d->have_full_history_source = 0;
  }

  if (!unloaded_message_ids.empty()) {
    send_closure_later(
        G()->td(), &Td::send_update,
        td_api::make_object<td_api::updateDeleteMessages>(get_chat_id_object(dialog_id, "updateDeleteMessages"),
                                                          std::move(unloaded_message_ids), false, true));
  }

  if (has_left_to_unload_messages) {
    LOG(DEBUG) << "Need to unload more messages in " << dialog_id;
    pending_unload_dialog_timeout_.add_timeout_in(
        d->dialog_id.get(),
        to_unload_message_ids.size() >= MAX_UNLOADED_MESSAGES ? 1.0 : get_next_unload_dialog_delay(d));
  } else {
    d->has_unload_timeout = false;
  }
}

void MessagesManager::clear_dialog_message_list(Dialog *d, bool remove_from_dialog_list, int32 last_message_date) {
  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d, max_message_id, -1, "delete_all_dialog_messages 1");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "delete_all_dialog_messages 2");
    }
  }

  if (d->unread_mention_count > 0) {
    set_dialog_unread_mention_count(d, 0);
    send_update_chat_unread_mention_count(d);
  }
  if (d->unread_reaction_count > 0) {
    set_dialog_unread_reaction_count(d, 0);
    send_update_chat_unread_reaction_count(d, "delete_all_dialog_messages");
  }

  bool has_last_message_id = d->last_message_id != MessageId();
  MessageId last_clear_history_message_id;
  if (!remove_from_dialog_list) {
    if (has_last_message_id) {
      last_clear_history_message_id = d->last_message_id;
    } else {
      last_message_date = d->last_clear_history_date;
      last_clear_history_message_id = d->last_clear_history_message_id;
    }
  }

  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }

  set_dialog_first_database_message_id(d, MessageId(), "delete_all_dialog_messages 4");
  set_dialog_last_database_message_id(d, MessageId(), "delete_all_dialog_messages 5");
  set_dialog_last_clear_history_date(d, last_message_date, last_clear_history_message_id,
                                     "delete_all_dialog_messages 6");
  d->last_read_all_mentions_message_id = MessageId();  // it is not needed anymore
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);

  if (has_last_message_id) {
    set_dialog_last_message_id(d, MessageId(), "delete_all_dialog_messages 7");
    send_update_chat_last_message(d, "delete_all_dialog_messages 8");
  }
  if (remove_from_dialog_list) {
    set_dialog_order(d, DEFAULT_ORDER, true, false, "delete_all_dialog_messages 9");
  } else {
    update_dialog_pos(d, "delete_all_dialog_messages 10");
  }
}

void MessagesManager::delete_all_dialog_messages(Dialog *d, bool remove_from_dialog_list, bool is_permanently_deleted) {
  CHECK(d != nullptr);
  LOG(INFO) << "Delete all messages in " << d->dialog_id
            << " with remove_from_dialog_list = " << remove_from_dialog_list
            << " and is_permanently_deleted = " << is_permanently_deleted;
  if (!td_->auth_manager_->is_bot()) {
    int32 last_message_date = 0;
    if (!remove_from_dialog_list && d->last_message_id.is_valid()) {
      auto m = get_message(d, d->last_message_id);
      CHECK(m != nullptr);
      last_message_date = m->date;
    }
    clear_dialog_message_list(d, remove_from_dialog_list, last_message_date);
  }

  vector<int64> deleted_message_ids;
  d->messages.foreach([&](const MessageId &message_id, unique_ptr<Message> &message) {
    CHECK(message_id == message->message_id);
    Message *m = message.get();

    static_cast<ListNode *>(m)->remove();

    LOG(INFO) << "Delete " << message_id;
    deleted_message_ids.push_back(message_id.get());

    delete_active_live_location(d->dialog_id, m);
    remove_message_file_sources(d->dialog_id, m);

    on_message_deleted(d, m, is_permanently_deleted, "do_delete_all_dialog_messages");

    if (is_permanently_deleted) {
      d->deleted_message_ids.insert(m->message_id);
    }
  });
  Scheduler::instance()->destroy_on_scheduler(G()->get_gc_scheduler_id(), d->messages, d->ordered_messages);

  delete_all_dialog_messages_from_database(d, MessageId::max(), "delete_all_dialog_messages 3");

  if (d->notification_info != nullptr) {
    // they aren't needed anymore
    delete_all_dialog_notifications(d, MessageId::max(), "delete_all_dialog_messages 4");

    d->notification_info->message_notification_group_.drop_max_removed_notification_id();
    d->notification_info->mention_notification_group_.drop_max_removed_notification_id();
    d->notification_info->notification_id_to_message_id_.clear();
  }

  on_dialog_updated(d->dialog_id, "delete_all_dialog_messages 11");

  send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), is_permanently_deleted);
}

void MessagesManager::on_dialog_deleted(DialogId dialog_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Delete " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "on_dialog_deleted");
  if (d == nullptr) {
    return promise.set_value(Unit());
  }

  delete_all_dialog_messages(d, true, false);
  if (dialog_id.get_type() != DialogType::SecretChat) {
    d->have_full_history = false;
    d->have_full_history_source = 0;
    d->is_empty = false;
    d->need_restore_reply_markup = true;
    on_dialog_updated(dialog_id, "on_dialog_deleted");
  }
  if (!td_->auth_manager_->is_bot()) {
    recently_found_dialogs_.remove_dialog(dialog_id);
    recently_opened_dialogs_.remove_dialog(dialog_id);
  }
  if (dialog_id.get_type() == DialogType::Channel) {
    G()->td_db()->get_binlog_pmc()->erase(get_channel_pts_key(dialog_id));
  }

  close_dialog(d);

  td_->forum_topic_manager_->delete_all_dialog_topics(dialog_id);

  promise.set_value(Unit());
}

void MessagesManager::on_update_dialog_group_call_rights(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  if (d->active_group_call_id.is_valid()) {
    td_->group_call_manager_->on_update_group_call_rights(d->active_group_call_id);
  }
}

void MessagesManager::read_all_dialog_mentions(DialogId dialog_id, MessageId top_thread_message_id,
                                               Promise<Unit> &&promise) {
  TRY_RESULT_PROMISE(promise, d, check_dialog_access(dialog_id, true, AccessRights::Read, "read_all_dialog_mentions"));
  TRY_STATUS_PROMISE(promise, can_use_top_thread_message_id(d, top_thread_message_id, MessageInputReplyTo()));

  if (top_thread_message_id.is_valid()) {
    LOG(INFO) << "Receive readAllChatMentions request in thread of " << top_thread_message_id << " in " << dialog_id;
    AffectedHistoryQuery query = [td = td_, top_thread_message_id](DialogId dialog_id,
                                                                   Promise<AffectedHistory> &&query_promise) {
      td->create_handler<ReadMentionsQuery>(std::move(query_promise))->send(dialog_id, top_thread_message_id);
    };
    run_affected_history_query_until_complete(dialog_id, std::move(query), true, std::move(promise));
    return;
  } else {
    LOG(INFO) << "Receive readAllChatMentions request in " << dialog_id << " with " << d->unread_mention_count
              << " unread mentions";
  }
  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(d->unread_mention_count == 0);
    return promise.set_value(Unit());
  }

  if (d->last_new_message_id > d->last_read_all_mentions_message_id) {
    d->last_read_all_mentions_message_id = d->last_new_message_id;
    on_dialog_updated(dialog_id, "read_all_dialog_mentions");
  }

  auto message_ids = find_dialog_messages(d, [](const Message *m) { return m->contains_unread_mention; });

  LOG(INFO) << "Found " << message_ids.size() << " messages with unread mentions in memory";
  bool is_update_sent = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->contains_unread_mention);
    CHECK(m->message_id == message_id);
    CHECK(m->message_id.is_valid());
    remove_message_notification_id(d, m, true, false);  // must be called before contains_unread_mention is updated
    m->contains_unread_mention = false;

    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateMessageMentionRead>(
                     get_chat_id_object(dialog_id, "updateMessageMentionRead"), m->message_id.get(), 0));
    is_update_sent = true;
    on_message_changed(d, m, true, "read_all_dialog_mentions");
  }

  if (d->unread_mention_count != 0) {
    set_dialog_unread_mention_count(d, 0);
    if (!is_update_sent) {
      send_update_chat_unread_mention_count(d);
    } else {
      LOG(INFO) << "Update unread mention message count in " << dialog_id << " to " << d->unread_mention_count;
      on_dialog_updated(dialog_id, "read_all_dialog_mentions");
    }
  }
  remove_message_dialog_notifications(d, MessageId::max(), true, "read_all_dialog_mentions");

  read_all_dialog_mentions_on_server(dialog_id, 0, std::move(promise));
}

class MessagesManager::ReadAllDialogMentionsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_read_all_dialog_mentions_on_server_log_event(DialogId dialog_id) {
  ReadAllDialogMentionsOnServerLogEvent log_event{dialog_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadAllDialogMentionsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::read_all_dialog_mentions_on_server(DialogId dialog_id, uint64 log_event_id,
                                                         Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_read_all_dialog_mentions_on_server_log_event(dialog_id);
  }

  AffectedHistoryQuery query = [td = td_](DialogId dialog_id, Promise<AffectedHistory> &&query_promise) {
    td->create_handler<ReadMentionsQuery>(std::move(query_promise))->send(dialog_id, MessageId());
  };
  run_affected_history_query_until_complete(dialog_id, std::move(query), false,
                                            get_erase_log_event_promise(log_event_id, std::move(promise)));
}

void MessagesManager::read_all_dialog_reactions(DialogId dialog_id, MessageId top_thread_message_id,
                                                Promise<Unit> &&promise) {
  TRY_RESULT_PROMISE(promise, d, check_dialog_access(dialog_id, true, AccessRights::Read, "read_all_dialog_reactions"));
  TRY_STATUS_PROMISE(promise, can_use_top_thread_message_id(d, top_thread_message_id, MessageInputReplyTo()));

  if (top_thread_message_id.is_valid()) {
    LOG(INFO) << "Receive readAllChatReactions request in thread of " << top_thread_message_id << " in " << dialog_id;
    AffectedHistoryQuery query = [td = td_, top_thread_message_id](DialogId dialog_id,
                                                                   Promise<AffectedHistory> &&query_promise) {
      td->create_handler<ReadReactionsQuery>(std::move(query_promise))->send(dialog_id, top_thread_message_id);
    };
    run_affected_history_query_until_complete(dialog_id, std::move(query), true, std::move(promise));
    return;
  } else {
    LOG(INFO) << "Receive readAllChatReactions request in " << dialog_id << " with " << d->unread_reaction_count
              << " unread reactions";
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(d->unread_reaction_count == 0);
    return promise.set_value(Unit());
  }

  auto message_ids = find_dialog_messages(
      d, [this, dialog_id](const Message *m) { return has_unread_message_reactions(dialog_id, m); });

  LOG(INFO) << "Found " << message_ids.size() << " messages with unread reactions in memory";
  bool is_update_sent = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(has_unread_message_reactions(dialog_id, m));
    CHECK(m->message_id == message_id);
    CHECK(m->message_id.is_valid());
    // remove_message_notification_id(d, m, true, false);  // must be called before unread_reactions are cleared
    m->reactions->unread_reactions_.clear();

    send_update_message_unread_reactions(dialog_id, m, 0);
    is_update_sent = true;
    on_message_changed(d, m, true, "read_all_dialog_reactions");
  }

  if (d->unread_reaction_count != 0) {
    set_dialog_unread_reaction_count(d, 0);
    if (!is_update_sent) {
      send_update_chat_unread_reaction_count(d, "read_all_dialog_reactions");
    } else {
      LOG(INFO) << "Update unread reaction message count in " << dialog_id << " to " << d->unread_reaction_count;
      on_dialog_updated(dialog_id, "read_all_dialog_reactions");
    }
  }
  // remove_message_dialog_notifications(d, MessageId::max(), true, "read_all_dialog_reactions");

  read_all_dialog_reactions_on_server(dialog_id, 0, std::move(promise));
}

class MessagesManager::ReadAllDialogReactionsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_read_all_dialog_reactions_on_server_log_event(DialogId dialog_id) {
  ReadAllDialogReactionsOnServerLogEvent log_event{dialog_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadAllDialogReactionsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::read_all_dialog_reactions_on_server(DialogId dialog_id, uint64 log_event_id,
                                                          Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_read_all_dialog_reactions_on_server_log_event(dialog_id);
  }

  AffectedHistoryQuery query = [td = td_](DialogId dialog_id, Promise<AffectedHistory> &&query_promise) {
    td->create_handler<ReadReactionsQuery>(std::move(query_promise))->send(dialog_id, MessageId());
  };
  run_affected_history_query_until_complete(dialog_id, std::move(query), false,
                                            get_erase_log_event_promise(log_event_id, std::move(promise)));
}

void MessagesManager::read_message_content_from_updates(MessageId message_id, int32 read_date) {
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id;
    return;
  }

  Dialog *d = get_dialog_by_message_id(message_id);
  if (d != nullptr) {
    Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    read_message_content(d, m, false, read_date, "read_message_content_from_updates");
  }
}

void MessagesManager::read_channel_message_content_from_updates(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id << " in " << d->dialog_id;
    return;
  }
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Message *m = get_message_force(d, message_id, "read_channel_message_content_from_updates");
  if (m != nullptr) {
    read_message_content(d, m, false, 0, "read_channel_message_content_from_updates");
  } else {
    if (!td_->dialog_manager_->have_input_peer(d->dialog_id, false, AccessRights::Read)) {
      LOG(INFO) << "Ignore updateChannelReadMessagesContents in inaccessible " << d->dialog_id;
      if (d->unread_mention_count != 0) {
        set_dialog_unread_mention_count(d, 0);
      }
      return;
    }
    if (message_id > d->last_new_message_id && d->last_new_message_id.is_valid()) {
      get_channel_difference(d->dialog_id, d->pts, 0, message_id, true, "read_channel_message_content_from_updates");
    } else {
      // there is no message, so the update can be ignored
      if (d->unread_mention_count > 0) {
        // but if the chat has unread mentions, then number of unread mentions could have been changed
        repair_dialog_unread_mention_count(d, "read_channel_message_content_from_updates");
      }
    }
  }
}

bool MessagesManager::read_message_content(Dialog *d, Message *m, bool is_local_read, int32 read_date,
                                           const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  bool is_mention_read = update_message_contains_unread_mention(d, m, false, "read_message_content");
  bool is_content_read = update_opened_message_content(m->content.get());
  if (ttl_on_open(d, m, Time::now(), is_local_read, read_date)) {
    is_content_read = true;
  }

  LOG(INFO) << "Read message content of " << m->message_id << " in " << d->dialog_id
            << ": is_mention_read = " << is_mention_read << ", is_content_read = " << is_content_read;
  if (is_mention_read || is_content_read) {
    on_message_changed(d, m, true, "read_message_content");
    if (is_content_read) {
      send_closure(G()->td(), &Td::send_update,
                   td_api::make_object<td_api::updateMessageContentOpened>(
                       get_chat_id_object(d->dialog_id, "updateMessageContentOpened"), m->message_id.get()));
    }
    return true;
  }
  return false;
}

bool MessagesManager::has_incoming_notification(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (m->is_from_scheduled) {
    return true;
  }
  return !m->message_id.is_scheduled() && !m->is_outgoing && dialog_id != td_->dialog_manager_->get_my_dialog_id();
}

int32 MessagesManager::calc_new_unread_count_from_last_unread(Dialog *d, MessageId max_message_id,
                                                              MessageType type) const {
  CHECK(!max_message_id.is_scheduled());
  auto it = d->ordered_messages.get_const_iterator(max_message_id);
  if (*it == nullptr || (*it)->get_message_id() != max_message_id) {
    return -1;
  }

  int32 unread_count = type == MessageType::Server ? d->server_unread_count : d->local_unread_count;
  while (*it != nullptr && (*it)->get_message_id() > d->last_read_inbox_message_id) {
    auto message_id = (*it)->get_message_id();
    if (message_id.get_type() == type && has_incoming_notification(d->dialog_id, get_message(d, message_id))) {
      unread_count--;
    }
    --it;
  }
  if (*it == nullptr || (*it)->get_message_id() != d->last_read_inbox_message_id) {
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from last unread message";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count_from_the_end(Dialog *d, MessageId max_message_id, MessageType type,
                                                          int32 hint_unread_count) const {
  CHECK(!max_message_id.is_scheduled());
  int32 unread_count = 0;
  auto it = d->ordered_messages.get_const_iterator(MessageId::max());
  while (*it != nullptr && (*it)->get_message_id() > max_message_id) {
    auto message_id = (*it)->get_message_id();
    if (message_id.get_type() == type && has_incoming_notification(d->dialog_id, get_message(d, message_id))) {
      unread_count++;
    }
    --it;
  }

  bool is_count_exact = d->last_message_id.is_valid() && *it != nullptr;
  if (hint_unread_count >= 0) {
    if (is_count_exact) {
      if (hint_unread_count == unread_count) {
        return hint_unread_count;
      }
    } else {
      if (hint_unread_count >= unread_count) {
        return hint_unread_count;
      }
    }

    // hint_unread_count is definitely wrong, ignore it

    if (need_unread_counter(d->order)) {
      LOG(ERROR) << "Receive hint_unread_count = " << hint_unread_count << ", but found " << unread_count
                 << " unread messages in " << d->dialog_id;
    }
  }

  if (!is_count_exact) {
    // unread count is likely to be calculated wrong, so ignore it
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from the end";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count(Dialog *d, MessageId max_message_id, MessageType type,
                                             int32 hint_unread_count) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(!max_message_id.is_scheduled());
  if (d->is_empty) {
    return 0;
  }

  if (!d->last_read_inbox_message_id.is_valid()) {
    return calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  }

  if (!d->last_message_id.is_valid() ||
      (d->last_message_id.get() - max_message_id.get() > max_message_id.get() - d->last_read_inbox_message_id.get())) {
    int32 unread_count = calc_new_unread_count_from_last_unread(d, max_message_id, type);
    return unread_count >= 0 ? unread_count
                             : calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  } else {
    int32 unread_count = calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
    return unread_count >= 0 ? unread_count : calc_new_unread_count_from_last_unread(d, max_message_id, type);
  }
}

void MessagesManager::repair_server_unread_count(DialogId dialog_id, int32 unread_count, const char *source) {
  if (td_->auth_manager_->is_bot() || !td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return;
  }
  if (pending_read_history_timeout_.has_timeout(dialog_id.get())) {
    return;  // postpone until read history request is sent
  }

  LOG(INFO) << "Repair server unread count in " << dialog_id << " from " << unread_count << " from " << source;
  create_actor<SleepActor>("RepairServerUnreadCountSleepActor", 0.2,
                           PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Result<Unit> result) {
                             send_closure(actor_id, &MessagesManager::send_get_dialog_query, dialog_id, Promise<Unit>(),
                                          0, "repair_server_unread_count");
                           }))
      .release();
}

void MessagesManager::repair_channel_server_unread_count(Dialog *d) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::Channel);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (d->last_read_inbox_message_id >= d->last_new_message_id) {
    // all messages are already read
    return;
  }
  if (!need_unread_counter(d->order)) {
    // there are no unread counters in left channels
    return;
  }
  if (!d->need_repair_channel_server_unread_count) {
    d->need_repair_channel_server_unread_count = true;
    on_dialog_updated(d->dialog_id, "repair_channel_server_unread_count");
  }

  LOG(INFO) << "Reload ChannelFull for " << d->dialog_id << " to repair unread message counts";
  td_->dialog_manager_->get_dialog_info_full(d->dialog_id, Auto(), "repair_channel_server_unread_count");
}

void MessagesManager::repair_dialog_unread_reaction_count(Dialog *d, Promise<Unit> &&promise, const char *source) {
  CHECK(d != nullptr);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!d->need_repair_unread_reaction_count) {
    d->need_repair_unread_reaction_count = true;
    on_dialog_updated(d->dialog_id, "repair_dialog_unread_reaction_count");
  }

  send_get_dialog_query(d->dialog_id, std::move(promise), 0, source);
}

void MessagesManager::repair_dialog_unread_mention_count(Dialog *d, const char *source) {
  CHECK(d != nullptr);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!d->need_repair_unread_mention_count) {
    d->need_repair_unread_mention_count = true;
    on_dialog_updated(d->dialog_id, "repair_dialog_unread_mention_count");
  }

  send_get_dialog_query(d->dialog_id, Promise<Unit>(), 0, source);
}

void MessagesManager::read_history_inbox(DialogId dialog_id, MessageId max_message_id, int32 unread_count,
                                         const char *source) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "read_history_inbox");
  if (d != nullptr) {
    read_history_inbox(d, max_message_id, unread_count, source);
  } else {
    LOG(INFO) << "Receive read inbox about unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::read_history_inbox(Dialog *d, MessageId max_message_id, int32 unread_count, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto dialog_id = d->dialog_id;
  if (d->need_repair_channel_server_unread_count) {
    d->need_repair_channel_server_unread_count = false;
    on_dialog_updated(dialog_id, "read_history_inbox");
  }

  // there can be updateReadHistoryInbox up to message 0, if messages where read and then all messages where deleted
  if (!max_message_id.is_valid() && max_message_id != MessageId()) {
    LOG(ERROR) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source;
    return;
  }
  if (d->is_last_read_inbox_message_id_inited && max_message_id <= d->last_read_inbox_message_id) {
    LOG(INFO) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source
              << ", but all messages have already been read up to " << d->last_read_inbox_message_id;
    if (max_message_id == d->last_read_inbox_message_id && unread_count >= 0 &&
        unread_count != d->server_unread_count) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), unread_count, d->local_unread_count, true, source);
    }
    return;
  }

  if (max_message_id != MessageId() && max_message_id.is_yet_unsent()) {
    LOG(ERROR) << "Tried to update last read inbox message in " << dialog_id << " with " << max_message_id << " from "
               << source;
    return;
  }

  if (max_message_id != MessageId() && unread_count > 0 && max_message_id >= d->last_new_message_id &&
      max_message_id >= d->last_message_id && max_message_id >= d->last_database_message_id) {
    if (d->last_new_message_id.is_valid()) {
      LOG(ERROR) << "Have unknown " << unread_count << " unread messages up to " << max_message_id << " in "
                 << dialog_id << " with last_new_message_id = " << d->last_new_message_id
                 << ", last_message_id = " << d->last_message_id
                 << ", last_database_message_id = " << d->last_database_message_id << ", and " << d->server_unread_count
                 << " unread messages up to " << d->last_read_inbox_message_id << " from " << source;
      unread_count = d->server_unread_count;
    } else {
      unread_count = 0;
    }
  }

  LOG_IF(INFO, d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
                   (d->notification_info != nullptr &&
                    max_message_id > d->notification_info->max_push_notification_message_id_) &&
                   max_message_id.is_server() && dialog_id.get_type() != DialogType::Channel &&
                   !running_get_difference_)
      << "Receive read inbox update up to unknown " << max_message_id << " in " << dialog_id << " from " << source
      << ". Last new is " << d->last_new_message_id << ", unread_count = " << unread_count
      << ". Possible only for deleted incoming message";

  if (dialog_id.get_type() == DialogType::SecretChat) {
    ttl_read_history(d, false, max_message_id, d->last_read_inbox_message_id, Time::now());
  }

  if (max_message_id > d->last_new_message_id && dialog_id.get_type() == DialogType::Channel) {
    schedule_get_channel_difference(dialog_id, 0, max_message_id, 0.001, "read_history_inbox");
  }

  int32 server_unread_count = calc_new_unread_count(d, max_message_id, MessageType::Server, unread_count);
  int32 local_unread_count =
      d->local_unread_count == 0 ? 0 : calc_new_unread_count(d, max_message_id, MessageType::Local, -1);

  if (server_unread_count < 0) {
    server_unread_count = unread_count >= 0 ? unread_count : d->server_unread_count;
    if (td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read) && need_unread_counter(d->order)) {
      d->need_repair_server_unread_count = true;
      on_dialog_updated(dialog_id, "read_history_inbox");
      repair_server_unread_count(dialog_id, server_unread_count, "read_history_inbox");
    }
  }
  if (local_unread_count < 0) {
    // TODO repair local unread count
    local_unread_count = d->local_unread_count;
  }

  set_dialog_last_read_inbox_message_id(d, max_message_id, server_unread_count, local_unread_count, true, source);

  if (d->is_marked_as_unread && max_message_id != MessageId()) {
    set_dialog_is_marked_as_unread(d, false);
  }
}

void MessagesManager::read_history_outbox(DialogId dialog_id, MessageId max_message_id, int32 read_date) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "read_history_outbox");
  if (d != nullptr) {
    read_history_outbox(d, max_message_id, read_date);
  } else {
    LOG(INFO) << "Receive read outbox update about unknown " << dialog_id;
  }
}

void MessagesManager::read_history_outbox(Dialog *d, MessageId max_message_id, int32 read_date) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto dialog_id = d->dialog_id;
  if (!max_message_id.is_valid()) {
    LOG(ERROR) << "Receive read outbox update in " << dialog_id << " with " << max_message_id;
    return;
  }
  if (max_message_id <= d->last_read_outbox_message_id) {
    LOG(INFO) << "Receive read outbox update up to " << max_message_id
              << ", but all messages have already been read up to " << d->last_read_outbox_message_id;
    return;
  }

  if (max_message_id.is_yet_unsent()) {
    LOG(ERROR) << "Tried to update last read outbox message with " << max_message_id << " in " << dialog_id;
    return;
  }

  // it is impossible for just sent outgoing messages because updates are ordered by PTS
  if (d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
      dialog_id.get_type() != DialogType::Channel) {
    LOG(INFO) << "Receive read outbox update about unknown " << max_message_id << " in " << dialog_id
              << " with last new " << d->last_new_message_id << ". Possible only for deleted outgoing message";
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    double server_time = G()->server_time();
    double read_time = Time::now();
    if (read_date <= 0) {
      LOG(ERROR) << "Receive wrong read date " << read_date << " in " << dialog_id;
    } else if (read_date < server_time) {
      read_time -= (server_time - read_date);
    }
    ttl_read_history(d, true, max_message_id, d->last_read_outbox_message_id, read_time);
  }

  set_dialog_last_read_outbox_message_id(d, max_message_id);
}

bool MessagesManager::need_unread_counter(int64 dialog_order) {
  return dialog_order != DEFAULT_ORDER;
}

int32 MessagesManager::get_dialog_total_count(const DialogList &list) const {
  int32 sponsored_dialog_count = 0;
  if (sponsored_dialog_id_.is_valid() && list.dialog_list_id == DialogListId(FolderId::main())) {
    auto d = get_dialog(sponsored_dialog_id_);
    CHECK(d != nullptr);
    if (is_dialog_sponsored(d)) {
      sponsored_dialog_count = 1;
    }
  }
  if (list.server_dialog_total_count_ != -1 && list.secret_chat_total_count_ != -1) {
    return std::max(list.server_dialog_total_count_ + list.secret_chat_total_count_,
                    list.in_memory_dialog_total_count_) +
           sponsored_dialog_count;
  }
  if (list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    return list.in_memory_dialog_total_count_ + sponsored_dialog_count;
  }
  return list.in_memory_dialog_total_count_ + sponsored_dialog_count + 1;
}

void MessagesManager::repair_server_dialog_total_count(DialogListId dialog_list_id) {
  if (G()->close_flag()) {
    return;
  }
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!dialog_list_id.is_folder()) {
    // can repair total count only in folders
    return;
  }

  LOG(INFO) << "Repair total chat count in " << dialog_list_id;
  td_->create_handler<GetDialogListQuery>(Promise<Unit>())
      ->send(dialog_list_id.get_folder_id(), 2147483647, ServerMessageId(), DialogId(), 1);
}

void MessagesManager::repair_secret_chat_total_count(DialogListId dialog_list_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (G()->use_message_database() && dialog_list_id.is_folder()) {
    // race-prone
    G()->td_db()->get_dialog_db_async()->get_secret_chat_count(
        dialog_list_id.get_folder_id(),
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_list_id](Result<int32> result) {
          if (result.is_error()) {
            return;
          }
          send_closure(actor_id, &MessagesManager::on_get_secret_chat_total_count, dialog_list_id, result.move_as_ok());
        }));
  } else {
    int32 total_count = 0;
    auto *list = get_dialog_list(dialog_list_id);
    CHECK(list != nullptr);
    for (auto &folder_id : get_dialog_list_folder_ids(*list)) {
      const auto *folder_list = get_dialog_list(DialogListId(folder_id));
      CHECK(folder_list != nullptr);
      if (folder_list->need_unread_count_recalc_) {
        // can't repair total secret chat count yet
        return;
      }

      const auto *folder = get_dialog_folder(folder_id);
      CHECK(folder != nullptr);
      for (const auto &dialog_date : folder->ordered_dialogs_) {
        auto dialog_id = dialog_date.get_dialog_id();
        if (dialog_id.get_type() == DialogType::SecretChat && dialog_date.get_order() != DEFAULT_ORDER) {
          total_count++;
        }
      }
    }
    on_get_secret_chat_total_count(dialog_list_id, total_count);
  }
}

void MessagesManager::on_get_secret_chat_total_count(DialogListId dialog_list_id, int32 total_count) {
  if (G()->close_flag()) {
    return;
  }

  CHECK(!td_->auth_manager_->is_bot());
  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    // just in case
    return;
  }
  CHECK(total_count >= 0);
  if (list->secret_chat_total_count_ != total_count) {
    auto old_dialog_total_count = get_dialog_total_count(*list);
    list->secret_chat_total_count_ = total_count;
    if (list->is_dialog_unread_count_inited_) {
      if (old_dialog_total_count != get_dialog_total_count(*list)) {
        send_update_unread_chat_count(*list, DialogId(), true, "on_get_secret_chat_total_count");
      } else {
        save_unread_chat_count(*list);
      }
    }
  }
}

void MessagesManager::recalc_unread_count(DialogListId dialog_list_id, int32 old_dialog_total_count, bool force) {
  if (G()->close_flag() || td_->auth_manager_->is_bot() || !G()->use_message_database()) {
    return;
  }

  auto *list_ptr = get_dialog_list(dialog_list_id);
  CHECK(list_ptr != nullptr);
  auto &list = *list_ptr;
  if (!list.need_unread_count_recalc_ && !force) {
    return;
  }
  LOG(INFO) << "Recalculate unread counts in " << dialog_list_id;
  list.need_unread_count_recalc_ = false;
  list.is_message_unread_count_inited_ = true;
  list.is_dialog_unread_count_inited_ = true;

  int32 message_total_count = 0;
  int32 message_muted_count = 0;
  int32 dialog_total_count = 0;
  int32 dialog_muted_count = 0;
  int32 dialog_marked_count = 0;
  int32 dialog_muted_marked_count = 0;
  int32 server_dialog_total_count = 0;
  int32 secret_chat_total_count = 0;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    const auto &folder = *get_dialog_folder(folder_id);
    for (const auto &dialog_date : folder.ordered_dialogs_) {
      if (dialog_date.get_order() == DEFAULT_ORDER) {
        break;
      }

      auto dialog_id = dialog_date.get_dialog_id();
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      if (!is_dialog_in_list(d, dialog_list_id)) {
        continue;
      }

      int unread_count = d->server_unread_count + d->local_unread_count;
      if (need_unread_counter(d->order) && (unread_count > 0 || d->is_marked_as_unread)) {
        message_total_count += unread_count;
        dialog_total_count++;
        if (unread_count == 0 && d->is_marked_as_unread) {
          dialog_marked_count++;
        }

        LOG(DEBUG) << "Have " << unread_count << " messages in " << dialog_id;
        if (is_dialog_muted(d)) {
          message_muted_count += unread_count;
          dialog_muted_count++;
          if (unread_count == 0 && d->is_marked_as_unread) {
            dialog_muted_marked_count++;
          }
        }
      }
      if (d->order != DEFAULT_ORDER) {  // must not count sponsored dialog, which is added independently
        if (dialog_id.get_type() == DialogType::SecretChat) {
          secret_chat_total_count++;
        } else {
          server_dialog_total_count++;
        }
      }
    }
  }

  if (list.unread_message_total_count_ != message_total_count ||
      list.unread_message_muted_count_ != message_muted_count) {
    list.unread_message_total_count_ = message_total_count;
    list.unread_message_muted_count_ = message_muted_count;
    send_update_unread_message_count(list, DialogId(), true, "recalc_unread_count");
  }

  if (old_dialog_total_count == -1) {
    old_dialog_total_count = get_dialog_total_count(list);
  }
  bool need_save = false;
  if (list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    if (server_dialog_total_count != list.server_dialog_total_count_ ||
        secret_chat_total_count != list.secret_chat_total_count_) {
      list.server_dialog_total_count_ = server_dialog_total_count;
      list.secret_chat_total_count_ = secret_chat_total_count;
      need_save = true;
    }
  } else {
    if (list.server_dialog_total_count_ == -1) {
      // recalc_unread_count is called only after getDialogs request; it is unneeded to call getDialogs again
      repair_server_dialog_total_count(dialog_list_id);
    }

    if (list.secret_chat_total_count_ == -1) {
      repair_secret_chat_total_count(dialog_list_id);
    }
  }
  if (list.unread_dialog_total_count_ != dialog_total_count || list.unread_dialog_muted_count_ != dialog_muted_count ||
      list.unread_dialog_marked_count_ != dialog_marked_count ||
      list.unread_dialog_muted_marked_count_ != dialog_muted_marked_count ||
      old_dialog_total_count != get_dialog_total_count(list)) {
    list.unread_dialog_total_count_ = dialog_total_count;
    list.unread_dialog_muted_count_ = dialog_muted_count;
    list.unread_dialog_marked_count_ = dialog_marked_count;
    list.unread_dialog_muted_marked_count_ = dialog_muted_marked_count;
    send_update_unread_chat_count(list, DialogId(), true, "recalc_unread_count");
  } else if (need_save) {
    save_unread_chat_count(list);
  }
}

void MessagesManager::set_dialog_last_read_inbox_message_id(Dialog *d, MessageId message_id, int32 server_unread_count,
                                                            int32 local_unread_count, bool force_update,
                                                            const char *source) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read inbox message in " << d->dialog_id << " from " << d->last_read_inbox_message_id
            << " to " << message_id << " and update unread message count from " << d->server_unread_count << " + "
            << d->local_unread_count << " to " << server_unread_count << " + " << local_unread_count << " from "
            << source;
  if (message_id != MessageId::min()) {
    d->last_read_inbox_message_id = message_id;
    d->is_last_read_inbox_message_id_inited = true;
  }
  int32 old_unread_count = d->server_unread_count + d->local_unread_count;
  d->server_unread_count = server_unread_count;
  d->local_unread_count = local_unread_count;

  if (need_unread_counter(d->order)) {
    const int32 new_unread_count = d->server_unread_count + d->local_unread_count;
    for (auto &list : get_dialog_lists(d)) {
      int32 delta = new_unread_count - old_unread_count;
      if (delta != 0 && list.is_message_unread_count_inited_) {
        list.unread_message_total_count_ += delta;
        if (is_dialog_muted(d)) {
          list.unread_message_muted_count_ += delta;
        }
        send_update_unread_message_count(list, d->dialog_id, force_update, source);
      }
      delta = static_cast<int32>(new_unread_count != 0) - static_cast<int32>(old_unread_count != 0);
      if (delta != 0 && list.is_dialog_unread_count_inited_) {
        if (d->is_marked_as_unread) {
          list.unread_dialog_marked_count_ -= delta;
        } else {
          list.unread_dialog_total_count_ += delta;
        }
        if (is_dialog_muted(d)) {
          if (d->is_marked_as_unread) {
            list.unread_dialog_muted_marked_count_ -= delta;
          } else {
            list.unread_dialog_muted_count_ += delta;
          }
        }
        send_update_unread_chat_count(list, d->dialog_id, force_update, source);
      }
    }

    bool was_unread = old_unread_count != 0 || d->is_marked_as_unread;
    bool is_unread = new_unread_count != 0 || d->is_marked_as_unread;
    if (td_->dialog_filter_manager_->have_dialog_filters() && was_unread != is_unread) {
      update_dialog_lists(d, get_dialog_positions(d), true, false, "set_dialog_last_read_inbox_message_id");
    }
  }

  if (message_id != MessageId::min() && d->last_read_inbox_message_id.is_valid() &&
      (d->order != DEFAULT_ORDER || is_dialog_sponsored(d))) {
    VLOG(notifications) << "Remove some notifications in " << d->dialog_id
                        << " after updating last read inbox message to " << message_id
                        << " and unread message count to " << server_unread_count << " + " << local_unread_count
                        << " from " << source;
    if (d->notification_info != nullptr && d->notification_info->message_notification_group_.is_valid()) {
      auto total_count = get_dialog_pending_notification_count(d, false);
      if (total_count == 0) {
        set_dialog_last_notification(d->dialog_id, d->notification_info->message_notification_group_, 0,
                                     NotificationId(), source);
      }
      if (!d->notification_info->pending_new_message_notifications_.empty()) {
        for (auto &it : d->notification_info->pending_new_message_notifications_) {
          if (it.second <= message_id) {
            it.first = DialogId();
          }
        }
        flush_pending_new_message_notifications(d->dialog_id, false, DialogId(UserId(static_cast<int64>(1))));
      }
      total_count -= static_cast<int32>(d->notification_info->pending_new_message_notifications_.size());
      if (total_count < 0) {
        LOG(ERROR) << "Total message notification count is " << total_count << " in " << d->dialog_id
                   << " with old unread_count = " << old_unread_count << " and "
                   << d->notification_info->pending_new_message_notifications_
                   << " pending new message notifications after reading history up to " << message_id;
        total_count = 0;
      }
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification_group,
                         d->notification_info->message_notification_group_.get_group_id(), NotificationId(),
                         d->last_read_inbox_message_id, total_count, Slice(source) == Slice("view_messages"),
                         Promise<Unit>());
    }

    if (d->notification_info != nullptr && d->notification_info->mention_notification_group_.is_valid() &&
        d->notification_info->pinned_message_notification_message_id_.is_valid() &&
        d->notification_info->pinned_message_notification_message_id_ <= d->last_read_inbox_message_id) {
      // remove pinned message notification when it is read
      remove_dialog_pinned_message_notification(d, source);
    }
  }

  on_dialog_updated(d->dialog_id, source);
  send_update_chat_read_inbox(d, force_update, source);
}

void MessagesManager::set_dialog_last_read_outbox_message_id(Dialog *d, MessageId message_id) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read outbox message in " << d->dialog_id << " from " << d->last_read_outbox_message_id
            << " to " << message_id;
  d->last_read_outbox_message_id = message_id;
  d->is_last_read_outbox_message_id_inited = true;
  send_update_chat_read_outbox(d);
}

void MessagesManager::set_dialog_max_unavailable_message_id(DialogId dialog_id, MessageId max_unavailable_message_id,
                                                            bool from_update, const char *source) {
  CHECK(!max_unavailable_message_id.is_scheduled());

  Dialog *d = get_dialog_force(dialog_id, source);
  if (d != nullptr) {
    if (d->last_new_message_id.is_valid() && max_unavailable_message_id > d->last_new_message_id && from_update) {
      // possible if the last unavailable message has already been deleted
      LOG(INFO) << "Tried to set " << dialog_id << " max unavailable message to " << max_unavailable_message_id
                << " from " << source << ", but last new message is " << d->last_new_message_id;
      max_unavailable_message_id = d->last_new_message_id;
    }

    if (d->max_unavailable_message_id == max_unavailable_message_id) {
      return;
    }

    if (max_unavailable_message_id.is_valid() && max_unavailable_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Tried to update " << dialog_id << " max unavailable message with " << max_unavailable_message_id
                 << " from " << source;
      return;
    }
    LOG(INFO) << "Set max unavailable message to " << max_unavailable_message_id << " in " << dialog_id << " from "
              << source;

    on_dialog_updated(dialog_id, "set_dialog_max_unavailable_message_id");

    if (d->max_unavailable_message_id > max_unavailable_message_id) {
      d->max_unavailable_message_id = max_unavailable_message_id;
      return;
    }

    d->max_unavailable_message_id = max_unavailable_message_id;

    auto message_ids = d->ordered_messages.find_older_messages(max_unavailable_message_id);

    vector<int64> deleted_message_ids;
    bool need_update_dialog_pos = false;
    for (auto message_id : message_ids) {
      if (message_id.is_yet_unsent()) {
        continue;
      }

      auto m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id <= max_unavailable_message_id);
      CHECK(m->message_id == message_id);
      auto message =
          delete_message(d, message_id, !from_update, &need_update_dialog_pos, "set_dialog_max_unavailable_message_id");
      CHECK(message.get() == m);
      deleted_message_ids.push_back(m->message_id.get());
    }

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "set_dialog_max_unavailable_message_id");
    }

    send_update_delete_messages(dialog_id, std::move(deleted_message_ids), !from_update);

    if (d->server_unread_count + d->local_unread_count > 0) {
      read_history_inbox(d, max_unavailable_message_id, -1, "set_dialog_max_unavailable_message_id");
    }
  } else {
    LOG(INFO) << "Receive max unavailable message in unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::on_update_viewed_messages_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  LOG(INFO) << "Expired timeout for updating of recently viewed messages in " << dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (d->open_count == 0) {
    return;
  }

  auto it = dialog_viewed_messages_.find(dialog_id);
  if (it == dialog_viewed_messages_.end() || !td_->is_online()) {
    return;
  }

  auto &info = it->second;
  CHECK(info != nullptr);
  vector<MessageId> reaction_message_ids;
  vector<MessageId> views_message_ids;
  vector<MessageId> extended_media_message_ids;
  int32 newest_message_date = 0;
  for (auto &message_it : info->message_id_to_view_id) {
    Message *m = get_message_force(d, message_it.first, "on_update_viewed_messages_timeout");
    CHECK(m != nullptr);
    CHECK(m->message_id.is_valid());
    CHECK(m->message_id.is_server());
    if (need_poll_message_reactions(d, m)) {
      reaction_message_ids.push_back(m->message_id);
    }
    if (m->view_count > 0 && !m->has_get_message_views_query) {
      m->has_get_message_views_query = true;
      views_message_ids.push_back(m->message_id);
    }
    if (need_poll_message_content_extended_media(m->content.get()) && !m->has_get_extended_media_query) {
      m->has_get_extended_media_query = true;
      extended_media_message_ids.push_back(m->message_id);
    }
    if (m->date > newest_message_date) {
      newest_message_date = m->date;
    }
  }

  if (!reaction_message_ids.empty()) {
    queue_message_reactions_reload(dialog_id, reaction_message_ids);
  }
  if (!views_message_ids.empty()) {
    td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(views_message_ids), false);
  }
  if (!extended_media_message_ids.empty()) {
    td_->create_handler<GetExtendedMediaQuery>()->send(dialog_id, std::move(extended_media_message_ids));
  }

  int divisor = 5 - min(max(G()->unix_time() - newest_message_date, 0) / UPDATE_VIEWED_MESSAGES_PERIOD, 4);
  update_viewed_messages_timeout_.add_timeout_in(dialog_id.get(), UPDATE_VIEWED_MESSAGES_PERIOD / divisor);
}

void MessagesManager::on_send_update_chat_read_inbox_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  if (postponed_chat_read_inbox_updates_.erase(dialog_id) > 0) {
    send_update_chat_read_inbox(get_dialog(dialog_id), true, "on_send_update_chat_read_inbox_timeout");
  }
}

int32 MessagesManager::get_message_date(const tl_object_ptr<telegram_api::Message> &message_ptr) {
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      return 0;
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return message->date_;
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return message->date_;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

vector<UserId> MessagesManager::get_message_user_ids(const Message *m) const {
  vector<UserId> user_ids;
  if (m->sender_user_id.is_valid()) {
    user_ids.push_back(m->sender_user_id);
  }
  if (m->via_bot_user_id.is_valid()) {
    user_ids.push_back(m->via_bot_user_id);
  }
  if (m->via_business_bot_user_id.is_valid()) {
    user_ids.push_back(m->via_business_bot_user_id);
  }
  if (m->forward_info != nullptr) {
    m->forward_info->add_min_user_ids(user_ids);
  }
  append(user_ids, get_message_content_min_user_ids(td_, m->content.get()));
  if (!m->replied_message_info.is_empty()) {
    append(user_ids, m->replied_message_info.get_min_user_ids(td_));
  }
  return user_ids;
}

vector<ChannelId> MessagesManager::get_message_channel_ids(const Message *m) const {
  vector<ChannelId> channel_ids;
  if (m->sender_dialog_id.is_valid() && m->sender_dialog_id.get_type() == DialogType::Channel) {
    channel_ids.push_back(m->sender_dialog_id.get_channel_id());
  }
  if (m->forward_info != nullptr) {
    m->forward_info->add_min_channel_ids(channel_ids);
  }
  append(channel_ids, get_message_content_min_channel_ids(td_, m->content.get()));
  if (!m->replied_message_info.is_empty()) {
    append(channel_ids, m->replied_message_info.get_min_channel_ids(td_));
  }
  return channel_ids;
}

void MessagesManager::ttl_read_history(Dialog *d, bool is_outgoing, MessageId from_message_id,
                                       MessageId till_message_id, double view_date) {
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  // TODO: protect with log event
  suffix_load_till_message_id(d, till_message_id,
                              PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = d->dialog_id, is_outgoing,
                                                      from_message_id, till_message_id, view_date](Result<Unit>) {
                                send_closure(actor_id, &MessagesManager::ttl_read_history_impl, dialog_id, is_outgoing,
                                             from_message_id, till_message_id, view_date);
                              }));
}

void MessagesManager::ttl_read_history_impl(DialogId dialog_id, bool is_outgoing, MessageId from_message_id,
                                            MessageId till_message_id, double view_date) {
  CHECK(dialog_id.get_type() == DialogType::SecretChat);
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto now = Time::now();
  for (auto it = d->ordered_messages.get_const_iterator(from_message_id);
       *it && (*it)->get_message_id() >= till_message_id; --it) {
    auto *m = get_message(d, (*it)->get_message_id());
    CHECK(m != nullptr);
    if (m->is_outgoing == is_outgoing) {
      ttl_on_view(d, m, view_date, now);
    }
  }
}

void MessagesManager::ttl_on_view(const Dialog *d, Message *m, double view_date, double now) {
  if (!m->ttl.is_empty() && m->ttl_expires_at == 0 && !m->message_id.is_scheduled() && !m->message_id.is_yet_unsent() &&
      !m->is_failed_to_send && !m->is_content_secret) {
    m->ttl_expires_at = m->ttl.get_input_ttl() + view_date;
    ttl_register_message(d->dialog_id, m, now);
    on_message_changed(d, m, true, "ttl_on_view");
  }
}

bool MessagesManager::ttl_on_open(Dialog *d, Message *m, double now, bool is_local_read, int32 read_date) {
  CHECK(!m->message_id.is_scheduled());
  if (!m->ttl.is_empty() && m->ttl_expires_at == 0) {
    int32 passed_after_read_time = 0;
    auto can_destroy_immediately = [&] {
      if (m->ttl.is_immediate()) {
        return true;
      }
      if (is_local_read) {
        return false;
      }
      if (read_date <= 0) {
        return d->dialog_id.get_type() != DialogType::SecretChat;
      }
      passed_after_read_time = max(G()->unix_time() - read_date, 0);
      if (m->ttl.get_input_ttl() <= passed_after_read_time) {
        return true;
      }
      return false;
    }();
    if (can_destroy_immediately) {
      on_message_ttl_expired(d, m);
    } else {
      m->ttl_expires_at = m->ttl.get_input_ttl() + now - passed_after_read_time;
      ttl_register_message(d->dialog_id, m, now);
    }
    return true;
  }
  return false;
}

void MessagesManager::ttl_register_message(DialogId dialog_id, const Message *m, double now) {
  CHECK(m != nullptr);
  CHECK(m->ttl_expires_at != 0);
  CHECK(!m->message_id.is_scheduled());

  auto it_flag = ttl_nodes_.emplace(dialog_id, m->message_id, false);
  CHECK(it_flag.second);
  auto it = it_flag.first;

  ttl_heap_.insert(m->ttl_expires_at, it->as_heap_node());
  ttl_update_timeout(now);
}

void MessagesManager::ttl_period_register_message(DialogId dialog_id, const Message *m, double server_time) {
  CHECK(m != nullptr);
  CHECK(m->ttl_period != 0);
  CHECK(!m->message_id.is_scheduled());

  auto it_flag = ttl_nodes_.emplace(dialog_id, m->message_id, true);
  CHECK(it_flag.second);
  auto it = it_flag.first;

  auto now = Time::now();
  ttl_heap_.insert(now + (m->date + m->ttl_period - server_time), it->as_heap_node());
  ttl_update_timeout(now);
}

void MessagesManager::ttl_unregister_message(DialogId dialog_id, const Message *m, const char *source) {
  if (m->ttl_expires_at == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  auto it = ttl_nodes_.find(TtlNode(dialog_id, m->message_id, false));

  // expect m->ttl.is_empty(), but m->ttl_expires_at > 0 from binlog
  CHECK(it != ttl_nodes_.end());

  auto *heap_node = it->as_heap_node();
  if (heap_node->in_heap()) {
    ttl_heap_.erase(heap_node);
  }
  ttl_nodes_.erase(it);
  ttl_update_timeout(Time::now());
}

void MessagesManager::ttl_period_unregister_message(DialogId dialog_id, const Message *m) {
  if (m->ttl_period == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  auto it = ttl_nodes_.find(TtlNode(dialog_id, m->message_id, true));

  CHECK(it != ttl_nodes_.end());

  auto *heap_node = it->as_heap_node();
  if (heap_node->in_heap()) {
    ttl_heap_.erase(heap_node);
  }
  ttl_nodes_.erase(it);
  ttl_update_timeout(Time::now());
}

void MessagesManager::ttl_loop(double now) {
  FlatHashMap<DialogId, std::vector<MessageId>, DialogIdHash> to_delete;
  while (!ttl_heap_.empty() && ttl_heap_.top_key() < now) {
    TtlNode *ttl_node = TtlNode::from_heap_node(ttl_heap_.pop());
    auto message_full_id = ttl_node->message_full_id_;
    auto dialog_id = message_full_id.get_dialog_id();
    CHECK(dialog_id.is_valid());
    if (dialog_id.get_type() == DialogType::SecretChat || ttl_node->by_ttl_period_) {
      to_delete[dialog_id].push_back(message_full_id.get_message_id());
    } else {
      auto d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      auto m = get_message(d, message_full_id.get_message_id());
      CHECK(m != nullptr);
      on_message_ttl_expired(d, m);
      on_message_changed(d, m, true, "ttl_loop");
    }
  }
  for (auto &it : to_delete) {
    delete_dialog_messages(it.first, it.second, false, "ttl_loop");
  }
  ttl_update_timeout(now);
}

void MessagesManager::ttl_update_timeout(double now) {
  if (ttl_heap_.empty()) {
    if (!ttl_slot_.empty()) {
      ttl_slot_.cancel_timeout();
    }
    return;
  }
  ttl_slot_.set_event(EventCreator::yield(actor_id()));
  ttl_slot_.set_timeout_in(ttl_heap_.top_key() - now);
}

void MessagesManager::on_message_ttl_expired(Dialog *d, Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->ttl.is_valid());
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  ttl_unregister_message(d->dialog_id, m, "on_message_ttl_expired");
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_ttl_expired");
  remove_message_file_sources(d->dialog_id, m);
  on_message_ttl_expired_impl(d, m, true);
  register_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_ttl_expired");
  send_update_message_content(d, m, true, "on_message_ttl_expired");
  // the caller must call on_message_changed
}

void MessagesManager::on_message_ttl_expired_impl(Dialog *d, Message *m, bool is_message_in_dialog) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(!m->message_id.is_yet_unsent());
  CHECK(m->ttl.is_valid());
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  delete_message_files(d->dialog_id, m);
  update_expired_message_content(m->content);
  m->ttl = {};
  m->ttl_expires_at = 0;
  if (m->reply_markup != nullptr) {
    if (m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
      if (d->reply_markup_message_id == m->message_id) {
        set_dialog_reply_markup(d, MessageId());
      }
      m->had_reply_markup = true;
    }
    m->reply_markup = nullptr;
  }
  remove_message_notification_id(d, m, true, true);
  update_message_contains_unread_mention(d, m, false, "on_message_ttl_expired_impl");
  remove_message_unread_reactions(d, m, "on_message_ttl_expired_impl");
  set_message_reply(d, m, MessageInputReplyTo(), is_message_in_dialog);
  m->noforwards = false;
  m->contains_mention = false;
  m->linked_top_thread_message_id = MessageId();
  m->is_content_secret = false;
  m->invert_media = false;
}

void MessagesManager::loop() {
  auto token = get_link_token();
  if (token == YieldType::TtlDb) {
    ttl_db_loop();
  } else {
    ttl_loop(Time::now());
  }
}

void MessagesManager::tear_down() {
  parent_.reset();

  LOG(DEBUG) << "Have " << dialogs_.calc_size() << " chats with " << added_message_count_ << " messages to free";
}

void MessagesManager::hangup() {
  postponed_channel_updates_.clear();

  if (!G()->use_message_database()) {
    while (!being_uploaded_files_.empty()) {
      auto it = being_uploaded_files_.begin();
      auto message_full_id = it->second.first;
      being_uploaded_files_.erase(it);
      if (message_full_id.get_message_id().is_yet_unsent()) {
        fail_send_message(message_full_id, Global::request_aborted_error());
      }
    }
    while (!being_uploaded_thumbnails_.empty()) {
      auto it = being_uploaded_thumbnails_.begin();
      auto message_full_id = it->second.message_full_id;
      being_uploaded_thumbnails_.erase(it);
      if (message_full_id.get_message_id().is_yet_unsent()) {
        fail_send_message(message_full_id, Global::request_aborted_error());
      }
    }
    while (!being_loaded_secret_thumbnails_.empty()) {
      auto it = being_loaded_secret_thumbnails_.begin();
      auto message_full_id = it->second.message_full_id;
      being_loaded_secret_thumbnails_.erase(it);
      if (message_full_id.get_message_id().is_yet_unsent()) {
        fail_send_message(message_full_id, Global::request_aborted_error());
      }
    }
    while (!yet_unsent_media_queues_.empty()) {
      auto it = yet_unsent_media_queues_.begin();
      auto queue = std::move(it->second);
      yet_unsent_media_queues_.erase(it);
      for (auto &promise_it : queue.queue_) {
        auto message_id = promise_it.first;
        if (message_id.is_yet_unsent()) {
          fail_send_message({queue.dialog_id_, message_id}, Global::request_aborted_error());
        }
      }
    }
    while (!being_sent_messages_.empty()) {
      on_send_message_fail(being_sent_messages_.begin()->first, Global::request_aborted_error());
    }
    while (!update_message_ids_.empty()) {
      auto it = update_message_ids_.begin();
      auto message_full_id = MessageFullId(it->first.get_dialog_id(), it->second);
      update_message_ids_.erase(it);

      // the message was sent successfully, but can't be added yet
      fail_send_message(message_full_id, Global::request_aborted_error());
    }
    while (!update_scheduled_message_ids_.empty()) {
      auto it = update_scheduled_message_ids_.begin();
      auto dialog_id = it->first;
      auto message_ids = std::move(it->second);
      update_scheduled_message_ids_.erase(it);
      for (auto &message_id_it : message_ids) {
        // the message was sent successfully, but can't be added yet
        fail_send_message({dialog_id, message_id_it.second}, Global::request_aborted_error());
      }
    }
  }

  fail_promises(load_active_live_location_messages_queries_, Global::request_aborted_error());
  auto fail_promise_map = [](auto &queries) {
    while (!queries.empty()) {
      auto it = queries.begin();
      auto promises = std::move(it->second);
      queries.erase(it);
      fail_promises(promises, Global::request_aborted_error());
    }
  };
  fail_promise_map(get_dialog_queries_);
  fail_promise_map(load_scheduled_messages_from_database_queries_);
  fail_promise_map(run_after_get_channel_difference_);
  fail_promise_map(search_public_dialogs_queries_);
  fail_promise_map(get_history_queries_);
  while (!pending_channel_on_get_dialogs_.empty()) {
    auto it = pending_channel_on_get_dialogs_.begin();
    auto promise = std::move(it->second.promise);
    pending_channel_on_get_dialogs_.erase(it);
    promise.set_error(Global::request_aborted_error());
  }
  while (!get_dialogs_tasks_.empty()) {
    auto it = get_dialogs_tasks_.begin();
    auto promise = std::move(it->second.promise);
    get_dialogs_tasks_.erase(it);
    promise.set_error(Global::request_aborted_error());
  }

  stop();
}

void MessagesManager::start_up() {
  init();
}

void MessagesManager::create_folders(int source) {
  LOG(INFO) << "Create folders";
  create_folders_source_ = source;
  dialog_folders_[FolderId::main()].folder_id = FolderId::main();
  dialog_folders_[FolderId::archive()].folder_id = FolderId::archive();

  add_dialog_list(DialogListId(FolderId::main()));
  add_dialog_list(DialogListId(FolderId::archive()));
  create_folders_source_ = source + 1;
}

void MessagesManager::init() {
  if (is_inited_) {
    return;
  }
  is_inited_ = true;

  td_->notification_settings_manager_->init();  // load scope notification settings
  td_->reaction_manager_->init();               // load available reactions

  start_time_ = Time::now();
  last_channel_pts_jump_warning_time_ = start_time_ - 3600;

  bool was_authorized_user = td_->auth_manager_->was_authorized() && !td_->auth_manager_->is_bot();
  if (was_authorized_user) {
    create_folders(10);  // ensure that Main and Archive dialog lists are created
  }
  authorization_date_ = td_->option_manager_->get_option_integer("authorization_date");

  td_->dialog_filter_manager_->init();  // load dialog filters

  if (G()->use_message_database() && was_authorized_user) {
    // erase old keys
    G()->td_db()->get_binlog_pmc()->erase("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase("unread_dialog_count");

    auto last_database_server_dialog_dates = G()->td_db()->get_binlog_pmc()->prefix_get("last_server_dialog_date");
    for (auto &it : last_database_server_dialog_dates) {
      auto r_folder_id = to_integer_safe<int32>(it.first);
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }

      string order_str;
      string dialog_id_str;
      std::tie(order_str, dialog_id_str) = split(it.second);

      auto r_order = to_integer_safe<int64>(order_str);
      auto r_dialog_id = to_integer_safe<int64>(dialog_id_str);
      if (r_order.is_error() || r_dialog_id.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        FolderId folder_id(r_folder_id.ok());
        auto *folder = get_dialog_folder(folder_id);
        CHECK(folder != nullptr);
        DialogDate dialog_date(r_order.ok(), DialogId(r_dialog_id.ok()));
        if (dialog_date.get_date() == 0 && dialog_date != MAX_DIALOG_DATE) {
          LOG(ERROR) << "Ignore incorrect last database server dialog date " << dialog_date << " in " << folder_id;
        } else {
          if (folder->last_database_server_dialog_date_ < dialog_date) {
            folder->last_database_server_dialog_date_ = dialog_date;
          }
          LOG(INFO) << "Loaded last_database_server_dialog_date_ " << folder->last_database_server_dialog_date_
                    << " in " << folder_id;
        }
      }
    }

    auto sponsored_dialog_id_string = G()->td_db()->get_binlog_pmc()->get("sponsored_dialog_id");
    if (!sponsored_dialog_id_string.empty()) {
      auto dialog_id_source = split(Slice(sponsored_dialog_id_string));
      auto r_dialog_id = to_integer_safe<int64>(dialog_id_source.first);
      auto r_source = DialogSource::unserialize(dialog_id_source.second);
      if (r_dialog_id.is_error() || r_source.is_error()) {
        LOG(ERROR) << "Can't parse " << sponsored_dialog_id_string;
      } else {
        DialogId dialog_id(r_dialog_id.ok());

        const Dialog *d = get_dialog_force(dialog_id, "init");
        if (d != nullptr) {
          LOG(INFO) << "Loaded sponsored " << dialog_id;
          add_sponsored_dialog(d, r_source.move_as_ok());
        } else {
          LOG(ERROR) << "Can't load " << dialog_id;
        }
      }
    }

    auto pinned_dialog_ids = G()->td_db()->get_binlog_pmc()->prefix_get("pinned_dialog_ids");
    for (auto &it : pinned_dialog_ids) {
      auto r_folder_id = to_integer_safe<int32>(it.first);
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }
      FolderId folder_id(r_folder_id.ok());

      auto r_dialog_ids = transform(full_split(Slice(it.second), ','), [](Slice str) -> Result<DialogId> {
        TRY_RESULT(dialog_id_int, to_integer_safe<int64>(str));
        DialogId dialog_id(dialog_id_int);
        if (!dialog_id.is_valid()) {
          return Status::Error("Have invalid dialog ID");
        }
        return dialog_id;
      });
      if (any_of(r_dialog_ids, [](const auto &r_dialog_id) { return r_dialog_id.is_error(); })) {
        LOG(ERROR) << "Can't parse " << it.second;
        reload_pinned_dialogs(DialogListId(folder_id), Auto());
      } else {
        auto *list = get_dialog_list(DialogListId(folder_id));
        CHECK(list != nullptr);
        CHECK(list->pinned_dialogs_.empty());
        for (auto &r_dialog_id : reversed(r_dialog_ids)) {
          auto dialog_id = r_dialog_id.move_as_ok();
          if (!dialog_id.is_valid()) {
            LOG(ERROR) << "Loaded " << dialog_id << " as a pinned dialog";
            continue;
          }
          auto order = get_next_pinned_dialog_order();
          list->pinned_dialogs_.emplace_back(order, dialog_id);
          list->pinned_dialog_id_orders_.emplace(dialog_id, order);
        }
        std::reverse(list->pinned_dialogs_.begin(), list->pinned_dialogs_.end());
        list->are_pinned_dialogs_inited_ = true;
        update_list_last_pinned_dialog_date(*list);

        LOG(INFO) << "Loaded pinned chats " << list->pinned_dialogs_ << " in " << folder_id;
      }
    }

    auto unread_message_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_message_count");
    for (auto &it : unread_message_counts) {
      auto r_dialog_list_id = to_integer_safe<int64>(it.first);
      if (r_dialog_list_id.is_error()) {
        LOG(ERROR) << "Can't parse dialog list ID from " << it.first;
        continue;
      }
      string total_count;
      string muted_count;
      std::tie(total_count, muted_count) = split(it.second);

      auto r_total_count = to_integer_safe<int32>(total_count);
      auto r_muted_count = to_integer_safe<int32>(muted_count);
      if (r_total_count.is_error() || r_muted_count.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        DialogListId dialog_list_id(r_dialog_list_id.ok());
        auto *list = get_dialog_list(dialog_list_id);
        if (list != nullptr) {
          list->unread_message_total_count_ = r_total_count.ok();
          list->unread_message_muted_count_ = r_muted_count.ok();
          list->is_message_unread_count_inited_ = true;
          send_update_unread_message_count(*list, DialogId(), true, "load unread_message_count", true);
        } else {
          G()->td_db()->get_binlog_pmc()->erase("unread_message_count" + it.first);
        }
      }
    }

    auto unread_dialog_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_dialog_count");
    for (auto &it : unread_dialog_counts) {
      auto r_dialog_list_id = to_integer_safe<int64>(it.first);
      if (r_dialog_list_id.is_error()) {
        LOG(ERROR) << "Can't parse dialog list ID from " << it.first;
        continue;
      }

      auto counts = transform(full_split(Slice(it.second)), [](Slice str) { return to_integer_safe<int32>(str); });
      if ((counts.size() != 4 && counts.size() != 6) || any_of(counts, [](const auto &c) { return c.is_error(); })) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        DialogListId dialog_list_id(r_dialog_list_id.ok());
        auto *list = get_dialog_list(dialog_list_id);
        if (list != nullptr) {
          list->unread_dialog_total_count_ = counts[0].ok();
          list->unread_dialog_muted_count_ = counts[1].ok();
          list->unread_dialog_marked_count_ = counts[2].ok();
          list->unread_dialog_muted_marked_count_ = counts[3].ok();
          if (counts.size() == 6) {
            list->server_dialog_total_count_ = counts[4].ok();
            list->secret_chat_total_count_ = counts[5].ok();
          }
          if (list->server_dialog_total_count_ == -1) {
            repair_server_dialog_total_count(dialog_list_id);
          }
          if (list->secret_chat_total_count_ == -1) {
            repair_secret_chat_total_count(dialog_list_id);
          }
          list->is_dialog_unread_count_inited_ = true;
          send_update_unread_chat_count(*list, DialogId(), true, "load unread_dialog_count", true);
        } else {
          G()->td_db()->get_binlog_pmc()->erase("unread_dialog_count" + it.first);
        }
      }
    }
  } else if (!td_->auth_manager_->is_bot()) {
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("pinned_dialog_ids");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_dialog_count");
    G()->td_db()->get_binlog_pmc()->erase("sponsored_dialog_id");
    G()->td_db()->get_binlog_pmc()->erase("fetched_marks_as_unread");
  }
  G()->td_db()->get_binlog_pmc()->erase("dialog_pinned_current_order");

  if (G()->use_message_database()) {
    ttl_db_loop();
  }

  load_calls_db_state();

  auto auth_notification_ids_string = G()->td_db()->get_binlog_pmc()->get("auth_notification_ids");
  if (!auth_notification_ids_string.empty()) {
    VLOG(notifications) << "Loaded auth_notification_ids = " << auth_notification_ids_string;
    auto stored_ids = full_split(auth_notification_ids_string, ',');
    CHECK(stored_ids.size() % 2 == 0);
    bool is_changed = false;
    auto min_date = G()->unix_time() - AUTH_NOTIFICATION_ID_CACHE_TIME;
    for (size_t i = 0; i < stored_ids.size(); i += 2) {
      auto date = to_integer_safe<int32>(stored_ids[i + 1]).ok();
      if (date < min_date || stored_ids[i].empty()) {
        is_changed = true;
        continue;
      }
      if (auth_notification_id_date_.size() == MAX_SAVED_AUTH_NOTIFICATION_IDS) {
        is_changed = true;
        break;
      }
      auth_notification_id_date_.emplace(std::move(stored_ids[i]), date);
    }
    if (is_changed) {
      save_auth_notification_ids();
    }
  }
}

void MessagesManager::on_authorization_success() {
  CHECK(td_->auth_manager_->is_authorized());
  authorization_date_ = td_->option_manager_->get_option_integer("authorization_date");

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  create_folders(20);
}

void MessagesManager::ttl_db_loop() {
  if (ttl_db_has_query_) {
    return;
  }

  auto now = Time::now();
  if (now < ttl_db_next_request_time_) {
    ttl_db_slot_.set_event(EventCreator::yield(actor_shared(this, YieldType::TtlDb)));
    auto wakeup_in = ttl_db_next_request_time_ - now;
    ttl_db_slot_.set_timeout_in(wakeup_in);
    LOG(INFO) << "Set ttl_db timeout in " << wakeup_in;
    return;
  }

  ttl_db_has_query_ = true;
  LOG(INFO) << "Send ttl_db query with limit " << ttl_db_next_limit_;
  G()->td_db()->get_message_db_async()->get_expiring_messages(
      G()->unix_time() - 1, ttl_db_next_limit_,
      PromiseCreator::lambda([actor_id = actor_id(this)](Result<std::vector<MessageDbMessage>> result) {
        send_closure(actor_id, &MessagesManager::ttl_db_on_result, std::move(result), false);
      }));
}

void MessagesManager::ttl_db_on_result(Result<std::vector<MessageDbMessage>> r_result, bool dummy) {
  if (G()->close_flag()) {
    return;
  }

  CHECK(r_result.is_ok());
  auto result = r_result.move_as_ok();
  ttl_db_has_query_ = false;

  int32 next_request_delay;
  if (result.size() == static_cast<size_t>(ttl_db_next_limit_)) {
    CHECK(ttl_db_next_limit_ < (1 << 30));
    ttl_db_next_limit_ *= 2;
    next_request_delay = 1;
  } else {
    ttl_db_next_limit_ = DEFAULT_LOADED_EXPIRED_MESSAGES;
    next_request_delay = Random::fast(3000, 4200);
  }
  ttl_db_next_request_time_ = Time::now() + next_request_delay;

  LOG(INFO) << "Receive " << result.size() << " expired messages from ttl_db with next request in "
            << next_request_delay << " seconds";
  for (auto &dialog_message : result) {
    on_get_message_from_database(dialog_message, false, "ttl_db_on_result");
  }
  ttl_db_loop();
}

void MessagesManager::on_send_secret_message_error(int64 random_id, Status error, Promise<Unit> promise) {
  promise.set_value(Unit());  // TODO: set after error is saved

  auto it = being_sent_messages_.find(random_id);
  if (it != being_sent_messages_.end()) {
    auto message_full_id = it->second;
    auto *m = get_message(message_full_id);
    if (m != nullptr) {
      auto file_id = get_message_content_upload_file_id(m->content.get());
      if (file_id.is_valid()) {
        if (G()->close_flag() && G()->use_message_database()) {
          // do not send error, message will be re-sent after restart
          return;
        }
        auto bad_parts = FileManager::get_missing_file_parts(error);
        if (!bad_parts.empty()) {
          on_send_message_file_parts_missing(random_id, std::move(bad_parts));
          return;
        }

        td_->file_manager_->delete_partial_remote_location_if_needed(file_id, error);
      }
    }
  }

  on_send_message_fail(random_id, std::move(error));
}

void MessagesManager::on_send_secret_message_success(int64 random_id, MessageId message_id, int32 date,
                                                     unique_ptr<EncryptedFile> file, Promise<Unit> promise) {
  promise.set_value(Unit());  // TODO: set after message is saved

  FileId new_file_id;
  if (file != nullptr) {
    if (!DcId::is_valid(file->dc_id_)) {
      LOG(ERROR) << "Wrong dc_id = " << file->dc_id_ << " in file " << *file;
    } else {
      DialogId owner_dialog_id;
      auto it = being_sent_messages_.find(random_id);
      if (it != being_sent_messages_.end()) {
        owner_dialog_id = it->second.get_dialog_id();
      }

      new_file_id = td_->file_manager_->register_remote(
          FullRemoteFileLocation(FileType::Encrypted, file->id_, file->access_hash_, DcId::internal(file->dc_id_), ""),
          FileLocationSource::FromServer, owner_dialog_id, 0, file->size_, to_string(static_cast<uint64>(file->id_)));
    }
  }

  on_send_message_success(random_id, message_id, date, 0, new_file_id, "on_send_secret_message_success");
}

void MessagesManager::delete_secret_messages(SecretChatId secret_chat_id, std::vector<int64> random_ids,
                                             Promise<Unit> promise) {
  LOG(DEBUG) << "On delete messages in " << secret_chat_id << " with random_ids " << random_ids;
  CHECK(secret_chat_id.is_valid());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id, "delete_secret_messages")) {
    LOG(ERROR) << "Ignore delete secret messages in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteMessages;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->random_ids = std::move(random_ids);

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_messages(DialogId dialog_id, std::vector<int64> random_ids,
                                                    Promise<Unit> promise) {
  LOG(INFO) << "Delete messages with random_ids " << random_ids << " in " << dialog_id;
  promise.set_value(Unit());  // TODO: set after event is saved

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  vector<MessageId> to_delete_message_ids;
  for (auto &random_id : random_ids) {
    auto message_id = get_message_id_by_random_id(d, random_id, "finish_delete_secret_messages");
    if (!message_id.is_valid()) {
      LOG(INFO) << "Can't find message with random_id " << random_id;
      continue;
    }
    const Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    if (!is_service_message_content(m->content->get_type())) {
      to_delete_message_ids.push_back(message_id);
    } else {
      LOG(INFO) << "Skip deletion of service " << message_id;
    }
  }
  delete_dialog_messages(d, to_delete_message_ids, true, "finish_delete_secret_messages");
}

void MessagesManager::delete_secret_chat_history(SecretChatId secret_chat_id, bool remove_from_dialog_list,
                                                 MessageId last_message_id, Promise<Unit> promise) {
  LOG(DEBUG) << "Delete history in " << secret_chat_id << " up to " << last_message_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(!last_message_id.is_scheduled());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id, "delete_secret_chat_history")) {
    LOG(ERROR) << "Ignore delete history in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteHistory;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->last_message_id = last_message_id;
  pending_secret_message->remove_from_dialog_list = remove_from_dialog_list;

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_chat_history(DialogId dialog_id, bool remove_from_dialog_list,
                                                        MessageId last_message_id, Promise<Unit> promise) {
  LOG(DEBUG) << "Delete history in " << dialog_id << " up to " << last_message_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  // TODO: probably last_message_id is not needed
  delete_all_dialog_messages(d, remove_from_dialog_list, true);
  promise.set_value(Unit());  // TODO: set after event is saved
}

void MessagesManager::read_secret_chat_outbox(SecretChatId secret_chat_id, int32 up_to_date, int32 read_date) {
  if (!secret_chat_id.is_valid()) {
    LOG(ERROR) << "Receive read secret chat outbox in the invalid " << secret_chat_id;
    return;
  }
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  auto dialog_id = DialogId(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id, "read_secret_chat_outbox");
  if (d == nullptr) {
    return;
  }

  if (read_date > 0) {
    auto user_id = td_->user_manager_->get_secret_chat_user_id(secret_chat_id);
    if (user_id.is_valid()) {
      td_->user_manager_->on_update_user_local_was_online(user_id, read_date);
    }
  }

  // TODO: protect with log event
  suffix_load_till_date(
      d, up_to_date,
      PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, up_to_date, read_date](Result<Unit> result) {
        send_closure(actor_id, &MessagesManager::read_secret_chat_outbox_inner, dialog_id, up_to_date, read_date);
      }));
}

void MessagesManager::read_secret_chat_outbox_inner(DialogId dialog_id, int32 up_to_date, int32 read_date) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto end = d->ordered_messages.get_const_iterator(MessageId::max());
  while (*end &&
         (get_message(d, (*end)->get_message_id())->date > up_to_date || (*end)->get_message_id().is_yet_unsent())) {
    --end;
  }
  if (!*end) {
    LOG(INFO) << "Ignore read_secret_chat_outbox in " << dialog_id << " at " << up_to_date
              << ": no messages with such date are known";
    return;
  }
  auto max_message_id = (*end)->get_message_id();
  read_history_outbox(d, max_message_id, read_date);
}

void MessagesManager::open_secret_message(SecretChatId secret_chat_id, int64 random_id, Promise<Unit> promise) {
  promise.set_value(Unit());  // TODO: set after event is saved
  DialogId dialog_id(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id, "open_secret_message");
  if (d == nullptr) {
    LOG(ERROR) << "Ignore opening secret chat message in unknown " << dialog_id;
    return;
  }

  auto message_id = get_message_id_by_random_id(d, random_id, "open_secret_message");
  if (!message_id.is_valid()) {
    return;
  }
  Message *m = get_message(d, message_id);
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent() || m->is_failed_to_send || !m->is_outgoing) {
    LOG(ERROR) << "Peer has opened wrong " << message_id << " in " << dialog_id;
    return;
  }

  read_message_content(d, m, false, 0, "open_secret_message");
}

void MessagesManager::on_update_secret_chat_state(SecretChatId secret_chat_id, SecretChatState state) {
  if (state == SecretChatState::Closed && !td_->auth_manager_->is_bot()) {
    DialogId dialog_id(secret_chat_id);
    Dialog *d = get_dialog_force(dialog_id, "on_update_secret_chat_state");
    if (d != nullptr && d->notification_info != nullptr) {
      if (d->notification_info->new_secret_chat_notification_id_.is_valid()) {
        remove_new_secret_chat_notification(d, true);
      }
      if (d->notification_info->message_notification_group_.is_valid() &&
          get_dialog_pending_notification_count(d, false) == 0 &&
          !d->notification_info->message_notification_group_.get_last_notification_id().is_valid()) {
        d->notification_info->message_notification_group_.try_reuse();
        on_dialog_updated(d->dialog_id, "on_update_secret_chat_state");
      }
      CHECK(!d->notification_info->mention_notification_group_
                 .is_valid());  // there can't be unread mentions in secret chats
    }
  }
}

void MessagesManager::on_get_secret_message(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                            int32 date, unique_ptr<EncryptedFile> file,
                                            tl_object_ptr<secret_api::decryptedMessage> message,
                                            Promise<Unit> promise) {
  LOG(DEBUG) << "On get " << to_string(message);
  CHECK(message != nullptr);
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  if (message->random_id_ == 0) {
    LOG(ERROR) << "Ignore secret message with random_id == 0";
    promise.set_error(Status::Error(400, "Invalid random_id"));
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = message->random_id_;
  message_info.ttl = MessageSelfDestructType(message->ttl_, false);
  message_info.has_unread_content = true;
  message_info.is_silent = message->silent_;
  message_info.media_album_id = message->grouped_id_;

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_get_secret_message");
  if (d == nullptr && td_->dialog_manager_->have_dialog_info_force(message_info.dialog_id, "on_get_secret_message")) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  pending_secret_message_ids_[message_info.dialog_id][message_info.random_id] = message_id;

  pending_secret_message->load_data_multipromise.add_promise(Auto());
  auto lock_promise = pending_secret_message->load_data_multipromise.get_promise();

  if ((message->flags_ & secret_api::decryptedMessage::REPLY_TO_RANDOM_ID_MASK) != 0) {
    auto reply_to_message_id = get_message_id_by_random_id(d, message->reply_to_random_id_, "on_get_secret_message");
    if (!reply_to_message_id.is_valid()) {
      auto dialog_it = pending_secret_message_ids_.find(message_info.dialog_id);
      if (dialog_it != pending_secret_message_ids_.end()) {
        auto message_it = dialog_it->second.find(message->reply_to_random_id_);
        if (message_it != dialog_it->second.end()) {
          reply_to_message_id = message_it->second;
        }
      }
    }
    message_info.reply_header.replied_message_info_ = RepliedMessageInfo::legacy(reply_to_message_id);
  }

  if (!clean_input_string(message->via_bot_name_)) {
    LOG(WARNING) << "Receive invalid bot username " << message->via_bot_name_;
    message->via_bot_name_.clear();
  }
  if (!message->via_bot_name_.empty()) {
    auto request_promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), via_bot_username = message->via_bot_name_, message_info_ptr = &message_info,
         promise = pending_secret_message->load_data_multipromise.get_promise()](Unit) mutable {
          send_closure(actor_id, &MessagesManager::on_resolve_secret_chat_message_via_bot_username, via_bot_username,
                       message_info_ptr, std::move(promise));
        });
    td_->dialog_manager_->search_public_dialog(message->via_bot_name_, false, std::move(request_promise));
  }

  message_info.content = get_secret_message_content(
      td_, std::move(message->message_), std::move(file), std::move(message->media_), std::move(message->entities_),
      message_info.dialog_id, pending_secret_message->load_data_multipromise,
      td_->user_manager_->is_user_premium(user_id));

  add_secret_message(std::move(pending_secret_message), std::move(lock_promise));
}

void MessagesManager::on_resolve_secret_chat_message_via_bot_username(const string &via_bot_username,
                                                                      MessageInfo *message_info_ptr,
                                                                      Promise<Unit> &&promise) {
  if (!G()->close_flag()) {
    auto dialog_id = td_->dialog_manager_->get_resolved_dialog_by_username(via_bot_username);
    if (dialog_id.is_valid() && dialog_id.get_type() == DialogType::User) {
      auto user_id = dialog_id.get_user_id();
      auto r_bot_data = td_->user_manager_->get_bot_data(user_id);
      if (r_bot_data.is_ok() && r_bot_data.ok().is_inline) {
        message_info_ptr->via_bot_user_id = user_id;
      }
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_secret_chat_screenshot_taken(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                      int32 date, int64 random_id, Promise<Unit> promise) {
  LOG(DEBUG) << "On screenshot taken in " << secret_chat_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.content = create_screenshot_taken_message_content();

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_secret_chat_screenshot_taken");
  if (d == nullptr &&
      td_->dialog_manager_->have_dialog_info_force(message_info.dialog_id, "on_secret_chat_screenshot_taken")) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::on_secret_chat_ttl_changed(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                 int32 date, int32 ttl, int64 random_id, Promise<Unit> promise) {
  LOG(DEBUG) << "On self-destruct timer set in " << secret_chat_id << " to " << ttl;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);
  if (ttl < 0) {
    LOG(WARNING) << "Receive wrong self-destruct time = " << ttl;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.content = create_chat_set_ttl_message_content(ttl, UserId());

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_secret_chat_ttl_changed");
  if (d == nullptr &&
      td_->dialog_manager_->have_dialog_info_force(message_info.dialog_id, "on_secret_chat_ttl_changed")) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message,
                                         Promise<Unit> lock_promise) {
  auto &multipromise = pending_secret_message->load_data_multipromise;
  multipromise.set_ignore_errors(true);
  int64 token = pending_secret_messages_.add(std::move(pending_secret_message));

  multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), token](Result<Unit> result) {
    if (result.is_ok()) {
      send_closure(actor_id, &MessagesManager::on_add_secret_message_ready, token);
    }
  }));

  if (!lock_promise) {
    lock_promise = multipromise.get_promise();
  }
  lock_promise.set_value(Unit());
}

void MessagesManager::on_add_secret_message_ready(int64 token) {
  if (G()->close_flag()) {
    return;
  }

  pending_secret_messages_.finish(
      token, [actor_id = actor_id(this)](unique_ptr<PendingSecretMessage> pending_secret_message) {
        send_closure_later(actor_id, &MessagesManager::finish_add_secret_message, std::move(pending_secret_message));
      });
}

void MessagesManager::finish_add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message) {
  if (G()->close_flag()) {
    return;
  }

  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteMessages) {
    return finish_delete_secret_messages(pending_secret_message->dialog_id,
                                         std::move(pending_secret_message->random_ids),
                                         std::move(pending_secret_message->success_promise));
  }
  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteHistory) {
    return finish_delete_secret_chat_history(
        pending_secret_message->dialog_id, pending_secret_message->remove_from_dialog_list,
        pending_secret_message->last_message_id, std::move(pending_secret_message->success_promise));
  }

  auto d = get_dialog(pending_secret_message->message_info.dialog_id);
  CHECK(d != nullptr);
  auto random_id = pending_secret_message->message_info.random_id;
  auto message_id = get_message_id_by_random_id(d, random_id, "finish_add_secret_message");
  if (message_id.is_valid()) {
    if (message_id != pending_secret_message->message_info.message_id) {
      LOG(WARNING) << "Ignore duplicate " << pending_secret_message->message_info.message_id
                   << " received earlier with " << message_id << " and random_id " << random_id;
    }
  } else {
    if (!td_->user_manager_->is_user_premium(pending_secret_message->message_info.sender_user_id)) {
      auto message_text = get_message_content_text_mutable(pending_secret_message->message_info.content.get());
      if (message_text != nullptr) {
        remove_premium_custom_emoji_entities(td_, message_text->entities, true);
      }
    }

    on_get_message(std::move(pending_secret_message->message_info), true, false, "finish add secret message");
  }
  auto dialog_it = pending_secret_message_ids_.find(d->dialog_id);
  if (dialog_it != pending_secret_message_ids_.end()) {
    auto message_it = dialog_it->second.find(random_id);
    if (message_it != dialog_it->second.end() && message_it->second == message_id) {
      dialog_it->second.erase(message_it);
      if (dialog_it->second.empty()) {
        pending_secret_message_ids_.erase(dialog_it);
      }
    }
  }

  pending_secret_message->success_promise.set_value(Unit());  // TODO: set after message is saved
}

MessagesManager::MessageInfo MessagesManager::parse_telegram_api_message(
    Td *td, tl_object_ptr<telegram_api::Message> message_ptr, bool is_scheduled, const char *source) {
  LOG(DEBUG) << "Receive from " << source << " " << to_string(message_ptr);
  LOG_CHECK(message_ptr != nullptr) << source;

  MessageInfo message_info;
  message_info.message_id = MessageId::get_message_id(message_ptr, is_scheduled);
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      message_info.message_id = MessageId();
      break;
    case telegram_api::message::ID: {
      auto message = move_tl_object_as<telegram_api::message>(message_ptr);
      if (message->quick_reply_shortcut_id_ != 0) {
        LOG(ERROR) << "Receive shortcut " << message->quick_reply_shortcut_id_ << " from " << source;
        message_info.message_id = MessageId();
        break;
      }

      message_info.dialog_id = DialogId(message->peer_id_);
      if (message->from_id_ != nullptr) {
        message_info.sender_dialog_id = DialogId(message->from_id_);
      } else {
        message_info.sender_dialog_id = message_info.dialog_id;
      }
      message_info.date = message->date_;
      message_info.forward_header = std::move(message->fwd_from_);
      bool can_have_thread = message_info.dialog_id.get_type() == DialogType::Channel &&
                             !td->dialog_manager_->is_broadcast_channel(message_info.dialog_id);
      message_info.reply_header = MessageReplyHeader(td, std::move(message->reply_to_), message_info.dialog_id,
                                                     message_info.message_id, message_info.date, can_have_thread);
      if (message->flags_ & telegram_api::message::VIA_BOT_ID_MASK) {
        message_info.via_bot_user_id = UserId(message->via_bot_id_);
        if (!message_info.via_bot_user_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << message_info.via_bot_user_id << " from " << source;
          message_info.via_bot_user_id = UserId();
        }
      }
      if (message->flags2_ & telegram_api::message::VIA_BUSINESS_BOT_ID_MASK) {
        message_info.via_business_bot_user_id = UserId(message->via_business_bot_id_);
        if (!message_info.via_business_bot_user_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << message_info.via_business_bot_user_id << " from " << source;
          message_info.via_business_bot_user_id = UserId();
        }
      }
      message_info.view_count = message->views_;
      message_info.forward_count = message->forwards_;
      message_info.reply_info = std::move(message->replies_);
      message_info.reactions = std::move(message->reactions_);
      message_info.edit_date = message->edit_date_;
      message_info.media_album_id = message->grouped_id_;
      message_info.ttl_period = message->ttl_period_;
      message_info.is_outgoing = message->out_;
      message_info.is_silent = message->silent_;
      message_info.is_channel_post = message->post_;
      message_info.is_legacy = message->legacy_;
      message_info.hide_edit_date = message->edit_hide_;
      message_info.is_from_scheduled = message->from_scheduled_;
      message_info.is_from_offline = message->offline_;
      message_info.is_pinned = message->pinned_;
      message_info.noforwards = message->noforwards_;
      message_info.has_mention = message->mentioned_;
      message_info.has_unread_content = message->media_unread_;
      message_info.invert_media = message->invert_media_;

      bool is_content_read = true;
      if (!td->auth_manager_->is_bot()) {
        if (is_scheduled) {
          is_content_read = false;
        } else if (td->messages_manager_->is_message_auto_read(message_info.dialog_id, message_info.is_outgoing)) {
          is_content_read = true;
        } else {
          is_content_read = !message_info.has_unread_content;
        }
      }
      auto new_source = PSTRING() << MessageFullId(message_info.dialog_id, message_info.message_id) << " sent by "
                                  << message_info.sender_dialog_id << " from " << source;
      message_info.content = get_message_content(
          td,
          get_message_text(td->user_manager_.get(), std::move(message->message_), std::move(message->entities_), true,
                           td->auth_manager_->is_bot(),
                           message_info.forward_header ? message_info.forward_header->date_ : message_info.date,
                           message_info.media_album_id != 0, new_source.c_str()),
          std::move(message->media_), message_info.dialog_id, message_info.date, is_content_read,
          message_info.via_bot_user_id, &message_info.ttl, &message_info.disable_web_page_preview, new_source.c_str());
      message_info.reply_markup = std::move(message->reply_markup_);
      message_info.restriction_reasons = get_restriction_reasons(std::move(message->restriction_reason_));
      message_info.author_signature = std::move(message->post_author_);
      message_info.sender_boost_count = message->from_boosts_applied_;
      if (message->saved_peer_id_ != nullptr) {
        message_info.saved_messages_topic_id = SavedMessagesTopicId(DialogId(message->saved_peer_id_));
      }
      break;
    }
    case telegram_api::messageService::ID: {
      auto message = move_tl_object_as<telegram_api::messageService>(message_ptr);

      message_info.dialog_id = DialogId(message->peer_id_);
      if (message->from_id_ != nullptr) {
        message_info.sender_dialog_id = DialogId(message->from_id_);
      } else {
        message_info.sender_dialog_id = message_info.dialog_id;
      }
      message_info.date = message->date_;
      message_info.ttl_period = message->ttl_period_;
      message_info.is_outgoing = message->out_;
      message_info.is_silent = message->silent_;
      message_info.is_channel_post = message->post_;
      message_info.is_legacy = message->legacy_;
      message_info.has_mention = message->mentioned_;
      message_info.has_unread_content = message->media_unread_;
      bool can_have_thread = message_info.dialog_id.get_type() == DialogType::Channel &&
                             !td->dialog_manager_->is_broadcast_channel(message_info.dialog_id);
      message_info.reply_header = MessageReplyHeader(td, std::move(message->reply_to_), message_info.dialog_id,
                                                     message_info.message_id, message_info.date, can_have_thread);
      message_info.content =
          get_action_message_content(td, std::move(message->action_), message_info.dialog_id, message_info.date,
                                     message_info.reply_header.replied_message_info_);
      message_info.reply_header.replied_message_info_ = RepliedMessageInfo();
      message_info.reply_header.story_full_id_ = StoryFullId();
      if (message_info.dialog_id == td->dialog_manager_->get_my_dialog_id()) {
        message_info.saved_messages_topic_id = SavedMessagesTopicId(message_info.dialog_id);
      }
      break;
    }
    default:
      UNREACHABLE();
      break;
  }
  if (message_info.sender_dialog_id.is_valid() && message_info.sender_dialog_id.get_type() == DialogType::User) {
    message_info.sender_user_id = message_info.sender_dialog_id.get_user_id();
    message_info.sender_dialog_id = DialogId();
  }
  return message_info;
}

std::pair<DialogId, unique_ptr<MessagesManager::Message>> MessagesManager::create_message(
    Td *td, MessageInfo &&message_info, bool is_channel_message, bool is_business_message, const char *source) {
  DialogId dialog_id = message_info.dialog_id;
  MessageId message_id = message_info.message_id;
  if ((!message_id.is_valid() && !message_id.is_valid_scheduled()) || !dialog_id.is_valid()) {
    if (message_id != MessageId() || dialog_id != DialogId()) {
      LOG(ERROR) << "Receive " << message_id << " in " << dialog_id;
    }
    return {DialogId(), nullptr};
  }
  if (message_id.is_yet_unsent() || message_id.is_local()) {
    LOG(ERROR) << "Receive " << message_id;
    return {DialogId(), nullptr};
  }

  CHECK(message_info.content != nullptr);

  auto dialog_type = dialog_id.get_type();
  UserId sender_user_id = message_info.sender_user_id;
  DialogId sender_dialog_id = message_info.sender_dialog_id;
  if (!sender_user_id.is_valid()) {
    if (sender_user_id != UserId()) {
      LOG(ERROR) << "Receive invalid " << sender_user_id;
      sender_user_id = UserId();
    }
    if (!td->dialog_manager_->is_broadcast_channel(dialog_id) && td->auth_manager_->is_bot()) {
      if (dialog_id == sender_dialog_id) {
        td->user_manager_->add_anonymous_bot_user();
      } else {
        td->user_manager_->add_service_notifications_user();
        td->user_manager_->add_channel_bot_user();
      }
    }
  }
  if (sender_dialog_id.is_valid()) {
    if (dialog_type == DialogType::User || dialog_type == DialogType::SecretChat) {
      LOG(ERROR) << "Receive " << message_id << " sent by " << sender_dialog_id << " in " << dialog_id;
      return {DialogId(), nullptr};
    }
  } else if (sender_dialog_id != DialogId()) {
    LOG(ERROR) << "Receive invalid " << sender_dialog_id;
    sender_dialog_id = DialogId();
  }
  if (message_id.is_scheduled()) {
    is_channel_message = (dialog_type == DialogType::Channel);
  }

  LOG_IF(ERROR, is_channel_message != (dialog_type == DialogType::Channel))
      << "Receive wrong is_channel_message for " << message_id << " in " << dialog_id;

  bool is_channel_post = message_info.is_channel_post;
  if (is_channel_post && !td->dialog_manager_->is_broadcast_channel(dialog_id)) {
    LOG(ERROR) << "Receive is_channel_post for " << message_id << " in " << dialog_id;
    is_channel_post = false;
  }

  UserId my_id = td->user_manager_->get_my_id();
  DialogId my_dialog_id = DialogId(my_id);
  if (dialog_id == my_dialog_id && (sender_user_id != my_id || sender_dialog_id.is_valid())) {
    LOG(ERROR) << "Receive " << sender_user_id << "/" << sender_dialog_id << " as a sender of " << message_id
               << " instead of self";
    sender_user_id = my_id;
    sender_dialog_id = DialogId();
  }

  bool is_outgoing = message_info.is_outgoing;
  bool supposed_to_be_outgoing = sender_user_id == my_id && !(dialog_id == my_dialog_id && !message_id.is_scheduled());
  if (sender_user_id.is_valid() && supposed_to_be_outgoing != is_outgoing && !is_business_message) {
    LOG(ERROR) << "Receive wrong out flag for " << message_id << " in " << dialog_id << ": me is " << my_id
               << ", but message is from " << sender_user_id;
    is_outgoing = supposed_to_be_outgoing;

    /*
    // it is useless to call getChannelDifference, because the channel PTS will be increased already
    if (dialog_type == DialogType::Channel && !running_get_difference_ && !running_get_channel_difference(dialog_id) &&
        !message_id.is_scheduled() && get_channel_difference_to_log_event_id_.count(dialog_id) == 0) {
      // it is safer to completely ignore the message and re-get it through getChannelDifference
      Dialog *d = get_dialog(dialog_id);
      if (d != nullptr) {
        schedule_get_channel_difference(dialog_id, 0, message_id, 0.001, source);
        return {DialogId(), nullptr};
      }
    }
    */
  }

  int32 date = message_info.date;
  if (date <= 0) {
    LOG(ERROR) << "Wrong date = " << date << " received in " << message_id << " in " << dialog_id;
    date = 1;
  }

  MessageId top_thread_message_id = message_info.reply_header.top_thread_message_id_;
  bool is_topic_message = message_info.reply_header.is_topic_message_;
  auto reply_to_story_full_id = message_info.reply_header.story_full_id_;
  if (reply_to_story_full_id != StoryFullId()) {
    auto story_dialog_id = reply_to_story_full_id.get_dialog_id();
    if (story_dialog_id != my_dialog_id && story_dialog_id != dialog_id &&
        story_dialog_id != DialogId(sender_user_id)) {
      LOG(ERROR) << "Receive reply to " << reply_to_story_full_id << " in " << dialog_id;
      reply_to_story_full_id = {};
    }
  }

  UserId via_bot_user_id = message_info.via_bot_user_id;
  if (!via_bot_user_id.is_valid()) {
    via_bot_user_id = UserId();
  }

  int32 edit_date = message_info.edit_date;
  if (edit_date < 0) {
    LOG(ERROR) << "Wrong edit_date = " << edit_date << " received in " << message_id << " in " << dialog_id;
    edit_date = 0;
  }

  auto content_type = message_info.content->get_type();
  if (content_type == MessageContentType::Sticker &&
      get_message_content_sticker_type(td, message_info.content.get()) == StickerType::CustomEmoji) {
    LOG(INFO) << "Replace emoji sticker with an empty message";
    message_info.content =
        create_text_message_content("Invalid sticker", {}, WebPageId(), false, false, false, string());
    message_info.invert_media = false;
    content_type = message_info.content->get_type();
  }

  bool hide_edit_date = message_info.hide_edit_date;
  if (hide_edit_date && td->auth_manager_->is_bot()) {
    hide_edit_date = false;
  }
  if (hide_edit_date && content_type == MessageContentType::LiveLocation) {
    hide_edit_date = false;
  }

  int32 ttl_period = message_info.ttl_period;
  if (ttl_period < 0 || (message_id.is_scheduled() && ttl_period != 0)) {
    LOG(ERROR) << "Wrong auto-delete time " << ttl_period << " received in " << message_id << " in " << dialog_id;
    ttl_period = 0;
  }

  auto ttl = message_info.ttl;
  bool is_content_secret = ttl.is_secret_message_content(content_type);  // must be calculated before TTL is adjusted
  if (!ttl.is_empty()) {
    if (!ttl.is_valid() || message_id.is_scheduled()) {
      LOG(ERROR) << "Wrong " << ttl << " received in " << message_id << " in " << dialog_id;
      ttl = {};
    } else {
      ttl.ensure_at_least(get_message_content_duration(message_info.content.get(), td) + 1);
    }
  }

  if (message_id.is_scheduled()) {
    if (message_info.reply_info != nullptr) {
      LOG(ERROR) << "Receive " << message_id << " in " << dialog_id << " with " << to_string(message_info.reply_info);
      message_info.reply_info = nullptr;
    }
    if (message_info.reactions != nullptr) {
      LOG(ERROR) << "Receive " << message_id << " in " << dialog_id << " with " << to_string(message_info.reactions);
      message_info.reactions = nullptr;
    }
  }
  int32 view_count = message_info.view_count;
  if (view_count < 0) {
    LOG(ERROR) << "Wrong view_count = " << view_count << " received in " << message_id << " in " << dialog_id;
    view_count = 0;
  }
  int32 forward_count = message_info.forward_count;
  if (forward_count < 0) {
    LOG(ERROR) << "Wrong forward_count = " << forward_count << " received in " << message_id << " in " << dialog_id;
    forward_count = 0;
  }
  MessageReplyInfo reply_info(td, std::move(message_info.reply_info), td->auth_manager_->is_bot());
  if (!top_thread_message_id.is_valid() &&
      td->messages_manager_->is_thread_message(dialog_id, message_id, reply_info, content_type)) {
    top_thread_message_id = message_id;
    is_topic_message = (content_type == MessageContentType::TopicCreate);
  }
  if (top_thread_message_id.is_valid() && dialog_type != DialogType::Channel) {
    // just in case
    top_thread_message_id = MessageId();
  }
  if (!top_thread_message_id.is_valid()) {
    // just in case
    is_topic_message = false;
  }
  auto reactions =
      MessageReactions::get_message_reactions(td, std::move(message_info.reactions), td->auth_manager_->is_bot());
  if (reactions != nullptr) {
    reactions->sort_reactions(td->messages_manager_->active_reaction_pos_);
    reactions->fix_chosen_reaction();
    reactions->fix_my_recent_chooser_dialog_id(my_dialog_id);
  }

  bool has_forward_info = message_info.forward_header != nullptr;
  bool noforwards = message_info.noforwards;
  bool is_expired = is_expired_message_content(content_type);
  if (is_expired) {
    CHECK(ttl.is_empty());  // self-destruct time is ignored/set to 0 if the message has already been expired
    message_info.reply_header.replied_message_info_ = {};
    reply_to_story_full_id = StoryFullId();
    noforwards = false;
    is_content_secret = false;
    message_info.invert_media = false;
  }

  bool is_pinned = message_info.is_pinned;
  if (is_pinned && message_id.is_scheduled()) {
    LOG(ERROR) << "Receive pinned " << message_id << " in " << dialog_id;
    is_pinned = false;
  }

  bool has_mention =
      message_info.has_mention || (content_type == MessageContentType::PinMessage &&
                                   td->option_manager_->get_option_boolean("process_pinned_messages_as_mentions"));

  LOG(INFO) << "Receive " << message_id << " in " << dialog_id << " from " << sender_user_id << "/" << sender_dialog_id;

  auto message = make_unique<Message>();
  message->message_id = message_id;
  message->sender_user_id = sender_user_id;
  message->sender_dialog_id = sender_dialog_id;
  message->date = date;
  message->ttl_period = ttl_period;
  message->ttl = ttl;
  message->disable_web_page_preview = message_info.disable_web_page_preview;
  message->edit_date = edit_date;
  message->random_id = message_info.random_id;
  message->forward_info = MessageForwardInfo::get_message_forward_info(td, std::move(message_info.forward_header));
  message->replied_message_info = std::move(message_info.reply_header.replied_message_info_);
  message->top_thread_message_id = top_thread_message_id;
  message->is_topic_message = is_topic_message;
  message->via_bot_user_id = via_bot_user_id;
  message->via_business_bot_user_id = message_info.via_business_bot_user_id;
  message->reply_to_story_full_id = reply_to_story_full_id;
  message->restriction_reasons = std::move(message_info.restriction_reasons);
  message->author_signature = std::move(message_info.author_signature);
  message->sender_boost_count = message_info.sender_boost_count;
  message->saved_messages_topic_id = message_info.saved_messages_topic_id;
  message->is_outgoing = is_outgoing;
  message->is_channel_post = is_channel_post;
  message->contains_mention =
      !is_outgoing && dialog_type != DialogType::User && !is_expired && has_mention && !td->auth_manager_->is_bot();
  message->contains_unread_mention =
      !message_id.is_scheduled() && message_id.is_server() && message->contains_mention &&
      message_info.has_unread_content &&
      (dialog_type == DialogType::Chat ||
       (dialog_type == DialogType::Channel && !td->dialog_manager_->is_broadcast_channel(dialog_id)));
  message->disable_notification = message_info.is_silent;
  message->is_content_secret = is_content_secret;
  message->hide_edit_date = hide_edit_date;
  message->is_from_scheduled = message_info.is_from_scheduled;
  message->is_from_offline = message_info.is_from_offline;
  message->is_pinned = is_pinned;
  message->noforwards = noforwards;
  message->interaction_info_update_date = G()->unix_time();
  message->view_count = view_count;
  message->forward_count = forward_count;
  message->reply_info = std::move(reply_info);
  message->reactions = std::move(reactions);
  message->legacy_layer = (message_info.is_legacy ? MTPROTO_LAYER : 0);
  message->invert_media = message_info.invert_media;
  message->content = std::move(message_info.content);
  message->reply_markup = get_reply_markup(std::move(message_info.reply_markup), td->auth_manager_->is_bot(), false,
                                           message->contains_mention || dialog_type == DialogType::User);
  if (message->reply_markup != nullptr && is_expired) {
    // just in case
    if (message->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
      message->had_reply_markup = true;
    }
    message->reply_markup = nullptr;
  }

  if (message_info.media_album_id != 0) {
    if (!is_allowed_media_group_content(content_type)) {
      if (content_type != MessageContentType::Unsupported) {
        LOG(ERROR) << "Receive media group identifier " << message_info.media_album_id << " in " << message_id
                   << " from " << dialog_id << " with content "
                   << oneline(to_string(
                          td->messages_manager_->get_message_message_content_object(dialog_id, message.get())));
      }
    } else {
      message->media_album_id = message_info.media_album_id;
    }
  }

  if (message->forward_info == nullptr && has_forward_info) {
    message->had_forward_info = true;
  }

  if (dialog_id == my_dialog_id) {
    if (!message->saved_messages_topic_id.is_valid()) {
      LOG(ERROR) << "Receive no Saved Messages topic for " << message_id << " in " << dialog_id;
      message->saved_messages_topic_id = SavedMessagesTopicId(my_dialog_id, message->forward_info.get(), DialogId());
    }
  } else {
    if (message->saved_messages_topic_id.is_valid()) {
      LOG(ERROR) << "Receive Saved Messages " << message_info.saved_messages_topic_id << " for " << message_id << " in "
                 << dialog_id;
      message->saved_messages_topic_id = SavedMessagesTopicId();
    }
  }

  Dependencies dependencies;
  td->messages_manager_->add_message_dependencies(dependencies, message.get());
  for (auto dependent_dialog_id : dependencies.get_dialog_ids()) {
    if (dependent_dialog_id != dialog_id) {
      td->dialog_manager_->force_create_dialog(dependent_dialog_id, source, true);
    }
  }

  return {dialog_id, std::move(message)};
}

MessageId MessagesManager::find_old_message_id(DialogId dialog_id, MessageId message_id) const {
  if (message_id.is_scheduled()) {
    CHECK(message_id.is_scheduled_server());
    auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
    if (dialog_it != update_scheduled_message_ids_.end()) {
      auto it = dialog_it->second.find(message_id.get_scheduled_server_message_id());
      if (it != dialog_it->second.end()) {
        return it->second;
      }
    }
  } else {
    CHECK(message_id.is_server());
    auto it = update_message_ids_.find(MessageFullId(dialog_id, message_id));
    if (it != update_message_ids_.end()) {
      return it->second;
    }
  }
  return MessageId();
}

void MessagesManager::delete_update_message_id(DialogId dialog_id, MessageId message_id) {
  if (message_id.is_scheduled()) {
    CHECK(message_id.is_scheduled_server());
    auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
    CHECK(dialog_it != update_scheduled_message_ids_.end());
    auto erased_count = dialog_it->second.erase(message_id.get_scheduled_server_message_id());
    CHECK(erased_count > 0);
    if (dialog_it->second.empty()) {
      update_scheduled_message_ids_.erase(dialog_it);
    }
  } else {
    CHECK(message_id.is_server());
    auto erased_count = update_message_ids_.erase(MessageFullId(dialog_id, message_id));
    CHECK(erased_count > 0);
  }
}

MessageFullId MessagesManager::on_get_message(tl_object_ptr<telegram_api::Message> message_ptr, bool from_update,
                                              bool is_channel_message, bool is_scheduled, const char *source) {
  return on_get_message(parse_telegram_api_message(td_, std::move(message_ptr), is_scheduled, source), from_update,
                        is_channel_message, source);
}

MessageFullId MessagesManager::on_get_message(MessageInfo &&message_info, const bool from_update,
                                              const bool is_channel_message, const char *source) {
  DialogId dialog_id;
  unique_ptr<Message> new_message;
  std::tie(dialog_id, new_message) = create_message(td_, std::move(message_info), is_channel_message, false, source);
  if (new_message == nullptr) {
    return MessageFullId();
  }
  MessageId message_id = new_message->message_id;

  bool need_update = from_update;
  bool need_update_dialog_pos = false;

  Dialog *d = get_dialog_force(dialog_id, source);

  MessageId old_message_id = find_old_message_id(dialog_id, message_id);
  bool is_sent_message = false;
  if (old_message_id.is_valid() || old_message_id.is_valid_scheduled()) {
    LOG(INFO) << "Found temporary " << old_message_id << " for " << MessageFullId{dialog_id, message_id};
    CHECK(d != nullptr);

    if (!from_update && !message_id.is_scheduled()) {
      if (message_id <= d->last_new_message_id) {
        if (get_message_force(d, message_id, "receive missed unsent message not from update") != nullptr) {
          LOG(ERROR) << "New " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                     << " has identifier less than last_new_message_id = " << d->last_new_message_id;
          return MessageFullId();
        }
        // if there is no message yet, then it is likely was missed because of a server bug and is being repaired via
        // get_message_from_server from after_get_difference
        // TODO move to INFO
        LOG(ERROR) << "Receive " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                   << " with identifier less than last_new_message_id = " << d->last_new_message_id
                   << " and trying to add it anyway";
      } else {
        LOG(INFO) << "Ignore " << old_message_id << "/" << message_id << " received not through update from " << source
                  << ": " << oneline(to_string(get_message_object(dialog_id, new_message.get(), "on_get_message")));
        if (dialog_id.get_type() == DialogType::Channel &&
            td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
          schedule_get_channel_difference(dialog_id, 0, message_id, 0.001, "on_get_message");
        }
        return MessageFullId();
      }
    }

    delete_update_message_id(dialog_id, message_id);

    if (!new_message->is_outgoing && dialog_id != td_->dialog_manager_->get_my_dialog_id()) {
      // sent message is not from me
      LOG(ERROR) << "Sent in " << dialog_id << " " << message_id << " is sent by " << new_message->sender_user_id << "/"
                 << new_message->sender_dialog_id;
      return MessageFullId();
    }

    // must be called before delete_message
    update_reply_to_message_id(dialog_id, old_message_id, message_id, true, "on_get_message");

    being_readded_message_id_ = {dialog_id, old_message_id};
    unique_ptr<Message> old_message =
        delete_message(d, old_message_id, false, &need_update_dialog_pos, "add sent message");
    if (old_message == nullptr) {
      delete_sent_message_on_server(dialog_id, message_id, old_message_id);
      being_readded_message_id_ = MessageFullId();
      return MessageFullId();
    }
    old_message_id = old_message->message_id;

    need_update = false;

    new_message->message_id = old_message_id;
    update_message(d, old_message.get(), std::move(new_message), false);
    new_message = std::move(old_message);

    auto reply_message_full_id = new_message->replied_message_info.get_reply_message_full_id(dialog_id, false);
    auto reply_message_id = reply_message_full_id.get_message_id();
    if (reply_message_id.is_valid() && reply_message_id.is_yet_unsent()) {
      set_message_reply(d, new_message.get(), MessageInputReplyTo(), false);
    }

    new_message->message_id = message_id;
    send_update_message_send_succeeded(d, old_message_id, new_message.get(), &need_update_dialog_pos);

    if (!message_id.is_scheduled()) {
      is_sent_message = true;
    }
  }
  if (d == nullptr) {
    d = add_dialog_for_new_message(dialog_id, from_update, &need_update_dialog_pos, source);
  }

  const Message *m = add_message_to_dialog(d, std::move(new_message), false, from_update, &need_update,
                                           &need_update_dialog_pos, source);
  being_readded_message_id_ = MessageFullId();
  if (m == nullptr) {
    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "on_get_message");
    }
    if (old_message_id.is_valid() || old_message_id.is_valid_scheduled()) {
      if (!old_message_id.is_valid() || !message_id.is_valid() || old_message_id <= message_id) {
        LOG(ERROR) << "Failed to add just sent " << old_message_id << " to " << dialog_id << " as " << message_id
                   << " from " << source << ": " << debug_add_message_to_dialog_fail_reason_;
      }
      send_update_delete_messages(dialog_id, {message_id.get()}, true);
    }

    return MessageFullId();
  }

  if (need_update) {
    send_update_new_message(d, m);
  }

  if (is_sent_message) {
    try_add_active_live_location(dialog_id, m);

    // add_message_to_dialog will not update counts, because need_update == false
    update_message_count_by_index(d, +1, m);
  }

  if (is_sent_message || (need_update && !message_id.is_scheduled())) {
    update_reply_count_by_message(d, +1, m);
    update_forward_count(dialog_id, m);
  }

  if (dialog_id.get_type() == DialogType::Channel &&
      !td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    auto message = delete_message(d, message_id, false, &need_update_dialog_pos, "get a message in inaccessible chat");
    CHECK(message.get() == m);
    send_update_delete_messages(dialog_id, {m->message_id.get()}, false);
    // don't need to update chat position
    return MessageFullId();
  }

  if (m->message_id.is_scheduled()) {
    send_update_chat_has_scheduled_messages(d, false);
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_message");
  }

  // set dialog reply markup only after updateNewMessage and updateChatLastMessage are sent
  if (need_update && m->reply_markup != nullptr && !m->message_id.is_scheduled() &&
      m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard && m->reply_markup->is_personal &&
      !td_->auth_manager_->is_bot()) {
    set_dialog_reply_markup(d, message_id);
  }

  if (from_update) {
    auto it = pending_created_dialogs_.find(dialog_id);
    if (it != pending_created_dialogs_.end()) {
      auto pending_created_dialog = std::move(it->second);
      pending_created_dialogs_.erase(it);

      if (pending_created_dialog.chat_promise_) {
        pending_created_dialog.chat_promise_.set_value(td_api::make_object<td_api::createdBasicGroupChat>(
            get_chat_id_object(dialog_id, "on_get_message"), std::move(pending_created_dialog.failed_to_add_members_)));
      } else {
        pending_created_dialog.channel_promise_.set_value(get_chat_object(d, "on_get_message"));
      }
    }
  }

  return MessageFullId(dialog_id, message_id);
}

void MessagesManager::set_dialog_last_message_id(Dialog *d, MessageId last_message_id, const char *source,
                                                 const Message *m) {
  CHECK(!last_message_id.is_scheduled());
  CHECK(!td_->auth_manager_->is_bot());

  LOG(INFO) << "Set " << d->dialog_id << " last message to " << last_message_id << " from " << source;
  d->last_message_id = last_message_id;

  if (m != nullptr) {
    d->last_media_album_id = m->media_album_id;
  } else if (!last_message_id.is_valid()) {
    d->last_media_album_id = 0;
  } else {
    m = get_message(d, last_message_id);
    if (m == nullptr) {
      LOG(ERROR) << "Failed to find last " << last_message_id << " in " << d->dialog_id;
      d->last_media_album_id = 0;
    } else {
      d->last_media_album_id = m->media_album_id;
    }
  }
  if (!last_message_id.is_valid()) {
    auto it = dialog_suffix_load_queries_.find(d->dialog_id);
    if (it != dialog_suffix_load_queries_.end()) {
      it->second->suffix_load_first_message_id_ = MessageId();
      it->second->suffix_load_done_ = false;
    }
  }
  if (last_message_id.is_valid() && d->delete_last_message_date != 0) {
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;
    on_dialog_updated(d->dialog_id, "update_delete_last_message_date");
  }
  d->pending_order = DEFAULT_ORDER;
}

void MessagesManager::set_dialog_first_database_message_id(Dialog *d, MessageId first_database_message_id,
                                                           const char *source) {
  CHECK(!first_database_message_id.is_scheduled());
  CHECK(!td_->auth_manager_->is_bot());
  if (first_database_message_id == d->first_database_message_id) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " first database message to " << first_database_message_id << " from "
            << source;
  d->first_database_message_id = first_database_message_id;
  on_dialog_updated(d->dialog_id, "set_dialog_first_database_message_id");
}

void MessagesManager::set_dialog_last_database_message_id(Dialog *d, MessageId last_database_message_id,
                                                          const char *source, bool is_loaded_from_database) {
  CHECK(!last_database_message_id.is_scheduled());
  CHECK(!td_->auth_manager_->is_bot());
  if (last_database_message_id == d->last_database_message_id) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " last database message to " << last_database_message_id << " from " << source;
  d->debug_set_dialog_last_database_message_id = source;
  d->last_database_message_id = last_database_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_database_message_id");
  }
}

void MessagesManager::remove_dialog_newer_messages(Dialog *d, MessageId from_message_id, const char *source) {
  LOG(INFO) << "Remove messages in " << d->dialog_id << " newer than " << from_message_id << " from " << source;
  CHECK(!d->last_new_message_id.is_valid());
  CHECK(!td_->auth_manager_->is_bot());

  // the function is called to handle channel gaps, and hence must always delete all database messages,
  // even first_database_message_id and last_database_message_id are empty
  delete_all_dialog_messages_from_database(d, MessageId::max(), "remove_dialog_newer_messages");
  set_dialog_first_database_message_id(d, MessageId(), "remove_dialog_newer_messages");
  set_dialog_last_database_message_id(d, MessageId(), source);
  if (d->dialog_id.get_type() != DialogType::SecretChat && !d->is_empty) {
    d->have_full_history = false;
    d->have_full_history_source = 0;
  }
  invalidate_message_indexes(d);

  auto to_delete_message_ids = d->ordered_messages.find_newer_messages(from_message_id);
  td::remove_if(to_delete_message_ids, [](MessageId message_id) { return message_id.is_yet_unsent(); });
  if (!to_delete_message_ids.empty()) {
    LOG(INFO) << "Delete " << format::as_array(to_delete_message_ids) << " newer than " << from_message_id << " in "
              << d->dialog_id << " from " << source;

    vector<int64> deleted_message_ids;
    bool need_update_dialog_pos = false;
    for (auto message_id : to_delete_message_ids) {
      auto message = delete_message(d, message_id, false, &need_update_dialog_pos, "remove_dialog_newer_messages");
      if (message != nullptr) {
        deleted_message_ids.push_back(message->message_id.get());
      }
    }
    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "remove_dialog_newer_messages");
    }
    send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), false);
  }
}

void MessagesManager::set_dialog_last_new_message_id(Dialog *d, MessageId last_new_message_id, const char *source) {
  CHECK(!last_new_message_id.is_scheduled());
  CHECK(!td_->auth_manager_->is_bot());

  LOG_CHECK(last_new_message_id > d->last_new_message_id)
      << last_new_message_id << " " << d->last_new_message_id << " " << source;
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat || last_new_message_id.is_server());
  if (!d->last_new_message_id.is_valid()) {
    remove_dialog_newer_messages(d, last_new_message_id, source);

    auto last_new_message = get_message(d, last_new_message_id);
    if (last_new_message != nullptr) {
      add_message_to_database(d, last_new_message, source);
      set_dialog_first_database_message_id(d, last_new_message_id, source);
      set_dialog_last_database_message_id(d, last_new_message_id, source);
      try_restore_dialog_reply_markup(d, last_new_message);
    }
  }

  LOG(INFO) << "Set " << d->dialog_id << " last new message to " << last_new_message_id << " from " << source;
  d->last_new_message_id = last_new_message_id;
  on_dialog_updated(d->dialog_id, source);
}

void MessagesManager::set_dialog_last_clear_history_date(Dialog *d, int32 date, MessageId last_clear_history_message_id,
                                                         const char *source, bool is_loaded_from_database) {
  CHECK(!last_clear_history_message_id.is_scheduled());

  if (d->last_clear_history_message_id == last_clear_history_message_id && d->last_clear_history_date == date) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " last clear history date to " << date << " of "
            << last_clear_history_message_id << " from " << source;
  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_.erase(d->last_clear_history_message_id);
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }

  d->last_clear_history_date = date;
  d->last_clear_history_message_id = last_clear_history_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_clear_history_date");
  }

  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_[d->last_clear_history_message_id] = d->dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
}

void MessagesManager::set_dialog_unread_mention_count(Dialog *d, int32 unread_mention_count) {
  CHECK(d->unread_mention_count != unread_mention_count);
  CHECK(unread_mention_count >= 0);

  d->unread_mention_count = unread_mention_count;
  d->message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadMention)] = unread_mention_count;
}

void MessagesManager::set_dialog_unread_reaction_count(Dialog *d, int32 unread_reaction_count) {
  CHECK(d->unread_reaction_count != unread_reaction_count);
  CHECK(unread_reaction_count >= 0);

  d->unread_reaction_count = unread_reaction_count;
  d->message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadReaction)] = unread_reaction_count;
}

void MessagesManager::set_dialog_is_empty(Dialog *d, const char *source) {
  LOG(INFO) << "Set " << d->dialog_id << " is_empty to true from " << source;
  CHECK(d->have_full_history);
  if (!d->is_empty && d->order != DEFAULT_ORDER) {
    td_->dialog_manager_->reload_dialog_info_full(d->dialog_id, "set_dialog_is_empty");
  }
  d->is_empty = true;

  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d, max_message_id, -1, "set_dialog_is_empty");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "set_dialog_is_empty");
    }
  }
  if (d->unread_mention_count > 0) {
    set_dialog_unread_mention_count(d, 0);
    send_update_chat_unread_mention_count(d);
  }
  if (d->unread_reaction_count > 0) {
    set_dialog_unread_reaction_count(d, 0);
    send_update_chat_unread_reaction_count(d, "set_dialog_is_empty");
  }
  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);
  if (d->notification_info != nullptr) {
    d->notification_info->notification_id_to_message_id_.clear();
  }

  if (d->delete_last_message_date != 0) {
    if (d->is_last_message_deleted_locally && d->last_clear_history_date == 0) {
      set_dialog_last_clear_history_date(d, d->delete_last_message_date, d->deleted_last_message_id,
                                         "set_dialog_is_empty");
    }
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;

    on_dialog_updated(d->dialog_id, "set_dialog_is_empty");
  }
  d->pending_order = DEFAULT_ORDER;
  if (d->last_database_message_id.is_valid()) {
    set_dialog_first_database_message_id(d, MessageId(), "set_dialog_is_empty");
    set_dialog_last_database_message_id(d, MessageId(), "set_dialog_is_empty");
  }

  update_dialog_pos(d, source);
}

bool MessagesManager::is_dialog_pinned(DialogListId dialog_list_id, DialogId dialog_id) const {
  if (get_dialog_pinned_order(dialog_list_id, dialog_id) != DEFAULT_ORDER) {
    return true;
  }
  if (dialog_list_id.is_filter()) {
    return td_->dialog_filter_manager_->is_dialog_pinned(dialog_list_id.get_filter_id(), dialog_id);
  }
  return false;
}

int64 MessagesManager::get_dialog_pinned_order(DialogListId dialog_list_id, DialogId dialog_id) const {
  return get_dialog_pinned_order(get_dialog_list(dialog_list_id), dialog_id);
}

int64 MessagesManager::get_dialog_pinned_order(const DialogList *list, DialogId dialog_id) {
  if (list != nullptr && !list->pinned_dialog_id_orders_.empty()) {
    auto it = list->pinned_dialog_id_orders_.find(dialog_id);
    if (it != list->pinned_dialog_id_orders_.end()) {
      return it->second;
    }
  }
  return DEFAULT_ORDER;
}

bool MessagesManager::set_dialog_is_pinned(DialogId dialog_id, bool is_pinned) {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return set_dialog_is_pinned(DialogListId(d->folder_id), d, is_pinned);
}

// only removes the Dialog from the dialog list, but changes nothing in the corresponding DialogFilter
bool MessagesManager::set_dialog_is_pinned(DialogListId dialog_list_id, Dialog *d, bool is_pinned,
                                           bool need_update_dialog_lists) {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  CHECK(d != nullptr);
  if (d->order == DEFAULT_ORDER && is_pinned) {
    // the chat can't be pinned
    return false;
  }

  auto positions = get_dialog_positions(d);
  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return false;
  }
  if (!list->are_pinned_dialogs_inited_) {
    return false;
  }
  auto dialog_id = d->dialog_id;
  auto is_changed_dialog = [dialog_id](const DialogDate &dialog_date) {
    return dialog_date.get_dialog_id() == dialog_id;
  };
  if (is_pinned) {
    if (!list->pinned_dialogs_.empty() && is_changed_dialog(list->pinned_dialogs_[0])) {
      return false;
    }
    auto order = get_next_pinned_dialog_order();
    DialogDate dialog_date(order, dialog_id);
    add_to_top_if(list->pinned_dialogs_, list->pinned_dialogs_.size() + 1, dialog_date, is_changed_dialog);
    auto it = list->pinned_dialog_id_orders_.find(dialog_id);
    if (it != list->pinned_dialog_id_orders_.end()) {
      CHECK(list->pinned_dialogs_[0] != dialog_date);
      list->pinned_dialogs_[0] = dialog_date;
      it->second = order;
    } else {
      CHECK(list->pinned_dialogs_[0] == dialog_date);
      list->pinned_dialog_id_orders_.emplace(dialog_id, order);
    }
  } else {
    if (!td::remove_if(list->pinned_dialogs_, is_changed_dialog)) {
      return false;
    }
    list->pinned_dialog_id_orders_.erase(dialog_id);
  }

  LOG(INFO) << "Set " << d->dialog_id << " is pinned in " << dialog_list_id << " to " << is_pinned;

  save_pinned_folder_dialog_ids(*list);

  if (need_update_dialog_lists) {
    update_dialog_lists(d, std::move(positions), true, false, "set_dialog_is_pinned");
  }
  return true;
}

void MessagesManager::save_pinned_folder_dialog_ids(const DialogList &list) const {
  if (!list.dialog_list_id.is_folder() || !G()->use_message_database()) {
    return;
  }
  G()->td_db()->get_binlog_pmc()->set(
      PSTRING() << "pinned_dialog_ids" << list.dialog_list_id.get_folder_id().get(),
      implode(transform(list.pinned_dialogs_,
                        [](auto &pinned_dialog) { return PSTRING() << pinned_dialog.get_dialog_id().get(); }),
              ','));
}

void MessagesManager::set_dialog_reply_markup(Dialog *d, MessageId message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!message_id.is_scheduled());

  if (d->reply_markup_message_id != message_id) {
    on_dialog_updated(d->dialog_id, "set_dialog_reply_markup");
  }

  d->need_restore_reply_markup = false;

  if (d->reply_markup_message_id.is_valid() || message_id.is_valid()) {
    LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_reply_markup";
    d->reply_markup_message_id = message_id;
    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateChatReplyMarkup>(
                     get_chat_id_object(d->dialog_id, "updateChatReplyMarkup"), message_id.get()));
  }
}

void MessagesManager::try_restore_dialog_reply_markup(Dialog *d, const Message *m) {
  if (!d->need_restore_reply_markup || td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!m->message_id.is_scheduled());
  if (m->had_reply_markup) {
    LOG(INFO) << "Restore deleted reply markup in " << d->dialog_id;
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup != nullptr && m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard &&
             m->reply_markup->is_personal) {
    LOG(INFO) << "Restore reply markup in " << d->dialog_id << " to " << m->message_id;
    set_dialog_reply_markup(d, m->message_id);
  }
}

void MessagesManager::set_dialog_pinned_message_notification(Dialog *d, MessageId message_id, const char *source) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  if (d->notification_info == nullptr && message_id == MessageId()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());
  auto notification_info = add_dialog_notification_info(d);
  auto old_message_id = notification_info->pinned_message_notification_message_id_;
  if (old_message_id == message_id) {
    return;
  }
  VLOG(notifications) << "Change pinned message notification in " << d->dialog_id << " from " << old_message_id
                      << " to " << message_id;
  if (old_message_id.is_valid()) {
    auto m = get_message_force(d, old_message_id, source);
    if (m != nullptr && m->notification_id.is_valid() && is_message_notification_active(d, m)) {
      // Can't remove pinned_message_notification_message_id before the call,
      // because the notification needs to be still active inside remove_message_notification_id
      remove_message_notification_id(d, m, true, false, true);
      on_message_changed(d, m, false, source);
    } else {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_temporary_notification_by_object_id,
                         notification_info->mention_notification_group_.get_group_id(), old_message_id, false, source);
    }
  }
  notification_info->pinned_message_notification_message_id_ = message_id;
  on_dialog_updated(d->dialog_id, source);
}

void MessagesManager::remove_dialog_pinned_message_notification(Dialog *d, const char *source) {
  set_dialog_pinned_message_notification(d, MessageId(), source);
}

void MessagesManager::remove_scope_pinned_message_notifications(NotificationSettingsScope scope) {
  VLOG(notifications) << "Remove pinned message notifications in " << scope;
  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    Dialog *d = dialog.get();
    if (d->notification_settings.use_default_disable_pinned_message_notifications && d->notification_info != nullptr &&
        d->notification_info->mention_notification_group_.is_valid() &&
        d->notification_info->pinned_message_notification_message_id_.is_valid() &&
        td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id) == scope) {
      remove_dialog_pinned_message_notification(d, "remove_scope_pinned_message_notifications");
    }
  });
}

void MessagesManager::on_update_scope_mention_notifications(NotificationSettingsScope scope,
                                                            bool disable_mention_notifications) {
  VLOG(notifications) << "Remove mention notifications in " << scope;
  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    Dialog *d = dialog.get();
    if (d->notification_settings.use_default_disable_mention_notifications &&
        td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id) == scope) {
      if (!disable_mention_notifications) {
        update_dialog_mention_notification_count(d);
      } else {
        remove_dialog_mention_notifications(d);
      }
    }
  });
}

void MessagesManager::remove_dialog_mention_notifications(Dialog *d) {
  if (d->notification_info == nullptr || !d->notification_info->mention_notification_group_.is_valid() ||
      d->unread_mention_count == 0) {
    return;
  }
  CHECK(!d->being_added_message_id.is_valid());

  VLOG(notifications) << "Remove mention notifications in " << d->dialog_id;

  auto message_ids = find_dialog_messages(d, [](const Message *m) { return m->contains_unread_mention; });
  VLOG(notifications) << "Found unread mentions in " << message_ids;
  FlatHashSet<NotificationId, NotificationIdHash> removed_notification_ids_set;
  for (auto &message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->message_id.is_valid());
    if (m->notification_id.is_valid() && is_message_notification_active(d, m) &&
        is_from_mention_notification_group(m)) {
      removed_notification_ids_set.insert(m->notification_id);
    }
  }

  auto notification_group_id = d->notification_info->mention_notification_group_.get_group_id();
  CHECK(notification_group_id.is_valid());
  message_ids = td_->notification_manager_->get_notification_group_message_ids(notification_group_id);
  VLOG(notifications) << "Found active mention notifications in " << message_ids;
  for (auto &message_id : message_ids) {
    CHECK(!message_id.is_scheduled());
    if (message_id != d->notification_info->pinned_message_notification_message_id_) {
      auto m = get_message_force(d, message_id, "remove_dialog_mention_notifications");
      if (m != nullptr && m->notification_id.is_valid()) {
        CHECK(is_from_mention_notification_group(m));
        if (is_message_notification_active(d, m)) {
          removed_notification_ids_set.insert(m->notification_id);
        }
      }
    }
  }

  vector<NotificationId> removed_notification_ids;
  for (auto notification_id : removed_notification_ids_set) {
    removed_notification_ids.push_back(notification_id);
  }
  for (size_t i = 0; i < removed_notification_ids.size(); i++) {
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification, notification_group_id,
                       removed_notification_ids[i], false, i + 1 == removed_notification_ids.size(), Promise<Unit>(),
                       "remove_dialog_mention_notifications");
  }
}

void MessagesManager::set_dialog_last_notification(DialogId dialog_id, NotificationGroupInfo &group_info,
                                                   int32 last_notification_date, NotificationId last_notification_id,
                                                   const char *source) {
  if (group_info.set_last_notification(last_notification_date, last_notification_id, source)) {
    on_dialog_updated(dialog_id, "set_dialog_last_notification");
  }
}

void MessagesManager::set_dialog_last_notification_checked(DialogId dialog_id, NotificationGroupInfo &group_info,
                                                           int32 last_notification_date,
                                                           NotificationId last_notification_id, const char *source) {
  bool is_changed = group_info.set_last_notification(last_notification_date, last_notification_id, source);
  CHECK(is_changed);
  on_dialog_updated(dialog_id, "set_dialog_last_notification_checked");
}

void MessagesManager::on_update_sent_text_message(int64 random_id,
                                                  tl_object_ptr<telegram_api::MessageMedia> message_media,
                                                  vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities) {
  int32 message_media_id = message_media == nullptr ? telegram_api::messageMediaEmpty::ID : message_media->get_id();
  LOG_IF(ERROR, message_media_id != telegram_api::messageMediaWebPage::ID &&
                    message_media_id != telegram_api::messageMediaEmpty::ID)
      << "Receive non web-page media for text message: " << oneline(to_string(message_media));

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // result of sending message has already been received through getDifference
    return;
  }

  auto message_full_id = it->second;
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  auto m = get_message_force(d, message_full_id.get_message_id(), "on_update_sent_text_message");
  if (m == nullptr) {
    // message has already been deleted
    return;
  }
  CHECK(m->message_id.is_yet_unsent());
  message_full_id = MessageFullId(dialog_id, m->message_id);

  if (m->content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has been already changed to " << m->content->get_type();
    return;
  }

  const FormattedText *old_message_text = get_message_content_text(m->content.get());
  CHECK(old_message_text != nullptr);
  FormattedText new_message_text = get_message_text(
      td_->user_manager_.get(), old_message_text->text, std::move(entities), true, td_->auth_manager_->is_bot(),
      get_message_original_date(m), m->media_album_id != 0, "on_update_sent_text_message");
  auto new_content = get_message_content(td_, std::move(new_message_text), std::move(message_media), dialog_id, m->date,
                                         true /*likely ignored*/, UserId() /*likely ignored*/, nullptr /*ignored*/,
                                         nullptr, "on_update_sent_text_message");
  if (new_content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has changed to " << new_content->get_type();
    return;
  }

  bool need_update = false;
  bool is_content_changed = false;
  merge_message_contents(td_, m->content.get(), new_content.get(), need_message_changed_warning(m), dialog_id, false,
                         is_content_changed, need_update);
  compare_message_contents(td_, m->content.get(), new_content.get(), is_content_changed, need_update);

  if (is_content_changed || need_update) {
    reregister_message_content(td_, m->content.get(), new_content.get(), message_full_id,
                               "on_update_sent_text_message");
    m->content = std::move(new_content);
    m->is_content_secret = m->ttl.is_secret_message_content(MessageContentType::Text);

    if (need_update) {
      send_update_message_content(d, m, true, "on_update_sent_text_message");
    }
    on_message_changed(d, m, need_update, "on_update_sent_text_message");
  }
}

void MessagesManager::delete_pending_message_web_page(MessageFullId message_full_id) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  Message *m = get_message(d, message_full_id.get_message_id());
  CHECK(m != nullptr);

  MessageContent *content = m->content.get();
  CHECK(has_message_content_web_page(content));
  unregister_message_content(td_, content, message_full_id, "delete_pending_message_web_page");
  remove_message_content_web_page(content);
  register_message_content(td_, content, message_full_id, "delete_pending_message_web_page");

  // don't need to send an updateMessageContent, because the web page was pending

  on_message_changed(d, m, false, "delete_pending_message_web_page");
}

void MessagesManager::on_get_dialogs(FolderId folder_id, vector<tl_object_ptr<telegram_api::Dialog>> &&dialog_folders,
                                     int32 total_count, vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                     Promise<Unit> &&promise) {
  if (td_->updates_manager_->running_get_difference()) {
    LOG(INFO) << "Postpone result of getDialogs";
    pending_on_get_dialogs_.push_back(PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count,
                                                          std::move(messages), std::move(promise)});
    return;
  }
  bool from_dialog_list = total_count >= 0;
  bool from_get_dialog = total_count == -1;
  bool from_pinned_dialog_list = total_count == -2;

  if (from_get_dialog && dialog_folders.size() == 1 && dialog_folders[0]->get_id() == telegram_api::dialog::ID) {
    DialogId dialog_id(static_cast<const telegram_api::dialog *>(dialog_folders[0].get())->peer_);
    if (dialog_id.is_valid() && running_get_channel_difference(dialog_id)) {
      LOG(INFO) << "Postpone result of channels getDialogs for " << dialog_id;
      pending_channel_on_get_dialogs_.emplace(
          dialog_id, PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count, std::move(messages),
                                         std::move(promise)});
      return;
    }
  }

  vector<tl_object_ptr<telegram_api::dialog>> dialogs;
  for (auto &dialog_folder : dialog_folders) {
    switch (dialog_folder->get_id()) {
      case telegram_api::dialog::ID:
        dialogs.push_back(telegram_api::move_object_as<telegram_api::dialog>(dialog_folder));
        break;
      case telegram_api::dialogFolder::ID: {
        auto folder = telegram_api::move_object_as<telegram_api::dialogFolder>(dialog_folder);
        if (from_pinned_dialog_list) {
          // TODO update unread_muted_peers_count:int unread_unmuted_peers_count:int
          // unread_muted_messages_count:int unread_unmuted_messages_count:int
          FolderId folder_folder_id(folder->folder_->id_);
          if (folder_folder_id == FolderId::archive()) {
            // archive is expected in pinned dialogs list
            break;
          }
        }
        LOG(ERROR) << "Receive unexpected " << to_string(folder);
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  const char *source = nullptr;
  if (from_get_dialog) {
    LOG(INFO) << "Process " << dialogs.size() << " chats";
    source = "get chat";
  } else if (from_pinned_dialog_list) {
    LOG(INFO) << "Process " << dialogs.size() << " pinned chats in " << folder_id;
    source = "get pinned chats";
  } else {
    LOG(INFO) << "Process " << dialogs.size() << " chats out of " << total_count << " in " << folder_id;
    source = "get chat list";
  }
  FlatHashMap<MessageFullId, DialogDate, MessageFullIdHash> message_full_id_to_dialog_date;
  FlatHashMap<MessageFullId, tl_object_ptr<telegram_api::Message>, MessageFullIdHash> message_full_id_to_message;
  for (auto &message : messages) {
    auto message_full_id = MessageFullId::get_message_full_id(message, false);
    if (!message_full_id.get_message_id().is_valid()) {  // must not check dialog_id because of messageEmpty
      continue;
    }
    if (from_dialog_list) {
      auto message_date = get_message_date(message);
      int64 order = get_dialog_order(message_full_id.get_message_id(), message_date);
      message_full_id_to_dialog_date.emplace(message_full_id, DialogDate(order, message_full_id.get_dialog_id()));
    }
    message_full_id_to_message[message_full_id] = std::move(message);
  }

  DialogDate max_dialog_date = MIN_DIALOG_DATE;
  for (auto &dialog : dialogs) {
    //    LOG(INFO) << to_string(dialog);
    DialogId dialog_id(dialog->peer_);
    bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;

    if (!dialog_id.is_valid()) {
      LOG(ERROR) << "Receive wrong " << dialog_id;
      return promise.set_error(Status::Error(500, "Wrong query result returned: receive wrong chat identifier"));
    }
    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        if (has_pts) {
          LOG(ERROR) << "Receive user or group " << dialog_id << " with PTS";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive user or basic group chat with PTS"));
        }
        break;
      case DialogType::Channel:
        if (!has_pts) {
          LOG(ERROR) << "Receive channel " << dialog_id << " without PTS";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive supergroup chat without PTS"));
        }
        break;
      case DialogType::SecretChat:
      case DialogType::None:
      default:
        UNREACHABLE();
        return promise.set_error(Status::Error(500, "UNREACHABLE"));
    }

    if (from_dialog_list) {
      MessageId last_message_id(ServerMessageId(dialog->top_message_));
      if (last_message_id.is_valid()) {
        MessageFullId message_full_id(dialog_id, last_message_id);
        auto it = message_full_id_to_dialog_date.find(message_full_id);
        if (it == message_full_id_to_dialog_date.end() && dialog_id.get_type() != DialogType::Channel) {
          it = message_full_id_to_dialog_date.find({DialogId(), last_message_id});
        }
        if (it == message_full_id_to_dialog_date.end()) {
          LOG(ERROR) << "Last " << last_message_id << " in " << dialog_id << " not found";
          return promise.set_error(Status::Error(500, "Wrong query result returned: last message not found"));
        }
        FolderId dialog_folder_id(dialog->folder_id_);
        if (dialog_folder_id != folder_id) {
          LOG(ERROR) << "Receive " << dialog_id << " in " << dialog_folder_id << " instead of " << folder_id;
          continue;
        }

        DialogDate dialog_date = it->second;
        if (dialog_date.get_date() > 0 && dialog_date.get_dialog_id() == dialog_id && max_dialog_date < dialog_date) {
          max_dialog_date = dialog_date;
        }
      } else {
        LOG(ERROR) << "Receive " << last_message_id << " as last chat message";
        continue;
      }
    }
  }

  if (from_dialog_list && total_count < narrow_cast<int32>(dialogs.size())) {
    LOG(ERROR) << "Receive chat total_count = " << total_count << ", but " << dialogs.size() << " chats";
    total_count = narrow_cast<int32>(dialogs.size());
  }

  vector<DialogId> added_dialog_ids;
  for (auto &dialog : dialogs) {
    MessageId last_message_id(ServerMessageId(dialog->top_message_));
    if (!last_message_id.is_valid() && from_dialog_list) {
      // skip dialogs without messages
      total_count--;
      continue;
    }

    DialogId dialog_id(dialog->peer_);
    if (td::contains(added_dialog_ids, dialog_id)) {
      LOG(ERROR) << "Receive " << dialog_id << " twice in result of getChats with total_count = " << total_count;
      continue;
    }
    added_dialog_ids.push_back(dialog_id);
    Dialog *d = get_dialog_force(dialog_id, source);
    bool need_update_dialog_pos = false;
    CHECK(!being_added_dialog_id_.is_valid());
    being_added_dialog_id_ = dialog_id;
    if (d == nullptr) {
      d = add_dialog(dialog_id, source);
      need_update_dialog_pos = true;
    } else {
      LOG(INFO) << "Receive already created " << dialog_id;
      CHECK(d->dialog_id == dialog_id);
    }
    bool is_new = d->last_new_message_id == MessageId();
    auto positions = get_dialog_positions(d);

    set_dialog_folder_id(d, FolderId(dialog->folder_id_));
    if (dialog_id.get_type() == DialogType::Channel) {
      set_dialog_view_as_messages(d, dialog->view_forum_as_messages_, "on_get_dialogs");
    }

    on_update_dialog_notify_settings(dialog_id, std::move(dialog->notify_settings_), source);
    if (!d->notification_settings.is_synchronized && !td_->auth_manager_->is_bot()) {
      LOG(ERROR) << "Failed to synchronize settings in " << dialog_id;
      d->notification_settings.is_synchronized = true;
      on_dialog_updated(dialog_id, "set notification_settings.is_synchronized");
    }

    if (dialog->unread_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_count_ << " as number of unread messages in " << dialog_id;
      dialog->unread_count_ = 0;
    }
    MessageId read_inbox_max_message_id = MessageId(ServerMessageId(dialog->read_inbox_max_id_));
    if (!read_inbox_max_message_id.is_valid() && read_inbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_inbox_max_message_id << " as last read inbox message in " << dialog_id;
      read_inbox_max_message_id = MessageId();
    }
    MessageId read_outbox_max_message_id = MessageId(ServerMessageId(dialog->read_outbox_max_id_));
    if (!read_outbox_max_message_id.is_valid() && read_outbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_outbox_max_message_id << " as last read outbox message in " << dialog_id;
      read_outbox_max_message_id = MessageId();
    }
    if (dialog->unread_mentions_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_mentions_count_ << " as number of unread mention messages in "
                 << dialog_id;
      dialog->unread_mentions_count_ = 0;
    }
    if (dialog->unread_reactions_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_reactions_count_ << " as number of messages with unread reactions in "
                 << dialog_id;
      dialog->unread_reactions_count_ = 0;
    }
    if (dialog->ttl_period_ < 0) {
      LOG(ERROR) << "Receive " << dialog->ttl_period_ << " as message auto-delete time in " << dialog_id;
      dialog->ttl_period_ = 0;
    }
    if (!td_->auth_manager_->is_bot()) {
      const char *reload_source = nullptr;
      if (!d->is_is_blocked_for_stories_inited) {
        reload_source = "on_get_dialogs init is_blocked_for_stories";
      } else if (!d->is_has_bots_inited) {
        reload_source = "on_get_dialogs init has_bots";
      } else if (!d->is_background_inited) {
        reload_source = "on_get_dialogs init background";
      } else if (!d->is_theme_name_inited) {
        reload_source = "on_get_dialogs init theme_name";
      } else if (!d->is_available_reactions_inited) {
        reload_source = "on_get_dialogs init available_reactions";
      } else if (!d->is_last_pinned_message_id_inited) {
        get_dialog_pinned_message(dialog_id, Auto());
      }
      if (reload_source != nullptr) {
        // asynchronously get dialog info from the server
        // TODO add is_blocked/is_blocked_for_stories/has_bots/background/theme_name/available_reactions
        // to telegram_api::dialog
        td_->dialog_manager_->reload_dialog_info_full(dialog_id, reload_source);
      }
    }

    need_update_dialog_pos |=
        update_dialog_draft_message(d, get_draft_message(td_, std::move(dialog->draft_)), true, false);
    if (is_new) {
      bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;
      if (last_message_id.is_valid() && !td_->auth_manager_->is_bot()) {
        MessageFullId message_full_id(dialog_id, last_message_id);
        auto it = message_full_id_to_message.find(message_full_id);
        if (it == message_full_id_to_message.end() && dialog_id.get_type() != DialogType::Channel) {
          it = message_full_id_to_message.find({DialogId(), last_message_id});
        }
        if (it == message_full_id_to_message.end()) {
          LOG(ERROR) << "Last " << message_full_id << " not found";
        } else if (!has_pts || d->pts == 0 || dialog->pts_ <= d->pts || d->is_channel_difference_finished) {
          auto added_message_full_id = on_get_message(std::move(it->second), false, has_pts, false, source);
          message_full_id_to_message.erase(it);
          CHECK(d->last_new_message_id == MessageId());
          set_dialog_last_new_message_id(d, last_message_id, source);
          if (d->last_new_message_id > d->last_message_id && added_message_full_id.get_message_id().is_valid()) {
            CHECK(added_message_full_id.get_message_id() == d->last_new_message_id);
            set_dialog_last_message_id(d, d->last_new_message_id, source);
            send_update_chat_last_message(d, source);
          }
        } else if (dialog_id.get_type() == DialogType::Channel) {
          get_channel_difference(dialog_id, d->pts, dialog->pts_, last_message_id, true, source);
        }
      }

      if (has_pts && !running_get_channel_difference(dialog_id)) {
        set_channel_pts(d, dialog->pts_, source);
      }
    }
    bool is_marked_as_unread = dialog->unread_mark_;
    if (is_marked_as_unread != d->is_marked_as_unread) {
      set_dialog_is_marked_as_unread(d, is_marked_as_unread);
    }

    if (need_update_dialog_pos) {
      update_dialog_pos(d, source);
    }

    if (!td_->auth_manager_->is_bot() && !from_pinned_dialog_list) {
      // set is_pinned only after updating chat position to ensure that order is initialized
      bool is_pinned = (dialog->flags_ & DIALOG_FLAG_IS_PINNED) != 0;
      bool was_pinned = is_dialog_pinned(DialogListId(d->folder_id), dialog_id);
      if (is_pinned != was_pinned) {
        set_dialog_is_pinned(DialogListId(d->folder_id), d, is_pinned);
      }
    }

    if (!G()->use_message_database() || is_new || !d->is_last_read_inbox_message_id_inited ||
        d->need_repair_server_unread_count) {
      if (d->last_read_inbox_message_id.is_valid() && !d->last_read_inbox_message_id.is_server() &&
          read_inbox_max_message_id == d->last_read_inbox_message_id.get_prev_server_message_id()) {
        read_inbox_max_message_id = d->last_read_inbox_message_id;
      }
      if (d->need_repair_server_unread_count &&
          (d->last_read_inbox_message_id <= read_inbox_max_message_id || !need_unread_counter(d->order) ||
           !td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read))) {
        LOG(INFO) << "Repaired server unread count in " << dialog_id << " from " << d->last_read_inbox_message_id << "/"
                  << d->server_unread_count << " to " << read_inbox_max_message_id << "/" << dialog->unread_count_;
        d->need_repair_server_unread_count = false;
        on_dialog_updated(dialog_id, "repaired dialog server unread count");
      }
      if (d->need_repair_server_unread_count) {
        auto &previous_message_id = previous_repaired_read_inbox_max_message_id_[dialog_id];
        if (previous_message_id >= read_inbox_max_message_id) {
          // protect from sending the request in a loop
          if (d->server_unread_count != 0) {
            LOG(ERROR) << "Failed to repair server unread count in " << dialog_id
                       << ", because receive read_inbox_max_message_id = " << read_inbox_max_message_id << " after "
                       << previous_message_id << ", but messages are read up to " << d->last_read_inbox_message_id;
          } else {
            LOG(INFO) << "Failed to repair server unread count in " << dialog_id
                      << ", because receive read_inbox_max_message_id = " << read_inbox_max_message_id
                      << ", but messages are read up to " << d->last_read_inbox_message_id
                      << ". Likely all messages after " << read_inbox_max_message_id << " are outgoing";
          }
          d->need_repair_server_unread_count = false;
          on_dialog_updated(dialog_id, "failed to repair dialog server unread count");
        } else {
          LOG(INFO) << "Have last_read_inbox_message_id = " << d->last_read_inbox_message_id << ", but received only "
                    << read_inbox_max_message_id << " from the server, trying to repair server unread count again";
          previous_message_id = read_inbox_max_message_id;
          repair_server_unread_count(dialog_id, d->server_unread_count, source);
        }
      }
      if (!d->need_repair_server_unread_count) {
        previous_repaired_read_inbox_max_message_id_.erase(dialog_id);
      }
      if ((d->server_unread_count != dialog->unread_count_ &&
           d->last_read_inbox_message_id == read_inbox_max_message_id) ||
          d->last_read_inbox_message_id < read_inbox_max_message_id) {
        set_dialog_last_read_inbox_message_id(d, read_inbox_max_message_id, dialog->unread_count_,
                                              d->local_unread_count, true, source);
      }
      if (!d->is_last_read_inbox_message_id_inited) {
        d->is_last_read_inbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_inbox_message_id_inited");
      }
    }

    if (!G()->use_message_database() || is_new || !d->is_last_read_outbox_message_id_inited) {
      if (d->last_read_outbox_message_id < read_outbox_max_message_id) {
        set_dialog_last_read_outbox_message_id(d, read_outbox_max_message_id);
      }
      if (!d->is_last_read_outbox_message_id_inited) {
        d->is_last_read_outbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_outbox_message_id_inited");
      }
    }

    if (!G()->use_message_database() || is_new || d->need_repair_unread_mention_count) {
      if (d->need_repair_unread_mention_count) {
        if (d->unread_mention_count != dialog->unread_mentions_count_) {
          LOG(INFO) << "Repaired unread mention count in " << dialog_id << " from " << d->unread_mention_count << " to "
                    << dialog->unread_mentions_count_;
        }
        d->need_repair_unread_mention_count = false;
        on_dialog_updated(dialog_id, "repaired dialog unread mention count");
      }
      if (d->unread_mention_count != dialog->unread_mentions_count_ && !td_->auth_manager_->is_bot()) {
        set_dialog_unread_mention_count(d, dialog->unread_mentions_count_);
        update_dialog_mention_notification_count(d);
        send_update_chat_unread_mention_count(d);
      }
    }
    if (!G()->use_message_database() || is_new || d->need_repair_unread_reaction_count) {
      if (d->need_repair_unread_reaction_count) {
        if (d->unread_reaction_count != dialog->unread_reactions_count_) {
          LOG(INFO) << "Repaired unread reaction count in " << dialog_id << " from " << d->unread_reaction_count
                    << " to " << dialog->unread_reactions_count_;
        }
        d->need_repair_unread_reaction_count = false;
        on_dialog_updated(dialog_id, "repaired dialog unread reaction count");
      }
      if (d->unread_reaction_count != dialog->unread_reactions_count_ && !td_->auth_manager_->is_bot()) {
        set_dialog_unread_reaction_count(d, dialog->unread_reactions_count_);
        // update_dialog_reaction_notification_count(d);
        send_update_chat_unread_reaction_count(d, source);
      }
    }

    set_dialog_message_ttl(d, MessageTtl(dialog->ttl_period_));

    being_added_dialog_id_ = DialogId();

    update_dialog_lists(d, std::move(positions), true, false, source);

    if ((from_dialog_list || from_pinned_dialog_list) && d->order == DEFAULT_ORDER) {
      load_last_dialog_message(d, "on_get_dialog");
    }
  }

  if (from_dialog_list) {
    CHECK(!td_->auth_manager_->is_bot());
    CHECK(total_count >= 0);

    auto &folder_list = add_dialog_list(DialogListId(folder_id));
    if (folder_list.server_dialog_total_count_ != total_count) {
      auto old_dialog_total_count = get_dialog_total_count(folder_list);
      folder_list.server_dialog_total_count_ = total_count;
      if (folder_list.is_dialog_unread_count_inited_) {
        if (old_dialog_total_count != get_dialog_total_count(folder_list)) {
          send_update_unread_chat_count(folder_list, DialogId(), true, source);
        } else {
          save_unread_chat_count(folder_list);
        }
      }
    }

    auto *folder = get_dialog_folder(folder_id);
    CHECK(folder != nullptr);
    if (dialogs.empty()) {
      // if there are no more dialogs on the server
      max_dialog_date = MAX_DIALOG_DATE;
    }
    if (folder->last_server_dialog_date_ < max_dialog_date) {
      folder->last_server_dialog_date_ = max_dialog_date;
      update_last_dialog_date(folder_id);
    } else if (promise) {
      LOG(ERROR) << "Last server dialog date didn't increased from " << folder->last_server_dialog_date_ << " to "
                 << max_dialog_date << " after receiving " << dialogs.size() << " chats " << added_dialog_ids
                 << " from " << total_count << " in " << folder_id
                 << ". last_dialog_date = " << folder->folder_last_dialog_date_
                 << ", last_loaded_database_dialog_date = " << folder->last_loaded_database_dialog_date_;
    }
  }
  if (from_pinned_dialog_list) {
    CHECK(!td_->auth_manager_->is_bot());
    auto *folder_list = get_dialog_list(DialogListId(folder_id));
    CHECK(folder_list != nullptr);
    auto pinned_dialog_ids = DialogId::remove_secret_chat_dialog_ids(get_pinned_dialog_ids(DialogListId(folder_id)));
    bool are_pinned_dialogs_saved = folder_list->are_pinned_dialogs_inited_;
    folder_list->are_pinned_dialogs_inited_ = true;
    if (pinned_dialog_ids != added_dialog_ids) {
      if (set_folder_pinned_dialogs(folder_id, std::move(pinned_dialog_ids), std::move(added_dialog_ids))) {
        are_pinned_dialogs_saved = true;
      }
    } else {
      LOG(INFO) << "Pinned chats are not changed";
    }
    update_list_last_pinned_dialog_date(*folder_list);

    if (!are_pinned_dialogs_saved && G()->use_message_database()) {
      LOG(INFO) << "Save empty pinned chat list in " << folder_id;
      G()->td_db()->get_binlog_pmc()->set(PSTRING() << "pinned_dialog_ids" << folder_id.get(), "");
    }
  }
  promise.set_value(Unit());
}

bool MessagesManager::is_message_unload_enabled() const {
  return G()->use_message_database() || td_->auth_manager_->is_bot();
}

bool MessagesManager::can_unload_message(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  MessageFullId message_full_id{d->dialog_id, m->message_id};
  if (td_->auth_manager_->is_bot() && !G()->use_message_database()) {
    return !m->message_id.is_yet_unsent() && replied_by_yet_unsent_messages_.count(message_full_id) == 0 &&
           m->edited_content == nullptr && m->message_id != d->last_pinned_message_id &&
           m->message_id != d->last_edited_message_id;
  }
  // don't want to unload messages from opened dialogs
  // don't want to unload messages to which there are replies in yet unsent messages
  // don't want to unload message with active reply markup
  // don't want to unload the newest pinned message
  // don't want to unload last edited message, because server can send updateEditChannelMessage again
  // don't want to unload messages from the last album
  // can't unload from memory last dialog, last database messages, yet unsent messages, being edited media messages and active live locations
  // can't unload messages in dialog with active suffix load query
  {
    auto it = dialog_suffix_load_queries_.find(d->dialog_id);
    if (it != dialog_suffix_load_queries_.end() && !it->second->suffix_load_queries_.empty()) {
      return false;
    }
  }
  return d->open_count == 0 && m->message_id != d->last_message_id && m->message_id != d->last_database_message_id &&
         !m->message_id.is_yet_unsent() && active_live_location_message_full_ids_.count(message_full_id) == 0 &&
         replied_by_yet_unsent_messages_.count(message_full_id) == 0 && m->edited_content == nullptr &&
         m->message_id != d->reply_markup_message_id && m->message_id != d->last_pinned_message_id &&
         m->message_id != d->last_edited_message_id &&
         (m->media_album_id != d->last_media_album_id || m->media_album_id == 0);
}

unique_ptr<MessagesManager::Message> MessagesManager::unload_message(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(message_id.is_valid());
  bool need_update_dialog_pos = false;
  auto message = do_delete_message(d, message_id, false, true, &need_update_dialog_pos, "unload_message");
  CHECK(!need_update_dialog_pos);
  return message;
}

unique_ptr<MessagesManager::Message> MessagesManager::delete_message(Dialog *d, MessageId message_id,
                                                                     bool is_permanently_deleted,
                                                                     bool *need_update_dialog_pos, const char *source) {
  return do_delete_message(d, message_id, is_permanently_deleted, false, need_update_dialog_pos, source);
}

void MessagesManager::add_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat || message_id.is_yet_unsent());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it == d->random_id_to_message_id.end() || it->second.get() < message_id.get()) {
    LOG(INFO) << "Add correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    if (random_id != 0) {
      d->random_id_to_message_id[random_id] = message_id;
    }
  }
}

void MessagesManager::delete_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat || message_id.is_yet_unsent());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it != d->random_id_to_message_id.end() && it->second == message_id) {
    LOG(INFO) << "Delete correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    d->random_id_to_message_id.erase(it);
  }
}

void MessagesManager::add_notification_id_to_message_id_correspondence(NotificationInfo *notification_info,
                                                                       NotificationId notification_id,
                                                                       MessageId message_id) {
  CHECK(notification_info != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = notification_info->notification_id_to_message_id_.find(notification_id);
  if (it == notification_info->notification_id_to_message_id_.end()) {
    VLOG(notifications) << "Add correspondence from " << notification_id << " to " << message_id;
    notification_info->notification_id_to_message_id_.emplace(notification_id, message_id);
  } else if (it->second != message_id) {
    LOG(ERROR) << "Have the same " << notification_id << " for " << message_id << " and " << it->second;
    if (it->second < message_id) {
      it->second = message_id;
    }
  }
}

void MessagesManager::delete_notification_id_to_message_id_correspondence(NotificationInfo *notification_info,
                                                                          NotificationId notification_id,
                                                                          MessageId message_id) {
  CHECK(notification_info != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = notification_info->notification_id_to_message_id_.find(notification_id);
  if (it != notification_info->notification_id_to_message_id_.end() && it->second == message_id) {
    VLOG(notifications) << "Delete correspondence from " << notification_id << " to " << message_id;
    notification_info->notification_id_to_message_id_.erase(it);
  } else {
    LOG(ERROR) << "Can't find " << notification_id << " from " << message_id;
  }
}

bool MessagesManager::is_notification_info_group_id(const NotificationInfo *notification_info,
                                                    NotificationGroupId group_id) {
  if (!group_id.is_valid() || notification_info == nullptr) {
    return false;
  }
  return notification_info->message_notification_group_.has_group_id(group_id) ||
         notification_info->mention_notification_group_.has_group_id(group_id);
}

bool MessagesManager::is_dialog_notification_group_id(const Dialog *d, NotificationGroupId group_id) {
  if (d == nullptr) {
    return false;
  }
  return is_notification_info_group_id(d->notification_info.get(), group_id);
}

void MessagesManager::remove_message_notification_id(Dialog *d, Message *m, bool is_permanent, bool force_update,
                                                     bool ignore_pinned_message_notification_removal) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  if (!m->notification_id.is_valid()) {
    return;
  }

  auto from_mentions = is_from_mention_notification_group(m);
  auto &group_info = get_notification_group_info(d, m);
  if (!group_info.is_valid()) {
    return;
  }

  bool had_active_notification = is_message_notification_active(d, m);

  auto notification_id = m->notification_id;
  VLOG(notifications) << "Remove " << notification_id << " from " << m->message_id << " in "
                      << group_info.get_group_id() << '/' << d->dialog_id
                      << " from database, was_active = " << had_active_notification
                      << ", is_permanent = " << is_permanent;
  delete_notification_id_to_message_id_correspondence(d->notification_info.get(), notification_id, m->message_id);
  m->removed_notification_id = m->notification_id;
  m->notification_id = NotificationId();
  if (d->notification_info->pinned_message_notification_message_id_ == m->message_id && is_permanent &&
      !ignore_pinned_message_notification_removal) {
    remove_dialog_pinned_message_notification(
        d, "remove_message_notification_id");  // must be called after notification_id is removed
  }
  if (group_info.get_last_notification_id() == notification_id) {
    // last notification is deleted, need to find new last notification
    fix_dialog_last_notification_id(d, from_mentions, m->message_id);
  }

  if (is_permanent) {
    if (had_active_notification) {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification,
                         group_info.get_group_id(), notification_id, is_permanent, force_update, Promise<Unit>(),
                         "remove_message_notification_id");
    }

    // on_message_changed will be called by the caller
    // don't need to call there to not save twice/or to save just deleted message
  } else {
    on_message_changed(d, m, false, "remove_message_notification_id");
  }
}

void MessagesManager::remove_new_secret_chat_notification(Dialog *d, bool is_permanent) {
  CHECK(d != nullptr);
  CHECK(d->notification_info != nullptr);
  auto notification_id = d->notification_info->new_secret_chat_notification_id_;
  CHECK(notification_id.is_valid());
  VLOG(notifications) << "Remove " << notification_id << " about new secret " << d->dialog_id << " from "
                      << d->notification_info->message_notification_group_.get_group_id();
  d->notification_info->new_secret_chat_notification_id_ = NotificationId();
  set_dialog_last_notification_checked(d->dialog_id, d->notification_info->message_notification_group_, 0,
                                       NotificationId(), "remove_new_secret_chat_notification");
  if (is_permanent) {
    CHECK(d->notification_info->message_notification_group_.is_valid());
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification,
                       d->notification_info->message_notification_group_.get_group_id(), notification_id, true, true,
                       Promise<Unit>(), "remove_new_secret_chat_notification");
  }
}

void MessagesManager::fix_dialog_last_notification_id(Dialog *d, bool from_mentions, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  if (d->notification_info == nullptr) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());
  auto &group_info = get_notification_group_info(d, from_mentions);
  CHECK(group_info.is_valid());
  auto it = d->ordered_messages.get_const_iterator(message_id);
  VLOG(notifications) << "Trying to fix last notification identifier in " << group_info.get_group_id() << " from "
                      << d->dialog_id << " from " << message_id << '/' << group_info.get_last_notification_id();
  if (*it != nullptr && ((*it)->get_message_id() == message_id || (*it)->have_next())) {
    while (*it != nullptr) {
      const Message *m = get_message(d, (*it)->get_message_id());
      if (is_from_mention_notification_group(m) == from_mentions && m->notification_id.is_valid() &&
          is_message_notification_active(d, m) && m->message_id != message_id) {
        set_dialog_last_notification(d->dialog_id, group_info, m->date, m->notification_id,
                                     "fix_dialog_last_notification_id");
        return;
      }
      --it;
    }
  }
  if (G()->use_message_database()) {
    get_message_notifications_from_database(
        d->dialog_id, group_info.get_group_id(), group_info.get_last_notification_id(), message_id, 1,
        PromiseCreator::lambda(
            [actor_id = actor_id(this), dialog_id = d->dialog_id, from_mentions,
             prev_last_notification_id = group_info.get_last_notification_id()](Result<vector<Notification>> result) {
              send_closure(actor_id, &MessagesManager::do_fix_dialog_last_notification_id, dialog_id, from_mentions,
                           prev_last_notification_id, std::move(result));
            }));
  }
}

void MessagesManager::do_fix_dialog_last_notification_id(DialogId dialog_id, bool from_mentions,
                                                         NotificationId prev_last_notification_id,
                                                         Result<vector<Notification>> result) {
  if (result.is_error()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (d->notification_info == nullptr) {
    return;
  }
  auto &group_info = get_notification_group_info(d, from_mentions);
  if (!group_info.is_valid()) {
    return;
  }
  VLOG(notifications) << "Receive " << result.ok().size() << " message notifications in " << group_info.get_group_id()
                      << '/' << dialog_id << " from " << prev_last_notification_id;
  if (group_info.get_last_notification_id() != prev_last_notification_id) {
    // last_notification_id was changed
    return;
  }

  auto notifications = result.move_as_ok();
  CHECK(notifications.size() <= 1);

  int32 last_notification_date = 0;
  NotificationId last_notification_id;
  if (!notifications.empty()) {
    last_notification_date = notifications[0].date;
    last_notification_id = notifications[0].notification_id;
  }

  set_dialog_last_notification(dialog_id, group_info, last_notification_date, last_notification_id,
                               "do_fix_dialog_last_notification_id");
}

// DO NOT FORGET TO ADD ALL CHANGES OF THIS FUNCTION AS WELL TO delete_all_dialog_messages
unique_ptr<MessagesManager::Message> MessagesManager::do_delete_message(Dialog *d, MessageId message_id,
                                                                        bool is_permanently_deleted,
                                                                        bool only_from_memory,
                                                                        bool *need_update_dialog_pos,
                                                                        const char *source) {
  CHECK(d != nullptr);
  if (!message_id.is_valid()) {
    if (message_id.is_valid_scheduled()) {
      return do_delete_scheduled_message(d, message_id, is_permanently_deleted, source);
    }

    LOG(ERROR) << "Trying to delete " << message_id << " in " << d->dialog_id << " from " << source;
    return nullptr;
  }

  MessageFullId message_full_id(d->dialog_id, message_id);
  const Message *m = get_message(d, message_id);
  if (m == nullptr) {
    if (only_from_memory) {
      return nullptr;
    }

    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    m = get_message_force(d, message_id, "do_delete_message");
    if (m == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted, source);

      if (is_permanently_deleted && d->last_clear_history_message_id == message_id) {
        set_dialog_last_clear_history_date(d, 0, MessageId(), "do_delete_message");
        *need_update_dialog_pos = true;
      }

      /*
      can't do this because the message may be never received in the dialog, unread count will became negative
      // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
      if (message_id.is_valid() && !message_id.is_yet_unsent() && d->is_last_read_inbox_message_id_inited &&
          message_id > d->last_read_inbox_message_id && !td_->auth_manager_->is_bot()) {
        int32 server_unread_count = d->server_unread_count;
        int32 local_unread_count = d->local_unread_count;
        int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
        if (unread_count == 0) {
          LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                     << ". Last read is " << d->last_read_inbox_message_id;
        } else {
          unread_count--;
          set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                                source);
        }
      }
      */
      return nullptr;
    }
  }

  if (only_from_memory && !can_unload_message(d, m)) {
    return nullptr;
  }

  LOG_CHECK(!d->being_deleted_message_id.is_valid())
      << d->being_deleted_message_id << " " << message_id << " " << source;
  d->being_deleted_message_id = message_id;

  bool need_get_history = false;
  if (!only_from_memory) {
    LOG(INFO) << "Deleting " << message_full_id << " from " << source;

    delete_message_from_database(d, message_id, m, is_permanently_deleted, source);

    delete_active_live_location(d->dialog_id, m);
    remove_message_file_sources(d->dialog_id, m);

    if (message_id == d->last_message_id) {
      auto it = d->ordered_messages.get_const_iterator(message_id);
      CHECK(*it != nullptr);
      CHECK((*it)->get_message_id() == message_id);
      --it;
      if (*it != nullptr) {
        set_dialog_last_message_id(d, (*it)->get_message_id(), "do_delete_message");
      } else {
        need_get_history = true;
        set_dialog_last_message_id(d, MessageId(), "do_delete_message");
        d->delete_last_message_date = m->date;
        d->deleted_last_message_id = message_id;
        d->is_last_message_deleted_locally = Slice(source) == Slice(DELETE_MESSAGE_USER_REQUEST_SOURCE);
        on_dialog_updated(d->dialog_id, "do delete last message");
      }
      *need_update_dialog_pos = true;
    }

    if (message_id == d->last_database_message_id) {
      auto it = d->ordered_messages.get_const_iterator(message_id);
      CHECK(*it != nullptr);
      CHECK((*it)->get_message_id() == message_id);
      do {
        --it;
      } while (*it != nullptr && (*it)->get_message_id().is_yet_unsent());

      if (*it != nullptr) {
        if ((*it)->get_message_id() < d->first_database_message_id && d->dialog_id.get_type() == DialogType::Channel) {
          // possible if messages were deleted from database, but not from memory after updateChannelTooLong
          set_dialog_last_database_message_id(d, MessageId(), "do_delete_message 1");
        } else {
          set_dialog_last_database_message_id(d, (*it)->get_message_id(), "do_delete_message 2");
          if (d->last_database_message_id < d->first_database_message_id) {
            LOG(ERROR) << "Last database " << d->last_database_message_id << " became less than first database "
                       << d->first_database_message_id << " after deletion of " << message_full_id;
            set_dialog_first_database_message_id(d, d->last_database_message_id, "do_delete_message 2");
          }
        }
      } else {
        if (d->first_database_message_id == d->last_database_message_id) {
          // database definitely has no more messages
          set_dialog_last_database_message_id(d, MessageId(), "do_delete_message 3");
        } else {
          LOG(INFO) << "Need to get history to repair last_database_message_id in " << d->dialog_id;
          need_get_history = true;
        }
      }
    }
    if (d->last_database_message_id.is_valid()) {
      CHECK(d->first_database_message_id.is_valid());
    } else if (d->first_database_message_id.is_valid()) {
      set_dialog_first_database_message_id(d, MessageId(), "do_delete_message");
    }

    auto suffix_load_queries_it = dialog_suffix_load_queries_.find(d->dialog_id);
    if (suffix_load_queries_it != dialog_suffix_load_queries_.end() &&
        message_id == suffix_load_queries_it->second->suffix_load_first_message_id_) {
      auto it = d->ordered_messages.get_const_iterator(message_id);
      CHECK(*it != nullptr);
      CHECK((*it)->get_message_id() == message_id);
      --it;
      if (*it != nullptr) {
        suffix_load_queries_it->second->suffix_load_first_message_id_ = (*it)->get_message_id();
      } else {
        suffix_load_queries_it->second->suffix_load_first_message_id_ = MessageId();
        suffix_load_queries_it->second->suffix_load_done_ = false;
      }
    }
  } else {
    auto suffix_load_queries_it = dialog_suffix_load_queries_.find(d->dialog_id);
    if (suffix_load_queries_it != dialog_suffix_load_queries_.end() &&
        message_id >= suffix_load_queries_it->second->suffix_load_first_message_id_) {
      suffix_load_queries_it->second->suffix_load_first_message_id_ = MessageId();
      suffix_load_queries_it->second->suffix_load_done_ = false;
    }
  }

  auto result = std::move(d->messages[message_id]);
  CHECK(m == result.get());
  d->messages.erase(message_id);

  static_cast<ListNode *>(result.get())->remove();

  if (!td_->auth_manager_->is_bot()) {
    d->ordered_messages.erase(message_id, only_from_memory);
  }

  d->being_deleted_message_id = MessageId();

  if (need_get_history) {
    send_closure_later(actor_id(this), &MessagesManager::load_last_dialog_message_later, d->dialog_id);
  }

  on_message_deleted(d, result.get(), is_permanently_deleted, source);

  return result;
}

void MessagesManager::on_message_deleted_from_database(Dialog *d, const Message *m, const char *source) {
  CHECK(d != nullptr);
  if (m == nullptr || m->message_id.is_scheduled() || td_->auth_manager_->is_bot()) {
    return;
  }

  auto message_id = m->message_id;
  if (d->reply_markup_message_id == message_id) {
    set_dialog_reply_markup(d, MessageId());
  }
  // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
  if (has_incoming_notification(d->dialog_id, m) && message_id > d->last_read_inbox_message_id &&
      d->is_last_read_inbox_message_id_inited) {
    int32 server_unread_count = d->server_unread_count;
    int32 local_unread_count = d->local_unread_count;
    int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
    if (unread_count == 0) {
      if (need_unread_counter(d->order)) {
        LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                   << ". Last read is " << d->last_read_inbox_message_id;
      }
    } else {
      unread_count--;
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                            source);
    }
  }
  if (m->contains_unread_mention) {
    if (d->unread_mention_count == 0) {
      if (is_dialog_inited(d)) {
        LOG(ERROR) << "Unread mention count became negative in " << d->dialog_id << " after deletion of " << message_id;
      }
    } else {
      set_dialog_unread_mention_count(d, d->unread_mention_count - 1);
      send_update_chat_unread_mention_count(d);
    }
  }
  if (has_unread_message_reactions(d->dialog_id, m)) {
    if (d->unread_reaction_count == 0) {
      if (is_dialog_inited(d)) {
        LOG(ERROR) << "Unread reaction count became negative in " << d->dialog_id << " after deletion of "
                   << message_id;
      }
    } else {
      set_dialog_unread_reaction_count(d, d->unread_reaction_count - 1);
      send_update_chat_unread_reaction_count(d, source);
    }
  }

  update_message_count_by_index(d, -1, m);
  update_reply_count_by_message(d, -1, m);

  td_->reaction_manager_->update_saved_messages_tags(m->saved_messages_topic_id, get_chosen_tags(m->reactions), {});
}

void MessagesManager::on_message_deleted(Dialog *d, Message *m, bool is_permanently_deleted, const char *source) {
  // also called for unloaded messages, but not for scheduled messages
  CHECK(m->message_id.is_valid());

  if (m->message_id.is_yet_unsent() && !m->message_id.is_scheduled() && m->top_thread_message_id.is_valid() &&
      !td_->auth_manager_->is_bot()) {
    auto it = yet_unsent_thread_message_ids_.find({d->dialog_id, m->top_thread_message_id});
    CHECK(it != yet_unsent_thread_message_ids_.end());
    auto is_deleted = it->second.erase(m->message_id) > 0;
    CHECK(is_deleted);
    if (it->second.empty()) {
      yet_unsent_thread_message_ids_.erase(it);
    }
  }
  if (d->open_count > 0) {
    auto it = dialog_viewed_messages_.find(d->dialog_id);
    if (it != dialog_viewed_messages_.end()) {
      auto &info = it->second;
      CHECK(info != nullptr);
      auto message_it = info->message_id_to_view_id.find(m->message_id);
      if (message_it != info->message_id_to_view_id.end()) {
        info->recently_viewed_messages.erase(message_it->second);
        info->message_id_to_view_id.erase(message_it);
      }
    }
  }

  cancel_send_deleted_message(d->dialog_id, m, is_permanently_deleted);

  auto dialog_type = d->dialog_id.get_type();
  switch (dialog_type) {
    case DialogType::User:
    case DialogType::Chat:
      if (m->message_id.is_server()) {
        message_id_to_dialog_id_.erase(m->message_id);
      }
      break;
    case DialogType::Channel:
      if (m->message_id.is_server() && !td_->auth_manager_->is_bot()) {
        td_->user_manager_->unregister_message_users({d->dialog_id, m->message_id}, get_message_user_ids(m));
        td_->chat_manager_->unregister_message_channels({d->dialog_id, m->message_id}, get_message_channel_ids(m));
      }
      break;
    case DialogType::SecretChat:
      // nothing to do
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }
  ttl_unregister_message(d->dialog_id, m, source);
  ttl_period_unregister_message(d->dialog_id, m);
  delete_bot_command_message_id(d->dialog_id, m->message_id);
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_deleted");
  unregister_message_reply(d->dialog_id, m);
  if (m->notification_id.is_valid()) {
    delete_notification_id_to_message_id_correspondence(d->notification_info.get(), m->notification_id, m->message_id);
  }
  if (m->message_id.is_yet_unsent() || dialog_type == DialogType::SecretChat) {
    delete_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);
  }
  if (m->is_topic_message) {
    td_->forum_topic_manager_->on_topic_message_count_changed(d->dialog_id, m->top_thread_message_id, -1);
  }
  if (is_permanently_deleted && !td_->auth_manager_->is_bot() && m->saved_messages_topic_id.is_valid()) {
    CHECK(d->dialog_id == td_->dialog_manager_->get_my_dialog_id());
    td_->saved_messages_manager_->on_topic_message_deleted(m->saved_messages_topic_id, m->message_id);
  }

  added_message_count_--;
}

bool MessagesManager::is_deleted_message(const Dialog *d, MessageId message_id) {
  if (message_id.is_scheduled() && message_id.is_valid_scheduled() && message_id.is_scheduled_server()) {
    return d->scheduled_messages != nullptr && d->scheduled_messages->deleted_scheduled_server_message_ids_.count(
                                                   message_id.get_scheduled_server_message_id()) > 0;
  } else {
    return d->deleted_message_ids.count(message_id) > 0;
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::do_delete_scheduled_message(Dialog *d, MessageId message_id,
                                                                                  bool is_permanently_deleted,
                                                                                  const char *source) {
  CHECK(d != nullptr);
  LOG_CHECK(message_id.is_valid_scheduled()) << d->dialog_id << ' ' << message_id << ' ' << source;

  if (d->scheduled_messages == nullptr) {
    auto message = get_message_force(d, message_id, "do_delete_scheduled_message");
    if (message == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted, source);
      return nullptr;
    }
    CHECK(d->scheduled_messages != nullptr);
  }
  auto it = d->scheduled_messages->scheduled_messages_.find(message_id);
  if (it == d->scheduled_messages->scheduled_messages_.end()) {
    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    auto message = get_message_force(d, message_id, "do_delete_scheduled_message");
    if (message == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted, source);
      return nullptr;
    }

    message_id = message->message_id;
    it = d->scheduled_messages->scheduled_messages_.find(message_id);
    CHECK(it != d->scheduled_messages->scheduled_messages_.end());
  }

  const Message *m = it->second.get();
  CHECK(m->message_id == message_id);

  LOG(INFO) << "Deleting " << MessageFullId{d->dialog_id, message_id} << " from " << source;

  delete_message_from_database(d, message_id, m, is_permanently_deleted, source);

  remove_message_file_sources(d->dialog_id, m);

  it = d->scheduled_messages->scheduled_messages_.find(message_id);
  auto result = std::move(it->second);
  d->scheduled_messages->scheduled_messages_.erase(it);
  CHECK(m == result.get());

  if (message_id.is_scheduled_server()) {
    size_t erased_count =
        d->scheduled_messages->scheduled_message_date_.erase(message_id.get_scheduled_server_message_id());
    CHECK(erased_count != 0);
  }

  cancel_send_deleted_message(d->dialog_id, result.get(), is_permanently_deleted);

  unregister_message_content(td_, m->content.get(), {d->dialog_id, message_id}, "do_delete_scheduled_message");
  unregister_message_reply(d->dialog_id, m);
  if (message_id.is_yet_unsent()) {
    delete_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);
  }
  if (m->is_topic_message) {
    td_->forum_topic_manager_->on_topic_message_count_changed(d->dialog_id, m->top_thread_message_id, -1);
  }

  return result;
}

bool MessagesManager::have_dialog(DialogId dialog_id) const {
  return dialogs_.count(dialog_id) > 0;
}

void MessagesManager::load_dialogs(vector<DialogId> dialog_ids, Promise<vector<DialogId>> &&promise) {
  LOG(INFO) << "Load chats " << format::as_array(dialog_ids);

  Dependencies dependencies;
  for (auto dialog_id : dialog_ids) {
    if (!have_dialog(dialog_id)) {
      dependencies.add_dialog_dependencies(dialog_id);
    }
  }
  dependencies.resolve_force(td_, "load_dialogs", true);

  td::remove_if(dialog_ids, [this](DialogId dialog_id) { return !td_->dialog_manager_->have_dialog_info(dialog_id); });

  for (auto dialog_id : dialog_ids) {
    force_create_dialog(dialog_id, "load_dialogs");
  }

  LOG(INFO) << "Loaded chats " << format::as_array(dialog_ids);
  promise.set_value(std::move(dialog_ids));
}

bool MessagesManager::load_dialog(DialogId dialog_id, int left_tries, Promise<Unit> &&promise) {
  if (!dialog_id.is_valid()) {
    promise.set_error(Status::Error(400, "Invalid chat identifier specified"));
    return false;
  }

  if (!have_dialog_force(dialog_id, "load_dialog")) {  // TODO remove _force
    if (G()->use_message_database()) {
      //      TODO load dialog from database, DialogLoader
      //      send_closure_later(actor_id(this), &MessagesManager::load_dialog_from_database, dialog_id,
      //      std::move(promise));
      //      return false;
    }
    if (td_->auth_manager_->is_bot()) {
      switch (dialog_id.get_type()) {
        case DialogType::User: {
          auto user_id = dialog_id.get_user_id();
          auto have_user = td_->user_manager_->get_user(user_id, left_tries, std::move(promise));
          if (!have_user) {
            return false;
          }
          break;
        }
        case DialogType::Chat: {
          auto have_chat = td_->chat_manager_->get_chat(dialog_id.get_chat_id(), left_tries, std::move(promise));
          if (!have_chat) {
            return false;
          }
          break;
        }
        case DialogType::Channel: {
          auto have_channel =
              td_->chat_manager_->get_channel(dialog_id.get_channel_id(), left_tries, std::move(promise));
          if (!have_channel) {
            return false;
          }
          break;
        }
        case DialogType::SecretChat:
          promise.set_error(Status::Error(400, "Chat not found"));
          return false;
        case DialogType::None:
        default:
          UNREACHABLE();
      }
      if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
        return false;
      }

      add_dialog(dialog_id, "load_dialog");
      return true;
    }

    promise.set_error(Status::Error(400, "Chat not found"));
    return false;
  }

  promise.set_value(Unit());
  return true;
}

Result<DialogDate> MessagesManager::get_dialog_list_last_date(DialogListId dialog_list_id) {
  CHECK(!td_->auth_manager_->is_bot());

  auto *list_ptr = get_dialog_list(dialog_list_id);
  if (list_ptr == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  return list_ptr->list_last_dialog_date_;
}

vector<DialogId> MessagesManager::get_dialogs(DialogListId dialog_list_id, DialogDate offset, int32 limit,
                                              bool exact_limit, bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  auto *list_ptr = get_dialog_list(dialog_list_id);
  if (list_ptr == nullptr) {
    promise.set_error(Status::Error(400, "Chat list not found"));
    return {};
  }
  auto &list = *list_ptr;

  LOG(INFO) << "Get chats in " << dialog_list_id << " with offset " << offset << " and limit " << limit
            << ". last_dialog_date = " << list.list_last_dialog_date_
            << ", last_pinned_dialog_date_ = " << list.last_pinned_dialog_date_
            << ", are_pinned_dialogs_inited_ = " << list.are_pinned_dialogs_inited_;

  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }

  vector<DialogId> result;
  if (dialog_list_id == DialogListId(FolderId::main()) && sponsored_dialog_id_.is_valid()) {
    auto d = get_dialog(sponsored_dialog_id_);
    CHECK(d != nullptr);
    if (is_dialog_sponsored(d)) {
      DialogDate date(get_dialog_private_order(&list, d), d->dialog_id);
      if (offset < date) {
        result.push_back(sponsored_dialog_id_);
        offset = date;
        limit--;
      }
    }
  }

  if (!list.are_pinned_dialogs_inited_) {
    if (limit == 0 || force) {
      promise.set_value(Unit());
      return result;
    } else {
      if (dialog_list_id.is_folder()) {
        auto &folder = *get_dialog_folder(dialog_list_id.get_folder_id());
        if (folder.last_loaded_database_dialog_date_ == folder.last_database_server_dialog_date_ &&
            folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
          load_dialog_list(list, limit, std::move(promise));
          return {};
        }
      }
      reload_pinned_dialogs(dialog_list_id, std::move(promise));
      return {};
    }
  }
  if (dialog_list_id.is_filter()) {
    auto dialog_filter_id = dialog_list_id.get_filter_id();
    vector<InputDialogId> input_dialog_ids;
    for (const auto &input_dialog_id : td_->dialog_filter_manager_->get_pinned_input_dialog_ids(dialog_filter_id)) {
      auto dialog_id = input_dialog_id.get_dialog_id();
      if (!have_dialog_force(dialog_id, "get_dialogs")) {
        if (dialog_id.get_type() == DialogType::SecretChat) {
          if (td_->dialog_manager_->have_dialog_info_force(dialog_id, "get_dialogs")) {
            force_create_dialog(dialog_id, "get_dialogs");
          }
        } else {
          input_dialog_ids.push_back(input_dialog_id);
        }
      }
    }

    if (!input_dialog_ids.empty()) {
      if (limit == 0 || force) {
        promise.set_value(Unit());
        return result;
      } else {
        td_->dialog_filter_manager_->load_dialog_filter_dialogs(dialog_filter_id, std::move(input_dialog_ids),
                                                                std::move(promise));
        return {};
      }
    }
  }

  if (!list.pinned_dialogs_.empty() && offset < list.pinned_dialogs_.back() && limit > 0) {
    bool need_reload_pinned_dialogs = false;
    bool need_remove_unknown_secret_chats = false;
    for (auto &pinned_dialog : list.pinned_dialogs_) {
      if (offset < pinned_dialog) {
        auto dialog_id = pinned_dialog.get_dialog_id();
        auto d = get_dialog_force(dialog_id, "get_dialogs");
        if (d == nullptr) {
          LOG(ERROR) << "Failed to load pinned " << dialog_id << " from " << dialog_list_id;
          if (dialog_id.get_type() != DialogType::SecretChat) {
            need_reload_pinned_dialogs = true;
          } else {
            need_remove_unknown_secret_chats = true;
          }
          continue;
        }
        if (d->order == DEFAULT_ORDER) {
          LOG(INFO) << "Loaded pinned " << dialog_id << " with default order in " << dialog_list_id;
          continue;
        }
        result.push_back(dialog_id);
        offset = pinned_dialog;
        limit--;
        if (limit == 0) {
          break;
        }
      }
    }
    if (need_reload_pinned_dialogs) {
      reload_pinned_dialogs(dialog_list_id, Auto());
    }
    if (need_remove_unknown_secret_chats) {
      td::remove_if(list.pinned_dialogs_, [this, &list](const DialogDate &dialog_date) {
        auto dialog_id = dialog_date.get_dialog_id();
        if (dialog_id.get_type() == DialogType::SecretChat && !have_dialog_force(dialog_id, "get_dialogs 2")) {
          list.pinned_dialog_id_orders_.erase(dialog_id);
          return true;
        }
        return false;
      });
      save_pinned_folder_dialog_ids(list);
    }
  }
  update_list_last_pinned_dialog_date(list);

  vector<const DialogFolder *> folders;
  vector<std::set<DialogDate>::const_iterator> folder_iterators;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    folders.push_back(get_dialog_folder(folder_id));
    folder_iterators.push_back(folders.back()->ordered_dialogs_.upper_bound(offset));
  }
  while (limit > 0) {
    size_t best_pos = 0;
    DialogDate best_dialog_date = MAX_DIALOG_DATE;
    for (size_t i = 0; i < folders.size(); i++) {
      while (folder_iterators[i] != folders[i]->ordered_dialogs_.end() &&
             *folder_iterators[i] <= list.list_last_dialog_date_ &&
             (!is_dialog_in_list(get_dialog(folder_iterators[i]->get_dialog_id()), dialog_list_id) ||
              get_dialog_pinned_order(&list, folder_iterators[i]->get_dialog_id()) != DEFAULT_ORDER)) {
        ++folder_iterators[i];
      }
      if (folder_iterators[i] != folders[i]->ordered_dialogs_.end() &&
          *folder_iterators[i] <= list.list_last_dialog_date_ && *folder_iterators[i] < best_dialog_date) {
        best_pos = i;
        best_dialog_date = *folder_iterators[i];
      }
    }
    if (best_dialog_date == MAX_DIALOG_DATE || best_dialog_date.get_order() == DEFAULT_ORDER) {
      break;
    }

    limit--;
    result.push_back(folder_iterators[best_pos]->get_dialog_id());
    ++folder_iterators[best_pos];
  }

  if ((!result.empty() && (!exact_limit || limit == 0)) || force || list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    if (limit > 0 && list.list_last_dialog_date_ != MAX_DIALOG_DATE) {
      LOG(INFO) << "Preload next " << limit << " chats in " << dialog_list_id;
      load_dialog_list(list, limit, Promise<Unit>());
    }

    promise.set_value(Unit());
    return result;
  } else {
    if (!result.empty()) {
      LOG(INFO) << "Have only " << result.size() << " chats, but " << limit << " chats more are needed";
    }
    load_dialog_list(list, limit, std::move(promise));
    return {};
  }
}

void MessagesManager::load_dialog_list(DialogList &list, int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  if (limit > MAX_GET_DIALOGS + 2) {
    limit = MAX_GET_DIALOGS + 2;
  }
  bool is_request_sent = false;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    const auto &folder = *get_dialog_folder(folder_id);
    if (folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
      load_folder_dialog_list(folder_id, limit, false);
      is_request_sent = true;
    }
  }
  if (is_request_sent) {
    LOG(INFO) << "Wait for loading of " << limit << " chats in " << list.dialog_list_id;
    list.load_list_queries_.push_back(std::move(promise));
  } else {
    LOG(ERROR) << "There is nothing to load for " << list.dialog_list_id << " with folders "
               << get_dialog_list_folder_ids(list);
    promise.set_error(Status::Error(404, "Not Found"));
  }
}

void MessagesManager::load_folder_dialog_list(FolderId folder_id, int32 limit, bool only_local) {
  if (G()->close_flag()) {
    return;
  }

  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  if (folder.folder_last_dialog_date_ == MAX_DIALOG_DATE) {
    return;
  }

  bool use_database = G()->use_message_database() &&
                      folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_;
  if (only_local && !use_database) {
    return;
  }

  auto &multipromise = folder.load_folder_dialog_list_multipromise_;
  if (multipromise.promise_count() != 0) {
    // queries have already been sent, just wait for the result
    LOG(INFO) << "Skip loading of dialog list in " << folder_id << " with limit " << limit
              << ", because it is already being loaded";
    if (use_database && folder.load_dialog_list_limit_max_ != 0) {
      folder.load_dialog_list_limit_max_ = max(folder.load_dialog_list_limit_max_, limit);
    }
    return;
  }
  LOG(INFO) << "Load chat list in " << folder_id << " with limit " << limit;
  multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<Unit> result) {
    send_closure_later(actor_id, &MessagesManager::on_load_folder_dialog_list, folder_id, std::move(result));
  }));

  bool is_query_sent = false;
  if (use_database) {
    load_folder_dialog_list_from_database(folder_id, limit, multipromise.get_promise());
    is_query_sent = true;
  } else {
    LOG(INFO) << "Get chats from " << folder.last_server_dialog_date_;
    multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<Unit> result) {
      if (result.is_ok()) {
        send_closure(actor_id, &MessagesManager::recalc_unread_count, DialogListId(folder_id), -1, true);
      }
    }));
    auto lock = multipromise.get_promise();
    reload_pinned_dialogs(DialogListId(folder_id), multipromise.get_promise());
    if (folder.folder_last_dialog_date_ == folder.last_server_dialog_date_) {
      td_->create_handler<GetDialogListQuery>(multipromise.get_promise())
          ->send(folder_id, folder.last_server_dialog_date_.get_date(),
                 folder.last_server_dialog_date_.get_message_id().get_next_server_message_id().get_server_message_id(),
                 folder.last_server_dialog_date_.get_dialog_id(), int32{MAX_GET_DIALOGS});
      is_query_sent = true;
    }
    if (folder_id == FolderId::main() && folder.last_server_dialog_date_ == MIN_DIALOG_DATE) {
      // do not pass promise to not wait for drafts before showing the chat list
      load_all_draft_messages(td_);
    }
    lock.set_value(Unit());
  }
  CHECK(is_query_sent);
}

void MessagesManager::on_load_folder_dialog_list(FolderId folder_id, Result<Unit> &&result) {
  if (G()->close_flag()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());

  const auto &folder = *get_dialog_folder(folder_id);
  if (result.is_ok()) {
    LOG(INFO) << "Successfully loaded chats in " << folder_id;
    if (folder.last_server_dialog_date_ == MAX_DIALOG_DATE) {
      return;
    }

    bool need_new_get_dialog_list = false;
    for (const auto &list_it : dialog_lists_) {
      auto &list = list_it.second;
      if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder)) {
        LOG(INFO) << "Need to load more chats in " << folder_id << " for " << list_it.first;
        need_new_get_dialog_list = true;
      }
    }
    if (need_new_get_dialog_list) {
      load_folder_dialog_list(folder_id, int32{MAX_GET_DIALOGS}, false);
    }
    return;
  }

  LOG(WARNING) << "Failed to load chats in " << folder_id << ": " << result.error();
  vector<Promise<Unit>> promises;
  for (auto &list_it : dialog_lists_) {
    auto &list = list_it.second;
    if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder)) {
      append(promises, std::move(list.load_list_queries_));
      list.load_list_queries_.clear();
    }
  }

  fail_promises(promises, result.move_as_error());
}

void MessagesManager::load_folder_dialog_list_from_database(FolderId folder_id, int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  LOG(INFO) << "Load " << limit << " chats in " << folder_id << " from database from "
            << folder.last_loaded_database_dialog_date_
            << ", last database server dialog date = " << folder.last_database_server_dialog_date_;

  CHECK(folder.load_dialog_list_limit_max_ == 0);
  folder.load_dialog_list_limit_max_ = limit;
  G()->td_db()->get_dialog_db_async()->get_dialogs(
      folder_id, folder.last_loaded_database_dialog_date_.get_order(),
      folder.last_loaded_database_dialog_date_.get_dialog_id(), limit,
      PromiseCreator::lambda([actor_id = actor_id(this), folder_id, limit,
                              promise = std::move(promise)](DialogDbGetDialogsResult result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_dialogs_from_database, folder_id, limit, std::move(result),
                     std::move(promise));
      }));
}

void MessagesManager::on_get_dialogs_from_database(FolderId folder_id, int32 limit, DialogDbGetDialogsResult &&dialogs,
                                                   Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  LOG(INFO) << "Receive " << dialogs.dialogs.size() << " from expected " << limit << " chats in " << folder_id
            << " in from database with next order " << dialogs.next_order << " and next " << dialogs.next_dialog_id;
  int32 new_get_dialogs_limit = 0;
  int32 have_more_dialogs_in_database = (limit == static_cast<int32>(dialogs.dialogs.size()));
  if (have_more_dialogs_in_database && limit < folder.load_dialog_list_limit_max_) {
    new_get_dialogs_limit = folder.load_dialog_list_limit_max_ - limit;
  }
  folder.load_dialog_list_limit_max_ = 0;

  size_t dialogs_skipped = 0;
  for (auto &dialog : dialogs.dialogs) {
    Dialog *d = on_load_dialog_from_database(DialogId(), std::move(dialog), "on_get_dialogs_from_database");
    if (d == nullptr) {
      dialogs_skipped++;
      continue;
    }
    if (d->folder_id != folder_id) {
      LOG(INFO) << "Skip " << d->dialog_id << " received from database, because it is in " << d->folder_id
                << " instead of " << folder_id;
      dialogs_skipped++;
      continue;
    }

    LOG(INFO) << "Loaded from database " << d->dialog_id << " with order " << d->order;
  }

  DialogDate max_dialog_date(dialogs.next_order, dialogs.next_dialog_id);
  if (!have_more_dialogs_in_database) {
    folder.last_loaded_database_dialog_date_ = MAX_DIALOG_DATE;
    LOG(INFO) << "Set last loaded database dialog date to " << folder.last_loaded_database_dialog_date_;
    folder.last_server_dialog_date_ = max(folder.last_server_dialog_date_, folder.last_database_server_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << folder.last_server_dialog_date_;
    update_last_dialog_date(folder_id);
  } else if (folder.last_loaded_database_dialog_date_ < max_dialog_date) {
    folder.last_loaded_database_dialog_date_ = min(max_dialog_date, folder.last_database_server_dialog_date_);
    LOG(INFO) << "Set last loaded database dialog date to " << folder.last_loaded_database_dialog_date_;
    folder.last_server_dialog_date_ = max(folder.last_server_dialog_date_, folder.last_loaded_database_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << folder.last_server_dialog_date_;
    update_last_dialog_date(folder_id);

    for (const auto &list_it : dialog_lists_) {
      auto &list = list_it.second;
      if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder) && new_get_dialogs_limit < limit) {
        new_get_dialogs_limit = limit;
      }
    }
  } else {
    LOG(ERROR) << "Last loaded database dialog date didn't increased, skipped " << dialogs_skipped << " chats out of "
               << dialogs.dialogs.size();
  }

  if (!(folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_)) {
    // have_more_dialogs_in_database = false;
    new_get_dialogs_limit = 0;
  }

  if (new_get_dialogs_limit == 0) {
    preload_folder_dialog_list_timeout_.add_timeout_in(folder_id.get(), 0.2);
    promise.set_value(Unit());
  } else {
    load_folder_dialog_list_from_database(folder_id, new_get_dialogs_limit, std::move(promise));
  }
}

void MessagesManager::preload_folder_dialog_list(FolderId folder_id) {
  if (G()->close_flag()) {
    LOG(INFO) << "Skip chat list preload in " << folder_id << " because of closing";
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());

  auto &folder = *get_dialog_folder(folder_id);
  CHECK(G()->use_message_database());
  if (folder.load_folder_dialog_list_multipromise_.promise_count() != 0) {
    LOG(INFO) << "Skip chat list preload in " << folder_id << ", because there is a pending load chat list request";
    return;
  }

  if (folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_) {
    // if there are some dialogs in database, preload some of them
    load_folder_dialog_list(folder_id, 20, true);
  } else if (folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
    // otherwise load more dialogs from the server
    load_folder_dialog_list(folder_id, MAX_GET_DIALOGS, false);
  } else {
    recalc_unread_count(DialogListId(folder_id), -1, false);
  }
}

void MessagesManager::get_dialogs_from_list(DialogListId dialog_list_id, int32 limit,
                                            Promise<td_api::object_ptr<td_api::chats>> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  if (get_dialog_list(dialog_list_id) == nullptr) {
    return promise.set_error(Status::Error(400, "Chat list not found"));
  }

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }

  auto task_id = ++current_get_dialogs_task_id_;
  auto &task = get_dialogs_tasks_[task_id];
  task.dialog_list_id = dialog_list_id;
  task.retry_count = 5;
  task.limit = limit;
  task.promise = std::move(promise);
  get_dialogs_from_list_impl(task_id);
}

void MessagesManager::get_dialogs_from_list_impl(int64 task_id) {
  auto task_it = get_dialogs_tasks_.find(task_id);
  CHECK(task_it != get_dialogs_tasks_.end());
  auto &task = task_it->second;
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), task_id](Result<Unit> &&result) {
    // on_get_dialogs_from_list can delete get_dialogs_tasks_[task_id], so it must be called later
    send_closure_later(actor_id, &MessagesManager::on_get_dialogs_from_list, task_id, std::move(result));
  });
  auto dialog_ids = get_dialogs(task.dialog_list_id, MIN_DIALOG_DATE, task.limit, true, false, std::move(promise));
  auto &list = *get_dialog_list(task.dialog_list_id);
  auto total_count = get_dialog_total_count(list);
  LOG(INFO) << "Receive " << dialog_ids.size() << " chats instead of " << task.limit << " out of " << total_count
            << " in " << task.dialog_list_id;
  CHECK(dialog_ids.size() <= static_cast<size_t>(total_count));
  CHECK(dialog_ids.size() <= static_cast<size_t>(task.limit));
  if (dialog_ids.size() == static_cast<size_t>(min(total_count, task.limit)) ||
      list.list_last_dialog_date_ == MAX_DIALOG_DATE || task.retry_count == 0) {
    auto task_promise = std::move(task.promise);
    get_dialogs_tasks_.erase(task_it);
    if (!task_promise) {
      dialog_ids.clear();
    }
    return task_promise.set_value(
        td_->dialog_manager_->get_chats_object(total_count, dialog_ids, "get_dialogs_from_list_impl"));
  }
  // nor the limit, nor the end of the list were reached; wait for the promise
}

void MessagesManager::on_get_dialogs_from_list(int64 task_id, Result<Unit> &&result) {
  auto task_it = get_dialogs_tasks_.find(task_id);
  if (task_it == get_dialogs_tasks_.end()) {
    // the task has already been completed
    LOG(INFO) << "Chat list load task " << task_id << " has already been completed";
    return;
  }
  auto &task = task_it->second;
  if (result.is_error()) {
    LOG(INFO) << "Chat list load task " << task_id << " failed with the error " << result.error();
    auto task_promise = std::move(task.promise);
    get_dialogs_tasks_.erase(task_it);
    return task_promise.set_error(result.move_as_error());
  }

  auto list_ptr = get_dialog_list(task.dialog_list_id);
  CHECK(list_ptr != nullptr);
  auto &list = *list_ptr;
  if (task.last_dialog_date == list.list_last_dialog_date_) {
    // no new chats were loaded
    task.retry_count--;
  } else {
    CHECK(task.last_dialog_date < list.list_last_dialog_date_);
    task.last_dialog_date = list.list_last_dialog_date_;
    task.retry_count = 5;
  }
  get_dialogs_from_list_impl(task_id);
}

void MessagesManager::mark_dialog_as_read(Dialog *d) {
  if (td_->dialog_manager_->is_forum_channel(d->dialog_id)) {
    // TODO read forum topics
  }
  if (d->server_unread_count + d->local_unread_count > 0 && d->last_message_id.is_valid()) {
    auto it = d->ordered_messages.get_const_iterator(d->last_message_id);
    while (*it != nullptr) {
      auto message_id = (*it)->get_message_id();
      if (message_id.is_server() || message_id.is_local()) {
        read_dialog_inbox(d, message_id);
        break;
      }
      --it;
    }
    if (*it == nullptr) {
      // if can't find the last message, then d->last_new_message_id is the best guess
      read_dialog_inbox(d, d->last_new_message_id);
    }
  }
  if (d->is_marked_as_unread) {
    set_dialog_is_marked_as_unread(d, false);
  }
}

void MessagesManager::read_all_dialogs_from_list(DialogListId dialog_list_id, Promise<Unit> &&promise,
                                                 bool is_recursive) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (get_dialog_list(dialog_list_id) == nullptr) {
    return promise.set_error(Status::Error(400, "Chat list not found"));
  }

  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    Dialog *d = dialog.get();
    if (is_dialog_in_list(d, dialog_list_id)) {
      mark_dialog_as_read(d);
    }
  });

  if (is_recursive) {
    return promise.set_value(Unit());
  }

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_list_id, promise = std::move(promise)](
                                                  Result<td_api::object_ptr<td_api::chats>> &&) mutable {
    // it is expected that loading of so many chats will fail, so just ignore the error and result itself
    send_closure(actor_id, &MessagesManager::read_all_dialogs_from_list, dialog_list_id, std::move(promise), true);
  });
  get_dialogs_from_list(dialog_list_id, 10000, std::move(query_promise));
}

vector<DialogId> MessagesManager::get_pinned_dialog_ids(DialogListId dialog_list_id) const {
  CHECK(!td_->auth_manager_->is_bot());
  if (dialog_list_id.is_filter()) {
    return td_->dialog_filter_manager_->get_pinned_dialog_ids(dialog_list_id.get_filter_id());
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr || !list->are_pinned_dialogs_inited_) {
    return {};
  }
  return transform(list->pinned_dialogs_, [](auto &pinned_dialog) { return pinned_dialog.get_dialog_id(); });
}

void MessagesManager::reload_pinned_dialogs(DialogListId dialog_list_id, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  CHECK(!td_->auth_manager_->is_bot());

  if (dialog_list_id.is_folder()) {
    td_->create_handler<GetPinnedDialogsQuery>(std::move(promise))->send(dialog_list_id.get_folder_id());
  } else if (dialog_list_id.is_filter()) {
    td_->dialog_filter_manager_->schedule_reload_dialog_filters(std::move(promise));
  }
}

vector<DialogId> MessagesManager::search_public_dialogs(const string &query, Promise<Unit> &&promise) {
  LOG(INFO) << "Search public chats with query = \"" << query << '"';

  auto query_length = utf8_length(query);
  if (query_length < MIN_SEARCH_PUBLIC_DIALOG_PREFIX_LEN ||
      (query_length == MIN_SEARCH_PUBLIC_DIALOG_PREFIX_LEN && query[0] == '@')) {
    string username = clean_username(query);
    if (username[0] == '@') {
      username = username.substr(1);
    }

    for (auto &short_username : get_valid_short_usernames()) {
      if (2 * username.size() > short_username.size() && begins_with(short_username, username)) {
        username = short_username.str();
        auto dialog_id = td_->dialog_manager_->resolve_dialog_username(username, promise);
        if (!dialog_id.is_valid()) {
          return {};
        }

        force_create_dialog(dialog_id, "search_public_dialogs");

        auto d = get_dialog(dialog_id);
        if (d == nullptr || d->order != DEFAULT_ORDER ||
            (dialog_id.get_type() == DialogType::User &&
             td_->user_manager_->is_user_contact(dialog_id.get_user_id()))) {
          continue;
        }

        promise.set_value(Unit());
        return {dialog_id};
      }
    }
    promise.set_value(Unit());
    return {};
  }

  auto it = found_public_dialogs_.find(query);
  if (it != found_public_dialogs_.end()) {
    promise.set_value(Unit());
    return it->second;
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return {};
}

void MessagesManager::send_search_public_dialogs_query(const string &query, Promise<Unit> &&promise) {
  CHECK(!query.empty());
  auto &promises = search_public_dialogs_queries_[query];
  promises.push_back(std::move(promise));
  if (promises.size() != 1) {
    // query has already been sent, just wait for the result
    return;
  }

  td_->create_handler<SearchPublicDialogsQuery>()->send(query);
}

std::pair<int32, vector<DialogId>> MessagesManager::search_dialogs(const string &query, int32 limit,
                                                                   Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats with query \"" << query << "\" and limit " << limit;
  CHECK(!td_->auth_manager_->is_bot());

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (query.empty()) {
    return recently_found_dialogs_.get_dialogs(limit, std::move(promise));
  }

  auto result = dialogs_hints_.search(query, limit);
  vector<DialogId> dialog_ids;
  dialog_ids.reserve(result.second.size());
  for (auto key : result.second) {
    dialog_ids.push_back(DialogId(-key));
  }

  promise.set_value(Unit());
  return {narrow_cast<int32>(result.first), std::move(dialog_ids)};
}

std::pair<int32, vector<DialogId>> MessagesManager::search_recently_found_dialogs(const string &query, int32 limit,
                                                                                  Promise<Unit> &&promise) {
  auto result = recently_found_dialogs_.get_dialogs(query.empty() ? limit : 50, std::move(promise));
  if (result.first == 0 || query.empty()) {
    return result;
  }

  Hints hints;
  int rating = 1;
  for (auto dialog_id : result.second) {
    hints.add(dialog_id.get(), td_->dialog_manager_->get_dialog_search_text(dialog_id));
    hints.set_rating(dialog_id.get(), ++rating);
  }

  auto hints_result = hints.search(query, limit, false);
  return {narrow_cast<int32>(hints_result.first),
          transform(hints_result.second, [](int64 key) { return DialogId(key); })};
}

std::pair<int32, vector<DialogId>> MessagesManager::get_recently_opened_dialogs(int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  return recently_opened_dialogs_.get_dialogs(limit, std::move(promise));
}

vector<DialogId> MessagesManager::sort_dialogs_by_order(const vector<DialogId> &dialog_ids, int32 limit) const {
  CHECK(!td_->auth_manager_->is_bot());
  auto fake_order = static_cast<int64>(dialog_ids.size()) + 1;
  auto dialog_dates = transform(dialog_ids, [this, &fake_order](DialogId dialog_id) {
    const Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    auto order = get_dialog_base_order(d);
    if (is_dialog_inited(d) || order != DEFAULT_ORDER) {
      return DialogDate(order, dialog_id);
    }
    // if the dialog is not inited yet, we need to assume that server knows better and the dialog needs to be returned
    return DialogDate(fake_order--, dialog_id);
  });
  if (static_cast<size_t>(limit) >= dialog_dates.size()) {
    std::sort(dialog_dates.begin(), dialog_dates.end());
  } else {
    std::partial_sort(dialog_dates.begin(), dialog_dates.begin() + limit, dialog_dates.end());
    dialog_dates.resize(limit, MAX_DIALOG_DATE);
  }
  while (!dialog_dates.empty() && dialog_dates.back().get_order() == DEFAULT_ORDER) {
    dialog_dates.pop_back();
  }
  return transform(dialog_dates, [](auto dialog_date) { return dialog_date.get_dialog_id(); });
}

vector<DialogId> MessagesManager::search_dialogs_on_server(const string &query, int32 limit, Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats on server with query \"" << query << "\" and limit " << limit;

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }

  auto it = found_on_server_dialogs_.find(query);
  if (it != found_on_server_dialogs_.end()) {
    promise.set_value(Unit());
    return sort_dialogs_by_order(it->second, limit);
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return {};
}

void MessagesManager::block_message_sender_from_replies(MessageId message_id, bool need_delete_message,
                                                        bool need_delete_all_messages, bool report_spam,
                                                        Promise<Unit> &&promise) {
  auto dialog_id = DialogId(UserManager::get_replies_bot_user_id());
  TRY_RESULT_PROMISE(promise, d,
                     check_dialog_access(dialog_id, false, AccessRights::Read, "block_message_sender_from_replies"));

  auto *m = get_message_force(d, message_id, "block_message_sender_from_replies");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }
  if (m->is_outgoing || m->message_id.is_scheduled() || !m->message_id.is_server()) {
    return promise.set_error(Status::Error(400, "Wrong message specified"));
  }

  DialogId sender_dialog_id;
  if (m->forward_info != nullptr) {
    sender_dialog_id = m->forward_info->get_origin().get_sender();
  }
  vector<MessageId> message_ids;
  if (need_delete_all_messages && sender_dialog_id.is_valid()) {
    message_ids = find_dialog_messages(d, [sender_dialog_id](const Message *m) {
      return !m->is_outgoing && m->forward_info != nullptr &&
             m->forward_info->get_origin().get_sender() == sender_dialog_id;
    });
    CHECK(td::contains(message_ids, message_id));
  } else if (need_delete_message) {
    message_ids.push_back(message_id);
  }

  delete_dialog_messages(d, message_ids, false, DELETE_MESSAGE_USER_REQUEST_SOURCE);

  block_message_sender_from_replies_on_server(message_id, need_delete_message, need_delete_all_messages, report_spam, 0,
                                              std::move(promise));
}

class MessagesManager::BlockMessageSenderFromRepliesOnServerLogEvent {
 public:
  MessageId message_id_;
  bool delete_message_;
  bool delete_all_messages_;
  bool report_spam_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(delete_message_);
    STORE_FLAG(delete_all_messages_);
    STORE_FLAG(report_spam_);
    END_STORE_FLAGS();

    td::store(message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(delete_message_);
    PARSE_FLAG(delete_all_messages_);
    PARSE_FLAG(report_spam_);
    END_PARSE_FLAGS();

    td::parse(message_id_, parser);
  }
};

uint64 MessagesManager::save_block_message_sender_from_replies_on_server_log_event(MessageId message_id,
                                                                                   bool need_delete_message,
                                                                                   bool need_delete_all_messages,
                                                                                   bool report_spam) {
  BlockMessageSenderFromRepliesOnServerLogEvent log_event{message_id, need_delete_message, need_delete_all_messages,
                                                          report_spam};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::BlockMessageSenderFromRepliesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::block_message_sender_from_replies_on_server(MessageId message_id, bool need_delete_message,
                                                                  bool need_delete_all_messages, bool report_spam,
                                                                  uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0) {
    log_event_id = save_block_message_sender_from_replies_on_server_log_event(message_id, need_delete_message,
                                                                              need_delete_all_messages, report_spam);
  }

  td_->create_handler<BlockFromRepliesQuery>(get_erase_log_event_promise(log_event_id, std::move(promise)))
      ->send(message_id, need_delete_message, need_delete_all_messages, report_spam);
}

bool MessagesManager::is_dialog_blocked(DialogId dialog_id) const {
  const Dialog *d = get_dialog(dialog_id);
  return d != nullptr && d->is_blocked;
}

void MessagesManager::get_blocked_dialogs(const td_api::object_ptr<td_api::BlockList> &block_list, int32 offset,
                                          int32 limit, Promise<td_api::object_ptr<td_api::messageSenders>> &&promise) {
  if (offset < 0) {
    return promise.set_error(Status::Error(400, "Parameter offset must be non-negative"));
  }

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }

  auto block_list_id = BlockListId(block_list);
  if (!block_list_id.is_valid()) {
    return promise.set_error(Status::Error(400, "Block list must be non-empty"));
  }

  td_->create_handler<GetBlockedDialogsQuery>(std::move(promise))->send(block_list_id, offset, limit);
}

void MessagesManager::on_get_blocked_dialogs(int32 offset, int32 limit, int32 total_count,
                                             vector<tl_object_ptr<telegram_api::peerBlocked>> &&blocked_peers,
                                             Promise<td_api::object_ptr<td_api::messageSenders>> &&promise) {
  LOG(INFO) << "Receive " << blocked_peers.size() << " blocked chats from offset " << offset << " out of "
            << total_count;
  auto peers = transform(std::move(blocked_peers), [](tl_object_ptr<telegram_api::peerBlocked> &&blocked_peer) {
    return std::move(blocked_peer->peer_id_);
  });
  auto dialog_ids = get_message_sender_dialog_ids(td_, std::move(peers));
  if (!dialog_ids.empty() && offset + dialog_ids.size() > static_cast<size_t>(total_count)) {
    LOG(ERROR) << "Fix total count of blocked chats from " << total_count << " to " << offset + dialog_ids.size();
    total_count = offset + narrow_cast<int32>(dialog_ids.size());
  }

  auto senders = transform(dialog_ids, [td = td_](DialogId dialog_id) {
    return get_message_sender_object(td, dialog_id, "on_get_blocked_dialogs");
  });
  promise.set_value(td_api::make_object<td_api::messageSenders>(total_count, std::move(senders)));
}

DialogId MessagesManager::get_dialog_message_sender(MessageFullId message_full_id) {
  const auto *m = get_message_force(message_full_id, "get_dialog_message_sender");
  if (m == nullptr) {
    return DialogId();
  }
  return get_message_sender(m);
}

bool MessagesManager::have_message_force(MessageFullId message_full_id, const char *source) {
  return get_message_force(message_full_id, source) != nullptr;
}

bool MessagesManager::have_message_force(Dialog *d, MessageId message_id, const char *source) {
  return get_message_force(d, message_id, source) != nullptr;
}

MessagesManager::Message *MessagesManager::get_message(MessageFullId message_full_id) {
  Dialog *d = get_dialog(message_full_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, message_full_id.get_message_id());
}

const MessagesManager::Message *MessagesManager::get_message(MessageFullId message_full_id) const {
  const Dialog *d = get_dialog(message_full_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, message_full_id.get_message_id());
}

MessagesManager::Message *MessagesManager::get_message_force(MessageFullId message_full_id, const char *source) {
  Dialog *d = get_dialog_force(message_full_id.get_dialog_id(), source);
  if (d == nullptr) {
    return nullptr;
  }

  return get_message_force(d, message_full_id.get_message_id(), source);
}

MessageFullId MessagesManager::get_replied_message_id(DialogId dialog_id, const Message *m) {
  if (m->reply_to_story_full_id.is_valid()) {
    return {};
  }
  auto message_full_id = get_message_content_replied_message_id(dialog_id, m->content.get());
  if (message_full_id.get_message_id().is_valid()) {
    CHECK(m->replied_message_info.is_empty());
    return message_full_id;
  }
  auto reply_message_full_id = m->replied_message_info.get_reply_message_full_id(dialog_id, true);
  if (reply_message_full_id.get_message_id() != MessageId()) {
    return reply_message_full_id;
  }
  if (m->top_thread_message_id.is_valid() && m->top_thread_message_id != m->message_id) {
    return {dialog_id, m->top_thread_message_id};
  }
  return {};
}

void MessagesManager::get_message_force_from_server(Dialog *d, MessageId message_id, Promise<Unit> &&promise,
                                                    tl_object_ptr<telegram_api::InputMessage> input_message) {
  LOG(INFO) << "Get " << message_id << " in " << d->dialog_id << " using " << to_string(input_message);
  auto dialog_type = d->dialog_id.get_type();
  auto m = get_message_force(d, message_id, "get_message_force_from_server");
  if (m == nullptr && !is_deleted_message(d, message_id) && dialog_type != DialogType::SecretChat) {
    if (message_id.is_valid() && message_id.is_server()) {
      if (d->last_new_message_id != MessageId() && message_id > d->last_new_message_id &&
          dialog_type != DialogType::Channel && !td_->auth_manager_->is_bot()) {
        // message will not be added to the dialog anyway
        return promise.set_value(Unit());
      }

      return get_message_from_server({d->dialog_id, message_id}, std::move(promise), "get_message_force_from_server",
                                     std::move(input_message));
    }
    if (message_id.is_valid_scheduled() && message_id.is_scheduled_server() && input_message == nullptr) {
      return get_message_from_server({d->dialog_id, message_id}, std::move(promise), "get_message_force_from_server");
    }
  }

  promise.set_value(Unit());
}

void MessagesManager::get_message(MessageFullId message_full_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(message_full_id.get_dialog_id(), "get_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  get_message_force_from_server(d, message_full_id.get_message_id(), std::move(promise));
}

MessageFullId MessagesManager::get_replied_message(DialogId dialog_id, MessageId message_id, bool force,
                                                   Promise<Unit> &&promise) {
  LOG(INFO) << "Get replied message to " << message_id << " in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "get_replied_message");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return MessageFullId();
  }

  message_id = get_persistent_message_id(d, message_id);
  auto m = get_message_force(d, message_id, "get_replied_message");
  if (m == nullptr) {
    if (force) {
      promise.set_value(Unit());
    } else {
      get_message_force_from_server(d, message_id, std::move(promise));
    }
    return MessageFullId();
  }

  tl_object_ptr<telegram_api::InputMessage> input_message;
  auto replied_message_id = get_replied_message_id(dialog_id, m);
  if (replied_message_id.get_dialog_id() != dialog_id) {
    dialog_id = replied_message_id.get_dialog_id();
    if (!td_->dialog_manager_->have_dialog_info_force(dialog_id, "get_replied_message")) {
      promise.set_value(Unit());
      return {};
    }
    if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
      promise.set_value(Unit());
      return {};
    }

    force_create_dialog(dialog_id, "get_replied_message");
    d = get_dialog_force(dialog_id, "get_replied_message");
    if (d == nullptr) {
      promise.set_error(Status::Error(500, "Chat with replied message not found"));
      return {};
    }
  } else if (m->message_id.is_valid() && m->message_id.is_server()) {
    input_message = make_tl_object<telegram_api::inputMessageReplyTo>(m->message_id.get_server_message_id().get());
  }
  get_message_force_from_server(d, replied_message_id.get_message_id(), std::move(promise), std::move(input_message));

  return replied_message_id;
}

Result<MessageFullId> MessagesManager::get_top_thread_message_full_id(DialogId dialog_id, const Message *m,
                                                                      bool allow_non_root) const {
  CHECK(m != nullptr);
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Message is scheduled");
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    return Status::Error(400, "Chat can't have message threads");
  }
  if (!m->reply_info.is_empty() && m->reply_info.is_comment_) {
    if (!is_visible_message_reply_info(dialog_id, m)) {
      return Status::Error(400, "Message has no comments");
    }
    if (m->message_id.is_yet_unsent()) {
      return Status::Error(400, "Message is not sent yet");
    }
    return MessageFullId{DialogId(m->reply_info.channel_id_), m->linked_top_thread_message_id};
  } else {
    if (!m->top_thread_message_id.is_valid()) {
      return Status::Error(400, "Message has no thread");
    }
    if (!allow_non_root && m->top_thread_message_id != m->message_id &&
        !td_->chat_manager_->get_channel_has_linked_channel(dialog_id.get_channel_id())) {
      return Status::Error(400, "Root message must be used to get the message thread");
    }
    return MessageFullId{dialog_id, m->top_thread_message_id};
  }
}

void MessagesManager::get_message_thread(DialogId dialog_id, MessageId message_id,
                                         Promise<MessageThreadInfo> &&promise) {
  LOG(INFO) << "Get message thread from " << message_id << " in " << dialog_id;
  TRY_RESULT_PROMISE(promise, d, check_dialog_access(dialog_id, false, AccessRights::Read, "get_message_thread"));
  if (dialog_id.get_type() != DialogType::Channel) {
    return promise.set_error(Status::Error(400, "Chat is not a supergroup or a channel"));
  }
  if (message_id.is_scheduled()) {
    return promise.set_error(Status::Error(400, "Scheduled messages can't have message threads"));
  }

  MessageFullId top_thread_message_full_id;
  if (message_id == MessageId(ServerMessageId(1)) && td_->dialog_manager_->is_forum_channel(dialog_id)) {
    top_thread_message_full_id = MessageFullId{dialog_id, message_id};
  } else {
    message_id = get_persistent_message_id(d, message_id);
    auto m = get_message_force(d, message_id, "get_message_thread");
    if (m == nullptr) {
      return promise.set_error(Status::Error(400, "Message not found"));
    }

    TRY_RESULT_PROMISE_ASSIGN(promise, top_thread_message_full_id, get_top_thread_message_full_id(dialog_id, m, true));
    if ((m->reply_info.is_empty() || !m->reply_info.is_comment_) &&
        top_thread_message_full_id.get_message_id() != m->message_id) {
      CHECK(dialog_id == top_thread_message_full_id.get_dialog_id());
      // get information about the thread from the top message
      message_id = top_thread_message_full_id.get_message_id();
      CHECK(message_id.is_valid());
    }
  }

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, message_id,
                                               promise = std::move(promise)](Result<MessageThreadInfo> result) mutable {
    if (result.is_error()) {
      return promise.set_error(result.move_as_error());
    }
    send_closure(actor_id, &MessagesManager::on_get_discussion_message, dialog_id, message_id, result.move_as_ok(),
                 std::move(promise));
  });

  td_->create_handler<GetDiscussionMessageQuery>(std::move(query_promise))
      ->send(dialog_id, message_id, top_thread_message_full_id.get_dialog_id(),
             top_thread_message_full_id.get_message_id());
}

void MessagesManager::process_discussion_message(
    telegram_api::object_ptr<telegram_api::messages_discussionMessage> &&result, DialogId dialog_id,
    MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id,
    Promise<MessageThreadInfo> promise) {
  LOG(INFO) << "Receive discussion message for " << message_id << " in " << dialog_id << " with expected "
            << expected_message_id << " in " << expected_dialog_id << ": " << to_string(result);
  td_->user_manager_->on_get_users(std::move(result->users_), "process_discussion_message");
  td_->chat_manager_->on_get_chats(std::move(result->chats_), "process_discussion_message");

  for (auto &message : result->messages_) {
    auto message_dialog_id = DialogId::get_message_dialog_id(message);
    if (message_dialog_id != expected_dialog_id) {
      return promise.set_error(Status::Error(500, "Expected messages in a different chat"));
    }
  }

  for (auto &message : result->messages_) {
    if (need_channel_difference_to_add_message(expected_dialog_id, message)) {
      auto max_message_id = MessageId::get_max_message_id(result->messages_);
      return run_after_channel_difference(
          expected_dialog_id, max_message_id,
          PromiseCreator::lambda([actor_id = actor_id(this), result = std::move(result), dialog_id, message_id,
                                  expected_dialog_id, expected_message_id,
                                  promise = std::move(promise)](Unit ignored) mutable {
            send_closure(actor_id, &MessagesManager::process_discussion_message_impl, std::move(result), dialog_id,
                         message_id, expected_dialog_id, expected_message_id, std::move(promise));
          }),
          "process_discussion_message");
    }
  }

  process_discussion_message_impl(std::move(result), dialog_id, message_id, expected_dialog_id, expected_message_id,
                                  std::move(promise));
}

void MessagesManager::process_discussion_message_impl(
    telegram_api::object_ptr<telegram_api::messages_discussionMessage> &&result, DialogId dialog_id,
    MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id,
    Promise<MessageThreadInfo> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  MessageThreadInfo message_thread_info;
  message_thread_info.dialog_id = expected_dialog_id;
  message_thread_info.unread_message_count = max(0, result->unread_count_);
  MessageId top_message_id;
  for (auto &message : result->messages_) {
    auto message_full_id = on_get_message(std::move(message), false, true, false, "process_discussion_message_impl");
    if (message_full_id.get_message_id().is_valid()) {
      CHECK(message_full_id.get_dialog_id() == expected_dialog_id);
      message_thread_info.message_ids.push_back(message_full_id.get_message_id());
      if (message_full_id.get_message_id() == expected_message_id) {
        top_message_id = expected_message_id;
      }
    }
  }
  if (!message_thread_info.message_ids.empty() && !top_message_id.is_valid()) {
    top_message_id = message_thread_info.message_ids.back();
  }
  auto max_message_id = MessageId(ServerMessageId(result->max_id_));
  auto last_read_inbox_message_id = MessageId(ServerMessageId(result->read_inbox_max_id_));
  auto last_read_outbox_message_id = MessageId(ServerMessageId(result->read_outbox_max_id_));
  if (top_message_id.is_valid()) {
    on_update_read_message_comments(expected_dialog_id, top_message_id, max_message_id, last_read_inbox_message_id,
                                    last_read_outbox_message_id, message_thread_info.unread_message_count);
  }
  if (expected_dialog_id != dialog_id) {
    on_update_read_message_comments(dialog_id, message_id, max_message_id, last_read_inbox_message_id,
                                    last_read_outbox_message_id, message_thread_info.unread_message_count);
  }
  promise.set_value(std::move(message_thread_info));
}

void MessagesManager::on_get_discussion_message(DialogId dialog_id, MessageId message_id,
                                                MessageThreadInfo &&message_thread_info,
                                                Promise<MessageThreadInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Dialog *d = get_dialog_force(dialog_id, "on_get_discussion_message");
  CHECK(d != nullptr);

  CHECK(message_id.is_valid());
  auto m = get_message_force(d, message_id, "on_get_discussion_message");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }
  if (message_thread_info.message_ids.empty()) {
    if (message_thread_info.dialog_id != dialog_id &&
        !td_->dialog_manager_->have_input_peer(message_thread_info.dialog_id, false, AccessRights::Read)) {
      return promise.set_error(Status::Error(400, "Can't access message comments"));
    }
    return promise.set_error(Status::Error(400, "Message has no thread"));
  }

  DialogId expected_dialog_id;
  if (m->reply_info.is_comment_) {
    if (!is_active_message_reply_info(dialog_id, m->reply_info)) {
      return promise.set_error(Status::Error(400, "Message has no comments"));
    }
    expected_dialog_id = DialogId(m->reply_info.channel_id_);
  } else if (message_id == MessageId(ServerMessageId(1)) && td_->dialog_manager_->is_forum_channel(dialog_id)) {
    // General forum topic
    expected_dialog_id = dialog_id;
  } else {
    if (!m->top_thread_message_id.is_valid()) {
      return promise.set_error(Status::Error(400, "Message has no thread"));
    }
    expected_dialog_id = dialog_id;
  }

  if (expected_dialog_id != dialog_id && m->reply_info.is_comment_ &&
      m->linked_top_thread_message_id != message_thread_info.message_ids.back()) {
    auto linked_d = get_dialog_force(expected_dialog_id, "on_get_discussion_message 2");
    CHECK(linked_d != nullptr);
    td::remove_if(message_thread_info.message_ids, [&](MessageId linked_message_id) {
      return !have_message_force(linked_d, linked_message_id, "on_get_discussion_message 4");
    });
    if (message_thread_info.message_ids.empty()) {
      return promise.set_error(Status::Error(400, "Message has no thread"));
    }
    auto linked_message_id = message_thread_info.message_ids.back();
    Message *linked_m = get_message_force(linked_d, linked_message_id, "on_get_discussion_message 3");
    CHECK(linked_m != nullptr && linked_m->message_id.is_server());
    if (linked_m->top_thread_message_id == linked_m->message_id &&
        is_active_message_reply_info(expected_dialog_id, linked_m->reply_info)) {
      if (m->linked_top_thread_message_id.is_valid()) {
        LOG(ERROR) << "Comment message identifier for " << message_id << " in " << dialog_id << " changed from "
                   << m->linked_top_thread_message_id << " to " << linked_message_id;
      }
      m->linked_top_thread_message_id = linked_message_id;
      on_dialog_updated(dialog_id, "on_get_discussion_message");
    }
  }
  promise.set_value(std::move(message_thread_info));
}

td_api::object_ptr<td_api::messageThreadInfo> MessagesManager::get_message_thread_info_object(
    const MessageThreadInfo &info) {
  if (info.message_ids.empty()) {
    return nullptr;
  }

  Dialog *d = get_dialog(info.dialog_id);
  CHECK(d != nullptr);
  td_api::object_ptr<td_api::messageReplyInfo> reply_info;
  vector<td_api::object_ptr<td_api::message>> messages;
  messages.reserve(info.message_ids.size());
  bool is_forum_topic = false;
  for (auto message_id : info.message_ids) {
    const Message *m = get_message_force(d, message_id, "get_message_thread_info_object");
    auto message = get_message_object(d->dialog_id, m, "get_message_thread_info_object");
    if (message != nullptr) {
      if (message->interaction_info_ != nullptr && message->interaction_info_->reply_info_ != nullptr) {
        reply_info = m->reply_info.get_message_reply_info_object(td_, d->last_read_inbox_message_id);
        CHECK(reply_info != nullptr);
      }
      is_forum_topic = message->is_topic_message_;
      messages.push_back(std::move(message));
    }
  }
  if (messages.size() != 1) {
    is_forum_topic = false;
  } else if (info.message_ids[0] == MessageId(ServerMessageId(1)) &&
             td_->dialog_manager_->is_forum_channel(info.dialog_id)) {
    // General forum topic
    is_forum_topic = true;
  }
  if (reply_info == nullptr && !is_forum_topic) {
    return nullptr;
  }

  MessageId top_thread_message_id;
  td_api::object_ptr<td_api::draftMessage> draft_message;
  if (!info.message_ids.empty()) {
    top_thread_message_id = info.message_ids.back();
    if (can_send_message(d->dialog_id).is_ok()) {
      const Message *m = get_message_force(d, top_thread_message_id, "get_message_thread_info_object 2");
      if (m != nullptr && !m->reply_info.is_comment_ && is_active_message_reply_info(d->dialog_id, m->reply_info)) {
        draft_message = get_draft_message_object(td_, m->thread_draft_message);
      }
    }
  }
  return td_api::make_object<td_api::messageThreadInfo>(
      get_chat_id_object(d->dialog_id, "messageThreadInfo"), top_thread_message_id.get(), std::move(reply_info),
      info.unread_message_count, std::move(messages), std::move(draft_message));
}

Status MessagesManager::can_get_message_read_date(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "User is bot");
  }
  CHECK(m != nullptr);
  if (!m->is_outgoing) {
    return Status::Error(400, "Can't get read date of incoming messages");
  }
  if (G()->unix_time() - m->date > td_->option_manager_->get_option_integer("pm_read_date_expire_period")) {
    return Status::Error(400, "Message is too old");
  }

  if (dialog_id.get_type() != DialogType::User) {
    return Status::Error(400, "Read date can be received only in private chats");
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }
  auto user_id = dialog_id.get_user_id();
  if (td_->user_manager_->is_user_bot(user_id)) {
    return Status::Error(400, "The user is a bot");
  }
  if (td_->user_manager_->is_user_support(user_id)) {
    return Status::Error(400, "The user is a Telegram support account");
  }

  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Scheduled messages can't be read");
  }
  if (m->message_id.is_yet_unsent()) {
    return Status::Error(400, "Yet unsent messages can't be read");
  }
  if (m->message_id.is_local()) {
    return Status::Error(400, "Local messages can't be read");
  }
  CHECK(m->message_id.is_server());

  return Status::OK();
}

void MessagesManager::get_message_read_date(MessageFullId message_full_id,
                                            Promise<td_api::object_ptr<td_api::MessageReadDate>> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "get_message_read_date");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  auto m = get_message_force(d, message_full_id.get_message_id(), "get_message_read_date");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  TRY_STATUS_PROMISE(promise, can_get_message_read_date(dialog_id, m));

  if (d->last_read_outbox_message_id < m->message_id) {
    return promise.set_value(td_api::make_object<td_api::messageReadDateUnread>());
  }
  if (td_->user_manager_->get_user_read_dates_private(dialog_id.get_user_id())) {
    return promise.set_value(td_api::make_object<td_api::messageReadDateUserPrivacyRestricted>());
  }

  td_->create_handler<GetOutboxReadDateQuery>(std::move(promise))
      ->send(message_full_id.get_dialog_id(), message_full_id.get_message_id());
}

Status MessagesManager::can_get_message_viewers(MessageFullId message_full_id) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "can_get_message_viewers");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  auto m = get_message_force(d, message_full_id.get_message_id(), "can_get_message_viewers");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  return can_get_message_viewers(dialog_id, m);
}

Status MessagesManager::can_get_message_viewers(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "User is bot");
  }
  CHECK(m != nullptr);
  if (!m->is_outgoing) {
    return Status::Error(400, "Can't get viewers of incoming messages");
  }
  if (G()->unix_time() - m->date >
      td_->option_manager_->get_option_integer("chat_read_mark_expire_period", 7 * 86400)) {
    return Status::Error(400, "Message is too old");
  }

  int32 participant_count = 0;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return Status::Error(400, "Can't get message viewers in private chats");
    case DialogType::Chat:
      if (!td_->chat_manager_->get_chat_is_active(dialog_id.get_chat_id())) {
        return Status::Error(400, "Chat is deactivated");
      }
      participant_count = td_->chat_manager_->get_chat_participant_count(dialog_id.get_chat_id());
      break;
    case DialogType::Channel:
      if (td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
        return Status::Error(400, "Can't get message viewers in channel chats");
      }
      if (td_->chat_manager_->get_channel_effective_has_hidden_participants(dialog_id.get_channel_id(),
                                                                            "can_get_message_viewers")) {
        return Status::Error(400, "Participant list is hidden in the chat");
      }
      participant_count = td_->chat_manager_->get_channel_participant_count(dialog_id.get_channel_id());
      break;
    case DialogType::SecretChat:
      return Status::Error(400, "Can't get message viewers in secret chats");
    case DialogType::None:
    default:
      UNREACHABLE();
      return Status::OK();
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }
  if (participant_count == 0) {
    return Status::Error(400, "Chat is empty or have unknown number of members");
  }
  if (participant_count > td_->option_manager_->get_option_integer("chat_read_mark_size_threshold", 100)) {
    return Status::Error(400, "Chat is too big");
  }

  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Scheduled messages can't have viewers");
  }
  if (m->message_id.is_yet_unsent()) {
    return Status::Error(400, "Yet unsent messages can't have viewers");
  }
  if (m->message_id.is_local()) {
    return Status::Error(400, "Local messages can't have viewers");
  }
  CHECK(m->message_id.is_server());

  if (m->content->get_type() == MessageContentType::Poll &&
      get_message_content_poll_is_anonymous(td_, m->content.get())) {
    return Status::Error(400, "Anonymous poll viewers are unavailable");
  }

  return Status::OK();
}

void MessagesManager::get_message_viewers(MessageFullId message_full_id,
                                          Promise<td_api::object_ptr<td_api::messageViewers>> &&promise) {
  TRY_STATUS_PROMISE(promise, can_get_message_viewers(message_full_id));

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = message_full_id.get_dialog_id(),
                                               promise = std::move(promise)](Result<MessageViewers> result) mutable {
    if (result.is_error()) {
      return promise.set_error(result.move_as_error());
    }
    send_closure(actor_id, &MessagesManager::on_get_message_viewers, dialog_id, result.move_as_ok(), false,
                 std::move(promise));
  });

  td_->create_handler<GetMessageReadParticipantsQuery>(std::move(query_promise))
      ->send(message_full_id.get_dialog_id(), message_full_id.get_message_id());
}

void MessagesManager::on_get_message_viewers(DialogId dialog_id, MessageViewers message_viewers, bool is_recursive,
                                             Promise<td_api::object_ptr<td_api::messageViewers>> &&promise) {
  if (!is_recursive) {
    bool need_participant_list = false;
    for (auto user_id : message_viewers.get_user_ids()) {
      if (!td_->user_manager_->have_user_force(user_id, "on_get_message_viewers")) {
        need_participant_list = true;
      }
    }
    if (need_participant_list) {
      auto query_promise =
          PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, message_viewers = std::move(message_viewers),
                                  promise = std::move(promise)](Unit result) mutable {
            send_closure(actor_id, &MessagesManager::on_get_message_viewers, dialog_id, std::move(message_viewers),
                         true, std::move(promise));
          });

      switch (dialog_id.get_type()) {
        case DialogType::Chat:
          return td_->chat_manager_->reload_chat_full(dialog_id.get_chat_id(), std::move(query_promise),
                                                      "on_get_message_viewers");
        case DialogType::Channel:
          return td_->dialog_participant_manager_->get_channel_participants(
              dialog_id.get_channel_id(), td_api::make_object<td_api::supergroupMembersFilterRecent>(), string(), 0,
              200, 200, PromiseCreator::lambda([query_promise = std::move(query_promise)](DialogParticipants) mutable {
                query_promise.set_value(Unit());
              }));
        default:
          UNREACHABLE();
          return;
      }
    }
  }
  promise.set_value(message_viewers.get_message_viewers_object(td_->user_manager_.get()));
}

void MessagesManager::translate_message_text(MessageFullId message_full_id, const string &to_language_code,
                                             Promise<td_api::object_ptr<td_api::formattedText>> &&promise) {
  auto m = get_message_force(message_full_id, "translate_message_text");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  auto text = get_message_content_text(m->content.get());
  if (text == nullptr) {
    return promise.set_value(td_api::make_object<td_api::formattedText>());
  }

  auto skip_bot_commands = need_skip_bot_commands(message_full_id.get_dialog_id(), m);
  auto max_media_timestamp = get_message_max_media_timestamp(m);
  td_->translation_manager_->translate_text(*text, skip_bot_commands, max_media_timestamp, to_language_code,
                                            std::move(promise));
}

void MessagesManager::reload_dialog_notification_settings(DialogId dialog_id, Promise<Unit> &&promise,
                                                          const char *source) {
  LOG(INFO) << "Reload notification settings for " << dialog_id << " from " << source;
  const Dialog *d = get_dialog(dialog_id);
  if (d != nullptr) {
    td_->notification_settings_manager_->send_get_dialog_notification_settings_query(dialog_id, MessageId(),
                                                                                     std::move(promise));
  } else {
    send_get_dialog_query(dialog_id, std::move(promise), 0, source);
  }
}

MessageId MessagesManager::get_dialog_pinned_message(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_pinned_message");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return MessageId();
  }

  LOG(INFO) << "Get pinned message in " << dialog_id << " with "
            << (d->is_last_pinned_message_id_inited ? "inited" : "unknown") << " pinned " << d->last_pinned_message_id;

  if (!d->is_last_pinned_message_id_inited) {
    // must call get_dialog_info_full as expected in fix_new_dialog
    td_->dialog_manager_->get_dialog_info_full(dialog_id, std::move(promise), "get_dialog_pinned_message 1");
    return MessageId();
  }

  td_->dialog_manager_->get_dialog_info_full(dialog_id, Auto(), "get_dialog_pinned_message 2");

  if (d->last_pinned_message_id.is_valid()) {
    tl_object_ptr<telegram_api::InputMessage> input_message;
    if (dialog_id.get_type() == DialogType::Channel) {
      input_message = make_tl_object<telegram_api::inputMessagePinned>();
    }
    get_message_force_from_server(d, d->last_pinned_message_id, std::move(promise), std::move(input_message));
  } else {
    promise.set_value(Unit());
  }

  return d->last_pinned_message_id;
}

void MessagesManager::get_callback_query_message(DialogId dialog_id, MessageId message_id, int64 callback_query_id,
                                                 Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_callback_query_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (!message_id.is_valid() || !message_id.is_server()) {
    return promise.set_error(Status::Error(400, "Invalid message identifier specified"));
  }

  LOG(INFO) << "Get callback query " << message_id << " in " << dialog_id << " for query " << callback_query_id;

  auto input_message = make_tl_object<telegram_api::inputMessageCallbackQuery>(message_id.get_server_message_id().get(),
                                                                               callback_query_id);
  get_message_force_from_server(d, message_id, std::move(promise), std::move(input_message));
}

bool MessagesManager::get_messages(DialogId dialog_id, const vector<MessageId> &message_ids, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return false;
  }

  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  vector<MessageFullId> missed_message_ids;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      promise.set_error(Status::Error(400, "Invalid message identifier"));
      return false;
    }

    auto *m = get_message_force(d, message_id, "get_messages");
    if (m == nullptr && message_id.is_any_server() && !is_secret) {
      missed_message_ids.emplace_back(dialog_id, message_id);
      continue;
    }
  }

  if (!missed_message_ids.empty()) {
    get_messages_from_server(std::move(missed_message_ids), std::move(promise), "get_messages");
    return false;
  }

  promise.set_value(Unit());
  return true;
}

void MessagesManager::get_message_from_server(MessageFullId message_full_id, Promise<Unit> &&promise,
                                              const char *source,
                                              tl_object_ptr<telegram_api::InputMessage> input_message) {
  get_messages_from_server({message_full_id}, std::move(promise), source, std::move(input_message));
}

void MessagesManager::get_messages_from_server(vector<MessageFullId> &&message_ids, Promise<Unit> &&promise,
                                               const char *source,
                                               tl_object_ptr<telegram_api::InputMessage> input_message) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (message_ids.empty()) {
    LOG(ERROR) << "Empty message_ids from " << source;
    return promise.set_error(Status::Error(500, "There are no messages specified to fetch"));
  }

  if (input_message != nullptr) {
    CHECK(message_ids.size() == 1);
  }

  vector<tl_object_ptr<telegram_api::InputMessage>> ordinary_message_ids;
  FlatHashMap<ChannelId, vector<tl_object_ptr<telegram_api::InputMessage>>, ChannelIdHash> channel_message_ids;
  FlatHashMap<DialogId, vector<int32>, DialogIdHash> scheduled_message_ids;
  for (auto &message_full_id : message_ids) {
    auto dialog_id = message_full_id.get_dialog_id();
    auto message_id = message_full_id.get_message_id();
    if (!message_id.is_valid() || !message_id.is_server()) {
      if (message_id.is_valid_scheduled() && message_id.is_scheduled_server() && dialog_id.is_valid()) {
        scheduled_message_ids[dialog_id].push_back(message_id.get_scheduled_server_message_id().get());
      }
      continue;
    }

    if (input_message == nullptr) {
      input_message = make_tl_object<telegram_api::inputMessageID>(message_id.get_server_message_id().get());
    }

    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        ordinary_message_ids.push_back(std::move(input_message));
        break;
      case DialogType::Channel:
        channel_message_ids[dialog_id.get_channel_id()].push_back(std::move(input_message));
        break;
      case DialogType::SecretChat:
        LOG(ERROR) << "Can't get " << message_full_id << " from server from " << source;
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }

  MultiPromiseActorSafe mpas{"GetMessagesOnServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));
  auto lock = mpas.get_promise();

  if (!ordinary_message_ids.empty()) {
    td_->create_handler<GetMessagesQuery>(mpas.get_promise())->send(std::move(ordinary_message_ids));
  }

  for (auto &it : scheduled_message_ids) {
    auto dialog_id = it.first;
    have_dialog_force(dialog_id, "get_messages_from_server");
    auto input_peer = td_->dialog_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't find info about " << dialog_id << " to get a message from it from " << source;
      mpas.get_promise().set_error(Status::Error(400, "Can't access the chat"));
      continue;
    }
    td_->create_handler<GetScheduledMessagesQuery>(mpas.get_promise())
        ->send(dialog_id, std::move(input_peer), std::move(it.second));
  }

  for (auto &it : channel_message_ids) {
    td_->chat_manager_->have_channel_force(it.first, "get_messages_from_server");
    auto input_channel = td_->chat_manager_->get_input_channel(it.first);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Can't find info about " << it.first << " to get a message from it from " << source;
      mpas.get_promise().set_error(Status::Error(400, "Can't access the chat"));
      continue;
    }
    const auto *d = get_dialog_force(DialogId(it.first));
    td_->create_handler<GetChannelMessagesQuery>(mpas.get_promise())
        ->send(it.first, std::move(input_channel), std::move(it.second),
               d == nullptr ? MessageId() : d->last_new_message_id);
  }
  lock.set_value(Unit());
}

bool MessagesManager::is_message_edited_recently(MessageFullId message_full_id, int32 seconds) {
  if (seconds < 0) {
    return false;
  }
  if (!message_full_id.get_message_id().is_valid()) {
    return false;
  }

  auto m = get_message_force(message_full_id, "is_message_edited_recently");
  if (m == nullptr) {
    return true;
  }

  return m->edit_date >= G()->unix_time() - seconds;
}

MessagesManager::ReportDialogFromActionBar MessagesManager::report_dialog_from_action_bar(DialogId dialog_id,
                                                                                          Promise<Unit> &promise) {
  ReportDialogFromActionBar result;
  Dialog *d = nullptr;
  if (dialog_id.get_type() == DialogType::SecretChat) {
    auto user_dialog_id = DialogId(td_->user_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
    d = get_dialog_force(user_dialog_id, "report_dialog_from_action_bar");
    if (d == nullptr) {
      promise.set_error(Status::Error(400, "Chat with the user not found"));
      result.is_reported_ = true;
      return result;
    }
  } else {
    d = get_dialog(dialog_id);
    CHECK(d != nullptr);
  }
  result.know_action_bar_ = d->know_action_bar;

  if (d->know_action_bar && d->action_bar != nullptr && d->action_bar->can_report_spam()) {
    result.is_reported_ = true;
    hide_dialog_action_bar(d);
    toggle_dialog_report_spam_state_on_server(dialog_id, true, 0, std::move(promise));
  }
  return result;
}

Status MessagesManager::can_get_media_timestamp_link(DialogId dialog_id, const Message *m) {
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  if (dialog_id.get_type() != DialogType::Channel) {
    if (!can_message_content_have_media_timestamp(m->content.get()) || m->forward_info == nullptr ||
        m->forward_info->is_imported()) {
      return Status::Error(400, "Message links are available only for messages in supergroups and channel chats");
    }
    auto origin_message_full_id = m->forward_info->get_origin_message_full_id();
    auto origin_message_id = origin_message_full_id.get_message_id();
    if (!origin_message_id.is_valid() || !origin_message_id.is_server()) {
      return Status::Error(400, "Message links are available only for messages in supergroups and channel chats");
    }
    return Status::OK();
  }

  if (m->message_id.is_yet_unsent()) {
    return Status::Error(400, "Message is not sent yet");
  }
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Message is scheduled");
  }
  if (!m->message_id.is_server()) {
    return Status::Error(400, "Message is local");
  }
  return Status::OK();
}

bool MessagesManager::can_report_message_reactions(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (dialog_id.get_type() != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(dialog_id) ||
      !td_->chat_manager_->is_channel_public(dialog_id.get_channel_id())) {
    return false;
  }
  if (m->message_id.is_scheduled() || !m->message_id.is_server()) {
    return false;
  }
  if (is_discussion_message(dialog_id, m)) {
    return false;
  }

  return true;
}

Result<std::pair<string, bool>> MessagesManager::get_message_link(MessageFullId message_full_id, int32 media_timestamp,
                                                                  bool for_group, bool in_message_thread) {
  auto dialog_id = message_full_id.get_dialog_id();
  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "get_message_link"));

  auto *m = get_message_force(d, message_full_id.get_message_id(), "get_message_link");
  TRY_STATUS(can_get_media_timestamp_link(dialog_id, m));

  if (media_timestamp <= 0 || !can_message_content_have_media_timestamp(m->content.get())) {
    media_timestamp = 0;
  }
  if (media_timestamp != 0) {
    for_group = false;
    auto duration = get_message_content_media_duration(m->content.get(), td_);
    if (duration != 0 && media_timestamp > duration) {
      media_timestamp = 0;
    }
  }

  auto message_id = m->message_id;
  if (dialog_id.get_type() != DialogType::Channel) {
    CHECK(m->forward_info != nullptr);

    auto origin_message_full_id = m->forward_info->get_origin_message_full_id();
    dialog_id = origin_message_full_id.get_dialog_id();
    message_id = origin_message_full_id.get_message_id();
    CHECK(dialog_id.get_type() == DialogType::Channel);
    for_group = false;
    in_message_thread = false;
    auto channel_message = get_message(origin_message_full_id);
    if (channel_message != nullptr && channel_message->media_album_id == 0) {
      for_group = true;  // default is true
    }
  } else {
    if (m->media_album_id == 0) {
      for_group = true;  // default is true
    }
  }

  bool is_forum = td_->dialog_manager_->is_forum_channel(dialog_id);
  if (!is_forum &&
      (!m->top_thread_message_id.is_valid() || !m->top_thread_message_id.is_server() ||
       is_deleted_message(d, m->top_thread_message_id) || td_->dialog_manager_->is_broadcast_channel(dialog_id))) {
    in_message_thread = false;
  }

  if (!td_->auth_manager_->is_bot()) {
    td_->create_handler<ExportChannelMessageLinkQuery>(Promise<Unit>())
        ->send(dialog_id.get_channel_id(), message_id, for_group, true);
  }

  SliceBuilder sb;
  sb << LinkManager::get_t_me_url();

  if (in_message_thread && !is_forum) {
    // try to generate a comment link
    auto *top_m = get_message_force(d, m->top_thread_message_id, "get_public_message_link");
    if (is_discussion_message(dialog_id, top_m) && is_active_message_reply_info(dialog_id, top_m->reply_info)) {
      auto linked_message_full_id = top_m->forward_info->get_last_message_full_id();
      auto linked_dialog_id = linked_message_full_id.get_dialog_id();
      auto linked_message_id = linked_message_full_id.get_message_id();
      auto linked_d = get_dialog(linked_dialog_id);
      CHECK(linked_d != nullptr);
      CHECK(linked_dialog_id.get_type() == DialogType::Channel);
      auto *linked_m = get_message_force(linked_d, linked_message_id, "get_public_message_link");
      auto channel_username = td_->chat_manager_->get_channel_first_username(linked_dialog_id.get_channel_id());
      if (linked_m != nullptr && is_active_message_reply_info(linked_dialog_id, linked_m->reply_info) &&
          linked_message_id.is_server() &&
          td_->dialog_manager_->have_input_peer(linked_dialog_id, false, AccessRights::Read) &&
          !channel_username.empty()) {
        sb << channel_username << '/' << linked_message_id.get_server_message_id().get()
           << "?comment=" << message_id.get_server_message_id().get();
        if (!for_group) {
          sb << "&single";
        }
        if (media_timestamp > 0) {
          sb << "&t=" << media_timestamp;
        }
        return std::make_pair(sb.as_cslice().str(), true);
      }
    }
  }

  auto dialog_username = td_->chat_manager_->get_channel_first_username(dialog_id.get_channel_id());
  bool is_public = !dialog_username.empty();
  if (m->content->get_type() == MessageContentType::VideoNote &&
      td_->dialog_manager_->is_broadcast_channel(dialog_id) && is_public) {
    return std::make_pair(
        PSTRING() << "https://telesco.pe/" << dialog_username << '/' << message_id.get_server_message_id().get(), true);
  }

  if (is_public) {
    sb << dialog_username;
  } else {
    sb << "c/" << dialog_id.get_channel_id().get();
  }
  if (in_message_thread && is_forum) {
    auto top_thread_message_id = m->is_topic_message ? m->top_thread_message_id : MessageId(ServerMessageId(1));
    if (top_thread_message_id != message_id) {
      sb << '/' << top_thread_message_id.get_server_message_id().get();
    }
    in_message_thread = false;
  }
  sb << '/' << message_id.get_server_message_id().get();

  char separator = '?';
  if (in_message_thread) {
    sb << separator << "thread=" << m->top_thread_message_id.get_server_message_id().get();
    separator = '&';
  }
  if (!for_group) {
    sb << separator << "single";
    separator = '&';
  }
  if (media_timestamp > 0) {
    sb << separator << "t=" << media_timestamp;
    separator = '&';
  }
  CHECK(separator == '?' || separator == '&');

  return std::make_pair(sb.as_cslice().str(), is_public);
}

string MessagesManager::get_message_embedding_code(MessageFullId message_full_id, bool for_group,
                                                   Promise<Unit> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  auto r_d = check_dialog_access(dialog_id, false, AccessRights::Read, "get_message_embedding_code");
  if (r_d.is_error()) {
    promise.set_error(r_d.move_as_error());
    return {};
  }
  auto d = r_d.move_as_ok();
  if (dialog_id.get_type() != DialogType::Channel ||
      td_->chat_manager_->get_channel_first_username(dialog_id.get_channel_id()).empty()) {
    promise.set_error(Status::Error(
        400, "Message embedding code is available only for messages in public supergroups and channel chats"));
    return {};
  }

  auto *m = get_message_force(d, message_full_id.get_message_id(), "get_message_embedding_code");
  if (m == nullptr) {
    promise.set_error(Status::Error(400, "Message not found"));
    return {};
  }
  if (m->message_id.is_yet_unsent()) {
    promise.set_error(Status::Error(400, "Message is not sent yet"));
    return {};
  }
  if (m->message_id.is_scheduled()) {
    promise.set_error(Status::Error(400, "Message is scheduled"));
    return {};
  }
  if (!m->message_id.is_server()) {
    promise.set_error(Status::Error(400, "Message is local"));
    return {};
  }

  if (m->media_album_id == 0) {
    for_group = true;  // default is true
  }

  auto &links = message_embedding_codes_[for_group][dialog_id].embedding_codes_;
  auto it = links.find(m->message_id);
  if (it == links.end()) {
    td_->create_handler<ExportChannelMessageLinkQuery>(std::move(promise))
        ->send(dialog_id.get_channel_id(), m->message_id, for_group, false);
    return {};
  }

  promise.set_value(Unit());
  return it->second;
}

void MessagesManager::on_get_public_message_link(MessageFullId message_full_id, bool for_group, string url,
                                                 string html) {
  LOG_IF(ERROR, url.empty() && html.empty()) << "Receive empty public link for " << message_full_id;
  auto dialog_id = message_full_id.get_dialog_id();
  auto message_id = message_full_id.get_message_id();
  message_embedding_codes_[for_group][dialog_id].embedding_codes_[message_id] = std::move(html);
}

void MessagesManager::get_message_link_info(Slice url, Promise<MessageLinkInfo> &&promise) {
  auto r_message_link_info = LinkManager::get_message_link_info(url);
  if (r_message_link_info.is_error()) {
    return promise.set_error(Status::Error(400, r_message_link_info.error().message()));
  }

  auto info = r_message_link_info.move_as_ok();
  auto query_promise = PromiseCreator::lambda(
      [actor_id = actor_id(this), info, promise = std::move(promise)](Result<DialogId> &&result) mutable {
        if (result.is_error()) {
          return promise.set_value(std::move(info));
        }
        send_closure(actor_id, &MessagesManager::on_get_message_link_dialog, std::move(info), result.ok(),
                     std::move(promise));
      });
  td_->dialog_manager_->resolve_dialog(info.username, info.channel_id, std::move(query_promise));
}

void MessagesManager::on_get_message_link_dialog(MessageLinkInfo &&info, DialogId dialog_id,
                                                 Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Dialog *d = get_dialog_force(dialog_id, "on_get_message_link_dialog");
  CHECK(d != nullptr);
  get_message_force_from_server(d, info.message_id,
                                PromiseCreator::lambda([actor_id = actor_id(this), info = std::move(info), dialog_id,
                                                        promise = std::move(promise)](Result<Unit> &&result) mutable {
                                  if (result.is_error()) {
                                    return promise.set_value(std::move(info));
                                  }
                                  send_closure(actor_id, &MessagesManager::on_get_message_link_message, std::move(info),
                                               dialog_id, std::move(promise));
                                }));
}

void MessagesManager::on_get_message_link_message(MessageLinkInfo &&info, DialogId dialog_id,
                                                  Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto message_id = info.message_id;
  Message *m = get_message_force({dialog_id, message_id}, "on_get_message_link_message");
  if (info.comment_message_id == MessageId() || m == nullptr ||
      !td_->dialog_manager_->is_broadcast_channel(dialog_id) || !m->reply_info.is_comment_ ||
      !is_active_message_reply_info(dialog_id, m->reply_info)) {
    return promise.set_value(std::move(info));
  }

  if (td_->chat_manager_->have_channel_force(m->reply_info.channel_id_, "on_get_message_link_message")) {
    force_create_dialog(DialogId(m->reply_info.channel_id_), "on_get_message_link_message");
    on_get_message_link_discussion_message(std::move(info), DialogId(m->reply_info.channel_id_), std::move(promise));
    return;
  }

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), info = std::move(info),
                                               promise = std::move(promise)](Result<MessageThreadInfo> result) mutable {
    if (result.is_error() || result.ok().message_ids.empty()) {
      return promise.set_value(std::move(info));
    }
    send_closure(actor_id, &MessagesManager::on_get_message_link_discussion_message, std::move(info),
                 result.ok().dialog_id, std::move(promise));
  });

  td_->create_handler<GetDiscussionMessageQuery>(std::move(query_promise))
      ->send(dialog_id, message_id, DialogId(m->reply_info.channel_id_), MessageId());
}

void MessagesManager::on_get_message_link_discussion_message(MessageLinkInfo &&info, DialogId comment_dialog_id,
                                                             Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  CHECK(comment_dialog_id.is_valid());
  info.comment_dialog_id = comment_dialog_id;

  Dialog *d = get_dialog_force(comment_dialog_id, "on_get_message_link_discussion_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(500, "Chat not found"));
  }

  auto comment_message_id = info.comment_message_id;
  get_message_force_from_server(
      d, comment_message_id,
      PromiseCreator::lambda([info = std::move(info), promise = std::move(promise)](Result<Unit> &&result) mutable {
        return promise.set_value(std::move(info));
      }));
}

td_api::object_ptr<td_api::messageLinkInfo> MessagesManager::get_message_link_info_object(
    const MessageLinkInfo &info) const {
  CHECK(info.username.empty() == info.channel_id.is_valid());

  bool is_public = !info.username.empty();
  DialogId dialog_id = info.comment_dialog_id.is_valid()
                           ? info.comment_dialog_id
                           : (is_public ? td_->dialog_manager_->get_resolved_dialog_by_username(info.username)
                                        : DialogId(info.channel_id));
  MessageId top_thread_message_id;
  MessageId message_id = info.comment_dialog_id.is_valid() ? info.comment_message_id : info.message_id;
  td_api::object_ptr<td_api::message> message;
  int32 media_timestamp = 0;
  bool for_album = false;

  const Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    dialog_id = DialogId();
    top_thread_message_id = MessageId();
  } else {
    const Message *m = get_message(d, message_id);
    if (m != nullptr) {
      message = get_message_object(dialog_id, m, "get_message_link_info_object");
      for_album = !info.is_single && m->media_album_id != 0;
      if (info.comment_dialog_id.is_valid() || info.for_comment || m->is_topic_message) {
        top_thread_message_id = m->top_thread_message_id;
      } else if (td_->dialog_manager_->is_forum_channel(dialog_id) &&
                 info.top_thread_message_id == MessageId(ServerMessageId(1))) {
        // General topic
        top_thread_message_id = info.top_thread_message_id;
      } else {
        top_thread_message_id = MessageId();
      }
      if (can_message_content_have_media_timestamp(m->content.get())) {
        auto duration = get_message_content_media_duration(m->content.get(), td_);
        if (duration == 0 || info.media_timestamp <= duration) {
          media_timestamp = info.media_timestamp;
        }
      }
      if (m->content->get_type() == MessageContentType::TopicCreate && top_thread_message_id.is_valid()) {
        message = nullptr;
        CHECK(!for_album);
        CHECK(media_timestamp == 0);
      }
    } else if (!info.comment_dialog_id.is_valid() && dialog_id.get_type() == DialogType::Channel &&
               !td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
      top_thread_message_id = info.top_thread_message_id;
    }
  }

  return td_api::make_object<td_api::messageLinkInfo>(is_public, get_chat_id_object(dialog_id, "messageLinkInfo"),
                                                      top_thread_message_id.get(), std::move(message), media_timestamp,
                                                      for_album);
}

Status MessagesManager::can_add_dialog_to_filter(DialogId dialog_id) {
  if (!dialog_id.is_valid()) {
    return Status::Error(400, "Invalid chat identifier specified");
  }
  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "can_add_dialog_to_filter"));
  if (d->order == DEFAULT_ORDER) {
    return Status::Error(400, "Chat is not in the chat list");
  }
  return Status::OK();
}

Status MessagesManager::delete_dialog_reply_markup(DialogId dialog_id, MessageId message_id) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't delete chat reply markup");
  }
  if (message_id.is_scheduled()) {
    return Status::Error(400, "Wrong message identifier specified");
  }
  if (!message_id.is_valid()) {
    return Status::Error(400, "Invalid message identifier specified");
  }

  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_reply_markup");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (d->reply_markup_message_id != message_id) {
    return Status::OK();
  }

  Message *m = get_message_force(d, message_id, "delete_dialog_reply_markup");
  CHECK(m != nullptr);
  CHECK(m->reply_markup != nullptr);

  if (m->reply_markup->type == ReplyMarkup::Type::ForceReply) {
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup->type == ReplyMarkup::Type::ShowKeyboard) {
    if (!m->reply_markup->is_one_time_keyboard) {
      return Status::Error(400, "Do not need to delete non one-time keyboard");
    }
    if (m->reply_markup->is_personal) {
      m->reply_markup->is_personal = false;
      set_dialog_reply_markup(d, message_id);

      on_message_changed(d, m, true, "delete_dialog_reply_markup");
    }
  } else {
    // non-bots can't have messages with RemoveKeyboard
    UNREACHABLE();
  }
  return Status::OK();
}

class MessagesManager::SaveDialogDraftMessageOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

Status MessagesManager::set_dialog_draft_message(DialogId dialog_id, MessageId top_thread_message_id,
                                                 tl_object_ptr<td_api::draftMessage> &&draft_message) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't change chat draft message");
  }

  Dialog *d = get_dialog_force(dialog_id, "set_dialog_draft_message");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  TRY_STATUS(can_send_message(dialog_id));

  TRY_STATUS(can_use_top_thread_message_id(d, top_thread_message_id, MessageInputReplyTo()));

  TRY_RESULT(new_draft_message,
             DraftMessage::get_draft_message(td_, dialog_id, top_thread_message_id, std::move(draft_message)));

  if (top_thread_message_id != MessageId()) {
    CHECK(top_thread_message_id.is_valid());
    CHECK(top_thread_message_id.is_server());
    auto m = get_message_force(d, top_thread_message_id, "set_dialog_draft_message");
    if (m == nullptr || m->reply_info.is_comment_ || !is_active_message_reply_info(dialog_id, m->reply_info)) {
      return Status::OK();
    }

    if (need_update_draft_message(m->thread_draft_message, new_draft_message, false)) {
      m->thread_draft_message = std::move(new_draft_message);
      on_message_changed(d, m, false, "set_dialog_draft_message");
    }

    return Status::OK();
  }

  if (update_dialog_draft_message(d, std::move(new_draft_message), false, true)) {
    if (dialog_id.get_type() != DialogType::SecretChat && !is_local_draft_message(d->draft_message)) {
      if (G()->use_message_database()) {
        SaveDialogDraftMessageOnServerLogEvent log_event;
        log_event.dialog_id_ = dialog_id;
        add_log_event(d->save_draft_message_log_event_id, get_log_event_storer(log_event),
                      LogEvent::HandlerType::SaveDialogDraftMessageOnServer, "draft");
      }

      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), d->open_count > 0 ? MIN_SAVE_DRAFT_DELAY : 0);
    }
  }
  return Status::OK();
}

void MessagesManager::save_dialog_draft_message_on_server(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  Promise<Unit> promise;
  if (d->save_draft_message_log_event_id.log_event_id != 0) {
    d->save_draft_message_log_event_id.generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->save_draft_message_log_event_id.generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_saved_dialog_draft_message, dialog_id, generation);
      }
    });
  }

  save_draft_message(td_, dialog_id, d->draft_message, std::move(promise));
}

void MessagesManager::on_saved_dialog_draft_message(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  delete_log_event(d->save_draft_message_log_event_id, generation, "draft");
}

void MessagesManager::clear_all_draft_messages(bool exclude_secret_chats, Promise<Unit> &&promise) {
  if (!exclude_secret_chats) {
    dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
      Dialog *d = dialog.get();
      if (dialog_id.get_type() == DialogType::SecretChat) {
        update_dialog_draft_message(d, nullptr, false, true);
      }
    });
  }
  ::td::clear_all_draft_messages(td_, std::move(promise));
}

int32 MessagesManager::get_pinned_dialogs_limit(DialogListId dialog_list_id) const {
  if (dialog_list_id.is_filter()) {
    return DialogFilter::get_max_filter_dialogs();
  }

  Slice key{"pinned_chat_count_max"};
  int32 default_limit = 5;
  if (!dialog_list_id.is_folder() || dialog_list_id.get_folder_id() != FolderId::main()) {
    key = Slice("pinned_archived_chat_count_max");
    default_limit = 100;
  }
  int32 limit = clamp(narrow_cast<int32>(td_->option_manager_->get_option_integer(key)), 0, 1000);
  if (limit <= 0) {
    if (td_->option_manager_->get_option_boolean("is_premium")) {
      default_limit *= 2;
    }
    return default_limit;
  }
  return limit;
}

Status MessagesManager::toggle_dialog_is_pinned(DialogListId dialog_list_id, DialogId dialog_id, bool is_pinned) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't change chat pin state");
  }

  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "toggle_dialog_is_pinned"));
  if (d->order == DEFAULT_ORDER && is_pinned) {
    return Status::Error(400, "The chat can't be pinned");
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  if (!list->are_pinned_dialogs_inited_) {
    return Status::Error(400, "Pinned chats must be loaded first");
  }

  bool was_pinned = is_dialog_pinned(dialog_list_id, dialog_id);
  if (is_pinned == was_pinned) {
    return Status::OK();
  }

  if (dialog_list_id.is_filter()) {
    return td_->dialog_filter_manager_->set_dialog_is_pinned(
        dialog_list_id.get_filter_id(), td_->dialog_manager_->get_input_dialog_id(dialog_id), is_pinned);
  }

  CHECK(dialog_list_id.is_folder());
  auto folder_id = dialog_list_id.get_folder_id();
  if (is_pinned) {
    if (d->folder_id != folder_id) {
      return Status::Error(400, "Chat not in the list");
    }

    auto pinned_dialog_ids = get_pinned_dialog_ids(dialog_list_id);
    auto pinned_dialog_count = pinned_dialog_ids.size();
    auto secret_pinned_dialog_count =
        std::count_if(pinned_dialog_ids.begin(), pinned_dialog_ids.end(),
                      [](DialogId dialog_id) { return dialog_id.get_type() == DialogType::SecretChat; });
    size_t dialog_count = dialog_id.get_type() == DialogType::SecretChat
                              ? secret_pinned_dialog_count
                              : pinned_dialog_count - secret_pinned_dialog_count;

    if (dialog_count >= static_cast<size_t>(get_pinned_dialogs_limit(dialog_list_id))) {
      return Status::Error(400, "The maximum number of pinned chats exceeded");
    }
  }

  if (set_dialog_is_pinned(dialog_list_id, d, is_pinned)) {
    toggle_dialog_is_pinned_on_server(dialog_id, is_pinned, 0);
  }
  return Status::OK();
}

class MessagesManager::ToggleDialogIsPinnedOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_pinned_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_pinned_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_pinned_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_pinned_on_server_log_event(DialogId dialog_id, bool is_pinned) {
  ToggleDialogIsPinnedOnServerLogEvent log_event{dialog_id, is_pinned};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsPinnedOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_pinned_on_server(DialogId dialog_id, bool is_pinned, uint64 log_event_id) {
  CHECK(!td_->auth_manager_->is_bot());
  if (log_event_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_is_pinned_on_server_log_event(dialog_id, is_pinned);
  }

  td_->create_handler<ToggleDialogPinQuery>(get_erase_log_event_promise(log_event_id))->send(dialog_id, is_pinned);
}

bool MessagesManager::set_folder_pinned_dialogs(FolderId folder_id, vector<DialogId> old_dialog_ids,
                                                vector<DialogId> new_dialog_ids) {
  LOG(INFO) << "Reorder pinned chats in " << folder_id << " from " << old_dialog_ids << " to " << new_dialog_ids;

  std::reverse(old_dialog_ids.begin(), old_dialog_ids.end());
  std::reverse(new_dialog_ids.begin(), new_dialog_ids.end());

  FlatHashSet<DialogId, DialogIdHash> all_old_pinned_dialog_ids;
  for (auto dialog_id : old_dialog_ids) {
    CHECK(dialog_id.is_valid());
    all_old_pinned_dialog_ids.insert(dialog_id);
  }

  bool are_pinned_dialogs_saved = false;
  auto old_it = old_dialog_ids.begin();
  for (auto dialog_id : new_dialog_ids) {
    all_old_pinned_dialog_ids.erase(dialog_id);
    while (old_it < old_dialog_ids.end()) {
      if (*old_it == dialog_id) {
        break;
      }
      ++old_it;
    }
    if (old_it < old_dialog_ids.end()) {
      // leave dialog where it is
      ++old_it;
      continue;
    }
    if (set_dialog_is_pinned(dialog_id, true)) {
      are_pinned_dialogs_saved = true;
    }
  }
  for (auto dialog_id : all_old_pinned_dialog_ids) {
    Dialog *d = get_dialog_force(dialog_id, "set_folder_pinned_dialogs 1");
    if (d == nullptr) {
      LOG(ERROR) << "Failed to find " << dialog_id << " to unpin in " << folder_id;
      force_create_dialog(dialog_id, "set_folder_pinned_dialogs 2", true);
      d = get_dialog_force(dialog_id, "set_folder_pinned_dialogs 3");
    }
    if (d != nullptr && set_dialog_is_pinned(DialogListId(folder_id), d, false)) {
      are_pinned_dialogs_saved = true;
    }
  }
  return are_pinned_dialogs_saved;
}

Status MessagesManager::set_pinned_dialogs(DialogListId dialog_list_id, vector<DialogId> dialog_ids) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't reorder pinned chats");
  }

  int32 dialog_count = 0;
  int32 secret_dialog_count = 0;
  auto dialog_count_limit = get_pinned_dialogs_limit(dialog_list_id);
  FlatHashSet<DialogId, DialogIdHash> new_pinned_dialog_ids;
  for (auto dialog_id : dialog_ids) {
    TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "set_pinned_dialogs"));
    if (d->order == DEFAULT_ORDER) {
      return Status::Error(400, "The chat can't be pinned");
    }
    if (dialog_list_id.is_folder() && d->folder_id != dialog_list_id.get_folder_id()) {
      return Status::Error(400, "Chat not in the list");
    }
    if (dialog_id.get_type() == DialogType::SecretChat) {
      secret_dialog_count++;
    } else {
      dialog_count++;
    }

    if (dialog_count > dialog_count_limit || secret_dialog_count > dialog_count_limit) {
      return Status::Error(400, "The maximum number of pinned chats exceeded");
    }

    new_pinned_dialog_ids.insert(dialog_id);
  }
  if (new_pinned_dialog_ids.size() != dialog_ids.size()) {
    return Status::Error(400, "Duplicate chats in the list of pinned chats");
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  if (!list->are_pinned_dialogs_inited_) {
    return Status::Error(400, "Pinned chats must be loaded first");
  }

  auto pinned_dialog_ids = get_pinned_dialog_ids(dialog_list_id);
  if (pinned_dialog_ids == dialog_ids) {
    return Status::OK();
  }

  auto server_old_dialog_ids = DialogId::remove_secret_chat_dialog_ids(pinned_dialog_ids);
  auto server_new_dialog_ids = DialogId::remove_secret_chat_dialog_ids(dialog_ids);

  if (dialog_list_id.is_filter()) {
    return td_->dialog_filter_manager_->set_pinned_dialog_ids(
        dialog_list_id.get_filter_id(),
        transform(dialog_ids,
                  [this](DialogId dialog_id) { return td_->dialog_manager_->get_input_dialog_id(dialog_id); }),
        server_old_dialog_ids != server_new_dialog_ids);
  }

  auto folder_id = dialog_list_id.get_folder_id();
  set_folder_pinned_dialogs(folder_id, std::move(pinned_dialog_ids), std::move(dialog_ids));

  if (server_old_dialog_ids != server_new_dialog_ids) {
    reorder_pinned_dialogs_on_server(folder_id, server_new_dialog_ids, 0);
  }
  return Status::OK();
}

class MessagesManager::ReorderPinnedDialogsOnServerLogEvent {
 public:
  FolderId folder_id_;
  vector<DialogId> dialog_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(folder_id_, storer);
    td::store(dialog_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    if (parser.version() >= static_cast<int32>(Version::AddFolders)) {
      td::parse(folder_id_, parser);
    } else {
      folder_id_ = FolderId();
    }
    td::parse(dialog_ids_, parser);
  }
};

uint64 MessagesManager::save_reorder_pinned_dialogs_on_server_log_event(FolderId folder_id,
                                                                        const vector<DialogId> &dialog_ids) {
  ReorderPinnedDialogsOnServerLogEvent log_event{folder_id, dialog_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReorderPinnedDialogsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::reorder_pinned_dialogs_on_server(FolderId folder_id, const vector<DialogId> &dialog_ids,
                                                       uint64 log_event_id) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_reorder_pinned_dialogs_on_server_log_event(folder_id, dialog_ids);
  }

  td_->create_handler<ReorderPinnedDialogsQuery>(get_erase_log_event_promise(log_event_id))
      ->send(folder_id, dialog_ids);
}

Status MessagesManager::toggle_dialog_view_as_messages(DialogId dialog_id, bool view_as_messages) {
  TRY_RESULT(d, check_dialog_access(dialog_id, false, AccessRights::Read, "toggle_dialog_view_as_messages"));
  bool is_saved_messages = dialog_id == td_->dialog_manager_->get_my_dialog_id();
  if (!is_saved_messages && !td_->dialog_manager_->is_forum_channel(dialog_id)) {
    return Status::Error(400, "The method is available only in forum channels");
  }

  if (view_as_messages == d->view_as_messages) {
    return Status::OK();
  }

  set_dialog_view_as_messages(d, view_as_messages, "toggle_dialog_view_as_messages");

  if (!is_saved_messages) {
    toggle_dialog_view_as_messages_on_server(dialog_id, view_as_messages, 0);
  }
  return Status::OK();
}

class MessagesManager::ToggleDialogViewAsMessagesOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool view_as_messages_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(view_as_messages_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(view_as_messages_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_view_as_messages_on_server_log_event(DialogId dialog_id,
                                                                                bool view_as_messages) {
  ToggleDialogViewAsMessagesOnServerLogEvent log_event{dialog_id, view_as_messages};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogViewAsMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_view_as_messages_on_server(DialogId dialog_id, bool view_as_messages,
                                                               uint64 log_event_id) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_view_as_messages_on_server_log_event(dialog_id, view_as_messages);
  }

  td_->create_handler<ToggleViewForumAsMessagesQuery>(get_erase_log_event_promise(log_event_id))
      ->send(dialog_id, view_as_messages);
}

Status MessagesManager::toggle_dialog_is_marked_as_unread(DialogId dialog_id, bool is_marked_as_unread) {
  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "toggle_dialog_is_marked_as_unread"));
  if (is_marked_as_unread == d->is_marked_as_unread) {
    return Status::OK();
  }

  set_dialog_is_marked_as_unread(d, is_marked_as_unread);

  toggle_dialog_is_marked_as_unread_on_server(dialog_id, is_marked_as_unread, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsMarkedAsUnreadOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_marked_as_unread_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_marked_as_unread_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_marked_as_unread_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_marked_as_unread_on_server_log_event(DialogId dialog_id,
                                                                                   bool is_marked_as_unread) {
  ToggleDialogIsMarkedAsUnreadOnServerLogEvent log_event{dialog_id, is_marked_as_unread};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsMarkedAsUnreadOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_marked_as_unread_on_server(DialogId dialog_id, bool is_marked_as_unread,
                                                                  uint64 log_event_id) {
  if (log_event_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_is_marked_as_unread_on_server_log_event(dialog_id, is_marked_as_unread);
  }

  td_->create_handler<ToggleDialogUnreadMarkQuery>(get_erase_log_event_promise(log_event_id))
      ->send(dialog_id, is_marked_as_unread);
}

Status MessagesManager::toggle_dialog_is_translatable(DialogId dialog_id, bool is_translatable) {
  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "toggle_dialog_is_translatable"));
  if (is_translatable == d->is_translatable) {
    return Status::OK();
  }

  set_dialog_is_translatable(d, is_translatable);

  toggle_dialog_is_translatable_on_server(dialog_id, is_translatable, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsTranslatableOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_translatable_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_translatable_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_translatable_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_translatable_on_server_log_event(DialogId dialog_id,
                                                                               bool is_translatable) {
  ToggleDialogIsTranslatableOnServerLogEvent log_event{dialog_id, is_translatable};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsTranslatableOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_translatable_on_server(DialogId dialog_id, bool is_translatable,
                                                              uint64 log_event_id) {
  if (log_event_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_is_translatable_on_server_log_event(dialog_id, is_translatable);
  }

  td_->create_handler<ToggleDialogTranslationsQuery>(get_erase_log_event_promise(log_event_id))
      ->send(dialog_id, is_translatable);
}

Status MessagesManager::set_message_sender_block_list(const td_api::object_ptr<td_api::MessageSender> &sender,
                                                      const td_api::object_ptr<td_api::BlockList> &block_list) {
  TRY_RESULT(dialog_id, get_message_sender_dialog_id(td_, sender, true, false));
  BlockListId block_list_id(block_list);
  bool is_blocked = block_list_id == BlockListId::main();
  bool is_blocked_for_stories = block_list_id == BlockListId::stories();
  switch (dialog_id.get_type()) {
    case DialogType::User:
      if (dialog_id == td_->dialog_manager_->get_my_dialog_id()) {
        return Status::Error(
            400, is_blocked || is_blocked_for_stories ? Slice("Can't block self") : Slice("Can't unblock self"));
      }
      break;
    case DialogType::Chat:
      return Status::Error(400, "Basic group chats can't be blocked");
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat: {
      auto user_id = td_->user_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
      if (!user_id.is_valid() || !td_->user_manager_->have_user_force(user_id, "set_message_sender_block_list")) {
        return Status::Error(400, "The secret chat can't be blocked");
      }
      dialog_id = DialogId(user_id);
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  Dialog *d = get_dialog_force(dialog_id, "set_message_sender_block_list");
  if (!td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Know)) {
    return Status::Error(400, "Message sender isn't accessible");
  }
  if (d != nullptr) {
    if (is_blocked == d->is_blocked && is_blocked_for_stories == d->is_blocked_for_stories) {
      return Status::OK();
    }
    set_dialog_is_blocked(d, is_blocked, is_blocked_for_stories);
  } else {
    CHECK(dialog_id.get_type() == DialogType::User);
    td_->user_manager_->on_update_user_is_blocked(dialog_id.get_user_id(), is_blocked, is_blocked_for_stories);
  }

  toggle_dialog_is_blocked_on_server(dialog_id, is_blocked, is_blocked_for_stories, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsBlockedOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_blocked_;
  bool is_blocked_for_stories_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_blocked_);
    STORE_FLAG(is_blocked_for_stories_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_blocked_);
    PARSE_FLAG(is_blocked_for_stories_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_blocked_on_server_log_event(DialogId dialog_id, bool is_blocked,
                                                                          bool is_blocked_for_stories) {
  ToggleDialogIsBlockedOnServerLogEvent log_event{dialog_id, is_blocked, is_blocked_for_stories};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsBlockedOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_blocked_on_server(DialogId dialog_id, bool is_blocked,
                                                         bool is_blocked_for_stories, uint64 log_event_id) {
  if (log_event_id == 0 && G()->use_message_database()) {
    log_event_id = save_toggle_dialog_is_blocked_on_server_log_event(dialog_id, is_blocked, is_blocked_for_stories);
  }

  td_->create_handler<ToggleDialogIsBlockedQuery>(get_erase_log_event_promise(log_event_id))
      ->send(dialog_id, is_blocked, is_blocked_for_stories);
}

Status MessagesManager::toggle_dialog_silent_send_message(DialogId dialog_id, bool silent_send_message) {
  CHECK(!td_->auth_manager_->is_bot());

  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "toggle_dialog_silent_send_message"));

  if (update_dialog_silent_send_message(d, silent_send_message)) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }

  return Status::OK();
}

class MessagesManager::UpdateDialogNotificationSettingsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

void MessagesManager::update_dialog_notification_settings_on_server(DialogId dialog_id, bool from_binlog) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (!from_binlog && td_->notification_settings_manager_->get_input_notify_peer(dialog_id, MessageId()) == nullptr) {
    // don't even create new binlog events
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!from_binlog && G()->use_message_database()) {
    UpdateDialogNotificationSettingsOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    add_log_event(d->save_notification_settings_log_event_id, get_log_event_storer(log_event),
                  LogEvent::HandlerType::UpdateDialogNotificationSettingsOnServer, "notification settings");
  }

  Promise<Unit> promise;
  if (d->save_notification_settings_log_event_id.log_event_id != 0) {
    d->save_notification_settings_log_event_id.generation++;
    promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id,
         generation = d->save_notification_settings_log_event_id.generation](Result<Unit> result) {
          if (!G()->close_flag()) {
            send_closure(actor_id, &MessagesManager::on_updated_dialog_notification_settings, dialog_id, generation);
          }
        });
  }

  send_update_dialog_notification_settings_query(d, std::move(promise));
}

void MessagesManager::send_update_dialog_notification_settings_query(const Dialog *d, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  // TODO do not send two queries simultaneously or use InvokeAfter
  td_->notification_settings_manager_->update_dialog_notify_settings(d->dialog_id, MessageId(),
                                                                     d->notification_settings, std::move(promise));
}

void MessagesManager::on_updated_dialog_notification_settings(DialogId dialog_id, uint64 generation) {
  CHECK(!td_->auth_manager_->is_bot());
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  delete_log_event(d->save_notification_settings_log_event_id, generation, "notification settings");
}

Status MessagesManager::set_dialog_client_data(DialogId dialog_id, string &&client_data) {
  Dialog *d = get_dialog_force(dialog_id, "set_dialog_client_data");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  d->client_data = std::move(client_data);
  on_dialog_updated(d->dialog_id, "set_dialog_client_data");
  return Status::OK();
}

bool MessagesManager::is_dialog_inited(const Dialog *d) {
  return d != nullptr && d->notification_settings.is_synchronized && d->is_last_read_inbox_message_id_inited &&
         d->is_last_read_outbox_message_id_inited;
}

int32 MessagesManager::get_dialog_mute_until(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_mute_until) {
    auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id);
    return td_->notification_settings_manager_->get_scope_mute_until(scope);
  } else {
    return d->notification_settings.mute_until;
  }
}

bool MessagesManager::is_dialog_muted(const Dialog *d) const {
  return get_dialog_mute_until(d) != 0;
}

bool MessagesManager::is_dialog_pinned_message_notifications_disabled(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_pinned_message_notifications) {
    auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id);
    return td_->notification_settings_manager_->get_scope_disable_pinned_message_notifications(scope);
  }

  return d->notification_settings.disable_pinned_message_notifications;
}

bool MessagesManager::is_dialog_mention_notifications_disabled(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_mention_notifications) {
    auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(d->dialog_id);
    return td_->notification_settings_manager_->get_scope_disable_mention_notifications(scope);
  }

  return d->notification_settings.disable_mention_notifications;
}

void MessagesManager::create_dialog(DialogId dialog_id, bool force, Promise<Unit> &&promise) {
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    if (!td_->dialog_manager_->have_dialog_info_force(dialog_id, "create dialog")) {
      return promise.set_error(Status::Error(400, "Chat info not found"));
    }
    if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
      return promise.set_error(Status::Error(400, "Can't access the chat"));
    }
  }

  if (force || td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    force_create_dialog(dialog_id, "create dialog");
  } else {
    const Dialog *d = get_dialog_force(dialog_id, "create_dialog");
    if (!is_dialog_inited(d)) {
      return send_get_dialog_query(dialog_id, std::move(promise), 0, "create_dialog");
    }
  }

  promise.set_value(Unit());
}

bool MessagesManager::is_dialog_opened(DialogId dialog_id) const {
  const Dialog *d = get_dialog(dialog_id);
  return d != nullptr && d->open_count > 0;
}

Status MessagesManager::open_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "open_dialog");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  open_dialog(d);
  return Status::OK();
}

Status MessagesManager::close_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "close_dialog");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  close_dialog(d);
  return Status::OK();
}

Status MessagesManager::view_messages(DialogId dialog_id, vector<MessageId> message_ids, MessageSource source,
                                      bool force_read) {
  CHECK(!td_->auth_manager_->is_bot());

  TRY_RESULT(d, check_dialog_access(dialog_id, true, AccessRights::Read, "view_messages"));

  if (source == MessageSource::Auto) {
    if (d->open_count > 0) {
      source = MessageSource::DialogHistory;
    } else {
      source = MessageSource::Other;
    }
  }
  bool is_dialog_history = source == MessageSource::DialogHistory || source == MessageSource::MessageThreadHistory ||
                           source == MessageSource::ForumTopicHistory;
  bool need_read = force_read || is_dialog_history;
  bool need_update_view_count = is_dialog_history || source == MessageSource::HistoryPreview ||
                                source == MessageSource::DialogList || source == MessageSource::Other;
  bool need_mark_download_as_viewed = is_dialog_history || source == MessageSource::HistoryPreview ||
                                      source == MessageSource::Search || source == MessageSource::Other;
  bool need_invalidate_authentication_code =
      dialog_id == DialogId(UserManager::get_service_notifications_user_id()) && source == MessageSource::Screenshot;
  auto dialog_type = dialog_id.get_type();
  bool need_screenshot_notification = source == MessageSource::Screenshot &&
                                      (dialog_type == DialogType::User || dialog_type == DialogType::SecretChat) &&
                                      can_send_message(dialog_id).is_ok();

  if (source == MessageSource::DialogList && dialog_type == DialogType::User) {
    td_->story_manager_->on_view_dialog_active_stories({dialog_id});
  }

  // keep only valid message identifiers
  size_t pos = 0;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid()) {
      if (message_id.is_valid_scheduled()) {
        // nothing to do for scheduled messages
        continue;
      }
      if (message_id.is_valid_sponsored()) {
        if (is_dialog_history) {
          td_->sponsored_message_manager_->view_sponsored_message(dialog_id, message_id);
          continue;
        } else if (source == MessageSource::HistoryPreview || source == MessageSource::Other) {
          continue;
        } else {
          return Status::Error(400, "Can't view the message from the specified source");
        }
      }
      return Status::Error(400, "Invalid message identifier");
    }
    message_ids[pos++] = message_id;
  }
  message_ids.resize(pos);
  if (message_ids.empty()) {
    // nothing to do
    return Status::OK();
  }

  if (source == MessageSource::DialogEventLog) {
    // nothing more to do
    return Status::OK();
  }

  for (auto message_id : message_ids) {
    auto *m = get_message_force(d, message_id, "view_messages 20");
    if (m != nullptr) {
      auto file_ids = get_message_file_ids(m);
      for (auto file_id : file_ids) {
        td_->file_manager_->check_local_location_async(file_id, true);
      }
    }
  }

  // get information about thread of the messages
  MessageId top_thread_message_id;
  if (source == MessageSource::MessageThreadHistory) {
    if (dialog_type != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
      return Status::Error(400, "There are no message threads in the chat");
    }

    std::sort(message_ids.begin(), message_ids.end());

    vector<MessageId> top_thread_message_ids;
    vector<int64> media_album_ids;
    for (auto message_id : message_ids) {
      auto *m = get_message_force(d, message_id, "view_messages 1");
      if (m != nullptr) {
        top_thread_message_ids.push_back(m->top_thread_message_id);
        media_album_ids.push_back(m->media_album_id);
      }
    }

    if (!top_thread_message_ids.empty()) {
      // first we expect a root album, then messages from the thread
      // the root album can have all messages from their own threads,
      // or all messages except the first one without thread for automatic forwards
      size_t thread_start = 0;
      if (media_album_ids[0] != 0) {
        thread_start++;
        while (thread_start < media_album_ids.size() && media_album_ids[thread_start] == media_album_ids[0]) {
          thread_start++;
        }
      }
      if (thread_start < media_album_ids.size()) {
        top_thread_message_id = top_thread_message_ids[thread_start];
        for (size_t i = thread_start; i < top_thread_message_ids.size(); i++) {
          if (!top_thread_message_ids[i].is_valid()) {
            return Status::Error(400, "Messages must be from a message thread");
          }
          if (top_thread_message_ids[i] != top_thread_message_id) {
            return Status::Error(400, "All messages must be from the same message thread");
          }
        }
        // do not check album messages to belong to the thread top_thread_message_id
      } else {
        // all messages are from the same album; thread of the first message is always the best guess
        top_thread_message_id = top_thread_message_ids[0];
      }
    }
  }

  MessageId max_thread_message_id;
  if (top_thread_message_id.is_valid()) {
    const auto *top_m = get_message_force(d, top_thread_message_id, "view_messages 2");
    if (top_m != nullptr && !top_m->reply_info.is_comment_) {
      max_thread_message_id = top_m->reply_info.max_message_id_;
    }
  }

  // get forum topic identifier for the messages
  MessageId forum_topic_id;
  if (source == MessageSource::ForumTopicHistory) {
    if (!td_->dialog_manager_->is_forum_channel(dialog_id)) {
      return Status::Error(400, "Chat has no topics");
    }

    for (auto message_id : message_ids) {
      auto *m = get_message_force(d, message_id, "view_messages 3");
      if (m != nullptr) {
        auto message_forum_topic_id = m->is_topic_message ? m->top_thread_message_id : MessageId(ServerMessageId(1));
        if (forum_topic_id.is_valid()) {
          if (message_forum_topic_id != forum_topic_id) {
            return Status::Error(400, "All messages must be from the same forum topic");
          }
        } else {
          forum_topic_id = message_forum_topic_id;
        }
      }
    }
  }

  MessageId max_message_id;  // max server or local viewed message_id
  vector<MessageId> read_content_message_ids;
  vector<MessageId> read_reaction_message_ids;
  vector<MessageId> new_viewed_message_ids;
  vector<MessageId> viewed_reaction_message_ids;
  vector<string> authentication_codes;
  vector<MessageId> screenshotted_secret_message_ids;
  for (auto message_id : message_ids) {
    auto *m = get_message_force(d, message_id, "view_messages 4");
    if (m != nullptr) {
      if (m->message_id.is_server() && m->view_count > 0 && need_update_view_count) {
        pending_message_views_[dialog_id].message_ids_.insert(m->message_id);
      }

      if (!m->message_id.is_yet_unsent() && m->message_id > max_message_id) {
        max_message_id = m->message_id;
      }

      auto message_content_type = m->content->get_type();
      if (message_content_type == MessageContentType::LiveLocation) {
        on_message_live_location_viewed(d, m);
      }

      if (need_read && message_content_type != MessageContentType::VoiceNote &&
          message_content_type != MessageContentType::VideoNote &&
          update_message_contains_unread_mention(d, m, false, "view_messages 5")) {
        CHECK(m->message_id.is_server());
        read_content_message_ids.push_back(m->message_id);
        on_message_changed(d, m, true, "view_messages 6");
      }

      if (need_read && remove_message_unread_reactions(d, m, "view_messages 7")) {
        CHECK(m->message_id.is_server());
        read_reaction_message_ids.push_back(m->message_id);
        on_message_changed(d, m, true, "view_messages 8");
      }

      auto file_source_id = message_full_id_to_file_source_id_.get({dialog_id, m->message_id});
      if (file_source_id.is_valid() && need_mark_download_as_viewed) {
        LOG(INFO) << "Have " << file_source_id << " for " << m->message_id;
        CHECK(file_source_id.is_valid());
        for (auto file_id : get_message_file_ids(m)) {
          auto file_view = td_->file_manager_->get_file_view(file_id);
          CHECK(!file_view.empty());
          send_closure(td_->download_manager_actor_, &DownloadManager::update_file_viewed, file_view.get_main_file_id(),
                       file_source_id);
        }
      }

      auto story_full_id = get_message_content_story_full_id(td_, m->content.get());
      if (story_full_id.is_valid()) {
        td_->story_manager_->view_story_message(story_full_id);
      }

      if (m->message_id.is_server() && d->open_count > 0) {
        auto &info = dialog_viewed_messages_[dialog_id];
        if (info == nullptr) {
          info = make_unique<ViewedMessagesInfo>();
        }
        auto &view_id = info->message_id_to_view_id[message_id];
        if (view_id == 0) {
          new_viewed_message_ids.push_back(message_id);
          if (need_poll_message_reactions(d, m)) {
            viewed_reaction_message_ids.push_back(message_id);
          }
        } else {
          info->recently_viewed_messages.erase(view_id);
        }
        view_id = ++info->current_view_id;
        info->recently_viewed_messages[view_id] = message_id;
      }

      if (need_invalidate_authentication_code) {
        extract_authentication_codes(dialog_id, m, authentication_codes);
      }
      if (need_screenshot_notification && !m->is_outgoing) {
        if ((dialog_type == DialogType::User && m->is_content_secret) || dialog_type == DialogType::SecretChat) {
          screenshotted_secret_message_ids.push_back(m->message_id);
        }
      }
    } else if (!message_id.is_yet_unsent() && message_id > max_message_id) {
      if ((d->notification_info != nullptr && message_id <= d->notification_info->max_push_notification_message_id_) ||
          message_id <= d->last_new_message_id || message_id <= max_thread_message_id) {
        max_message_id = message_id;
      }
    }
  }
  if (pending_message_views_.count(dialog_id) != 0) {
    pending_message_views_timeout_.add_timeout_in(dialog_id.get(), MAX_MESSAGE_VIEW_DELAY);
    if (is_dialog_history) {
      pending_message_views_[dialog_id].increment_view_counter_ = true;
    }
  }
  if (!read_content_message_ids.empty()) {
    read_message_contents_on_server(dialog_id, std::move(read_content_message_ids), 0, Auto());
  }
  if (!read_reaction_message_ids.empty()) {
    for (auto message_id : read_reaction_message_ids) {
      pending_read_reactions_[{dialog_id, message_id}]++;
    }
    auto promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id, read_reaction_message_ids](Result<Unit> &&result) mutable {
          send_closure(actor_id, &MessagesManager::on_read_message_reactions, dialog_id,
                       std::move(read_reaction_message_ids), std::move(result));
        });
    read_message_contents_on_server(dialog_id, std::move(read_reaction_message_ids), 0, std::move(promise));
  }
  if (!new_viewed_message_ids.empty()) {
    LOG(INFO) << "Have new viewed " << new_viewed_message_ids;
    auto &info = dialog_viewed_messages_[dialog_id];
    CHECK(info != nullptr);
    CHECK(info->message_id_to_view_id.size() == info->recently_viewed_messages.size());
    constexpr size_t MAX_RECENTLY_VIEWED_MESSAGES = 25;
    while (info->recently_viewed_messages.size() > MAX_RECENTLY_VIEWED_MESSAGES) {
      auto it = info->recently_viewed_messages.begin();
      info->message_id_to_view_id.erase(it->second);
      info->recently_viewed_messages.erase(it);
    }
    if (!viewed_reaction_message_ids.empty()) {
      queue_message_reactions_reload(dialog_id, viewed_reaction_message_ids);
    }
  }
  if (td_->is_online() && dialog_viewed_messages_.count(dialog_id) != 0) {
    update_viewed_messages_timeout_.add_timeout_in(dialog_id.get(), UPDATE_VIEWED_MESSAGES_PERIOD);
  }
  if (!authentication_codes.empty()) {
    td_->account_manager_->invalidate_authentication_codes(std::move(authentication_codes));
  }
  if (!screenshotted_secret_message_ids.empty()) {
    send_screenshot_taken_notification_message(d);
  }

  if (!need_read) {
    return Status::OK();
  }

  if (source == MessageSource::MessageThreadHistory) {
    if (!top_thread_message_id.is_valid() || !max_message_id.is_valid()) {
      return Status::OK();
    }
    MessageId prev_last_read_inbox_message_id;
    max_thread_message_id = MessageId();
    Message *top_m = get_message_force(d, top_thread_message_id, "view_messages 9");
    if (top_m != nullptr && is_active_message_reply_info(dialog_id, top_m->reply_info)) {
      prev_last_read_inbox_message_id = top_m->reply_info.last_read_inbox_message_id_;
      if (top_m->reply_info.update_max_message_ids(MessageId(), max_message_id, MessageId())) {
        on_message_reply_info_changed(dialog_id, top_m);
        on_message_changed(d, top_m, true, "view_messages 10");
      }
      max_thread_message_id = top_m->reply_info.max_message_id_;

      if (is_discussion_message(dialog_id, top_m)) {
        auto linked_message_full_id = top_m->forward_info->get_last_message_full_id();
        auto linked_dialog_id = linked_message_full_id.get_dialog_id();
        auto linked_d = get_dialog(linked_dialog_id);
        CHECK(linked_d != nullptr);
        CHECK(linked_dialog_id.get_type() == DialogType::Channel);
        auto *linked_m = get_message_force(linked_d, linked_message_full_id.get_message_id(), "view_messages 11");
        if (linked_m != nullptr && is_active_message_reply_info(linked_dialog_id, linked_m->reply_info)) {
          if (linked_m->reply_info.last_read_inbox_message_id_ < prev_last_read_inbox_message_id) {
            prev_last_read_inbox_message_id = linked_m->reply_info.last_read_inbox_message_id_;
          }
          if (linked_m->reply_info.update_max_message_ids(MessageId(), max_message_id, MessageId())) {
            on_message_reply_info_changed(linked_dialog_id, linked_m);
            on_message_changed(linked_d, linked_m, true, "view_messages 12");
          }
          if (linked_m->reply_info.max_message_id_ > max_thread_message_id) {
            max_thread_message_id = linked_m->reply_info.max_message_id_;
          }
        }
      }
    }

    if (max_message_id.get_prev_server_message_id().get() >
        prev_last_read_inbox_message_id.get_prev_server_message_id().get()) {
      read_message_thread_history_on_server(d, top_thread_message_id, max_message_id.get_prev_server_message_id(),
                                            max_thread_message_id.get_prev_server_message_id());
    }

    return Status::OK();
  }
  if (source == MessageSource::ForumTopicHistory) {
    if (!forum_topic_id.is_valid() || !max_message_id.is_valid()) {
      return Status::OK();
    }

    td_->forum_topic_manager_->read_forum_topic_messages(dialog_id, forum_topic_id, max_message_id);
    return Status::OK();
  }

  CHECK(source == MessageSource::DialogHistory || force_read);
  read_dialog_inbox(d, max_message_id);
  if (d->is_marked_as_unread) {
    set_dialog_is_marked_as_unread(d, false);
  }
  return Status::OK();
}

void MessagesManager::read_dialog_inbox(Dialog *d, MessageId max_message_id) {
  if (max_message_id == MessageId()) {
    return;
  }
  CHECK(d != nullptr);
  CHECK(max_message_id.is_valid());
  CHECK(max_message_id.is_server() || max_message_id.is_local());
  if (max_message_id <= d->last_read_inbox_message_id) {
    return;
  }

  const MessageId last_read_message_id = max_message_id;
  const MessageId prev_last_read_inbox_message_id = d->last_read_inbox_message_id;
  MessageId read_history_on_server_message_id;
  if (d->dialog_id.get_type() != DialogType::SecretChat) {
    if (last_read_message_id.get_prev_server_message_id().get() >
        prev_last_read_inbox_message_id.get_prev_server_message_id().get()) {
      read_history_on_server_message_id = last_read_message_id.get_prev_server_message_id();
    }
  } else {
    if (last_read_message_id > prev_last_read_inbox_message_id) {
      read_history_on_server_message_id = last_read_message_id;
    }
  }

  if (read_history_on_server_message_id.is_valid()) {
    // add dummy timeout to not try to repair unread_count in read_history_inbox before server request succeeds
    // the timeout will be overwritten in the read_history_on_server call
    pending_read_history_timeout_.add_timeout_in(d->dialog_id.get(), 0);
  }
  read_history_inbox(d, last_read_message_id, -1, "read_dialog_inbox");
  if (read_history_on_server_message_id.is_valid()) {
    // call read_history_on_server after read_history_inbox to not have delay before request if all messages are read
    read_history_on_server(d, read_history_on_server_message_id);
  }
}

void MessagesManager::finish_get_message_views(DialogId dialog_id, const vector<MessageId> &message_ids) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  for (auto message_id : message_ids) {
    auto *m = get_message(d, message_id);
    if (m != nullptr) {
      m->has_get_message_views_query = false;
      m->need_view_counter_increment = false;
    }
  }
}

void MessagesManager::finish_get_message_extended_media(DialogId dialog_id, const vector<MessageId> &message_ids) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  for (auto message_id : message_ids) {
    auto *m = get_message(d, message_id);
    if (m != nullptr) {
      m->has_get_extended_media_query = false;
    }
  }
}

Status MessagesManager::open_message_content(MessageFullId message_full_id) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "open_message_content");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  auto *m = get_message_force(d, message_full_id.get_message_id(), "open_message_content");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  if (m->message_id.is_scheduled() || m->message_id.is_yet_unsent() || m->is_outgoing) {
    return Status::OK();
  }

  if (read_message_content(d, m, true, 0, "open_message_content") &&
      (m->message_id.is_server() || dialog_id.get_type() == DialogType::SecretChat)) {
    read_message_contents_on_server(dialog_id, {m->message_id}, 0, Auto());
  }

  if (m->content->get_type() == MessageContentType::LiveLocation) {
    on_message_live_location_viewed(d, m);
  }

  auto file_ids = get_message_file_ids(m);
  for (auto file_id : file_ids) {
    td_->file_manager_->check_local_location_async(file_id, true);
  }

  return Status::OK();
}

class MessagesManager::ReadMessageContentsOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_read_message_contents_on_server_log_event(DialogId dialog_id,
                                                                       const vector<MessageId> &message_ids) {
  ReadMessageContentsOnServerLogEvent log_event{dialog_id, message_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadMessageContentsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::read_message_contents_on_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                      uint64 log_event_id, Promise<Unit> &&promise,
                                                      bool skip_log_event) {
  CHECK(!message_ids.empty());

  LOG(INFO) << "Read contents of " << format::as_array(message_ids) << " in " << dialog_id << " on server";

  if (log_event_id == 0 && G()->use_message_database() && !skip_log_event) {
    log_event_id = save_read_message_contents_on_server_log_event(dialog_id, message_ids);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadMessagesContentsQuery>(std::move(promise))->send(std::move(message_ids));
      break;
    case DialogType::Channel:
      td_->create_handler<ReadChannelMessagesContentsQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), std::move(message_ids));
      break;
    case DialogType::SecretChat: {
      CHECK(message_ids.size() == 1);
      auto m = get_message_force({dialog_id, message_ids[0]}, "read_message_contents_on_server");
      if (m != nullptr) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_open_message,
                     dialog_id.get_secret_chat_id(), m->random_id, std::move(promise));
      } else {
        promise.set_error(Status::Error(400, "Message not found"));
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::click_animated_emoji_message(MessageFullId message_full_id,
                                                   Promise<td_api::object_ptr<td_api::sticker>> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "click_animated_emoji_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  auto message_id = get_persistent_message_id(d, message_full_id.get_message_id());
  auto *m = get_message_force(d, message_id, "click_animated_emoji_message");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  if (m->message_id.is_scheduled() || dialog_id.get_type() != DialogType::User || !m->message_id.is_server()) {
    return promise.set_value(nullptr);
  }

  get_message_content_animated_emoji_click_sticker(m->content.get(), message_full_id, td_, std::move(promise));
}

void MessagesManager::open_dialog(Dialog *d) {
  CHECK(!td_->auth_manager_->is_bot());
  DialogId dialog_id = d->dialog_id;
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    return;
  }
  recently_opened_dialogs_.add_dialog(dialog_id);
  if (d->open_count == std::numeric_limits<uint32>::max()) {
    return;
  }
  d->open_count++;
  if (d->open_count != 1) {
    return;
  }
  d->was_opened = true;

  auto min_message_id = MessageId(ServerMessageId(1));
  if (d->last_message_id == MessageId() && d->last_read_outbox_message_id < min_message_id) {
    auto it = d->ordered_messages.get_const_iterator(MessageId::max());
    if (*it != nullptr && (*it)->get_message_id() < min_message_id) {
      read_history_inbox(d, (*it)->get_message_id(), -1, "open_dialog");
    }
  }

  if (d->has_unload_timeout) {
    LOG(INFO) << "Cancel unload timeout for " << dialog_id;
    pending_unload_dialog_timeout_.cancel_timeout(dialog_id.get());
    d->has_unload_timeout = false;
  }

  if (d->notification_info != nullptr && d->notification_info->new_secret_chat_notification_id_.is_valid()) {
    remove_new_secret_chat_notification(d, true);
  }

  get_dialog_pinned_message(dialog_id, Auto());

  if (d->active_group_call_id.is_valid()) {
    td_->group_call_manager_->reload_group_call(d->active_group_call_id, Auto());
  }
  if (d->need_drop_default_send_message_as_dialog_id) {
    CHECK(d->default_send_message_as_dialog_id.is_valid());
    d->need_drop_default_send_message_as_dialog_id = false;
    d->default_send_message_as_dialog_id = DialogId();
    LOG(INFO) << "Set message sender in " << d->dialog_id << " to " << d->default_send_message_as_dialog_id;
    on_dialog_updated(dialog_id, "open_dialog");
    send_update_chat_message_sender(d);
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      td_->story_manager_->on_view_dialog_active_stories({dialog_id});
      break;
    case DialogType::Chat:
      td_->chat_manager_->repair_chat_participants(dialog_id.get_chat_id());
      reget_dialog_action_bar(dialog_id, "open_dialog", false);
      break;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      if (!td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
        auto participant_count = td_->chat_manager_->get_channel_participant_count(channel_id);
        auto has_hidden_participants = td_->chat_manager_->get_channel_effective_has_hidden_participants(
            dialog_id.get_channel_id(), "open_dialog");
        if (participant_count < 195 && !has_hidden_participants) {  // include unknown participant_count
          td_->dialog_participant_manager_->get_channel_participants(
              channel_id, td_api::make_object<td_api::supergroupMembersFilterRecent>(), string(), 0, 200, 200, Auto());
        }
      } else {
        td_->story_manager_->on_view_dialog_active_stories({dialog_id});
      }
      get_channel_difference(dialog_id, d->pts, 0, MessageId(), true, "open_dialog");
      reget_dialog_action_bar(dialog_id, "open_dialog", false);

      if (td_->chat_manager_->get_channel_has_linked_channel(channel_id)) {
        auto linked_channel_id = td_->chat_manager_->get_channel_linked_channel_id(channel_id, "open_dialog");
        if (!linked_channel_id.is_valid()) {
          // load linked_channel_id
          send_closure_later(G()->chat_manager(), &ChatManager::load_channel_full, channel_id, false, Promise<Unit>(),
                             "open_dialog");
        } else {
          td_->dialog_manager_->get_dialog_info_full(DialogId(linked_channel_id), Auto(), "open_dialog");
        }
      }
      break;
    }
    case DialogType::SecretChat: {
      // to repair dialog action bar
      auto user_id = td_->user_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
      if (user_id.is_valid()) {
        td_->user_manager_->reload_user_full(user_id, Promise<Unit>(), "open_dialog");
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    td_->dialog_participant_manager_->on_dialog_opened(dialog_id);

    if (d->has_scheduled_database_messages && !d->is_has_scheduled_database_messages_checked) {
      CHECK(G()->use_message_database());

      LOG(INFO) << "Send check has_scheduled_database_messages request";
      d->is_has_scheduled_database_messages_checked = true;
      G()->td_db()->get_message_db_async()->get_scheduled_messages(
          dialog_id, 1,
          PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](vector<MessageDbDialogMessage> messages) {
            if (messages.empty()) {
              send_closure(actor_id, &MessagesManager::set_dialog_has_scheduled_database_messages, dialog_id, false);
            }
          }));
    }
  }
}

void MessagesManager::close_dialog(Dialog *d) {
  if (d->open_count == 0) {
    return;
  }
  d->open_count--;
  if (d->open_count > 0) {
    return;
  }

  auto dialog_id = d->dialog_id;
  if (td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Write)) {
    if (pending_draft_message_timeout_.has_timeout(dialog_id.get())) {
      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_draft_message_timeout_.cancel_timeout(dialog_id.get());
  }

  if (td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    if (pending_message_views_timeout_.has_timeout(dialog_id.get())) {
      pending_message_views_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
    if (pending_read_history_timeout_.has_timeout(dialog_id.get())) {
      pending_read_history_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_message_views_timeout_.cancel_timeout(dialog_id.get());
    pending_message_views_.erase(dialog_id);
    pending_read_history_timeout_.cancel_timeout(dialog_id.get());
  }

  if (is_message_unload_enabled()) {
    CHECK(!d->has_unload_timeout);
    pending_unload_dialog_timeout_.set_timeout_in(dialog_id.get(), get_next_unload_dialog_delay(d));
    d->has_unload_timeout = true;

    if (d->need_unload_on_close) {
      unload_dialog(dialog_id, 0);
      d->need_unload_on_close = false;
    }
  }

  dialog_viewed_messages_.erase(dialog_id);
  update_viewed_messages_timeout_.cancel_timeout(dialog_id.get());

  auto live_locations_it = pending_viewed_live_locations_.find(dialog_id);
  if (live_locations_it != pending_viewed_live_locations_.end()) {
    for (auto &it : live_locations_it->second) {
      auto live_location_task_id = it.second;
      auto erased_count = viewed_live_location_tasks_.erase(live_location_task_id);
      CHECK(erased_count > 0);
    }
    pending_viewed_live_locations_.erase(live_locations_it);
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      break;
    case DialogType::Channel:
      channel_get_difference_timeout_.cancel_timeout(dialog_id.get());
      break;
    case DialogType::SecretChat:
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    if (postponed_chat_read_inbox_updates_.erase(dialog_id) > 0) {
      send_update_chat_read_inbox(d, false, "close_dialog");
    }

    td_->dialog_participant_manager_->on_dialog_closed(dialog_id);
  }
}

int64 MessagesManager::get_chat_id_object(DialogId dialog_id, const char *source) const {
  const Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    if (dialog_id != DialogId()) {
      LOG(ERROR) << "Can't find " << dialog_id << ", needed from " << source;
    }
  } else if (!d->is_update_new_chat_sent && !d->is_update_new_chat_being_sent) {
    LOG(ERROR) << "Didn't send updateNewChat for " << dialog_id << ", needed from " << source;
  }
  return dialog_id.get();
}

td_api::object_ptr<td_api::ChatActionBar> MessagesManager::get_chat_action_bar_object(const Dialog *d) const {
  CHECK(d != nullptr);
  auto dialog_type = d->dialog_id.get_type();
  if (dialog_type == DialogType::SecretChat) {
    auto user_id = td_->user_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return nullptr;
    }
    const Dialog *user_d = get_dialog(DialogId(user_id));
    if (user_d == nullptr || user_d->action_bar == nullptr) {
      return nullptr;
    }
    return user_d->action_bar->get_chat_action_bar_object(DialogType::User, d->folder_id != FolderId::archive());
  }

  if (d->action_bar == nullptr) {
    return nullptr;
  }
  return d->action_bar->get_chat_action_bar_object(dialog_type, false);
}

td_api::object_ptr<td_api::businessBotManageBar> MessagesManager::get_business_bot_manage_bar_object(
    const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->business_bot_manage_bar == nullptr) {
    return nullptr;
  }
  return d->business_bot_manage_bar->get_business_bot_manage_bar_object(td_);
}

td_api::object_ptr<td_api::chatBackground> MessagesManager::get_chat_background_object(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    auto user_id = td_->user_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return nullptr;
    }
    d = get_dialog(DialogId(user_id));
    if (d == nullptr) {
      return nullptr;
    }
  }
  return d->background_info.get_chat_background_object(td_);
}

string MessagesManager::get_dialog_theme_name(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    auto user_id = td_->user_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return string();
    }
    d = get_dialog(DialogId(user_id));
    if (d == nullptr) {
      return string();
    }
  }
  return d->theme_name;
}

td_api::object_ptr<td_api::chatJoinRequestsInfo> MessagesManager::get_chat_join_requests_info_object(
    const Dialog *d) const {
  if (d->pending_join_request_count == 0) {
    return nullptr;
  }
  return td_api::make_object<td_api::chatJoinRequestsInfo>(
      d->pending_join_request_count,
      td_->user_manager_->get_user_ids_object(d->pending_join_request_user_ids, "get_chat_join_requests_info_object"));
}

td_api::object_ptr<td_api::videoChat> MessagesManager::get_video_chat_object(const Dialog *d) const {
  auto active_group_call_id = td_->group_call_manager_->get_group_call_id(d->active_group_call_id, d->dialog_id);
  auto default_participant_alias =
      d->default_join_group_call_as_dialog_id.is_valid()
          ? get_message_sender_object_const(td_, d->default_join_group_call_as_dialog_id, "get_video_chat_object")
          : nullptr;
  return make_tl_object<td_api::videoChat>(active_group_call_id.get(),
                                           active_group_call_id.is_valid() ? !d->is_group_call_empty : false,
                                           std::move(default_participant_alias));
}

td_api::object_ptr<td_api::MessageSender> MessagesManager::get_default_message_sender_object(const Dialog *d) const {
  auto as_dialog_id = d->default_send_message_as_dialog_id;
  return as_dialog_id.is_valid()
             ? get_message_sender_object_const(td_, as_dialog_id, "get_default_message_sender_object")
             : nullptr;
}

td_api::object_ptr<td_api::chat> MessagesManager::get_chat_object(const Dialog *d, const char *source) const {
  CHECK(d != nullptr);

  bool is_premium = td_->option_manager_->get_option_boolean("is_premium");
  auto chat_source = is_dialog_sponsored(d) ? sponsored_dialog_source_.get_chat_source_object() : nullptr;
  auto can_delete = can_delete_dialog(d);
  // TODO hide/show draft message when need_hide_dialog_draft_message changes
  auto draft_message = !need_hide_dialog_draft_message(d) ? get_draft_message_object(td_, d->draft_message) : nullptr;
  auto available_reactions = get_dialog_active_reactions(d).get_chat_available_reactions_object();
  auto is_translatable = d->is_translatable && is_premium;
  auto block_list_id = BlockListId(d->is_blocked, d->is_blocked_for_stories);
  auto chat_lists = transform(get_dialog_list_ids(d),
                              [](DialogListId dialog_list_id) { return dialog_list_id.get_chat_list_object(); });
  return make_tl_object<td_api::chat>(
      d->dialog_id.get(), td_->dialog_manager_->get_chat_type_object(d->dialog_id, source),
      td_->dialog_manager_->get_dialog_title(d->dialog_id),
      get_chat_photo_info_object(td_->file_manager_.get(), td_->dialog_manager_->get_dialog_photo(d->dialog_id)),
      td_->dialog_manager_->get_dialog_accent_color_id_object(d->dialog_id),
      td_->dialog_manager_->get_dialog_background_custom_emoji_id(d->dialog_id).get(),
      td_->dialog_manager_->get_dialog_profile_accent_color_id_object(d->dialog_id),
      td_->dialog_manager_->get_dialog_profile_background_custom_emoji_id(d->dialog_id).get(),
      td_->dialog_manager_->get_dialog_default_permissions(d->dialog_id).get_chat_permissions_object(),
      get_message_object(d->dialog_id, get_message(d, d->last_message_id), source), get_chat_positions_object(d),
      std::move(chat_lists), get_default_message_sender_object(d), block_list_id.get_block_list_object(),
      td_->dialog_manager_->get_dialog_has_protected_content(d->dialog_id), is_translatable, d->is_marked_as_unread,
      get_dialog_view_as_topics(d), get_dialog_has_scheduled_messages(d), can_delete.for_self_,
      can_delete.for_all_users_, td_->dialog_manager_->can_report_dialog(d->dialog_id),
      d->notification_settings.silent_send_message, d->server_unread_count + d->local_unread_count,
      d->last_read_inbox_message_id.get(), d->last_read_outbox_message_id.get(), d->unread_mention_count,
      d->unread_reaction_count, get_chat_notification_settings_object(&d->notification_settings),
      std::move(available_reactions), d->message_ttl.get_message_auto_delete_time_object(),
      td_->dialog_manager_->get_dialog_emoji_status_object(d->dialog_id), get_chat_background_object(d),
      get_dialog_theme_name(d), get_chat_action_bar_object(d), get_business_bot_manage_bar_object(d),
      get_video_chat_object(d), get_chat_join_requests_info_object(d), d->reply_markup_message_id.get(),
      std::move(draft_message), d->client_data);
}

td_api::object_ptr<td_api::chat> MessagesManager::get_chat_object(DialogId dialog_id, const char *source) {
  const Dialog *d = get_dialog(dialog_id);
  if (postponed_chat_read_inbox_updates_.erase(dialog_id) > 0) {
    send_update_chat_read_inbox(d, true, source);
  }
  return get_chat_object(d, source);
}

td_api::object_ptr<td_api::draftMessage> MessagesManager::get_my_dialog_draft_message_object() const {
  const Dialog *d = get_dialog(td_->dialog_manager_->get_my_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }
  return get_draft_message_object(td_, d->draft_message);
}

std::pair<bool, int32> MessagesManager::get_dialog_mute_until(DialogId dialog_id, const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  if (d == nullptr || !d->notification_settings.is_synchronized) {
    auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id);
    return {false, td_->notification_settings_manager_->get_scope_mute_until(scope)};
  }

  return {d->notification_settings.is_use_default_fixed, get_dialog_mute_until(d)};
}

int64 MessagesManager::get_dialog_notification_ringtone_id(DialogId dialog_id, const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  if (d == nullptr || !d->notification_settings.is_synchronized ||
      is_notification_sound_default(d->notification_settings.sound)) {
    auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id);
    return get_notification_sound_ringtone_id(td_->notification_settings_manager_->get_scope_notification_sound(scope));
  }

  return get_notification_sound_ringtone_id(d->notification_settings.sound);
}

StoryNotificationSettings MessagesManager::get_story_notification_settings(DialogId dialog_id) {
  bool need_dialog_settings = false;
  bool need_top_dialogs = false;
  bool are_muted = false;
  bool hide_sender = false;
  int64 ringtone_id = 0;

  const Dialog *d = get_dialog_force(dialog_id, "get_story_notification_settings");
  if (d == nullptr || !d->notification_settings.is_synchronized) {
    need_dialog_settings = true;
  }
  auto scope = td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id);
  if (need_dialog_settings || d->notification_settings.use_default_mute_stories) {
    bool use_default;
    std::tie(use_default, are_muted) = td_->notification_settings_manager_->get_scope_mute_stories(scope);
    if (use_default) {
      auto is_top_dialog = td_->top_dialog_manager_->is_top_dialog(TopDialogCategory::Correspondent, 5, dialog_id);
      if (is_top_dialog == -1) {
        need_top_dialogs = true;
      } else {
        are_muted = is_top_dialog != 0;
      }
    }
  } else {
    are_muted = d->notification_settings.mute_stories;
  }
  if (need_dialog_settings || d->notification_settings.use_default_hide_story_sender) {
    hide_sender = td_->notification_settings_manager_->get_scope_hide_story_sender(scope);
  } else {
    hide_sender = d->notification_settings.hide_story_sender;
  }
  if (need_dialog_settings || is_notification_sound_default(d->notification_settings.story_sound)) {
    ringtone_id = get_notification_sound_ringtone_id(
        td_->notification_settings_manager_->get_scope_story_notification_sound(scope));
  } else {
    ringtone_id = get_notification_sound_ringtone_id(d->notification_settings.story_sound);
  }

  return {need_dialog_settings, need_top_dialogs, are_muted, hide_sender, ringtone_id};
}

vector<DialogId> MessagesManager::get_dialog_notification_settings_exceptions(NotificationSettingsScope scope,
                                                                              bool filter_scope, bool compare_sound,
                                                                              bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  bool have_all_dialogs = true;
  for (const auto &list : dialog_folders_) {
    if (list.second.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
      have_all_dialogs = false;
    }
  }

  if (have_all_dialogs || force) {
    vector<DialogDate> ordered_dialogs;
    auto my_dialog_id = td_->dialog_manager_->get_my_dialog_id();
    for (const auto &list : dialog_folders_) {
      for (const auto &dialog_date : list.second.ordered_dialogs_) {
        auto dialog_id = dialog_date.get_dialog_id();
        if (filter_scope && td_->dialog_manager_->get_dialog_notification_setting_scope(dialog_id) != scope) {
          continue;
        }
        if (dialog_id == my_dialog_id) {
          continue;
        }

        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        LOG_CHECK(d->folder_id == list.first)
            << list.first << ' ' << dialog_id << ' ' << d->folder_id << ' ' << d->order;
        if (d->order == DEFAULT_ORDER) {
          break;
        }
        if (are_default_dialog_notification_settings(d->notification_settings, compare_sound)) {
          continue;
        }
        if (is_dialog_message_notification_disabled(dialog_id, std::numeric_limits<int32>::max())) {
          continue;
        }
        ordered_dialogs.push_back(DialogDate(get_dialog_base_order(d), dialog_id));
      }
    }
    std::sort(ordered_dialogs.begin(), ordered_dialogs.end());

    vector<DialogId> result;
    for (auto &dialog_date : ordered_dialogs) {
      CHECK(result.empty() || result.back() != dialog_date.get_dialog_id());
      result.push_back(dialog_date.get_dialog_id());
    }
    promise.set_value(Unit());
    return result;
  }

  for (const auto &folder : dialog_folders_) {
    load_folder_dialog_list(folder.first, MAX_GET_DIALOGS, true);
  }

  td_->notification_settings_manager_->get_notify_settings_exceptions(scope, filter_scope, compare_sound,
                                                                      std::move(promise));
  return {};
}

DialogNotificationSettings *MessagesManager::get_dialog_notification_settings(DialogId dialog_id, bool force) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_notification_settings");
  if (d == nullptr) {
    return nullptr;
  }
  if (!force && !td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    return nullptr;
  }
  return &d->notification_settings;
}

Status MessagesManager::set_dialog_notification_settings(
    DialogId dialog_id, tl_object_ptr<td_api::chatNotificationSettings> &&notification_settings) {
  CHECK(!td_->auth_manager_->is_bot());
  auto current_settings = get_dialog_notification_settings(dialog_id, false);
  if (current_settings == nullptr) {
    return Status::Error(400, "Wrong chat identifier specified");
  }
  if (dialog_id == td_->dialog_manager_->get_my_dialog_id()) {
    return Status::Error(400, "Notification settings of the Saved Messages chat can't be changed");
  }

  TRY_RESULT(new_settings, ::td::get_dialog_notification_settings(std::move(notification_settings), current_settings));
  if (update_dialog_notification_settings(dialog_id, current_settings, std::move(new_settings))) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }
  return Status::OK();
}

void MessagesManager::reset_all_notification_settings() {
  CHECK(!td_->auth_manager_->is_bot());

  dialogs_.foreach([&](const DialogId &dialog_id, unique_ptr<Dialog> &dialog) {
    DialogNotificationSettings new_dialog_settings;
    new_dialog_settings.is_synchronized = true;
    Dialog *d = dialog.get();
    update_dialog_notification_settings(dialog_id, &d->notification_settings, std::move(new_dialog_settings));
  });

  td_->notification_settings_manager_->reset_scope_notification_settings();

  reset_all_notification_settings_on_server(0);
}

class MessagesManager::ResetAllNotificationSettingsOnServerLogEvent {
 public:
  template <class StorerT>
  void store(StorerT &storer) const {
  }

  template <class ParserT>
  void parse(ParserT &parser) {
  }
};

uint64 MessagesManager::save_reset_all_notification_settings_on_server_log_event() {
  ResetAllNotificationSettingsOnServerLogEvent log_event;
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ResetAllNotificationSettingsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::reset_all_notification_settings_on_server(uint64 log_event_id) {
  CHECK(!td_->auth_manager_->is_bot());
  if (log_event_id == 0) {
    log_event_id = save_reset_all_notification_settings_on_server_log_event();
  }

  LOG(INFO) << "Reset all notification settings";
  td_->notification_settings_manager_->reset_notify_settings(get_erase_log_event_promise(log_event_id));
}

tl_object_ptr<td_api::messages> MessagesManager::get_dialog_history(DialogId dialog_id, MessageId from_message_id,
                                                                    int32 offset, int32 limit, int left_tries,
                                                                    bool only_local, Promise<Unit> &&promise) {
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return nullptr;
  }
  if (limit > MAX_GET_HISTORY) {
    limit = MAX_GET_HISTORY;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return nullptr;
  }
  if (offset <= -MAX_GET_HISTORY) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than -100"));
    return nullptr;
  }
  if (offset < -limit) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than or equal to -limit"));
    return nullptr;
  }

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_history");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return nullptr;
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return nullptr;
  }

  if (from_message_id == MessageId() || from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }
  if (!from_message_id.is_valid()) {
    promise.set_error(Status::Error(400, "Invalid value of parameter from_message_id specified"));
    return nullptr;
  }

  LOG(INFO) << "Get " << (only_local ? "local " : "") << "history in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ", " << left_tries
            << " tries left, is_empty = " << d->is_empty << ", have_full_history = " << d->have_full_history
            << ", have_full_history_source = " << d->have_full_history_source;

  auto message_ids = d->ordered_messages.get_history(d->last_message_id, from_message_id, offset, limit,
                                                     left_tries == 0 && !only_local);
  if (!message_ids.empty()) {
    // maybe need some messages
    CHECK(offset == 0);
    preload_newer_messages(d, message_ids[0]);
    preload_older_messages(d, message_ids.back());
  } else if (limit > 0 && left_tries != 0 && !(d->is_empty && d->have_full_history && left_tries < 3)) {
    // there can be more messages in the database or on the server, need to load them
    send_closure_later(actor_id(this), &MessagesManager::load_messages, dialog_id, from_message_id, offset, limit,
                       left_tries, only_local, std::move(promise));
    return nullptr;
  }

  LOG(INFO) << "Return " << message_ids << " in result to getChatHistory";
  promise.set_value(Unit());  // can return some messages
  return get_messages_object(-1, dialog_id, message_ids, true,
                             "get_dialog_history");  // TODO return real total_count of messages in the dialog
}

class MessagesManager::ReadHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

class MessagesManager::ReadHistoryInSecretChatLogEvent {
 public:
  DialogId dialog_id_;
  int32 max_date_ = 0;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_date_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_date_, parser);
  }
};

class MessagesManager::ReadMessageThreadHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId top_thread_message_id_;
  MessageId max_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(top_thread_message_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(top_thread_message_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

void MessagesManager::read_history_on_server(Dialog *d, MessageId max_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  CHECK(!max_message_id.is_scheduled());

  auto dialog_id = d->dialog_id;
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  bool need_delay = d->open_count > 0 && !is_secret &&
                    (d->server_unread_count > 0 || (!need_unread_counter(d->order) && d->last_message_id.is_valid() &&
                                                    max_message_id < d->last_message_id));
  LOG(INFO) << "Read history in " << dialog_id << " on server up to " << max_message_id << " with"
            << (need_delay ? "" : "out") << " delay";

  if (is_secret) {
    auto *m = get_message_force(d, max_message_id, "read_history_on_server");
    if (m == nullptr) {
      LOG(ERROR) << "Failed to read history in " << dialog_id << " up to " << max_message_id;
      return;
    }

    ReadHistoryInSecretChatLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.max_date_ = m->date;
    add_log_event(read_history_log_event_ids_[dialog_id][0], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadHistoryInSecretChat, "read history");

    d->last_read_inbox_message_date = m->date;
  } else if (G()->use_message_database()) {
    ReadHistoryOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.max_message_id_ = max_message_id;
    add_log_event(read_history_log_event_ids_[dialog_id][0], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadHistoryOnServer, "read history");
  }

  updated_read_history_message_ids_[dialog_id].insert(MessageId());

  pending_read_history_timeout_.set_timeout_in(dialog_id.get(), need_delay ? MIN_READ_HISTORY_DELAY : 0);
}

void MessagesManager::read_message_thread_history_on_server(Dialog *d, MessageId top_thread_message_id,
                                                            MessageId max_message_id, MessageId last_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  CHECK(top_thread_message_id.is_valid());
  CHECK(top_thread_message_id.is_server());
  CHECK(max_message_id.is_server());

  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Read history in thread of " << top_thread_message_id << " in " << dialog_id << " on server up to "
            << max_message_id;

  if (G()->use_message_database()) {
    ReadMessageThreadHistoryOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.top_thread_message_id_ = top_thread_message_id;
    log_event.max_message_id_ = max_message_id;
    add_log_event(read_history_log_event_ids_[dialog_id][top_thread_message_id.get()], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadMessageThreadHistoryOnServer, "read history");
  }

  updated_read_history_message_ids_[dialog_id].insert(top_thread_message_id);

  bool need_delay = d->open_count > 0 && last_message_id.is_valid() && max_message_id < last_message_id;
  pending_read_history_timeout_.set_timeout_in(dialog_id.get(), need_delay ? MIN_READ_HISTORY_DELAY : 0);
}

void MessagesManager::do_read_history_on_server(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = updated_read_history_message_ids_.find(dialog_id);
  if (it != updated_read_history_message_ids_.end()) {
    auto top_thread_message_ids = std::move(it->second);
    updated_read_history_message_ids_.erase(it);
    for (auto top_thread_message_id : top_thread_message_ids) {
      if (!top_thread_message_id.is_valid()) {
        read_history_on_server_impl(d, MessageId());
      } else {
        read_message_thread_history_on_server_impl(d, top_thread_message_id, MessageId());
      }
    }
  }
}

void MessagesManager::read_history_on_server_impl(Dialog *d, MessageId max_message_id) {
  CHECK(d != nullptr);
  CHECK(max_message_id == MessageId() || max_message_id.is_valid());
  auto dialog_id = d->dialog_id;

  {
    auto message_id = d->last_read_inbox_message_id;
    if (dialog_id.get_type() != DialogType::SecretChat) {
      message_id = message_id.get_prev_server_message_id();
    }
    if (message_id > max_message_id) {
      max_message_id = message_id;
    }
  }

  Promise<Unit> promise;
  auto &log_event_id = read_history_log_event_ids_[dialog_id][0];
  if (log_event_id.log_event_id != 0) {
    log_event_id.generation++;
    promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id, generation = log_event_id.generation](Result<Unit> result) {
          if (!G()->close_flag()) {
            send_closure(actor_id, &MessagesManager::on_read_history_finished, dialog_id, MessageId(), generation);
          }
        });
  }
  if (d->need_repair_server_unread_count && need_unread_counter(d->order)) {
    repair_server_unread_count(dialog_id, d->server_unread_count, "read_history_on_server_impl");
  }

  if (!max_message_id.is_valid() || !td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Send read history request in " << dialog_id << " up to " << max_message_id;
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadHistoryQuery>(std::move(promise))->send(dialog_id, max_message_id);
      break;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      td_->create_handler<ReadChannelHistoryQuery>(std::move(promise))->send(channel_id, max_message_id);
      break;
    }
    case DialogType::SecretChat: {
      auto secret_chat_id = dialog_id.get_secret_chat_id();
      auto date = d->last_read_inbox_message_date;
      auto *m = get_message_force(d, max_message_id, "read_history_on_server_impl");
      if (m != nullptr && m->date > date) {
        date = m->date;
      }
      if (date == 0) {
        LOG(ERROR) << "Don't know last read inbox message date in " << dialog_id;
        return promise.set_value(Unit());
      }
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_read_history, secret_chat_id, date,
                   std::move(promise));
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::read_message_thread_history_on_server_impl(Dialog *d, MessageId top_thread_message_id,
                                                                 MessageId max_message_id) {
  CHECK(d != nullptr);
  CHECK(max_message_id == MessageId() || max_message_id.is_valid());
  auto dialog_id = d->dialog_id;
  CHECK(dialog_id.get_type() == DialogType::Channel);

  const Message *m = get_message_force(d, top_thread_message_id, "read_message_thread_history_on_server_impl");
  if (m != nullptr) {
    auto message_id = m->reply_info.last_read_inbox_message_id_.get_prev_server_message_id();
    if (message_id > max_message_id) {
      max_message_id = message_id;
    }
  }

  Promise<Unit> promise;
  auto &log_event_id = read_history_log_event_ids_[dialog_id][top_thread_message_id.get()];
  if (log_event_id.log_event_id != 0) {
    log_event_id.generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, top_thread_message_id,
                                      generation = log_event_id.generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_read_history_finished, dialog_id, top_thread_message_id,
                     generation);
      }
    });
  }

  if (!max_message_id.is_valid() || !td_->dialog_manager_->have_input_peer(dialog_id, false, AccessRights::Read)) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Send read history request in thread of " << top_thread_message_id << " in " << dialog_id << " up to "
            << max_message_id;
  td_->create_handler<ReadDiscussionQuery>(std::move(promise))->send(dialog_id, top_thread_message_id, max_message_id);
}

void MessagesManager::on_read_history_finished(DialogId dialog_id, MessageId top_thread_message_id, uint64 generation) {
  auto dialog_it = read_history_log_event_ids_.find(dialog_id);
  if (dialog_it == read_history_log_event_ids_.end()) {
    return;
  }
  auto it = dialog_it->second.find(top_thread_message_id.get());
  if (it == dialog_it->second.end()) {
    return;
  }
  delete_log_event(it->second, generation, "read history");
  if (it->second.log_event_id == 0) {
    dialog_it->second.erase(it);
    if (dialog_it->second.empty()) {
      read_history_log_event_ids_.erase(dialog_it);
    }
  }
}

template <class T, class It>
vector<MessageId> MessagesManager::get_message_history_slice(const T &begin, It it, const T &end,
                                                             MessageId from_message_id, int32 offset, int32 limit) {
  int32 left_offset = -offset;
  int32 left_limit = limit + offset;
  while (left_offset > 0 && it != end) {
    ++it;
    left_offset--;
    left_limit++;
  }

  vector<MessageId> message_ids;
  while (left_limit > 0 && it != begin) {
    --it;
    left_limit--;
    message_ids.push_back(*it);
  }
  return message_ids;
}

std::pair<DialogId, vector<MessageId>> MessagesManager::get_message_thread_history(
    DialogId dialog_id, MessageId message_id, MessageId from_message_id, int32 offset, int32 limit, int64 &random_id,
    Promise<Unit> &&promise) {
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }
  if (limit > MAX_GET_HISTORY) {
    limit = MAX_GET_HISTORY;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return {};
  }
  if (offset <= -MAX_GET_HISTORY) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than -100"));
    return {};
  }
  if (offset < -limit) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than or equal to -limit"));
    return {};
  }
  bool is_limit_increased = false;
  if (limit == -offset) {
    limit++;
    is_limit_increased = true;
  }
  CHECK(0 < limit && limit <= MAX_GET_HISTORY);
  CHECK(-limit < offset && offset <= 0);

  Dialog *d = get_dialog_force(dialog_id, "get_message_thread_history");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    promise.set_error(Status::Error(400, "Can't get message thread history in the chat"));
    return {};
  }

  if (from_message_id == MessageId() || from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }
  if (!from_message_id.is_valid()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return {};
  }

  MessageFullId top_thread_message_full_id;
  if (message_id == MessageId(ServerMessageId(1)) && td_->dialog_manager_->is_forum_channel(dialog_id)) {
    top_thread_message_full_id = MessageFullId{dialog_id, message_id};
  } else {
    message_id = get_persistent_message_id(d, message_id);
    Message *m = get_message_force(d, message_id, "get_message_thread_history 1");
    if (m == nullptr) {
      promise.set_error(Status::Error(400, "Message not found"));
      return {};
    }

    auto r_top_thread_message_full_id = get_top_thread_message_full_id(dialog_id, m, true);
    if (r_top_thread_message_full_id.is_error()) {
      promise.set_error(r_top_thread_message_full_id.move_as_error());
      return {};
    }
    top_thread_message_full_id = r_top_thread_message_full_id.move_as_ok();
    if ((m->reply_info.is_empty() || !m->reply_info.is_comment_) &&
        top_thread_message_full_id.get_message_id() != m->message_id) {
      CHECK(dialog_id == top_thread_message_full_id.get_dialog_id());
      // get information about the thread from the top message
      message_id = top_thread_message_full_id.get_message_id();
      CHECK(message_id.is_valid());
    }

    if (!top_thread_message_full_id.get_message_id().is_valid()) {
      CHECK(m->reply_info.is_comment_);
      get_message_thread(
          dialog_id, message_id,
          PromiseCreator::lambda([promise = std::move(promise)](Result<MessageThreadInfo> &&result) mutable {
            if (result.is_error()) {
              promise.set_error(result.move_as_error());
            } else {
              promise.set_value(Unit());
            }
          }));
      return {};
    }
  }

  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    CHECK(it != found_dialog_messages_.end());
    auto result = std::move(it->second.message_ids);
    found_dialog_messages_.erase(it);

    auto dialog_id_it = found_dialog_messages_dialog_id_.find(random_id);
    if (dialog_id_it != found_dialog_messages_dialog_id_.end()) {
      dialog_id = dialog_id_it->second;
      found_dialog_messages_dialog_id_.erase(dialog_id_it);

      d = get_dialog(dialog_id);
      CHECK(d != nullptr);
    }
    if (dialog_id != top_thread_message_full_id.get_dialog_id()) {
      promise.set_error(Status::Error(500, "Receive messages in an unexpected chat"));
      return {};
    }

    auto yet_unsent_it = yet_unsent_thread_message_ids_.find(top_thread_message_full_id);
    if (yet_unsent_it != yet_unsent_thread_message_ids_.end()) {
      const std::set<MessageId> &message_ids = yet_unsent_it->second;
      auto merge_message_ids = get_message_history_slice(message_ids.begin(), message_ids.lower_bound(from_message_id),
                                                         message_ids.end(), from_message_id, offset, limit);
      vector<MessageId> new_result(result.size() + merge_message_ids.size());
      std::merge(result.begin(), result.end(), merge_message_ids.begin(), merge_message_ids.end(), new_result.begin(),
                 std::greater<>());
      result = std::move(new_result);
    }

    Message *top_m = get_message_force(d, top_thread_message_full_id.get_message_id(), "get_message_thread_history 2");
    if (top_m != nullptr && !top_m->local_thread_message_ids.empty()) {
      vector<MessageId> &message_ids = top_m->local_thread_message_ids;
      vector<MessageId> merge_message_ids;
      while (true) {
        merge_message_ids = get_message_history_slice(
            message_ids.begin(), std::lower_bound(message_ids.begin(), message_ids.end(), from_message_id),
            message_ids.end(), from_message_id, offset, limit);
        bool found_deleted = false;
        for (auto local_message_id : merge_message_ids) {
          Message *local_m = get_message_force(d, local_message_id, "get_message_thread_history 3");
          if (local_m == nullptr) {
            auto local_it = std::lower_bound(message_ids.begin(), message_ids.end(), local_message_id);
            CHECK(local_it != message_ids.end() && *local_it == local_message_id);
            message_ids.erase(local_it);
            found_deleted = true;
          }
        }
        if (!found_deleted) {
          break;
        }
        on_message_changed(d, top_m, false, "get_message_thread_history");
      }
      vector<MessageId> new_result(result.size() + merge_message_ids.size());
      std::merge(result.begin(), result.end(), merge_message_ids.begin(), merge_message_ids.end(), new_result.begin(),
                 std::greater<>());
      result = std::move(new_result);
    }

    if (is_limit_increased) {
      limit--;
    }

    std::reverse(result.begin(), result.end());
    result = get_message_history_slice(result.begin(), std::lower_bound(result.begin(), result.end(), from_message_id),
                                       result.end(), from_message_id, offset, limit);

    LOG(INFO) << "Return " << result.size() << " messages in result to getMessageThreadHistory";

    promise.set_value(Unit());
    return {dialog_id, std::move(result)};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.count(random_id) > 0);
  found_dialog_messages_[random_id];  // reserve place for result

  td_->create_handler<SearchMessagesQuery>(std::move(promise))
      ->send(dialog_id, SavedMessagesTopicId(), string(), DialogId(), from_message_id.get_next_server_message_id(),
             offset, limit, MessageSearchFilter::Empty, message_id, ReactionType(), random_id);
  return {};
}

td_api::object_ptr<td_api::messageCalendar> MessagesManager::get_dialog_message_calendar(
    DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageId from_message_id,
    MessageSearchFilter filter, int64 &random_id, bool use_db, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_message_calendars_.find(random_id);
    if (it != found_dialog_message_calendars_.end()) {
      auto result = std::move(it->second);
      found_dialog_message_calendars_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Get message calendar in " << dialog_id << " with " << saved_messages_topic_id << " filtered by "
            << filter << " from " << from_message_id;

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return {};
  }
  from_message_id = from_message_id.get_next_server_message_id();

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_calendar");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  {
    auto status = saved_messages_topic_id.is_valid_in(td_, dialog_id);
    if (status.is_error()) {
      promise.set_error(std::move(status));
      return {};
    }
  }

  CHECK(filter != MessageSearchFilter::Call && filter != MessageSearchFilter::MissedCall);
  if (filter == MessageSearchFilter::Empty || filter == MessageSearchFilter::Mention ||
      filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::UnreadReaction) {
    if (filter != MessageSearchFilter::Empty && saved_messages_topic_id.is_valid()) {
      return td_api::make_object<td_api::messageCalendar>();
    }
    promise.set_error(Status::Error(400, "The filter is not supported"));
    return {};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_message_calendars_.count(random_id) > 0);
  found_dialog_message_calendars_[random_id];  // reserve place for result

  // Trying to use database
  if (use_db && G()->use_message_database() && !saved_messages_topic_id.is_valid()) {
    MessageId first_db_message_id = get_first_database_message_id_by_index(d, filter);
    int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    LOG(INFO) << "Get message calendar in " << dialog_id << " from " << fixed_from_message_id << ", have up to "
              << first_db_message_id << ", message_count = " << message_count;
    if (first_db_message_id < fixed_from_message_id && message_count != -1) {
      LOG(INFO) << "Get message calendar from database in " << dialog_id << " from " << fixed_from_message_id;
      auto new_promise =
          PromiseCreator::lambda([random_id, dialog_id, fixed_from_message_id, first_db_message_id, filter,
                                  promise = std::move(promise)](Result<MessageDbCalendar> r_calendar) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_get_message_calendar_from_database, random_id,
                         dialog_id, fixed_from_message_id, first_db_message_id, filter, std::move(r_calendar),
                         std::move(promise));
          });
      MessageDbDialogCalendarQuery db_query;
      db_query.dialog_id = dialog_id;
      db_query.filter = filter;
      db_query.from_message_id = fixed_from_message_id;
      db_query.tz_offset = static_cast<int32>(td_->option_manager_->get_option_integer("utc_time_offset"));
      G()->td_db()->get_message_db_async()->get_dialog_message_calendar(db_query, std::move(new_promise));
      return {};
    }
  }
  if (filter == MessageSearchFilter::FailedToSend) {
    found_dialog_message_calendars_.erase(random_id);
    promise.set_value(Unit());
    return td_api::make_object<td_api::messageCalendar>();
  }

  switch (dialog_id.get_type()) {
    case DialogType::None:
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchResultCalendarQuery>(std::move(promise))
          ->send(dialog_id, saved_messages_topic_id, from_message_id, filter, random_id);
      break;
    case DialogType::SecretChat:
      promise.set_value(Unit());
      break;
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Message search is not supported"));
  }
  return {};
}

void MessagesManager::on_get_message_calendar_from_database(int64 random_id, DialogId dialog_id,
                                                            MessageId from_message_id, MessageId first_db_message_id,
                                                            MessageSearchFilter filter,
                                                            Result<MessageDbCalendar> r_calendar,
                                                            Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (r_calendar.is_error()) {
    LOG(ERROR) << "Failed to get message calendar from the database: " << r_calendar.error();
    if (first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat &&
        filter != MessageSearchFilter::FailedToSend) {
      found_dialog_message_calendars_.erase(random_id);
    }
    return promise.set_value(Unit());
  }
  CHECK(!from_message_id.is_scheduled());
  CHECK(!first_db_message_id.is_scheduled());

  auto calendar = r_calendar.move_as_ok();

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());
  CHECK(it->second == nullptr);

  vector<std::pair<MessageId, int32>> periods;
  periods.reserve(calendar.messages.size());
  for (size_t i = 0; i < calendar.messages.size(); i++) {
    auto m = on_get_message_from_database(d, calendar.messages[i], false, "on_get_message_calendar_from_database");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      CHECK(!m->message_id.is_scheduled());
      periods.emplace_back(m->message_id, calendar.total_counts[i]);
    }
  }

  if (periods.empty() && first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(INFO) << "No messages found in database";
    found_dialog_message_calendars_.erase(it);
  } else {
    auto total_count = d->message_count_by_index[message_search_filter_index(filter)];
    vector<td_api::object_ptr<td_api::messageCalendarDay>> days;
    for (auto &period : periods) {
      const auto *m = get_message(d, period.first);
      CHECK(m != nullptr);
      days.push_back(td_api::make_object<td_api::messageCalendarDay>(
          period.second, get_message_object(dialog_id, m, "on_get_message_calendar_from_database")));
    }
    it->second = td_api::make_object<td_api::messageCalendar>(total_count, std::move(days));
  }
  promise.set_value(Unit());
}

MessagesManager::FoundDialogMessages MessagesManager::search_dialog_messages(
    DialogId dialog_id, const string &query, const td_api::object_ptr<td_api::MessageSender> &sender,
    MessageId from_message_id, int32 offset, int32 limit, MessageSearchFilter filter, MessageId top_thread_message_id,
    SavedMessagesTopicId saved_messages_topic_id, const ReactionType &tag, int64 &random_id, bool use_db,
    Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    if (it != found_dialog_messages_.end()) {
      CHECK(found_dialog_messages_dialog_id_.count(random_id) == 0);
      auto result = std::move(it->second);
      found_dialog_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search messages with query \"" << query << "\" in " << dialog_id << " sent by "
            << oneline(to_string(sender)) << " in thread of " << top_thread_message_id << " and in "
            << saved_messages_topic_id << " filtered by " << filter << " from " << from_message_id << " with offset "
            << offset << " and limit " << limit;

  FoundDialogMessages result;
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }
  if (limit <= -offset) {
    promise.set_error(Status::Error(400, "Parameter limit must be greater than -offset"));
    return result;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return result;
  }

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return result;
  }
  from_message_id = from_message_id.get_next_server_message_id();

  const Dialog *d = get_dialog_force(dialog_id, "search_dialog_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return result;
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return result;
  }

  auto r_sender_dialog_id = get_message_sender_dialog_id(td_, sender, true, true);
  if (r_sender_dialog_id.is_error()) {
    promise.set_error(r_sender_dialog_id.move_as_error());
    return result;
  }
  auto sender_dialog_id = r_sender_dialog_id.move_as_ok();
  if (sender_dialog_id != DialogId() &&
      !td_->dialog_manager_->have_input_peer(sender_dialog_id, false, AccessRights::Know)) {
    promise.set_error(Status::Error(400, "Invalid message sender specified"));
    return result;
  }
  if (sender_dialog_id == dialog_id && td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
    sender_dialog_id = DialogId();
  }

  if (filter == MessageSearchFilter::FailedToSend && sender_dialog_id.is_valid()) {
    if (sender_dialog_id != td_->dialog_manager_->get_my_dialog_id()) {
      promise.set_value(Unit());
      return result;
    }
    sender_dialog_id = DialogId();
  }

  if (top_thread_message_id != MessageId()) {
    if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
      promise.set_error(Status::Error(400, "Invalid message thread specified"));
      return result;
    }
    if (dialog_id.get_type() != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
      promise.set_error(Status::Error(400, "Can't filter by message thread in the chat"));
      return result;
    }
  }
  {
    auto status = saved_messages_topic_id.is_valid_in(td_, dialog_id);
    if (status.is_error()) {
      promise.set_error(std::move(status));
      return result;
    }
  }

  if (sender_dialog_id.get_type() == DialogType::SecretChat) {
    promise.set_value(Unit());
    return result;
  }

  if (filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::UnreadReaction) {
    if (!query.empty()) {
      promise.set_error(Status::Error(400, "Non-empty query is unsupported with the specified filter"));
      return result;
    }
    if (sender_dialog_id.is_valid()) {
      promise.set_error(Status::Error(400, "Filtering by sender is unsupported with the specified filter"));
      return result;
    }
    if (saved_messages_topic_id.is_valid()) {
      promise.set_value(Unit());
      return result;
    }
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.count(random_id) > 0);
  found_dialog_messages_[random_id];  // reserve place for result

  // Trying to use database
  if (use_db && query.empty() && G()->use_message_database() && filter != MessageSearchFilter::Empty &&
      !sender_dialog_id.is_valid() && top_thread_message_id == MessageId() &&
      saved_messages_topic_id == SavedMessagesTopicId() && tag.is_empty()) {
    MessageId first_db_message_id = get_first_database_message_id_by_index(d, filter);
    int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    LOG(INFO) << "Search messages in " << dialog_id << " from " << fixed_from_message_id << ", have up to "
              << first_db_message_id << ", message_count = " << message_count;
    if ((first_db_message_id < fixed_from_message_id || (first_db_message_id == fixed_from_message_id && offset < 0)) &&
        message_count != -1) {
      LOG(INFO) << "Search messages in database in " << dialog_id << " from " << fixed_from_message_id
                << " and with limit " << limit;
      auto new_promise = PromiseCreator::lambda(
          [random_id, dialog_id, fixed_from_message_id, first_db_message_id, filter, offset, limit,
           promise = std::move(promise)](Result<vector<MessageDbDialogMessage>> r_messages) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_search_dialog_message_db_result, random_id,
                         dialog_id, fixed_from_message_id, first_db_message_id, filter, offset, limit,
                         std::move(r_messages), std::move(promise));
          });
      MessageDbMessagesQuery db_query;
      db_query.dialog_id = dialog_id;
      db_query.filter = filter;
      db_query.from_message_id = fixed_from_message_id;
      db_query.offset = offset;
      db_query.limit = limit;
      G()->td_db()->get_message_db_async()->get_messages(db_query, std::move(new_promise));
      return result;
    }
  }
  if (filter == MessageSearchFilter::FailedToSend) {
    found_dialog_messages_.erase(random_id);
    promise.set_value(Unit());
    return result;
  }

  LOG(DEBUG) << "Search messages on server in " << dialog_id << " with query \"" << query << "\" from "
             << sender_dialog_id << " in thread of " << top_thread_message_id << " from " << from_message_id
             << " and with limit " << limit;

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<SearchMessagesQuery>(std::move(promise))
          ->send(dialog_id, saved_messages_topic_id, query, sender_dialog_id, from_message_id, offset, limit, filter,
                 top_thread_message_id, tag, random_id);
      break;
    case DialogType::SecretChat:
      if (filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::Pinned ||
          filter == MessageSearchFilter::UnreadReaction) {
        promise.set_value(Unit());
      } else {
        promise.set_error(Status::Error(500, "Message search is not supported in secret chats"));
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Message search is not supported"));
  }
  return result;
}

MessagesManager::FoundMessages MessagesManager::search_call_messages(const string &offset, int32 limit,
                                                                     bool only_missed, int64 &random_id, bool use_db,
                                                                     Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_call_messages_.find(random_id);
    if (it != found_call_messages_.end()) {
      auto result = std::move(it->second);
      found_call_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search call messages from " << offset << " with limit " << limit;

  FoundMessages result;
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  MessageId offset_message_id;
  if (!offset.empty()) {
    auto r_offset_server_message_id = to_integer_safe<int32>(offset);
    if (r_offset_server_message_id.is_error()) {
      promise.set_error(Status::Error(400, "Invalid offset specified"));
      return result;
    }

    offset_message_id = MessageId(ServerMessageId(r_offset_server_message_id.ok()));
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_call_messages_.count(random_id) > 0);
  found_call_messages_[random_id];  // reserve place for result

  auto filter = only_missed ? MessageSearchFilter::MissedCall : MessageSearchFilter::Call;

  if (use_db && G()->use_message_database()) {
    // try to use database
    MessageId first_db_message_id =
        calls_db_state_.first_calls_database_message_id_by_index[call_message_search_filter_index(filter)];
    int32 message_count = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];
    auto fixed_from_message_id = offset_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    CHECK(fixed_from_message_id.is_valid() && fixed_from_message_id.is_server());
    LOG(INFO) << "Search call messages from " << fixed_from_message_id << ", have up to " << first_db_message_id
              << ", message_count = " << message_count;
    if (first_db_message_id < fixed_from_message_id && message_count != -1) {
      LOG(INFO) << "Search messages in database from " << fixed_from_message_id << " and with limit " << limit;

      MessageDbCallsQuery db_query;
      db_query.filter = filter;
      db_query.from_unique_message_id = fixed_from_message_id.get_server_message_id().get();
      db_query.limit = limit;
      G()->td_db()->get_message_db_async()->get_calls(
          db_query, PromiseCreator::lambda([random_id, first_db_message_id, filter, promise = std::move(promise)](
                                               Result<MessageDbCallsResult> calls_result) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_message_db_calls_result, std::move(calls_result),
                         random_id, first_db_message_id, filter, std::move(promise));
          }));
      return result;
    }
  }

  td_->create_handler<SearchMessagesQuery>(std::move(promise))
      ->send(DialogId(), SavedMessagesTopicId(), string(), DialogId(), offset_message_id, 0, limit, filter, MessageId(),
             ReactionType(), random_id);
  return result;
}

void MessagesManager::search_outgoing_document_messages(const string &query, int32 limit,
                                                        Promise<td_api::object_ptr<td_api::foundMessages>> &&promise) {
  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  td_->create_handler<SearchSentMediaQuery>(std::move(promise))->send(query, limit);
}

void MessagesManager::search_dialog_recent_location_messages(DialogId dialog_id, int32 limit,
                                                             Promise<td_api::object_ptr<td_api::messages>> &&promise) {
  LOG(INFO) << "Search recent location messages in " << dialog_id << " with limit " << limit;

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  const Dialog *d = get_dialog_force(dialog_id, "search_dialog_recent_location_messages");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      return td_->create_handler<GetRecentLocationsQuery>(std::move(promise))->send(dialog_id, limit);
    case DialogType::SecretChat:
      return promise.set_value(get_messages_object(0, vector<td_api::object_ptr<td_api::message>>(), false));
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Message search is not supported"));
  }
}

vector<MessageFullId> MessagesManager::get_active_live_location_messages(Promise<Unit> &&promise) {
  if (!G()->use_message_database()) {
    are_active_live_location_messages_loaded_ = true;
  }

  if (!are_active_live_location_messages_loaded_) {
    load_active_live_location_messages_queries_.push_back(std::move(promise));
    if (load_active_live_location_messages_queries_.size() == 1u) {
      LOG(INFO) << "Trying to load active live location messages from database";
      G()->td_db()->get_sqlite_pmc()->get(
          "di_active_live_location_messages", PromiseCreator::lambda([](string value) {
            send_closure(G()->messages_manager(),
                         &MessagesManager::on_load_active_live_location_message_full_ids_from_database,
                         std::move(value));
          }));
    }
    return {};
  }

  promise.set_value(Unit());
  vector<MessageFullId> result;
  for (auto &message_full_id : active_live_location_message_full_ids_) {
    auto m = get_message(message_full_id);
    CHECK(m != nullptr);
    CHECK(m->content->get_type() == MessageContentType::LiveLocation);
    CHECK(!m->message_id.is_scheduled());

    if (m->is_failed_to_send) {
      continue;
    }

    auto live_period = get_message_content_live_location_period(m->content.get());
    if (live_period <= G()->unix_time() - m->date) {  // bool is_expired flag?
      // live location is expired
      continue;
    }
    result.push_back(message_full_id);
  }

  return result;
}

void MessagesManager::on_load_active_live_location_message_full_ids_from_database(string value) {
  if (G()->close_flag()) {
    return;
  }
  if (value.empty()) {
    LOG(INFO) << "Active live location messages aren't found in the database";
    on_load_active_live_location_messages_finished();

    if (!active_live_location_message_full_ids_.empty()) {
      save_active_live_locations();
    }
    return;
  }

  LOG(INFO) << "Successfully loaded active live location messages list of size " << value.size() << " from database";

  auto new_message_full_ids = std::move(active_live_location_message_full_ids_);
  vector<MessageFullId> old_message_full_ids;
  log_event_parse(old_message_full_ids, value).ensure();

  // TODO asynchronously load messages from database
  active_live_location_message_full_ids_.clear();
  for (const auto &message_full_id : old_message_full_ids) {
    Message *m = get_message_force(message_full_id, "on_load_active_live_location_message_full_ids_from_database");
    if (m != nullptr) {
      try_add_active_live_location(message_full_id.get_dialog_id(), m);
    }
  }

  for (const auto &message_full_id : new_message_full_ids) {
    add_active_live_location(message_full_id);
  }

  on_load_active_live_location_messages_finished();

  if (!new_message_full_ids.empty() || old_message_full_ids.size() != active_live_location_message_full_ids_.size()) {
    save_active_live_locations();
  }
}

void MessagesManager::on_load_active_live_location_messages_finished() {
  are_active_live_location_messages_loaded_ = true;
  set_promises(load_active_live_location_messages_queries_);
}

void MessagesManager::try_add_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (m->content->get_type() != MessageContentType::LiveLocation || m->message_id.is_scheduled() ||
      m->message_id.is_local() || m->via_bot_user_id.is_valid() || m->via_business_bot_user_id.is_valid() ||
      m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {  // bool is_expired flag?
    // live location is expired
    return;
  }
  add_active_live_location({dialog_id, m->message_id});
}

void MessagesManager::add_active_live_location(MessageFullId message_full_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  CHECK(message_full_id.get_message_id().is_valid());
  if (!active_live_location_message_full_ids_.insert(message_full_id).second) {
    return;
  }

  // TODO add timer for live location expiration

  if (!G()->use_message_database()) {
    return;
  }

  if (are_active_live_location_messages_loaded_) {
    save_active_live_locations();
  } else if (load_active_live_location_messages_queries_.empty()) {
    // load active live locations and save after that
    get_active_live_location_messages(Auto());
  }
}

bool MessagesManager::delete_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  return active_live_location_message_full_ids_.erase(MessageFullId{dialog_id, m->message_id}) != 0;
}

void MessagesManager::save_active_live_locations() {
  CHECK(are_active_live_location_messages_loaded_);
  LOG(INFO) << "Save active live locations of size " << active_live_location_message_full_ids_.size() << " to database";
  if (G()->use_message_database()) {
    G()->td_db()->get_sqlite_pmc()->set("di_active_live_location_messages",
                                        log_event_store(active_live_location_message_full_ids_).as_slice().str(),
                                        Auto());
  }
}

void MessagesManager::on_message_live_location_viewed(Dialog *d, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->content->get_type() == MessageContentType::LiveLocation);
  CHECK(!m->message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return;
    default:
      UNREACHABLE();
      return;
  }
  if (d->open_count == 0) {
    return;
  }

  if (m->is_outgoing || !m->message_id.is_server() || m->via_bot_user_id.is_valid() ||
      m->via_business_bot_user_id.is_valid() || !m->sender_user_id.is_valid() ||
      td_->user_manager_->is_user_bot(m->sender_user_id) || m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {
    // live location is expired
    return;
  }

  auto &live_location_task_id = pending_viewed_live_locations_[d->dialog_id][m->message_id];
  if (live_location_task_id != 0) {
    return;
  }

  live_location_task_id = ++viewed_live_location_task_id_;
  auto &message_full_id = viewed_live_location_tasks_[live_location_task_id];
  message_full_id = MessageFullId(d->dialog_id, m->message_id);
  view_message_live_location_on_server_impl(live_location_task_id, message_full_id);
}

void MessagesManager::view_message_live_location_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  auto message_full_id = it->second;
  Dialog *d = get_dialog(message_full_id.get_dialog_id());
  const Message *m = get_message_force(d, message_full_id.get_message_id(), "view_message_live_location_on_server");
  if (m == nullptr || get_message_content_live_location_period(m->content.get()) <= G()->unix_time() - m->date + 1) {
    // the message was deleted or live location is expired
    viewed_live_location_tasks_.erase(it);
    auto live_locations_it = pending_viewed_live_locations_.find(d->dialog_id);
    CHECK(live_locations_it != pending_viewed_live_locations_.end());
    auto erased_count = live_locations_it->second.erase(message_full_id.get_message_id());
    CHECK(erased_count > 0);
    if (live_locations_it->second.empty()) {
      pending_viewed_live_locations_.erase(live_locations_it);
    }
    return;
  }

  view_message_live_location_on_server_impl(task_id, message_full_id);
}

void MessagesManager::view_message_live_location_on_server_impl(int64 task_id, MessageFullId message_full_id) {
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), task_id](Unit result) {
    send_closure(actor_id, &MessagesManager::on_message_live_location_viewed_on_server, task_id);
  });
  read_message_contents_on_server(message_full_id.get_dialog_id(), {message_full_id.get_message_id()}, 0,
                                  std::move(promise), true);
}

void MessagesManager::on_message_live_location_viewed_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  pending_message_live_location_view_timeout_.add_timeout_in(task_id, LIVE_LOCATION_VIEW_PERIOD);
}

void MessagesManager::try_add_bot_command_message_id(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || !td_->dialog_manager_->is_group_dialog(dialog_id) ||
      m->message_id.is_scheduled() || !has_bot_commands(get_message_content_text(m->content.get()))) {
    return;
  }

  dialog_bot_command_message_ids_[dialog_id].message_ids.insert(m->message_id);
}

void MessagesManager::delete_bot_command_message_id(DialogId dialog_id, MessageId message_id) {
  if (message_id.is_scheduled()) {
    return;
  }
  auto it = dialog_bot_command_message_ids_.find(dialog_id);
  if (it == dialog_bot_command_message_ids_.end()) {
    return;
  }
  if (it->second.message_ids.erase(message_id) && it->second.message_ids.empty()) {
    dialog_bot_command_message_ids_.erase(it);
  }
}

FileSourceId MessagesManager::get_message_file_source_id(MessageFullId message_full_id, bool force) {
  if (!force) {
    if (td_->auth_manager_->is_bot()) {
      return FileSourceId();
    }

    auto dialog_id = message_full_id.get_dialog_id();
    auto message_id = message_full_id.get_message_id();
    if (!dialog_id.is_valid() || !(message_id.is_valid() || message_id.is_valid_scheduled()) ||
        dialog_id.get_type() == DialogType::SecretChat || !message_id.is_any_server()) {
      return FileSourceId();
    }
  }

  auto &file_source_id = message_full_id_to_file_source_id_[message_full_id];
  if (!file_source_id.is_valid()) {
    file_source_id = td_->file_reference_manager_->create_message_file_source(message_full_id);
  }
  return file_source_id;
}

void MessagesManager::add_message_file_sources(DialogId dialog_id, const Message *m) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (dialog_id.get_type() != DialogType::SecretChat && m->is_content_secret) {
    // return;
  }

  auto file_ids = get_message_file_ids(m);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(MessageFullId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      td_->file_manager_->add_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::remove_message_file_sources(DialogId dialog_id, const Message *m) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto file_ids = get_message_file_ids(m);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(MessageFullId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      auto file_view = td_->file_manager_->get_file_view(file_id);
      send_closure(td_->download_manager_actor_, &DownloadManager::remove_file, file_view.get_main_file_id(),
                   file_source_id, false, Promise<Unit>());
      td_->file_manager_->remove_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::change_message_files(DialogId dialog_id, const Message *m, const vector<FileId> &old_file_ids) {
  if (dialog_id.get_type() != DialogType::SecretChat && m->is_content_secret) {
    // return;
  }

  auto new_file_ids = get_message_file_ids(m);
  if (new_file_ids == old_file_ids) {
    return;
  }

  MessageFullId message_full_id{dialog_id, m->message_id};
  bool need_delete_files = need_delete_message_files(dialog_id, m);
  auto file_source_id = get_message_file_source_id(message_full_id);
  for (auto file_id : old_file_ids) {
    if (!td::contains(new_file_ids, file_id)) {
      if (need_delete_files && need_delete_file(message_full_id, file_id)) {
        send_closure(G()->file_manager(), &FileManager::delete_file, file_id, Promise<Unit>(), "change_message_files");
      }
      if (file_source_id.is_valid()) {
        auto file_view = td_->file_manager_->get_file_view(file_id);
        send_closure(td_->download_manager_actor_, &DownloadManager::remove_file, file_view.get_main_file_id(),
                     file_source_id, false, Promise<Unit>());
      }
    }
  }

  if (file_source_id.is_valid()) {
    td_->file_manager_->change_files_source(file_source_id, old_file_ids, new_file_ids);
  }
}

MessageId MessagesManager::get_first_database_message_id_by_index(const Dialog *d, MessageSearchFilter filter) {
  CHECK(d != nullptr);
  auto message_id = filter == MessageSearchFilter::Empty
                        ? d->first_database_message_id
                        : d->first_database_message_id_by_index[message_search_filter_index(filter)];
  CHECK(!message_id.is_scheduled());
  if (!message_id.is_valid()) {
    if (d->dialog_id.get_type() == DialogType::SecretChat) {
      LOG(ERROR) << "Invalid first_database_message_id_by_index in " << d->dialog_id;
      return MessageId::min();
    }
    return MessageId::max();
  }
  return message_id;
}

void MessagesManager::on_search_dialog_message_db_result(int64 random_id, DialogId dialog_id, MessageId from_message_id,
                                                         MessageId first_db_message_id, MessageSearchFilter filter,
                                                         int32 offset, int32 limit,
                                                         Result<vector<MessageDbDialogMessage>> r_messages,
                                                         Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (r_messages.is_error()) {
    LOG(ERROR) << "Failed to get messages from the database: " << r_messages.error();
    if (first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat &&
        filter != MessageSearchFilter::FailedToSend) {
      found_dialog_messages_.erase(random_id);
    }
    return promise.set_value(Unit());
  }
  CHECK(!from_message_id.is_scheduled());
  CHECK(!first_db_message_id.is_scheduled());

  auto messages = r_messages.move_as_ok();

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  auto &res = it->second.message_ids;

  MessageId next_from_message_id;
  res.reserve(messages.size());
  for (auto &message : messages) {
    auto m = on_get_message_from_database(d, message, false, "on_search_dialog_message_db_result");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      if (!next_from_message_id.is_valid() || m->message_id < next_from_message_id) {
        next_from_message_id = m->message_id;
      }
      if (filter == MessageSearchFilter::UnreadMention && !m->contains_unread_mention) {
        // skip already read by d->last_read_all_mentions_message_id mentions
      } else {
        CHECK(!m->message_id.is_scheduled());
        res.push_back(m->message_id);
      }
    }
  }

  auto &message_count = d->message_count_by_index[message_search_filter_index(filter)];
  auto result_size = narrow_cast<int32>(res.size());
  bool from_the_end =
      from_message_id == MessageId::max() || (offset < 0 && (result_size == 0 || res[0] < from_message_id));
  if ((message_count != -1 && message_count < result_size) ||
      (message_count > result_size && from_the_end && first_db_message_id == MessageId::min() &&
       result_size < limit + offset)) {
    LOG(INFO) << "Fix found message count in " << dialog_id << " from " << message_count << " to " << result_size;
    message_count = result_size;
    if (filter == MessageSearchFilter::UnreadMention) {
      d->unread_mention_count = message_count;
      update_dialog_mention_notification_count(d);
      send_update_chat_unread_mention_count(d);
    }
    if (filter == MessageSearchFilter::UnreadReaction) {
      d->unread_reaction_count = message_count;
      // update_dialog_mention_notification_count(d);
      send_update_chat_unread_reaction_count(d, "on_search_dialog_message_db_result");
    }
    on_dialog_updated(dialog_id, "on_search_dialog_message_db_result");
  }
  it->second.total_count = message_count;
  it->second.next_from_message_id = next_from_message_id;
  if (res.empty() && first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(INFO) << "No messages found in database";
    found_dialog_messages_.erase(it);
  } else {
    LOG(INFO) << "Found " << res.size() << " messages out of " << message_count << " in database";
    if (from_the_end && filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, res.empty() ? MessageId() : res[0]);
    }
  }
  promise.set_value(Unit());
}

td_api::object_ptr<td_api::foundChatMessages> MessagesManager::get_found_chat_messages_object(
    DialogId dialog_id, const FoundDialogMessages &found_dialog_messages, const char *source) {
  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  vector<tl_object_ptr<td_api::message>> result;
  result.reserve(found_dialog_messages.message_ids.size());
  for (const auto &message_id : found_dialog_messages.message_ids) {
    auto message = get_message_object(dialog_id, get_message_force(d, message_id, source), source);
    if (message != nullptr) {
      result.push_back(std::move(message));
    }
  }

  return td_api::make_object<td_api::foundChatMessages>(found_dialog_messages.total_count, std::move(result),
                                                        found_dialog_messages.next_from_message_id.get());
}

td_api::object_ptr<td_api::foundMessages> MessagesManager::get_found_messages_object(
    const FoundMessages &found_messages, const char *source) {
  vector<tl_object_ptr<td_api::message>> result;
  result.reserve(found_messages.message_full_ids.size());
  for (const auto &message_full_id : found_messages.message_full_ids) {
    auto message = get_message_object(message_full_id, source);
    if (message != nullptr) {
      result.push_back(std::move(message));
    }
  }

  return td_api::make_object<td_api::foundMessages>(found_messages.total_count, std::move(result),
                                                    found_messages.next_offset);
}

MessagesManager::FoundMessages MessagesManager::offline_search_messages(DialogId dialog_id, const string &query,
                                                                        string offset, int32 limit,
                                                                        MessageSearchFilter filter, int64 &random_id,
                                                                        Promise<Unit> &&promise) {
  if (!G()->use_message_database()) {
    promise.set_error(Status::Error(400, "Message database is required to search messages in secret chats"));
    return {};
  }

  if (random_id != 0) {
    // request has already been sent before
    auto it = found_fts_messages_.find(random_id);
    CHECK(it != found_fts_messages_.end());
    auto result = std::move(it->second);
    found_fts_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }
  if (dialog_id != DialogId() && !have_dialog_force(dialog_id, "offline_search_messages")) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Limit must be positive"));
    return {};
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  MessageDbFtsQuery fts_query;
  fts_query.query = query;
  fts_query.dialog_id = dialog_id;
  fts_query.filter = filter;
  if (!offset.empty()) {
    auto r_from_search_id = to_integer_safe<int64>(offset);
    if (r_from_search_id.is_error()) {
      promise.set_error(Status::Error(400, "Invalid offset specified"));
      return {};
    }
    fts_query.from_search_id = r_from_search_id.ok();
  }
  fts_query.limit = limit;

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_fts_messages_.count(random_id) > 0);
  found_fts_messages_[random_id];  // reserve place for result

  G()->td_db()->get_message_db_async()->get_messages_fts(
      std::move(fts_query),
      PromiseCreator::lambda([random_id, offset = std::move(offset), limit,
                              promise = std::move(promise)](Result<MessageDbFtsResult> fts_result) mutable {
        send_closure(G()->messages_manager(), &MessagesManager::on_message_db_fts_result, std::move(fts_result),
                     std::move(offset), limit, random_id, std::move(promise));
      }));

  return {};
}

void MessagesManager::on_message_db_fts_result(Result<MessageDbFtsResult> result, string offset, int32 limit,
                                               int64 random_id, Promise<Unit> &&promise) {
  G()->ignore_result_if_closing(result);
  if (result.is_error()) {
    found_fts_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto fts_result = result.move_as_ok();

  auto it = found_fts_messages_.find(random_id);
  CHECK(it != found_fts_messages_.end());
  auto &res = it->second.message_full_ids;

  res.reserve(fts_result.messages.size());
  for (auto &message : fts_result.messages) {
    auto m = on_get_message_from_database(message, false, "on_message_db_fts_result");
    if (m != nullptr) {
      res.emplace_back(message.dialog_id, m->message_id);
    }
  }

  it->second.next_offset = fts_result.next_search_id <= 1 ? string() : to_string(fts_result.next_search_id);
  it->second.total_count = offset.empty() && fts_result.messages.size() < static_cast<size_t>(limit)
                               ? static_cast<int32>(fts_result.messages.size())
                               : -1;

  promise.set_value(Unit());
}

void MessagesManager::on_message_db_calls_result(Result<MessageDbCallsResult> result, int64 random_id,
                                                 MessageId first_db_message_id, MessageSearchFilter filter,
                                                 Promise<Unit> &&promise) {
  G()->ignore_result_if_closing(result);
  if (result.is_error()) {
    found_call_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto calls_result = result.move_as_ok();

  auto it = found_call_messages_.find(random_id);
  CHECK(it != found_call_messages_.end());
  auto &res = it->second.message_full_ids;

  CHECK(!first_db_message_id.is_scheduled());
  res.reserve(calls_result.messages.size());
  MessageId next_offset_message_id;
  for (auto &message : calls_result.messages) {
    auto m = on_get_message_from_database(message, false, "on_message_db_calls_result");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      if (!next_offset_message_id.is_valid() || m->message_id < next_offset_message_id) {
        next_offset_message_id = m->message_id;
      }
      res.emplace_back(message.dialog_id, m->message_id);
    }
  }
  it->second.total_count = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];
  if (next_offset_message_id.is_valid()) {
    it->second.next_offset = PSTRING() << next_offset_message_id.get_server_message_id().get();
  }

  if (res.empty() && first_db_message_id != MessageId::min()) {
    LOG(INFO) << "No messages found in database";
    found_call_messages_.erase(it);
  }

  promise.set_value(Unit());
}

MessagesManager::FoundMessages MessagesManager::search_messages(FolderId folder_id, bool ignore_folder_id,
                                                                const string &query, const string &offset, int32 limit,
                                                                MessageSearchFilter filter, int32 min_date,
                                                                int32 max_date, int64 &random_id,
                                                                Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_messages_.find(random_id);
    CHECK(it != found_messages_.end());
    auto result = std::move(it->second);
    found_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  int32 offset_date = std::numeric_limits<int32>::max();
  DialogId offset_dialog_id;
  MessageId offset_message_id;
  bool is_offset_valid = [&] {
    if (offset.empty()) {
      return true;
    }

    auto parts = full_split(offset, ',');
    if (parts.size() != 3) {
      return false;
    }
    auto r_offset_date = to_integer_safe<int32>(parts[0]);
    auto r_offset_dialog_id = to_integer_safe<int64>(parts[1]);
    auto r_offset_message_id = to_integer_safe<int32>(parts[2]);
    if (r_offset_date.is_error() || r_offset_date.ok() <= 0 || r_offset_message_id.is_error() ||
        r_offset_dialog_id.is_error()) {
      return false;
    }
    offset_date = r_offset_date.ok();
    offset_message_id = MessageId(ServerMessageId(r_offset_message_id.ok()));
    offset_dialog_id = DialogId(r_offset_dialog_id.ok());
    if (!offset_message_id.is_valid() || !offset_dialog_id.is_valid() ||
        DialogManager::get_input_peer_force(offset_dialog_id)->get_id() == telegram_api::inputPeerEmpty::ID) {
      return false;
    }
    return true;
  }();
  if (!is_offset_valid) {
    promise.set_error(Status::Error(400, "Invalid offset specified"));
    return {};
  }

  CHECK(filter != MessageSearchFilter::Call && filter != MessageSearchFilter::MissedCall);
  if (filter == MessageSearchFilter::Mention || filter == MessageSearchFilter::UnreadMention ||
      filter == MessageSearchFilter::UnreadReaction || filter == MessageSearchFilter::FailedToSend ||
      filter == MessageSearchFilter::Pinned) {
    promise.set_error(Status::Error(400, "The filter is not supported"));
    return {};
  }

  if (query.empty() && filter == MessageSearchFilter::Empty) {
    promise.set_value(Unit());
    return {};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_messages_.count(random_id) > 0);
  found_messages_[random_id];  // reserve place for result

  LOG(DEBUG) << "Search all messages filtered by " << filter << " with query = \"" << query << "\" from offset "
             << offset << " and limit " << limit;

  td_->create_handler<SearchMessagesGlobalQuery>(std::move(promise))
      ->send(folder_id, ignore_folder_id, query, offset_date, offset_dialog_id, offset_message_id, limit, filter,
             min_date, max_date, random_id);
  return {};
}

int64 MessagesManager::get_dialog_message_by_date(DialogId dialog_id, int32 date, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_by_date");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return 0;
  }

  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return 0;
  }

  if (date <= 0) {
    date = 1;
  }

  int64 random_id = 0;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || get_dialog_message_by_date_results_.count(random_id) > 0);
  get_dialog_message_by_date_results_[random_id];  // reserve place for result

  auto message_id = d->ordered_messages.find_message_by_date(date, get_get_message_date(d));
  if (message_id.is_valid() &&
      (message_id == d->last_message_id || (*d->ordered_messages.get_const_iterator(message_id))->have_next())) {
    get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
    promise.set_value(Unit());
    return random_id;
  }

  if (G()->use_message_database() && d->last_database_message_id != MessageId()) {
    CHECK(d->first_database_message_id != MessageId());
    G()->td_db()->get_message_db_async()->get_dialog_message_by_date(
        dialog_id, d->first_database_message_id, d->last_database_message_id, date,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, date, random_id,
                                promise = std::move(promise)](Result<MessageDbDialogMessage> result) mutable {
          send_closure(actor_id, &MessagesManager::on_get_dialog_message_by_date_from_database, dialog_id, date,
                       random_id, std::move(result), std::move(promise));
        }));
  } else {
    get_dialog_message_by_date_from_server(d, date, random_id, false, std::move(promise));
  }
  return random_id;
}

void MessagesManager::run_affected_history_query_until_complete(DialogId dialog_id, AffectedHistoryQuery query,
                                                                bool get_affected_messages, Promise<Unit> &&promise) {
  CHECK(!G()->close_flag());
  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, query, get_affected_messages,
                                               promise = std::move(promise)](Result<AffectedHistory> &&result) mutable {
    if (result.is_error()) {
      return promise.set_error(result.move_as_error());
    }

    send_closure(actor_id, &MessagesManager::on_get_affected_history, dialog_id, query, get_affected_messages,
                 result.move_as_ok(), std::move(promise));
  });
  query(dialog_id, std::move(query_promise));
}

void MessagesManager::on_get_affected_history(DialogId dialog_id, AffectedHistoryQuery query,
                                              bool get_affected_messages, AffectedHistory affected_history,
                                              Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  LOG(INFO) << "Receive " << (affected_history.is_final_ ? "final " : "partial ")
            << "affected history with PTS = " << affected_history.pts_
            << " and pts_count = " << affected_history.pts_count_;

  if (affected_history.pts_count_ > 0) {
    if (get_affected_messages) {
      affected_history.pts_count_ = 0;
    }
    auto update_promise = affected_history.is_final_ ? std::move(promise) : Promise<Unit>();
    if (dialog_id.get_type() == DialogType::Channel) {
      add_pending_channel_update(dialog_id, make_tl_object<dummyUpdate>(), affected_history.pts_,
                                 affected_history.pts_count_, std::move(update_promise), "on_get_affected_history");
    } else {
      td_->updates_manager_->add_pending_pts_update(
          make_tl_object<dummyUpdate>(), affected_history.pts_, affected_history.pts_count_,
          Time::now() - (get_affected_messages ? 10.0 : 0.0), std::move(update_promise), "on_get_affected_history");
    }
  } else if (affected_history.is_final_) {
    promise.set_value(Unit());
  }

  if (!affected_history.is_final_) {
    run_affected_history_query_until_complete(dialog_id, std::move(query), get_affected_messages, std::move(promise));
  }
}

std::function<int32(MessageId)> MessagesManager::get_get_message_date(const Dialog *d) const {
  return [d](MessageId message_id) {
    const auto *m = get_message_static(d, message_id);
    CHECK(m != nullptr);
    return m->date;
  };
}

void MessagesManager::on_get_dialog_message_by_date_from_database(DialogId dialog_id, int32 date, int64 random_id,
                                                                  Result<MessageDbDialogMessage> result,
                                                                  Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (result.is_ok()) {
    Message *m = on_get_message_from_database(d, result.ok(), false, "on_get_dialog_message_by_date_from_database");
    if (m != nullptr) {
      auto message_id = d->ordered_messages.find_message_by_date(date, get_get_message_date(d));
      if (!message_id.is_valid()) {
        LOG(ERROR) << "Failed to find " << m->message_id << " in " << dialog_id << " by date " << date;
        message_id = m->message_id;
      }
      get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
      promise.set_value(Unit());
      return;
    }
    // TODO if m == nullptr, we need to just adjust it to the next non-nullptr message, not get from server
  }

  return get_dialog_message_by_date_from_server(d, date, random_id, true, std::move(promise));
}

void MessagesManager::get_dialog_message_by_date_from_server(const Dialog *d, int32 date, int64 random_id,
                                                             bool after_database_search, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  if (d->have_full_history) {
    // request can always be done locally/in memory. There is no need to send request to the server
    if (after_database_search) {
      return promise.set_value(Unit());
    }

    auto message_id = d->ordered_messages.find_message_by_date(date, get_get_message_date(d));
    if (message_id.is_valid()) {
      get_dialog_message_by_date_results_[random_id] = {d->dialog_id, message_id};
    }
    promise.set_value(Unit());
    return;
  }
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    // there is no way to send request to the server
    return promise.set_value(Unit());
  }

  td_->create_handler<GetDialogMessageByDateQuery>(std::move(promise))->send(d->dialog_id, date, random_id);
}

void MessagesManager::on_get_dialog_message_by_date_success(DialogId dialog_id, int32 date, int64 random_id,
                                                            vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                            Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto &result = it->second;
  CHECK(result == MessageFullId());

  for (auto &message : messages) {
    auto message_date = get_message_date(message);
    auto message_dialog_id = DialogId::get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive message in wrong " << message_dialog_id << " instead of " << dialog_id;
      continue;
    }
    if (message_date != 0 && message_date <= date) {
      result = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel, false,
                              "on_get_dialog_message_by_date_success");
      if (result != MessageFullId()) {
        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        auto message_id = d->ordered_messages.find_message_by_date(date, get_get_message_date(d));
        if (!message_id.is_valid()) {
          LOG(ERROR) << "Failed to find " << result.get_message_id() << " in " << dialog_id << " by date " << date;
          message_id = result.get_message_id();
        }
        get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
        // TODO result must be adjusted by local messages
        promise.set_value(Unit());
        return;
      }
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_get_dialog_message_by_date_fail(int64 random_id) {
  auto erased_count = get_dialog_message_by_date_results_.erase(random_id);
  CHECK(erased_count > 0);
}

tl_object_ptr<td_api::message> MessagesManager::get_dialog_message_by_date_object(int64 random_id) {
  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto message_full_id = std::move(it->second);
  get_dialog_message_by_date_results_.erase(it);
  return get_message_object(message_full_id, "get_dialog_message_by_date_object");
}

void MessagesManager::get_dialog_sparse_message_positions(
    DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageSearchFilter filter,
    MessageId from_message_id, int32 limit, Promise<td_api::object_ptr<td_api::messagePositions>> &&promise) {
  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_sparse_message_positions");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (limit < 50 || limit > 2000) {  // server-side limits
    return promise.set_error(Status::Error(400, "Invalid limit specified"));
  }

  CHECK(filter != MessageSearchFilter::Call && filter != MessageSearchFilter::MissedCall);
  if (filter == MessageSearchFilter::Empty || filter == MessageSearchFilter::Mention ||
      filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::UnreadReaction ||
      filter == MessageSearchFilter::Pinned) {
    return promise.set_error(Status::Error(400, "The filter is not supported"));
  }

  if (from_message_id.is_scheduled()) {
    return promise.set_error(Status::Error(400, "Invalid from_message_id specified"));
  }
  if (!from_message_id.is_valid() || from_message_id > d->last_new_message_id) {
    if (d->last_new_message_id.is_valid()) {
      from_message_id = d->last_new_message_id.get_next_message_id(MessageType::Server);
    } else {
      from_message_id = MessageId::max();
    }
  } else {
    from_message_id = from_message_id.get_next_server_message_id();
  }
  TRY_STATUS_PROMISE(promise, saved_messages_topic_id.is_valid_in(td_, dialog_id));

  if (filter == MessageSearchFilter::FailedToSend || dialog_id.get_type() == DialogType::SecretChat) {
    if (saved_messages_topic_id.is_valid()) {
      return promise.set_value(td_api::make_object<td_api::messagePositions>());
    }
    if (!G()->use_message_database()) {
      return promise.set_error(Status::Error(400, "Unsupported without message database"));
    }

    LOG(INFO) << "Get sparse message positions from database";
    auto new_promise =
        PromiseCreator::lambda([promise = std::move(promise)](Result<MessageDbMessagePositions> result) mutable {
          TRY_STATUS_PROMISE(promise, G()->close_status());
          if (result.is_error()) {
            return promise.set_error(result.move_as_error());
          }

          auto positions = result.move_as_ok();
          promise.set_value(td_api::make_object<td_api::messagePositions>(
              positions.total_count, transform(positions.positions, [](const MessageDbMessagePosition &position) {
                return td_api::make_object<td_api::messagePosition>(position.position, position.message_id.get(),
                                                                    position.date);
              })));
        });
    MessageDbGetDialogSparseMessagePositionsQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.filter = filter;
    db_query.from_message_id = from_message_id;
    db_query.limit = limit;
    G()->td_db()->get_message_db_async()->get_dialog_sparse_message_positions(db_query, std::move(new_promise));
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchResultPositionsQuery>(std::move(promise))
          ->send(dialog_id, saved_messages_topic_id, filter, from_message_id, limit);
      break;
    case DialogType::SecretChat:
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::on_get_dialog_sparse_message_positions(
    DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id, MessageSearchFilter filter,
    telegram_api::object_ptr<telegram_api::messages_searchResultsPositions> positions,
    Promise<td_api::object_ptr<td_api::messagePositions>> &&promise) {
  auto message_positions = transform(
      positions->positions_, [](const telegram_api::object_ptr<telegram_api::searchResultPosition> &position) {
        return td_api::make_object<td_api::messagePosition>(
            position->offset_, MessageId(ServerMessageId(position->msg_id_)).get(), position->date_);
      });
  promise.set_value(td_api::make_object<td_api::messagePositions>(positions->count_, std::move(message_positions)));
}

void MessagesManager::get_dialog_message_count(DialogId dialog_id, SavedMessagesTopicId saved_messages_topic_id,
                                               MessageSearchFilter filter, bool return_local,
                                               Promise<int32> &&promise) {
  LOG(INFO) << "Get " << (return_local ? "local " : "") << "number of messages in " << dialog_id << " filtered by "
            << filter;

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_count");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (filter == MessageSearchFilter::Empty) {
    return promise.set_error(Status::Error(400, "Can't use searchMessagesFilterEmpty"));
  }

  TRY_STATUS_PROMISE(promise, saved_messages_topic_id.is_valid_in(td_, dialog_id));
  if (saved_messages_topic_id.is_valid()) {
    if (filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::UnreadReaction ||
        filter == MessageSearchFilter::FailedToSend) {
      return promise.set_value(static_cast<int32>(0));
    }
    if (return_local) {
      return promise.set_value(static_cast<int32>(-1));
    }
  } else {
    auto dialog_type = dialog_id.get_type();
    int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
    if (message_count == -1 && filter == MessageSearchFilter::UnreadMention) {
      message_count = d->unread_mention_count;
    }
    if (message_count == -1 && filter == MessageSearchFilter::UnreadReaction) {
      message_count = d->unread_reaction_count;
    }
    if (message_count != -1 || return_local || dialog_type == DialogType::SecretChat ||
        filter == MessageSearchFilter::FailedToSend) {
      return promise.set_value(std::move(message_count));
    }
  }

  get_dialog_message_count_from_server(dialog_id, saved_messages_topic_id, filter, std::move(promise));
}

void MessagesManager::get_dialog_message_count_from_server(DialogId dialog_id,
                                                           SavedMessagesTopicId saved_messages_topic_id,
                                                           MessageSearchFilter filter, Promise<int32> &&promise) {
  LOG(INFO) << "Get number of messages in " << dialog_id << " with " << saved_messages_topic_id << " filtered by "
            << filter << " from the server";

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchCountersQuery>(std::move(promise))->send(dialog_id, saved_messages_topic_id, filter);
      break;
    case DialogType::None:
    case DialogType::SecretChat:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::get_dialog_message_position(MessageFullId message_full_id, MessageSearchFilter filter,
                                                  MessageId top_thread_message_id,
                                                  SavedMessagesTopicId saved_messages_topic_id,
                                                  Promise<int32> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  TRY_RESULT_PROMISE(promise, d,
                     check_dialog_access(dialog_id, true, AccessRights::Read, "get_dialog_message_position"));

  auto message_id = message_full_id.get_message_id();
  const Message *m = get_message_force(d, message_id, "get_dialog_message_position");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }
  if (!m->message_id.is_valid() || !m->message_id.is_server() ||
      (filter != MessageSearchFilter::Empty &&
       (get_message_index_mask(d->dialog_id, m) & message_search_filter_index_mask(filter)) == 0)) {
    return promise.set_error(Status::Error(400, "Message can't be found in the filter"));
  }

  if (top_thread_message_id != MessageId()) {
    if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
      return promise.set_error(Status::Error(400, "Invalid message thread identifier specified"));
    }
    if (dialog_id.get_type() != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(dialog_id)) {
      return promise.set_error(Status::Error(400, "Can't filter by message thread identifier in the chat"));
    }
    if (m->top_thread_message_id != top_thread_message_id ||
        (m->message_id == top_thread_message_id && !m->is_topic_message)) {
      return promise.set_error(Status::Error(400, "Message doesn't belong to the message thread"));
    }
  }
  TRY_STATUS_PROMISE(promise, saved_messages_topic_id.is_valid_in(td_, dialog_id));

  if (dialog_id.get_type() == DialogType::SecretChat) {
    return promise.set_error(Status::Error(400, "The method can't be used in secret chats"));
  }

  if (filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::UnreadReaction ||
      filter == MessageSearchFilter::FailedToSend) {
    return promise.set_error(Status::Error(400, "The filter is not supported"));
  }

  td_->create_handler<GetMessagePositionQuery>(std::move(promise))
      ->send(dialog_id, message_id, filter, top_thread_message_id, saved_messages_topic_id);
}

void MessagesManager::preload_newer_messages(const Dialog *d, MessageId max_message_id) {
  CHECK(d != nullptr);
  CHECK(max_message_id.is_valid());
  CHECK(!td_->auth_manager_->is_bot());

  auto it = d->ordered_messages.get_const_iterator(max_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10;
  while (*it != nullptr && limit-- > 0) {
    ++it;
    if (*it) {
      max_message_id = (*it)->get_message_id();
    }
  }
  if (limit > 0 && (d->last_message_id == MessageId() || max_message_id < d->last_message_id)) {
    // need to preload some new messages
    LOG(INFO) << "Preloading newer after " << max_message_id;
    load_messages_impl(d, max_message_id, -MAX_GET_HISTORY + 1, MAX_GET_HISTORY, 3, false, Promise<Unit>());
  }
}

void MessagesManager::preload_older_messages(const Dialog *d, MessageId min_message_id) {
  CHECK(d != nullptr);
  CHECK(min_message_id.is_valid());
  CHECK(!td_->auth_manager_->is_bot());

  /*
    if (d->first_remote_message_id == -1) {
      // nothing left to preload from server
      return;
    }
  */
  auto it = d->ordered_messages.get_const_iterator(min_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10 + 1;
  while (*it != nullptr && limit-- > 0) {
    min_message_id = (*it)->get_message_id();
    --it;
  }
  if (limit > 0) {
    // need to preload some old messages
    LOG(INFO) << "Preloading older before " << min_message_id;
    load_messages_impl(d, min_message_id, 0, MAX_GET_HISTORY / 2, 3, false, Promise<Unit>());
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::parse_message(Dialog *d, MessageId expected_message_id,
                                                                    const BufferSlice &value, bool is_scheduled) {
  CHECK(d != nullptr);
  auto dialog_id = d->dialog_id;
  auto message = make_unique<Message>();
  auto *m = message.get();

  auto status = log_event_parse(*m, value.as_slice());
  bool is_message_id_valid = [&] {
    if (is_scheduled) {
      if (!expected_message_id.is_valid_scheduled()) {
        return false;
      }
      if (m->message_id == expected_message_id) {
        return true;
      }
      return m->message_id.is_valid_scheduled() && expected_message_id.is_scheduled_server() &&
             m->message_id.is_scheduled_server() &&
             m->message_id.get_scheduled_server_message_id() == expected_message_id.get_scheduled_server_message_id();
    } else {
      if (!expected_message_id.is_valid()) {
        return false;
      }
      return m->message_id == expected_message_id;
    }
  }();
  if (status.is_error() || !is_message_id_valid) {
    // can't happen unless the database is broken, but has been seen in the wild
    LOG(ERROR) << "Receive invalid message from database: " << expected_message_id << ' ' << m->message_id << ' '
               << status << ' ' << format::as_hex_dump<4>(value.as_slice());
    if (!is_scheduled && dialog_id.get_type() != DialogType::SecretChat) {
      // trying to repair the message
      if (expected_message_id.is_valid() && expected_message_id.is_server()) {
        get_message_from_server({dialog_id, expected_message_id}, Auto(), "parse_message");
      }
      if (m->message_id.is_valid() && m->message_id.is_server()) {
        get_message_from_server({dialog_id, m->message_id}, Auto(), "parse_message");
      }
    }
    return nullptr;
  }
  if (m->reactions != nullptr) {
    if (td_->auth_manager_->is_bot() || m->available_reactions_generation < d->available_reactions_generation) {
      m->reactions = nullptr;
      m->available_reactions_generation = 0;
    } else if (m->available_reactions_generation > d->available_reactions_generation &&
               m->available_reactions_generation - d->available_reactions_generation < 1000000000) {
      switch (dialog_id.get_type()) {
        case DialogType::Chat:
        case DialogType::Channel:
          LOG(ERROR) << "Fix available_reactions_generation in " << dialog_id << " from "
                     << d->available_reactions_generation << " to " << m->available_reactions_generation;
          hide_dialog_message_reactions(d);
          set_dialog_next_available_reactions_generation(d, m->available_reactions_generation);
          on_dialog_updated(dialog_id, "parse_message");
          break;
        case DialogType::User:
        case DialogType::SecretChat:
        default:
          LOG(ERROR) << "Receive available_reactions_generation = " << m->available_reactions_generation << " in "
                     << m->message_id << " in " << dialog_id;
          break;
      }
    }
    if (m->reactions != nullptr) {
      m->reactions->fix_my_recent_chooser_dialog_id(td_->dialog_manager_->get_my_dialog_id());
    }
  }
  if (m->contains_mention && td_->auth_manager_->is_bot()) {
    m->contains_mention = false;
    m->contains_unread_mention = false;
  }
  if (m->history_generation > d->history_generation && m->history_generation - d->history_generation < 1000000000) {
    switch (dialog_id.get_type()) {
      case DialogType::Channel:
        LOG(ERROR) << "Fix history_generation in " << dialog_id << " from " << d->history_generation << " to "
                   << m->history_generation;
        d->history_generation = m->history_generation + 1;
        on_dialog_updated(dialog_id, "parse_message");
        break;
      case DialogType::Chat:
      case DialogType::User:
      case DialogType::SecretChat:
      default:
        LOG(ERROR) << "Receive history_generation = " << m->history_generation << " in " << m->message_id << " in "
                   << dialog_id;
        break;
    }
  }
  if (m->is_pinned && is_scheduled) {
    m->is_pinned = false;
  }
  if (dialog_id == td_->dialog_manager_->get_my_dialog_id() && !m->saved_messages_topic_id.is_valid()) {
    m->saved_messages_topic_id = SavedMessagesTopicId(dialog_id, m->forward_info.get(), m->real_forward_from_dialog_id);
  }

  LOG(INFO) << "Loaded " << m->message_id << " in " << dialog_id << " of size " << value.size() << " from database";
  return message;
}

void MessagesManager::on_get_history_from_database(DialogId dialog_id, MessageId from_message_id,
                                                   MessageId old_last_database_message_id, int32 offset, int32 limit,
                                                   bool only_local, vector<MessageDbDialogMessage> &&messages,
                                                   Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  bool from_the_end = from_message_id == MessageId::max();
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  CHECK(!from_message_id.is_scheduled());

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (old_last_database_message_id < d->last_database_message_id && old_last_database_message_id < from_message_id) {
    // new messages where added to the database since the request was sent
    // they should have been received from the database, so we must repeat the request to get them
    LOG(INFO) << "Reget chat history from database in " << dialog_id
              << ", because last database message was changed from " << old_last_database_message_id << " to "
              << d->last_database_message_id;
    get_history_impl(d, from_message_id, offset, limit, true, only_local, std::move(promise),
                     "on_get_history_from_database 20");
    return;
  }

  LOG(INFO) << "Receive " << messages.size() << " history messages from database "
            << (from_the_end ? "from the end " : "") << "in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ". First database message is "
            << d->first_database_message_id << ", last database message is " << d->last_database_message_id
            << ", have_full_history = " << d->have_full_history
            << ", have_full_history_source = " << d->have_full_history_source;

  bool had_full_history = d->have_full_history;
  auto debug_first_database_message_id = d->first_database_message_id;
  auto debug_last_message_id = d->last_message_id;
  auto debug_last_new_message_id = d->last_new_message_id;

  bool have_error = false;
  auto message_ids = on_get_messages_from_database(
      d, std::move(messages), d->have_full_history ? MessageId::min() : d->first_database_message_id, have_error,
      "on_get_history_from_database");
  if (have_error) {
    // database is broken
    message_ids.clear();  // ignore received messages just in case
    if (d->have_full_history) {
      d->have_full_history = false;
      d->have_full_history_source = 0;
      d->is_empty = false;  // just in case
      on_dialog_updated(dialog_id, "drop have_full_history in on_get_history_from_database");
    }
  }

  vector<bool> is_added_to_dialog;
  for (auto &message_id : message_ids) {
    Message *m = get_message(d, message_id);
    is_added_to_dialog.push_back(m != nullptr);
  }

  bool have_next = false;
  MessageId first_added_message_id;
  MessageId last_added_message_id;
  MessageId next_message_id;
  auto first_received_message_id = MessageId::max();
  MessageId last_received_message_id;
  for (size_t pos = 0; pos < message_ids.size(); pos++) {
    auto message_id = message_ids[pos];
    bool is_added = is_added_to_dialog[pos];
    first_received_message_id = message_id;
    if (!last_received_message_id.is_valid()) {
      last_received_message_id = message_id;
    }

    if (!have_next && (from_the_end || (pos == 0 && offset < -1 && message_id <= from_message_id)) &&
        message_id < d->last_message_id) {
      // last message in the dialog must be attached to the next local message
      have_next = true;
    }

    if (is_added) {
      if (have_next) {
        d->ordered_messages.attach_message_to_next(message_id, "on_get_history");
      }
      first_added_message_id = message_id;
      if (!last_added_message_id.is_valid()) {
        last_added_message_id = message_id;
      }
      if (next_message_id.is_valid()) {
        CHECK(message_id < next_message_id);
        d->ordered_messages.attach_message_to_previous(
            next_message_id, (PSLICE() << "on_get_history_from_database 1 " << message_id << ' ' << from_message_id
                                       << ' ' << offset << ' ' << limit << ' ' << d->first_database_message_id << ' '
                                       << d->have_full_history << ' ' << pos)
                                 .c_str());
      }

      have_next = true;
      next_message_id = message_id;
    }
  }

  if (from_the_end && messages.empty() && d->ordered_messages.empty()) {
    if (d->have_full_history) {
      set_dialog_is_empty(d, "on_get_history_from_database empty");
    } else if (d->last_database_message_id.is_valid()) {
      set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database empty");
      set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database empty");
    }
  }

  if (from_the_end && !last_added_message_id.is_valid() && d->first_database_message_id.is_valid() &&
      !d->have_full_history) {
    if (first_received_message_id <= d->first_database_message_id) {
      // database definitely has no messages from first_database_message_id to last_database_message_id; drop them
      set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database 8");
      set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database 9");
    } else {
      CHECK(first_received_message_id.is_valid());
      // if a message was received, but wasn't added, then it is likely to be already deleted
      // if it is less than d->last_database_message_id, then we can adjust d->last_database_message_id and
      // try again database search without chance to loop
      if (first_received_message_id < d->last_database_message_id) {
        set_dialog_last_database_message_id(d, first_received_message_id, "on_get_history_from_database 12");

        get_history_impl(d, MessageId::max(), 0, -1, true, only_local, std::move(promise),
                         "on_get_history_from_database 21");
        return;
      }

      if (limit > 1) {
        // we expected to have messages [first_database_message_id, last_database_message_id] in the database, but
        // received messages [first_received_message_id, last_received_message_id], none of which can be added
        // first_database_message_id and last_database_message_id are very wrong, so it is better to drop them,
        // pretending that the database has no usable messages
        if (!is_deleted_message(d, d->first_database_message_id) ||
            !is_deleted_message(d, d->last_database_message_id)) {
          if (first_received_message_id == MessageId::max()) {
            CHECK(last_received_message_id == MessageId());
            LOG(ERROR) << "Receive no usable messages in " << dialog_id
                       << " from database from the end, but expected messages from " << d->first_database_message_id
                       << " up to " << d->last_database_message_id
                       << ". Have old last_database_message_id = " << old_last_database_message_id << " and "
                       << messages.size() << " received messages";
          } else {
            LOG(ERROR) << "Receive " << messages.size() << " unusable messages [" << first_received_message_id
                       << " ... " << last_received_message_id << "] in " << dialog_id
                       << " from database from the end, but expected messages from " << d->first_database_message_id
                       << " up to " << d->last_database_message_id;
          }
        }
        set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database 13");
        set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database 14");
      }
    }
  }

  if (!first_added_message_id.is_valid() && !only_local && dialog_id.get_type() != DialogType::SecretChat) {
    load_messages_impl(d, from_message_id, offset, limit, 1, false, std::move(promise));
    return;
  }

  bool need_update_dialog_pos = false;
  if (from_the_end && last_added_message_id.is_valid()) {
    CHECK(next_message_id.is_valid());
    // CHECK(d->first_database_message_id.is_valid());
    // CHECK(last_added_message_id >= d->first_database_message_id);
    if ((had_full_history || d->have_full_history) && !d->last_new_message_id.is_valid() &&
        (last_added_message_id.is_server() || d->dialog_id.get_type() == DialogType::SecretChat)) {
      LOG(ERROR) << "Trying to hard fix " << d->dialog_id << " last new message to " << last_added_message_id
                 << " from on_get_history_from_database 2";
      d->last_new_message_id = last_added_message_id;
      on_dialog_updated(d->dialog_id, "on_get_history_from_database 3");
    }
    if (last_added_message_id > d->last_message_id && d->last_new_message_id.is_valid()) {
      set_dialog_last_message_id(d, last_added_message_id, "on_get_history_from_database 4");
      need_update_dialog_pos = true;
    }
    if (last_added_message_id != d->last_database_message_id && d->last_new_message_id.is_valid()) {
      auto debug_last_database_message_id = d->last_database_message_id;
      auto debug_set_dialog_last_database_message_id = d->debug_set_dialog_last_database_message_id;
      if (!d->first_database_message_id.is_valid() && !d->last_database_message_id.is_valid()) {
        set_dialog_first_database_message_id(d, next_message_id, "on_get_history_from_database 5");
      }
      set_dialog_last_database_message_id(d, last_added_message_id, "on_get_history_from_database 5");
      if (last_added_message_id < d->first_database_message_id || !d->first_database_message_id.is_valid()) {
        LOG_CHECK(had_full_history || d->have_full_history)
            << had_full_history << ' ' << d->have_full_history << ' ' << next_message_id << ' ' << last_added_message_id
            << ' ' << d->first_database_message_id << ' ' << debug_first_database_message_id << ' '
            << d->last_database_message_id << ' ' << debug_last_database_message_id << ' ' << dialog_id << ' '
            << d->last_new_message_id << ' ' << debug_last_new_message_id << ' ' << d->last_message_id << ' '
            << debug_last_message_id << ' ' << debug_set_dialog_last_database_message_id << ' '
            << d->debug_set_dialog_last_database_message_id << ' ' << first_received_message_id << ' '
            << last_received_message_id << ' ' << d->debug_first_database_message_id << ' '
            << d->debug_last_database_message_id << ' ' << d->debug_last_new_message_id << ' '
            << d->have_full_history_source;
        CHECK(next_message_id <= d->last_database_message_id);
        LOG(ERROR) << "Fix first database message in " << dialog_id << " from " << d->first_database_message_id
                   << " to " << next_message_id;
        set_dialog_first_database_message_id(d, next_message_id, "on_get_history_from_database 6");
      }
    }
  }
  if (first_added_message_id.is_valid() && first_added_message_id != d->first_database_message_id &&
      first_received_message_id < d->first_database_message_id && d->last_new_message_id.is_valid() &&
      !d->have_full_history) {
    CHECK(first_added_message_id > d->first_database_message_id);
    set_dialog_first_database_message_id(d, first_added_message_id, "on_get_history_from_database 10");
    if (d->last_database_message_id < d->first_database_message_id) {
      set_dialog_last_database_message_id(d, d->first_database_message_id, "on_get_history_from_database 11");
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_history_from_database 7");
  }

  promise.set_value(Unit());
}

void MessagesManager::load_last_dialog_message_later(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }
  load_last_dialog_message(get_dialog(dialog_id), "load_last_dialog_message");
}

void MessagesManager::load_last_dialog_message(const Dialog *d, const char *source) {
  get_history_impl(d, MessageId::max(), 0, -1, true, false, Promise<Unit>(), source);
}

void MessagesManager::on_get_history_finished(const PendingGetHistoryQuery &query, Result<Unit> &&result) {
  G()->ignore_result_if_closing(result);

  auto it = get_history_queries_.find(query);
  if (it == get_history_queries_.end()) {
    return;
  }
  auto promises = std::move(it->second);
  CHECK(!promises.empty());
  get_history_queries_.erase(it);

  if (result.is_ok()) {
    set_promises(promises);
  } else {
    fail_promises(promises, result.move_as_error());
  }
}

void MessagesManager::get_history_impl(const Dialog *d, MessageId from_message_id, int32 offset, int32 limit,
                                       bool from_database, bool only_local, Promise<Unit> &&promise,
                                       const char *source) {
  CHECK(d != nullptr);
  bool from_the_end = from_message_id == MessageId() || from_message_id == MessageId::max();
  if (!from_the_end) {
    CHECK(from_message_id.is_valid());
  } else {
    from_message_id = MessageId::max();
  }

  auto dialog_id = d->dialog_id;
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    // can't get history in dialogs without read access
    return promise.set_value(Unit());
  }
  if ((!d->first_database_message_id.is_valid() || from_message_id <= d->first_database_message_id) &&
      !d->have_full_history) {
    from_database = false;
  }
  if (!G()->use_message_database()) {
    from_database = false;
  }

  if (from_the_end) {
    // load only 10 messages when repairing the last message and can't save the result to the database
    limit = !promise && (from_database || !G()->use_message_database()) ? max(limit, 10) : MAX_GET_HISTORY;
    offset = 0;
  } else if (offset >= -1) {
    // get history before some server or local message
    limit = clamp(limit + offset + 1, MAX_GET_HISTORY / 2, MAX_GET_HISTORY);
    offset = -1;
  } else {
    // get history around some server or local message
    int32 messages_to_load = max(MAX_GET_HISTORY, limit);
    int32 max_add = max(messages_to_load - limit - 2, 0);
    offset -= max_add;
    limit = MAX_GET_HISTORY;
  }

  PendingGetHistoryQuery query;
  query.dialog_id_ = dialog_id;
  query.from_message_id_ = from_message_id;
  query.offset_ = offset;
  query.limit_ = limit;
  query.from_database_ = from_database;
  query.only_local_ = only_local;

  if (from_database) {
    LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
              << " and limit " << limit << " from database from " << source;

    query.old_last_message_id_ = d->last_database_message_id;

    auto &promises = get_history_queries_[query];
    promises.push_back(std::move(promise));
    if (promises.size() != 1) {
      // query has already been sent, just wait for the result
      return;
    }

    auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), query](Result<Unit> &&result) {
      send_closure(actor_id, &MessagesManager::on_get_history_finished, query, std::move(result));
    });

    MessageDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.from_message_id = from_message_id;
    db_query.offset = offset;
    db_query.limit = limit;
    G()->td_db()->get_message_db_async()->get_messages(
        db_query,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, from_message_id,
                                old_last_database_message_id = d->last_database_message_id, offset, limit, only_local,
                                promise = std::move(query_promise)](vector<MessageDbDialogMessage> messages) mutable {
          send_closure(actor_id, &MessagesManager::on_get_history_from_database, dialog_id, from_message_id,
                       old_last_database_message_id, offset, limit, only_local, std::move(messages),
                       std::move(promise));
        }));
  } else {
    if (only_local || dialog_id.get_type() == DialogType::SecretChat ||
        (from_the_end && d->last_message_id.is_valid())) {
      // if the last message is known, there are no reasons to get message history from server from the end
      return promise.set_value(Unit());
    }

    query.old_last_message_id_ = d->last_new_message_id;

    auto &promises = get_history_queries_[query];
    promises.push_back(std::move(promise));
    if (promises.size() != 1) {
      // query has already been sent, just wait for the result
      return;
    }

    auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), query](Result<Unit> &&result) {
      send_closure(actor_id, &MessagesManager::on_get_history_finished, query, std::move(result));
    });

    if (from_the_end) {
      LOG(INFO) << "Get history from the end of " << dialog_id << " from server from " << source;
      td_->create_handler<GetHistoryQuery>(std::move(query_promise))
          ->send_get_from_the_end(dialog_id, d->last_new_message_id, limit);
    } else {
      LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
                << " and limit " << limit << " from server from " << source;
      td_->create_handler<GetHistoryQuery>(std::move(query_promise))
          ->send(dialog_id, from_message_id.get_next_server_message_id(), d->last_new_message_id, offset, limit);
    }
  }
}

void MessagesManager::load_messages(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                    int left_tries, bool only_local, Promise<Unit> &&promise) {
  load_messages_impl(get_dialog(dialog_id), from_message_id, offset, limit, left_tries, only_local, std::move(promise));
}

void MessagesManager::load_messages_impl(const Dialog *d, MessageId from_message_id, int32 offset, int32 limit,
                                         int left_tries, bool only_local, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  CHECK(offset <= 0);
  CHECK(left_tries > 0);
  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Load " << (only_local ? "local " : "") << "messages in " << dialog_id << " from " << from_message_id
            << " with offset = " << offset << " and limit = " << limit << ". " << left_tries << " tries left";
  only_local |= dialog_id.get_type() == DialogType::SecretChat;
  if (!only_local && d->have_full_history) {
    LOG(INFO) << "Have full history in " << dialog_id << ", so don't need to get chat history from server";
    only_local = true;
  }
  bool from_database = (left_tries > 2 || only_local) && G()->use_message_database();
  get_history_impl(d, from_message_id, offset, limit, from_database, only_local, std::move(promise),
                   "load_messages_impl");
}

vector<MessageId> MessagesManager::get_dialog_scheduled_messages(DialogId dialog_id, bool force, bool ignore_result,
                                                                 Promise<Unit> &&promise) {
  if (G()->close_flag()) {
    promise.set_error(Global::request_aborted_error());
    return {};
  }

  LOG(INFO) << "Get scheduled messages in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_scheduled_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  if (td_->dialog_manager_->is_broadcast_channel(dialog_id) &&
      !td_->chat_manager_->get_channel_status(dialog_id.get_channel_id()).can_post_messages()) {
    promise.set_error(Status::Error(400, "Not enough rights to get scheduled messages"));
    return {};
  }
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    promise.set_value(Unit());
    return {};
  }

  if (!d->has_loaded_scheduled_messages_from_database) {
    load_dialog_scheduled_messages(dialog_id, true, 0, std::move(promise));
    return {};
  }

  vector<MessageId> message_ids;
  if (d->scheduled_messages != nullptr) {
    for (const auto &it : d->scheduled_messages->scheduled_messages_) {
      message_ids.push_back(it.first);
    };
    std::sort(message_ids.begin(), message_ids.end(), std::greater<>());
  }

  if (G()->use_message_database()) {
    bool has_scheduled_database_messages = false;
    for (auto &message_id : message_ids) {
      CHECK(message_id.is_valid_scheduled());
      if (!message_id.is_yet_unsent()) {
        has_scheduled_database_messages = true;
        break;
      }
    }
    set_dialog_has_scheduled_database_messages(d->dialog_id, has_scheduled_database_messages);
  }

  if (d->scheduled_messages_sync_generation != scheduled_messages_sync_generation_) {
    vector<uint64> numbers;
    for (auto &message_id : message_ids) {
      if (!message_id.is_scheduled_server()) {
        continue;
      }

      numbers.push_back(message_id.get_scheduled_server_message_id().get());
      const Message *m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id.get_scheduled_server_message_id() == message_id.get_scheduled_server_message_id());
      numbers.push_back(m->edit_date);
      numbers.push_back(m->date);
    }
    auto hash = get_vector_hash(numbers);

    if (!force && (d->has_scheduled_server_messages ||
                   (d->scheduled_messages_sync_generation == 0 && !G()->use_message_database()))) {
      load_dialog_scheduled_messages(dialog_id, false, hash, std::move(promise));
      return {};
    }
    load_dialog_scheduled_messages(dialog_id, false, hash, Promise<Unit>());
  }

  if (!ignore_result) {
    d->sent_scheduled_messages = true;
  }

  promise.set_value(Unit());
  return message_ids;
}

void MessagesManager::load_dialog_scheduled_messages(DialogId dialog_id, bool from_database, int64 hash,
                                                     Promise<Unit> &&promise) {
  CHECK(dialog_id.get_type() != DialogType::SecretChat);
  if (G()->use_message_database() && from_database) {
    LOG(INFO) << "Load scheduled messages from database in " << dialog_id;
    auto &queries = load_scheduled_messages_from_database_queries_[dialog_id];
    queries.push_back(std::move(promise));
    if (queries.size() == 1) {
      G()->td_db()->get_message_db_async()->get_scheduled_messages(
          dialog_id, 1000,
          PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](vector<MessageDbDialogMessage> messages) {
            send_closure(actor_id, &MessagesManager::on_get_scheduled_messages_from_database, dialog_id,
                         std::move(messages));
          }));
    }
  } else {
    td_->create_handler<GetAllScheduledMessagesQuery>(std::move(promise))
        ->send(dialog_id, hash, scheduled_messages_sync_generation_);
  }
}

void MessagesManager::on_get_scheduled_messages_from_database(DialogId dialog_id,
                                                              vector<MessageDbDialogMessage> &&messages) {
  if (G()->close_flag()) {
    auto it = load_scheduled_messages_from_database_queries_.find(dialog_id);
    CHECK(it != load_scheduled_messages_from_database_queries_.end());
    CHECK(!it->second.empty());
    auto promises = std::move(it->second);
    load_scheduled_messages_from_database_queries_.erase(it);

    fail_promises(promises, Global::request_aborted_error());
    return;
  }
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  d->has_loaded_scheduled_messages_from_database = true;

  LOG(INFO) << "Receive " << messages.size() << " scheduled messages from database in " << dialog_id;

  Dependencies dependencies;
  vector<MessageId> added_message_ids;
  for (auto &message_slice : messages) {
    auto message = parse_message(d, message_slice.message_id, message_slice.data, true);
    if (message == nullptr) {
      continue;
    }

    if (get_message(d, message->message_id) != nullptr) {
      continue;
    }

    bool need_update = false;
    Message *m = add_scheduled_message_to_dialog(d, std::move(message), true, false, &need_update,
                                                 "on_get_scheduled_messages_from_database");
    if (m != nullptr) {
      add_message_dependencies(dependencies, m);
      added_message_ids.push_back(m->message_id);
    }
  }
  dependencies.resolve_force(td_, "on_get_scheduled_messages_from_database");

  // for (auto message_id : added_message_ids) {
  //   send_update_new_message(d, get_message(d, message_id));
  // }
  send_update_chat_has_scheduled_messages(d, false);

  auto it = load_scheduled_messages_from_database_queries_.find(dialog_id);
  CHECK(it != load_scheduled_messages_from_database_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  load_scheduled_messages_from_database_queries_.erase(it);

  set_promises(promises);
}

bool MessagesManager::can_add_message_tag(DialogId dialog_id, const MessageReactions *reactions) const {
  return dialog_id == td_->dialog_manager_->get_my_dialog_id() &&
         (reactions == nullptr || reactions->reactions_.empty() || reactions->are_tags_);
}

Result<td_api::object_ptr<td_api::availableReactions>> MessagesManager::get_message_available_reactions(
    MessageFullId message_full_id, int32 row_size) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "get_message_available_reactions");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  const Message *m = get_message_force(d, message_full_id.get_message_id(), "get_message_available_reactions");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  bool is_tag = can_add_message_tag(dialog_id, m->reactions.get());
  ReactionUnavailabilityReason unavailability_reason = ReactionUnavailabilityReason::None;
  auto available_reactions = get_message_available_reactions(d, m, false, &unavailability_reason);
  return td_->reaction_manager_->get_sorted_available_reactions(
      std::move(available_reactions), get_message_active_reactions(d, m), row_size, is_tag, unavailability_reason);
}

ChatReactions MessagesManager::get_message_available_reactions(const Dialog *d, const Message *m,
                                                               bool disallow_custom_for_non_premium,
                                                               ReactionUnavailabilityReason *unavailability_reason) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  auto active_reactions = get_message_active_reactions(d, m);
  if (active_reactions.empty()) {
    return {};
  }

  bool can_use_reactions = true;
  if (d->dialog_id.get_type() == DialogType::Channel) {
    auto channel_id = d->dialog_id.get_channel_id();
    if (td_->chat_manager_->is_megagroup_channel(channel_id) &&
        !td_->chat_manager_->get_channel_status(channel_id).is_member() && can_send_message(d->dialog_id).is_error()) {
      // can't use reactions if can't send messages to the group without joining
      if (unavailability_reason != nullptr) {
        *unavailability_reason = ReactionUnavailabilityReason::Guest;
      }
      can_use_reactions = false;
    } else if (td_->dialog_manager_->is_anonymous_administrator(d->dialog_id, nullptr) &&
               !td_->dialog_manager_->is_broadcast_channel(d->dialog_id) &&
               !td_->chat_manager_->get_channel_status(channel_id).is_creator()) {
      // only creator can react as the chat
      if (unavailability_reason != nullptr) {
        *unavailability_reason = ReactionUnavailabilityReason::AnonymousAdministrator;
      }
      can_use_reactions = false;
    }
  }

  int64 reactions_uniq_max = td_->option_manager_->get_option_integer("reactions_uniq_max", 11);
  bool can_add_new_reactions =
      m->reactions == nullptr || static_cast<int64>(m->reactions->reactions_.size()) < reactions_uniq_max;

  if (!can_use_reactions || !can_add_new_reactions) {
    active_reactions = ChatReactions();
  }

  if (active_reactions.allow_all_regular_) {
    if (can_add_message_tag(d->dialog_id, m->reactions.get())) {
      auto default_tag_reactions = td_->reaction_manager_->get_default_tag_reactions();
      active_reactions.reaction_types_ = default_tag_reactions;
      if (td_->option_manager_->get_option_boolean("is_premium")) {
        for (auto &reaction_type : active_reaction_types_) {
          if (!td::contains(default_tag_reactions, reaction_type)) {
            active_reactions.reaction_types_.push_back(reaction_type);
          }
        }
      } else {
        disallow_custom_for_non_premium = true;
      }
    } else {
      active_reactions.reaction_types_ = active_reaction_types_;
    }
    active_reactions.allow_all_regular_ = false;
  }
  if (can_use_reactions && m->reactions != nullptr) {
    for (const auto &reaction : m->reactions->reactions_) {
      // an already used reaction can be added if it is an active reaction
      const auto &reaction_type = reaction.get_reaction_type();
      if (reaction_type.is_active_reaction(active_reaction_pos_) &&
          !td::contains(active_reactions.reaction_types_, reaction_type)) {
        active_reactions.reaction_types_.push_back(reaction_type);
      }
    }
  }
  if (disallow_custom_for_non_premium && !td_->option_manager_->get_option_boolean("is_premium")) {
    active_reactions.allow_all_custom_ = false;
  }
  return active_reactions;
}

DialogId MessagesManager::get_my_reaction_dialog_id(const Dialog *d) const {
  auto my_dialog_id = td_->dialog_manager_->get_my_dialog_id();
  auto reaction_dialog_id =
      d->default_send_message_as_dialog_id.is_valid() ? d->default_send_message_as_dialog_id : my_dialog_id;
  if (reaction_dialog_id == my_dialog_id && td_->dialog_manager_->is_anonymous_administrator(d->dialog_id, nullptr) &&
      !td_->dialog_manager_->is_broadcast_channel(d->dialog_id)) {
    // react as the supergroup
    return d->dialog_id;
  }
  return reaction_dialog_id;
}

void MessagesManager::add_message_reaction(MessageFullId message_full_id, ReactionType reaction_type, bool is_big,
                                           bool add_to_recent, Promise<Unit> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "add_message_reaction");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  Message *m = get_message_force(d, message_full_id.get_message_id(), "add_message_reaction");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  if (!get_message_available_reactions(d, m, true, nullptr).is_allowed_reaction_type(reaction_type)) {
    return promise.set_error(Status::Error(400, "The reaction isn't available for the message"));
  }

  bool have_recent_choosers =
      !td_->dialog_manager_->is_broadcast_channel(dialog_id) && !is_discussion_message(dialog_id, m);
  if (m->reactions == nullptr) {
    m->reactions = make_unique<MessageReactions>();
    m->reactions->can_get_added_reactions_ = have_recent_choosers && dialog_id.get_type() != DialogType::User;
    m->available_reactions_generation = d->available_reactions_generation;
  }

  LOG(INFO) << "Have message with " << *m->reactions;
  bool is_tag = can_add_message_tag(dialog_id, m->reactions.get());
  auto old_chosen_tags = get_chosen_tags(m->reactions);
  if (!m->reactions->add_my_reaction(reaction_type, is_big, get_my_reaction_dialog_id(d), have_recent_choosers,
                                     is_tag)) {
    return promise.set_value(Unit());
  }

  set_message_reactions(d, m, is_big, add_to_recent, std::move(promise));

  if (is_tag) {
    td_->reaction_manager_->update_saved_messages_tags(m->saved_messages_topic_id, old_chosen_tags,
                                                       get_chosen_tags(m->reactions));
  } else if (add_to_recent) {
    td_->reaction_manager_->add_recent_reaction(reaction_type);
  }
}

void MessagesManager::remove_message_reaction(MessageFullId message_full_id, ReactionType reaction_type,
                                              Promise<Unit> &&promise) {
  auto dialog_id = message_full_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "remove_message_reaction");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  Message *m = get_message_force(d, message_full_id.get_message_id(), "remove_message_reaction");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  if (reaction_type.is_empty()) {
    return promise.set_error(Status::Error(400, "Invalid reaction specified"));
  }

  if (m->reactions == nullptr) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Have message with " << *m->reactions;
  auto old_chosen_tags = get_chosen_tags(m->reactions);
  if (!m->reactions->remove_my_reaction(reaction_type, get_my_reaction_dialog_id(d))) {
    return promise.set_value(Unit());
  }

  set_message_reactions(d, m, false, false, std::move(promise));

  if (!old_chosen_tags.empty()) {
    td_->reaction_manager_->update_saved_messages_tags(m->saved_messages_topic_id, old_chosen_tags,
                                                       get_chosen_tags(m->reactions));
  }
}

void MessagesManager::set_message_reactions(Dialog *d, Message *m, bool is_big, bool add_to_recent,
                                            Promise<Unit> &&promise) {
  CHECK(m->reactions != nullptr);
  m->reactions->sort_reactions(active_reaction_pos_);

  LOG(INFO) << "Update message reactions to " << *m->reactions;

  MessageFullId message_full_id{d->dialog_id, m->message_id};
  pending_reactions_[message_full_id].query_count++;

  send_update_message_interaction_info(d->dialog_id, m);
  on_message_changed(d, m, true, "set_message_reactions");

  // TODO cancel previous queries, log event
  auto query_promise = PromiseCreator::lambda(
      [actor_id = actor_id(this), message_full_id, promise = std::move(promise)](Result<Unit> &&result) mutable {
        send_closure(actor_id, &MessagesManager::on_set_message_reactions, message_full_id, std::move(result),
                     std::move(promise));
      });
  send_message_reaction(td_, message_full_id, m->reactions->get_chosen_reaction_types(), is_big, add_to_recent,
                        std::move(query_promise));
}

void MessagesManager::on_set_message_reactions(MessageFullId message_full_id, Result<Unit> result,
                                               Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  bool need_reload = result.is_error();
  auto it = pending_reactions_.find(message_full_id);
  CHECK(it != pending_reactions_.end());
  if (--it->second.query_count == 0) {
    // need_reload |= it->second.was_updated;
    pending_reactions_.erase(it);
  }

  if (!have_message_force(message_full_id, "on_set_message_reactions")) {
    return promise.set_value(Unit());
  }

  if (need_reload) {
    queue_message_reactions_reload(message_full_id);
  }

  promise.set_result(std::move(result));
}

void MessagesManager::on_read_message_reactions(DialogId dialog_id, vector<MessageId> &&message_ids,
                                                Result<Unit> &&result) {
  for (auto message_id : message_ids) {
    MessageFullId message_full_id{dialog_id, message_id};
    auto it = pending_read_reactions_.find(message_full_id);
    CHECK(it != pending_read_reactions_.end());
    if (--it->second == 0) {
      pending_read_reactions_.erase(it);
    }

    if (!have_message_force(message_full_id, "on_read_message_reactions")) {
      continue;
    }

    if (result.is_error()) {
      queue_message_reactions_reload(message_full_id);
    }
  }
}

Result<int32> MessagesManager::get_message_schedule_date(
    td_api::object_ptr<td_api::MessageSchedulingState> &&scheduling_state) {
  if (scheduling_state == nullptr) {
    return 0;
  }

  switch (scheduling_state->get_id()) {
    case td_api::messageSchedulingStateSendWhenOnline::ID: {
      auto send_date = SCHEDULE_WHEN_ONLINE_DATE;
      return send_date;
    }
    case td_api::messageSchedulingStateSendAtDate::ID: {
      auto send_at_date = td_api::move_object_as<td_api::messageSchedulingStateSendAtDate>(scheduling_state);
      auto send_date = send_at_date->send_date_;
      if (send_date <= 0) {
        return Status::Error(400, "Invalid send date specified");
      }
      if (send_date <= G()->unix_time() + 10) {
        return 0;
      }
      if (send_date - G()->unix_time() > 367 * 86400) {
        return Status::Error(400, "Send date is too far in the future");
      }
      return send_date;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

tl_object_ptr<td_api::MessageSendingState> MessagesManager::get_message_sending_state_object(const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    return td_api::make_object<td_api::messageSendingStatePending>(m->sending_id);
  }
  if (m->is_failed_to_send) {
    auto can_retry = can_resend_message(m);
    auto error_code = m->send_error_code > 0 ? m->send_error_code : 400;
    auto need_another_sender =
        can_retry && error_code == 400 && m->send_error_message == CSlice("SEND_AS_PEER_INVALID");
    auto need_another_reply_quote =
        can_retry && error_code == 400 && m->send_error_message == CSlice("QUOTE_TEXT_INVALID");
    auto need_drop_reply =
        can_retry && error_code == 400 && m->send_error_message == CSlice("REPLY_MESSAGE_ID_INVALID");
    return td_api::make_object<td_api::messageSendingStateFailed>(
        td_api::make_object<td_api::error>(error_code, m->send_error_message), can_retry, need_another_sender,
        need_another_reply_quote, need_drop_reply, max(m->try_resend_at - Time::now(), 0.0));
  }
  return nullptr;
}

tl_object_ptr<td_api::MessageSchedulingState> MessagesManager::get_message_scheduling_state_object(int32 send_date) {
  if (send_date == SCHEDULE_WHEN_ONLINE_DATE) {
    return td_api::make_object<td_api::messageSchedulingStateSendWhenOnline>();
  }
  return td_api::make_object<td_api::messageSchedulingStateSendAtDate>(send_date);
}

td_api::object_ptr<td_api::MessageContent> MessagesManager::get_message_message_content_object(DialogId dialog_id,
                                                                                               const Message *m) const {
  auto live_location_date = m->is_failed_to_send ? 0 : m->date;
  return get_message_content_object(m->content.get(), td_, dialog_id, live_location_date, m->is_content_secret,
                                    need_skip_bot_commands(dialog_id, m), get_message_max_media_timestamp(m),
                                    m->invert_media, m->disable_web_page_preview);
}

td_api::object_ptr<td_api::message> MessagesManager::get_dialog_event_log_message_object(
    DialogId dialog_id, tl_object_ptr<telegram_api::Message> &&message, DialogId &sender_dialog_id) {
  auto dialog_message = create_message(
      td_, parse_telegram_api_message(td_, std::move(message), false, "get_dialog_event_log_message_object"),
      dialog_id.get_type() == DialogType::Channel, false, "get_dialog_event_log_message_object");
  const Message *m = dialog_message.second.get();
  if (m == nullptr || dialog_message.first != dialog_id) {
    LOG(ERROR) << "Failed to create event log message in " << dialog_id;
    return nullptr;
  }
  sender_dialog_id = get_message_sender(m);

  auto sender = get_message_sender_object_const(td_, m->sender_user_id, m->sender_dialog_id,
                                                "get_dialog_event_log_message_object");
  auto forward_info =
      m->forward_info == nullptr ? nullptr : m->forward_info->get_message_forward_info_object(td_, false);
  auto import_info = m->forward_info == nullptr ? nullptr : m->forward_info->get_message_import_info_object();
  auto interaction_info = get_message_interaction_info_object(dialog_id, m);
  auto can_be_saved = can_save_message(dialog_id, m);
  auto via_bot_user_id =
      td_->user_manager_->get_user_id_object(m->via_bot_user_id, "get_dialog_event_log_message_object via_bot_user_id");
  auto edit_date = m->hide_edit_date ? 0 : m->edit_date;
  auto reply_markup = get_reply_markup_object(td_->user_manager_.get(), m->reply_markup);
  auto content =
      get_message_content_object(m->content.get(), td_, dialog_id, 0, false, true,
                                 get_message_own_max_media_timestamp(m), m->invert_media, m->disable_web_page_preview);
  return td_api::make_object<td_api::message>(
      m->message_id.get(), std::move(sender), get_chat_id_object(dialog_id, "get_dialog_event_log_message_object"),
      nullptr, nullptr, m->is_outgoing, m->is_pinned, m->is_from_offline, false, false, false, can_be_saved, false,
      false, false, false, false, false, false, false, false, true, m->is_channel_post, m->is_topic_message, false,
      m->date, edit_date, std::move(forward_info), std::move(import_info), std::move(interaction_info), Auto(), nullptr,
      0, 0, nullptr, 0.0, 0.0, via_bot_user_id, 0, m->sender_boost_count, m->author_signature, 0,
      get_restriction_reason_description(m->restriction_reasons), std::move(content), std::move(reply_markup));
}

td_api::object_ptr<td_api::businessMessage> MessagesManager::get_business_message_object(
    telegram_api::object_ptr<telegram_api::Message> &&message,
    telegram_api::object_ptr<telegram_api::Message> &&reply_to_message) {
  auto message_object = get_business_message_message_object(std::move(message));
  if (message_object == nullptr) {
    LOG(ERROR) << "Failed to create a business message";
    return nullptr;
  }
  return td_api::make_object<td_api::businessMessage>(std::move(message_object),
                                                      get_business_message_message_object(std::move(reply_to_message)));
}

td_api::object_ptr<td_api::message> MessagesManager::get_business_message_message_object(
    telegram_api::object_ptr<telegram_api::Message> &&message) {
  CHECK(td_->auth_manager_->is_bot());
  if (message == nullptr) {
    return nullptr;
  }
  auto dialog_message = create_message(
      td_, parse_telegram_api_message(td_, std::move(message), false, "get_business_message_message_object"), false,
      true, "get_business_message_message_object");
  const Message *m = dialog_message.second.get();
  if (m == nullptr) {
    return nullptr;
  }

  auto dialog_id = dialog_message.first;
  if (dialog_id.get_type() != DialogType::User) {
    LOG(ERROR) << "Receive a business message in " << dialog_id;
    return nullptr;
  }
  force_create_dialog(dialog_id, "get_business_message_message_object chat", true);

  auto sender = get_message_sender_object_const(td_, m->sender_user_id, m->sender_dialog_id,
                                                "get_business_message_message_object");
  auto forward_info =
      m->forward_info == nullptr ? nullptr : m->forward_info->get_message_forward_info_object(td_, false);
  auto import_info = m->forward_info == nullptr ? nullptr : m->forward_info->get_message_import_info_object();
  auto can_be_saved = can_save_message(dialog_id, m);
  auto via_bot_user_id =
      td_->user_manager_->get_user_id_object(m->via_bot_user_id, "get_business_message_message_object via_bot_user_id");
  auto via_business_bot_user_id = td_->user_manager_->get_user_id_object(
      m->via_business_bot_user_id, "get_business_message_message_object via_business_bot_user_id");
  auto reply_to = [&]() -> td_api::object_ptr<td_api::MessageReplyTo> {
    if (!m->replied_message_info.is_empty()) {
      return m->replied_message_info.get_message_reply_to_message_object(td_, dialog_id);
    }
    if (m->reply_to_story_full_id.is_valid()) {
      return td_api::make_object<td_api::messageReplyToStory>(
          get_chat_id_object(m->reply_to_story_full_id.get_dialog_id(),
                             "get_business_message_message_object messageReplyToStory"),
          m->reply_to_story_full_id.get_story_id().get());
    }
    return nullptr;
  }();
  auto reply_markup = get_reply_markup_object(td_->user_manager_.get(), m->reply_markup);
  auto content = get_message_message_content_object(dialog_id, m);
  auto self_destruct_type = m->ttl.get_message_self_destruct_type_object();

  return td_api::make_object<td_api::message>(
      m->message_id.get(), std::move(sender), get_chat_id_object(dialog_id, "get_business_message_message_object"),
      nullptr, nullptr, m->is_outgoing, m->is_from_offline, false, false, false, false, can_be_saved, false, false,
      false, false, false, false, false, false, false, false, false, false, false, m->date, m->edit_date,
      std::move(forward_info), std::move(import_info), nullptr, Auto(), std::move(reply_to), 0, 0,
      std::move(self_destruct_type), 0.0, 0.0, via_bot_user_id, via_business_bot_user_id, 0, string(),
      m->media_album_id, get_restriction_reason_description(m->restriction_reasons), std::move(content),
      std::move(reply_markup));
}

td_api::object_ptr<td_api::message> MessagesManager::get_message_object(MessageFullId message_full_id,
                                                                        const char *source) {
  return get_message_object(message_full_id.get_dialog_id(), get_message_force(message_full_id, source), source);
}

td_api::object_ptr<td_api::message> MessagesManager::get_message_object(DialogId dialog_id, const Message *m,
                                                                        const char *source) const {
  if (m == nullptr) {
    return nullptr;
  }
  LOG_CHECK(have_dialog(dialog_id)) << source;

  auto is_bot = td_->auth_manager_->is_bot();
  auto sending_state = get_message_sending_state_object(m);
  if (sending_state == nullptr || !is_bot) {
    m->is_update_sent = true;
  }
  bool can_delete = can_delete_message(dialog_id, m);
  bool is_scheduled = m->message_id.is_scheduled();
  DialogId my_dialog_id = td_->dialog_manager_->get_my_dialog_id();
  bool can_delete_for_self = false;
  bool can_delete_for_all_users = can_delete && can_revoke_message(dialog_id, m);
  if (can_delete) {
    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        // TODO allow to delete yet unsent message just for self
        can_delete_for_self = !m->message_id.is_yet_unsent() || dialog_id == my_dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        can_delete_for_self = !can_delete_for_all_users;
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
  if (is_scheduled) {
    can_delete_for_self = (dialog_id == my_dialog_id);
    can_delete_for_all_users = !can_delete_for_self;
  }

  bool is_outgoing = m->is_outgoing;
  if (dialog_id == my_dialog_id) {
    // in Saved Messages all non-forwarded messages must be outgoing
    // a forwarded message is outgoing, only if it doesn't have from_dialog_id and its sender isn't hidden
    // i.e. a message is incoming only if it's a forwarded message with known from_dialog_id or with a hidden sender
    is_outgoing =
        is_scheduled || m->forward_info == nullptr ||
        (!m->forward_info->get_last_dialog_id().is_valid() && !m->forward_info->get_origin().is_sender_hidden());
  }

  double ttl_expires_in =
      m->ttl_expires_at != 0 ? clamp(m->ttl_expires_at - Time::now(), 1e-3, m->ttl.get_input_ttl() - 1e-3) : 0.0;
  double auto_delete_in =
      m->ttl_period == 0 ? 0.0 : clamp(m->date + m->ttl_period - G()->server_time(), 1e-3, m->ttl_period - 1e-3);
  auto sender = get_message_sender_object_const(td_, m->sender_user_id, m->sender_dialog_id, source);
  auto scheduling_state = is_scheduled ? get_message_scheduling_state_object(m->date) : nullptr;
  auto forward_info = m->forward_info == nullptr
                          ? nullptr
                          : m->forward_info->get_message_forward_info_object(td_, dialog_id == my_dialog_id);
  auto import_info = m->forward_info == nullptr ? nullptr : m->forward_info->get_message_import_info_object();
  auto interaction_info = is_bot ? nullptr : get_message_interaction_info_object(dialog_id, m);
  auto unread_reactions = get_unread_reactions_object(dialog_id, m);
  auto can_be_saved = can_save_message(dialog_id, m);
  auto can_be_edited = can_edit_message(dialog_id, m, false, is_bot);
  auto can_be_forwarded = can_be_saved && can_forward_message(dialog_id, m);
  auto can_be_replied_in_another_chat = can_be_forwarded && m->message_id.is_server();
  auto can_get_added_reactions = m->reactions != nullptr && m->reactions->can_get_added_reactions_;
  auto can_get_statistics = can_get_message_statistics(dialog_id, m);
  auto can_get_message_thread = get_top_thread_message_full_id(dialog_id, m, false).is_ok();
  auto can_get_read_date = can_get_message_read_date(dialog_id, m).is_ok();
  auto can_get_viewers = can_get_message_viewers(dialog_id, m).is_ok();
  auto can_get_media_timestamp_links = can_get_media_timestamp_link(dialog_id, m).is_ok();
  auto can_report_reactions = can_report_message_reactions(dialog_id, m);
  auto via_bot_user_id =
      td_->user_manager_->get_user_id_object(m->via_bot_user_id, "get_message_object via_bot_user_id");
  auto via_business_bot_user_id = td_->user_manager_->get_user_id_object(m->via_business_bot_user_id,
                                                                         "get_message_object via_business_bot_user_id");
  auto reply_to = [&]() -> td_api::object_ptr<td_api::MessageReplyTo> {
    if (!m->replied_message_info.is_empty()) {
      if (!is_bot && m->is_topic_message &&
          m->replied_message_info.get_same_chat_reply_to_message_id(false) == m->top_thread_message_id) {
        return nullptr;
      }
      return m->replied_message_info.get_message_reply_to_message_object(td_, dialog_id);
    }
    if (m->reply_to_story_full_id.is_valid()) {
      return td_api::make_object<td_api::messageReplyToStory>(
          get_chat_id_object(m->reply_to_story_full_id.get_dialog_id(), "get_message_object messageReplyToStory"),
          m->reply_to_story_full_id.get_story_id().get());
    }
    return nullptr;
  }();
  auto top_thread_message_id = m->top_thread_message_id.get();
  auto date = is_scheduled ? 0 : m->date;
  auto edit_date = m->hide_edit_date ? 0 : m->edit_date;
  auto has_timestamped_media = reply_to == nullptr || m->max_own_media_timestamp >= 0;
  auto reply_markup = get_reply_markup_object(td_->user_manager_.get(), m->reply_markup);
  auto content = get_message_message_content_object(dialog_id, m);
  auto self_destruct_type = m->ttl.get_message_self_destruct_type_object();

  return td_api::make_object<td_api::message>(
      m->message_id.get(), std::move(sender), get_chat_id_object(dialog_id, "get_message_object"),
      std::move(sending_state), std::move(scheduling_state), is_outgoing, m->is_pinned, m->is_from_offline,
      can_be_edited, can_be_forwarded, can_be_replied_in_another_chat, can_be_saved, can_delete_for_self,
      can_delete_for_all_users, can_get_added_reactions, can_get_statistics, can_get_message_thread, can_get_read_date,
      can_get_viewers, can_get_media_timestamp_links, can_report_reactions, has_timestamped_media, m->is_channel_post,
      m->is_topic_message, m->contains_unread_mention, date, edit_date, std::move(forward_info), std::move(import_info),
      std::move(interaction_info), std::move(unread_reactions), std::move(reply_to), top_thread_message_id,
      td_->saved_messages_manager_->get_saved_messages_topic_id_object(m->saved_messages_topic_id),
      std::move(self_destruct_type), ttl_expires_in, auto_delete_in, via_bot_user_id, via_business_bot_user_id,
      m->sender_boost_count, m->author_signature, m->media_album_id,
      get_restriction_reason_description(m->restriction_reasons), std::move(content), std::move(reply_markup));
}

td_api::object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count, DialogId dialog_id,
                                                                          const vector<MessageId> &message_ids,
                                                                          bool skip_not_found, const char *source) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto message_objects = transform(message_ids, [this, dialog_id, d, source](MessageId message_id) {
    return get_message_object(dialog_id, get_message_force(d, message_id, source), source);
  });
  return get_messages_object(total_count, std::move(message_objects), skip_not_found);
}

td_api::object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count,
                                                                          const vector<MessageFullId> &message_full_ids,
                                                                          bool skip_not_found, const char *source) {
  auto message_objects = transform(message_full_ids, [this, source](MessageFullId message_full_id) {
    return get_message_object(message_full_id, source);
  });
  return get_messages_object(total_count, std::move(message_objects), skip_not_found);
}

td_api::object_ptr<td_api::messages> MessagesManager::get_messages_object(
    int32 total_count, vector<tl_object_ptr<td_api::message>> &&messages, bool skip_not_found) {
  auto message_count = narrow_cast<int32>(messages.size());
  if (total_count < message_count) {
    if (total_count != -1) {
      LOG(ERROR) << "Have wrong total_count = " << total_count << ", while having " << message_count << " messages";
    }
    total_count = message_count;
  }
  if (skip_not_found && td::remove(messages, nullptr)) {
    total_count -= message_count - static_cast<int32>(messages.size());
  }
  return td_api::make_object<td_api::messages>(total_count, std::move(messages));
}

bool MessagesManager::get_dialog_silent_send_message(DialogId dialog_id) const {
  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return d->notification_settings.silent_send_message;
}

bool MessagesManager::get_dialog_has_last_message(DialogId dialog_id) const {
  const auto *d = get_dialog(dialog_id);
  return d != nullptr && d->last_message_id.is_valid();
}

DialogId MessagesManager::get_dialog_default_send_message_as_dialog_id(DialogId dialog_id) const {
  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return d->default_send_message_as_dialog_id;
}

MessageInputReplyTo MessagesManager::create_message_input_reply_to(
    DialogId dialog_id, MessageId top_thread_message_id, td_api::object_ptr<td_api::InputMessageReplyTo> &&reply_to,
    bool for_draft) {
  return create_message_input_reply_to(get_dialog(dialog_id), top_thread_message_id, std::move(reply_to), for_draft);
}

int64 MessagesManager::generate_new_random_id(const Dialog *d) {
  int64 random_id;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || being_sent_messages_.count(random_id) > 0 ||
           d->random_id_to_message_id.count(random_id) > 0);
  return random_id;
}

unique_ptr<MessagesManager::Message> MessagesManager::create_message_to_send(
    Dialog *d, MessageId top_thread_message_id, MessageInputReplyTo &&input_reply_to, const MessageSendOptions &options,
    unique_ptr<MessageContent> &&content, bool invert_media, bool suppress_reply_info,
    unique_ptr<MessageForwardInfo> forward_info, DialogId real_forward_from_dialog_id, bool is_copy,
    DialogId send_as_dialog_id) const {
  CHECK(d != nullptr);
  CHECK(content != nullptr);

  bool is_scheduled = options.schedule_date != 0;
  DialogId dialog_id = d->dialog_id;

  auto dialog_type = dialog_id.get_type();
  auto my_id = td_->user_manager_->get_my_id();

  int64 reply_to_random_id = 0;
  bool is_topic_message = false;
  auto initial_top_thread_message_id = top_thread_message_id;
  auto same_chat_reply_to_message_id = input_reply_to.get_same_chat_reply_to_message_id();
  if (same_chat_reply_to_message_id.is_valid() || same_chat_reply_to_message_id.is_valid_scheduled()) {
    // the message was forcely preloaded in create_message_input_reply_to
    // it can be missing, only if it is unknown message from a push notification, or an unknown top thread message
    const Message *reply_m = get_message(d, same_chat_reply_to_message_id);
    if (reply_m != nullptr && !same_chat_reply_to_message_id.is_scheduled()) {
      if (reply_m->top_thread_message_id.is_valid()) {
        top_thread_message_id = reply_m->top_thread_message_id;
      }
      is_topic_message = reply_m->is_topic_message;
    }
    if (dialog_type == DialogType::SecretChat || same_chat_reply_to_message_id.is_yet_unsent()) {
      if (reply_m != nullptr) {
        reply_to_random_id = reply_m->random_id;
      } else {
        CHECK(dialog_type == DialogType::SecretChat);
        CHECK(top_thread_message_id == MessageId());
        input_reply_to = MessageInputReplyTo();
      }
    }
  }
  if (top_thread_message_id.is_valid()) {
    const Message *top_m = get_message(d, top_thread_message_id);
    if (top_m != nullptr) {
      is_topic_message = top_m->is_topic_message;
    }
  }

  auto message = make_unique<Message>();
  auto *m = message.get();
  bool is_channel_post = td_->dialog_manager_->is_broadcast_channel(dialog_id);
  if (is_channel_post) {
    // sender of the post can be hidden
    if (!is_scheduled && td_->chat_manager_->get_channel_sign_messages(dialog_id.get_channel_id())) {
      m->author_signature = td_->user_manager_->get_user_title(my_id);
    }
    m->sender_dialog_id = dialog_id;
  } else {
    if (send_as_dialog_id.is_valid()) {
      if (send_as_dialog_id.get_type() == DialogType::User) {
        m->sender_user_id = send_as_dialog_id.get_user_id();
      } else {
        m->sender_dialog_id = send_as_dialog_id;
      }
    } else if (d->default_send_message_as_dialog_id.is_valid()) {
      if (d->default_send_message_as_dialog_id.get_type() == DialogType::User) {
        m->sender_user_id = my_id;
      } else {
        m->sender_dialog_id = d->default_send_message_as_dialog_id;
      }
      m->has_explicit_sender = true;
    } else {
      if (td_->dialog_manager_->is_anonymous_administrator(dialog_id, &m->author_signature)) {
        m->sender_dialog_id = dialog_id;
      } else {
        m->sender_user_id = my_id;
      }
    }
  }
  m->message_id = options.schedule_date != 0 ? get_next_yet_unsent_scheduled_message_id(d, options.schedule_date)
                                             : get_next_yet_unsent_message_id(d);
  m->send_date = G()->unix_time();
  m->date = is_scheduled ? options.schedule_date : m->send_date;
  m->replied_message_info = RepliedMessageInfo(td_, input_reply_to);
  m->reply_to_story_full_id = input_reply_to.get_story_full_id();
  m->input_reply_to = std::move(input_reply_to);
  m->reply_to_random_id = reply_to_random_id;
  m->top_thread_message_id = top_thread_message_id;
  m->initial_top_thread_message_id = initial_top_thread_message_id;
  m->is_topic_message = is_topic_message;
  m->is_channel_post = is_channel_post;
  m->is_outgoing = is_scheduled || dialog_id != DialogId(my_id);
  m->from_background = options.from_background;
  m->update_stickersets_order = options.update_stickersets_order;
  m->noforwards = options.protect_content;
  m->view_count = is_channel_post && !is_scheduled ? 1 : 0;
  m->forward_count = 0;
  if ([&] {
        if (suppress_reply_info) {
          return false;
        }
        if (is_scheduled) {
          return false;
        }
        if (dialog_type != DialogType::Channel) {
          return false;
        }
        if (td_->auth_manager_->is_bot()) {
          return false;
        }
        if (is_channel_post) {
          return td_->chat_manager_->get_channel_has_linked_channel(dialog_id.get_channel_id());
        }
        return !m->input_reply_to.is_valid();
      }()) {
    m->reply_info.reply_count_ = 0;
    if (is_channel_post) {
      auto linked_channel_id =
          td_->chat_manager_->get_channel_linked_channel_id(dialog_id.get_channel_id(), "create_message_to_send");
      if (linked_channel_id.is_valid()) {
        m->reply_info.is_comment_ = true;
        m->reply_info.channel_id_ = linked_channel_id;
      }
    }
  }
  if (m->sender_user_id == my_id && dialog_type == DialogType::Channel) {
    m->sender_boost_count = td_->chat_manager_->get_channel_my_boost_count(dialog_id.get_channel_id());
  }
  m->content = std::move(content);
  m->invert_media = invert_media;
  m->forward_info = std::move(forward_info);
  m->real_forward_from_dialog_id = real_forward_from_dialog_id;
  m->is_copy = is_copy || m->forward_info != nullptr;
  m->sending_id = options.sending_id;

  if (td_->auth_manager_->is_bot() || options.disable_notification ||
      td_->option_manager_->get_option_boolean("ignore_default_disable_notification")) {
    m->disable_notification = options.disable_notification;
  } else {
    m->disable_notification = d->notification_settings.silent_send_message;
  }

  if (dialog_type == DialogType::SecretChat) {
    CHECK(!is_scheduled);
    if (is_service_message_content(m->content->get_type())) {
      m->ttl = {};
    } else {
      m->ttl = MessageSelfDestructType(td_->user_manager_->get_secret_chat_ttl(dialog_id.get_secret_chat_id()), false);
    }
    m->is_content_secret = m->ttl.is_secret_message_content(m->content->get_type());
  }
  if (dialog_id == DialogId(my_id)) {
    m->saved_messages_topic_id = SavedMessagesTopicId(dialog_id, m->forward_info.get(), m->real_forward_from_dialog_id);
  }

  return message;
}

MessagesManager::Message *MessagesManager::get_message_to_send(
    Dialog *d, MessageId top_thread_message_id, MessageInputReplyTo &&input_reply_to, const MessageSendOptions &options,
    unique_ptr<MessageContent> &&content, bool invert_media, bool *need_update_dialog_pos, bool suppress_reply_info,
    unique_ptr<MessageForwardInfo> forward_info, DialogId real_forward_from_dialog_id, bool is_copy,
    DialogId send_as_dialog_id) {
  d->was_opened = true;

  auto message = create_message_to_send(d, top_thread_message_id, std::move(input_reply_to), options,
                                        std::move(content), invert_media, suppress_reply_info, std::move(forward_info),
                                        real_forward_from_dialog_id, is_copy, send_as_dialog_id);

  auto message_id = message->message_id;
  message->random_id = generate_new_random_id(d);

  bool need_update = false;
  CHECK(td_->dialog_manager_->have_input_peer(d->dialog_id, true, AccessRights::Read));
  auto result =
      add_message_to_dialog(d, std::move(message), false, true, &need_update, need_update_dialog_pos, "send message");
  LOG_CHECK(result != nullptr) << message_id << " " << debug_add_message_to_dialog_fail_reason_;
  if (result->message_id.is_scheduled()) {
    send_update_chat_has_scheduled_messages(d, false);
  }
  if (options.update_stickersets_order && !td_->auth_manager_->is_bot()) {
    move_message_content_sticker_set_to_top(td_, result->content.get());
  }
  return result;
}

int64 MessagesManager::begin_send_message(DialogId dialog_id, const Message *m) {
  LOG(INFO) << "Begin to send " << MessageFullId(dialog_id, m->message_id) << " with random_id = " << m->random_id;
  CHECK(m->random_id != 0);
  CHECK(m->message_id.is_yet_unsent());
  bool is_inserted = being_sent_messages_.emplace(m->random_id, MessageFullId(dialog_id, m->message_id)).second;
  CHECK(is_inserted);
  return m->random_id;
}

Status MessagesManager::can_send_message(DialogId dialog_id) const {
  if (!td_->dialog_manager_->have_input_peer(dialog_id, true, AccessRights::Write)) {
    return Status::Error(400, "Have no write access to the chat");
  }

  if (dialog_id.get_type() == DialogType::Channel) {
    auto channel_id = dialog_id.get_channel_id();
    auto channel_type = td_->chat_manager_->get_channel_type(channel_id);
    auto channel_status = td_->chat_manager_->get_channel_permissions(channel_id);

    switch (channel_type) {
      case ChannelType::Unknown:
      case ChannelType::Megagroup:
        break;
      case ChannelType::Broadcast: {
        if (!channel_status.can_post_messages()) {
          return Status::Error(400, "Need administrator rights in the channel chat");
        }
        break;
      }
      default:
        UNREACHABLE();
    }
  }
  return Status::OK();
}

MessageId MessagesManager::get_persistent_message_id(const Dialog *d, MessageId message_id) const {
  if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
    return MessageId();
  }
  if (message_id.is_yet_unsent()) {
    // it is possible that user tries to do something with an already sent message by its temporary identifier
    // we need to use real message in this case and transparently replace message_id
    auto it = yet_unsent_message_full_id_to_persistent_message_id_.find({d->dialog_id, message_id});
    if (it != yet_unsent_message_full_id_to_persistent_message_id_.end()) {
      return it->second;
    }
  }

  return message_id;
}

MessageInputReplyTo MessagesManager::create_message_input_reply_to(
    Dialog *d, MessageId top_thread_message_id, td_api::object_ptr<td_api::InputMessageReplyTo> &&reply_to,
    bool for_draft) {
  CHECK(d != nullptr);
  if (top_thread_message_id.is_valid() &&
      !have_message_force(d, top_thread_message_id, "create_message_input_reply_to 1")) {
    LOG(INFO) << "Have reply in the thread of unknown " << top_thread_message_id;
  }
  if (reply_to == nullptr) {
    if (!for_draft && top_thread_message_id.is_valid() && top_thread_message_id.is_server()) {
      return MessageInputReplyTo{top_thread_message_id, DialogId(), MessageQuote()};
    }
    return {};
  }
  switch (reply_to->get_id()) {
    case td_api::inputMessageReplyToStory::ID: {
      if (for_draft) {
        return {};
      }
      auto reply_to_story = td_api::move_object_as<td_api::inputMessageReplyToStory>(reply_to);
      auto story_id = StoryId(reply_to_story->story_id_);
      auto sender_dialog_id = DialogId(reply_to_story->story_sender_chat_id_);
      if (d->dialog_id != sender_dialog_id || td_->dialog_manager_->is_broadcast_channel(sender_dialog_id)) {
        LOG(INFO) << "Ignore reply to story from " << sender_dialog_id << " in a wrong " << d->dialog_id;
        return {};
      }
      if (!story_id.is_server()) {
        LOG(INFO) << "Ignore reply to invalid " << story_id;
        return {};
      }
      return MessageInputReplyTo{StoryFullId(sender_dialog_id, story_id)};
    }
    case td_api::inputMessageReplyToMessage::ID: {
      auto reply_to_message = td_api::move_object_as<td_api::inputMessageReplyToMessage>(reply_to);
      auto message_id = MessageId(reply_to_message->message_id_);
      if (!message_id.is_valid()) {
        if (!for_draft && message_id == MessageId() && top_thread_message_id.is_valid() &&
            top_thread_message_id.is_server()) {
          return MessageInputReplyTo{top_thread_message_id, DialogId(), MessageQuote()};
        }
        return {};
      }
      auto *reply_d = d;
      auto reply_dialog_id = DialogId(reply_to_message->chat_id_);
      if (reply_dialog_id != DialogId()) {
        reply_d = get_dialog_force(reply_dialog_id, "create_message_input_reply_to");
        if (reply_d == nullptr) {
          return {};
        }
        if (d->dialog_id.get_type() == DialogType::SecretChat) {
          return {};
        }
      }
      message_id = get_persistent_message_id(reply_d, message_id);
      if (message_id == MessageId(ServerMessageId(1)) && reply_d->dialog_id.get_type() == DialogType::Channel) {
        return {};
      }
      const Message *m = get_message_force(reply_d, message_id, "create_message_input_reply_to 2");
      if (m == nullptr || m->message_id.is_yet_unsent() ||
          (m->message_id.is_local() && reply_d->dialog_id.get_type() != DialogType::SecretChat)) {
        if (message_id.is_server() && reply_d->dialog_id.get_type() != DialogType::SecretChat &&
            reply_dialog_id == DialogId() && message_id > reply_d->last_new_message_id &&
            (reply_d->notification_info != nullptr &&
             message_id <= reply_d->notification_info->max_push_notification_message_id_)) {
          // allow to reply yet unreceived server message in the same chat
          return MessageInputReplyTo{message_id, reply_dialog_id,
                                     MessageQuote{td_, std::move(reply_to_message->quote_)}};
        }
        if (!for_draft && top_thread_message_id.is_valid() && top_thread_message_id.is_server()) {
          return MessageInputReplyTo{top_thread_message_id, DialogId(), MessageQuote()};
        }
        LOG(INFO) << "Can't find " << message_id << " in " << reply_d->dialog_id;

        // TODO local replies to local messages can be allowed
        // TODO replies to yet unsent messages can be allowed with special handling of them on application restart
        return {};
      }
      if (reply_dialog_id != DialogId() && (!can_forward_message(reply_dialog_id, m) || !m->message_id.is_server())) {
        LOG(INFO) << "Can't reply in another chat " << m->message_id << " in " << reply_d->dialog_id;
        return {};
      }
      return MessageInputReplyTo{m->message_id, reply_dialog_id,
                                 MessageQuote{td_, std::move(reply_to_message->quote_)}};
    }
    default:
      UNREACHABLE();
      return {};
  }
}

MessagesManager::ForwardedMessageInfo MessagesManager::get_forwarded_message_info(MessageFullId message_full_id) {
  ForwardedMessageInfo result;
  auto *m = get_message_force(message_full_id, "get_forwarded_message_info");
  if (m == nullptr || m->message_id.is_scheduled()) {
    return result;
  }
  auto dialog_id = message_full_id.get_dialog_id();
  result.origin_date_ = get_message_original_date(m);
  result.origin_ = get_forwarded_message_origin(dialog_id, m);
  result.content_ = dup_message_content(td_, td_->dialog_manager_->get_my_dialog_id(), m->content.get(),
                                        MessageContentDupType::Forward, MessageCopyOptions());
  return result;
}

const MessageInputReplyTo *MessagesManager::get_message_input_reply_to(const Message *m) {
  CHECK(m != nullptr);
  CHECK(!m->message_id.is_any_server());
  return &m->input_reply_to;
}

vector<FileId> MessagesManager::get_message_file_ids(const Message *m) const {
  CHECK(m != nullptr);
  auto file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (!m->replied_message_info.is_empty()) {
    append(file_ids, m->replied_message_info.get_file_ids(td_));
  }
  return file_ids;
}

void MessagesManager::cancel_upload_message_content_files(const MessageContent *content) {
  auto file_id = get_message_content_upload_file_id(content);
  // always cancel file upload, it should be a no-op in the worst case
  if (being_uploaded_files_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id, "cancel_upload_message_content_files");
  }
  file_id = get_message_content_thumbnail_file_id(content, td_);
  if (being_uploaded_thumbnails_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id, "cancel_upload_message_content_files");
  }
}

void MessagesManager::cancel_upload_file(FileId file_id, const char *source) {
  // send the request later so they doesn't interfere with other actions
  // for example merge, supposed to happen soon, can auto-cancel the upload
  LOG(INFO) << "Cancel upload of file " << file_id << " from " << source;
  send_closure_later(G()->file_manager(), &FileManager::cancel_upload, file_id);
}

void MessagesManager::cancel_send_message_query(DialogId dialog_id, Message *m) {
  CHECK(m != nullptr);
  CHECK(m->content != nullptr);
  CHECK(m->message_id.is_valid() || m->message_id.is_valid_scheduled());
  CHECK(m->message_id.is_yet_unsent());
  LOG(INFO) << "Cancel send message query for " << m->message_id;

  cancel_upload_message_content_files(m->content.get());

  CHECK(m->edited_content == nullptr);

  if (!m->send_query_ref.empty()) {
    LOG(INFO) << "Cancel send query for " << m->message_id;
    cancel_query(m->send_query_ref);
    m->send_query_ref = NetQueryRef();
  }

  if (m->send_message_log_event_id != 0) {
    LOG(INFO) << "Delete send message log event for " << m->message_id;
    binlog_erase(G()->td_db()->get_binlog(), m->send_message_log_event_id);
    m->send_message_log_event_id = 0;
  }

  update_replied_by_message_count(dialog_id, m, false);
  {
    auto it = replied_yet_unsent_messages_.find({dialog_id, m->message_id});
    if (it != replied_yet_unsent_messages_.end()) {
      for (auto message_full_id : it->second) {
        auto reply_d = get_dialog(message_full_id.get_dialog_id());
        CHECK(reply_d != nullptr);
        auto replied_m = get_message(reply_d, message_full_id.get_message_id());
        CHECK(replied_m != nullptr);
        const auto *input_reply_to = get_message_input_reply_to(replied_m);
        CHECK(input_reply_to != nullptr);
        CHECK(input_reply_to->get_reply_message_full_id(reply_d->dialog_id) == MessageFullId(dialog_id, m->message_id));
        set_message_reply(reply_d, replied_m,
                          MessageInputReplyTo{replied_m->top_thread_message_id, DialogId(), MessageQuote()}, true);
      }
      replied_yet_unsent_messages_.erase(it);
    }
  }

  if (m->media_album_id != 0) {
    send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id, dialog_id,
                       m->message_id, Status::OK());
  }

  if (!m->message_id.is_scheduled() && G()->keep_media_order() && !m->is_copy) {
    auto queue_id = ChainId(dialog_id, m->content->get_type()).get();
    if (queue_id & 1) {
      auto queue_it = yet_unsent_media_queues_.find(queue_id);
      if (queue_it != yet_unsent_media_queues_.end()) {
        auto &queue = queue_it->second.queue_;
        LOG(INFO) << "Delete " << m->message_id << " from queue " << queue_id;
        if (queue.erase(m->message_id) != 0) {
          if (queue.empty()) {
            yet_unsent_media_queues_.erase(queue_it);
          } else {
            // send later, because delete_all_dialog_messages can be called right now
            send_closure_later(actor_id(this), &MessagesManager::on_yet_unsent_media_queue_updated, dialog_id);
          }
        }
      }
    }
  }
}

void MessagesManager::cancel_send_deleted_message(DialogId dialog_id, Message *m, bool is_permanently_deleted) {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    cancel_send_message_query(dialog_id, m);
  } else if (is_permanently_deleted || !m->message_id.is_scheduled()) {
    cancel_edit_message_media(dialog_id, m, "Message was deleted");
  }
}

bool MessagesManager::is_message_auto_read(DialogId dialog_id, bool is_outgoing) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      auto user_id = dialog_id.get_user_id();
      if (user_id == td_->user_manager_->get_my_id()) {
        return true;
      }
      if (is_outgoing && td_->user_manager_->is_user_bot(user_id) && !td_->user_manager_->is_user_support(user_id)) {
        return true;
      }
      return false;
    }
    case DialogType::Chat:
      // TODO auto_read message content and messages sent to group with bots only
      return false;
    case DialogType::Channel:
      return is_outgoing && td_->dialog_manager_->is_broadcast_channel(dialog_id);
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::add_message_dependencies(Dependencies &dependencies, const Message *m) {
  auto is_bot = td_->auth_manager_->is_bot();
  dependencies.add(m->sender_user_id);
  dependencies.add_dialog_and_dependencies(m->sender_dialog_id);
  m->saved_messages_topic_id.add_dependencies(dependencies);
  m->replied_message_info.add_dependencies(dependencies, is_bot);
  dependencies.add_dialog_and_dependencies(m->reply_to_story_full_id.get_dialog_id());
  dependencies.add_dialog_and_dependencies(m->real_forward_from_dialog_id);
  dependencies.add(m->via_bot_user_id);
  dependencies.add(m->via_business_bot_user_id);
  if (m->forward_info != nullptr) {
    m->forward_info->add_dependencies(dependencies);
  }
  for (const auto &replier_min_channel : m->reply_info.replier_min_channels_) {
    LOG(INFO) << "Add min replied " << replier_min_channel.first;
    td_->chat_manager_->add_min_channel(replier_min_channel.first, replier_min_channel.second);
  }
  for (auto recent_replier_dialog_id : m->reply_info.recent_replier_dialog_ids_) {
    dependencies.add_message_sender_dependencies(recent_replier_dialog_id);
  }
  if (m->reactions != nullptr) {
    m->reactions->add_min_channels(td_);
    m->reactions->add_dependencies(dependencies);
  }
  add_message_content_dependencies(dependencies, m->content.get(), is_bot);
  add_reply_markup_dependencies(dependencies, m->reply_markup.get());
  add_draft_message_dependencies(dependencies, m->thread_draft_message);
}

void MessagesManager::get_dialog_send_message_as_dialog_ids(
    DialogId dialog_id, Promise<td_api::object_ptr<td_api::chatMessageSenders>> &&promise, bool is_recursive) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  TRY_RESULT_PROMISE(promise, d,
                     check_dialog_access(dialog_id, true, AccessRights::Read, "get_dialog_send_message_as_dialog_ids"));
  if (!d->default_send_message_as_dialog_id.is_valid()) {
    return promise.set_value(td_api::make_object<td_api::chatMessageSenders>());
  }
  CHECK(dialog_id.get_type() == DialogType::Channel);

  if (created_public_broadcasts_inited_) {
    auto senders = td_api::make_object<td_api::chatMessageSenders>();
    if (!created_public_broadcasts_.empty()) {
      auto add_sender = [&senders, td = td_](DialogId dialog_id, bool needs_premium) {
        auto sender = get_message_sender_object(td, dialog_id, "add_sender");
        senders->senders_.push_back(td_api::make_object<td_api::chatMessageSender>(std::move(sender), needs_premium));
      };
      if (td_->dialog_manager_->is_anonymous_administrator(dialog_id, nullptr)) {
        add_sender(dialog_id, false);
      } else {
        add_sender(td_->dialog_manager_->get_my_dialog_id(), false);
      }

      struct Sender {
        ChannelId channel_id;
        bool needs_premium;
      };
      std::multimap<int64, Sender> sorted_senders;

      bool is_premium = td_->option_manager_->get_option_boolean("is_premium");
      auto linked_channel_id = td_->chat_manager_->get_channel_linked_channel_id(
          dialog_id.get_channel_id(), "get_dialog_send_message_as_dialog_ids");
      for (auto channel_id : created_public_broadcasts_) {
        int64 score = td_->chat_manager_->get_channel_participant_count(channel_id);
        bool needs_premium =
            !is_premium && channel_id != linked_channel_id && !td_->chat_manager_->get_channel_is_verified(channel_id);
        if (needs_premium) {
          score -= static_cast<int64>(1) << 40;
        }
        if (channel_id == linked_channel_id) {
          score += static_cast<int64>(1) << 32;
        }
        sorted_senders.emplace(-score, Sender{channel_id, needs_premium});
      };

      for (auto &sender : sorted_senders) {
        add_sender(DialogId(sender.second.channel_id), sender.second.needs_premium);
      }
    }
    return promise.set_value(std::move(senders));
  }

  CHECK(!is_recursive);
  auto new_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, promise = std::move(promise)](
                                                Result<td_api::object_ptr<td_api::chats>> &&result) mutable {
    if (result.is_error()) {
      promise.set_error(result.move_as_error());
    } else {
      send_closure_later(actor_id, &MessagesManager::get_dialog_send_message_as_dialog_ids, dialog_id,
                         std::move(promise), true);
    }
  });
  td_->chat_manager_->get_created_public_dialogs(PublicDialogType::ForPersonalDialog, std::move(new_promise), true);
}

void MessagesManager::set_dialog_default_send_message_as_dialog_id(DialogId dialog_id,
                                                                   DialogId message_sender_dialog_id,
                                                                   Promise<Unit> &&promise) {
  TRY_RESULT_PROMISE(
      promise, d,
      check_dialog_access(dialog_id, false, AccessRights::Read, "set_dialog_default_send_message_as_dialog_id"));
  if (!d->default_send_message_as_dialog_id.is_valid()) {
    return promise.set_error(Status::Error(400, "Can't change message sender in the chat"));
  }
  // checked in on_update_dialog_default_send_message_as_dialog_id
  CHECK(dialog_id.get_type() == DialogType::Channel && !td_->dialog_manager_->is_broadcast_channel(dialog_id));

  bool is_anonymous = td_->dialog_manager_->is_anonymous_administrator(dialog_id, nullptr);
  switch (message_sender_dialog_id.get_type()) {
    case DialogType::User:
      if (message_sender_dialog_id != td_->dialog_manager_->get_my_dialog_id()) {
        return promise.set_error(Status::Error(400, "Can't send messages as another user"));
      }
      if (is_anonymous) {
        return promise.set_error(Status::Error(400, "Can't send messages as self"));
      }
      break;
    case DialogType::Chat:
    case DialogType::Channel:
    case DialogType::SecretChat:
      if (is_anonymous && dialog_id == message_sender_dialog_id) {
        break;
      }
      if (!td_->dialog_manager_->is_broadcast_channel(message_sender_dialog_id) ||
          td_->chat_manager_->get_channel_first_username(message_sender_dialog_id.get_channel_id()).empty()) {
        return promise.set_error(Status::Error(400, "Message sender chat must be a public channel"));
      }
      break;
    default:
      return promise.set_error(Status::Error(400, "Invalid message sender specified"));
  }
  if (!td_->dialog_manager_->have_input_peer(message_sender_dialog_id, true, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access specified message sender chat"));
  }

  td_->dialog_action_manager_->cancel_send_dialog_action_queries(dialog_id);

  td_->create_handler<SaveDefaultSendAsQuery>(std::move(promise))->send(dialog_id, message_sender_dialog_id);

  on_update_dialog_default_send_message_as_dialog_id(dialog_id, message_sender_dialog_id, true);
}

class MessagesManager::SendMessageLogEvent {
 public:
  DialogId dialog_id;
  const Message *m_in;
  unique_ptr<Message> message_out;

  SendMessageLogEvent() : dialog_id(), m_in(nullptr) {
  }

  SendMessageLogEvent(DialogId dialog_id, const Message *m) : dialog_id(dialog_id), m_in(m) {
  }

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id, parser);
    td::parse(message_out, parser);
  }
};

Result<td_api::object_ptr<td_api::message>> MessagesManager::send_message(
    DialogId dialog_id, const MessageId top_thread_message_id,
    td_api::object_ptr<td_api::InputMessageReplyTo> &&reply_to, tl_object_ptr<td_api::messageSendOptions> &&options,
    tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
    tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(400, "Can't send message without content");
  }

  Dialog *d = get_dialog_force(dialog_id, "send_message");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  auto input_reply_to = create_message_input_reply_to(d, top_thread_message_id, std::move(reply_to), false);

  if (input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    auto input_message = td_api::move_object_as<td_api::inputMessageForwarded>(input_message_content);
    TRY_RESULT(copy_options, process_message_copy_options(dialog_id, std::move(input_message->copy_options_)));
    copy_options.input_reply_to = std::move(input_reply_to);
    TRY_RESULT_ASSIGN(copy_options.reply_markup, get_dialog_reply_markup(dialog_id, std::move(reply_markup)));
    return forward_message(dialog_id, top_thread_message_id, DialogId(input_message->from_chat_id_),
                           MessageId(input_message->message_id_), std::move(options), input_message->in_game_share_,
                           std::move(copy_options));
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(message_reply_markup, get_dialog_reply_markup(dialog_id, std::move(reply_markup)));
  TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
  TRY_RESULT(message_send_options, process_message_send_options(dialog_id, std::move(options), true));
  TRY_STATUS(can_use_message_send_options(message_send_options, message_content));
  TRY_STATUS(can_use_top_thread_message_id(d, top_thread_message_id, input_reply_to));

  // there must be no errors after get_message_to_send call

  auto content = dup_message_content(td_, dialog_id, message_content.content.get(), MessageContentDupType::Send,
                                     MessageCopyOptions());
  bool need_update_dialog_pos = false;
  unique_ptr<Message> message;
  Message *m;
  if (message_send_options.only_preview) {
    message = create_message_to_send(d, top_thread_message_id, std::move(input_reply_to), message_send_options,
                                     std::move(content), message_content.invert_media, false, nullptr, DialogId(),
                                     message_content.via_bot_user_id.is_valid(), DialogId());
    m = message.get();
  } else {
    m = get_message_to_send(d, top_thread_message_id, std::move(input_reply_to), message_send_options,
                            std::move(content), message_content.invert_media, &need_update_dialog_pos, false, nullptr,
                            DialogId(), message_content.via_bot_user_id.is_valid());
  }
  m->reply_markup = std::move(message_reply_markup);
  m->via_bot_user_id = message_content.via_bot_user_id;
  m->disable_web_page_preview = message_content.disable_web_page_preview;
  m->clear_draft = message_content.clear_draft;
  if (message_content.ttl.is_valid()) {
    m->ttl = message_content.ttl;
    m->is_content_secret = m->ttl.is_secret_message_content(m->content->get_type());
  }
  m->send_emoji = std::move(message_content.emoji);

  if (message_send_options.only_preview) {
    return get_message_object(dialog_id, m, "send_message");
  }

  clear_dialog_draft_by_sent_message(d, m, !need_update_dialog_pos);

  save_send_message_log_event(dialog_id, m);
  do_send_message(dialog_id, m);

  if (!td_->auth_manager_->is_bot()) {
    send_update_new_message(d, m);
    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "send_message");
    }
  }

  return get_message_object(dialog_id, m, "send_message");
}

Result<InputMessageContent> MessagesManager::process_input_message_content(
    DialogId dialog_id, tl_object_ptr<td_api::InputMessageContent> &&input_message_content, bool check_permissions) {
  if (input_message_content != nullptr && input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    // for sendMessageAlbum/editMessageMedia/addLocalMessage
    auto input_message = td_api::move_object_as<td_api::inputMessageForwarded>(input_message_content);
    TRY_RESULT(copy_options, process_message_copy_options(dialog_id, std::move(input_message->copy_options_)));
    if (!copy_options.send_copy) {
      return Status::Error(400, "Can't use forwarded message");
    }

    DialogId from_dialog_id(input_message->from_chat_id_);
    Dialog *from_dialog = get_dialog_force(from_dialog_id, "send_message copy");
    if (from_dialog == nullptr) {
      return Status::Error(400, "Chat to copy message from not found");
    }
    if (!td_->dialog_manager_->have_input_peer(from_dialog_id, false, AccessRights::Read)) {
      return Status::Error(400, "Can't access the chat to copy message from");
    }
    MessageId message_id = get_persistent_message_id(from_dialog, MessageId(input_message->message_id_));

    const Message *copied_message = get_message_force(from_dialog, message_id, "process_input_message_content");
    if (copied_message == nullptr) {
      return Status::Error(400, "Can't find message to copy");
    }
    if (!can_forward_message(from_dialog_id, copied_message)) {
      return Status::Error(400, "Can't copy message");
    }
    if (!can_save_message(from_dialog_id, copied_message) && !td_->auth_manager_->is_bot()) {
      return Status::Error(400, "Message copying is restricted");
    }

    unique_ptr<MessageContent> content = dup_message_content(td_, dialog_id, copied_message->content.get(),
                                                             MessageContentDupType::Copy, std::move(copy_options));
    if (content == nullptr) {
      return Status::Error(400, "Can't copy message content");
    }

    return InputMessageContent(std::move(content), get_message_disable_web_page_preview(copied_message),
                               copied_message->invert_media, false, MessageSelfDestructType(), UserId(),
                               copied_message->send_emoji);
  }

  bool is_premium = td_->option_manager_->get_option_boolean("is_premium");
  TRY_RESULT(content, get_input_message_content(dialog_id, std::move(input_message_content), td_, is_premium));

  if (dialog_id != DialogId()) {
    TRY_STATUS(can_send_message_content(dialog_id, content.content.get(), false, check_permissions, td_));
  }

  return std::move(content);
}

Result<MessageCopyOptions> MessagesManager::process_message_copy_options(
    DialogId dialog_id, tl_object_ptr<td_api::messageCopyOptions> &&options) const {
  if (options == nullptr || !options->send_copy_) {
    return MessageCopyOptions();
  }
  MessageCopyOptions result;
  result.send_copy = true;
  result.replace_caption = options->replace_caption_;
  if (result.replace_caption) {
    TRY_RESULT_ASSIGN(result.new_caption, get_formatted_text(td_, dialog_id, std::move(options->new_caption_),
                                                             td_->auth_manager_->is_bot(), true, false, false));
  }
  return std::move(result);
}

Result<MessagesManager::MessageSendOptions> MessagesManager::process_message_send_options(
    DialogId dialog_id, tl_object_ptr<td_api::messageSendOptions> &&options,
    bool allow_update_stickersets_order) const {
  MessageSendOptions result;
  if (options == nullptr) {
    return std::move(result);
  }

  result.disable_notification = options->disable_notification_;
  result.from_background = options->from_background_;
  if (allow_update_stickersets_order) {
    result.update_stickersets_order = options->update_order_of_installed_sticker_sets_;
  }
  if (td_->auth_manager_->is_bot()) {
    result.protect_content = options->protect_content_;
  }
  result.only_preview = options->only_preview_;
  TRY_RESULT_ASSIGN(result.schedule_date, get_message_schedule_date(std::move(options->scheduling_state_)));
  result.sending_id = options->sending_id_;

  if (result.schedule_date != 0) {
    auto dialog_type = dialog_id.get_type();
    if (dialog_type == DialogType::SecretChat) {
      return Status::Error(400, "Can't schedule messages in secret chats");
    }
    if (td_->auth_manager_->is_bot()) {
      return Status::Error(400, "Bots can't send scheduled messages");
    }

    if (result.schedule_date == SCHEDULE_WHEN_ONLINE_DATE) {
      if (dialog_type != DialogType::User) {
        return Status::Error(400, "Messages can be scheduled till online only in private chats");
      }
      if (dialog_id == td_->dialog_manager_->get_my_dialog_id()) {
        return Status::Error(400, "Can't scheduled till online messages in chat with self");
      }
    }
  }

  return std::move(result);
}

Status MessagesManager::can_use_message_send_options(const MessageSendOptions &options,
                                                     const unique_ptr<MessageContent> &content,
                                                     MessageSelfDestructType ttl) {
  if (options.schedule_date != 0) {
    if (ttl.is_valid()) {
      return Status::Error(400, "Can't send scheduled self-destructing messages");
    }
    if (content->get_type() == MessageContentType::LiveLocation) {
      return Status::Error(400, "Can't send scheduled live location messages");
    }
  }

  return Status::OK();
}

Status MessagesManager::can_use_message_send_options(const MessageSendOptions &options,
                                                     const InputMessageContent &content) {
  return can_use_message_send_options(options, content.content, content.ttl);
}

Status MessagesManager::can_use_top_thread_message_id(Dialog *d, MessageId top_thread_message_id,
                                                      const MessageInputReplyTo &input_reply_to) {
  if (top_thread_message_id == MessageId()) {
    return Status::OK();
  }

  if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
    return Status::Error(400, "Invalid message thread identifier specified");
  }
  if (d->dialog_id.get_type() != DialogType::Channel || td_->dialog_manager_->is_broadcast_channel(d->dialog_id)) {
    return Status::Error(400, "Chat doesn't have threads");
  }
  if (input_reply_to.get_story_full_id().is_valid()) {
    return Status::Error(400, "Can't send story replies to the thread");
  }
  auto same_chat_reply_to_message_id = input_reply_to.get_same_chat_reply_to_message_id();
  if (same_chat_reply_to_message_id.is_valid()) {
    const Message *reply_m = get_message_force(d, same_chat_reply_to_message_id, "can_use_top_thread_message_id 1");
    if (reply_m != nullptr && top_thread_message_id != reply_m->top_thread_message_id) {
      if (reply_m->top_thread_message_id.is_valid() || reply_m->media_album_id == 0) {
        return Status::Error(400, "The message to be replied is not in the specified message thread");
      }

      // if the message is in an album and not in the thread, it can be in the album of top_thread_message_id
      const Message *top_m = get_message_force(d, top_thread_message_id, "can_use_top_thread_message_id 2");
      if (top_m != nullptr &&
          (top_m->media_album_id != reply_m->media_album_id || top_m->top_thread_message_id != top_m->message_id)) {
        return Status::Error(400, "The message to be replied is not in the specified message thread root album");
      }
    }
  }

  return Status::OK();
}

int64 MessagesManager::generate_new_media_album_id() {
  int64 media_album_id = 0;
  do {
    media_album_id = Random::secure_int64();
  } while (media_album_id >= 0 || pending_message_group_sends_.count(media_album_id) != 0);
  return media_album_id;
}

Result<td_api::object_ptr<td_api::messages>> MessagesManager::send_message_group(
    DialogId dialog_id, const MessageId top_thread_message_id,
    td_api::object_ptr<td_api::InputMessageReplyTo> &&reply_to, tl_object_ptr<td_api::messageSendOptions> &&options,
    vector<tl_object_ptr<td_api::InputMessageContent>> &&input_message_contents) {
  if (input_message_contents.size() > MAX_GROUPED_MESSAGES) {
    return Status::Error(400, "Too many messages to send as an album");
  }
  if (input_message_contents.empty()) {
    return Status::Error(400, "There are no messages to send");
  }

  Dialog *d = get_dialog_force(dialog_id, "send_message_group");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(message_send_options, process_message_send_options(dialog_id, std::move(options), true));

  vector<InputMessageContent> message_contents;
  std::unordered_set<MessageContentType, MessageContentTypeHash> message_content_types;
  for (auto &input_message_content : input_message_contents) {
    TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
    TRY_STATUS(can_use_message_send_options(message_send_options, message_content));
    auto message_content_type = message_content.content->get_type();
    if (!is_allowed_media_group_content(message_content_type)) {
      return Status::Error(400, "Invalid message content type");
    }
    message_content_types.insert(message_content_type);

    message_contents.push_back(std::move(message_content));
  }
  if (message_content_types.size() > 1) {
    for (auto message_content_type : message_content_types) {
      if (is_homogenous_media_group_content(message_content_type)) {
        return Status::Error(400, PSLICE() << message_content_type << " can't be mixed with other media types");
      }
    }
  }

  auto input_reply_to = create_message_input_reply_to(d, top_thread_message_id, std::move(reply_to), false);
  TRY_STATUS(can_use_top_thread_message_id(d, top_thread_message_id, input_reply_to));

  int64 media_album_id = 0;
  if (message_contents.size() > 1) {
    media_album_id = generate_new_media_album_id();
  }

  // there must be no errors after get_message_to_send calls

  vector<tl_object_ptr<td_api::message>> result;
  bool need_update_dialog_pos = false;
  for (size_t i = 0; i < message_contents.size(); i++) {
    auto &message_content = message_contents[i];
    unique_ptr<Message> message;
    Message *m;
    if (message_send_options.only_preview) {
      message = create_message_to_send(d, top_thread_message_id, input_reply_to.clone(), message_send_options,
                                       std::move(message_content.content), message_content.invert_media, i != 0,
                                       nullptr, DialogId(), false, DialogId());
      m = message.get();
    } else {
      m = get_message_to_send(d, top_thread_message_id, input_reply_to.clone(), message_send_options,
                              dup_message_content(td_, dialog_id, message_content.content.get(),
                                                  MessageContentDupType::Send, MessageCopyOptions()),
                              message_content.invert_media, &need_update_dialog_pos, i != 0);
    }

    auto ttl = message_content.ttl;
    if (ttl.is_valid()) {
      m->ttl = ttl;
      m->is_content_secret = m->ttl.is_secret_message_content(m->content->get_type());
    }
    m->media_album_id = media_album_id;

    result.push_back(get_message_object(dialog_id, m, "send_message_group"));

    if (!message_send_options.only_preview) {
      save_send_message_log_event(dialog_id, m);
      do_send_message(dialog_id, m);

      if (!td_->auth_manager_->is_bot()) {
        send_update_new_message(d, m);
      }
    }
  }

  if (need_update_dialog_pos) {
    CHECK(!message_send_options.only_preview);
    send_update_chat_last_message(d, "send_message_group");
  }

  return get_messages_object(-1, std::move(result), false);
}

void MessagesManager::save_send_message_log_event(DialogId dialog_id, const Message *m) {
  if (!G()->use_message_database()) {
    return;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << MessageFullId(dialog_id, m->message_id) << " to binlog";
  auto log_event = SendMessageLogEvent(dialog_id, m);
  CHECK(m->send_message_log_event_id == 0);
  m->send_message_log_event_id =
      binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendMessage, get_log_event_storer(log_event));
}

void MessagesManager::do_send_message(DialogId dialog_id, const Message *m, vector<int> bad_parts) {
  bool is_edit = m->message_id.is_any_server();
  LOG(INFO) << "Do " << (is_edit ? "edit" : "send") << ' ' << MessageFullId(dialog_id, m->message_id);
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;

  if (m->media_album_id != 0 && bad_parts.empty() && !is_secret && !is_edit) {
    auto &request = pending_message_group_sends_[m->media_album_id];
    request.dialog_id = dialog_id;
    request.message_ids.push_back(m->message_id);
    request.is_finished.push_back(false);

    request.results.push_back(Status::OK());
  }

  auto content = is_edit ? m->edited_content.get() : m->content.get();
  CHECK(content != nullptr);
  auto content_type = content->get_type();
  if (content_type == MessageContentType::Text) {
    CHECK(!is_edit);
    send_closure_later(actor_id(this), &MessagesManager::on_text_message_ready_to_send, dialog_id, m->message_id);
    return;
  }

  FileId file_id = get_message_content_any_file_id(content);  // any_file_id, because it could be a photo sent by ID
  FileId thumbnail_file_id = get_message_content_thumbnail_file_id(content, td_);
  LOG(DEBUG) << "Need to send file " << file_id << " with thumbnail " << thumbnail_file_id;
  if (is_secret) {
    CHECK(!is_edit);
    auto layer = td_->user_manager_->get_secret_chat_layer(dialog_id.get_secret_chat_id());
    auto secret_input_media = get_secret_input_media(content, td_, nullptr, BufferSlice(), layer);
    if (secret_input_media.empty()) {
      LOG(INFO) << "Ask to upload encrypted file " << file_id;
      CHECK(file_id.is_valid());
      FileView file_view = td_->file_manager_->get_file_view(file_id);
      CHECK(file_view.is_encrypted_secret());
      bool is_inserted =
          being_uploaded_files_
              .emplace(file_id, std::make_pair(MessageFullId(dialog_id, m->message_id), thumbnail_file_id))
              .second;
      CHECK(is_inserted);
      // need to call resume_upload synchronously to make upload process consistent with being_uploaded_files_
      td_->file_manager_->resume_up