//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Numeric/DSpinBox.cpp
//! @brief     Implements class DSpinBox.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Numeric/DSpinBox.h"
#include "Base/Math/Numeric.h"
#include "Base/Util/Assert.h"
#include "Base/Util/StringUtil.h"
#include "GUI/Model/Descriptor/DoubleProperty.h"
#include <QLineEdit>
#include <QRegularExpression>
#include <QWheelEvent>
#include <iostream>

namespace {

static constexpr double step0 = 0.1;

QString toString(double val, int decimal_points = 6)
{
    if (val == 0)
        return "0";

    QString result = (std::abs(val) >= 10000 || std::abs(val) < 0.1)
                         ? QString::number(val, 'e', decimal_points)
                         : QString::number(val, 'f', decimal_points);

    // suppress ".0" in mantissa; normalize exponent
    return result.replace(QRegularExpression("(\\.?0+)?((e)([\\+]?)([-]?)(0*)([1-9].*))?$"),
                          "\\3\\5\\7");
}

} // namespace


DSpinBox::DSpinBox(DoubleProperty* d)
    : m_step(::step0)
{
    replaceProperty(d);

    connect(this, &QAbstractSpinBox::editingFinished, [this] {
        ASSERT(m_property);
        setValue(fromDisplay());
        m_old_dir = 0;
        m_step = ::step0;
    });
    // setSingleStep(m_property->step());
}

void DSpinBox::replaceProperty(DoubleProperty* d)
{
    if (m_property)
        disconnect(m_property);
    m_property = d;
    if (m_property) {
        setFocusPolicy(Qt::StrongFocus);
        setToolTip(d->tooltip());
        setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
        lineEdit()->setText(toString(m_property->dVal()));
        connect(d, &DoubleProperty::setAndNotifyCalled, this, &DSpinBox::updateValue);
        //		[this] { updateValue(); });
    }
    setReadOnly(!m_property);
    updateValue();
}

void DSpinBox::updateValue()
{
    if (m_property)
        lineEdit()->setText(toString(m_property->dVal()));
    else
        lineEdit()->setText("");
}

QAbstractSpinBox::StepEnabled DSpinBox::stepEnabled() const
{
    return StepUpEnabled | StepDownEnabled;
}

double DSpinBox::fromDisplay() const
{
    double result;
    if (Base::String::to_double(lineEdit()->text().toStdString(), &result))
        return result;
    // invalid input => restore last valid value
    return m_property->dVal();
}

void DSpinBox::setValue(double val)
{
    ASSERT(m_property);
    const double oldval = m_property->dVal();
    val = m_property->limits().clamp(val);
    lineEdit()->setText(toString(val));
    m_property->setDVal(fromDisplay());
    if (m_property->dVal() != oldval)
        emit valueChanged(m_property->dVal());
}

void DSpinBox::wheelEvent(QWheelEvent* event)
{
    if (hasFocus())
        QAbstractSpinBox::wheelEvent(event);
    else
        event->ignore();
}

void DSpinBox::stepBy(int steps)
{
    ASSERT(m_property);

    const int new_dir = steps > 0 ? +1 : -1;
    if (m_old_dir == new_dir)
        m_step = std::min(m_step * (std::abs(steps) == 1 ? 1.2 : 2.) * (1 + m_step), 9.);
    else if (m_old_dir == -new_dir)
        m_step = std::abs(steps) == 1 ? std::max(m_step / 9, 1e-6) : m_step;

    const double fac = (steps > 0) ^ (m_property->dVal() < 0) ? 1 + m_step : 1 / (1 + m_step);
    const int decimals = std::min(6, std::max(2, (int)(2 - std::log10(m_step))));

    // std::cout << "DEBUG steps=" << steps << " m_s=" << m_step << " fac=" << fac
    //	      << " dec=" << decimals << " old=" << m_property->dVal()
    //	      << " new=" << m_property->dVal() * fac
    //	      << " rnd=" << Numeric::round_decimal(m_property->dVal() * fac, decimals)
    //	      << std::endl;
    setValue(Numeric::round_decimal(m_property->dVal() * fac, decimals));

    m_old_dir = new_dir;
}
