//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Device/AlphaScanEditor.cpp
//! @brief     Defines class AlphaScanEditor.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Device/AlphaScanEditor.h"
#include "GUI/Model/Beam/BeamDistributionItem.h"
#include "GUI/Model/Beam/DistributionItems.h"
#include "GUI/Model/Beam/SourceItems.h"
#include "GUI/View/Device/DistributionPlot.h"
#include "GUI/View/Device/DistributionSelector.h"
#include "GUI/View/Device/ScanRangeForm.h"

AlphaScanEditor::AlphaScanEditor(QWidget* parent, ScanItem* item, bool allow_distr)
    : StaticGroupBox("Grazing angles (deg)", parent)
    , m_item(item)
    , m_plot(new DistributionPlot(this))
{
    auto* layout = new QFormLayout(body());

    //... axis type combo
    auto* typeComboBox = new QComboBox;
    typeComboBox->addItem("Uniform axis");
    typeComboBox->addItem("Non-uniform axis"); // for use with experimental data ?
    const int idx = m_item->pointwiseAlphaAxisSelected() ? 1 : 0;
    layout->addRow("Axis type:", typeComboBox);

    //... axis parameters
    m_form = new ScanRangeForm(layout);
    connect(m_form, &ScanRangeForm::dataChanged, this, &AlphaScanEditor::dataChanged);

    //... beam distribution
    m_selector = new DistributionSelector(std::nullopt, DistributionSelector::Category::Symmetric,
                                          this, m_item->grazingScanItem(), allow_distr);
    connect(m_selector, &DistributionSelector::distributionChanged, this,
            &AlphaScanEditor::dataChanged);
    connect(m_selector, &DistributionSelector::distributionChanged, this,
            &AlphaScanEditor::updatePlot);
    layout->addRow(m_selector);

    //... update axis type combo (needs m_form)
    typeComboBox->setCurrentIndex(idx);
    onAxisTypeSelected(idx); // enable currently selected axis

    typeComboBox->setEnabled(m_item->pointwiseAlphaAxisDefined());
    connect(typeComboBox, &QComboBox::currentIndexChanged, this,
            &AlphaScanEditor::onAxisTypeSelected);

    //... distribution plot

    m_plot->setFixedHeight(170);
    m_plot->setShowMouseCoords(false);
    layout->addRow(m_plot);
    setFixedWidth(300);

    updatePlot();
}

void AlphaScanEditor::updateIndicators()
{
    m_form->updateData();
}

void AlphaScanEditor::onAxisTypeSelected(int index)
{
    if (m_item) {
        if (index == 0 && m_item->pointwiseAlphaAxisSelected()) {
            m_item->selectUniformAxis();
            emit dataChanged();
        } else if (index == 1 && !m_item->pointwiseAlphaAxisSelected()) {
            m_item->selectListScan();
            emit dataChanged();
        }

        m_form->setAxisItem(m_item->alphaAxisItem());
        m_form->setEnabled(index == 0);
    }
}

void AlphaScanEditor::updatePlot()
{
    auto* d = m_selector->item()->distributionItem();
    m_plot->setVisible(!dynamic_cast<const DistributionNoneItem*>(d));
    m_plot->setDistItem(d);
    m_plot->plotItem();
}
