!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubesimulate_primary
  use cube_types
  use cubetools_parameters
  use cubetools_structure
  use cubetools_axis_types
  use cubeadm_cubeid_types
  use cubeadm_cubeprod_types
  use cubesimulate_messaging
  use cubesimulate_observatory
  use cubesimulate_tuning
  !
  public :: primary
  private
  !
  type :: primary_comm_t
     type(option_t),    pointer :: comm
     type(cube_prod_t), pointer :: primbeam
     type(tuning_comm_t),      pointer :: tuning => null()
     type(observatory_comm_t), pointer :: observ => null()
   contains
     procedure, private :: init     => cubesimulate_primary_comm_init
     procedure, public  :: register => cubesimulate_primary_comm_register
     procedure, private :: parse    => cubesimulate_primary_comm_parse
     procedure, private :: main     => cubesimulate_primary_comm_main
  end type primary_comm_t
  type(primary_comm_t) :: primary
  !
  type primary_user_t
     type(cubeid_user_t) :: cubeids
   contains
     procedure, private :: toprog => cubesimulate_primary_user_toprog
  end type primary_user_t
  !
  type primary_prog_t
     type(tuning_prog_t),      pointer :: tuning => null()
     type(observatory_prog_t), pointer :: observ => null()
     type(cube_t),             pointer :: primbeam
     type(axis_t)                      :: freq
   contains
     procedure, private :: init   => cubesimulate_primary_prog_init
     procedure, private :: header => cubesimulate_primary_prog_header
     procedure, private :: data   => cubesimulate_primary_prog_data
     procedure, private :: loop   => cubesimulate_primary_prog_loop
     procedure, private :: act    => cubesimulate_primary_prog_act
  end type primary_prog_t
  !
contains
  !
  subroutine cubesimulate_primary_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(primary_user_t) :: user
    character(len=*), parameter :: rname='PRIMARY>COMMAND'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    call primary%parse(line,user,error)
    if (error) return
    call primary%main(user,error)
    if (error) continue
  end subroutine cubesimulate_primary_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubesimulate_primary_comm_init(comm,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_comm_t), intent(inout) :: comm
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='PRIMARY>COMM>INIT'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    comm%tuning => tuning_comm
    comm%observ => observatory_comm
  end subroutine cubesimulate_primary_comm_init
  !
  subroutine cubesimulate_primary_comm_register(comm,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_comm_t), intent(inout) :: comm
    logical,               intent(inout) :: error
    !
    type(cube_prod_t) :: oucube
    character(len=*), parameter :: rname='PRIMARY>COMM>REGISTER'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    ! Initialize pointers
    call comm%init(error)
    if (error) return
    !
    ! Syntax
    call cubetools_register_command(&
         'PRIMARY','',&
         'Simulate the primary beam of a telescope',&
         'The variation of the beam with frequency is taken into account',&
         cubesimulate_primary_command,&
         comm%comm,&
         error)
    if (error) return
    !
    ! Products
    call oucube%register(&
         'PRIMARY',&
         'Primary beam',&
         strg_id,&
         [flag_primary],&
         comm%primbeam,&
         error)
    if (error)  return
  end subroutine cubesimulate_primary_comm_register
  !
  subroutine cubesimulate_primary_comm_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! PRIMARY
    !----------------------------------------------------------------------
    class(primary_comm_t), intent(in)    :: comm
    character(len=*),      intent(in)    :: line
    type(primary_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='PRIMARY>COMM>PARSE'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    ! Nothing to do right now.
  end subroutine cubesimulate_primary_comm_parse
  !
  subroutine cubesimulate_primary_comm_main(comm,user,error) 
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_comm_t), intent(in)    :: comm
    type(primary_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(primary_prog_t) :: prog
    character(len=*), parameter :: rname='PRIMARY>COMM>MAIN'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(comm,error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubesimulate_primary_comm_main
  !
  !------------------------------------------------------------------------
  !
  subroutine cubesimulate_primary_user_toprog(user,comm,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_user_t), intent(in)    :: user
    type(primary_comm_t),  intent(in)    :: comm    
    type(primary_prog_t),  intent(out)   :: prog
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='PRIMARY>USER>TOPROG'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    ! Initialize pointers
    call prog%init(error)
    if (error) return
    !
    ! User feedback
    call prog%tuning%list(comm%tuning,error)
    if (error) return
    call prog%observ%list(comm%observ,error)
    if (error) return
  end subroutine cubesimulate_primary_user_toprog
  !
  !------------------------------------------------------------------------
  !
  subroutine cubesimulate_primary_prog_init(prog,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='PRIMARY>PROG>INIT'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    prog%tuning => tuning_prog
    prog%observ => observatory_prog
  end subroutine cubesimulate_primary_prog_init
  !
  subroutine cubesimulate_primary_prog_header(prog,comm,error)
    use phys_const
    !***JP: This command is not standard => This should not be cloned elsewhere
    use cubetools_header_interface
    use cubetools_header_types
    !***JP: This command is not standard => This should not be cloned elsewhere
    use cubetools_header_methods
    use cubetools_unit_setup
    use cubedag_allflags
    use cubeadm_create
    use cubedag_node
    !----------------------------------------------------------------------
    ! Create a cube header from scratch and initialize it
    !----------------------------------------------------------------------
    class(primary_prog_t), intent(inout) :: prog
    type(primary_comm_t),  intent(in)    :: comm
    logical,               intent(inout) :: error
    !
    integer(kind=ndim_k) :: ndim
    integer(kind=data_k) :: dims(maxdim)
    real(kind=coor_k) :: bbwidth,bbspacing
    real(kind=coor_k) :: driftwidth,driftspacing
    type(cube_header_interface_t) :: head
    character(len=*), parameter :: rname='PRIMARY>PROG>HEADER'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    bbwidth   = 3*8e+3 ! [MHz] Baseband width
    bbspacing = 195e-3 ! [MHz] Baseband spacing
    !
    driftwidth   = 200*rad_per_sec
    driftspacing =   4*rad_per_sec
    !
    ndim = 3
    dims(1:3) = [ceiling(driftwidth/driftspacing),1,ceiling(bbwidth/bbspacing)]
    dims(4:maxdim) = 0
    call cubeadm_create_header(&
         [flag_primary,flag_beam],code_access_imaset,ndim,dims,prog%primbeam,error)
    if (error) return
    !
    call cubetools_header_export(prog%primbeam%head,head,error)
    if (error) return
    !
    head%axset_name(1) = 'RA'
    head%axset_kind(1) = code_unit_fov
    head%axset_unit(1) = unitbuffer%prog_name(code_unit_fov)
    head%axset_convert(2:3,1) = [0d0,driftspacing]
    !
    head%axset_name(2) = 'DEC'
    head%axset_kind(2) = code_unit_fov
    head%axset_unit(2) = unitbuffer%prog_name(code_unit_fov)
    head%axset_convert(2:3,2) = [0d0,driftspacing]
    !
    head%axset_name(3) = 'FREQUENCY'
    head%axset_kind(3) = code_unit_freq
    head%axset_unit(3) = unitbuffer%prog_name(code_unit_freq)
    head%axset_convert(2:3,3) = [prog%tuning%freq,bbspacing]
    head%array_unit = "---"
    !
    head%spectral_frame_code = code_speframe_lsrk
    head%spectral_convention = code_speconv_radio
    head%spectral_line = prog%tuning%line
    head%spectral_code = code_spectral_frequency
    head%spectral_increment_value = bbspacing
    head%spectral_signal_value = prog%tuning%freq
    head%spectral_image_value = prog%tuning%fima
    call cubetools_header_import_and_derive(head,prog%primbeam%head,error)
    if (error) return
    !
    !***JP: Why the family name is not an input of the
    !***JP: cubeadm_create_header subroutine? This would avoid to use cubedag_node here!
    call cubedag_node_set_family(prog%primbeam,'coucou',error)
    if (error) return
    ! Prepare extrema processing. Assume no parallel put involved (1 pseudo-task)
!    call primbeam%proc%allocate_extrema(buffer%cube%head,1,error)
!    if (error)  return
    !
    call cubetools_header_get_axis_head_c(prog%primbeam%head,prog%freq,error)
    if (error) return
  end subroutine cubesimulate_primary_prog_header
  !
  subroutine cubesimulate_primary_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(primary_prog_t), intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='PRIMARY>PROG>DATA'
    !
    call cubesimulate_message(simulateseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(iter)
       if (.not.error) &
         call prog%loop(iter,error)
       !$OMP END TASK
    enddo
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubesimulate_primary_prog_data
  !   
  subroutine cubesimulate_primary_prog_loop(prog,iter,error)
    use cubeadm_taskloop
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_prog_t),    intent(inout) :: prog
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    !
    type(image_t) :: primbeam
    character(len=*), parameter :: rname='PRIMARY>PROG>LOOP'
    !
    call primbeam%allocate('primbeam',prog%primbeam,iter,error)
    if (error) return
    call primbeam%associate_xy(error)
    if (error) return
    !
    do while (iter%iterate_entry(error))
      call prog%act(iter%ie,primbeam,error)
      if (error) return
    enddo ! ie
  end subroutine cubesimulate_primary_prog_loop
  !   
  subroutine cubesimulate_primary_prog_act(prog,ie,primbeam,error)
    use phys_const
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(primary_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: primbeam
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: ix,iy
    real(kind=coor_k) :: fwhm
    character(len=*), parameter :: rname='PRIMARY>PROG>ACT'
    !
    fwhm = 20*rad_per_sec*prog%tuning%freq/prog%freq%coord(ie)
    !
    do iy=1,primbeam%ny
       do ix=1,primbeam%nx
          primbeam%val(ix,iy) = gauss1d(primbeam%x%coord(ix),1d0,0d0,fwhm)
       enddo ! ix
    enddo ! iy
    call primbeam%put(ie,error)
    if (error) return
    !
  contains
    !
    function gauss1d(x,peak,posi,fwhm) result(y)
      !----------------------------------------------------------------------
      ! Compute a Gaussian profile
      !----------------------------------------------------------------------
      real(kind=dble_k), intent(in) :: x
      real(kind=dble_k), intent(in) :: peak
      real(kind=dble_k), intent(in) :: posi
      real(kind=dble_k), intent(in) :: fwhm
      real(kind=dble_k)             :: y ! intent(out)
      !
      real(kind=dble_k), parameter :: sigma_per_fwhm = 1.665109
      real(kind=dble_k) :: sigma,arg
      !
      sigma = sigma_per_fwhm*fwhm
      arg = (x-posi)/sigma
      y = peak*exp(-arg**2)
    end function gauss1d
  end subroutine cubesimulate_primary_prog_act
end module cubesimulate_primary
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
