use crate::jf2;
use crate::parse;

#[derive(thiserror::Error, Debug)]
pub enum Error {
    #[error(transparent)]
    Parse(#[from] parse::Error),

    #[error(transparent)]
    Types(#[from] microformats_types::Error),

    #[error(transparent)]
    IO(#[from] std::io::Error),

    #[error("Failed to generate HTML: {0}")]
    HtmlCodegen(String),

    #[error(transparent)]
    FromUtf8(#[from] std::string::FromUtf8Error),

    #[error("The required property {name:} was not of the type {kind:?}")]
    InvalidRequiredProperty { name: String, kind: String },

    #[error(transparent)]
    Json(#[from] serde_json::Error),

    #[error(transparent)]
    Jf2Profile(#[from] jf2::profiles::Error),
}

impl PartialEq for Error {
    fn eq(&self, other: &Self) -> bool {
        self.to_string().eq(&other.to_string())
    }
}

impl From<url::ParseError> for Error {
    fn from(value: url::ParseError) -> Self {
        Self::Parse(parse::Error::from(value))
    }
}

impl From<microformats_types::temporal::Error> for Error {
    fn from(value: microformats_types::temporal::Error) -> Self {
        Self::Types(microformats_types::Error::from(value))
    }
}
