//
//  atari8c64draw
//  Part of ScottFree, an interpreter for adventures in Scott Adams format
//
//  Routines to draw Atari 8-bit and C64 RLE graphics
//  Based on Code by David Lodge 29/04/2005
//
//  Original code at https://github.com/tautology0/textadventuregraphics

#include <stdio.h>
#include <stdlib.h>

#include "sagagraphics.h"
#include "scott.h"

extern int x, y, count;
extern int xlen, ylen;
extern int xoff, yoff;
extern int size;

typedef uint8_t RGB[3];

static void DrawA8C64Pixels(int pattern, int pattern2)
{
    int pix1, pix2, pix3, pix4;

    if (x > (xlen - 3) * 8)
        return;

    pix1 = (pattern & 0xc0) >> 6;
    pix2 = (pattern & 0x30) >> 4;
    pix3 = (pattern & 0x0c) >> 2;
    pix4 = (pattern & 0x03);

    PutDoublePixel(x, y, pix1);
    x += 2;
    PutDoublePixel(x, y, pix2);
    x += 2;
    PutDoublePixel(x, y, pix3);
    x += 2;
    PutDoublePixel(x, y, pix4);
    x += 2;
    y++;
    x -= 8;

    pix1 = (pattern2 & 0xc0) >> 6;
    pix2 = (pattern2 & 0x30) >> 4;
    pix3 = (pattern2 & 0x0c) >> 2;
    pix4 = (pattern2 & 0x03);

    PutDoublePixel(x, y, pix1);
    x += 2;
    PutDoublePixel(x, y, pix2);
    x += 2;
    PutDoublePixel(x, y, pix3);
    x += 2;
    PutDoublePixel(x, y, pix4);
    x += 2;
    y++;
    x -= 8;

    if (y > ylen) {
        x += 8;
        y = yoff;
    }
}

/* C64 colors */
static const RGB black = { 0, 0, 0 };
static const RGB white = { 255, 255, 255 };
static const RGB red = { 191, 97, 72 };
static const RGB purple = { 177, 89, 185 };
static const RGB green = { 121, 213, 112 };
static const RGB blue = { 95, 72, 233 };
static const RGB yellow = { 247, 255, 108 };
static const RGB orange = { 186, 134, 32 };
static const RGB brown = { 131, 112, 0 };
static const RGB lred = { 231, 154, 132 };
static const RGB grey = { 167, 167, 167 };
static const RGB lgreen = { 192, 255, 185 };
static const RGB lblue = { 162, 143, 255 };

// clang-format off

/* Atari 8-bit colors */
RGB colors[256] = {
    { 0x00, 0x00, 0x00 }, // 0
    { 0x0e, 0x0e, 0x0e }, // 1
    { 0x1d, 0x1d, 0x1d }, // 2

    { 0x2c, 0x2c, 0x2c }, // 3
    { 0x3b, 0x3b, 0x3b }, // 4
    { 0x4a, 0x4a, 0x4a }, // 5
    { 0x59, 0x59, 0x59 }, // 6
    { 0x68, 0x68, 0x68 }, // 7
    { 0x77, 0x77, 0x77 }, // 8
    { 0x86, 0x86, 0x86 }, // 9
    { 0x95, 0x95, 0x95 }, // 10
    { 0xa4, 0xa4, 0xa4 }, // 11
    { 0xb3, 0xb3, 0xb3 }, // 12
    { 0xc2, 0xc2, 0xc2 }, // 13
//    { 0xd1, 0xd1, 0xd1 }, // 14
    { 0xe0, 0xe0, 0xe0 },
    { 0xe0, 0xe0, 0xe0 }, // 15
    { 0x34, 0x00, 0x00 }, // 16
    { 0x43, 0x0e, 0x00 }, // 17
//    { 0x52, 0x1d, 0x00 }, // 18
    { 0xad, 0x5f, 0x64 },
    { 0x61, 0x2c, 0x00 }, // 19
    { 0x70, 0x3b, 0x00 }, // 20
    { 0x7e, 0x4a, 0x00 }, // 21
    { 0x8d, 0x59, 0x00 }, // 22
    { 0x9c, 0x68, 0x00 }, // 23
    { 0xab, 0x77, 0x00 }, // 24
    { 0xba, 0x86, 0x00 }, // 25
    { 0xc9, 0x95, 0x0b }, // 26
    { 0xd8, 0xa4, 0x1a }, // 27
    { 0xe7, 0xb3, 0x29 }, // 28
    { 0xee, 0xc2, 0x38 }, // 29
    { 0xee, 0xd1, 0x46 }, // 30
    { 0xee, 0xe0, 0x55 }, // 31
    { 0x3a, 0x00, 0x00 }, // 32
    { 0x49, 0x01, 0x00 }, // 33
    { 0x58, 0x10, 0x00 }, // 34
    { 0x67, 0x1f, 0x00 }, // 35
    { 0x76, 0x2e, 0x00 }, // 36
//    { 0x85, 0x3d, 0x00 }, // 37
    { 0xad, 0x5f, 0x64 },
    { 0x94, 0x4c, 0x02 }, // 38
    { 0xa3, 0x5b, 0x11 }, // 39
    { 0xb2, 0x6a, 0x20 }, // 40
    { 0xc1, 0x79, 0x2f }, // 41
    { 0xd0, 0x88, 0x3e }, // 42
    { 0xdf, 0x97, 0x4d }, // 43
    { 0xee, 0xa6, 0x5c }, // 44
    { 0xee, 0xb5, 0x6b }, // 45
    { 0xee, 0xc4, 0x7a }, // 46
    { 0xee, 0xd2, 0x89 }, // 47
    { 0x36, 0x00, 0x00 }, // 48
    { 0x45, 0x00, 0x00 }, // 49
//    { 0x54, 0x05, 0x0b }, // 50
    { 0xad, 0x5f, 0x64 },
    { 0x62, 0x14, 0x1a }, // 51
    { 0x71, 0x23, 0x29 }, // 52
    { 0x80, 0x32, 0x38 }, // 53
//    { 0x8f, 0x41, 0x46 }, // 54
    { 0xad, 0x5f, 0x64 },
    { 0x9e, 0x50, 0x55 }, // 55
    { 0xad, 0x5f, 0x64 }, // 56
    { 0xbc, 0x6e, 0x73 }, // 57
//    { 0xcb, 0x7d, 0x82 }, // 58
    { 0xad, 0x5f, 0x64 },
    { 0xda, 0x8c, 0x91 }, // 59
    { 0xe9, 0x9a, 0xa0 }, // 60
    { 0xee, 0xa9, 0xaf }, // 61
    { 0xee, 0xb8, 0xbe }, // 62
    { 0xee, 0xc7, 0xcd }, // 63
    { 0x27, 0x00, 0x34 }, // 64
    { 0x36, 0x00, 0x43 }, // 65
    { 0x45, 0x00, 0x52 }, // 66
    { 0x54, 0x0f, 0x61 }, // 67
    { 0x62, 0x1e, 0x70 }, // 68
    { 0x71, 0x2d, 0x7e }, // 69
    { 0x80, 0x3c, 0x8d }, // 70
    { 0x8f, 0x4b, 0x9c }, // 71
    { 0x9e, 0x5a, 0xab }, // 72
    { 0xad, 0x69, 0xba }, // 73
    { 0xbc, 0x78, 0xc9 }, // 74
    { 0xcb, 0x87, 0xd8 }, // 75
    { 0xda, 0x96, 0xe7 }, // 76
    { 0xe9, 0xa5, 0xee }, // 77
    { 0xee, 0xb4, 0xee }, // 78
    { 0xee, 0xc3, 0xee }, // 79
    { 0x0f, 0x00, 0x71 }, // 80
    { 0x1e, 0x00, 0x80 }, // 81
    { 0x2d, 0x00, 0x8f }, // 82
    { 0x3c, 0x0f, 0x9e }, // 83
    { 0x4b, 0x1e, 0xad }, // 84
    { 0x5a, 0x2d, 0xbc }, // 85
//    { 0x69, 0x3c, 0xcb }, // 86
    { 0x4b, 0x1e, 0xad },
    { 0x78, 0x4b, 0xda }, // 87
    { 0x87, 0x5a, 0xe9 }, // 88
    { 0x96, 0x69, 0xee }, // 89
    { 0xa5, 0x78, 0xee }, // 90
    { 0xb4, 0x87, 0xee }, // 91
    { 0xc3, 0x96, 0xee }, // 92
    { 0xd2, 0xa5, 0xee }, // 93
    { 0xe0, 0xb4, 0xee }, // 94
    { 0xee, 0xc3, 0xee }, // 95
    { 0x00, 0x00, 0x99 }, // 96
    { 0x04, 0x00, 0xa7 }, // 97
    { 0x13, 0x07, 0xb6 }, // 98
    { 0x22, 0x16, 0xc5 }, // 99
    { 0x31, 0x25, 0xd4 }, // 100
    { 0x40, 0x34, 0xe3 }, // 101
    { 0x4f, 0x43, 0xee }, // 102
    { 0x5e, 0x52, 0xee }, // 103
    { 0x6d, 0x61, 0xee }, // 104
    { 0x7c, 0x70, 0xee }, // 105
    { 0x8b, 0x7e, 0xee }, // 106
    { 0x9a, 0x8d, 0xee }, // 107
    { 0xa8, 0x9c, 0xee }, // 108
    { 0xb7, 0xab, 0xee }, // 109
    { 0xc6, 0xba, 0xee }, // 110
    { 0xd5, 0xc9, 0xee }, // 111
    { 0x00, 0x00, 0xa1 }, // 112
    { 0x00, 0x02, 0xb0 }, // 113
    { 0x00, 0x11, 0xbf }, // 114
    { 0x0a, 0x20, 0xce }, // 115
    { 0x19, 0x2f, 0xdd }, // 116
    { 0x28, 0x3e, 0xec }, // 117
    { 0x37, 0x4d, 0xee }, // 118
    { 0x46, 0x5c, 0xee }, // 119
    { 0x54, 0x6b, 0xee }, // 120
    { 0x63, 0x7a, 0xee }, // 121
    { 0x72, 0x89, 0xee }, // 122
    { 0x81, 0x98, 0xee }, // 123
    { 0x90, 0xa7, 0xee }, // 124
    { 0x9f, 0xb6, 0xee }, // 125
    { 0xae, 0xc4, 0xee }, // 126
    { 0xbd, 0xd3, 0xee }, // 127
    { 0x00, 0x00, 0x8a }, // 128
    { 0x00, 0x0e, 0x99 }, // 129
    { 0x00, 0x1d, 0xa8 }, // 130
    { 0x00, 0x2c, 0xb6 }, // 131
    { 0x07, 0x3b, 0xc5 }, // 132
//    { 0x16, 0x4a, 0xd4 }, // 133
    { 0x34, 0x68, 0xee },
    { 0x25, 0x59, 0xe3 }, // 134
    { 0x34, 0x68, 0xee }, // 135
    { 0x43, 0x77, 0xee }, // 136
    { 0x52, 0x86, 0xed }, // 137
    { 0x61, 0x95, 0xee }, // 138
    { 0x70, 0xa4, 0xee }, // 139
    { 0x7e, 0xb3, 0xee }, // 140
    { 0x8d, 0xc2, 0xee }, // 141
    { 0x9c, 0xd1, 0xee }, // 142
    { 0xab, 0xe0, 0xee }, // 143
    { 0x00, 0x0d, 0x56 }, // 144
    { 0x00, 0x1c, 0x65 }, // 145
    { 0x00, 0x2a, 0x74 }, // 146
    { 0x00, 0x39, 0x83 }, // 147
    { 0x00, 0x48, 0x92 }, // 148
    { 0x0f, 0x57, 0xa1 }, // 149
    { 0x1e, 0x66, 0xb0 }, // 150
    { 0x2d, 0x75, 0xbf }, // 151
    { 0x3c, 0x84, 0xce }, // 152
    { 0x4b, 0x93, 0xdd }, // 153
    { 0x5a, 0xa2, 0xec }, // 154
    { 0x69, 0xb1, 0xee }, // 155
    { 0x78, 0xc0, 0xee }, // 156
    { 0x87, 0xcf, 0xee }, // 157
    { 0x96, 0xde, 0xee }, // 158
    { 0xa5, 0xed, 0xee }, // 159
    { 0x00, 0x18, 0x12 }, // 160
    { 0x00, 0x27, 0x21 }, // 161
    { 0x00, 0x36, 0x30 }, // 162
    { 0x00, 0x45, 0x3f }, // 163
    { 0x05, 0x54, 0x4e }, // 164
    { 0x14, 0x62, 0x5d }, // 165
    { 0x23, 0x71, 0x6c }, // 166
    { 0x32, 0x80, 0x7b }, // 167
    { 0x41, 0x8f, 0x8a }, // 168
    { 0x50, 0x9e, 0x99 }, // 169
    { 0x5f, 0xad, 0xa8 }, // 170
    { 0x6e, 0xbc, 0xb6 }, // 171
    { 0x7d, 0xcb, 0xc5 }, // 172
    { 0x8c, 0xda, 0xd4 }, // 173
    { 0x9a, 0xe9, 0xe3 }, // 174
    { 0xa9, 0xee, 0xee }, // 175
    { 0x00, 0x1c, 0x00 }, // 176
    { 0x00, 0x2b, 0x00 }, // 177
    { 0x00, 0x3a, 0x00 }, // 178
    { 0x05, 0x49, 0x00 }, // 179
    { 0x14, 0x58, 0x07 }, // 180
    { 0x23, 0x67, 0x16 }, // 181
    { 0x32, 0x76, 0x25 }, // 182
    { 0x41, 0x85, 0x34 }, // 183
    { 0x50, 0x94, 0x43 }, // 184
    { 0x5f, 0xa3, 0x52 }, // 185
    { 0x6e, 0xb2, 0x61 }, // 186
    { 0x7d, 0xc1, 0x70 }, // 187
    { 0x8c, 0xd0, 0x7e }, // 188
    { 0x9a, 0xdf, 0x8d }, // 189
    { 0xa9, 0xee, 0x9c }, // 190
    { 0xb8, 0xee, 0xab }, // 191
    { 0x00, 0x1c, 0x00 }, // 192
    { 0x00, 0x2b, 0x00 }, // 193
    { 0x0e, 0x3a, 0x00 }, // 194
    { 0x1c, 0x49, 0x00 }, // 195
    { 0x2b, 0x58, 0x00 }, // 196
    { 0x3a, 0x67, 0x00 }, // 197
//    { 0x49, 0x76, 0x00 }, // 198
    { 0x2b, 0x58, 0x00 },
//    { 0x58, 0x85, 0x00 }, // 199
    { 0x3a, 0x67, 0x00 }, // 197
    { 0x67, 0x94, 0x05 }, // 200
    { 0x76, 0xa3, 0x14 }, // 201
    { 0x85, 0xb2, 0x23 }, // 202
    { 0x94, 0xc1, 0x32 }, // 203
    { 0xa3, 0xd0, 0x41 }, // 204
    { 0xb2, 0xdf, 0x50 }, // 205
    { 0xc1, 0xee, 0x5f }, // 206
    { 0xd0, 0xee, 0x6e }, // 207
    { 0x0a, 0x16, 0x00 }, // 208
    { 0x19, 0x25, 0x00 }, // 209
    { 0x28, 0x34, 0x00 }, // 210
    { 0x37, 0x43, 0x00 }, // 211
    { 0x46, 0x52, 0x00 }, // 212
    { 0x54, 0x61, 0x00 }, // 213
    { 0x63, 0x70, 0x00 }, // 214
    { 0x72, 0x7e, 0x00 }, // 215
//    { 0x81, 0x8d, 0x00 }, // 216
    { 0x63, 0x70, 0x00 },
    { 0x90, 0x9c, 0x00 }, // 217
    { 0x9f, 0xab, 0x00 }, // 218
    { 0xae, 0xba, 0x0b }, // 219
    { 0xbd, 0xc9, 0x1a }, // 220
    { 0xcc, 0xd8, 0x29 }, // 221
    { 0xdb, 0xe7, 0x38 }, // 222
    { 0xea, 0xee, 0x46 }, // 223
    { 0x22, 0x0c, 0x00 }, // 224
    { 0x31, 0x1b, 0x00 }, // 225
    { 0x40, 0x2a, 0x00 }, // 226
    { 0x4f, 0x38, 0x00 }, // 227
//    { 0x5e, 0x47, 0x00 }, // 228
    { 0x94, 0x4c, 0x02 },
    { 0x6d, 0x56, 0x00 }, // 229
    { 0x7c, 0x65, 0x00 }, // 230
    { 0x8b, 0x74, 0x00 }, // 231
    { 0x9a, 0x83, 0x00 }, // 232
    { 0xa8, 0x92, 0x00 }, // 233
    { 0xb7, 0xa1, 0x00 }, // 234
    { 0xc6, 0xb0, 0x02 }, // 235
    { 0xd5, 0xbf, 0x11 }, // 236
    { 0xe4, 0xce, 0x20 }, // 237
    { 0xee, 0xdd, 0x2f }, // 238
    { 0xee, 0xec, 0x3e }, // 239
    { 0x34, 0x00, 0x00 }, // 240
    { 0x43, 0x0e, 0x00 }, // 241
    { 0x52, 0x1d, 0x00 }, // 242
    { 0x61, 0x2c, 0x00 }, // 243
    { 0x70, 0x3b, 0x00 }, // 244
    { 0x7e, 0x4a, 0x00 }, // 245
    { 0x8d, 0x59, 0x00 }, // 246
//    { 0x9c, 0x68, 0x00 }, // 247
    { 0xad, 0x5f, 0x64 },
//    { 0xab, 0x77, 0x00 }, // 248
    { 0x8d, 0x59, 0x00 },
    { 0xba, 0x86, 0x00 }, // 249
    { 0xc9, 0x95, 0x0b }, // 250
    { 0xd8, 0xa4, 0x1a }, // 251
    { 0xe7, 0xb3, 0x29 }, // 252
    { 0xee, 0xc2, 0x38 }, // 253
    { 0xee, 0xd1, 0x46 }, // 254
//    { 0xee, 0xe0, 0x55 }, // 255
    { 0xba, 0x86, 0x00 }
};

// clang-format on

static void TranslateAtariColorRGB(int index, uint8_t value)
{
    SetColor(index, &colors[value]);
}

/*
 The values below are determined by looking at the games
 running in the VICE C64 emulator.
 I have no idea how the original interpreter calculates
 them. I might have made some mistakes. */

static void TranslateC64Color(int index, uint8_t value)
{
    debug_print("Color %d: %d\n", index, value);
    switch (value) {
    case 2:
    case 3:
    case 4:
    case 8:
    case 9:
    case 10:
    case 12:
    case 14:
    case 15:
    case 255:
    case 137:
    case 142:
        SetColor(index, &white);
        break;
    case 35:
    case 36:
    case 38:
    case 40:
    case 244:
    case 246:
    case 248:
        SetColor(index, &brown);
        break;
    case 24:
    case 26:
    case 30:
    case 46:
    case 16:
    case 230:
    case 237:
    case 238:
    case 252:
        SetColor(index, &yellow);
        break;
    case 50:
    case 51:
    case 52:
    case 53:
    case 54:
    case 56:
    case 58:
    case 59:
    case 60:
    case 66:
    case 62:
        SetColor(index, &orange);
        break;
    case 67:
    case 68:
    case 69:
    case 70:
    case 71:
        SetColor(index, &red);
        break;
    case 0:
    case 77:
    case 81:
    case 84:
    case 85:
    case 86:
    case 87:
    case 97:
    case 101:
    case 102:
    case 103:
    case 105:
    case 224:
        SetColor(index, &purple);
        break;
    case 89:
        SetColor(index, &lred);
        break;
    case 1:
    case 7:
    case 116:
    case 135:
    case 148:
    case 151:
        SetColor(index, &blue);
        break;
    case 110:
    case 157:
        SetColor(index, &lblue);
        break;
    case 161:
        SetColor(index, &grey);
        break;
    case 17:
    case 20:
    case 179:
    case 182:
    case 183:
    case 194:
    case 195:
    case 196:
    case 197:
    case 198:
    case 199:
    case 200:
    case 212:
    case 214:
    case 215:
    case 216:
        SetColor(index, &green);
        break;
    case 201:
        SetColor(index, &lgreen);
        break;
    default:
        fprintf(stderr, "Unknown color %d ", value);
        break;
    }
}

int DrawAtariC64Image(USImage *image)
{
    uint8_t *ptr = image->imagedata;
    size_t datasize = image->datasize;

    int work, work2;
    int c;
    int i;
    int countflag = (CurrentGame == COUNT_US || CurrentGame == VOODOO_CASTLE_US);

    uint8_t *origptr = ptr;

    x = 0;
    y = 0;

    ptr += 2;

    work = *ptr++;
    size = work + *ptr++ * 256;
    // Get the offset
    xoff = *ptr++ - 3;
    yoff = *ptr++;
    x = xoff * 8;
    y = yoff;

    // Get the x length
    xlen = *ptr++;

    ylen = *ptr++;
    if (countflag)
        ylen -= 2;

    glui32 curheight, curwidth;
    glk_window_get_size(Graphics, &curwidth, &curheight);

    if (image->usage == IMG_ROOM || (xlen == 38 && xoff < 1)) {
        xlen = xlen - 1 - image->cropright / 8;
        left_margin = image->cropleft;

        ImageWidth = xlen * 8 - 17 + left_margin;
        ImageHeight = ylen + 2;

        if (image->index == 19 && image->systype == SYS_ATARI8) {
            xlen++;
            ImageWidth = 308;
        }

        int width_in_pixels = ImageWidth * pixel_size;
        while (width_in_pixels > curwidth && pixel_size > 1) {
            pixel_size--;
            width_in_pixels = ImageWidth * pixel_size;
        }

        x_offset = (curwidth - width_in_pixels) / 2;

        int optimal_height = ImageHeight * pixel_size;

        if (curheight != optimal_height) {
            winid_t parent = glk_window_get_parent(Graphics);
            if (parent)
                glk_window_set_arrangement(parent, winmethod_Above | winmethod_Fixed,
                    optimal_height, NULL);
        }
    }

    SetColor(0, &black);

    // Get the palette
    for (i = 1; i < 5; i++) {
        work = *ptr++;
        if (image->systype == SYS_C64)
            TranslateC64Color(i, work);
        else
            TranslateAtariColorRGB(i, work);
    }

    while (ptr - origptr < datasize - 2) {
        // First get count
        c = *ptr++;

        if (((c & 0x80) == 0x80) || countflag) { // is a counter
            if (countflag) {
                c -= 1;
            } else {
                c &= 0x7f;
            }
            work = *ptr++;
            work2 = *ptr++;
            for (i = 0; i < c + 1; i++) {
                DrawA8C64Pixels(work, work2);
            }
        } else {
            // Don't count on the next c characters
            for (i = 0; i < c + 1 && ptr - origptr < datasize - 1; i++) {
                work = *ptr++;
                work2 = *ptr++;
                DrawA8C64Pixels(work, work2);
            }
        }
    }
    return 1;
}
